﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>

#include <nn/nifm/nifm_Common.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/detail/nifm_CommonDetail.h>
#include <nn/nifm/nifm_TypesNetworkProfile.h>

#include <nn/util/util_Uuid.h>
#include <nn/util/util_StringUtil.h>


namespace
{
    static bool g_CheckPermanency = false;

    static const nn::nifm::NetworkProfileData g_UserNetworkProfile[] = {
        {
            { {  } },
            "ProfilePermanencyTest 1",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,
            true,
            false,
            { { { { 14,{ 0x6e,0x73,0x64,0x2d,0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x30,0x31 } }, false },{ { nn::nifm::Authentication_Open, nn::nifm::Encryption_None },{} } } },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { {  } },
            "ProfilePermanencyTest 2",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            false,
            true,
            { { { { 0,{ } }, false },{ { nn::nifm::Authentication_Invalid, nn::nifm::Encryption_Invalid },{} } } },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1300 },
        },
        {
            { {  } },
            "ProfilePermanencyTest 3",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,
            false,
            false,
            { { { { 10,{0x34,0x20,0x4a,0x55,0x43,0xa4,0x54,0x3a,0x45,0xae } }, true },{ { nn::nifm::Authentication_Wpa2Psk, nn::nifm::Encryption_Aes },{12, "testPassword"} } } },
            { { false,{{192,168,11,9}},{{255,255,168,0}},{{192,168,11,100}} },{ false,{{111,222,112,221}},{{123,123,123,123}} },{ false, 0, "",{ false, "", "" } }, 1500 },
        },
        {
            { {  } },
            "ProfilePermanencyTest 4",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,
            true,
            true,
            { { { { 12,{ 0x2a,0x45,0x14,0x6b,0x65,0x2d,0x30,0x31,0x32,0xaa,0x92,0x40 } }, false },{ { nn::nifm::Authentication_Open, nn::nifm::Encryption_Wep },{5, "test1"} } } },
            { { false,{{192,168,11,10}},{{255,255,255,0}},{{192,168,11,1}} },{ false,{{1,1,1,2}},{{3,3,4,5}} },{ true, 1000, "proxyname.net",{ true, "ethernet_username", "ethernet_password" } }, 1000 },
        },
    };

    void TestProfile(const nn::nifm::NetworkProfileData& networkProfileData, const nn::nifm::NetworkProfileBasicInfo& networkProfileBasicInfo)
    {
        EXPECT_EQ(0, nn::util::Strncmp(networkProfileData.name, networkProfileBasicInfo.name, nn::nifm::NetworkProfileBasicInfo::NameSize));
        //EXPECT_EQ(0, memcmp(networkProfileData.id.data, networkProfileBasicInfo.id.data, nn::util::Uuid::Size));
        EXPECT_EQ(networkProfileData.networkProfileType, networkProfileBasicInfo.networkProfileType);
        EXPECT_EQ(networkProfileData.wirelessSetting.ssidConfig.ssid.length, networkProfileBasicInfo.ssid.length);
        EXPECT_EQ(0, memcmp(networkProfileData.wirelessSetting.ssidConfig.ssid.hex, networkProfileBasicInfo.ssid.hex, networkProfileBasicInfo.ssid.length));
        EXPECT_EQ(networkProfileData.wirelessSetting.security.authEncryption.authentication, networkProfileBasicInfo.authentication);
        EXPECT_EQ(networkProfileData.wirelessSetting.security.authEncryption.encryption, networkProfileBasicInfo.encryption);

        nn::nifm::NetworkProfileData networkProfile;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetNetworkProfile(&networkProfile, networkProfileBasicInfo.id));

        EXPECT_EQ(networkProfileData.wirelessSetting.ssidConfig.ssid.length, networkProfile.wirelessSetting.ssidConfig.ssid.length);
        EXPECT_EQ(0, memcmp(networkProfileData.wirelessSetting.ssidConfig.ssid.hex, networkProfile.wirelessSetting.ssidConfig.ssid.hex, networkProfile.wirelessSetting.ssidConfig.ssid.length));
        EXPECT_EQ(networkProfileData.wirelessSetting.ssidConfig.nonBroadcast, networkProfile.wirelessSetting.ssidConfig.nonBroadcast);
        EXPECT_EQ(networkProfileData.wirelessSetting.security.authEncryption.authentication, networkProfile.wirelessSetting.security.authEncryption.authentication);
        EXPECT_EQ(networkProfileData.wirelessSetting.security.authEncryption.encryption, networkProfile.wirelessSetting.security.authEncryption.encryption);
        EXPECT_EQ(0, memcmp(networkProfileData.wirelessSetting.security.sharedKey.keyMaterial, networkProfile.wirelessSetting.security.sharedKey.keyMaterial, networkProfile.wirelessSetting.security.sharedKey.length));

        EXPECT_EQ(networkProfileData.ipSetting.ip.isAuto, networkProfile.ipSetting.ip.isAuto);
        EXPECT_EQ(0, memcmp(networkProfileData.ipSetting.ip.ipAddress.data, networkProfile.ipSetting.ip.ipAddress.data, nn::nifm::IpV4Address::Size));
        EXPECT_EQ(0, memcmp(networkProfileData.ipSetting.ip.subnetMask.data, networkProfile.ipSetting.ip.subnetMask.data, nn::nifm::IpV4Address::Size));
        EXPECT_EQ(0, memcmp(networkProfileData.ipSetting.ip.defaultGateway.data, networkProfile.ipSetting.ip.defaultGateway.data, nn::nifm::IpV4Address::Size));
        EXPECT_EQ(networkProfileData.ipSetting.dns.isAuto, networkProfile.ipSetting.dns.isAuto);
        EXPECT_EQ(0, memcmp(networkProfileData.ipSetting.dns.preferredDns.data, networkProfile.ipSetting.dns.preferredDns.data, nn::nifm::IpV4Address::Size));
        EXPECT_EQ(0, memcmp(networkProfileData.ipSetting.dns.alternateDns.data, networkProfile.ipSetting.dns.alternateDns.data, nn::nifm::IpV4Address::Size));
        EXPECT_EQ(networkProfileData.ipSetting.proxy.isEnabled, networkProfile.ipSetting.proxy.isEnabled);
        EXPECT_EQ(networkProfileData.ipSetting.proxy.port, networkProfile.ipSetting.proxy.port);
        EXPECT_EQ(0, strncmp(networkProfileData.ipSetting.proxy.proxy, networkProfile.ipSetting.proxy.proxy, nn::nifm::ProxySetting::ProxyNameSize));
        EXPECT_EQ(networkProfileData.ipSetting.proxy.authentication.isEnabled, networkProfile.ipSetting.proxy.authentication.isEnabled);
        EXPECT_EQ(0, strncmp(networkProfileData.ipSetting.proxy.authentication.username, networkProfile.ipSetting.proxy.authentication.username, nn::nifm::AuthenticationSetting::UsernameSize));
        EXPECT_EQ(0, strncmp(networkProfileData.ipSetting.proxy.authentication.password, networkProfile.ipSetting.proxy.authentication.password, nn::nifm::AuthenticationSetting::PasswordSize));
        EXPECT_EQ(networkProfileData.ipSetting.mtu, networkProfile.ipSetting.mtu);
    }
}

class ProfilePermanencyCaseTest : public ::testing::Test
{
protected:
    virtual void TearDown() { }
};

TEST_F(ProfilePermanencyCaseTest, CreateProfile)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    if (g_CheckPermanency)
    {
        return;
    }

    const int bufferSize = nn::nifm::UserNetworkProfileCountMax;

    // 削除
    {
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[bufferSize];
        int outCount;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize, nn::nifm::NetworkProfileType_User));
        EXPECT_LE(outCount, bufferSize);

        for (int i = outCount; i > 0; --i)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::RemoveNetworkProfile(networkProfileBasicInfo[i - 1].id));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize, nn::nifm::NetworkProfileType_User));
            EXPECT_EQ(i - 1, outCount);
        }
    }

    // 作成
    {
        nn::util::Uuid id;

        for (int i = 0; i < NN_ARRAY_SIZE(g_UserNetworkProfile); ++i)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetNetworkProfile(&id, g_UserNetworkProfile[i]));
        }
    }
}

TEST_F(ProfilePermanencyCaseTest, CheckProfile)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    if (!g_CheckPermanency)
    {
        return;
    }

    const int bufferSize = nn::nifm::UserNetworkProfileCountMax;

    // 確認
    {
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[bufferSize];
        int outCount;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize, nn::nifm::NetworkProfileType_User));
        EXPECT_LE(outCount, bufferSize);

        for (int i = 0; i < outCount; ++i)
        {
            NN_LOG("check %dth profile.\n", i);
            TestProfile(g_UserNetworkProfile[i], networkProfileBasicInfo[i]);
        }
    }

}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    for (int i = 0; i < argc; ++i)
    {
        if (std::strcmp(argv[i], "--check") == 0)
        {
            g_CheckPermanency = true;
        }
    }

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
