﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_ApiForLibcurl.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_ApiNetworkConnection.h>
#include <nn/nifm/nifm_ApiIpAddress.h>
#include <nn/nifm/nifm_ApiRequestPrivate.h>

#include <nn/nifm/nifm_Request.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_TemporaryNetworkProfile.h>
#include <nn/nifm/nifm_TypesRequest.h>
#include <nn/nifm/nifm_TypesRequestPrivate.h>
#include <nn/nifm/nifm_TypesNetworkProfile.h>
#include <nn/nifm/nifm_TypesNetworkInterface.h>
#include <nn/nifm/nifm_ApiWirelessCommunicationControl.h>
#include <nn/nifm/nifm_ApiEthernetCommunicationControl.h>
#include <nn/nifm/nifm_ApiCommunicationControlForTest.h>
#include <nn/nifm/nifm_ApiInternetConnectionStatus.h>
#include <nn/nifm/nifm_ApiWirelessConnectionInfo.h>

#include <nn/nifm/detail/nifm_CommonDetail.h>
#include <nn/nifm/detail/nifm_RequestClient.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include <nn/util/util_Uuid.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_ScopeExit.h>

#include <algorithm>
#include <cstring>

#include "../Common/nifm_TestUtility.h"

namespace
{
    // 基準となる優先度
    static const int BasePriority = 100;

    // 利用要求の即座の受理または却下を期待した待ち時間
    const int64_t TIME_OUT_IN_SECONDS_FOR_SUCCESS = 60;

    // ブロッキングしている利用要求を取り下げる
    void CleanUpThreadFunc(void* p)
    {
        nn::nifm::detail::RequestClient* pRequestClient = reinterpret_cast<nn::nifm::detail::RequestClient*>(p);
        nn::os::SystemEvent& systemEvent = pRequestClient->GetSystemEvent();

        while (NN_STATIC_CONDITION(true))
        {
            systemEvent.Wait();
            // systemEvent を clear してはいけない

            nn::nifm::RequestState requestState = pRequestClient->GetRequestState();

            switch (requestState)
            {
            case nn::nifm::RequestState::RequestState_Blocking:
                // cleanup
                pRequestClient->Cancel();
                return;

            default:
                break;
            }
        }
    }

    uint64_t g_LastWisprPassedTick = 0;

    // 最後の WISPr 認証から指定時間以上経過したか
    bool IsLongerThan(uint64_t seconds)
    {
        return g_LastWisprPassedTick == 0 ? true : ((nn::os::GetSystemTick().GetInt64Value() - g_LastWisprPassedTick) / nn::os::GetSystemTickFrequency()) > seconds;
    }

    //
    nn::Result ExpectAdditionalInfo(const nn::util::Uuid& id, const char* pRedirectUrl, nn::nifm::RequestHandle handle) NN_NOEXCEPT
    {
        nn::nifm::AdditionalInfo additionalInfo;
        uint32_t revision;
        NN_RESULT_DO(nn::nifm::GetRequestAdditionalInfo(&additionalInfo, &revision, handle));
        EXPECT_EQ(id, additionalInfo.profileId);
        EXPECT_EQ(0, nn::util::Strncmp(pRedirectUrl, additionalInfo.redirectUrl, nn::nifm::AdditionalInfo::RedirectUrlSize));
        NN_LOG("additionalInfo.redirectUrl=%s\n", additionalInfo.redirectUrl);
        NN_RESULT_SUCCESS;
    }

    // IpV4Address から ulong への変換
    unsigned long IpV4AddressToUlong(const nn::nifm::IpV4Address& address) NN_NOEXCEPT
    {
        const nn::Bit8* p = address.data;
        return (p[0] << 0 & 0x000000ff) | (p[1] << 8 & 0x0000ff00) | (p[2] << 16 & 0x00ff0000) | (p[3] << 24 & 0xff000000);
    }
}

class ConnectCaseTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
        nn::nifm::test::ElapsedTime(nn::os::GetSystemTick().GetInt64Value());

        // 接続制御が許可されているとしてテスト実行
        m_IsCommunicationControlEnabledBackup = nn::nifm::test::IsCommunicationControlEnabled();
        nn::nifm::test::SetCommunicationControlEnabled(true);
    }

    static void TearDownTestCase()
    {
        nn::nifm::Initialize();
        NN_UTIL_SCOPE_EXIT
        {
            nn::nifm::FinalizeForTest();
        };

        // 接続制御許可を元に戻す
        nn::nifm::test::SetCommunicationControlEnabled(m_IsCommunicationControlEnabledBackup);

        // 独占解除
        nn::nifm::test::SetExclusive<nn::nifm::Initialize, nn::nifm::FinalizeForTest>(false);

        NN_LOG("TotalTime [%s]\n", nn::nifm::test::ElapsedTime());
    }

private:
    static bool m_IsCommunicationControlEnabledBackup;
};
bool ConnectCaseTest::m_IsCommunicationControlEnabledBackup = false;

TEST_F(ConnectCaseTest, Initialize)
{
    NN_LOG("Initialize/FinalizeForTest:");
    for (int i = 0; i < 10; ++i)
    {
        NN_LOG(" %d", i);
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeForTest());
    }
    NN_LOG("\n");

    NN_LOG("InitializeSystem/FinalizeSystemForTest:");
    for (int i = 0; i < 10; ++i)
    {
        NN_LOG(" %d", i);
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeSystem());
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeSystemForTest());
    }
    NN_LOG("\n");

    NN_LOG("InitializeAdmin/FinalizeAdminForTest:");
    for (int i = 0; i < 10; ++i)
    {
        NN_LOG(" %d", i);
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
    }
    NN_LOG("\n");
}

TEST_F(ConnectCaseTest, NetworkInterfaceList)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::nifm::NetworkInterfaceInfo networkInterfaceInfo[8];
    int bufferSize;
    int outCount;

    // 利用対象の NIC を列挙
    bufferSize = nn::nifm::NetworkInterfaceCountMax;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ieee80211 | nn::nifm::EnumerateNetworkInterfacesFilter_Ethernet
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // 利用対象の無線 NIC を列挙
    bufferSize = nn::nifm::NetworkInterfaceCountMax;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ieee80211
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // 利用対象の有線 NIC を列挙
    bufferSize = nn::nifm::NetworkInterfaceCountMax;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ethernet
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // 無線 NIC を全部列挙
    bufferSize = 8;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ieee80211 | nn::nifm::EnumerateNetworkInterfacesFilter_Extended
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // 有線 NIC を全部列挙
    bufferSize = 8;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ethernet | nn::nifm::EnumerateNetworkInterfacesFilter_Extended
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // NIC を全部列挙
    bufferSize = 8;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ieee80211 | nn::nifm::EnumerateNetworkInterfacesFilter_Ethernet | nn::nifm::EnumerateNetworkInterfacesFilter_Extended
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // バッファサイズより大きい値が outCount に入る
    bufferSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ieee80211 | nn::nifm::EnumerateNetworkInterfacesFilter_Ethernet | nn::nifm::EnumerateNetworkInterfacesFilter_Extended
        )
    );
    ASSERT_GT(outCount, bufferSize);

    // フィルタを指定しなければ何も取れない
    bufferSize = nn::nifm::NetworkInterfaceCountMax;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            0
        )
    );
    ASSERT_EQ(outCount, 0);

    // 定義外のフィルタビットを立てると失敗
    bufferSize = nn::nifm::NetworkInterfaceCountMax;
    NNT_ASSERT_RESULT_FAILURE(
        nn::nifm::ResultInvalidArgument,
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            ~static_cast<nn::Bit32>(0)
        )
    );
}

// 実際の接続が行われるテストの中で最初に実行される
TEST_F(ConnectCaseTest, Basic)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    // Dispatch Loop 開始時のスキャンでつながるはず
    //NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    {
        // 非接続状態では，現在接続中のアクセスポイント情報を取得できない
        nn::nifm::AccessPointData accessPointData;
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetCurrentAccessPoint(&accessPointData));
        uint16_t channel;
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetCurrentAccessPointChannel(&channel));

        // 非接続状態では，現在利用中の接続設定を取得できない
        nn::nifm::NetworkProfileData networkProfileData;
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetCurrentNetworkProfile(&networkProfileData));
    }

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        // 接続 ～ 切断
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultInvalidRequestState, ExpectAdditionalInfo(nn::util::InvalidUuid, "", request1a.GetHandle()));

        request1a.Cancel();
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, request1a.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(ExpectAdditionalInfo(nn::util::InvalidUuid, "", request1a.GetHandle()));
    }

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersNone);
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(request1a.GetHandle(), nn::nifm::RequirementPreset_InternetGeneric));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        // 接続 ～ 切断
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultInvalidRequestState, ExpectAdditionalInfo(nn::util::InvalidUuid, "", request1a.GetHandle()));

        request1a.Cancel();
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, request1a.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(ExpectAdditionalInfo(nn::util::InvalidUuid, "", request1a.GetHandle()));
    }
}

// 特定の DNS を設定して疎通確認時のリダイレクトを検証1
TEST_F(ConnectCaseTest, WISPrAuthTest1)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::nifm::Scan();

    const nn::nifm::IpSettingData ipSetting = {
        {  // ip
            true,  // isAuto
            {},  // ipAddress
            {},  // subnetMask
            {}  // defaultGateway
        },
        {  // dns
            false,  // isAuto
            { { 52, 68, 203, 240 } },  // preferredDns
            {}  // alternateDns
        },
        {  // proxy
            false,  // isEnabled
            0,  // port
            "",  // proxy
            {  // authentication
                false,  // isEnabled
                "",  // username
                ""  // password
            }
        },
        1400  //mtu
    };

    nn::util::Uuid id;
    id.FromString("1a765924-cfb0-4833-a9da-aa81d83aea4a"); // mmt

    nn::nifm::IpSettingData defaultIpSetting;
    nn::nifm::GetDefaultIpSetting(&defaultIpSetting);

    nn::nifm::SetDefaultIpSetting(ipSetting);

    // 新規接続で WISPr 認証
    {
        nn::nifm::NetworkConnection networkConnection;

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), id));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultInvalidRequestState, ExpectAdditionalInfo(nn::util::InvalidUuid, "", networkConnection.GetRequestHandle()));

        nn::nifm::NetworkProfileData networkProfileData;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetCurrentNetworkProfile(&networkProfileData));
        EXPECT_EQ(id, networkProfileData.id);

        g_LastWisprPassedTick = nn::os::GetSystemTick().GetInt64Value();

        // TODO:
        // ...
    }

    nn::nifm::SetDefaultIpSetting(defaultIpSetting);
}

// WISPr 認証失敗するエントリを SSID リストに入れた場合のみテスト可能
#if 0
TEST_F(ConnectCaseTest, WISPrAuthTestFailed)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::nifm::Scan();

    const nn::nifm::IpSettingData ipSetting = {
        {  // ip
            true,  // isAuto
            {},  // ipAddress
            {},  // subnetMask
            {}  // defaultGateway
        },
        {  // dns
            false,  // isAuto
            { { 52, 68, 203, 240 } },  // preferredDns
            {}  // alternateDns
        },
        {  // proxy
            false,  // isEnabled
            0,  // port
            "",  // proxy
            {  // authentication
                false,  // isEnabled
                "",  // username
                ""  // password
            }
        },
        1400  //mtu
    };

    nn::util::Uuid id;
    id.FromString("1a765924-cfb0-4833-a9da-aa81d83aea4b"); // imatake-wpa2aes(SSIDlist)

    nn::nifm::Requirement requirement = { BasePriority, true, false, true, false, false, true, true, true, false, true, nn::nifm::NetworkType_Infrastructure,
        id, nn::nifm::ConnectionConfirmationOption_Forced };

    nn::nifm::IpSettingData defaultIpSetting;
    nn::nifm::GetDefaultIpSetting(&defaultIpSetting);

    nn::nifm::SetDefaultIpSetting(ipSetting);

    // 新規接続で WISPr 認証
    {
        nn::nifm::NetworkConnection networkConnection;

        nn::nifm::detail::RequestClient* pRequestClient = nn::nifm::GetRequestClientPointer(networkConnection.GetRequestHandle());
        ASSERT_NE(nullptr, pRequestClient);
        NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRequirement(requirement));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultHotspotAuthenticationViaWebAuthAppletNeeded, networkConnection.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(ExpectAdditionalInfo(id, "http://ctest.cdn.nintendo.net/", networkConnection.GetRequestHandle()));

        g_LastWisprPassedTick = nn::os::GetSystemTick().GetInt64Value();

        // TODO:
        // ...
    }

    nn::nifm::SetDefaultIpSetting(defaultIpSetting);
}
#endif

TEST_F(ConnectCaseTest, Scan)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    {
        // 通常のスキャンの成功と、スキャン結果が更新されていることの確認

        const int ListSize = 64;
        nn::nifm::AccessPointData accessPointDataList1[ListSize] = {};
        nn::nifm::AccessPointData accessPointDataList2[ListSize] = {};
        int count1, count2;

        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetScanData(accessPointDataList1, &count1, ListSize));

        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::Scan());

        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetScanData(accessPointDataList2, &count2, ListSize));

        {
            // WEP はスキャン結果からは認証タイプがわからない

            bool isWepFound = false;
            int count = std::min(std::min(count2, ListSize), nn::nifm::AccessPointCountMax);

            for (int i = 0; i < count; ++i)
            {
                const auto& accessPointData = accessPointDataList2[i];

                if (accessPointData.encryption == nn::nifm::Encryption_Wep)
                {
                    EXPECT_EQ(nn::nifm::Authentication_Unknown, accessPointData.authentication);

                    isWepFound = true;
                }
            }

            EXPECT_TRUE(isWepFound);
        }

        EXPECT_TRUE(std::memcmp(accessPointDataList1, accessPointDataList2, sizeof(nn::nifm::AccessPointData) * std::min(std::min(count1, count2), ListSize)) != 0);
    }

    {
        // 無線オフ時にはスキャンできない

        nn::nifm::SetWirelessCommunicationEnabled(false);
        EXPECT_FALSE(nn::nifm::IsWirelessCommunicationEnabled());

        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceNotAvailable, nn::nifm::Scan());

        nn::nifm::SetWirelessCommunicationEnabled(true);
        EXPECT_TRUE(nn::nifm::IsWirelessCommunicationEnabled());
    }

    {
        // 帯域を占有したい利用要求が受理されているときにはスキャンできる

        nn::nifm::NetworkConnection networkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnection.GetRequestHandle(), true));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(networkConnection.IsAvailable());

        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::Scan());
    }

    {
        // ローカル通信中でもローカル通信を切断してスキャンできる

        NN_ALIGNAS(nn::os::ThreadStackAlignment) static char stackOfThread[4 * 1024];
        nn::os::ThreadType CleanUpThread;

        nn::nifm::NetworkConnection networkConnectionLocal;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionLocal.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));

        networkConnectionLocal.SubmitRequestAndWait();
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionLocal.GetResult());

        NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&CleanUpThread,
            [](void* p) {
            auto pNetworkConnection = reinterpret_cast<nn::nifm::NetworkConnection*>(p);
            nn::os::SystemEvent& systemEvent = pNetworkConnection->GetSystemEvent();
            while (NN_STATIC_CONDITION(true))
            {
                systemEvent.Wait();
                nn::nifm::RequestState requestState = pNetworkConnection->GetRequestState();
                switch (requestState)
                {
                case nn::nifm::RequestState::RequestState_Blocking:
                    pNetworkConnection->CancelRequest();
                    return;
                default:
                    break;
                }
            }
        },
            &networkConnectionLocal, &stackOfThread, sizeof(stackOfThread), 24));
        nn::os::StartThread(&CleanUpThread);

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());
    }

    {
        // スキャンできる状態に確実に戻すために、通常のインターネット利用要求を出す

        nn::nifm::NetworkConnection networkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(networkConnection.IsAvailable());

        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::Scan());
    }
}

TEST_F(ConnectCaseTest, Recycle)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        // 接続 ～ 切断
        for( int i = 0; i < 10; ++i )
        {
            NN_LOG("Try %d\n", i);
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultInvalidRequestState, ExpectAdditionalInfo(nn::util::InvalidUuid, "", request1a.GetHandle()));

            request1a.Cancel();
            NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, request1a.GetResult());
        }
    }
}

TEST_F(ConnectCaseTest, TakeOverConnection)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    {
        nn::nifm::Request request0a(nn::nifm::test::RequestParametersNone);
        nn::nifm::Request request0b(nn::nifm::test::RequestParametersNone);
        nn::nifm::Request request1b(nn::nifm::test::RequestParametersNone);
        nn::nifm::Request request1c(nn::nifm::test::RequestParametersNone);
        nn::nifm::Request request1d(nn::nifm::test::RequestParametersNone);

        // 接続

        // requirement0
        nn::nifm::detail::RequestClient *pRequestClient = nn::nifm::GetRequestClientPointer(request0a.GetHandle());
        NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRequirementPreset(nn::nifm::RequirementPreset_LocalGeneric));
        NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRawPriority(BasePriority));
        NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRejectable(true));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request0a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request0a.GetResult());

        // ローカル通信接続では，現在接続中のアクセスポイント情報を取得できない
        nn::nifm::AccessPointData accessPointData;
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLocalConnection, nn::nifm::GetCurrentAccessPoint(&accessPointData));
        uint16_t channel;
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLocalConnection, nn::nifm::GetCurrentAccessPointChannel(&channel));

        // ローカル通信接続では，現在利用中の接続設定を取得できない
        nn::nifm::NetworkProfileData networkProfileData;
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLocalConnection, nn::nifm::GetCurrentNetworkProfile(&networkProfileData));

        // 優先度の高い異種要求で乗っ取り（Local:100 -> Internet:99）
        // requirement1
        pRequestClient = nn::nifm::GetRequestClientPointer(request1b.GetHandle());
        ASSERT_NE(nullptr, pRequestClient);
        NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRequirementPreset(nn::nifm::RequirementPreset_InternetGeneric));
        NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRawPriority(BasePriority - 1));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1b, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1b.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Free, request0a.GetRequestState());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLowPriority, request0a.GetResult());

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersNone);

            // 優先度の高い異種要求で乗っ取り
            // requirement2(Internet:99 -> Local:98)
            pRequestClient = nn::nifm::GetRequestClientPointer(request2a.GetHandle());
            ASSERT_NE(nullptr, pRequestClient);
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRequirementPreset(nn::nifm::RequirementPreset_LocalGeneric));
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRawPriority(BasePriority - 2));
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRejectable(true));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(request2a.GetResult());
            EXPECT_EQ(nn::nifm::RequestState_Free, request1b.GetRequestState());
            NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLowPriority, request1b.GetResult());

            // 優先度の低い同種要求で相乗り（相乗り可能なローカル通信要求）
            // requirement0(Local), requirement2(Local)
            pRequestClient = nn::nifm::GetRequestClientPointer(request0b.GetHandle());
            ASSERT_NE(nullptr, pRequestClient);
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRequirementPreset(nn::nifm::RequirementPreset_LocalGeneric));
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRawPriority(BasePriority));
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRejectable(true));
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetGreedy(false));
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetSharable(true));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request0b, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(request0b.GetResult());
            EXPECT_EQ(nn::nifm::RequestState_Accepted, request2a.GetRequestState());
            NNT_ASSERT_RESULT_SUCCESS(request2a.GetResult());

            // 優先度の低い異種要求で乗っ取り失敗
            // requirement0(Local), requirement2(Local)
            pRequestClient = nn::nifm::GetRequestClientPointer(request1c.GetHandle());
            ASSERT_NE(nullptr, pRequestClient);
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRequirementPreset(nn::nifm::RequirementPreset_InternetGeneric));
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRawPriority(BasePriority - 1));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1c, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLowPriority, request1c.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(ExpectAdditionalInfo(nn::util::InvalidUuid, "", request1c.GetHandle()));
            EXPECT_EQ(nn::nifm::RequestState_Accepted, request0b.GetRequestState());
            NNT_ASSERT_RESULT_SUCCESS(request0b.GetResult());
            EXPECT_EQ(nn::nifm::RequestState_Accepted, request2a.GetRequestState());
            NNT_ASSERT_RESULT_SUCCESS(request2a.GetResult());
        }

        // 優先度の高い要求が取り下げられたことで、以前却下された異種要求が乗っ取り成功
        // requirement1(Internet)
        pRequestClient = nn::nifm::GetRequestClientPointer(request1d.GetHandle());
        ASSERT_NE(nullptr, pRequestClient);
        NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRequirementPreset(nn::nifm::RequirementPreset_InternetGeneric));
        NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRawPriority(BasePriority - 1));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1d, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1d.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Free, request0b.GetRequestState());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLowPriority, request0b.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(ExpectAdditionalInfo(nn::util::InvalidUuid, "", request0b.GetHandle()));
    }
}

TEST_F(ConnectCaseTest, ConnectionFailed)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    // imatake-wpa2aes <- wpa-aes で接続
    {
        const nn::nifm::WirelessSettingData wirelessSetting1 = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_WpaPsk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting1 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile1 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting1
            },
            ipSetting1
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection.GetResult());
    }

    // imatake-wpa2aes <- open-wep で接続
    {
        const nn::nifm::WirelessSettingData wirelessSetting1 = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Open,  // authentication
                    nn::nifm::Encryption_Wep  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting1 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile1 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting1
            },
            ipSetting1
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection.GetResult());
    }

    // imatake-mixed <- open-wep で接続
    {
        const nn::nifm::WirelessSettingData wirelessSetting1 = {
            {  // ssidConfig
                {  // ssid
                    13,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6d,0x69,0x78,0x65,0x64 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Open,  // authentication
                    nn::nifm::Encryption_Wep  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting1 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile1 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting1
            },
            ipSetting1
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection.GetResult());
    }

    // imatake-mixed <- shared-wep で接続
    {
        const nn::nifm::WirelessSettingData wirelessSetting1 = {
            {  // ssidConfig
                {  // ssid
                    13,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6d,0x69,0x78,0x65,0x64 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Shared,  // authentication
                    nn::nifm::Encryption_Wep  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting1 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile1 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting1
            },
            ipSetting1
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection.GetResult());
    }

    // imatake-wpaaes <- wpa2-aes で接続
    {
        const nn::nifm::WirelessSettingData wirelessSetting1 = {
            {  // ssidConfig
                {  // ssid
                    14,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x61,0x65,0x73 } // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting1 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile1 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting1
            },
            ipSetting1
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection.GetResult());
    }

}  // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, TemporaryConnectEthernet)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    {
        const nn::nifm::WirelessSettingData wirelessSetting0 = nn::nifm::InvalidWirelessSettingData;
        const nn::nifm::IpSettingData ipSetting0 = { {true, {}, {}, {}}, {true, {}, {}}, {false, 0, "", {false, "", ""}}, 1400 };
        nn::nifm::NetworkProfileData networkProfile0 = { nn::util::InvalidUuid, {}, nn::nifm::NetworkProfileType_User, nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet, true, true,{ wirelessSetting0 }, ipSetting0 };
        // TODO: u"settings2"
        nn::util::ConvertStringUtf8ToUtf16Native( reinterpret_cast<uint16_t*>(networkProfile0.name), nn::nifm::NetworkProfileBasicInfo::NameSize, "setting2", sizeof("setting2") );

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile( networkProfile0 );
        ASSERT_NE( nn::util::InvalidUuid, temporaryNetworkProfile.GetId() );

        // テンポラリ接続要求0
        nn::nifm::NetworkConnection networkConnection0;

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection0.GetRequestHandle(), BasePriority));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection0.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection0.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        networkConnection0.SubmitRequest();
        EXPECT_TRUE(networkConnection0.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_ASSERT_RESULT_SUCCESS(networkConnection0.GetResult());

        // テンポラリ接続要求1
        nn::nifm::NetworkConnection networkConnection1;

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection1.GetRequestHandle(), BasePriority - 1));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection1.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        networkConnection1.SubmitRequest();
        EXPECT_TRUE(networkConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

        // 同一のテンポラリ接続設定を用いた、優先度だけが異なる要求は相乗りする
        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection0.GetRequestState());
        NNT_ASSERT_RESULT_SUCCESS(networkConnection0.GetResult());

        nn::nifm::NetworkProfileData networkProfileData;

        // 現在の接続情報の取得
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::GetCurrentNetworkProfile( &networkProfileData ) );

        EXPECT_TRUE( wirelessSetting0.ssidConfig.ssid.length == networkProfileData.wirelessSetting.ssidConfig.ssid.length );
        EXPECT_TRUE( memcmp( wirelessSetting0.ssidConfig.ssid.hex, networkProfileData.wirelessSetting.ssidConfig.ssid.hex, wirelessSetting0.ssidConfig.ssid.length ) == 0 );
        EXPECT_TRUE( wirelessSetting0.ssidConfig.nonBroadcast == networkProfileData.wirelessSetting.ssidConfig.nonBroadcast );
        EXPECT_TRUE( wirelessSetting0.security.authEncryption.authentication == networkProfileData.wirelessSetting.security.authEncryption.authentication );
        EXPECT_TRUE( wirelessSetting0.security.authEncryption.encryption == networkProfileData.wirelessSetting.security.authEncryption.encryption );
        EXPECT_TRUE( memcmp( wirelessSetting0.security.sharedKey.keyMaterial, networkProfileData.wirelessSetting.security.sharedKey.keyMaterial, networkProfileData.wirelessSetting.security.sharedKey.length ) == 0 );

        EXPECT_TRUE( ipSetting0.ip.isAuto == networkProfileData.ipSetting.ip.isAuto );
        EXPECT_TRUE( memcmp( ipSetting0.ip.ipAddress.data, networkProfileData.ipSetting.ip.ipAddress.data, nn::nifm::IpV4Address::Size ) == 0 );
        EXPECT_TRUE( memcmp( ipSetting0.ip.subnetMask.data, networkProfileData.ipSetting.ip.subnetMask.data, nn::nifm::IpV4Address::Size ) == 0 );
        EXPECT_TRUE( memcmp( ipSetting0.ip.defaultGateway.data, networkProfileData.ipSetting.ip.defaultGateway.data, nn::nifm::IpV4Address::Size ) == 0 );
        EXPECT_TRUE( ipSetting0.dns.isAuto == networkProfileData.ipSetting.dns.isAuto );
        EXPECT_TRUE( memcmp( ipSetting0.dns.preferredDns.data, networkProfileData.ipSetting.dns.preferredDns.data, nn::nifm::IpV4Address::Size ) == 0 );
        EXPECT_TRUE( memcmp( ipSetting0.dns.alternateDns.data, networkProfileData.ipSetting.dns.alternateDns.data, nn::nifm::IpV4Address::Size ) == 0 );
        EXPECT_TRUE( ipSetting0.proxy.isEnabled == networkProfileData.ipSetting.proxy.isEnabled );
        EXPECT_TRUE( ipSetting0.proxy.port == networkProfileData.ipSetting.proxy.port );
        EXPECT_TRUE( strncmp( ipSetting0.proxy.proxy, networkProfileData.ipSetting.proxy.proxy, nn::nifm::ProxySetting::ProxyNameSize ) == 0 );
        EXPECT_TRUE( ipSetting0.proxy.authentication.isEnabled == networkProfileData.ipSetting.proxy.authentication.isEnabled );
        EXPECT_TRUE( strncmp( ipSetting0.proxy.authentication.username, networkProfileData.ipSetting.proxy.authentication.username, nn::nifm::AuthenticationSetting::UsernameSize ) == 0 );
        EXPECT_TRUE( strncmp( ipSetting0.proxy.authentication.password, networkProfileData.ipSetting.proxy.authentication.password, nn::nifm::AuthenticationSetting::PasswordSize ) == 0 );
        EXPECT_TRUE( ipSetting0.mtu == networkProfileData.ipSetting.mtu );

        // 有線接続では，現在接続中のアクセスポイント情報を取得できない
        nn::nifm::AccessPointData accessPointData;
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultEthernetConnection, nn::nifm::GetCurrentAccessPoint(&accessPointData));
        uint16_t channel;
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultEthernetConnection, nn::nifm::GetCurrentAccessPointChannel(&channel));

        // Ethernet テンポラリ接続だと基本的に相乗りが発生しない
        // ...
    }
}

TEST_F(ConnectCaseTest, TemporaryConnectWifi)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    {
        ASSERT_TRUE(nn::nifm::test::Disconnect());

        {
            // 現在の接続情報の取得（接続前）
            nn::nifm::NetworkProfileData networkProfileData;
            NNT_ASSERT_RESULT_FAILURE( nn::nifm::ResultNotConnected, nn::nifm::GetCurrentNetworkProfile( &networkProfileData ) );

            nn::nifm::ProxySetting proxySetting;
            NNT_ASSERT_RESULT_FAILURE( nn::nifm::ResultNotConnected, nn::nifm::GetCurrentProxySetting( &proxySetting ) );
        }

        {
            // imatake-opennone-stealth
            const nn::nifm::WirelessSettingData wirelessSetting0 = {
                {  // ssidConfig
                    {  // ssid
                        24,  // length
                        {0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x6e,0x6f,0x6e,0x65,0x2d,0x73,0x74,0x65,0x61,0x6c,0x74,0x68}  // hex
                    },
                    true  // nonBroadcast
                },
                {  // security
                    {  //authEncryption
                        nn::nifm::Authentication_Open,  // authentication
                        nn::nifm::Encryption_None  // encryption
                    },
                    {  // sharedKey
                        7,  // length
                        "NotUsed"  // keyMaterial
                    }
                }
            };
            const nn::nifm::IpSettingData ipSetting0 = {
                {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    true,  // isEnabled
                    8080,  // port
                    "dummy.proxy.jp",  // proxy
                    {  // authentication
                        true,  // isEnabled
                        "dummy_user",  // username
                        "dummy_password"  // password
                    }
                },
                1400  //mtu
            };
            const nn::nifm::NetworkProfileData networkProfile0 = {
                nn::util::InvalidUuid,  // id
                {},  // name
                nn::nifm::NetworkProfileType_Temporary, // networkProfileType
                nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
                true, // isAutoConnect
                true, // isLargeCapacity
                {
                    wirelessSetting0
                },
                ipSetting0
            };

            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile( networkProfile0 );
            ASSERT_NE( nn::util::InvalidUuid, temporaryNetworkProfile.GetId() );

            // テンポラリ接続要求0
            nn::nifm::NetworkConnection networkConnection0;

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection0.GetRequestHandle(), BasePriority));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection0.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection0.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            networkConnection0.SubmitRequest();
            EXPECT_TRUE(networkConnection0.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
            NNT_ASSERT_RESULT_SUCCESS(networkConnection0.GetResult());

            // テンポラリ接続要求1
            nn::nifm::NetworkConnection networkConnection1;

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection1.GetRequestHandle(), BasePriority - 1));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection1.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            networkConnection1.SubmitRequest();
            EXPECT_TRUE(networkConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
            NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

            // 同一のテンポラリ接続設定を用いた、優先度だけが異なる要求は相乗りする
            EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection0.GetRequestState());
            NNT_ASSERT_RESULT_SUCCESS(networkConnection0.GetResult());

            // 現在の接続情報の取得
            nn::nifm::NetworkProfileData networkProfileData;
            NNT_ASSERT_RESULT_SUCCESS( nn::nifm::GetCurrentNetworkProfile( &networkProfileData ) );

            EXPECT_TRUE( wirelessSetting0.ssidConfig.ssid.length == networkProfileData.wirelessSetting.ssidConfig.ssid.length );
            EXPECT_TRUE( memcmp( wirelessSetting0.ssidConfig.ssid.hex, networkProfileData.wirelessSetting.ssidConfig.ssid.hex, wirelessSetting0.ssidConfig.ssid.length ) == 0 );
            EXPECT_TRUE( wirelessSetting0.ssidConfig.nonBroadcast == networkProfileData.wirelessSetting.ssidConfig.nonBroadcast );
            EXPECT_TRUE( wirelessSetting0.security.authEncryption.authentication == networkProfileData.wirelessSetting.security.authEncryption.authentication );
            EXPECT_TRUE( wirelessSetting0.security.authEncryption.encryption == networkProfileData.wirelessSetting.security.authEncryption.encryption );
            EXPECT_TRUE( memcmp( wirelessSetting0.security.sharedKey.keyMaterial, networkProfileData.wirelessSetting.security.sharedKey.keyMaterial, networkProfileData.wirelessSetting.security.sharedKey.length ) == 0 );

            EXPECT_TRUE( ipSetting0.ip.isAuto == networkProfileData.ipSetting.ip.isAuto );
            EXPECT_TRUE( memcmp( ipSetting0.ip.ipAddress.data, networkProfileData.ipSetting.ip.ipAddress.data, nn::nifm::IpV4Address::Size ) == 0 );
            EXPECT_TRUE( memcmp( ipSetting0.ip.subnetMask.data, networkProfileData.ipSetting.ip.subnetMask.data, nn::nifm::IpV4Address::Size ) == 0 );
            EXPECT_TRUE( memcmp( ipSetting0.ip.defaultGateway.data, networkProfileData.ipSetting.ip.defaultGateway.data, nn::nifm::IpV4Address::Size ) == 0 );
            EXPECT_TRUE( ipSetting0.dns.isAuto == networkProfileData.ipSetting.dns.isAuto );
            EXPECT_TRUE( memcmp( ipSetting0.dns.preferredDns.data, networkProfileData.ipSetting.dns.preferredDns.data, nn::nifm::IpV4Address::Size ) == 0 );
            EXPECT_TRUE( memcmp( ipSetting0.dns.alternateDns.data, networkProfileData.ipSetting.dns.alternateDns.data, nn::nifm::IpV4Address::Size ) == 0 );
            EXPECT_TRUE( ipSetting0.proxy.isEnabled == networkProfileData.ipSetting.proxy.isEnabled );
            EXPECT_TRUE( ipSetting0.proxy.port == networkProfileData.ipSetting.proxy.port );
            EXPECT_TRUE( strncmp( ipSetting0.proxy.proxy, networkProfileData.ipSetting.proxy.proxy, nn::nifm::ProxySetting::ProxyNameSize ) == 0 );
            EXPECT_TRUE( ipSetting0.proxy.authentication.isEnabled == networkProfileData.ipSetting.proxy.authentication.isEnabled );
            EXPECT_TRUE( strncmp( ipSetting0.proxy.authentication.username, networkProfileData.ipSetting.proxy.authentication.username, nn::nifm::AuthenticationSetting::UsernameSize ) == 0 );
            EXPECT_TRUE( strncmp( ipSetting0.proxy.authentication.password, networkProfileData.ipSetting.proxy.authentication.password, nn::nifm::AuthenticationSetting::PasswordSize ) == 0 ); // empty
            EXPECT_TRUE( ipSetting0.mtu == networkProfileData.ipSetting.mtu );

            // 現在のプロキシ設定情報の取得
            nn::nifm::ProxySetting proxySetting;
            NNT_ASSERT_RESULT_SUCCESS( nn::nifm::GetCurrentProxySetting( &proxySetting ) );

            EXPECT_TRUE( proxySetting.isEnabled == networkProfileData.ipSetting.proxy.isEnabled );
            EXPECT_TRUE( proxySetting.port == networkProfileData.ipSetting.proxy.port );
            EXPECT_TRUE( strncmp( proxySetting.proxy, networkProfileData.ipSetting.proxy.proxy, nn::nifm::ProxySetting::ProxyNameSize ) == 0 );
            EXPECT_TRUE( proxySetting.authentication.isEnabled == networkProfileData.ipSetting.proxy.authentication.isEnabled );
            EXPECT_TRUE( strncmp( proxySetting.authentication.username, networkProfileData.ipSetting.proxy.authentication.username, nn::nifm::AuthenticationSetting::UsernameSize ) == 0 );
            EXPECT_TRUE( strncmp( proxySetting.authentication.password, networkProfileData.ipSetting.proxy.authentication.password, nn::nifm::AuthenticationSetting::PasswordSize ) == 0 ); // empty
        }

        {
            // imatake-wpa2aes
            const nn::nifm::WirelessSettingData wirelessSetting1 = {
                {  // ssidConfig
                    {  // ssid
                        15,  // length
                        {0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73}  // hex
                    },
                    false  // nonBroadcast
                },
                {  // security
                    {  //authEncryption
                        nn::nifm::Authentication_Wpa2Psk,  // authentication
                        nn::nifm::Encryption_Aes  // encryption
                    },
                    {  // sharedKey
#if 1
                        11,  // length
                        "Shi2iTaiZen"  // keyMaterial
#else // 16 進数 64 桁 PMK をサポートしている場合
                        64,  // length
                        { 0x62,0x61,0x64,0x38,0x38,0x34,0x30,0x62,0x61,0x31,
                          0x38,0x37,0x31,0x61,0x65,0x37,0x66,0x37,0x36,0x66,
                          0x32,0x65,0x66,0x39,0x33,0x33,0x35,0x31,0x61,0x62,
                          0x36,0x66,0x35,0x30,0x65,0x64,0x65,0x33,0x35,0x63,
                          0x66,0x62,0x63,0x39,0x39,0x35,0x64,0x39,0x63,0x33,
                          0x37,0x37,0x32,0x64,0x61,0x32,0x31,0x63,0x61,0x63,
                          0x65,0x65,0x39,0x35 }  // keyMaterial
#endif
                    }
                }
            };
            const nn::nifm::IpSettingData ipSetting1 = {
                {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    false,  // isEnabled
                    0,  // port
                    "",  // proxy
                    {  // authentication
                        false,  // isEnabled
                        "",  // username
                        ""  // password
                    }
                },
                1400  //mtu
            };
            const nn::nifm::NetworkProfileData networkProfile1 = {
                nn::util::InvalidUuid,  // id
                {},  // name
                nn::nifm::NetworkProfileType_Temporary, // networkProfileType
                nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
                true,  // isAutoConnect
                true, // isLargeCapacity
                {
                    wirelessSetting1
                },
                ipSetting1
            };

            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile( networkProfile1 );
            ASSERT_NE( nn::util::InvalidUuid, temporaryNetworkProfile.GetId() );
            nn::util::Uuid id1 = temporaryNetworkProfile.GetId();

            // テンポラリ接続要求2
            nn::nifm::NetworkConnection networkConnection2;

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), BasePriority));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection2.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            networkConnection2.SubmitRequest();
            EXPECT_TRUE(networkConnection2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
            NNT_ASSERT_RESULT_SUCCESS(networkConnection2.GetResult());

            // テンポラリ接続要求3
            nn::nifm::NetworkConnection networkConnection3;

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection3.GetRequestHandle(), BasePriority - 2));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection3.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection3.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            networkConnection3.SubmitRequest();
            EXPECT_TRUE(networkConnection3.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
            NNT_ASSERT_RESULT_SUCCESS(networkConnection3.GetResult());

            // 現在の接続情報の取得
            nn::nifm::NetworkProfileData networkProfileData;
            NNT_ASSERT_RESULT_SUCCESS( nn::nifm::GetCurrentNetworkProfile( &networkProfileData ) );

            EXPECT_TRUE( wirelessSetting1.ssidConfig.ssid.length == networkProfileData.wirelessSetting.ssidConfig.ssid.length );
            EXPECT_TRUE( memcmp( wirelessSetting1.ssidConfig.ssid.hex, networkProfileData.wirelessSetting.ssidConfig.ssid.hex, wirelessSetting1.ssidConfig.ssid.length ) == 0 );
            EXPECT_TRUE( wirelessSetting1.ssidConfig.nonBroadcast == networkProfileData.wirelessSetting.ssidConfig.nonBroadcast );
            EXPECT_TRUE( wirelessSetting1.security.authEncryption.authentication == networkProfileData.wirelessSetting.security.authEncryption.authentication );
            EXPECT_TRUE( wirelessSetting1.security.authEncryption.encryption == networkProfileData.wirelessSetting.security.authEncryption.encryption );
            EXPECT_TRUE( memcmp( wirelessSetting1.security.sharedKey.keyMaterial, networkProfileData.wirelessSetting.security.sharedKey.keyMaterial, networkProfileData.wirelessSetting.security.sharedKey.length ) == 0 );

            EXPECT_TRUE( ipSetting1.ip.isAuto == networkProfileData.ipSetting.ip.isAuto );
            EXPECT_TRUE( memcmp( ipSetting1.ip.ipAddress.data, networkProfileData.ipSetting.ip.ipAddress.data, nn::nifm::IpV4Address::Size ) == 0 );
            EXPECT_TRUE( memcmp( ipSetting1.ip.subnetMask.data, networkProfileData.ipSetting.ip.subnetMask.data, nn::nifm::IpV4Address::Size ) == 0 );
            EXPECT_TRUE( memcmp( ipSetting1.ip.defaultGateway.data, networkProfileData.ipSetting.ip.defaultGateway.data, nn::nifm::IpV4Address::Size ) == 0 );
            EXPECT_TRUE( ipSetting1.dns.isAuto == networkProfileData.ipSetting.dns.isAuto );
            EXPECT_TRUE( memcmp( ipSetting1.dns.preferredDns.data, networkProfileData.ipSetting.dns.preferredDns.data, nn::nifm::IpV4Address::Size ) == 0 );
            EXPECT_TRUE( memcmp( ipSetting1.dns.alternateDns.data, networkProfileData.ipSetting.dns.alternateDns.data, nn::nifm::IpV4Address::Size ) == 0 );
            EXPECT_TRUE( ipSetting1.proxy.isEnabled == networkProfileData.ipSetting.proxy.isEnabled );
            EXPECT_TRUE( ipSetting1.proxy.port == networkProfileData.ipSetting.proxy.port );
            EXPECT_TRUE( strncmp( ipSetting1.proxy.proxy, networkProfileData.ipSetting.proxy.proxy, nn::nifm::ProxySetting::ProxyNameSize ) == 0 );
            EXPECT_TRUE( ipSetting1.proxy.authentication.isEnabled == networkProfileData.ipSetting.proxy.authentication.isEnabled );
            EXPECT_TRUE( strncmp( ipSetting1.proxy.authentication.username, networkProfileData.ipSetting.proxy.authentication.username, nn::nifm::AuthenticationSetting::UsernameSize ) == 0 );
            EXPECT_TRUE( strncmp( ipSetting1.proxy.authentication.password, networkProfileData.ipSetting.proxy.authentication.password, nn::nifm::AuthenticationSetting::PasswordSize ) == 0 ); // empty
            EXPECT_TRUE( ipSetting1.mtu == networkProfileData.ipSetting.mtu );

            // imatake-mixed(wpapsk)
            const nn::nifm::WirelessSettingData wirelessSetting2 = {
                {  // ssidConfig
                    {  // ssid
                        13,  // length
                        {0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6d,0x69,0x78,0x65,0x64}  // hex
                    },
                    false  // nonBroadcast
                },
                {  // security
                    {  //authEncryption
                        nn::nifm::Authentication_WpaPsk,  // authentication
                        nn::nifm::Encryption_Aes  // encryption
                    },
                    {  // sharedKey
                        11,  // length
                        "Shi2iTaiZen"  // keyMaterial
                    }
                }
            };
            const nn::nifm::IpSettingData ipSetting2 = {
                {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    false,  // isEnabled
                    0,  // port
                    "",  // proxy
                    {  // authentication
                        false,  // isEnabled
                        "",  // username
                        ""  // password
                    }
                },
                1400  //mtu
            };
            const nn::nifm::NetworkProfileData networkProfile2 = {
                nn::util::InvalidUuid,  // id
                {},  // name
                nn::nifm::NetworkProfileType_Temporary, // networkProfileType
                nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
                true, // isAutoConnect
                true, // isLargeCapacity
                {
                    wirelessSetting2
                },
                ipSetting2
            };

            //  テンポラリ接続オブジェクトを使いまわすと以前の接続設定は消える
            NNT_ASSERT_RESULT_SUCCESS(temporaryNetworkProfile.Update(networkProfile2));
            ASSERT_NE(id1, temporaryNetworkProfile.GetId());
            ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

            // テンポラリ接続要求4
            nn::nifm::NetworkConnection networkConnection4;

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection4.GetRequestHandle(), BasePriority - 3));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection4.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection4.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            networkConnection4.SubmitRequest();
            EXPECT_TRUE(networkConnection4.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
            NNT_ASSERT_RESULT_SUCCESS(networkConnection4.GetResult());

            // 異なるテンポラリ接続を用いた利用要求が accept されたので、以前の要求は reject
            EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection3.GetRequestState());
        }

        {
            // "imatake-openwep104"
            const nn::nifm::WirelessSettingData wirelessSetting3 = {
                {  // ssidConfig
                    {  // ssid
                        18,  // length
                        {0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x77,0x65,0x70,0x31,0x30,0x34}  // hex
                    },
                    false  // nonBroadcast
                },
                {  // security
                    {  //authEncryption
                        nn::nifm::Authentication_Open,  // authentication
                        nn::nifm::Encryption_Wep  // encryption
                    },
                    {  // sharedKey
                        13,  // length
                        "Shi2iTaiZenTo"  // keyMaterial
                    }
                }
            };
            const nn::nifm::IpSettingData ipSetting3 = {
                {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    false,  // isEnabled
                    0,  // port
                    "",  // proxy
                    {  // authentication
                        false,  // isEnabled
                        "",  // username
                        ""  // password
                    }
                },
                1400  //mtu
            };
            const nn::nifm::NetworkProfileData networkProfile3 = {
                nn::util::InvalidUuid,  // id
                {},  // name
                nn::nifm::NetworkProfileType_Temporary, // networkProfileType
                nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
                true, // isAutoConnect
                true, // isLargeCapacity
                {
                    wirelessSetting3
                },
                ipSetting3
            };

            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile( networkProfile3 );
            ASSERT_NE( nn::util::InvalidUuid, temporaryNetworkProfile.GetId() );

            // テンポラリ接続要求5
            nn::nifm::NetworkConnection networkConnection5;

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection5.GetRequestHandle(), BasePriority - 4));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection5.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection5.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            networkConnection5.SubmitRequest();
            EXPECT_TRUE(networkConnection5.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
            NNT_ASSERT_RESULT_SUCCESS(networkConnection5.GetResult());

            nn::nifm::AccessPointData accessPointData;
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetCurrentAccessPoint(&accessPointData));
            ASSERT_EQ(nn::nifm::Encryption_Wep, accessPointData.encryption);
            ASSERT_EQ(nn::nifm::Authentication_Open, accessPointData.authentication);

            uint16_t channel;
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetCurrentAccessPointChannel(&channel));
            ASSERT_EQ(accessPointData.channel, channel);
        }

        // sharedWep が存在しない CI 環境では動作しない
#if 0
        {
            // "imatake-sharedwep104"
            const nn::nifm::WirelessSettingData wirelessSetting3 = {
                {  // ssidConfig
                    {  // ssid
                        20,  // length
                        { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x73,0x68,0x61,0x72,0x65,0x64,0x77,0x65,0x70,0x31,0x30,0x34 }  // hex
                    },
                false  // nonBroadcast
                },
                {  // security
                    {  //authEncryption
                        nn::nifm::Authentication_Shared,  // authentication
                        nn::nifm::Encryption_Wep  // encryption
                    },
                    {  // sharedKey
                        13,  // length
                        "Shi2iTaiZenTo"  // keyMaterial
                    }
                }
            };
            const nn::nifm::IpSettingData ipSetting3 = {
                {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    false,  // isEnabled
                    0,  // port
                    "",  // proxy
                    {  // authentication
                        false,  // isEnabled
                        "",  // username
                        ""  // password
                    }
                },
                1400  //mtu
            };
            const nn::nifm::NetworkProfileData networkProfile3 = {
                nn::util::InvalidUuid,  // id
                {},  // name
                nn::nifm::NetworkProfileType_Temporary, // networkProfileType
                nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
                true, // isAutoConnect
                true, // isLargeCapacity
                {
                    wirelessSetting3
                },
                ipSetting3
            };

            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile3);
            ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

            // テンポラリ接続要求5
            nn::nifm::NetworkConnection networkConnection5;

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection5.GetRequestHandle(), BasePriority - 4));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection5.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection5.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            networkConnection5.SubmitRequest();
            EXPECT_TRUE(networkConnection5.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
            NNT_ASSERT_RESULT_SUCCESS(networkConnection5.GetResult());

            nn::nifm::AccessPointData accessPointData;
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetCurrentAccessPoint(&accessPointData));
            ASSERT_EQ(nn::nifm::Encryption_Wep, accessPointData.encryption);
            ASSERT_EQ(nn::nifm::Authentication_Shared, accessPointData.authentication);
        }
#endif

        // 残ったりリクエストが暗黙的に取り下げ
    }

    // TODO: APから切断されたことを検証
    // ...

} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, ResultRejected)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    {
        ASSERT_TRUE(nn::nifm::test::Disconnect());

        {
            // 現在の接続情報の取得（接続前）
            nn::nifm::NetworkProfileData networkProfileData;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetCurrentNetworkProfile(&networkProfileData));

            nn::nifm::ProxySetting proxySetting;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetCurrentProxySetting(&proxySetting));
        }

        {
            // imatake-none
            const nn::nifm::WirelessSettingData wirelessSetting0 = {
                {  // ssidConfig
                    {  // ssid
                        12,  // length
                        { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6e,0x6f,0x6e,0x65 }  // hex
                    },
                false  // nonBroadcast
                },
                {  // security
                    {  //authEncryption
                        nn::nifm::Authentication_Open,  // authentication
                        nn::nifm::Encryption_None  // encryption
                    },
                    {  // sharedKey
                        7,  // length
                        "NotUsed"  // keyMaterial
                    }
                }
            };
            const nn::nifm::IpSettingData ipSetting0 = {
                {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    true,  // isEnabled
                    8080,  // port
                    "dummy.proxy.jp",  // proxy
                    {  // authentication
                        true,  // isEnabled
                        "dummy_user",  // username
                        "dummy_password"  // password
                    }
                },
                1400  //mtu
            };
            const nn::nifm::NetworkProfileData networkProfile0 = {
                nn::util::InvalidUuid,  // id
                {},  // name
                nn::nifm::NetworkProfileType_Temporary, // networkProfileType
                nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
                true, // isAutoConnect
                true, // isLargeCapacity
                {
                    wirelessSetting0
                },
                ipSetting0
            };

            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile0);
            ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

            // テンポラリ接続要求0
            nn::nifm::NetworkConnection networkConnection0;

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection0.GetRequestHandle(), BasePriority));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection0.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection0.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            networkConnection0.SubmitRequest();
            EXPECT_TRUE(networkConnection0.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection0.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(ExpectAdditionalInfo(nn::util::InvalidUuid, "", networkConnection0.GetRequestHandle()));

            nn::nifm::detail::RequestClient* pRequestClient = nn::nifm::GetRequestClientPointer(networkConnection0.GetRequestHandle());
            ASSERT_NE(nullptr, pRequestClient);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
            {
                nn::applet::AppletId appletId = nn::applet::AppletId_None;
                nn::applet::LibraryAppletMode libraryAppletMode = nn::applet::LibraryAppletMode_AllForeground;
                size_t size = 0;
                char buffer[4 * 1024];
                NNT_EXPECT_RESULT_SUCCESS(pRequestClient->GetLibaryAppletArgument(&appletId, &libraryAppletMode, &size, buffer, sizeof(buffer)));
                EXPECT_EQ(nn::applet::AppletId_LibraryAppletNetConnect, appletId);
                EXPECT_EQ(nn::applet::LibraryAppletMode_AllForeground, libraryAppletMode);
            }
#endif
        }

        {
            // ido-opennone-stealth (存在しない)
            const nn::nifm::WirelessSettingData wirelessSetting1 = {
                {  // ssidConfig
                    {  // ssid
                        20,  // length
                        { 0x69,0x64,0x6F,0x2D,0x6F,0x70,0x65,0x6E,0x6E,0x6F,0x6E,0x65,0x2D,0x73,0x74,0x65,0x61,0x6C,0x74,0x68 }  // hex
                    },
                true   // nonBroadcast
                },
                {  // security
                    {  //authEncryption
                        nn::nifm::Authentication_Open,  // authentication
                        nn::nifm::Encryption_None  // encryption
                    },
                    {  // sharedKey
                        7,  // length
                        "NotUsed"  // keyMaterial
                    }
                }
            };
            const nn::nifm::IpSettingData ipSetting1 = {
                {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    true,  // isEnabled
                    8080,  // port
                    "dummy.proxy.jp",  // proxy
                    {  // authentication
                        true,  // isEnabled
                        "dummy_user",  // username
                        "dummy_password"  // password
                    }
                },
                1400  //mtu
            };
            const nn::nifm::NetworkProfileData networkProfile1 = {
                nn::util::InvalidUuid,  // id
                {},  // name
                nn::nifm::NetworkProfileType_Temporary, // networkProfileType
                nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
                true, // isAutoConnect
                true, // isLargeCapacity
                {
                    wirelessSetting1
                },
                ipSetting1
            };

            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);
            ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

            // テンポラリ接続要求1
            nn::nifm::NetworkConnection networkConnection1;

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection1.GetRequestHandle(), BasePriority));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection1.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            networkConnection1.SubmitRequest();
            EXPECT_TRUE(networkConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultBssNotFoundStealthIncluded, networkConnection1.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(ExpectAdditionalInfo(temporaryNetworkProfile.GetId(), "", networkConnection1.GetRequestHandle()));

            nn::nifm::detail::RequestClient* pRequestClient = nn::nifm::GetRequestClientPointer(networkConnection1.GetRequestHandle());
            ASSERT_NE(nullptr, pRequestClient);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
            {
                nn::applet::AppletId appletId = nn::applet::AppletId_None;
                nn::applet::LibraryAppletMode libraryAppletMode = nn::applet::LibraryAppletMode_AllForeground;
                size_t size = 0;
                char buffer[4 * 1024];
                NNT_EXPECT_RESULT_SUCCESS(pRequestClient->GetLibaryAppletArgument(&appletId, &libraryAppletMode, &size, buffer, sizeof(buffer)));
                EXPECT_EQ(nn::applet::AppletId_LibraryAppletNetConnect, appletId);
                EXPECT_EQ(nn::applet::LibraryAppletMode_AllForeground, libraryAppletMode);
            }
#endif
        }

        {
            // imatake-wpa2aes
            const nn::nifm::WirelessSettingData wirelessSetting1 = {
                {  // ssidConfig
                    {  // ssid
                        15,  // length
                        { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                    },
                false  // nonBroadcast
                },
                {  // security
                    {  //authEncryption
                        nn::nifm::Authentication_Wpa2Psk,  // authentication
                        nn::nifm::Encryption_Aes  // encryption
                    },
                    {  // sharedKey
                        15,  // length
                        "WrongPassphrase"  // keyMaterial
                    }
                }
            };
            const nn::nifm::IpSettingData ipSetting1 = {
                {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    false,  // isEnabled
                    0,  // port
                    "",  // proxy
                    {  // authentication
                        false,  // isEnabled
                        "",  // username
                        ""  // password
                    }
                },
                1400  //mtu
            };
            const nn::nifm::NetworkProfileData networkProfile1 = {
                nn::util::InvalidUuid,  // id
                {},  // name
                nn::nifm::NetworkProfileType_Temporary, // networkProfileType
                nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
                true,  // isAutoConnect
                true, // isLargeCapacity
                {
                    wirelessSetting1
                },
                ipSetting1
            };

            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);
            ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

            // テンポラリ接続要求2
            nn::nifm::NetworkConnection networkConnection2;

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), BasePriority));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection2.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            networkConnection2.SubmitRequest();
            EXPECT_TRUE(networkConnection2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS * 3)));
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultConnectionFailedWithReasonCode, networkConnection2.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(ExpectAdditionalInfo(temporaryNetworkProfile.GetId(), "", networkConnection2.GetRequestHandle()));

            nn::nifm::detail::RequestClient* pRequestClient = nn::nifm::GetRequestClientPointer(networkConnection2.GetRequestHandle());
            ASSERT_NE(nullptr, pRequestClient);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
            {
                nn::applet::AppletId appletId = nn::applet::AppletId_None;
                nn::applet::LibraryAppletMode libraryAppletMode = nn::applet::LibraryAppletMode_AllForeground;
                size_t size = 0;
                char buffer[4 * 1024];
                NNT_EXPECT_RESULT_SUCCESS(pRequestClient->GetLibaryAppletArgument(&appletId, &libraryAppletMode, &size, buffer, sizeof(buffer)));
                EXPECT_EQ(nn::applet::AppletId_LibraryAppletNetConnect, appletId);
                EXPECT_EQ(nn::applet::LibraryAppletMode_AllForeground, libraryAppletMode);
            }
#endif
        }

        {
            // "imatake-openwep104"
            const nn::nifm::WirelessSettingData wirelessSetting3 = {
                {  // ssidConfig
                    {  // ssid
                        18,  // length
                        { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x77,0x65,0x70,0x31,0x30,0x34 }  // hex
                    },
                false  // nonBroadcast
                },
                {  // security
                    {  //authEncryption
                        nn::nifm::Authentication_Open,  // authentication
                        nn::nifm::Encryption_Wep  // encryption
                    },
                    {  // sharedKey
                        5,  // length
                        "Wrong"  // keyMaterial
                    }
                }
            };
            const nn::nifm::IpSettingData ipSetting3 = {
                {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    false,  // isEnabled
                    0,  // port
                    "",  // proxy
                    {  // authentication
                        false,  // isEnabled
                        "",  // username
                        ""  // password
                    }
                },
                1400  //mtu
            };
            const nn::nifm::NetworkProfileData networkProfile3 = {
                nn::util::InvalidUuid,  // id
                {},  // name
                nn::nifm::NetworkProfileType_Temporary, // networkProfileType
                nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
                true, // isAutoConnect
                true, // isLargeCapacity
                {
                    wirelessSetting3
                },
                ipSetting3
            };

            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile3);
            ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

            // テンポラリ接続要求5
            nn::nifm::NetworkConnection networkConnection5;

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection5.GetRequestHandle(), BasePriority));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection5.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection5.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            networkConnection5.SubmitRequest();
            EXPECT_TRUE(networkConnection5.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS * 2)));
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultDhcpFailedOrPassphraseWrong, networkConnection5.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(ExpectAdditionalInfo(temporaryNetworkProfile.GetId(), "", networkConnection5.GetRequestHandle()));

            nn::nifm::detail::RequestClient* pRequestClient = nn::nifm::GetRequestClientPointer(networkConnection5.GetRequestHandle());
            ASSERT_NE(nullptr, pRequestClient);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
            {
                nn::applet::AppletId appletId = nn::applet::AppletId_None;
                nn::applet::LibraryAppletMode libraryAppletMode = nn::applet::LibraryAppletMode_AllForeground;
                size_t size = 0;
                char buffer[4 * 1024];
                NNT_EXPECT_RESULT_SUCCESS(pRequestClient->GetLibaryAppletArgument(&appletId, &libraryAppletMode, &size, buffer, sizeof(buffer)));
                EXPECT_EQ(nn::applet::AppletId_LibraryAppletNetConnect, appletId);
                EXPECT_EQ(nn::applet::LibraryAppletMode_AllForeground, libraryAppletMode);
            }
#endif
        }

        // 残ったリクエストが暗黙的に取り下げられる
    }

    // TODO: APから切断されたことを検証
    // ...

} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, NetworkConnection)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    bool isCompleted;

    // 一般的な使い方（ポーリング）
    {
        nn::nifm::NetworkConnection networkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        networkConnection.SubmitRequest();

        isCompleted = false;
        // 状態が変わるまで、最長で30秒待つ
        for( int i = 0; i < 3000; ++i )
        {
            if( !networkConnection.IsRequestOnHold() )
            {
                isCompleted = true;
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        }
        EXPECT_TRUE(isCompleted);

        EXPECT_FALSE(networkConnection.IsRequestOnHold());
        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_FALSE(networkConnection.GetSystemEvent().TryWait());     // AutoClear

        // 用が済んだら取り消し
        networkConnection.CancelRequest();

        EXPECT_FALSE(networkConnection.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
    }

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::NetworkConnection networkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        EXPECT_FALSE(networkConnection.IsRequestOnHold());
        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_FALSE(networkConnection.GetSystemEvent().TryWait());     // AutoClear

        // 用が済んだら取り消し
        networkConnection.CancelRequest();

        EXPECT_FALSE(networkConnection.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
    }

    // 一般的な使い方（イベント待ち）
    {
        nn::nifm::NetworkConnection networkConnection(nn::os::EventClearMode_ManualClear);
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        networkConnection.SubmitRequest();

        networkConnection.GetSystemEvent().Wait();

        EXPECT_FALSE(networkConnection.IsRequestOnHold());
        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        EXPECT_TRUE(networkConnection.GetSystemEvent().TryWait());      // ManualClear
        networkConnection.GetSystemEvent().Clear();

        // 用が済んだら取り消し
        networkConnection.CancelRequest();

        EXPECT_FALSE(networkConnection.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
    }

    // 細かい挙動の確認
    {
        nn::nifm::NetworkConnection networkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        // 初期状態の確認
        EXPECT_FALSE(networkConnection.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNotInitialized, networkConnection.GetResult());

        networkConnection.SubmitRequest();

        // TODO: EXPECT_EQ(nn::nifm::RequestState_OnHold, networkConnection.GetRequestState());
        // TODO: NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultProcessing, networkConnection.Submit());
        // TODO: NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultProcessing, networkConnection.GetResult());

        isCompleted = false;
        // 状態が変わるまで、最長で30秒待つ
        for( int i = 0; i < 3000; ++i )
        {
            if( !networkConnection.IsRequestOnHold() )
            {
                isCompleted = true;
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        }
        EXPECT_TRUE(isCompleted);

        EXPECT_FALSE(networkConnection.IsRequestOnHold());
        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        // もう1回リクエストしても何も起こらない
        networkConnection.SubmitRequest();

        // 用が済んだら取り消し
        networkConnection.CancelRequest();

        EXPECT_FALSE(networkConnection.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnection.GetResult());

        networkConnection.SubmitRequest();

        // TODO: 外部要因の切断から再接続
    }
}

TEST_F(ConnectCaseTest, Flat)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    bool isCompleted;

    // 一般的な使い方（ポーリング）
    {
        nn::nifm::SubmitNetworkRequest();

        isCompleted = false;
        // 状態が変わるまで、最長で30秒待つ
        for (int i = 0; i < 3000; ++i)
        {
            if (!nn::nifm::IsNetworkRequestOnHold())
            {
                isCompleted = true;
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        }
        EXPECT_TRUE(isCompleted);

        EXPECT_TRUE(nn::nifm::IsNetworkAvailable());
        //NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetResult());

        // 用が済んだら取り消し
        nn::nifm::CancelNetworkRequest();

        EXPECT_FALSE(nn::nifm::IsNetworkAvailable());
    }

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::SubmitNetworkRequestAndWait();

        EXPECT_TRUE(nn::nifm::IsNetworkAvailable());
        //NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetResult());

        // 用が済んだら取り消し
        nn::nifm::CancelNetworkRequest();

        EXPECT_FALSE(nn::nifm::IsNetworkAvailable());
    }

    // 細かい挙動の確認
    {
        // 初期状態の確認
        EXPECT_FALSE(nn::nifm::IsNetworkRequestOnHold());
        EXPECT_FALSE(nn::nifm::IsNetworkAvailable());
        //NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNotInitialized, nn::nifm::GetResult());

        nn::nifm::SubmitNetworkRequest();

        // TODO: EXPECT_EQ(nn::nifm::RequestState_OnHold, nn::nifm::GetRequestState());
        // TODO: NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultProcessing, nn::nifm::Submit());
        // TODO: NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultProcessing, nn::nifm::GetResult());

        isCompleted = false;
        // 状態が変わるまで、最長で30秒待つ
        for (int i = 0; i < 3000; ++i)
        {
            if (!nn::nifm::IsNetworkRequestOnHold())
            {
                isCompleted = true;
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        }
        EXPECT_TRUE(isCompleted);

        EXPECT_FALSE(nn::nifm::IsNetworkRequestOnHold());
        EXPECT_TRUE(nn::nifm::IsNetworkAvailable());
        //NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetResult());

        // もう1回リクエストしても何も起こらない
        nn::nifm::SubmitNetworkRequest();

        // 用が済んだら取り消し
        nn::nifm::CancelNetworkRequest();

        EXPECT_FALSE(nn::nifm::IsNetworkRequestOnHold());
        EXPECT_FALSE(nn::nifm::IsNetworkAvailable());
        //NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, nn::nifm::GetResult());

        nn::nifm::SubmitNetworkRequest();

        // TODO: 外部要因の切断から再接続
    }
}

TEST_F(ConnectCaseTest, SetLocalNetworkMode)
{
    // クローズドな LAN への接続確認
    {
        nn::nifm::test::ScopedInitializer<nn::nifm::Initialize, nn::nifm::FinalizeForTest> initializer(false);
        NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

        // imatake-opennone-stealth
        const nn::nifm::WirelessSettingData wirelessSetting0 = {
            {  // ssidConfig
                {  // ssid
                    24,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x6e,0x6f,0x6e,0x65,0x2d,0x73,0x74,0x65,0x61,0x6c,0x74,0x68 }  // hex
                },
            true  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Open,  // authentication
                    nn::nifm::Encryption_None  // encryption
                },
                {  // sharedKey
                    7,  // length
                    "NotUsed"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting0 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                true,  // isEnabled
                8080,  // port
                "dummy.proxy.jp",  // proxy
                {  // authentication
                    true,  // isEnabled
                    "dummy_user",  // username
                    "dummy_password"  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile0 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true, // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting0
            },
            ipSetting0
        };

        auto pNetworkConnection = nn::nifm::GetGlobalNetworkConnectionPointer();
        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile0);
        nn::nifm::SetRequestNetworkProfileId(pNetworkConnection->GetRequestHandle(), temporaryNetworkProfile.GetId());

        nn::nifm::SubmitNetworkRequestAndWait();
        EXPECT_FALSE(nn::nifm::IsNetworkRequestOnHold());
        EXPECT_FALSE(nn::nifm::IsNetworkAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultConnectionTestCommunicationFailure, pNetworkConnection->GetResult());
        nn::nifm::CancelNetworkRequest();

        nn::nifm::SetLocalNetworkMode(true);

        nn::nifm::SubmitNetworkRequestAndWait();
        EXPECT_FALSE(nn::nifm::IsNetworkRequestOnHold());
        EXPECT_TRUE(nn::nifm::IsNetworkAvailable());
        NNT_EXPECT_RESULT_SUCCESS(pNetworkConnection->GetResult());
        nn::nifm::CancelNetworkRequest();
    }

    // クローズドな LAN へ接続しようとすると WISPr 認証がおこなわれない
    {
        nn::nifm::test::ScopedInitializer<nn::nifm::Initialize, nn::nifm::FinalizeForTest> initializer(false);
        NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                false,  // isAuto
                { { 52, 68, 203, 240 } },  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };

        nn::util::Uuid id;
        id.FromString("1a765924-cfb0-4833-a9da-aa81d83aea4a"); // mmt

        nn::nifm::IpSettingData defaultIpSetting;
        nn::nifm::GetDefaultIpSetting(&defaultIpSetting);

        nn::nifm::SetDefaultIpSetting(ipSetting);

        NN_UTIL_SCOPE_EXIT
        {
            nn::nifm::SetDefaultIpSetting(defaultIpSetting);
        };

        auto pNetworkConnection = nn::nifm::GetGlobalNetworkConnectionPointer();
        nn::nifm::SetRequestNetworkProfileId(pNetworkConnection->GetRequestHandle(), id);

        nn::nifm::SetLocalNetworkMode(true);

        nn::nifm::SubmitNetworkRequestAndWait();
        EXPECT_FALSE(nn::nifm::IsNetworkRequestOnHold());
        EXPECT_TRUE(nn::nifm::IsNetworkAvailable());
        NNT_EXPECT_RESULT_SUCCESS(pNetworkConnection->GetResult());
        // TODO: 疎通確認サーバには行けないことを確認
        nn::nifm::CancelNetworkRequest();

        nn::nifm::SetLocalNetworkMode(false);

        nn::nifm::SubmitNetworkRequestAndWait();
        EXPECT_FALSE(nn::nifm::IsNetworkRequestOnHold());
        EXPECT_TRUE(nn::nifm::IsNetworkAvailable());
        NNT_EXPECT_RESULT_SUCCESS(pNetworkConnection->GetResult());
        nn::nifm::CancelNetworkRequest();
    }
} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, Rejectable)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    //////////////////////////////////////////
    // ローカル接続⇔インフラ接続での終了待ち
    //////////////////////////////////////////

    // ローカル通信
    nn::nifm::Request request1(nn::nifm::test::RequestParametersLocalGeneric);
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(request1.GetResult());

    // ローカル通信よりも低優先度のインフラ利用要求
    nn::nifm::NetworkConnection networkConnection2;
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection2.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnection2.GetResult());

    // ローカル通信よりも高優先度のインフラ利用要求１
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), 50));
    networkConnection2.SubmitRequest();

    // ローカル通信よりも高優先度のインフラ利用要求２
    nn::nifm::NetworkConnection networkConnection3;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection3.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection3.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection3.GetRequestHandle(), 51));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRejectable(networkConnection3.GetRequestHandle(), false));
    networkConnection3.SubmitRequest();

    // ローカル通信が終了するまでは OnHold のまま
    for( int i = 0; i < 100; ++i )
    {
        if( i % 20 == 0 )
        {
            NN_LOG("(%d/5)\n", i / 20 + 1 );
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        ASSERT_EQ(nn::nifm::RequestState_OnHold, networkConnection2.GetRequestState());
        ASSERT_EQ(nn::nifm::RequestState_OnHold, networkConnection3.GetRequestState());
    }
    ASSERT_EQ(nn::nifm::RequestState_Blocking, request1.GetRequestState());
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, request1.GetResult());  // Blocking の遷移時点で Result が変わる

    // ローカル通信の終了処理
    // ....
    request1.Cancel();

    ASSERT_TRUE(nn::nifm::test::WaitRequestCompletion(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    ASSERT_TRUE(nn::nifm::test::WaitRequestCompletion(&networkConnection3, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    NNT_ASSERT_RESULT_SUCCESS(networkConnection2.GetResult());
    NNT_ASSERT_RESULT_SUCCESS(networkConnection3.GetResult());

    // 低優先度のローカル通信利用要求は失敗
    nn::nifm::Request request4(nn::nifm::test::RequestParametersLocalGeneric);
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request4, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, request4.GetResult());

    // 高優先度のローカル通信
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(request4.GetHandle(), 40));
    request4.Submit();

    // インフラ通信が終了するまでは OnHold のまま
    for( int i = 0; i < 100; ++i )
    {
        if( i % 20 == 0 )
        {
            NN_LOG("(%d/5)\n", i / 20 + 1 );
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        ASSERT_EQ(nn::nifm::RequestState_OnHold, request4.GetRequestState());
    }

    // 取り下げ可能な利用要求は先に取り下げられる
    ASSERT_EQ(nn::nifm::RequestState_Free, networkConnection2.GetRequestState());

    ASSERT_EQ(nn::nifm::RequestState_Blocking, networkConnection3.GetRequestState());
    // インフラ通信の終了処理
    // ...
    networkConnection3.CancelRequest();

    ASSERT_TRUE(nn::nifm::test::WaitRequestCompletion(&request4, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(request4.GetResult());

    ASSERT_EQ(nn::nifm::RequestState_Free, networkConnection2.GetRequestState());
    ASSERT_EQ(nn::nifm::RequestState_Free, networkConnection3.GetRequestState());

    request4.Cancel();

    //////////////////////////////////////////
    // インフラ接続⇔インフラ接続での終了待ち
    //////////////////////////////////////////

    // 取り下げ依頼を受信する通常優先度のインフラ接続
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection3.GetRequestHandle(), 128));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection3, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection3.GetResult());

    const nn::nifm::WirelessSettingData wirelessSetting0 = nn::nifm::InvalidWirelessSettingData;
    const nn::nifm::IpSettingData ipSetting0 = { {true, {}, {}, {}}, {true, {}, {}}, {false, 0, "", {false, "", ""}}, 1400 };
    nn::nifm::NetworkProfileData networkProfile0 = { nn::util::InvalidUuid, {}, nn::nifm::NetworkProfileType_User, nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet, true, true, { wirelessSetting0 }, ipSetting0 };

    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile0);

    nn::nifm::NetworkConnection networkConnection5;
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection5.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection5.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection5.GetRequestHandle(), 100));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection5.GetRequestHandle(), temporaryNetworkProfile.GetId()));
    networkConnection5.SubmitRequest();

    // インフラ通信が終了するまでは OnHold のまま
    for( int i = 0; i < 100; ++i )
    {
        if( i % 20 == 0 )
        {
            NN_LOG("(%d/5)\n", i / 20 + 1 );
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        ASSERT_EQ(nn::nifm::RequestState_OnHold, networkConnection5.GetRequestState());
    }

    ASSERT_EQ(nn::nifm::RequestState_Blocking, networkConnection3.GetRequestState());
    // インフラ通信の終了処理
    // ...
    networkConnection3.CancelRequest();

    ASSERT_TRUE(nn::nifm::test::WaitRequestCompletion(&networkConnection5, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection5.GetResult());

    //
    networkConnection5.CancelRequest();
}

TEST_F(ConnectCaseTest, RejectableEvent)
{
    NN_ALIGNAS(nn::os::ThreadStackAlignment) static char stackOfThread[4 * 1024];
    ASSERT_EQ(0, reinterpret_cast<uintptr_t>(&stackOfThread) % nn::os::ThreadStackAlignment);

    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    //////////////////////////////////////////
    // ローカル接続⇔インフラ接続での終了待ち
    //////////////////////////////////////////

    // ローカル通信
    nn::nifm::Request request1(nn::nifm::test::RequestParametersLocalGeneric);
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(request1.GetResult());

    // ローカル通信を終了するためだけのスレッド
    nn::os::ThreadType CleanUpThread;
    NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&CleanUpThread, &CleanUpThreadFunc, &request1, &stackOfThread, 4 * 1024, 24 ) );
    nn::os::StartThread( &CleanUpThread );

    // ローカル通信よりも低優先度のインフラ利用要求
    nn::nifm::NetworkConnection networkConnection2;
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection2.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnection2.GetResult());

    // ローカル通信よりも高優先度のインフラ利用要求１
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), 50));
    networkConnection2.SubmitRequest();

    // ローカル通信よりも高優先度のインフラ利用要求２
    nn::nifm::NetworkConnection networkConnection3;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection3.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection3.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection3.GetRequestHandle(), 51));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRejectable(networkConnection3.GetRequestHandle(), false));
    networkConnection3.SubmitRequest();

    // ローカル通信終了
    nn::os::WaitThread(&CleanUpThread);
    nn::os::DestroyThread(&CleanUpThread);

    ASSERT_TRUE(nn::nifm::test::WaitRequestCompletion(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    ASSERT_TRUE(nn::nifm::test::WaitRequestCompletion(&networkConnection3, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    NNT_ASSERT_RESULT_SUCCESS(networkConnection2.GetResult());
    NNT_ASSERT_RESULT_SUCCESS(networkConnection3.GetResult());

    // インフラ通信２を終了するためだけのスレッド
    NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&CleanUpThread, &CleanUpThreadFunc, networkConnection3.GetRequestHandle()._value, &stackOfThread, 4 * 1024, 24));
    nn::os::StartThread( &CleanUpThread );

    // 低優先度のローカル通信利用要求は失敗
    nn::nifm::Request request4(nn::nifm::test::RequestParametersLocalGeneric);
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request4, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, request4.GetResult());

    // 高優先度のローカル通信
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(request4.GetHandle(), 40));
    request4.Submit();

    // インフラ通信２終了
    nn::os::WaitThread(&CleanUpThread);
    nn::os::DestroyThread(&CleanUpThread);

    ASSERT_TRUE(nn::nifm::test::WaitRequestCompletion(&request4, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(request4.GetResult());

    ASSERT_EQ(nn::nifm::RequestState_Free, networkConnection2.GetRequestState());
    ASSERT_EQ(nn::nifm::RequestState_Free, networkConnection3.GetRequestState());

    request4.Cancel();

    //////////////////////////////////////////
    // インフラ接続⇔インフラ接続での終了待ち
    //////////////////////////////////////////

    // 取り下げ依頼を受信する通常優先度のインフラ通信利用要求１
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection3.GetRequestHandle(), 128));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection3, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection3.GetResult());

    // インフラ通信１を終了するためだけのスレッド
    NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&CleanUpThread, &CleanUpThreadFunc, networkConnection3.GetRequestHandle()._value, &stackOfThread, 4 * 1024, 24));
    nn::os::StartThread( &CleanUpThread );

    const nn::nifm::WirelessSettingData wirelessSetting0 = nn::nifm::InvalidWirelessSettingData;
    const nn::nifm::IpSettingData ipSetting0 = { {true, {}, {}, {}}, {true, {}, {}}, {false, 0, "", {false, "", ""}}, 1400 };
    nn::nifm::NetworkProfileData networkProfile0 = { nn::util::InvalidUuid, {}, nn::nifm::NetworkProfileType_Temporary, nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet, true, true, { wirelessSetting0 }, ipSetting0 };

    // 高優先度のインフラ通信利用要求２（テンポラリ接続）
    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile0);

    nn::nifm::NetworkConnection networkConnection5;
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection5.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection5.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection5.GetRequestHandle(), 100));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection5.GetRequestHandle(), temporaryNetworkProfile.GetId()));
    networkConnection5.SubmitRequest();

    // インフラ通信１終了
    nn::os::WaitThread(&CleanUpThread);
    nn::os::DestroyThread(&CleanUpThread);

    ASSERT_TRUE(nn::nifm::test::WaitRequestCompletion(&networkConnection5, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection5.GetResult());

    //
    networkConnection5.CancelRequest();
}

TEST_F(ConnectCaseTest, GetCurrentPrimaryIpAddress)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::socket::InAddr inAddr;


    ASSERT_TRUE(nn::nifm::test::Disconnect());
    inAddr.S_addr = 0;
    EXPECT_TRUE(nn::nifm::GetCurrentPrimaryIpAddress(&inAddr).IsFailure());

    {
        nn::nifm::SubmitNetworkRequestAndWait();

        EXPECT_TRUE(nn::nifm::IsNetworkAvailable());

        inAddr.S_addr = 0;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetCurrentPrimaryIpAddress(&inAddr));
        EXPECT_TRUE(inAddr.S_addr != 0);
    }

    EXPECT_TRUE(nn::nifm::test::Disconnect());
    inAddr.S_addr = 0;
    EXPECT_TRUE(nn::nifm::GetCurrentPrimaryIpAddress(&inAddr).IsFailure());
}

TEST_F(ConnectCaseTest, GetCurrentIpStatus)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::socket::InAddr ipAddress;
    nn::socket::InAddr subnetMask;
    nn::socket::InAddr defaultGateway;
    nn::socket::InAddr preferredDns;
    nn::socket::InAddr alternateDns;

    ASSERT_TRUE(nn::nifm::test::Disconnect());
    ipAddress.S_addr = 0;
    subnetMask.S_addr = 0;
    defaultGateway.S_addr = 0;
    preferredDns.S_addr = 0;
    alternateDns.S_addr = 0;
    EXPECT_TRUE(nn::nifm::GetCurrentIpConfigInfo(&ipAddress, &subnetMask, &defaultGateway, &preferredDns, &alternateDns).IsFailure());

    {
        // imatake-mixed
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    13,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6d,0x69,0x78,0x65,0x64 } // hex
                },
            true  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                false,  // isAuto
                { {192, 168, 11, 30} },  // ipAddress
                { {255, 255, 255, 0} },  // subnetMask
                { {192, 168, 11, 1}}  // defaultGateway
            },
            {  // dns
                false,  // isAuto
                { {52, 68, 203, 240} },  // preferredDns
                { {52, 68, 203, 241} }  // alternateDns
            },
            {  // proxy
                true,  // isEnabled
                8080,  // port
                "dummy.proxy.jp",  // proxy
                {  // authentication
                    true,  // isEnabled
                    "dummy_user",  // username
                    "dummy_password"  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true, // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting
            },
            ipSetting
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());

        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetCurrentIpConfigInfo(&ipAddress, &subnetMask, &defaultGateway, &preferredDns, &alternateDns));

        EXPECT_EQ(IpV4AddressToUlong(ipSetting.ip.ipAddress), ipAddress.S_addr);
        EXPECT_EQ(IpV4AddressToUlong(ipSetting.ip.subnetMask), subnetMask.S_addr);
        EXPECT_EQ(IpV4AddressToUlong(ipSetting.ip.defaultGateway), defaultGateway.S_addr);
        EXPECT_EQ(IpV4AddressToUlong(ipSetting.dns.preferredDns), preferredDns.S_addr);
        EXPECT_EQ(IpV4AddressToUlong(ipSetting.dns.alternateDns), alternateDns.S_addr);

        nn::nifm::test::PrintInAddr(ipAddress);
        nn::nifm::test::PrintInAddr(subnetMask);
        nn::nifm::test::PrintInAddr(defaultGateway);
        nn::nifm::test::PrintInAddr(preferredDns);
        nn::nifm::test::PrintInAddr(alternateDns);
    }

    // AlternetDns のみ設定（IP 設定手動）
    {
        // imatake-mixed
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    13,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6d,0x69,0x78,0x65,0x64 } // hex
                },
            true  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                false,  // isAuto
                { { 192, 168, 11, 10 } },  // ipAddress
                { { 255, 255, 255, 0 } },  // subnetMask
                { { 192, 168, 11, 1 } }  // defaultGateway
            },
            {  // dns
                false,  // isAuto
                {},  // preferredDns
                { { 52, 68, 203, 242 } }  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true, // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting
            },
            ipSetting
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());

        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetCurrentIpConfigInfo(&ipAddress, &subnetMask, &defaultGateway, &preferredDns, &alternateDns));

        EXPECT_EQ(IpV4AddressToUlong(ipSetting.dns.alternateDns), preferredDns.S_addr); // 優先 DNS が無効なので、代替が繰り上げられる
        EXPECT_EQ(0, alternateDns.S_addr); // 代替 DNS は 0 になる

        nn::nifm::test::PrintInAddr(ipAddress);
        nn::nifm::test::PrintInAddr(subnetMask);
        nn::nifm::test::PrintInAddr(defaultGateway);
        nn::nifm::test::PrintInAddr(preferredDns);
        nn::nifm::test::PrintInAddr(alternateDns);
    }

    // AlternetDns のみ設定（IP 設定自動）
    {
        // imatake-mixed
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    13,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6d,0x69,0x78,0x65,0x64 } // hex
                },
            true  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                false,  // isAuto
                {},  // preferredDns
                { { 52, 68, 203, 243 } }  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true, // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting
            },
            ipSetting
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());

        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetCurrentIpConfigInfo(&ipAddress, &subnetMask, &defaultGateway, &preferredDns, &alternateDns));

        EXPECT_EQ(IpV4AddressToUlong(ipSetting.dns.alternateDns), preferredDns.S_addr); // 優先 DNS が無効なので、代替が繰り上げられる
        EXPECT_EQ(0, alternateDns.S_addr); // 代替 DNS は 0 になる

        nn::nifm::test::PrintInAddr(ipAddress);
        nn::nifm::test::PrintInAddr(subnetMask);
        nn::nifm::test::PrintInAddr(defaultGateway);
        nn::nifm::test::PrintInAddr(preferredDns);
        nn::nifm::test::PrintInAddr(alternateDns);
    }

    EXPECT_TRUE(nn::nifm::test::Disconnect());
    ipAddress.S_addr = 0;
    subnetMask.S_addr = 0;
    defaultGateway.S_addr = 0;
    preferredDns.S_addr = 0;
    alternateDns.S_addr = 0;
    EXPECT_TRUE(nn::nifm::GetCurrentIpConfigInfo(&ipAddress, &subnetMask, &defaultGateway, &preferredDns, &alternateDns).IsFailure());

} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, Exclusive)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    // 通常のインフラ通信要求 1
    nn::nifm::NetworkConnection networkConnection1;
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection1.GetRequestState());
    NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

    // ローカル通信要求（独占） 2
    nn::nifm::NetworkConnection networkConnection2;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection2.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), 100));
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection2.GetRequestState());
    NNT_ASSERT_RESULT_SUCCESS(networkConnection2.GetResult());

    // 独占により追い出された要求
    NNT_NIFM_WAIT_FOR_PROCESSING;
    EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection1.GetRequestState());
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnection1.GetResult());

    // より優先度の高いローカル通信要求（独占） 3
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection1.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection1.GetRequestHandle(), 90));
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    networkConnection1.SubmitRequest();

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5)); // Blocking への遷移待ち

    // ローカル通信が終了するまでは OnHold のまま
    ASSERT_EQ(nn::nifm::RequestState_OnHold, networkConnection1.GetRequestState());

    ASSERT_EQ(nn::nifm::RequestState_Blocking, networkConnection2.GetRequestState());

    // ローカル通信の終了処理
    // ....
    networkConnection2.CancelRequest();

    ASSERT_TRUE(nn::nifm::test::WaitRequestCompletion(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection1.GetRequestState());
    NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

    // 独占によりブロック状態に遷移しキャンセルした要求
    EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection2.GetRequestState());
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnection2.GetResult());

    // ローカル通信に後からインフラ通信（独占）が入る例
    // より優先度の高いインフラ通信要求（独占） 4
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection2.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), 80));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnection2.GetRequestHandle(), true));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnection2.GetRequestHandle(), false));
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    networkConnection2.SubmitRequest();

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5)); // Blocking への遷移待ち

    // ローカル通信が終了するまでは OnHold のまま
    ASSERT_EQ(nn::nifm::RequestState_OnHold, networkConnection2.GetRequestState());

    ASSERT_EQ(nn::nifm::RequestState_Blocking, networkConnection1.GetRequestState());

    // ローカル通信の終了処理
    // ....
    networkConnection1.CancelRequest();

    ASSERT_TRUE(nn::nifm::test::WaitRequestCompletion(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection2.GetRequestState());
    NNT_ASSERT_RESULT_SUCCESS(networkConnection2.GetResult());

    // より優先度の高いインフラ通信（非独占） 5
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection1.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection1.GetRequestHandle(), 70));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnection1.GetRequestHandle(), true));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnection1.GetRequestHandle(), true));
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection1.GetRequestState());
    NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

    // 非独占により追い出された独占要求
    NNT_NIFM_WAIT_FOR_PROCESSING;
    EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection2.GetRequestState());
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnection2.GetResult());

    // 非独占に後から独占が入る例
    // 優先度の等しいインフラ通信要求（独占） 6
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection2.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), 70));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnection2.GetRequestHandle(), true));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnection2.GetRequestHandle(), false));
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    // 優先度が等しい独占は後負け
    EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection2.GetRequestState());
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnection2.GetResult());

    const nn::nifm::WirelessSettingData wirelessSetting = nn::nifm::InvalidWirelessSettingData;
    const nn::nifm::IpSettingData ipSetting = { {true, {}, {}, {}}, {true, {}, {}}, {false, 0, "", {false, "", ""}}, 1400 };
    nn::nifm::NetworkProfileData networkProfile = { nn::util::InvalidUuid, {}, nn::nifm::NetworkProfileType_Temporary, nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet, true, true, { wirelessSetting }, ipSetting };

    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile( networkProfile );
    ASSERT_NE( nn::util::InvalidUuid, temporaryNetworkProfile.GetId() );

    // 非独占に後から独占が入る例
    // テンポラリ接続要求（独占）
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection2.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffort));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), 60));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection2.GetRequestHandle(), temporaryNetworkProfile.GetId()));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    // 独占により追い出された要求
    EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection1.GetRequestState());
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnection1.GetResult());

    // 優先度の等しいインフラ通信要求（非独占）
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection1.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection1.GetRequestHandle(), 60));
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    // 独占に後から非独占が入る例
    // 優先度が等しい非独占は後負け
    EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection1.GetRequestState());
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnection1.GetResult());

    // より優先度の高いローカル通信
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection1.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection1.GetRequestHandle(), 50));
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    // 独占により追い出された要求
    EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection2.GetRequestState());
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnection2.GetResult());
}

TEST_F(ConnectCaseTest, GreedyAndSharable)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    {
        // 通知などの基本的に切断されない BG 通信 1
        nn::nifm::NetworkConnection networkConnectionNpns;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionNpns.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnectionNpns.GetRequestHandle(), false));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnectionNpns.GetRequestHandle(), true));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestPersistent(networkConnectionNpns.GetRequestHandle(), true));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNpns, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnectionNpns.GetRequestState());
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionNpns.GetResult());

        // BCATなどの通常 BG 通信（または，通常 BG 通信を切断しないアプリ）2
        nn::nifm::NetworkConnection networkConnectionBgProcess;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBgProcess.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnectionBgProcess.GetRequestHandle(), false));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnectionBgProcess.GetRequestHandle(), false));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestPersistent(networkConnectionBgProcess.GetRequestHandle(), true));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionBgProcess, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        // networkConnection 1,2 共に受理
        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnectionNpns.GetRequestState());
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionNpns.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnectionBgProcess.GetRequestState());
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionBgProcess.GetResult());

        // 通常 BG 通信を切断するアプリ（優先度が等しいので後負け）3
        nn::nifm::NetworkConnection networkConnectionApp;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionApp.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnectionApp.GetRequestHandle(), true));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnectionApp.GetRequestHandle(), true));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApp, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        EXPECT_EQ(nn::nifm::RequestState_Free, networkConnectionApp.GetRequestState());
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionApp.GetResult());

        // 通常 BG 通信を切断するアプリ（優先度を高くして乗っ取る）4
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionApp.GetRequestHandle(), 90));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApp, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        // 通常 BG 通信が追い出される
        NNT_NIFM_WAIT_FOR_PROCESSING;
        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnectionApp.GetRequestState());
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApp.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionNpns.GetResult());
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionBgProcess.GetResult());

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));

        // 通常 BG 通信を切断するアプリをひっこめる
        networkConnectionApp.CancelRequest();

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));

        // Persistent な通常 BG 通信が復活
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionBgProcess.GetResult());
    }

    {
        nn::nifm::NetworkConnection networkConnectionGreedySharable;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionGreedySharable.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnectionGreedySharable.GetRequestHandle(), true));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnectionGreedySharable.GetRequestHandle(), true));

        nn::nifm::NetworkConnection networkConnectionGreedyNotSharable;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionGreedyNotSharable.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnectionGreedyNotSharable.GetRequestHandle(), true));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnectionGreedyNotSharable.GetRequestHandle(), false));

        nn::nifm::NetworkConnection networkConnectionNotGreedySharable;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionNotGreedySharable.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnectionNotGreedySharable.GetRequestHandle(), false));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnectionNotGreedySharable.GetRequestHandle(), true));

        nn::nifm::NetworkConnection networkConnectionNotGreedyNotSharable;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionNotGreedyNotSharable.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnectionNotGreedyNotSharable.GetRequestHandle(), false));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnectionNotGreedyNotSharable.GetRequestHandle(), false));

        {
            nn::nifm::NetworkConnection networkConnectionGreedySharable2;
            NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionGreedySharable2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnectionGreedySharable2.GetRequestHandle(), true));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnectionGreedySharable2.GetRequestHandle(), true));

            // Greedy && Sharable < Greedy && Sharable => 共存
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedySharable.GetRequestHandle(), 80));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedySharable2.GetRequestHandle(), 90));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedySharable2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedySharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedySharable2.GetResult());
            networkConnectionGreedySharable.CancelRequest();
            networkConnectionGreedySharable2.CancelRequest();
        }

        {
            // Greedy && Sharable < Greedy && NotSharable => Greedy && Sharable だけ受理
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedySharable.GetRequestHandle(), 80));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedyNotSharable.GetRequestHandle(), 90));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedySharable.GetResult());
            NNT_NIFM_WAIT_FOR_PROCESSING;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionGreedyNotSharable.GetResult());
            networkConnectionGreedySharable.CancelRequest();

            // Greedy && Sharable > Greedy && NotSharable => Greedy && NotSharable だけ受理
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedySharable.GetRequestHandle(), 90));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedyNotSharable.GetRequestHandle(), 80));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_NIFM_WAIT_FOR_PROCESSING;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionGreedySharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedyNotSharable.GetResult());
            networkConnectionGreedyNotSharable.CancelRequest();
        }

        {
            // Greedy && Sharable < NotGreedy && Sharable => 共存
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedySharable.GetRequestHandle(), 80));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedySharable.GetRequestHandle(), 90));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedySharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable.GetResult());
            networkConnectionGreedySharable.CancelRequest();
            networkConnectionNotGreedySharable.CancelRequest();

            // Greedy && Sharable > NotGreedy && Sharable => 共存
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedySharable.GetRequestHandle(), 90));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedySharable.GetRequestHandle(), 80));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedySharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable.GetResult());
            networkConnectionGreedySharable.CancelRequest();
            networkConnectionNotGreedySharable.CancelRequest();
        }

        {
            // Greedy && Sharable < NotGreedy && NotSharable => Greedy && Sharable だけ受理
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedySharable.GetRequestHandle(), 80));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedyNotSharable.GetRequestHandle(), 90));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedySharable.GetResult());
            NNT_NIFM_WAIT_FOR_PROCESSING;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionNotGreedyNotSharable.GetResult());
            networkConnectionGreedySharable.CancelRequest();

            // Greedy && Sharable > NotGreedy && NotSharable => NotGreedy && NotSharable だけ受理
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedySharable.GetRequestHandle(), 90));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedyNotSharable.GetRequestHandle(), 80));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_NIFM_WAIT_FOR_PROCESSING;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionGreedySharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedyNotSharable.GetResult());
            networkConnectionNotGreedyNotSharable.CancelRequest();
        }

        {
            nn::nifm::NetworkConnection networkConnectionGreedyNotSharable2;
            NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionGreedyNotSharable2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnectionGreedyNotSharable2.GetRequestHandle(), true));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnectionGreedyNotSharable2.GetRequestHandle(), false));

            // Greedy && NotSharable < Greedy && NotSharable => 優先度の高い Greedy && NotSharable だけ受理
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedyNotSharable.GetRequestHandle(), 80));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedyNotSharable2.GetRequestHandle(), 90));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedyNotSharable2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedyNotSharable.GetResult());
            NNT_NIFM_WAIT_FOR_PROCESSING;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionGreedyNotSharable2.GetResult());
            networkConnectionGreedyNotSharable.CancelRequest();

            // Greedy && NotSharable > Greedy && NotSharable => 優先度の高い Greedy && NotSharable だけ受理
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedyNotSharable.GetRequestHandle(), 90));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedyNotSharable2.GetRequestHandle(), 80));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedyNotSharable2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_NIFM_WAIT_FOR_PROCESSING;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionGreedyNotSharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedyNotSharable2.GetResult());
            networkConnectionGreedyNotSharable2.CancelRequest();
        }

        {
            // Greedy && NotSharable < NotGreedy && Sharable => 共存
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedyNotSharable.GetRequestHandle(), 80));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedySharable.GetRequestHandle(), 90));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedyNotSharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable.GetResult());
            networkConnectionGreedyNotSharable.CancelRequest();
            networkConnectionNotGreedySharable.CancelRequest();

            // Greedy && NotSharable > NotGreedy && Sharable => 共存
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedyNotSharable.GetRequestHandle(), 90));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedySharable.GetRequestHandle(), 80));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedyNotSharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable.GetResult());
            networkConnectionGreedyNotSharable.CancelRequest();
            networkConnectionNotGreedySharable.CancelRequest();
        }

        {
            // Greedy && NotSharable < NotGreedy && NotSharable => Greedy && NotSharable だけ受理
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedyNotSharable.GetRequestHandle(), 80));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedyNotSharable.GetRequestHandle(), 90));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedyNotSharable.GetResult());
            NNT_NIFM_WAIT_FOR_PROCESSING;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionNotGreedyNotSharable.GetResult());
            networkConnectionGreedyNotSharable.CancelRequest();

            // Greedy && NotSharable > NotGreedy && NotSharable => NotGreedy && NotSharable だけ受理
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedyNotSharable.GetRequestHandle(), 90));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedyNotSharable.GetRequestHandle(), 80));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_NIFM_WAIT_FOR_PROCESSING;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionGreedyNotSharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedyNotSharable.GetResult());
            networkConnectionNotGreedyNotSharable.CancelRequest();
        }

        {
            nn::nifm::NetworkConnection networkConnectionNotGreedySharable2;
            NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionNotGreedySharable2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnectionNotGreedySharable2.GetRequestHandle(), false));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnectionNotGreedySharable2.GetRequestHandle(), true));

            // NotGreedy && Sharable < NotGreedy && Sharable => 共存
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedySharable.GetRequestHandle(), 80));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedySharable2.GetRequestHandle(), 90));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedySharable2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable2.GetResult());
            networkConnectionNotGreedySharable.CancelRequest();
            networkConnectionNotGreedySharable2.CancelRequest();

            // NotGreedy && Sharable > NotGreedy && Sharable => 共存
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedySharable.GetRequestHandle(), 90));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedySharable2.GetRequestHandle(), 80));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedySharable2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable2.GetResult());
            networkConnectionNotGreedySharable.CancelRequest();
            networkConnectionNotGreedySharable2.CancelRequest();
        }

        {
            // NotGreedy && Sharable < NotGreedy && NotSharable => 共存
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedySharable.GetRequestHandle(), 80));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedyNotSharable.GetRequestHandle(), 90));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedyNotSharable.GetResult());
            networkConnectionNotGreedySharable.CancelRequest();
            networkConnectionNotGreedyNotSharable.CancelRequest();

            // NotGreedy && Sharable > NotGreedy && Sharable => 共存
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedySharable.GetRequestHandle(), 90));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedyNotSharable.GetRequestHandle(), 80));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedyNotSharable.GetResult());
            networkConnectionNotGreedySharable.CancelRequest();
            networkConnectionNotGreedyNotSharable.CancelRequest();
        }

        {
            nn::nifm::NetworkConnection networkConnectionNotGreedyNotSharable2;
            NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionNotGreedyNotSharable2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnectionNotGreedyNotSharable2.GetRequestHandle(), false));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnectionNotGreedyNotSharable2.GetRequestHandle(), false));

            // NotGreedy && NotSharable < NotGreedy && NotSharable => 共存
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedyNotSharable.GetRequestHandle(), 80));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedyNotSharable2.GetRequestHandle(), 90));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedyNotSharable2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedyNotSharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedyNotSharable2.GetResult());
            networkConnectionNotGreedyNotSharable.CancelRequest();
            networkConnectionNotGreedyNotSharable2.CancelRequest();

            // NotGreedy && NotSharable > NotGreedy && NotSharable => 共存
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedyNotSharable.GetRequestHandle(), 90));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedyNotSharable2.GetRequestHandle(), 80));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedyNotSharable2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedyNotSharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedyNotSharable2.GetResult());
            networkConnectionNotGreedyNotSharable.CancelRequest();
            networkConnectionNotGreedyNotSharable2.CancelRequest();
        }

        // より複雑な例
        {
            // Greedy && NotSharable(80) < NotGreedy && Sharable(90) => 共存
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedyNotSharable.GetRequestHandle(), 80));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedySharable.GetRequestHandle(), 90));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedyNotSharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable.GetResult());

            // Greedy && NotSharable(80) < NotGreedy && Sharable(90) < [Greedy && Sharable](100) => Greedy && Sharable だけ却下
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedySharable.GetRequestHandle(), 100));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedyNotSharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable.GetResult());
            NNT_NIFM_WAIT_FOR_PROCESSING;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionGreedySharable.GetResult());

            // [Greedy && Sharable](70) < Greedy && NotSharable(80) < NotGreedy && Sharable(90) => Greedy && NotSharable だけ却下
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedySharable.GetRequestHandle(), 70));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_NIFM_WAIT_FOR_PROCESSING;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionGreedyNotSharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedySharable.GetResult());

            // Greedy && Sharable(70) < [Greedy && NotSharable](80) < NotGreedy && Sharable(90) => Greedy && NotSharable だけ却下
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedyNotSharable.GetRequestHandle(), 80));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_NIFM_WAIT_FOR_PROCESSING;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionGreedyNotSharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedySharable.GetResult());

            networkConnectionNotGreedySharable.CancelRequest();
            networkConnectionGreedySharable.CancelRequest();
        }

        {
            // Greedy && Sharable(80) < NotGreedy && Sharable(90) => 共存
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedySharable.GetRequestHandle(), 80));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedySharable.GetRequestHandle(), 90));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedySharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable.GetResult());

            // [NotGreedy && NotSharable](70) < Greedy && Sharable(80) < NotGreedy && Sharable(90) => Greedy && Sharable だけ却下
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedyNotSharable.GetRequestHandle(), 70));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedyNotSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_NIFM_WAIT_FOR_PROCESSING;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionGreedySharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedyNotSharable.GetResult());

            {
                // NotGreedy && NotSharable(70) < NotGreedy && Sharable(90) < [NotGreedy && Sharable](100) => 共存
                nn::nifm::NetworkConnection networkConnectionNotGreedySharable2;
                NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionNotGreedySharable2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
                NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnectionNotGreedySharable2.GetRequestHandle(), false));
                NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnectionNotGreedySharable2.GetRequestHandle(), true));
                NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedySharable2.GetRequestHandle(), 100));
                EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedySharable2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
                NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable.GetResult());
                NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedyNotSharable.GetResult());
                NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable2.GetResult());
            }

            // NotGreedy && NotSharable(70) < NotGreedy && Sharable(90) => 共存
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedyNotSharable.GetResult());

            // NotGreedy && NotSharable(70) < [NotGreedy && NotSharable](80) < NotGreedy && Sharable(90) => 共存
            nn::nifm::NetworkConnection networkConnectionNotGreedyNotSharable2;
            NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionNotGreedyNotSharable2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnectionNotGreedyNotSharable2.GetRequestHandle(), false));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnectionNotGreedyNotSharable2.GetRequestHandle(), false));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedyNotSharable2.GetRequestHandle(), 80));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedyNotSharable2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedySharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedyNotSharable.GetResult());
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedyNotSharable2.GetResult());

            networkConnectionNotGreedySharable.CancelRequest();
            networkConnectionNotGreedyNotSharable.CancelRequest();
            networkConnectionNotGreedyNotSharable2.CancelRequest();
        }

        {
            // Greedy && Sharable を Persistent に
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionGreedySharable.GetRequestHandle(), 100));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestPersistent(networkConnectionGreedySharable.GetRequestHandle(), true));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionGreedySharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedySharable.GetResult());

            // Greedy && Sharable < NotGreedy && NotSharable => NotGreedy && NotSharable だけ受理
            {
                nn::nifm::NetworkConnection networkConnectionNotGreedyNotSharable2;
                NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionNotGreedyNotSharable2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
                NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnectionNotGreedyNotSharable2.GetRequestHandle(), false));
                NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnectionNotGreedyNotSharable2.GetRequestHandle(), false));
                NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnectionNotGreedyNotSharable2.GetRequestHandle(), 90));

                EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionNotGreedyNotSharable2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
                NNT_ASSERT_RESULT_SUCCESS(networkConnectionNotGreedyNotSharable2.GetResult());
                EXPECT_TRUE(networkConnectionGreedySharable.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
                NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionGreedySharable.GetResult());

                nn::os::SleepThread(nn::TimeSpan::FromSeconds(10)); // DispatchLoop が止まるまで待つ
                networkConnectionGreedySharable.GetSystemEvent().Clear();
            }

            // 即座に Greedy && Sharable が受理される
            int64_t startTick = nn::os::GetSystemTick().GetInt64Value();
            EXPECT_TRUE(networkConnectionGreedySharable.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(30)));
            int64_t t = (nn::os::GetSystemTick().GetInt64Value() - startTick) / nn::os::GetSystemTickFrequency();
            NN_LOG("Time to signal: %02d:%02d:%02d\n", static_cast<int>(t % 60), static_cast<int>((t / 60) % 60), static_cast<int>(t / (60 * 60)));
            NNT_ASSERT_RESULT_SUCCESS(networkConnectionGreedySharable.GetResult());

            networkConnectionGreedySharable.CancelRequest();
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestPersistent(networkConnectionGreedySharable.GetRequestHandle(), false));
        }
    }
} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, GreedyAndLargeCapacity)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::util::Uuid id;
    id.FromString("93188440-b6d5-14b7-056b-08cd57910967"); // imatake-mixed_small_capacity

    nn::nifm::NetworkConnection networkConnection;
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), id));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnection.GetRequestHandle(), true));

    // 大容量通信を必要とする要求は、大容量通信を許可しない接続設定では接続できない
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection.GetResult());


    id.FromString("93188440-b6d5-14b7-056b-08cd57910968"); // imatake-wpaaes

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), id));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnection.GetRequestHandle(), false));

    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());

    nn::nifm::NetworkProfileData networkProfileData;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetCurrentNetworkProfile(&networkProfileData));

    ASSERT_EQ(id, networkProfileData.id);
}

TEST_F(ConnectCaseTest, WirelessCommunicationControl)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::nifm::SetWirelessCommunicationEnabled(true);
    ASSERT_TRUE(nn::nifm::IsWirelessCommunicationEnabled());

    {
        // 無線インターネット通信のオフ

        // imatake-wpa2aes
        nn::util::Uuid id1 = {{ 0x93,0x18,0x84,0x40,0xb6,0xd5,0x14,0xb7,0x05,0x6b,0x08,0xcd,0x57,0x91,0x09,0x60 }};

        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), id1));

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        // 無線オフ時にもスキャン結果を残すための事前スキャン
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::Scan());

        // 無線通信をオフに
        nn::nifm::SetWirelessCommunicationEnabled(false);
        EXPECT_FALSE(nn::nifm::IsWirelessCommunicationEnabled());

        // 無線オフ後、すこし遅延して切られる
        EXPECT_TRUE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceNoLongerAvailable, request1a.GetResult());

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        {
            nn::applet::AppletId appletId = nn::applet::AppletId_None;
            nn::applet::LibraryAppletMode libraryAppletMode = nn::applet::LibraryAppletMode_AllForeground;
            size_t size = 0;
            char buffer[4 * 1024];
            nn::nifm::detail::RequestClient* pRequestClient = nn::nifm::GetRequestClientPointer(request1a.GetHandle());
            NN_ASSERT_NOT_NULL(pRequestClient);
            NNT_EXPECT_RESULT_SUCCESS(pRequestClient->GetLibaryAppletArgument(&appletId, &libraryAppletMode, &size, buffer, sizeof(buffer)));
            EXPECT_EQ(nn::applet::AppletId_LibraryAppletError, appletId);
            EXPECT_EQ(nn::applet::LibraryAppletMode_AllForeground, libraryAppletMode);
        }
#endif

        // 無線オフ状態で接続要求を再提出しても拒否される
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestPersistent(request1a.GetHandle(), true));
        request1a.GetSystemEvent().Clear();
        request1a.Submit();
        EXPECT_TRUE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceDisabled, request1a.GetResult());

        nn::nifm::SetWirelessCommunicationEnabled(true);
        ASSERT_TRUE(nn::nifm::IsWirelessCommunicationEnabled());

        // 解除後に、速やかに要求の再評価が走ることの確認
        EXPECT_TRUE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(50)));
        NNT_EXPECT_RESULT_SUCCESS(request1a.GetResult());
    }

    {
        // ローカル通信のオフ

        nn::nifm::Request request1a(nn::nifm::test::RequestParametersNone);
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(request1a.GetHandle(), nn::nifm::RequirementPreset_LocalGeneric));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRejectable(request1a.GetHandle(), true));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        // 無線通信をオフに
        nn::nifm::SetWirelessCommunicationEnabled(false);
        EXPECT_FALSE(nn::nifm::IsWirelessCommunicationEnabled());

        // 無線オフ後、すこし遅延して切られる
        EXPECT_TRUE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceNoLongerAvailable, request1a.GetResult());

        // 無線オフ状態で接続要求を再提出しても拒否される
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceDisabled, request1a.GetResult());

        nn::nifm::SetWirelessCommunicationEnabled(true);
        ASSERT_TRUE(nn::nifm::IsWirelessCommunicationEnabled());
    }

    {
        // 有線に影響を与えない確認

        // ethernet
        nn::util::Uuid id1 = {{ 0x93,0x18,0x84,0x40,0xb6,0xd5,0x14,0xb7,0x05,0x6b,0x08,0xcd,0x57,0x91,0x09,0x64 }};

        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), id1));

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        // 無線通信をオフに
        nn::nifm::SetWirelessCommunicationEnabled(false);
        EXPECT_FALSE(nn::nifm::IsWirelessCommunicationEnabled());

        // 無線オフされても影響はない
        EXPECT_FALSE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        NNT_EXPECT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::SetWirelessCommunicationEnabled(true);
        ASSERT_TRUE(nn::nifm::IsWirelessCommunicationEnabled());
    }

    {
        // テスト用操作

        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetWirelessCommunicationEnabledForTest(false));
        EXPECT_FALSE(nn::nifm::IsWirelessCommunicationEnabled());

        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetWirelessCommunicationEnabledForTest(true));
        EXPECT_TRUE(nn::nifm::IsWirelessCommunicationEnabled());
    }
} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, EthernetCommunicationControl)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::nifm::SetEthernetCommunicationEnabled(true);
    ASSERT_TRUE(nn::nifm::IsEthernetCommunicationEnabled());

    {
        // 有線インターネット通信のオフ

        // ethernet
        nn::util::Uuid id1 = { { 0x93,0x18,0x84,0x40,0xb6,0xd5,0x14,0xb7,0x05,0x6b,0x08,0xcd,0x57,0x91,0x09,0x64 } };

        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), id1));

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        // 有線通信をオフに
        nn::nifm::SetEthernetCommunicationEnabled(false);
        EXPECT_FALSE(nn::nifm::IsEthernetCommunicationEnabled());

        // 有線オフ後、すこし遅延して切られる
        EXPECT_TRUE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceNoLongerAvailable, request1a.GetResult());

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        {
            nn::applet::AppletId appletId = nn::applet::AppletId_None;
            nn::applet::LibraryAppletMode libraryAppletMode = nn::applet::LibraryAppletMode_AllForeground;
            size_t size = 0;
            char buffer[4 * 1024];
            nn::nifm::detail::RequestClient* pRequestClient = nn::nifm::GetRequestClientPointer(request1a.GetHandle());
            NN_ASSERT_NOT_NULL(pRequestClient);
            NNT_EXPECT_RESULT_SUCCESS(pRequestClient->GetLibaryAppletArgument(&appletId, &libraryAppletMode, &size, buffer, sizeof(buffer)));
            EXPECT_EQ(nn::applet::AppletId_LibraryAppletError, appletId);
            EXPECT_EQ(nn::applet::LibraryAppletMode_AllForeground, libraryAppletMode);
        }
#endif

        // 有線オフ状態で接続要求を再提出しても拒否される
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestPersistent(request1a.GetHandle(), true));
        request1a.GetSystemEvent().Clear();
        request1a.Submit();
        EXPECT_TRUE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceDisabled, request1a.GetResult());

        nn::nifm::SetEthernetCommunicationEnabled(true);
        ASSERT_TRUE(nn::nifm::IsEthernetCommunicationEnabled());

        // 解除後に、速やかに要求の再評価が走ることの確認
        EXPECT_TRUE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(15)));
        NNT_EXPECT_RESULT_SUCCESS(request1a.GetResult());
    }

    {
        // 無線に影響を与えない確認

        // imatake-wpa2aes
        nn::util::Uuid id1 = { { 0x93,0x18,0x84,0x40,0xb6,0xd5,0x14,0xb7,0x05,0x6b,0x08,0xcd,0x57,0x91,0x09,0x60 } };

        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), id1));

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        // 有線通信をオフに
        nn::nifm::SetEthernetCommunicationEnabled(false);
        EXPECT_FALSE(nn::nifm::IsEthernetCommunicationEnabled());

        // 有線オフされても影響はない
        EXPECT_FALSE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        NNT_EXPECT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::SetEthernetCommunicationEnabled(true);
        ASSERT_TRUE(nn::nifm::IsEthernetCommunicationEnabled());
    }

    {
        // テスト用操作

        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetEthernetCommunicationEnabledForTest(false));
        EXPECT_FALSE(nn::nifm::IsEthernetCommunicationEnabled());

        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetEthernetCommunicationEnabledForTest(true));
        EXPECT_TRUE(nn::nifm::IsEthernetCommunicationEnabled());
    }
} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, PersistentRequest)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    // インフラ通信要求（Greedy + Not Sharable）
    nn::nifm::NetworkConnection networkConnection1;
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnection1.GetRequestHandle(), true));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnection1.GetRequestHandle(), false));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection1.GetRequestState());
    NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

    // インフラ通信要求（Persistent）
    nn::nifm::NetworkConnection networkConnection2;
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestPersistent(networkConnection2.GetRequestHandle(), true));
    networkConnection2.SubmitRequest();

    // 継続的要求は受理はされないが OnHold のまま
    for( int i = 0; i < 100; ++i )
    {
        if( i % 20 == 0 )
        {
            NN_LOG("(%d/5)\n", i / 20 + 1 );
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        ASSERT_EQ(nn::nifm::RequestState_OnHold, networkConnection2.GetRequestState());
    }

    // OnHold -> OnHold でシグナルされている
    ASSERT_TRUE(networkConnection2.GetSystemEvent().TryWait());

    // 通常のインフラ通信（Greedy + Not Sharable）の終了処理
    networkConnection1.CancelRequest();

    // 継続的要求が受理される
    ASSERT_TRUE(nn::nifm::test::WaitRequestCompletion(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    // OnHold -> Accepted でシグナルされている
    ASSERT_TRUE(networkConnection2.GetSystemEvent().TryWait());
    NNT_ASSERT_RESULT_SUCCESS(networkConnection2.GetResult());

    // インフラ通信要求（独占，優先度高）
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection1.GetRequestHandle(), 100));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnection1.GetRequestHandle(), true));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnection1.GetRequestHandle(), false));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection1.GetRequestState());
    NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

    // 継続的要求は受理から OnHold に遷移
    for( int i = 0; i < 100; ++i )
    {
        if( i % 20 == 0 )
        {
            NN_LOG("(%d/5)\n", i / 20 + 1 );
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        ASSERT_EQ(nn::nifm::RequestState_OnHold, networkConnection2.GetRequestState());
    }

    // インフラ通信要求（独占，優先度高）の終了処理
    networkConnection1.CancelRequest();

    // 継続的要求が再度受理される
    ASSERT_TRUE(nn::nifm::test::WaitRequestCompletion(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection2.GetResult());

    // ローカル通信要求
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection1.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection1.GetRequestState());
    NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

    // 継続的要求は受理から OnHold に遷移
    for( int i = 0; i < 100; ++i )
    {
        if( i % 20 == 0 )
        {
            NN_LOG("(%d/5)\n", i / 20 + 1 );
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        ASSERT_EQ(nn::nifm::RequestState_OnHold, networkConnection2.GetRequestState());
    }

    // ローカル通信要求の終了処理
    networkConnection1.CancelRequest();

    // 継続的要求が再度受理される
    ASSERT_TRUE(nn::nifm::test::WaitRequestCompletion(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection2.GetResult());

    // 継続要求のキャンセル
    networkConnection2.CancelRequest();
}

TEST_F(ConnectCaseTest, DelayedRequest)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    // imatake-nowhere (見つからないはずの接続設定)
    const nn::nifm::WirelessSettingData wirelessSetting0 = {
        {  // ssidConfig
            {  // ssid
                15,  // length
                {0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6e,0x6f,0x77,0x68,0x65, 0x72, 0x65}  // hex
            },
            false  // nonBroadcast
        },
        {  // security
            {  //authEncryption
                nn::nifm::Authentication_Wpa2Psk,  // authentication
                nn::nifm::Encryption_Aes  // encryption
            },
            {  // sharedKey
                11,  // length
                "Shi2iTaiZen"  // keyMaterial
            }
        }
    };
    const nn::nifm::IpSettingData ipSetting0 = {
        {  // ip
            true,  // isAuto
            {},  // ipAddress
            {},  // subnetMask
            {}  // defaultGateway
        },
        {  // dns
            true,  // isAuto
            {},  // preferredDns
            {}  // alternateDns
        },
        {  // proxy
            false,  // isEnabled
            0,  // port
            "",  // proxy
            {  // authentication
                false,  // isEnabled
                "",  // username
                ""  // password
            }
        },
        1400  //mtu
    };
    const nn::nifm::NetworkProfileData networkProfile0 = {
        nn::util::InvalidUuid,  // id
        {},  // name
        nn::nifm::NetworkProfileType_Temporary, // networkProfileType
        nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
        true, // isAutoConnect
        true, // isLargeCapacity
        {
            wirelessSetting0
        },
        ipSetting0
    };

    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile( networkProfile0 );
    ASSERT_NE( nn::util::InvalidUuid, temporaryNetworkProfile.GetId() );

    {
        // インフラ通信要求（つながるはず）
        nn::nifm::NetworkConnection networkConnection1;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        networkConnection1.SubmitRequest();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));     // 次の利用要求とまとめられないように時間を置く

        // インフラ通信要求（つながらないはず）
        nn::nifm::NetworkConnection networkConnection2;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection2.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), 16));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_EXPECT_RESULT_SUCCESS(networkConnection1.GetResult());  // 先に評価されるのでつながっている
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection2.GetResult());
    }

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    {
        nn::nifm::AccessPointData accessPointData;
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetCurrentAccessPoint(&accessPointData));
        uint16_t channel;
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetCurrentAccessPointChannel(&channel));
    }

    // 前の処理によるシグナルで DispatchLoop が1周まわるのを待つ
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    {
        // インフラ通信要求（つながるはずだけど、評価遅延）
        nn::nifm::NetworkConnection networkConnection1;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestInstant(networkConnection1.GetRequestHandle(), false));

        // インフラ通信要求（つながらないはず）
        nn::nifm::NetworkConnection networkConnection2;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection2.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), 16));

        networkConnection1.SubmitRequest();
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));  // このタイミングで networkConnection1 も一緒に評価される

        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection2.GetResult());
        NNT_NIFM_WAIT_FOR_PROCESSING;
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection1.GetResult());    // つながらない優先度高の要求と同時評価されるのでつながらない
    }

    // 前の処理によるシグナルで DispatchLoop が1周まわるのを待つ
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    {
        // インフラ通信要求（つながるはずだけど、評価遅延）
        nn::nifm::NetworkConnection networkConnection1;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestInstant(networkConnection1.GetRequestHandle(), false));

        // インフラ通信要求（つながらないはず）
        nn::nifm::NetworkConnection networkConnection2;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection2.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), 16));

        networkConnection1.SubmitRequest();
        networkConnection2.CancelRequest(); // 未提出の利用要求のキャンセル
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
        EXPECT_TRUE(networkConnection1.IsRequestOnHold()); // 評価は実行されない
    }
} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, InternetBestEffort)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());
    {
        nn::nifm::NetworkConnection networkConnection1;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection1.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection1.GetRequestState());
        NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

        nn::nifm::NetworkConnection networkConnection2;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection2.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffort));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection2.GetRequestState());
        NNT_ASSERT_RESULT_SUCCESS(networkConnection2.GetResult());

        // InternetBestEffor は (greedy, sahred)で通常より優先度が高いので，通常のインターネット通信 (!greedy, !shared) は追い出される
        NNT_NIFM_WAIT_FOR_PROCESSING;
        EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection1.GetRequestState());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnection1.GetResult());
    }

    // アプリと通常アプレットは共存可能
    {
        nn::nifm::NetworkConnection networkConnectionApp;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApp.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffort));

        nn::nifm::NetworkConnection networkConnectionApplet;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApplet.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForApplet));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApp, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApplet, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApp.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApplet.GetResult());
    }

    EXPECT_TRUE(nn::nifm::test::Disconnect());

    // アプリと BG は共存不可（BG が却下）
    {
        nn::nifm::NetworkConnection networkConnectionApp;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApp.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffort));

        nn::nifm::NetworkConnection networkConnectionSystem;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionSystem.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApp, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionSystem, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApp.GetResult());
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionSystem.GetResult());
    }

    EXPECT_TRUE(nn::nifm::test::Disconnect());

    // 通常アプレットと BG は共存可能
    {
        nn::nifm::NetworkConnection networkConnectionApplet;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApplet.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForApplet));

        nn::nifm::NetworkConnection networkConnectionSystem;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionSystem.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApplet, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionSystem, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApplet.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionSystem.GetResult());
    }

    // BestEffort アプレットと BG は共存不可（BG が却下）
    {
        nn::nifm::NetworkConnection networkConnectionAppletBestEffort;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionAppletBestEffort.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffortForApplet));

        nn::nifm::NetworkConnection networkConnectionSystem;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionSystem.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionAppletBestEffort, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionSystem, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionAppletBestEffort.GetResult());
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionSystem.GetResult());
    }

    // BestEffortPersistent アプレットと BG は共存不可（BG が却下）
    {
        nn::nifm::NetworkConnection networkConnectionAppletBestEffortPersistent;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionAppletBestEffortPersistent.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffortForAppletPersistent));

        nn::nifm::NetworkConnection networkConnectionSystem;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionSystem.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionAppletBestEffortPersistent, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionSystem, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionAppletBestEffortPersistent.GetResult());
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionSystem.GetResult());
    }

    EXPECT_TRUE(nn::nifm::test::Disconnect());

    // 「アプリと BestEffort アプレット」と BG は共存不可（BG が却下）
    {
        nn::nifm::NetworkConnection networkConnectionApp;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApp.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffort));

        nn::nifm::NetworkConnection networkConnectionAppletBestEffort;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionAppletBestEffort.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffortForApplet));

        nn::nifm::NetworkConnection networkConnectionSystem;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionSystem.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));


        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApp, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionAppletBestEffort, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionSystem, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApp.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionAppletBestEffort.GetResult());
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionSystem.GetResult());
    }

    // 「アプリと BestEffortPersistent アプレット」と BG は共存不可（BG が却下）
    {
        nn::nifm::NetworkConnection networkConnectionApp;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApp.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffort));

        nn::nifm::NetworkConnection networkConnectionAppletBestEffortPersistent;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionAppletBestEffortPersistent.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffortForApplet));

        nn::nifm::NetworkConnection networkConnectionSystem;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionSystem.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));


        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApp, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionAppletBestEffortPersistent, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionSystem, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApp.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionAppletBestEffortPersistent.GetResult());
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionSystem.GetResult());
    }

    EXPECT_TRUE(nn::nifm::test::Disconnect());

    // 「アプリと BestEffort アプレット」と Sharable BG は共存可能
    {
        nn::nifm::NetworkConnection networkConnectionApp;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApp.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffort));

        nn::nifm::NetworkConnection networkConnectionAppletBestEffort;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionAppletBestEffort.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffortForApplet));

        nn::nifm::NetworkConnection networkConnectionSystemSharable;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionSystemSharable.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApp, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionAppletBestEffort, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionSystemSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApp.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionAppletBestEffort.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionSystemSharable.GetResult());
    }

    EXPECT_TRUE(nn::nifm::test::Disconnect());

    // 「アプリと BestEffortPersistent アプレット」と Sharable BG は共存可能
    {
        nn::nifm::NetworkConnection networkConnectionApp;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApp.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffort));

        nn::nifm::NetworkConnection networkConnectionAppletBestEffortPersistent;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionAppletBestEffortPersistent.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffortForApplet));

        nn::nifm::NetworkConnection networkConnectionSystemSharable;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionSystemSharable.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApp, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionAppletBestEffortPersistent, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionSystemSharable, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApp.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionAppletBestEffortPersistent.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionSystemSharable.GetResult());
    }

    // 「アプリと BestEffortPersistent アプレット と Continuous アプレット」と BG は共存不可（BG が却下）
    {
        nn::nifm::NetworkConnection networkConnectionApp;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApp.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffort));

        nn::nifm::NetworkConnection networkConnectionAppletBestEffortPersistent;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionAppletBestEffortPersistent.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffortForApplet));

        nn::nifm::NetworkConnection networkConnectionAppletContinuous;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionAppletContinuous.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForAppletContinuous));

        nn::nifm::NetworkConnection networkConnectionSystemPersistent;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionSystemPersistent.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessPersistent));


        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApp, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionAppletBestEffortPersistent, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionAppletContinuous, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        networkConnectionSystemPersistent.SubmitRequestAndWait();

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApp.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionAppletBestEffortPersistent.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionAppletContinuous.GetResult());
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionSystemPersistent.GetResult());

        networkConnectionSystemPersistent.GetSystemEvent().Clear();

        networkConnectionApp.CancelRequest();
        networkConnectionAppletBestEffortPersistent.CancelRequest();

        // アプリと BestEffortPersistent のキャンセル後は，BG が受理される
        ASSERT_TRUE(networkConnectionSystemPersistent.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionSystemPersistent.GetResult());
    }
} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, ConnectionStatus)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    nn::nifm::InternetConnectionStatus connectionStatus;

    // 切断時は情報を取得できない
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetInternetConnectionStatus(&connectionStatus));

    {
        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        // ローカル通信時は情報を取得できない
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNotInternetConnection, nn::nifm::GetInternetConnectionStatus(&connectionStatus));
    }

    {
        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&connectionStatus));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, connectionStatus.internetAvailability);
    }

    // 疎通確認中の状態取得
    {
        // imatake-wpa2aes
        const nn::nifm::WirelessSettingData wirelessSetting0 = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 } // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting0 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                8080,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile0 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true, // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting0
            },
            ipSetting0
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile0);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_Required));

        networkConnection.SubmitRequest();

        bool isCompleted = false;
        bool inProgress = false;
        // 疎通確認状態が変わるまで、最長で 50 秒待つ
        for (int i = 0; i < 1000; ++i)
        {
            nn::Result result = nn::nifm::GetInternetConnectionStatus(&connectionStatus);
            if (result.IsSuccess())
            {
                if (connectionStatus.internetAvailability == nn::nifm::InternetAvailability_InProgress)
                {
                    if (!inProgress)
                    {
                        inProgress = true;
                        NN_LOG("Checking internet connection .");
                    }
                    else
                    {
                        NN_LOG(".");
                    }
                }
                else if (connectionStatus.internetAvailability == nn::nifm::InternetAvailability_Confirmed)
                {
                    if (inProgress)
                    {
                        inProgress = false;
                        NN_LOG(" Done.\n");
                    }
                }
            }

            if (!networkConnection.IsRequestOnHold())
            {
                isCompleted = true;
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        }
        EXPECT_TRUE(isCompleted);
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&connectionStatus));
        EXPECT_EQ(nn::nifm::NetworkInterfaceType_Ieee80211, connectionStatus.networkInterfaceType);
        EXPECT_GT(connectionStatus.linkLevel, nn::nifm::LinkLevel_0);
        EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, connectionStatus.internetAvailability);
    }

    {
        // imatake-mixed
        const nn::nifm::WirelessSettingData wirelessSetting0 = {
            {  // ssidConfig
                {  // ssid
                    13,  // length
                    {0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6d,0x69,0x78,0x65,0x64} // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting0 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                8080,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile0 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true, // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting0
            },
            ipSetting0
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile0);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_Forced));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection.GetRequestState());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&connectionStatus));

        EXPECT_EQ(nn::nifm::NetworkInterfaceType_Ieee80211, connectionStatus.networkInterfaceType);
        EXPECT_GE(connectionStatus.linkLevel, nn::nifm::LinkLevel_0);
        EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, connectionStatus.internetAvailability);
    }
} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, SustainConnection)
{
    {
        nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
        NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

        nn::nifm::NetworkConnection networkConnection1;

        // imatake-wpa2aes
        nn::util::Uuid id_wpa2aes;
        id_wpa2aes.FromString("93188440-b6d5-14b7-056b-08cd57910960");

        nn::nifm::InternetConnectionStatus connectionStatus;
        nn::nifm::NetworkProfileData networkProfileData;

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection1.GetRequestHandle(), id_wpa2aes));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection1.GetRequestState());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection1.GetResult());

        // 通常の利用要求は、取り下げ後に接続が切れる
        networkConnection1.CancelRequest();
        EXPECT_FALSE(networkConnection1.IsAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnection1.GetResult());

        // 状態が変わるまで、最長で30秒待つ（切断される）
        NN_LOG("Waiting disconnect ");
        nn::Result connectionResult = nn::ResultSuccess();
        for (int i = 0; i < 30; ++i)
        {
            NN_LOG(".");
            connectionResult = nn::nifm::GetInternetConnectionStatus(&connectionStatus);
            if (connectionResult.IsFailure())
            {
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        }
        NN_LOG("Done.\n");
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNotConnected, connectionResult);

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSustainable(networkConnection1.GetRequestHandle(), true));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        // 接続が維持される
        networkConnection1.CancelRequest();
        EXPECT_FALSE(networkConnection1.IsAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnection1.GetResult());

        // 状態が変わるまで、最長で5秒待つ（切断されない）
        NN_LOG("Waiting disconnect ");
        for (int i = 0; i < 5; ++i)
        {
            NN_LOG(".");
            connectionResult = nn::nifm::GetInternetConnectionStatus(&connectionStatus);
            if (connectionResult.IsFailure())
            {
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        }
        NN_LOG("Done.\n");
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&connectionStatus));

        {
            // まだ、networkConnection2 の（内部複製）利用要求が残っているので，弱い優先度では失敗する

            nn::nifm::NetworkConnection networkConnection2;

            // ethernet
            nn::util::Uuid id_ethernet;
            id_ethernet.FromString("93188440-b6d5-14b7-056b-08cd57910964");

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection2.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));  // BG
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection2.GetRequestHandle(), id_ethernet));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), 160));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSustainable(networkConnection2.GetRequestHandle(), false));
            NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnection2.GetResult());
        }

        {
            // デフォルト設定の接続で相乗り

            nn::nifm::NetworkConnection networkConnection2;

            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

            EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection2.GetRequestState());
            NNT_EXPECT_RESULT_SUCCESS(networkConnection2.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&connectionStatus));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetCurrentNetworkProfile(&networkProfileData));
            ASSERT_EQ(id_wpa2aes, networkProfileData.id);
        }
    }

    {
        nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
        NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

        nn::nifm::NetworkConnection networkConnection;

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSustainable(networkConnection.GetRequestHandle(), true));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        // ローカル通信の利用要求では，継続維持できない
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInvalidRequirement, networkConnection.GetResult());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnection.GetRequestHandle(), true));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnection.GetRequestHandle(), false));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSustainable(networkConnection.GetRequestHandle(), true));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        // 独占利用要求では，継続維持できない
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInvalidRequirement, networkConnection.GetResult());
    }
} // NOLINT(impl/function_size)


TEST_F(ConnectCaseTest, RequirementPreset)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::util::Uuid id;
    id.FromString("93188440-b6d5-14b7-056b-08cd57910960"); // imatake-wpa2aes

    {
        // 最高優先度のアプリケーションプリセット利用要求
        nn::nifm::NetworkConnection networkConnectionApplication;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApplication.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionApplication.GetRequestHandle(), id));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestPriority(networkConnectionApplication.GetRequestHandle(), 0));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionApplication.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApplication, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApplication.GetResult());

        // 汎用アプレットプリセット利用要求（独占）
        nn::nifm::NetworkConnection networkConnectionApplet;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApplet.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForApplet));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionApplet.GetRequestHandle(), id));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestGreedy(networkConnectionApplet.GetRequestHandle(), true));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSharable(networkConnectionApplet.GetRequestHandle(), false));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionApplet.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApplet, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApplet.GetResult());

        // アプリが汎用アプレット（独占）に追い出される
        NNT_NIFM_WAIT_FOR_PROCESSING;
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionApplication.GetResult());

        {
            // imatake-wpa2aes
            const nn::nifm::WirelessSettingData wirelessSetting = {
                {  // ssidConfig
                    {  // ssid
                        15,  // length
                        { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                    },
                false  // nonBroadcast
                },
                {  // security
                    {  //authEncryption
                        nn::nifm::Authentication_Wpa2Psk,  // authentication
                        nn::nifm::Encryption_Aes  // encryption
                    },
                    {  // sharedKey
                        11,  // length
                        "Shi2iTaiZen"  // keyMaterial
                    }
                }
            };
            const nn::nifm::IpSettingData ipSetting = {
                {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    false,  // isEnabled
                    0,  // port
                    "",  // proxy
                    {  // authentication
                        false,  // isEnabled
                        "",  // username
                        ""  // password
                    }
                },
                1400  //mtu
            };

            nn::nifm::NetworkProfileData networkProfile = { nn::util::InvalidUuid,{}, nn::nifm::NetworkProfileType_Temporary, nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211, true, true,{ wirelessSetting }, ipSetting };

            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
            ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

            {
                // ネットワーク接続アプレット（テンポラリ接続 ID 指定により相乗り不可）
                nn::nifm::NetworkConnection networkConnectionAppletNc;
                NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionAppletNc.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForNetConnectApplet));
                NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionAppletNc.GetRequestHandle(), temporaryNetworkProfile.GetId()));
                NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionAppletNc.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
                EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionAppletNc, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

                NNT_ASSERT_RESULT_SUCCESS(networkConnectionAppletNc.GetResult());

                // ネットワーク接続アプレットに汎用アプレットが追い出される
                NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionApplet.GetResult());
            }

            // 状態が変わるまで、最長で30秒待つ（切断されない）
            nn::nifm::InternetConnectionStatus connectionStatus;
            nn::Result connectionResult = nn::ResultSuccess();
            NN_LOG("Waiting disconnect ");
            for (int i = 0; i < 30; ++i)
            {
                NN_LOG(".");
                connectionResult = nn::nifm::GetInternetConnectionStatus(&connectionStatus);
                if (connectionResult.IsFailure())
                {
                    break;
                }

                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
            }
            NN_LOG("Done.\n");
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&connectionStatus));
        }
    }

    {
        // アプリケーションのインターネット通信をアプレットのローカル通信が奪う

        // アプリケーションプリセット利用要求
        nn::nifm::NetworkConnection networkConnectionApplication;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApplication.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionApplication.GetRequestHandle(), id));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionApplication.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApplication, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_NIFM_WAIT_FOR_PROCESSING;
        EXPECT_TRUE(networkConnectionApplication.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApplication.GetResult());

        // 汎用アプレットプリセット利用要求
        nn::nifm::NetworkConnection networkConnectionApplet;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApplet.GetRequestHandle(), nn::nifm::RequirementPreset_LocalForApplet));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionApplet.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApplet, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApplet.GetResult());

        // アプリが汎用アプレットに追い出される
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, networkConnectionApplication.GetResult());
    }

    {
        // アプリケーションのローカル通信をアプレットのインターネット通信が奪う

        // アプリケーションプリセット利用要求
        nn::nifm::NetworkConnection networkConnectionApplication;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApplication.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionApplication.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApplication, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_NIFM_WAIT_FOR_PROCESSING;
        EXPECT_TRUE(networkConnectionApplication.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApplication.GetResult());

        // 汎用アプレットプリセット利用要求
        nn::nifm::NetworkConnection networkConnectionApplet;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApplet.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForApplet));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionApplet.GetRequestHandle(), id));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionApplet.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        networkConnectionApplet.SubmitRequest();

        // アプリケーションの要求がブロッキング状態に遷移したのを確認して取り下げ
        EXPECT_TRUE(networkConnectionApplication.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_EQ(nn::nifm::RequestState_Blocking, networkConnectionApplication.GetRequestState());
        networkConnectionApplication.CancelRequest();

        // アプリケーションの要求取り下げによってアプレットの要求が受理される
        EXPECT_TRUE(networkConnectionApplet.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApplet.GetResult());
    }

    {
        // アプリケーションのローカル通信をアプレットのローカル通信が奪う

        // アプリケーションプリセット利用要求
        nn::nifm::NetworkConnection networkConnectionApplication;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApplication.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionApplication.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApplication, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_NIFM_WAIT_FOR_PROCESSING;
        EXPECT_TRUE(networkConnectionApplication.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApplication.GetResult());

        // 汎用アプレットプリセット利用要求
        nn::nifm::NetworkConnection networkConnectionApplet;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApplet.GetRequestHandle(), nn::nifm::RequirementPreset_LocalForApplet));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionApplet.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        networkConnectionApplet.SubmitRequest();

        // アプリケーションの要求がブロッキング状態に遷移したのを確認して取り下げ
        EXPECT_TRUE(networkConnectionApplication.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_EQ(nn::nifm::RequestState_Blocking, networkConnectionApplication.GetRequestState());
        networkConnectionApplication.CancelRequest();

        // アプリケーションの要求取り下げによってアプレットの要求が受理される
        EXPECT_TRUE(networkConnectionApplet.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionApplet.GetResult());
    }
} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, RequirementPresetForSystemProcess)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::util::Uuid id;
    id.FromString("93188440-b6d5-14b7-056b-08cd57910960"); // imatake-wpa2aes

    // アプリケーションプリセット利用要求
    nn::nifm::NetworkConnection networkConnectionApplication;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApplication.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionApplication.GetRequestHandle(), id));
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionApplication.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApplication, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    // 汎用システムプロセス向けプリセット利用要求
    nn::nifm::NetworkConnection networkConnectionSystemProcess;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionSystemProcess.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionSystemProcess.GetRequestHandle(), id));
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionSystemProcess.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionSystemProcess, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    // TODO: 通知用のプリセットなども作ったらテスト
    // ...
}

TEST_F(ConnectCaseTest, CopyRequirement)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::nifm::Requirement requirement1 = {
        BasePriority - 1,                                   // _priority
        true,                                               // _isRejectable
        false,                                              // _isPersistent
        true,                                               // _isInstant
        false,                                              // _isSustainable
        false,                                              // _isGreedy
        true,                                               // _isSharable
        true,                                               // _isForeground
        true,                                               // _isAutomaticSwitchProhibited
        false,                                              // _isKeptInSleep
        true,                                               // _isInternalContextUpdatable
        nn::nifm::NetworkType_Infrastructure,               // _networkType
        nn::util::InvalidUuid,                              // _profileId
        nn::nifm::ConnectionConfirmationOption_Prohibited   // _connectionConfirmationOption
    };

    nn::nifm::NetworkConnection networkConnection1;
    nn::nifm::detail::RequestClient* pRequestClient1 = nn::nifm::GetRequestClientPointer(networkConnection1.GetRequestHandle());
    ASSERT_NE(nullptr, pRequestClient1);
    NNT_ASSERT_RESULT_SUCCESS(pRequestClient1->SetRequirement(requirement1));

    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

    nn::nifm::NetworkConnection networkConnection2;

    nn::nifm::detail::RequestClient* pRequestClient2 = nn::nifm::GetRequestClientPointer(networkConnection2.GetRequestHandle());
    ASSERT_NE(nullptr, pRequestClient2);

    uint32_t submitId;
    NNT_ASSERT_RESULT_SUCCESS(pRequestClient1->GetSubmitId(&submitId));

    NNT_ASSERT_RESULT_SUCCESS(pRequestClient2->SetRequirementBySubmitId(submitId));

    nn::nifm::Requirement requirement2;
    NNT_ASSERT_RESULT_SUCCESS(pRequestClient2->GetRequirement(&requirement2));

    ASSERT_EQ(requirement1._priority, requirement2._priority);
    ASSERT_EQ(requirement1._isRejectable, requirement2._isRejectable);
    ASSERT_EQ(requirement1._isPersistent, requirement2._isPersistent);
    ASSERT_EQ(requirement1._isInstant, requirement2._isInstant);
    ASSERT_EQ(requirement1._isSustainable, requirement2._isSustainable);
    ASSERT_EQ(requirement1._isGreedy, requirement2._isGreedy);
    ASSERT_EQ(requirement1._isSharable, requirement2._isSharable);
    ASSERT_EQ(requirement1._networkType, requirement2._networkType);
    ASSERT_EQ(requirement1._profileId, requirement2._profileId);
    ASSERT_EQ(requirement1._connectionConfirmationOption, requirement2._connectionConfirmationOption);

    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection2.GetResult());

    networkConnection1.CancelRequest();
    ASSERT_EQ(nn::nifm::RequestState_Free, networkConnection1.GetRequestState());

    networkConnection2.CancelRequest();
    ASSERT_EQ(nn::nifm::RequestState_Free, networkConnection2.GetRequestState());

    // 未提出状態の利用要求からもコピー可能
    NNT_ASSERT_RESULT_SUCCESS(pRequestClient2->SetRequirementBySubmitId(submitId));
}

// openWep* で疎通確認ができない CI 環境では動作しない
#if 0
TEST_F(ConnectCaseTest, NetworkProfileSpecified)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeSystem, nn::nifm::FinalizeSystemForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::nifm::NetworkConnection networkConnection;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_Forced));

    nn::util::Uuid id;
    id.FromString("93188440-b6d5-14b7-056b-08cd57910965"); // OpenWep40_2 (hexadecimal format key)
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), id));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());

    networkConnection.CancelRequest();

    id.FromString("93188440-b6d5-14b7-056b-08cd57910966"); // OpenWep104 (hexadecimal format key)
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), id));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());
}
#endif

// sharedWep が存在しない CI 環境では動作しない
#if 0
TEST_F(ConnectCaseTest, SharedWep)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::nifm::NetworkConnection networkConnection;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_Forced));

    nn::util::Uuid id;
    id.FromString("93188440-b6d5-14b7-056b-08cd57910969");
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), id));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    nn::nifm::AccessPointData accessPointData;
    nn::nifm::GetCurrentAccessPoint(&accessPointData);

    ASSERT_EQ(0, std::memcmp("imatake-sharedwep104", accessPointData.ssid.hex, accessPointData.ssid.length));
    ASSERT_EQ(nn::nifm::Authentication_Shared, accessPointData.authentication);
    ASSERT_EQ(nn::nifm::Encryption_Wep, accessPointData.encryption);

}
#endif

// 特定の DNS を設定して疎通確認時のリダイレクトを検証
// WISPrAuthTest1/2 成功の直後（1分以内）に動作すると失敗するので注意
TEST_F(ConnectCaseTest, WifiWebAuthTest)
{
    // WISPrAuthTest1 から 60 秒経過することを保証
    while (!IsLongerThan(60))
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    }

    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    // imatake-wpa2aes
    const nn::nifm::WirelessSettingData wirelessSetting = {
        {  // ssidConfig
            {  // ssid
                15,  // length
                { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
            },
        false  // nonBroadcast
        },
        {  // security
            {  //authEncryption
                nn::nifm::Authentication_Wpa2Psk,  // authentication
                nn::nifm::Encryption_Aes  // encryption
            },
            {  // sharedKey
                11,  // length
                "Shi2iTaiZen"  // keyMaterial
            }
        }
    };
    const nn::nifm::IpSettingData ipSetting = {
        {  // ip
            true,  // isAuto
            {},  // ipAddress
            {},  // subnetMask
            {}  // defaultGateway
        },
        {  // dns
            false,  // isAuto
            { { 52, 68, 203, 240 } },  // preferredDns
            {}  // alternateDns
        },
        {  // proxy
            false,  // isEnabled
            0,  // port
            "",  // proxy
            {  // authentication
                false,  // isEnabled
                "",  // username
                ""  // password
            }
        },
        1400  //mtu
    };
    const nn::nifm::NetworkProfileData networkProfile = {
        nn::util::InvalidUuid,  // id
        {},  // name
        nn::nifm::NetworkProfileType_Temporary, // networkProfileType
        nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
        true,  // isAutoConnect
        true, // isLargeCapacity
        {
            wirelessSetting
        },
        ipSetting
    };

    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
    ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

    // 新規接続で Hotspot 認証（実際に認証はしない）
    {
        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_Forced));

        // 他の実機テストと WISPr 認証のタイミングが重なった（1分以内）場合、3回までは再試行
        for(int i = 0; i < 3; ++i)
        {
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

            if (networkConnection.GetResult().IsSuccess())
            {
                networkConnection.CancelRequest();
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(60));
                continue;
            }

            break;
        }

        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultHotspotAuthenticationViaWebAuthAppletNeeded, networkConnection.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(ExpectAdditionalInfo(temporaryNetworkProfile.GetId(), "https://ssltest3.app.nintendowifi.net/login", networkConnection.GetRequestHandle()));

        uint32_t submitId0;
        nn::nifm::GetRequestClientPointer(networkConnection.GetRequestHandle())->GetSubmitId(&submitId0);

        {
            nn::nifm::AdditionalInfo additionalInfo;
            uint32_t submitId;
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetRequestAdditionalInfo(&additionalInfo, &submitId, networkConnection.GetRequestHandle()));

            char idStr[nn::util::Uuid::StringSize];
            NN_LOG("AdditionalInfo: { %s, %s }, Revision: %d\n", additionalInfo.redirectUrl, additionalInfo.profileId.ToString(idStr, nn::util::Uuid::StringSize), submitId);
            ASSERT_EQ(0, std::memcmp(additionalInfo.redirectUrl, "https://ssltest3.app.nintendowifi.net/login", sizeof("https://ssltest3.app.nintendowifi.net/login")));
            ASSERT_EQ(temporaryNetworkProfile.GetId(), additionalInfo.profileId);
            ASSERT_EQ(submitId0, submitId);
        }

        // TODO:
        // ...
    }

    // 相乗りで Hotspot 認証
    {
        nn::nifm::NetworkConnection networkConnection1;
        ASSERT_NE(nullptr, nn::nifm::GetRequestClientPointer(networkConnection1.GetRequestHandle()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection1.GetRequestHandle(), BasePriority));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection1.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

        nn::nifm::NetworkConnection networkConnection2;
        ASSERT_NE(nullptr, nn::nifm::GetRequestClientPointer(networkConnection2.GetRequestHandle()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), BasePriority - 10));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection2.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_Forced));

        // 他の実機テストと WISPr 認証のタイミングが重なった（1分以内）場合、3回までは再試行
        for (int i = 0; i < 3; ++i)
        {
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

            if (networkConnection2.GetResult().IsSuccess())
            {
                networkConnection2.CancelRequest();
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(60));
                continue;
            }

            break;
        }

        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultHotspotAuthenticationViaWebAuthAppletNeeded, networkConnection2.GetResult());

        {
            auto pRequestClient = nn::nifm::GetRequestClientPointer(networkConnection2.GetRequestHandle());
            ASSERT_NE(nullptr, pRequestClient);

            uint32_t submitId0;
            pRequestClient->GetSubmitId(&submitId0);

            nn::nifm::AdditionalInfo additionalInfo;
            uint32_t submitId;
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetRequestAdditionalInfo(&additionalInfo, &submitId, networkConnection2.GetRequestHandle()));

            char idStr[nn::util::Uuid::StringSize];
            NN_LOG("AdditionalInfo: { %s, %s }, Revision: %d\n", additionalInfo.redirectUrl, additionalInfo.profileId.ToString(idStr, nn::util::Uuid::StringSize), submitId);
            ASSERT_EQ(0, std::memcmp(additionalInfo.redirectUrl, "https://ssltest3.app.nintendowifi.net/login", sizeof("https://ssltest3.app.nintendowifi.net/login")));
            ASSERT_EQ(temporaryNetworkProfile.GetId(), additionalInfo.profileId);
            ASSERT_EQ(submitId0, submitId);
        }
        // TODO:
        // ...
    }
} // NOLINT(impl/function_size)


// 特定の DNS を設定して疎通確認時のリダイレクトを検証2（1 から 60s 以上経過している必要がある）
TEST_F(ConnectCaseTest, WISPrAuthTest2)
{
    // WifiWebAuthTest により WISPrAuthTest1 から 60 秒以上経過していることは保証される

    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::nifm::Scan();

    const nn::nifm::IpSettingData ipSetting = {
        {  // ip
            true,  // isAuto
            {},  // ipAddress
            {},  // subnetMask
            {}  // defaultGateway
        },
        {  // dns
            false,  // isAuto
            { { 52, 68, 203, 240 } },  // preferredDns
            {}  // alternateDns
        },
        {  // proxy
            false,  // isEnabled
            0,  // port
            "",  // proxy
            {  // authentication
                false,  // isEnabled
                "",  // username
                ""  // password
            }
        },
        1400  //mtu
    };

    nn::util::Uuid id;
    id.FromString("1a765924-cfb0-4833-a9da-aa81d83aea4a"); // mmt

    nn::nifm::IpSettingData defaultIpSetting;
    nn::nifm::GetDefaultIpSetting(&defaultIpSetting);

    nn::nifm::SetDefaultIpSetting(ipSetting);

    // 相乗りで WISPr 認証
    {
        nn::nifm::NetworkConnection networkConnection1;
        ASSERT_NE(nullptr, nn::nifm::GetRequestClientPointer(networkConnection1.GetRequestHandle()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection1.GetRequestHandle(), BasePriority));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection1.GetRequestHandle(), id));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

        nn::nifm::NetworkProfileData networkProfileData;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetCurrentNetworkProfile(&networkProfileData));
        EXPECT_EQ(id, networkProfileData.id);

        nn::nifm::NetworkConnection networkConnection2;
        ASSERT_NE(nullptr, nn::nifm::GetRequestClientPointer(networkConnection2.GetRequestHandle()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), BasePriority - 10));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection2.GetRequestHandle(), id));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_Forced));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_SUCCESS(networkConnection2.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetCurrentNetworkProfile(&networkProfileData));
        EXPECT_EQ(id, networkProfileData.id);

        // TODO:
        // ...
    }

    nn::nifm::SetDefaultIpSetting(defaultIpSetting);
}

TEST_F(ConnectCaseTest, GetCurrentAccessPoint)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    {
        // imatake-opennone-stealth
        const nn::nifm::WirelessSettingData wirelessSetting0 = {
            {  // ssidConfig
                {  // ssid
                    24,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x6e,0x6f,0x6e,0x65,0x2d,0x73,0x74,0x65,0x61,0x6c,0x74,0x68 }  // hex
                },
            true  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Open,  // authentication
                    nn::nifm::Encryption_None  // encryption
                },
                {  // sharedKey
                    0,  // length
                    ""  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting0 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile0 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true, // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting0
            },
            ipSetting0
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile0);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        nn::nifm::AccessPointData accessPointData = {};
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetCurrentAccessPoint(&accessPointData));

        uint16_t channel;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetCurrentAccessPointChannel(&channel));
        ASSERT_EQ(accessPointData.channel, channel);

        nn::nifm::test::PrintAccessPoint(accessPointData);

        ASSERT_EQ(wirelessSetting0.security.authEncryption.authentication, accessPointData.authentication);
        ASSERT_EQ(wirelessSetting0.security.authEncryption.encryption, accessPointData.encryption);
        //ASSERT_EQ(wirelessSetting0.ssidConfig.ssid.length, accessPointData.ssid.length);
        //ASSERT_EQ(0, memcmp(wirelessSetting0.ssidConfig.ssid.hex, accessPointData.ssid.hex, accessPointData.ssid.length));

        nn::nifm::MacAddress bssidZero = {};
        ASSERT_NE(bssidZero, accessPointData.bssid);
    }

    {
        // imatake-mixed
        const nn::nifm::WirelessSettingData wirelessSetting1 = {
            {  // ssidConfig
                {  // ssid
                    13,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6d,0x69,0x78,0x65,0x64 }  // hex
                },
            true  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting1 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile1 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true, // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting1
            },
            ipSetting1
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        nn::nifm::AccessPointData accessPointData = {};
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetCurrentAccessPoint(&accessPointData));

        uint16_t channel;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetCurrentAccessPointChannel(&channel));
        ASSERT_EQ(accessPointData.channel, channel);

        nn::nifm::test::PrintAccessPoint(accessPointData);

        ASSERT_EQ(nn::nifm::Authentication_Wpa2Psk, accessPointData.authentication);
        ASSERT_EQ(nn::nifm::Encryption_Aes, accessPointData.encryption);
        ASSERT_EQ(wirelessSetting1.ssidConfig.ssid.length, accessPointData.ssid.length);
        ASSERT_EQ(0, memcmp(wirelessSetting1.ssidConfig.ssid.hex, accessPointData.ssid.hex, accessPointData.ssid.length));

        nn::nifm::MacAddress bssidZero = {};
        ASSERT_NE(bssidZero, accessPointData.bssid);
    }
} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, ConnectionCancel)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // imatake-openwep40（パスワード間違い）
    nn::nifm::WirelessSettingData wirelessSetting0 = {
        {  // ssidConfig
            {  // ssid
                17,  // length
                { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x77,0x65,0x70,0x34,0x30 }  // hex
            },
        true  // nonBroadcast
        },
        {  // security
            {  //authEncryption
                nn::nifm::Authentication_Open,  // authentication
                nn::nifm::Encryption_Wep  // encryption
            },
            {  // sharedKey
                5,  // length
                "Wrong"  // keyMaterial
            }
        }
    };
    nn::nifm::IpSettingData ipSetting0 = {
        {  // ip
            true,  // isAuto
            {},  // ipAddress
            {},  // subnetMask
            {}  // defaultGateway
        },
        {  // dns
            true,  // isAuto
            {},  // preferredDns
            {}  // alternateDns
        },
        {  // proxy
            false,  // isEnabled
            0,  // port
            "",  // proxy
            {  // authentication
                false,  // isEnabled
                "",  // username
                ""  // password
            }
        },
        1400  //mtu
    };
    nn::nifm::NetworkProfileData networkProfile0 = {
        nn::util::InvalidUuid,  // id
        {},  // name
        nn::nifm::NetworkProfileType_Temporary, // networkProfileType
        nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
        true, // isAutoConnect
        true, // isLargeCapacity
        {
            wirelessSetting0
        },
        ipSetting0
    };

    // 現在の環境で，imatake-openwep40（パスワード間違い）の失敗にかかる時間の計測
    {
        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfileWirelwess(networkProfile0);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfileWirelwess.GetId());

        nn::nifm::NetworkConnection networkConnectionWireless;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionWireless.GetRequestHandle(), temporaryNetworkProfileWirelwess.GetId()));

        int64_t startTick = nn::os::GetSystemTick().GetInt64Value();

        networkConnectionWireless.SubmitRequestAndWait();

        ASSERT_TRUE(networkConnectionWireless.GetResult().IsFailure());

        int64_t t = (nn::os::GetSystemTick().GetInt64Value() - startTick) / nn::os::GetSystemTickFrequency();
        int s = static_cast<int>(t % 60);
        int m = static_cast<int>((t / 60) % 60);
        int h = static_cast<int>(t / (60 * 60));
        NN_LOG("Time to fail: %02d:%02d:%02d\n", h, m, s);
    }

    bool isCompletedOnTime = false;

    // 接続試行に時間のかかる利用要求の取り下げ後に提出された要求が評価されるケース
    for (int i = 0; i < 3; ++i)
    {
        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfileWirelwess(networkProfile0);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfileWirelwess.GetId());

        nn::nifm::NetworkConnection networkConnectionWireless;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionWireless.GetRequestHandle(), temporaryNetworkProfileWirelwess.GetId()));

        // WEP パスワード間違いでステルス許可，疎通確認ありなので接続試行に長時間かかる
        networkConnectionWireless.SubmitRequest();

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(nn::nifm::test::Random() % 15));

        networkConnectionWireless.CancelRequest();

        // 有線設定で接続
        nn::nifm::NetworkConnection networkConnectionEthernet;
        nn::util::Uuid id;
        id.FromString("93188440-b6d5-14b7-056b-08cd57910964");
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionEthernet.GetRequestHandle(), id));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestPriority(networkConnectionEthernet.GetRequestHandle(), 14));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionEthernet.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        int64_t startTick = nn::os::GetSystemTick().GetInt64Value();

        networkConnectionEthernet.SubmitRequestAndWait();

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionEthernet.GetResult());

        int64_t t = (nn::os::GetSystemTick().GetInt64Value() - startTick) / nn::os::GetSystemTickFrequency();
        int s = static_cast<int>(t % 60);
        int m = static_cast<int>((t / 60) % 60);
        int h = static_cast<int>(t / (60 * 60));
        NN_LOG("Time to connect: %02d:%02d:%02d\n", h, m, s);

        NN_UTIL_SCOPE_EXIT
        {
            ASSERT_TRUE(nn::nifm::test::Disconnect());
            EXPECT_TRUE(isCompletedOnTime);
        };

        // 経験的に 8 秒以内にはつながるはず（安全率 50%）
        if (s <= 12 && m == 0 && h == 0)
        {
            isCompletedOnTime = true;
            break;
        }
    }

    isCompletedOnTime = false;
    // 接続試行に時間のかかる利用要求の取り下げ後に，以前提出された Persistent な要求が評価されるケース
    for (int i = 0; i < 3; ++i)
    {
        // imatake-wpa2aes
        nn::nifm::NetworkConnection networkConnectionWireless1;
        nn::util::Uuid id;
        id.FromString("93188440-b6d5-14b7-056b-08cd57910960");
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionWireless1.GetRequestHandle(), id));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestPersistent(networkConnectionWireless1.GetRequestHandle(), true));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionWireless1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        networkConnectionWireless1.SubmitRequest();

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfileWirelwess(networkProfile0);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfileWirelwess.GetId());

        nn::nifm::NetworkConnection networkConnectionWireless2;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionWireless2.GetRequestHandle(), temporaryNetworkProfileWirelwess.GetId()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestPriority(networkConnectionWireless2.GetRequestHandle(), 14));

        // WEP パスワード間違いでステルス許可，疎通確認ありなので接続試行に長時間かかる
        networkConnectionWireless2.SubmitRequest();

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(nn::nifm::test::Random() % 15));

        // 要求の取り下げで，即時次の評価が行われるはず
        networkConnectionWireless2.CancelRequest();

        int64_t startTick = nn::os::GetSystemTick().GetInt64Value();

        ASSERT_TRUE(nn::nifm::test::WaitRequestCompletion(&networkConnectionWireless1, 60));

        int64_t t = (nn::os::GetSystemTick().GetInt64Value() - startTick) / nn::os::GetSystemTickFrequency();
        int s = static_cast<int>(t % 60);
        int m = static_cast<int>((t / 60) % 60);
        int h = static_cast<int>(t / (60 * 60));
        NN_LOG("Time to connect: %02d:%02d:%02d\n", h, m, s);

        NN_UTIL_SCOPE_EXIT
        {
            ASSERT_TRUE(nn::nifm::test::Disconnect());
            EXPECT_TRUE(isCompletedOnTime);
        };

        // 経験的に 12 秒以内にはつながるはず（安全率 50%）
        if (s <= 18 && m == 0 && h == 0)
        {
            isCompletedOnTime = true;
            break;
        }
    }
} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, PersistTemporaryNetworkProfile)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());


    nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[nn::nifm::UserNetworkProfileCountMax];
    int outCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, nn::nifm::UserNetworkProfileCountMax, nn::nifm::NetworkProfileType_User));
    NN_LOG("outCount=%d\n", outCount);

    nn::util::Uuid id;
    nn::nifm::NetworkConnection networkConnection;

    {
        // imatake-wpa2aes
        const nn::nifm::WirelessSettingData wirelessSetting1 = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting1 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile1 = {
            nn::util::InvalidUuid,  // id
            "FromTemporary",  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting1
            },
            ipSetting1
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());
        id = temporaryNetworkProfile.GetId();

        // NetConnect アプレット向けプリセット，テンポラリ接続設定 ID をセット
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForNetConnectApplet));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), id));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        // 接続成功
        NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());

        // テンポラリ接続設定を永続化
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::PersistTemporaryNetworkProfile(temporaryNetworkProfile.GetHandle()));

        // テンポラリ接続設定 ID は更新される
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());
        ASSERT_NE(id, temporaryNetworkProfile.GetId());

        // ユーザ接続設定の数は一つ増加
        int outCount2 = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount2, nn::nifm::UserNetworkProfileCountMax, nn::nifm::NetworkProfileType_User));
        ASSERT_EQ(outCount + 1, outCount2);
    }

    // テンポラリ接続設定は削除される

    // 状態が変わるまで、最長で30秒待つ（切断されない）
    nn::nifm::InternetConnectionStatus connectionStatus;
    NN_LOG("Waiting disconnect ");
    for (int i = 0; i < 30; ++i)
    {
        NN_LOG(".");
        nn::Result result = nn::nifm::GetInternetConnectionStatus(&connectionStatus);
        if (result.IsFailure())
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    }
    NN_LOG("Done.\n");

    nn::nifm::NetworkProfileData networkProfileData;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetCurrentNetworkProfile(&networkProfileData));

    // 永続化された設定で接続されている
    ASSERT_EQ(id, networkProfileData.id);
} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, Hex64PMK)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    {
        // imatake-wpa2aes
        const nn::nifm::WirelessSettingData wirelessSetting1 = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    64,  // length
                    {
                        0x62,0x61,0x64,0x38,0x38,0x34,0x30,0x62,0x61,0x31,
                        0x38,0x37,0x31,0x61,0x65,0x37,0x66,0x37,0x36,0x66,
                        0x32,0x65,0x66,0x39,0x33,0x33,0x35,0x31,0x61,0x62,
                        0x36,0x66,0x35,0x30,0x65,0x64,0x65,0x33,0x35,0x63,
                        0x66,0x62,0x63,0x39,0x39,0x35,0x64,0x39,0x63,0x33,
                        0x37,0x37,0x32,0x64,0x61,0x32,0x31,0x63,0x61,0x63,
                        0x65,0x65,0x39,0x35
                    }  // keyMaterial bad8840ba1871ae7f76f2ef93351ab6f50ede35cfbc995d9c3772da21cacee95
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting1 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile1 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting1
            },
            ipSetting1
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);
        nn::util::Uuid id1 = temporaryNetworkProfile.GetId();
        ASSERT_NE(nn::util::InvalidUuid, id1);


        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), id1));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::PersistTemporaryNetworkProfile(temporaryNetworkProfile.GetHandle()));

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::RemoveNetworkProfile(id1));
    }

    {
        // imatake-mixed
        const nn::nifm::WirelessSettingData wirelessSetting1 = {
            {  // ssidConfig
                {  // ssid
                    13,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6d,0x69,0x78,0x65,0x64 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    64,  // length
                    {
                        0x65,0x38,0x66,0x62,0x39,0x30,0x62,0x36,0x39,0x65,
                        0x36,0x34,0x64,0x35,0x66,0x36,0x36,0x32,0x33,0x63,
                        0x62,0x38,0x61,0x31,0x32,0x31,0x35,0x39,0x62,0x34,
                        0x36,0x65,0x66,0x39,0x35,0x34,0x62,0x63,0x65,0x39,
                        0x63,0x39,0x65,0x36,0x34,0x35,0x62,0x66,0x63,0x33,
                        0x36,0x63,0x65,0x63,0x35,0x36,0x31,0x31,0x36,0x61,
                        0x32,0x66,0x36,0x61
                    }  // keyMaterial e8fb90b69e64d5f6623cb8a12159b46ef954bce9c9e645bfc36cec56116a2f6a
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting1 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile1 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting1
            },
            ipSetting1
        };

        nn::util::Uuid networkProfileId;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetNetworkProfile(&networkProfileId, networkProfile1));

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), networkProfileId));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::RemoveNetworkProfile(networkProfileId));
    }
} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, UpdateAggregatedRequest)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    {
        // non-exist-ap
        const nn::nifm::WirelessSettingData wirelessSetting1 = {
            {  // ssidConfig
                {  // ssid
                    12,  // length
                    { 0x6e,0x6f,0x74,0x2d,0x65,0x78,0x69,0x73,0x74,0x2d,0x61,0x70 } // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    8,  // length
                    "password"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting1 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile1 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting1
            },
            ipSetting1
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);
        nn::util::Uuid id = temporaryNetworkProfile.GetId();
        ASSERT_NE(nn::util::InvalidUuid, id);

        // 通常のインターネット通信利用要求（存在しないAP）
        nn::nifm::NetworkConnection internetNetworkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(internetNetworkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(internetNetworkConnection.GetRequestHandle(), 110));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestPersistent(internetNetworkConnection.GetRequestHandle(), true));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestInstant(internetNetworkConnection.GetRequestHandle(), false));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(internetNetworkConnection.GetRequestHandle(), id));
        internetNetworkConnection.SubmitRequest();

        // 低優先度のローカル通信利用要求（ただし non-blocking）
        nn::nifm::NetworkConnection localNetworkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(localNetworkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(localNetworkConnection.GetRequestHandle(), 120));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestPersistent(localNetworkConnection.GetRequestHandle(), true));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRejectable(localNetworkConnection.GetRequestHandle(), true));
        localNetworkConnection.SubmitRequest();

        EXPECT_TRUE(internetNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, internetNetworkConnection.GetResult());

        EXPECT_TRUE(localNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_EXPECT_RESULT_SUCCESS(localNetworkConnection.GetResult());

        // 無線 OFF
        nn::nifm::SetWirelessCommunicationEnabled(false);
        EXPECT_FALSE(nn::nifm::IsWirelessCommunicationEnabled());

        EXPECT_TRUE(internetNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_TRUE(localNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));

        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, internetNetworkConnection.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceNoLongerAvailable, localNetworkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_OnHold, localNetworkConnection.GetRequestState());

        // 無線 ON
        nn::nifm::SetWirelessCommunicationEnabled(true);
        EXPECT_TRUE(nn::nifm::IsWirelessCommunicationEnabled());

        EXPECT_TRUE(internetNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_TRUE(localNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));

        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, internetNetworkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(localNetworkConnection.GetResult());

        internetNetworkConnection.CancelRequest();

        // 通常のインターネット通信利用要求
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestPersistent(internetNetworkConnection.GetRequestHandle(), false));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestInstant(internetNetworkConnection.GetRequestHandle(), true));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(internetNetworkConnection.GetRequestHandle(), nn::util::InvalidUuid));
        internetNetworkConnection.SubmitRequestAndWait();

        // インターネット通信利用要求が通って、ローカル通信利用要求が却下
        EXPECT_TRUE(internetNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(30)));
        EXPECT_TRUE(localNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(30)));
        NNT_EXPECT_RESULT_SUCCESS(internetNetworkConnection.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLowPriority, localNetworkConnection.GetResult());

        internetNetworkConnection.CancelRequest();
    }

    {
        // imatake-wpa2aes
        const nn::nifm::WirelessSettingData wirelessSetting1 = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting1 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile1 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting1
            },
            ipSetting1
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);
        nn::util::Uuid id = temporaryNetworkProfile.GetId();
        ASSERT_NE(nn::util::InvalidUuid, id);

        // 通常のインターネット通信利用要求
        nn::nifm::NetworkConnection internetNetworkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(internetNetworkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(internetNetworkConnection.GetRequestHandle(), 130));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestPersistent(internetNetworkConnection.GetRequestHandle(), true));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(internetNetworkConnection.GetRequestHandle(), id));
        internetNetworkConnection.SubmitRequest();

        // 低優先度のローカル通信利用要求（blocking）
        nn::nifm::NetworkConnection localNetworkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(localNetworkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(localNetworkConnection.GetRequestHandle(), 140));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestPersistent(localNetworkConnection.GetRequestHandle(), true));
        localNetworkConnection.SubmitRequest();

        EXPECT_TRUE(internetNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_TRUE(localNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));

        NNT_EXPECT_RESULT_SUCCESS(internetNetworkConnection.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLowPriority, localNetworkConnection.GetResult());

        // 無線 OFF
        nn::nifm::SetWirelessCommunicationEnabled(false);
        EXPECT_FALSE(nn::nifm::IsWirelessCommunicationEnabled());

        EXPECT_TRUE(internetNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_TRUE(localNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));

        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceNoLongerAvailable, internetNetworkConnection.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceDisabled, localNetworkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_OnHold, localNetworkConnection.GetRequestState());

        // 無線 ON
        nn::nifm::SetWirelessCommunicationEnabled(true);
        EXPECT_TRUE(nn::nifm::IsWirelessCommunicationEnabled());

        EXPECT_TRUE(internetNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_TRUE(localNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));

        NNT_EXPECT_RESULT_SUCCESS(internetNetworkConnection.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLowPriority, localNetworkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_OnHold, localNetworkConnection.GetRequestState());

        // インターネット通信利用要求の取り下げ
        internetNetworkConnection.CancelRequest();

        // ローカル通信利用要求が受理
        EXPECT_TRUE(localNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_EXPECT_RESULT_SUCCESS(localNetworkConnection.GetResult());

        // インターネット通信利用要求提出
        internetNetworkConnection.SubmitRequest();

        EXPECT_TRUE(localNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLowPriority, localNetworkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Blocking, localNetworkConnection.GetRequestState());

        // ブロック中のローカル通信利用要求の取り下げ
        internetNetworkConnection.GetSystemEvent().Clear();
        localNetworkConnection.CancelRequest();

        // インターネット通信利用要求の受理
        EXPECT_TRUE(internetNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_EXPECT_RESULT_SUCCESS(internetNetworkConnection.GetResult());
    }
} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, NeighborDetection)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

#if 1
    {
        // 通常のBGインターネット通信利用要求
        nn::nifm::NetworkConnection internetNetworkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(internetNetworkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessPersistent));
        internetNetworkConnection.SubmitRequest();

        // 相乗り可能なBGインターネット通信利用要求
        nn::nifm::NetworkConnection internetNetworkConnectionNpns;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(internetNetworkConnectionNpns.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForNpns));
        internetNetworkConnectionNpns.SubmitRequest();

        // BGすれちがい通信利用要求
        nn::nifm::NetworkConnection neighborDetectionConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(neighborDetectionConnection.GetRequestHandle(), nn::nifm::RequirementPreset_NeighborDetectionForSystemProcess));
        neighborDetectionConnection.SubmitRequest();

        EXPECT_TRUE(internetNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_TRUE(internetNetworkConnectionNpns.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_TRUE(neighborDetectionConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));

        // BGインターネット通信優先
        NNT_ASSERT_RESULT_SUCCESS(internetNetworkConnection.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(internetNetworkConnectionNpns.GetResult());
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, neighborDetectionConnection.GetResult());

        // インターネット通信利用要求が取り下げられれば、すれちがい可能
        internetNetworkConnection.CancelRequest();
        internetNetworkConnectionNpns.CancelRequest();
        EXPECT_TRUE(neighborDetectionConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_ASSERT_RESULT_SUCCESS(neighborDetectionConnection.GetResult());
    }

    {
        // 通常のBGインターネット通信利用要求
        nn::nifm::NetworkConnection internetNetworkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(internetNetworkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessPersistent));
        internetNetworkConnection.SubmitRequest();

        // 相乗り可能なBGインターネット通信利用要求
        nn::nifm::NetworkConnection internetNetworkConnectionNpns;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(internetNetworkConnectionNpns.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForNpns));
        internetNetworkConnectionNpns.SubmitRequest();

        // アプリすれちがい通信利用要求
        nn::nifm::NetworkConnection neighborDetectionConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(neighborDetectionConnection.GetRequestHandle(), nn::nifm::RequirementPreset_NeighborDetection));
        neighborDetectionConnection.SubmitRequest();

        EXPECT_TRUE(internetNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_TRUE(internetNetworkConnectionNpns.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_TRUE(neighborDetectionConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));

        // アプリすれちがい優先
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, internetNetworkConnection.GetResult());
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, internetNetworkConnectionNpns.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(neighborDetectionConnection.GetResult());

        internetNetworkConnection.CancelRequest();
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(internetNetworkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffort));
        neighborDetectionConnection.GetSystemEvent().Clear();
        internetNetworkConnection.SubmitRequest();

        // 通常アプリインターネット通信利用要求が優先
        EXPECT_TRUE(neighborDetectionConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_EQ(nn::nifm::RequestState_Blocking, neighborDetectionConnection.GetRequestState());
        internetNetworkConnection.GetSystemEvent().Clear();
        neighborDetectionConnection.CancelRequest();

        EXPECT_TRUE(internetNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_ASSERT_RESULT_SUCCESS(internetNetworkConnection.GetResult());
    }

    {
        // BGすれちがい通信利用要求
        nn::nifm::NetworkConnection neighborDetectionConnection1;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(neighborDetectionConnection1.GetRequestHandle(), nn::nifm::RequirementPreset_NeighborDetectionForSystemProcess));
        neighborDetectionConnection1.SubmitRequest();

        // アプリすれちがい通信利用要求
        nn::nifm::NetworkConnection neighborDetectionConnection2;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(neighborDetectionConnection2.GetRequestHandle(), nn::nifm::RequirementPreset_NeighborDetection));
        neighborDetectionConnection2.SubmitRequest();

        EXPECT_TRUE(neighborDetectionConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_TRUE(neighborDetectionConnection2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));

        // すれちがい同士は共存可能
        NNT_ASSERT_RESULT_SUCCESS(neighborDetectionConnection1.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(neighborDetectionConnection2.GetResult());

        nn::nifm::NetworkConnection internetNetworkConnection;
        internetNetworkConnection.SubmitRequest();

        // 通常のインターネット通信とは共存できない
        EXPECT_TRUE(neighborDetectionConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_TRUE(neighborDetectionConnection2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_EQ(nn::nifm::RequestState_Blocking, neighborDetectionConnection1.GetRequestState());
        EXPECT_EQ(nn::nifm::RequestState_Blocking, neighborDetectionConnection2.GetRequestState());
        internetNetworkConnection.GetSystemEvent().Clear();
        neighborDetectionConnection1.CancelRequest();
        neighborDetectionConnection2.CancelRequest();

        EXPECT_TRUE(internetNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_ASSERT_RESULT_SUCCESS(internetNetworkConnection.GetResult());

        // アプレットすれちがい
        EXPECT_TRUE(nn::nifm::IsAnyForegroundRequestAccepted());
        nn::nifm::NetworkConnection neighborDetectionConnection3;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(neighborDetectionConnection3.GetRequestHandle(), nn::nifm::RequirementPreset_NeighborDetectionForApplet));
        neighborDetectionConnection3.SubmitRequest();

        // アプレットすれちがいは，通常のインターネット通信に優先する
        EXPECT_TRUE(neighborDetectionConnection3.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_ASSERT_RESULT_SUCCESS(neighborDetectionConnection3.GetResult());
    }

    {
        // BGすれちがい通信利用要求
        nn::nifm::NetworkConnection neighborDetectionConnection1;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(neighborDetectionConnection1.GetRequestHandle(), nn::nifm::RequirementPreset_NeighborDetectionForSystemProcess));
        neighborDetectionConnection1.SubmitRequest();

        // アプリすれちがい通信利用要求
        nn::nifm::NetworkConnection neighborDetectionConnection2;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(neighborDetectionConnection2.GetRequestHandle(), nn::nifm::RequirementPreset_NeighborDetection));
        neighborDetectionConnection2.SubmitRequest();

        EXPECT_TRUE(neighborDetectionConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_TRUE(neighborDetectionConnection2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));

        // すれちがい同士は共存可能
        NNT_ASSERT_RESULT_SUCCESS(neighborDetectionConnection1.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(neighborDetectionConnection2.GetResult());

        nn::nifm::NetworkConnection localNetworkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(localNetworkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));
        localNetworkConnection.SubmitRequest();

        // 通常のローカル通信とは共存できない
        EXPECT_TRUE(neighborDetectionConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_TRUE(neighborDetectionConnection2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_EQ(nn::nifm::RequestState_Blocking, neighborDetectionConnection1.GetRequestState());
        EXPECT_EQ(nn::nifm::RequestState_Blocking, neighborDetectionConnection2.GetRequestState());
        localNetworkConnection.GetSystemEvent().Clear();
        neighborDetectionConnection1.CancelRequest();
        neighborDetectionConnection2.CancelRequest();

        EXPECT_TRUE(localNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_ASSERT_RESULT_SUCCESS(localNetworkConnection.GetResult());

        // アプレットすれちがい
        EXPECT_TRUE(nn::nifm::IsAnyForegroundRequestAccepted());
        nn::nifm::NetworkConnection neighborDetectionConnection3;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(neighborDetectionConnection3.GetRequestHandle(), nn::nifm::RequirementPreset_NeighborDetectionForApplet));
        neighborDetectionConnection3.SubmitRequest();

        EXPECT_TRUE(localNetworkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_EQ(nn::nifm::RequestState_Blocking, localNetworkConnection.GetRequestState());
        localNetworkConnection.CancelRequest();

        // アプレットすれちがいは，通常のローカル通信に優先する
        EXPECT_TRUE(neighborDetectionConnection3.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_ASSERT_RESULT_SUCCESS(neighborDetectionConnection3.GetResult());
    }

    {
        // BGすれちがい通信利用要求
        nn::nifm::NetworkConnection neighborDetectionConnection1;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(neighborDetectionConnection1.GetRequestHandle(), nn::nifm::RequirementPreset_NeighborDetectionForSystemProcess));
        neighborDetectionConnection1.SubmitRequest();

        // アプリすれちがい通信利用要求
        nn::nifm::NetworkConnection neighborDetectionConnection2;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(neighborDetectionConnection2.GetRequestHandle(), nn::nifm::RequirementPreset_NeighborDetection));
        neighborDetectionConnection2.SubmitRequest();

        // アプレットすれちがい通信利用要求
        nn::nifm::NetworkConnection neighborDetectionConnection3;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(neighborDetectionConnection3.GetRequestHandle(), nn::nifm::RequirementPreset_NeighborDetectionForApplet));
        neighborDetectionConnection3.SubmitRequest();

        EXPECT_TRUE(neighborDetectionConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_TRUE(neighborDetectionConnection2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_TRUE(neighborDetectionConnection3.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));

        // すれちがい同士は共存可能
        NNT_ASSERT_RESULT_SUCCESS(neighborDetectionConnection1.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(neighborDetectionConnection2.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(neighborDetectionConnection3.GetResult());

        // 無線通信をオフに
        nn::nifm::SetWirelessCommunicationEnabled(false);
        EXPECT_FALSE(nn::nifm::IsWirelessCommunicationEnabled());

        // 無線オフでは Blocking に遷移
        EXPECT_TRUE(neighborDetectionConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_TRUE(neighborDetectionConnection2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_TRUE(neighborDetectionConnection3.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        EXPECT_EQ(nn::nifm::RequestState_Blocking, neighborDetectionConnection1.GetRequestState());
        EXPECT_EQ(nn::nifm::RequestState_Blocking, neighborDetectionConnection2.GetRequestState());
        EXPECT_EQ(nn::nifm::RequestState_Blocking, neighborDetectionConnection3.GetRequestState());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceNoLongerAvailable, neighborDetectionConnection1.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceNoLongerAvailable, neighborDetectionConnection2.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceNoLongerAvailable, neighborDetectionConnection3.GetResult());

        neighborDetectionConnection1.CancelRequest();
        neighborDetectionConnection2.CancelRequest();
        neighborDetectionConnection3.CancelRequest();

        nn::nifm::SetWirelessCommunicationEnabled(true);
        EXPECT_TRUE(nn::nifm::IsWirelessCommunicationEnabled());
    }

#else
    // 手動確認限定：（接続設定が有線のみである状態が必要）
    // - 有線で接続後，LANケーブルの挿抜によって、交互に要求が受理されることを確認する
    // - 有線未接続状態で， 60 s 間隔でスキャンが実行され，即座にすれちがい通信状態に戻ることを確認する
    {
        // 通常のBGインターネット通信利用要求
        nn::nifm::NetworkConnection internetNetworkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(internetNetworkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessPersistent));
        internetNetworkConnection.SubmitRequest();

        // BGすれちがい通信利用要求
        nn::nifm::NetworkConnection neighborDetectionConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(neighborDetectionConnection.GetRequestHandle(), nn::nifm::RequirementPreset_NeighborDetectionForSystemProcess));
        neighborDetectionConnection.SubmitRequest();

        // すれちがい通信を終了し，直後に要求を再提出するスレッド
        NN_ALIGNAS(nn::os::ThreadStackAlignment) static char stackOfThread[4 * 1024];
        ASSERT_EQ(0, reinterpret_cast<uintptr_t>(&stackOfThread) % nn::os::ThreadStackAlignment);

        nn::os::ThreadType CleanUpAndSubmitThread;
        auto pRequest = nn::nifm::GetRequestClientPointer(neighborDetectionConnection.GetRequestHandle());
        NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&CleanUpAndSubmitThread,
            [](void* p)
        {
            while (NN_STATIC_CONDITION(true))
            {
                CleanUpThreadFunc(p);
                reinterpret_cast<nn::nifm::detail::RequestClient*>(p)->SubmitAndWait();
            }
        },
            pRequest, &stackOfThread, 4 * 1024, 24));
        nn::os::StartThread(&CleanUpAndSubmitThread);

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(10000));
    }
#endif

} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, CapabilityUser)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::Initialize, nn::nifm::FinalizeForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    {
        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        // Requirement の設定は不可能
        nn::nifm::detail::RequestClient *pRequestClient = nn::nifm::GetRequestClientPointer(networkConnection.GetRequestHandle());
        ASSERT_NE(nullptr, pRequestClient);
        nn::nifm::Requirement requirement = {}; // 設定できないので任意の値で可
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, pRequestClient->SetRequirement(requirement));

        // Rejectable フラグの設定は不可能
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetRequestRejectable(networkConnection.GetRequestHandle(), false));
        // Sustainable フラグの設定は不可能
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetRequestSustainable(networkConnection.GetRequestHandle(), false));
        // 優先度指定は可能
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestPriority(networkConnection.GetRequestHandle(), 10));
        // 優先度直値指定は不可能
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetRequestRawPriority(networkConnection.GetRequestHandle(), 10));
        // 権限が必要なプリセットは利用不可
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForApplet));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForNetConnectApplet));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForWifiWebAuthApplet));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_LocalForApplet));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessPersistent));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessContinuous));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForDevMenu));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForNpns));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForAppletContinuous));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffortForApplet));
    }

    {
        nn::nifm::NetworkProfileData networkProfile;
        nn::util::Uuid id;

        // 接続設定の新規作成，変更は不可能
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetNetworkProfile(&id, networkProfile));
        // 接続設定の削除は不可能
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::RemoveNetworkProfile(id));

        const int bufferSize = nn::nifm::UserNetworkProfileCountMax + 1;
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[bufferSize];
        int outCount;

        //  SSID リストの取得は不可能
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize, nn::nifm::NetworkProfileType_SsidList));
        //  ユーザ接続設定の取得は可能
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize, nn::nifm::NetworkProfileType_User));
    }

    {
        nn::nifm::NetworkProfileData networkProfile;
        nn::util::Uuid id;

        id.FromString("93188440-b6d5-14b7-056b-08cd57910962"); // dummy-profile

        // ネットワーク接続設定の取得は可能
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetNetworkProfile(&networkProfile, id));
        // パスフレーズやプロキシパスワードは取得できない
        EXPECT_EQ(0, networkProfile.wirelessSetting.security.sharedKey.length);
        const char zero[nn::nifm::SharedKey::KeyMaterialSize] = {};
        EXPECT_EQ(0, memcmp(zero, networkProfile.wirelessSetting.security.sharedKey.keyMaterial, nn::nifm::SharedKey::KeyMaterialSize));
        EXPECT_EQ(0, strncmp("", networkProfile.ipSetting.proxy.authentication.password, nn::nifm::AuthenticationSetting::PasswordSize));
    }

    {
        // imatake-none
        const nn::nifm::WirelessSettingData wirelessSetting0 = {
            {  // ssidConfig
                {  // ssid
                    12,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6e,0x6f,0x6e,0x65 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Open,  // authentication
                    nn::nifm::Encryption_None  // encryption
                },
                {  // sharedKey
                    7,  // length
                    "NotUsed"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting0 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                true,  // isEnabled
                8080,  // port
                "dummy.proxy.jp",  // proxy
                {  // authentication
                    true,  // isEnabled
                    "dummy_user",  // username
                    "dummy_password"  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile0 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true, // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting0
            },
            ipSetting0
        };
        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile0);

        // テンポラリ接続設定の永続化は不可能
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::PersistTemporaryNetworkProfile(temporaryNetworkProfile.GetHandle()));
    }

    // API が権限なしではアサーションに失敗するので Release ビルドでのみ確認
#if defined(NN_SDK_BUILD_RELEASE)
    // 無線オンオフは不可能
    bool isWirelessCommunicationEnabled = nn::nifm::IsWirelessCommunicationEnabled();
    nn::nifm::SetWirelessCommunicationEnabled(!isWirelessCommunicationEnabled);
    EXPECT_EQ(isWirelessCommunicationEnabled, nn::nifm::IsWirelessCommunicationEnabled());
#endif
} // NOLINT(impl/function_size)

TEST_F(ConnectCaseTest, CapabilitySystem)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeSystem, nn::nifm::FinalizeSystemForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    {
        nn::nifm::NetworkConnection networkConnection;

        // Requirement の設定は不可能
        nn::nifm::detail::RequestClient *pRequestClient = nn::nifm::GetRequestClientPointer(networkConnection.GetRequestHandle());
        ASSERT_NE(nullptr, pRequestClient);
        nn::nifm::Requirement requirement = {}; // 設定できないので任意の値で可
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, pRequestClient->SetRequirement(requirement));
        // Rejectable フラグの設定は不可能
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetRequestRejectable(networkConnection.GetRequestHandle(), false));
        // Sustainable フラグの設定は可能
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestSustainable(networkConnection.GetRequestHandle(), false));
        // 優先度指定は可能
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestPriority(networkConnection.GetRequestHandle(), 10));
        // 優先度直値指定は可能
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection.GetRequestHandle(), 10));
        // 権限が必要なプリセットは利用可能
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForApplet));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForNetConnectApplet));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForWifiWebAuthApplet));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_LocalForApplet));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessPersistent));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessContinuous));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForDevMenu));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForNpns));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForAppletContinuous));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffortForApplet));
    }

    {
        nn::nifm::NetworkProfileData networkProfile;
        nn::util::Uuid id;

        // 接続設定の新規作成，変更は不可能
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::SetNetworkProfile(&id, networkProfile));
        // 接続設定の削除は不可能
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::RemoveNetworkProfile(id));

        const int bufferSize = nn::nifm::UserNetworkProfileCountMax + 1;
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[bufferSize];
        int outCount;

        //  SSID リストの取得は不可能
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize, nn::nifm::NetworkProfileType_SsidList));
        //  ユーザ接続設定の取得は可能
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize, nn::nifm::NetworkProfileType_User));
    }

    {
        nn::nifm::NetworkProfileData networkProfile;
        nn::util::Uuid id;

        id.FromString("93188440-b6d5-14b7-056b-08cd57910962"); // dummy-profile

        // ネットワーク接続設定の取得は可能
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetNetworkProfile(&networkProfile, id));
        // パスフレーズやプロキシパスワードは取得できない
        EXPECT_EQ(0, networkProfile.wirelessSetting.security.sharedKey.length);
        const char zero[nn::nifm::SharedKey::KeyMaterialSize] = {};
        EXPECT_EQ(0, memcmp(zero, networkProfile.wirelessSetting.security.sharedKey.keyMaterial, nn::nifm::SharedKey::KeyMaterialSize));
        EXPECT_EQ(0, strncmp("", networkProfile.ipSetting.proxy.authentication.password, nn::nifm::AuthenticationSetting::PasswordSize));
    }

    {
        // imatake-none
        const nn::nifm::WirelessSettingData wirelessSetting0 = {
            {  // ssidConfig
                {  // ssid
                    12,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6e,0x6f,0x6e,0x65 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Open,  // authentication
                    nn::nifm::Encryption_None  // encryption
                },
                {  // sharedKey
                    7,  // length
                    "NotUsed"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting0 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                true,  // isEnabled
                8080,  // port
                "dummy.proxy.jp",  // proxy
                {  // authentication
                    true,  // isEnabled
                    "dummy_user",  // username
                    "dummy_password"  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile0 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true, // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting0
            },
            ipSetting0
        };
        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile0);

        // テンポラリ接続設定の永続化は不可能
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNoCapability, nn::nifm::PersistTemporaryNetworkProfile(temporaryNetworkProfile.GetHandle()));
    }

    // API が権限なしではアサーションに失敗するので Release ビルドでのみ確認
#if defined(NN_SDK_BUILD_RELEASE)
    // 無線オンオフは可能
    bool isWirelessCommunicationEnabled = nn::nifm::IsWirelessCommunicationEnabled();
    nn::nifm::SetWirelessCommunicationEnabled(!isWirelessCommunicationEnabled);
    EXPECT_EQ(!isWirelessCommunicationEnabled, nn::nifm::IsWirelessCommunicationEnabled());
    nn::nifm::SetWirelessCommunicationEnabled(isWirelessCommunicationEnabled);  // 戻す
#endif
} // NOLINT(impl/function_size)

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
