﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>

#include <algorithm>
#include <cstring>

#include "../Common/nifm_TestUtility.h"

namespace
{
    const int64_t TIME_OUT_IN_SECONDS_FOR_SUCCESS = 60;

    void SetConnectionConfirmationOptionNotRequired()
    {
        NN_ASSERT(nn::nifm::InitializeAdmin().IsSuccess());

        nn::nifm::NetworkConnection* pNetworkConnection = nn::nifm::GetGlobalNetworkConnectionPointer();
        NN_ASSERT_NOT_NULL(pNetworkConnection);
        NN_ASSERT(nn::nifm::SetRequestConnectionConfirmationOption(pNetworkConnection->GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired).IsSuccess());
    }
}

class ClientManagementCaseTest : public ::testing::Test
{
protected:
    virtual void SetUp()
    {
        SetConnectionConfirmationOptionNotRequired();
    }
    virtual void TearDown()
    {
        nn::nifm::FinalizeAdminForTest();
    }
};



TEST_F(ClientManagementCaseTest, IsAnyInternetRequestAccepted)
{
/*
    nn::nifm::ClientId clientId0 = { ~nn::nifm::InvalidClientIdValue };
    clientId0 = nn::nifm::GetClientId();
    EXPECT_TRUE(clientId0.value == nn::nifm::InvalidClientIdValue);
*/
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    nn::nifm::NetworkConnection* pNetworkConnection = nn::nifm::GetGlobalNetworkConnectionPointer();

    nn::nifm::ClientId clientId1 = {};
    clientId1 = nn::nifm::GetClientId();
    EXPECT_NE(nn::nifm::InvalidClientIdValue, clientId1.value);

    EXPECT_FALSE(nn::nifm::IsAnyInternetRequestAccepted(clientId1));

    nn::nifm::SubmitNetworkRequestAndWait();
    NNT_EXPECT_RESULT_SUCCESS(pNetworkConnection->GetResult());
    EXPECT_TRUE(nn::nifm::IsNetworkAvailable());

    EXPECT_TRUE(nn::nifm::IsAnyInternetRequestAccepted(clientId1));

    nn::nifm::CancelNetworkRequest();

    EXPECT_FALSE(nn::nifm::IsAnyInternetRequestAccepted(clientId1));

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    SetConnectionConfirmationOptionNotRequired();

    nn::nifm::ClientId clientId2 = {};
    clientId2 = nn::nifm::GetClientId();
    EXPECT_NE(nn::nifm::InvalidClientIdValue, clientId2.value);

    EXPECT_NE(clientId1.value, clientId2.value);

    EXPECT_FALSE(nn::nifm::IsAnyInternetRequestAccepted(clientId1));
    EXPECT_FALSE(nn::nifm::IsAnyInternetRequestAccepted(clientId2));

    nn::nifm::SubmitNetworkRequestAndWait();
    NNT_EXPECT_RESULT_SUCCESS(pNetworkConnection->GetResult());
    EXPECT_TRUE(nn::nifm::IsNetworkAvailable());

    EXPECT_FALSE(nn::nifm::IsAnyInternetRequestAccepted(clientId1));
    EXPECT_TRUE(nn::nifm::IsAnyInternetRequestAccepted(clientId2));

    nn::nifm::CancelNetworkRequest();

    EXPECT_FALSE(nn::nifm::IsAnyInternetRequestAccepted(clientId1));
    EXPECT_FALSE(nn::nifm::IsAnyInternetRequestAccepted(clientId2));

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

TEST_F(ClientManagementCaseTest, IsAnyForegroundRequestAccepted)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    {
        nn::nifm::NetworkConnection networkConnectionSystemInternet;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionSystemInternet.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionSystemInternet.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        nn::nifm::NetworkConnection networkConnectionApplicationInternet;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApplicationInternet.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionApplicationInternet.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        nn::nifm::NetworkConnection networkConnectionAppletInternet;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionAppletInternet.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForApplet));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionAppletInternet.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        nn::nifm::NetworkConnection networkConnectionApplicationLocal;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionApplicationLocal.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionApplicationLocal.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        EXPECT_FALSE(networkConnectionSystemInternet.IsAvailable());
        EXPECT_FALSE(networkConnectionApplicationInternet.IsAvailable());
        EXPECT_FALSE(networkConnectionAppletInternet.IsAvailable());
        EXPECT_FALSE(networkConnectionApplicationLocal.IsAvailable());

        EXPECT_FALSE(nn::nifm::IsAnyForegroundRequestAccepted());

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionSystemInternet, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        EXPECT_TRUE(networkConnectionSystemInternet.IsAvailable());
        EXPECT_FALSE(networkConnectionApplicationInternet.IsAvailable());
        EXPECT_FALSE(networkConnectionAppletInternet.IsAvailable());
        EXPECT_FALSE(networkConnectionApplicationLocal.IsAvailable());

        EXPECT_FALSE(nn::nifm::IsAnyForegroundRequestAccepted());

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionAppletInternet, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        EXPECT_TRUE(networkConnectionSystemInternet.IsAvailable());
        EXPECT_FALSE(networkConnectionApplicationInternet.IsAvailable());
        EXPECT_TRUE(networkConnectionAppletInternet.IsAvailable());
        EXPECT_FALSE(networkConnectionApplicationLocal.IsAvailable());

        EXPECT_TRUE(nn::nifm::IsAnyForegroundRequestAccepted());

        networkConnectionSystemInternet.CancelRequest();

        EXPECT_FALSE(networkConnectionSystemInternet.IsAvailable());
        EXPECT_FALSE(networkConnectionApplicationInternet.IsAvailable());
        EXPECT_TRUE(networkConnectionAppletInternet.IsAvailable());
        EXPECT_FALSE(networkConnectionApplicationLocal.IsAvailable());

        EXPECT_TRUE(nn::nifm::IsAnyForegroundRequestAccepted());

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApplicationInternet, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        EXPECT_FALSE(networkConnectionSystemInternet.IsAvailable());
        EXPECT_TRUE(networkConnectionApplicationInternet.IsAvailable());
        EXPECT_TRUE(networkConnectionAppletInternet.IsAvailable());
        EXPECT_FALSE(networkConnectionApplicationLocal.IsAvailable());

        EXPECT_TRUE(nn::nifm::IsAnyForegroundRequestAccepted());

        networkConnectionAppletInternet.CancelRequest();

        EXPECT_FALSE(networkConnectionSystemInternet.IsAvailable());
        EXPECT_TRUE(networkConnectionApplicationInternet.IsAvailable());
        EXPECT_FALSE(networkConnectionAppletInternet.IsAvailable());
        EXPECT_FALSE(networkConnectionApplicationLocal.IsAvailable());

        EXPECT_TRUE(nn::nifm::IsAnyForegroundRequestAccepted());

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionApplicationLocal, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        EXPECT_FALSE(networkConnectionSystemInternet.IsAvailable());
        EXPECT_FALSE(networkConnectionApplicationInternet.IsAvailable());
        EXPECT_FALSE(networkConnectionAppletInternet.IsAvailable());
        EXPECT_TRUE(networkConnectionApplicationLocal.IsAvailable());

        EXPECT_TRUE(nn::nifm::IsAnyForegroundRequestAccepted());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

TEST_F(ClientManagementCaseTest, SetExclusiveClient)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    nn::nifm::ClientId invalidClientId = { nn::nifm::InvalidClientIdValue };

    nn::nifm::ClientId ownClientId = {};
    ownClientId = nn::nifm::GetClientId();
    EXPECT_NE(nn::nifm::InvalidClientIdValue, ownClientId.value);

    nn::nifm::ClientId othersClientId = { ownClientId.value + 1000000 };

    {
        // 通常のリクエスト
        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        // 自分の ID を指定しても却下されない
        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(10)));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetExclusiveClient(ownClientId));
        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(10)));
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection.GetRequestState());

        // 無効な ID を指定しても却下されない
        networkConnection.GetSystemEvent().Clear();
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetExclusiveClient(invalidClientId));
        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(10)));
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection.GetRequestState());

        // 他人の ID を指定すると却下される
        networkConnection.GetSystemEvent().Clear();
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetExclusiveClient(othersClientId));
        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(10)));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLocked, networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection.GetRequestState());

        // 再提出しても通らない
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLocked, networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection.GetRequestState());

        // ローカル通信の要求も通らない
        nn::nifm::NetworkConnection networkConnectionLocal;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionLocal.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionLocal, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLocked, networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection.GetRequestState());

        // 無効な ID を指定すると再び通るようになる
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetExclusiveClient(invalidClientId));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection.GetRequestState());
    }

    {
        // persistent なリクエスト
        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestPersistent(networkConnection.GetRequestHandle(), true));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        // 自分の ID を指定しても却下されない
        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(10)));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetExclusiveClient(ownClientId));
        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(10)));
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection.GetRequestState());

        // 無効な ID を指定しても却下されない
        networkConnection.GetSystemEvent().Clear();
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetExclusiveClient(invalidClientId));
        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(10)));
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection.GetRequestState());

        // 他人の ID を指定すると却下される
        networkConnection.GetSystemEvent().Clear();
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetExclusiveClient(othersClientId));
        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(10)));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLocked, networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_OnHold, networkConnection.GetRequestState());
        networkConnection.CancelRequest();

        // 再提出しても通らない
        networkConnection.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(10)));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLocked, networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_OnHold, networkConnection.GetRequestState());
        networkConnection.CancelRequest();

        // 無効な ID を指定すると再び通るようになる
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetExclusiveClient(invalidClientId));
        networkConnection.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection.GetRequestState());
    }

    {
        // blocker なリクエスト
        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        // 自分の ID を指定しても却下されない
        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(10)));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetExclusiveClient(ownClientId));
        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(10)));
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection.GetRequestState());

        // 無効な ID を指定しても却下されない
        networkConnection.GetSystemEvent().Clear();
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetExclusiveClient(invalidClientId));
        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(10)));
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection.GetRequestState());

        // 他人の ID を指定すると却下される
        networkConnection.GetSystemEvent().Clear();
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetExclusiveClient(othersClientId));
        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(10)));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLocked, networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Blocking, networkConnection.GetRequestState());
        networkConnection.CancelRequest();

        // 再提出しても通らない
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLocked, networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection.GetRequestState());

        // 無効な ID を指定すると再び通るようになる
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetExclusiveClient(invalidClientId));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Accepted, networkConnection.GetRequestState());
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetExclusiveClient(invalidClientId));

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
} // NOLINT(impl/function_size)

TEST_F(ClientManagementCaseTest, SetBackgroundRequestEnabled)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    {
        // BG 利用要求
        nn::nifm::NetworkConnection networkConnectionBG;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBG.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        networkConnectionBG.SubmitRequestAndWait();
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        // BG 利用要求を禁止
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetBackgroundRequestEnabled(false));

        // 状態が変わるまで、最長で30秒待つ（切断される）
        NN_LOG("Waiting disconnect ");
        nn::Result connectionResult = nn::ResultSuccess();
        nn::nifm::InternetConnectionStatus internetConnectionStatus;
        for (int i = 0; i < 30; ++i)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

            NN_LOG(".");
            connectionResult = nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus);
            if (connectionResult.IsFailure())
            {
                break;
            }
        }
        NN_LOG("Done.\n");
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNotConnected, connectionResult);

        // すべての BG 利用要求が破棄され、切断された
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultBackgroundRequestProhibited, networkConnectionBG.GetResult());

        // FG の利用要求は受理可能
        nn::nifm::NetworkConnection networkConnectionFG;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionFG.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        networkConnectionFG.SubmitRequestAndWait();
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionFG.GetResult());

        // BG 属性の場合は、共存可能でも却下される
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBG.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        networkConnectionBG.SubmitRequestAndWait();
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultBackgroundRequestProhibited, networkConnectionBG.GetResult());

        // BG 利用要求を許可
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetBackgroundRequestEnabled(true));

        // BG 属性の場合も受理可能に
        networkConnectionBG.SubmitRequestAndWait();
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());
    }

    {
        // imatake-openwep40
        nn::nifm::WirelessSettingData wirelessSetting0 = {
            {  // ssidConfig
                {  // ssid
                    17,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x77,0x65,0x70,0x34,0x30 }  // hex
                },
            true  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Open,  // authentication
                    nn::nifm::Encryption_Wep  // encryption
                },
                {  // sharedKey
                    5,  // length
                    "Shi2i"  // keyMaterial
                }
            }
        };
        nn::nifm::IpSettingData ipSetting0 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        nn::nifm::NetworkProfileData networkProfile0 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true, // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting0
            },
            ipSetting0
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile0);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        // ネット接続アプレットからの利用要求
        nn::nifm::NetworkConnection networkConnectionFG;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionFG.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForNetConnectApplet));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionFG.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionFG.GetRequestHandle(), temporaryNetworkProfile.GetId()));

        // BG 利用要求
        nn::nifm::NetworkConnection networkConnectionBG;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBG.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        networkConnectionFG.SubmitRequest();
        networkConnectionBG.SubmitRequestAndWait();

        NNT_ASSERT_RESULT_SUCCESS(networkConnectionFG.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        // BG 利用要求を禁止
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetBackgroundRequestEnabled(false));

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));

        // FG は受理，BG は却下
        NNT_ASSERT_RESULT_SUCCESS(networkConnectionFG.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultBackgroundRequestProhibited, networkConnectionBG.GetResult());

        // FG を取り下げ
        networkConnectionFG.CancelRequest();
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnectionFG.GetResult());

        // 状態が変わるまで、最長で30秒待つ（切断されない）
        NN_LOG("Waiting disconnect ");
        nn::Result connectionResult = nn::ResultSuccess();
        nn::nifm::InternetConnectionStatus internetConnectionStatus;
        for (int i = 0; i < 30; ++i)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

            NN_LOG(".");
            connectionResult = nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus);
            if (connectionResult.IsFailure())
            {
                break;
            }
        }
        NN_LOG("Done.\n");
        NNT_ASSERT_RESULT_SUCCESS(connectionResult);

        // BG 利用要求を許可
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetBackgroundRequestEnabled(true));
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
} // NOLINT(impl/function_size)

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
