﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/nifm/nifm_Api.h>

#include <nn/nifm/detail/util/nifm_ArrayedFactory.h>

TEST(ArrayedFactoryCase, Basic)
{
    nn::nifm::detail::ArrayedFactory<uint64_t, 5> uint64Factory;
    uint64_t* pUint64;
    uint64_t** ppUint64;

    // 埋める
    EXPECT_NE(nullptr, new(uint64Factory.Allocate(sizeof(uint64_t))) uint64_t(10));
    EXPECT_NE(nullptr, new(uint64Factory.Allocate(sizeof(uint64_t))) uint64_t(11));
    EXPECT_NE(nullptr, new(uint64Factory.Allocate(sizeof(uint64_t))) uint64_t(12));
    EXPECT_NE(nullptr, new(uint64Factory.Allocate(sizeof(uint64_t))) uint64_t(13));
    EXPECT_NE(nullptr, new(uint64Factory.Allocate(sizeof(uint64_t))) uint64_t(14));
    EXPECT_EQ(5, uint64Factory.GetSize());
    EXPECT_EQ(5, uint64Factory.GetCount());

    // あふれる
    EXPECT_EQ(nullptr, uint64Factory.Allocate(sizeof(uint64_t)));

    // 格納された値を確認
    EXPECT_EQ(uint64_t(10), uint64Factory[0]);
    EXPECT_EQ(uint64_t(11), uint64Factory[1]);
    EXPECT_EQ(uint64_t(12), uint64Factory[2]);
    EXPECT_EQ(uint64_t(13), uint64Factory[3]);
    EXPECT_EQ(uint64_t(14), uint64Factory[4]);

    // いっぱいの状態で最後を抜く
    EXPECT_TRUE(uint64Factory.Free(14));
    EXPECT_EQ(uint64_t(10), uint64Factory[0]);
    EXPECT_EQ(uint64_t(11), uint64Factory[1]);
    EXPECT_EQ(uint64_t(12), uint64Factory[2]);
    EXPECT_EQ(uint64_t(13), uint64Factory[3]);
    EXPECT_EQ(4, uint64Factory.GetCount());

    // いっぱいに戻す
    EXPECT_NE(nullptr, new(uint64Factory.Allocate(sizeof(uint64_t))) uint64_t(14));
    EXPECT_EQ(5, uint64Factory.GetCount());

    // いっぱいの状態で途中を抜く
    EXPECT_TRUE(uint64Factory.Free(12));
    EXPECT_EQ(uint64_t(10), uint64Factory[0]);
    EXPECT_EQ(uint64_t(11), uint64Factory[1]);
    EXPECT_EQ(uint64_t(13), uint64Factory[2]);
    EXPECT_EQ(uint64_t(14), uint64Factory[3]);
    EXPECT_EQ(uint64_t(4), uint64Factory.GetCount());

    // 空きがある状態で最後を抜く
    EXPECT_TRUE(uint64Factory.Free(14));
    EXPECT_EQ(uint64_t(10), uint64Factory[0]);
    EXPECT_EQ(uint64_t(11), uint64Factory[1]);
    EXPECT_EQ(uint64_t(13), uint64Factory[2]);
    EXPECT_EQ(3, uint64Factory.GetCount());

    // 空きがある状態で途中を抜く
    EXPECT_TRUE(uint64Factory.Free(11));
    EXPECT_EQ(uint64_t(10), uint64Factory[0]);
    EXPECT_EQ(uint64_t(13), uint64Factory[1]);
    EXPECT_EQ(2, uint64Factory.GetCount());

    // 空きがある状態で頭を抜く
    EXPECT_TRUE(uint64Factory.Free(10));
    EXPECT_EQ(uint64_t(13), uint64Factory[0]);
    EXPECT_EQ(1, uint64Factory.GetCount());

    // 空にする
    EXPECT_TRUE(uint64Factory.Free(13));
    EXPECT_EQ(0, uint64Factory.GetCount());

    // 再度埋める
    EXPECT_NE(nullptr, pUint64 = reinterpret_cast<uint64_t*>(uint64Factory.Allocate(sizeof(uint64_t))));    *pUint64 = 10;
    EXPECT_NE(nullptr, pUint64 = reinterpret_cast<uint64_t*>(uint64Factory.Allocate(sizeof(uint64_t))));    *pUint64 = 11;
    EXPECT_NE(nullptr, pUint64 = reinterpret_cast<uint64_t*>(uint64Factory.Allocate(sizeof(uint64_t))));    *pUint64 = 12;
    EXPECT_NE(nullptr, pUint64 = reinterpret_cast<uint64_t*>(uint64Factory.Allocate(sizeof(uint64_t))));    *pUint64 = 13;
    EXPECT_NE(nullptr, pUint64 = reinterpret_cast<uint64_t*>(uint64Factory.Allocate(sizeof(uint64_t))));    *pUint64 = 14;
    EXPECT_EQ(5, uint64Factory.GetSize());
    EXPECT_EQ(5, uint64Factory.GetCount());

    // あふれる
    EXPECT_EQ(nullptr, pUint64 = reinterpret_cast<uint64_t*>(uint64Factory.Allocate(sizeof(uint64_t))));

    // 格納された値を確認
    ppUint64 = uint64Factory.begin();
    EXPECT_EQ(uint64_t(10), **ppUint64++);
    EXPECT_EQ(uint64_t(11), **ppUint64++);
    EXPECT_EQ(uint64_t(12), **ppUint64++);
    EXPECT_EQ(uint64_t(13), **ppUint64++);
    EXPECT_EQ(uint64_t(14), **ppUint64++);
    EXPECT_EQ(ppUint64, uint64Factory.end());
    EXPECT_EQ(ppUint64, uint64Factory.end());

    // いっぱいの状態で最後を抜く
    EXPECT_TRUE(uint64Factory.Free(&uint64Factory[4]));
    ppUint64 = uint64Factory.begin();
    EXPECT_EQ(uint64_t(10), **ppUint64++);
    EXPECT_EQ(uint64_t(11), **ppUint64++);
    EXPECT_EQ(uint64_t(12), **ppUint64++);
    EXPECT_EQ(uint64_t(13), **ppUint64++);
    EXPECT_EQ(ppUint64, uint64Factory.end());
    EXPECT_EQ(4, uint64Factory.GetCount());

    // いっぱいに戻す
    EXPECT_NE(nullptr, pUint64 = reinterpret_cast<uint64_t*>(uint64Factory.Allocate(sizeof(uint64_t))));    *pUint64 = 14;
    EXPECT_EQ(5, uint64Factory.GetCount());

    // いっぱいの状態で途中を抜く
    EXPECT_TRUE(uint64Factory.Free(&uint64Factory[2]));
    ppUint64 = uint64Factory.begin();
    EXPECT_EQ(uint64_t(10), **ppUint64++);
    EXPECT_EQ(uint64_t(11), **ppUint64++);
    EXPECT_EQ(uint64_t(13), **ppUint64++);
    EXPECT_EQ(uint64_t(14), **ppUint64++);
    EXPECT_EQ(ppUint64, uint64Factory.end());
    EXPECT_EQ(4, uint64Factory.GetCount());

    // 空きがある状態で最後を抜く
    EXPECT_TRUE(uint64Factory.Free(&uint64Factory[3]));
    ppUint64 = uint64Factory.begin();
    EXPECT_EQ(uint64_t(10), **ppUint64++);
    EXPECT_EQ(uint64_t(11), **ppUint64++);
    EXPECT_EQ(uint64_t(13), **ppUint64++);
    EXPECT_EQ(ppUint64, uint64Factory.end());
    EXPECT_EQ(3, uint64Factory.GetCount());

    // 空きがある状態で途中を抜く
    EXPECT_TRUE(uint64Factory.Free(&uint64Factory[1]));
    ppUint64 = uint64Factory.begin();
    EXPECT_EQ(uint64_t(10), **ppUint64++);
    EXPECT_EQ(uint64_t(13), **ppUint64++);
    EXPECT_EQ(ppUint64, uint64Factory.end());
    EXPECT_EQ(2, uint64Factory.GetCount());

    // 空きがある状態で頭を抜く
    EXPECT_TRUE(uint64Factory.Free(&uint64Factory[0]));
    ppUint64 = uint64Factory.begin();
    EXPECT_EQ(uint64_t(13), **ppUint64++);
    EXPECT_EQ(ppUint64, uint64Factory.end());
    EXPECT_EQ(1, uint64Factory.GetCount());

    // 空にする
    EXPECT_TRUE(uint64Factory.Free(&uint64Factory[0]));
    ppUint64 = uint64Factory.begin();
    EXPECT_EQ(ppUint64, uint64Factory.end());
    EXPECT_EQ(0, uint64Factory.GetCount());
} // NOLINT(readability/fn_size)


TEST(ArrayedFactoryCase, Move)
{
    nn::nifm::detail::ArrayedFactory<uint64_t, 5> uint64Factory;

    // 埋める
    EXPECT_NE(nullptr, new(uint64Factory.Allocate(sizeof(uint64_t))) uint64_t(10));
    EXPECT_NE(nullptr, new(uint64Factory.Allocate(sizeof(uint64_t))) uint64_t(11));
    EXPECT_NE(nullptr, new(uint64Factory.Allocate(sizeof(uint64_t))) uint64_t(12));
    EXPECT_NE(nullptr, new(uint64Factory.Allocate(sizeof(uint64_t))) uint64_t(13));
    EXPECT_NE(nullptr, new(uint64Factory.Allocate(sizeof(uint64_t))) uint64_t(14));
    EXPECT_EQ(5, uint64Factory.GetSize());
    EXPECT_EQ(5, uint64Factory.GetCount());

    // 格納された値を確認
    EXPECT_EQ(uint64_t(10), uint64Factory[0]);
    EXPECT_EQ(uint64_t(11), uint64Factory[1]);
    EXPECT_EQ(uint64_t(12), uint64Factory[2]);
    EXPECT_EQ(uint64_t(13), uint64Factory[3]);
    EXPECT_EQ(uint64_t(14), uint64Factory[4]);

    // 先頭を末尾に移動
    EXPECT_TRUE(uint64Factory.Move(0, uint64Factory.GetCount() - 1));
    EXPECT_EQ(uint64_t(11), uint64Factory[0]);
    EXPECT_EQ(uint64_t(12), uint64Factory[1]);
    EXPECT_EQ(uint64_t(13), uint64Factory[2]);
    EXPECT_EQ(uint64_t(14), uint64Factory[3]);
    EXPECT_EQ(uint64_t(10), uint64Factory[4]);

    // 末尾を先頭に移動
    EXPECT_TRUE(uint64Factory.Move(uint64Factory.GetCount() - 1, 0));
    EXPECT_EQ(uint64_t(10), uint64Factory[0]);
    EXPECT_EQ(uint64_t(11), uint64Factory[1]);
    EXPECT_EQ(uint64_t(12), uint64Factory[2]);
    EXPECT_EQ(uint64_t(13), uint64Factory[3]);
    EXPECT_EQ(uint64_t(14), uint64Factory[4]);

    // 途中を末尾に移動
    EXPECT_TRUE(uint64Factory.Move(2, uint64Factory.GetCount() - 1));
    EXPECT_EQ(uint64_t(10), uint64Factory[0]);
    EXPECT_EQ(uint64_t(11), uint64Factory[1]);
    EXPECT_EQ(uint64_t(13), uint64Factory[2]);
    EXPECT_EQ(uint64_t(14), uint64Factory[3]);
    EXPECT_EQ(uint64_t(12), uint64Factory[4]);

    // 末尾を途中に移動
    EXPECT_TRUE(uint64Factory.Move(uint64Factory.GetCount() - 1, 2));
    EXPECT_EQ(uint64_t(10), uint64Factory[0]);
    EXPECT_EQ(uint64_t(11), uint64Factory[1]);
    EXPECT_EQ(uint64_t(12), uint64Factory[2]);
    EXPECT_EQ(uint64_t(13), uint64Factory[3]);
    EXPECT_EQ(uint64_t(14), uint64Factory[4]);

    // 途中を先頭に移動
    EXPECT_TRUE(uint64Factory.Move(2, 0));
    EXPECT_EQ(uint64_t(12), uint64Factory[0]);
    EXPECT_EQ(uint64_t(10), uint64Factory[1]);
    EXPECT_EQ(uint64_t(11), uint64Factory[2]);
    EXPECT_EQ(uint64_t(13), uint64Factory[3]);
    EXPECT_EQ(uint64_t(14), uint64Factory[4]);

    // 先頭を途中に移動
    EXPECT_TRUE(uint64Factory.Move(0, 2));
    EXPECT_EQ(uint64_t(10), uint64Factory[0]);
    EXPECT_EQ(uint64_t(11), uint64Factory[1]);
    EXPECT_EQ(uint64_t(12), uint64Factory[2]);
    EXPECT_EQ(uint64_t(13), uint64Factory[3]);
    EXPECT_EQ(uint64_t(14), uint64Factory[4]);

    // 同じ個所に移動
    EXPECT_TRUE(uint64Factory.Move(3, 3));
    EXPECT_EQ(uint64_t(10), uint64Factory[0]);
    EXPECT_EQ(uint64_t(11), uint64Factory[1]);
    EXPECT_EQ(uint64_t(12), uint64Factory[2]);
    EXPECT_EQ(uint64_t(13), uint64Factory[3]);
    EXPECT_EQ(uint64_t(14), uint64Factory[4]);

    // 範囲外
    EXPECT_FALSE(uint64Factory.Move(0, uint64Factory.GetCount()));
    EXPECT_FALSE(uint64Factory.Move(uint64Factory.GetCount(), 0));
    EXPECT_FALSE(uint64Factory.Move(uint64Factory.GetCount(),uint64Factory.GetCount()));

    // 末尾を先頭に移動
    EXPECT_TRUE(uint64Factory.MoveToTop(uint64Factory[4]));
    EXPECT_EQ(uint64_t(14), uint64Factory[0]);
    EXPECT_EQ(uint64_t(10), uint64Factory[1]);
    EXPECT_EQ(uint64_t(11), uint64Factory[2]);
    EXPECT_EQ(uint64_t(12), uint64Factory[3]);
    EXPECT_EQ(uint64_t(13), uint64Factory[4]);

    // 先頭を末尾に移動
    EXPECT_TRUE(uint64Factory.MoveToEnd(uint64Factory[0]));
    EXPECT_EQ(uint64_t(10), uint64Factory[0]);
    EXPECT_EQ(uint64_t(11), uint64Factory[1]);
    EXPECT_EQ(uint64_t(12), uint64Factory[2]);
    EXPECT_EQ(uint64_t(13), uint64Factory[3]);
    EXPECT_EQ(uint64_t(14), uint64Factory[4]);

    // 11 を削除して 10 を追加
    EXPECT_TRUE(uint64Factory.Free(11));
    EXPECT_NE(nullptr, new(uint64Factory.Allocate(sizeof(uint64_t))) uint64_t(10));
    EXPECT_EQ(uint64_t(10), uint64Factory[0]);
    EXPECT_EQ(uint64_t(12), uint64Factory[1]);
    EXPECT_EQ(uint64_t(13), uint64Factory[2]);
    EXPECT_EQ(uint64_t(14), uint64Factory[3]);
    EXPECT_EQ(uint64_t(10), uint64Factory[4]);

    // 先頭を途中に移動
    EXPECT_TRUE(uint64Factory.Move(0, 2));
    EXPECT_EQ(uint64_t(12), uint64Factory[0]);
    EXPECT_EQ(uint64_t(13), uint64Factory[1]);
    EXPECT_EQ(uint64_t(10), uint64Factory[2]);
    EXPECT_EQ(uint64_t(14), uint64Factory[3]);
    EXPECT_EQ(uint64_t(10), uint64Factory[4]);

    // 最初の 10 を先頭に移動
    EXPECT_TRUE(uint64Factory.MoveToTop(10));
    EXPECT_EQ(uint64_t(10), uint64Factory[0]);
    EXPECT_EQ(uint64_t(12), uint64Factory[1]);
    EXPECT_EQ(uint64_t(13), uint64Factory[2]);
    EXPECT_EQ(uint64_t(14), uint64Factory[3]);
    EXPECT_EQ(uint64_t(10), uint64Factory[4]);

    // 最初の 10 を末尾に移動
    EXPECT_TRUE(uint64Factory.MoveToEnd(10));
    EXPECT_EQ(uint64_t(12), uint64Factory[0]);
    EXPECT_EQ(uint64_t(13), uint64Factory[1]);
    EXPECT_EQ(uint64_t(14), uint64Factory[2]);
    EXPECT_EQ(uint64_t(10), uint64Factory[3]);
    EXPECT_EQ(uint64_t(10), uint64Factory[4]);

    // 移動するものが見つからない
    EXPECT_FALSE(uint64Factory.MoveToTop(100));
    EXPECT_FALSE(uint64Factory.MoveToEnd(100));
}
