﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// テスト用のヘッダ
#include <cstdlib>
#include <string>
#include <memory>

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_ErrorResult.h>
#include <nn/util/util_Endian.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/fs.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/settings/settings_Language.h>
#include <nn/settings/system/settings_Language.h>

// テスト対象のヘッダ
#include <nn/ngc.h>

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
#pragma execution_character_set("utf-8")
#endif

NN_STATIC_ASSERT(sizeof(uint16_t) == sizeof(char16_t));

namespace {

// pWord の中に * がひとつでもあるなら true を返します
// NG ワードに重複がある場合、 * のみの文字列にならない場合があるため、
// すべての文字が * 埋めされる保証はない
// e.g. NG ワードとして badword bad ord の 3 つが登録されている場合
//      現在の仕様だと bad にマッチ -> badword がスキップ -> ord にマッチとなるため、
//      最終的な結果は ***w*** となる
bool CheckAsterisk(const char* pWord, size_t wordSize)
{
    for (size_t i = 0; i < wordSize; ++i)
    {
        if (pWord[i] == '*')
        {
            return true;
        }
    }
    return false;
}
bool CheckAsterisk(const uint16_t* pWord, size_t wordSize)
{
    for (size_t i = 0; i < wordSize; ++i)
    {
        if (pWord[i] == 0x002A)
        {
            return true;
        }
    }
    return false;
}

// UTF-16 の長さを返します
int Strlen16(uint16_t* pStr)
{
    int len = 0;
    while (pStr[len] != '\0')
    {
        if (pStr[len + 1] == '\0')
        {
            return len + 1;
        }
        if (pStr[len + 2] == '\0')
        {
            return len + 2;
        }
        if (pStr[len + 3] == '\0')
        {
            return len + 3;
        }
        len += 4;
    }
    return len;
}

// UTF-16 文字列のコピーを行います
void Strncpy16(uint16_t* pDst, const uint16_t* pSrc, size_t length)
{
    for (size_t i = 0; i < length; ++i)
    {
        pDst[i] = pSrc[i];
    }
}

// テストフィクスチャ
class NgcAllNgWordCheckTest : public ::testing::Test
{
protected:
    NgcAllNgWordCheckTest() NN_NOEXCEPT : m_pFsCache(NULL), m_pWorkMemory(NULL),
        m_pText(NULL), m_pTextUtf16(NULL),  m_pTextUtf8(NULL), m_WordUtf16Iter(0), m_WordUtf8Iter(0) {}

    // テスト開始前のセットアップ
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        // fs のマウント
        size_t cacheSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
        m_pFsCache = std::malloc(cacheSize);
        nn::fs::MountRom("rom", m_pFsCache, cacheSize);

        // NGC フィルターの初期化
        m_pWorkMemory = std::malloc(nn::ngc::ProfanityFilter::WorkMemorySize);
        NN_ABORT_UNLESS(m_pWorkMemory);
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Filter.Initialize(m_pWorkMemory,
            nn::ngc::ProfanityFilter::WorkMemorySize));
    }

    // テスト終了後の後処理
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fs::Unmount("rom");

        std::free(m_pFsCache);

        if (m_pText)
        {
            std::free(m_pText);
        }
        if (m_pTextUtf8)
        {
            std::free(m_pTextUtf8);
        }

        m_Filter.Finalize();
        std::free(m_pWorkMemory);
    }

    // 指定されたテスト用テキスト読み込みます
    // 読み込みに成功したら true を返します
    bool ReadText(int type) NN_NOEXCEPT
    {
        if (type >= nn::ngc::ProfanityFilterPatternList_Max || type < 0)
        {
            return false;
        }

        char pFileName[128];
        nn::util::SNPrintf(pFileName, 128, "rom:/%d.txt", type);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&m_File, pFileName, nn::fs::OpenMode_Read));

        int64_t fileSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, m_File));
        fileSize += 2;
        m_pText = std::malloc(static_cast<size_t>(fileSize));
        if (!m_pText)
        {
            return false;
        }
        memset(m_pText, 0, static_cast<size_t>(fileSize));

        size_t readSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(&readSize, m_File,
            0, m_pText, static_cast<size_t>(fileSize - 2)));

        nn::fs::CloseFile(m_File);

        // エンディアンの変換
        uint16_t* pTextIter = reinterpret_cast<uint16_t*>(m_pText);
        for (int i = 0; i < (fileSize / sizeof(uint16_t)); ++i)
        {
            nn::util::SwapEndian(pTextIter);
            pTextIter++;
        }
        m_pTextUtf16 = reinterpret_cast<uint16_t*>(m_pText);
        m_pTextUtf16 = &m_pTextUtf16[1];

        int utf8Length = 0;
        int utf16Length = Strlen16(m_pTextUtf16);

        // 読み込んだ UTF-16 文字列を UTF-8 に変換
        auto encodingResult = nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8(&utf8Length, m_pTextUtf16, utf16Length);
        NN_ABORT_UNLESS(nn::util::CharacterEncodingResult_Success == encodingResult);
        NN_ASSERT(utf8Length > 0);
        utf8Length++;   // 終端文字ぶんを追加
        m_pTextUtf8 = reinterpret_cast<char*>(std::malloc(static_cast<size_t>(utf8Length)));
        NN_ABORT_UNLESS(m_pTextUtf8);
        memset(m_pTextUtf8, 0, static_cast<size_t>(utf8Length));
        encodingResult = nn::util::ConvertStringUtf16NativeToUtf8(m_pTextUtf8, utf8Length, m_pTextUtf16, utf16Length);
        NN_ABORT_UNLESS(nn::util::CharacterEncodingResult_Success == encodingResult);

        m_WordUtf16Iter = 0;
        m_WordUtf8Iter = 0;

        return true;
    }

    // m_pTextUtf8/m_pText から 1 単語ずつ pOutWord に格納
    // 失敗するか、すべての単語を読み込んだら false を返す
    bool GetOneNgWord(char* pOutWordUtf8, uint16_t* pOutWordUtf16,
        size_t wordUtf8Size, size_t wordUtf16Size) NN_NOEXCEPT
    {
        if (!m_pTextUtf8 || !m_pTextUtf16)
        {
            return false;
        }

        // 文字列を見ていき正規表現を抜いた単語を作成
        // NgWord に使われている正規表現は以下の種類のみ
        // .* \. ^ $
        // よってこれらを排除するだけで良い
        if (static_cast<size_t>(m_WordUtf8Iter) >= strlen(m_pTextUtf8) ||
            m_WordUtf16Iter >= Strlen16(m_pTextUtf16))
        {
            return false;
        }

        // UTF-8 文字列を取得する
        {
            std::string word = "";
            while (static_cast<size_t>(m_WordUtf8Iter) < strlen(m_pTextUtf8) &&
                   m_pTextUtf8[m_WordUtf8Iter] != '\n' &&
                   m_pTextUtf8[m_WordUtf8Iter] != '\0')
            {
                if (m_pTextUtf8[m_WordUtf8Iter] != '*'  &&
                    m_pTextUtf8[m_WordUtf8Iter] != '\\' &&
                    m_pTextUtf8[m_WordUtf8Iter] != '^'  &&
                    m_pTextUtf8[m_WordUtf8Iter] != '$'  &&
                    (m_pTextUtf8[m_WordUtf8Iter] != '.' ||
                     (m_pTextUtf8[m_WordUtf8Iter] == '.' &&
                      m_WordUtf8Iter > 0 &&
                      m_pTextUtf8[m_WordUtf8Iter - 1] == '\\')
                    )
                   )
                {
                    word += m_pTextUtf8[m_WordUtf8Iter];
                }
                ++m_WordUtf8Iter;
            }
            ++m_WordUtf8Iter;
            if (word.length() > 0)
            {
                word += '\0';
                if (word.length() > wordUtf8Size)
                {
                    return false;
                }
                strncpy(pOutWordUtf8, word.c_str(), word.length());
            }
       }

        // UTF-16 文字列を取得する
        {
            std::u16string word;
            while (m_WordUtf16Iter < Strlen16(m_pTextUtf16) &&
                   m_pTextUtf16[m_WordUtf16Iter] != 0x000A &&
                   m_pTextUtf16[m_WordUtf16Iter] != 0)
            {
                if (m_pTextUtf16[m_WordUtf16Iter] != 0x002A  &&
                    m_pTextUtf16[m_WordUtf16Iter] != 0x005C &&
                    m_pTextUtf16[m_WordUtf16Iter] != 0x005E  &&
                    m_pTextUtf16[m_WordUtf16Iter] != 0x0024 &&
                    (m_pTextUtf16[m_WordUtf16Iter] != 0x002E ||
                     (m_pTextUtf16[m_WordUtf16Iter] == 0x002E &&
                      m_WordUtf16Iter > 0 &&
                      m_pTextUtf16[m_WordUtf16Iter - 1] == 0x005C)
                    )
                   )
                {
                    word += m_pTextUtf16[m_WordUtf16Iter];
                }
                ++m_WordUtf16Iter;
            }
            ++m_WordUtf16Iter;
            if (word.length() > 0)
            {
                word += static_cast<char16_t>(0x0000);
                if (word.length() > wordUtf16Size)
                {
                    return false;
                }
                Strncpy16(pOutWordUtf16,
                          reinterpret_cast<uint16_t*>(const_cast<char16_t*>(word.c_str())),
                          word.length());
            }
        }

        return true;
    }

    /**
     * @brief   指定した言語で CheckProfanityWords と MaskProfanityWordsInText のテストを行います。
     * @details Programs/Iris/Reources/NgWord にある NG ワードリストのすべての単語を読み込み
     *          正しく NG ワードとして引っかかるか確認します。
     */
    void DoApiTest(int pattern) NN_NOEXCEPT
    {
        const size_t MaxUtf8Length = nn::ngc::TextLengthMax * 4;    // UTF-8 文字列的にあり得る単語の最大サイズ
        const size_t MaxUtf16Length = nn::ngc::TextLengthMax * 2;   // UTF-16 文字列的にあり得る単語の最大サイズ
        char pNgWordsUtf8[MaxUtf8Length] = {};
        uint16_t pNgWordsUtf16[MaxUtf16Length] = {};

        int textNum = 0;
        for (textNum = 0; textNum < nn::ngc::ProfanityFilterPatternList_Max; ++textNum)
        {
            if ((pattern & (1 << textNum)) != 0)
            {
                break;
            }
        }
        NN_ASSERT(textNum < nn::ngc::ProfanityFilterPatternList_Max);
        ASSERT_TRUE(ReadText(textNum));
        while (GetOneNgWord(pNgWordsUtf8, pNgWordsUtf16, MaxUtf8Length, MaxUtf16Length))
        {
            // UTF-8
            {
                //NN_LOG("UTF-8  %s", pNgWordsUtf8);

                nn::Bit32 results = 0;
                const char* ppWords[] = { pNgWordsUtf8 };
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_Filter.CheckProfanityWords(&results,
                    pattern,
                    ppWords, 1));
                EXPECT_EQ(results & pattern, pattern);

                int count = 0;
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_Filter.MaskProfanityWordsInText(&count,
                    pNgWordsUtf8, pattern));

                //NN_LOG(" -> %s\n", pNgWordsUtf8);

                EXPECT_GE(count, 1);
                ASSERT_GT(strlen(pNgWordsUtf8), 0U);
                EXPECT_TRUE(CheckAsterisk(pNgWordsUtf8, strlen(pNgWordsUtf8)));
            }

            // UTF-16
            {
                char pConverted[MaxUtf8Length] = {};
                auto encodingResult = nn::util::ConvertStringUtf16NativeToUtf8(
                    pConverted, MaxUtf8Length, pNgWordsUtf16);
                NN_ABORT_UNLESS(nn::util::CharacterEncodingResult_Success == encodingResult);
                //NN_LOG("UTF-16 %s", pConverted);

                nn::Bit32 results = 0;
                const uint16_t* ppWords[] = { pNgWordsUtf16 };
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_Filter.CheckProfanityWords(&results,
                    pattern,
                    ppWords, 1));
                EXPECT_EQ(results & pattern, pattern);

                int count = 0;
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_Filter.MaskProfanityWordsInText(&count,
                    pNgWordsUtf16, pattern));

                encodingResult = nn::util::ConvertStringUtf16NativeToUtf8(
                    pConverted, MaxUtf8Length, pNgWordsUtf16);
                NN_ABORT_UNLESS(nn::util::CharacterEncodingResult_Success == encodingResult);

                //NN_LOG(" -> %s\n", pConverted);

                EXPECT_GE(count, 1);
                ASSERT_GT(Strlen16(pNgWordsUtf16), 0);
                EXPECT_TRUE(CheckAsterisk(pNgWordsUtf16, Strlen16(pNgWordsUtf16)));

                // UTF-8 と UTF-16 の結果は同じである
                ASSERT_EQ(strcmp(pNgWordsUtf8, pConverted), 0);
            }
        }
    }

    template<typename T>
    void MeasureCheckWordTimeCore(T** ppWords, int pattern, int utf)
    {
        // 実際に NG ワードに登録されている単語でのベンチマーク
        nn::Bit32 checkResults[nn::ngc::WordCountMax];
        nn::os::Tick startTick = nn::os::GetSystemTick();
        auto result = m_Filter.CheckProfanityWords(checkResults, pattern, const_cast<const T**>(ppWords), nn::ngc::WordCountMax);
        nn::os::Tick endTick = nn::os::GetSystemTick();
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        for (int i = 0; i < nn::ngc::WordCountMax; ++i)
        {
            EXPECT_EQ(checkResults[i] & pattern, pattern);
        }

        // 処理時間の表示
        nn::TimeSpan time = (endTick - startTick).ToTimeSpan();
        NN_LOG("Processing time(UTF-%2d listed words):      %lld us (%lld ms)\n", utf, time.GetMicroSeconds(), time.GetMilliSeconds());

        // 処理可能な最大文字数でのベンチマーク(aaa... を文字列として与える)
        for (int i = 0; i < nn::ngc::WordCountMax; ++i)
        {
            checkResults[i] = 0;
            for (int j = 0; j < nn::ngc::WordLengthMax - 1; ++j)
            {
                ppWords[i][j] = 41; // a
            }
            ppWords[i][nn::ngc::WordLengthMax - 1] = 0;
        }
        startTick = nn::os::GetSystemTick();
        result = m_Filter.CheckProfanityWords(checkResults, pattern, const_cast<const T**>(ppWords), nn::ngc::WordCountMax);
        endTick = nn::os::GetSystemTick();
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        // aaa... は NG ワードリストに登録されていない（はず）
        for (int i = 0; i < nn::ngc::WordCountMax; ++i)
        {
            EXPECT_EQ(checkResults[i] & pattern, 0);
        }

        // 処理時間の表示
        time = (endTick - startTick).ToTimeSpan();
        NN_LOG("Processing time(UTF-%2d max length words):  %lld us (%lld ms)\n", utf, time.GetMicroSeconds(), time.GetMilliSeconds());
    }

    /**
    * @brief   CheckProfanityWords にかかる時間を測定します。
    */
    void MeasureCheckWordTime(int pattern)
    {
        const size_t MaxUtf8Length = nn::ngc::WordLengthMax * 4;    // UTF-8 文字列的にあり得る単語の最大サイズ
        const size_t MaxUtf16Length = nn::ngc::WordLengthMax * 2;   // UTF-16 文字列的にあり得る単語の最大サイズ
        std::unique_ptr< char[] > ppWordsUtf8[nn::ngc::WordCountMax];           // チェック対象文字列(UTF-8)
        std::unique_ptr< uint16_t[] > ppWordsUtf16[nn::ngc::WordCountMax];      // チェック対象文字列(UTF-16)

        // 一度にチェック可能な最大の単語数ぶんの配列を作成
        int textNum = 0;
        for (textNum = 0; textNum < nn::ngc::ProfanityFilterPatternList_Max; ++textNum)
        {
            if ((pattern & (1 << textNum)) != 0)
            {
                break;
            }
        }
        NN_ASSERT(textNum < nn::ngc::ProfanityFilterPatternList_Max);
        ASSERT_TRUE(ReadText(textNum));

        for (int i = 0; i < nn::ngc::WordCountMax; ++i)
        {
            ppWordsUtf8[i].reset(new char[MaxUtf8Length]);
            ppWordsUtf16[i].reset(new uint16_t[MaxUtf16Length]);
            if(!GetOneNgWord(ppWordsUtf8[i].get(), ppWordsUtf16[i].get(), MaxUtf8Length, MaxUtf16Length))
            {
                // NG ワードリストの単語が nn::ngc::WordcountMax より少ないなら、
                // 同じ単語を繰り返し入れて ppWords を埋める
                m_WordUtf8Iter = 0;
                m_WordUtf16Iter = 0;
                NN_ABORT_UNLESS(GetOneNgWord(ppWordsUtf8[i].get(), ppWordsUtf16[i].get(), MaxUtf8Length, MaxUtf16Length));
            }
        }

        // UTF-8
        {
            //NN_LOG("ppWords(UTF-8):\n");
            //for (int i = 0; i < nn::ngc::WordCountMax; ++i)
            //{
            //    NN_LOG("%s\n", ppWordsUtf8[i].get());
            //}

            MeasureCheckWordTimeCore(reinterpret_cast<char**>(ppWordsUtf8), pattern, 8);
        }
        // UTF-16
        {
            //NN_LOG("ppWords(UTF-16):\n");
            //for (int i = 0; i < nn::ngc::WordCountMax; ++i)
            //{
            //    char pConverted[MaxUtf8Length];
            //    auto encodingResult = nn::util::ConvertStringUtf16NativeToUtf8(
            //        pConverted, MaxUtf8Length, ppWordsUtf16[i].get());
            //    NN_ABORT_UNLESS(nn::util::CharacterEncodingResult_Success == encodingResult);
            //    NN_LOG("%s\n", pConverted);
            //}

            MeasureCheckWordTimeCore(reinterpret_cast<uint16_t**>(ppWordsUtf16), pattern, 16);
        }
    }

    /**
     * @brief   MaskProfanityWordsInText にかかる時間を測定します。
     */
    void MeasureMaskTime(int pattern)
    {
        const size_t MaxUtf8Length = nn::ngc::TextLengthMax * 4;    // UTF-8 文字列的にあり得る単語の最大サイズ
        const size_t MaxUtf16Length = nn::ngc::TextLengthMax * 2;   // UTF-16 文字列的にあり得る単語の最大サイズ
        char pNgWordsUtf8[MaxUtf8Length] = {};
        uint16_t pNgWordsUtf16[MaxUtf16Length] = {};

        std::string ngTextUtf8;         // NG ワードと半角スペースのみで構成される文字列(UTF-8)

        // 制限ぎりぎりの長い文字列を作る
        int textNum = 0;
        for (textNum = 0; textNum < nn::ngc::ProfanityFilterPatternList_Max; ++textNum)
        {
            if ((pattern & (1 << textNum)) != 0)
            {
                break;
            }
        }
        NN_ASSERT(textNum < nn::ngc::ProfanityFilterPatternList_Max);
        ASSERT_TRUE(ReadText(textNum));
        const int repeat = 256;   // NG ワードに登録されている文字が少なかった時のためにループしておく
        for (int i = 0; i < repeat; ++i)
        {
            while (GetOneNgWord(pNgWordsUtf8, pNgWordsUtf16, MaxUtf8Length, MaxUtf16Length))
            {
                if ((ngTextUtf8 + pNgWordsUtf8).length() + 1 >= nn::ngc::TextLengthMax)
                {
                    break;
                }
                if ((ngTextUtf8 + pNgWordsUtf8).length() + 1 < nn::ngc::TextLengthMax)
                {
                    ngTextUtf8 += pNgWordsUtf8;
                    ngTextUtf8 += ' ';
                }
            }
            m_WordUtf8Iter = 0;
        }
        ngTextUtf8 += '\0';

        memset(pNgWordsUtf16, 0, MaxUtf16Length * sizeof(uint16_t));
        auto encodingResult = nn::util::ConvertStringUtf8ToUtf16Native(
            pNgWordsUtf16, MaxUtf16Length, ngTextUtf8.c_str());
        NN_ABORT_UNLESS(nn::util::CharacterEncodingResult_Success == encodingResult);

        // UTF-8
        {
            //NN_LOG("Before:\n%s\n", ngTextUtf8.c_str());

            nn::os::Tick startTick = nn::os::GetSystemTick();
            auto result = m_Filter.MaskProfanityWordsInText(nullptr,
                const_cast<char*>(ngTextUtf8.c_str()), pattern);
            nn::os::Tick endTick = nn::os::GetSystemTick();
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            //NN_LOG("After:\n%s\n", ngTextUtf8.c_str());

            // 処理時間の表示
            nn::TimeSpan time = (endTick - startTick).ToTimeSpan();
            NN_LOG("Processing time(UTF- 8): %lld us (%lld ms)\n", time.GetMicroSeconds(), time.GetMilliSeconds());

        }
        // UTF-16
        {
            char pConverted[MaxUtf8Length] = {};
            encodingResult = nn::util::ConvertStringUtf16NativeToUtf8(
                pConverted, MaxUtf8Length, pNgWordsUtf16);
            NN_ABORT_UNLESS(nn::util::CharacterEncodingResult_Success == encodingResult);
            //NN_LOG("Before:\n%s\n", pConverted);

            nn::os::Tick startTick = nn::os::GetSystemTick();
            auto result = m_Filter.MaskProfanityWordsInText(nullptr, pNgWordsUtf16, pattern);
            nn::os::Tick endTick = nn::os::GetSystemTick();
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            encodingResult = nn::util::ConvertStringUtf16NativeToUtf8(
                pConverted, MaxUtf8Length, pNgWordsUtf16);
            NN_ABORT_UNLESS(nn::util::CharacterEncodingResult_Success == encodingResult);
            //NN_LOG("After:\n%s\n", pConverted);

            // 処理時間の表示
            nn::TimeSpan time = (endTick - startTick).ToTimeSpan();
            NN_LOG("Processing time(UTF-16): %lld us (%lld ms)\n", time.GetMicroSeconds(), time.GetMilliSeconds());

            // UTF-8 と UTF-16 の結果は同じである
            ASSERT_EQ(strcmp(ngTextUtf8.c_str(), pConverted), 0);
        }

        EXPECT_TRUE(CheckAsterisk(ngTextUtf8.c_str(), ngTextUtf8.length()));
        EXPECT_TRUE(CheckAsterisk(pNgWordsUtf16, Strlen16(pNgWordsUtf16)));
    }

protected:
    nn::ngc::ProfanityFilter m_Filter;                  // NG ワードチェック

private:
    nn::fs::FileHandle m_File;                          // ファイルハンドル
    void* m_pFsCache;                                   // fs 用バッファ
    void* m_pWorkMemory;                                // ProfanityFilter 用バッファ
    void* m_pText;                                      // チェック対象のテキスト
    uint16_t* m_pTextUtf16;                             // チェック対象のテキスト(UTF-16)(= &m_pText[1])
    char* m_pTextUtf8;                                  // チェック対象のテキスト(UTF-8)
    int m_WordUtf16Iter;                                // m_pText の走査インデックス
    int m_WordUtf8Iter;                                 // m_pTextUtf8 の走査インデックス
};

}   // namespace

TEST_F(NgcAllNgWordCheckTest, ApiTest)
{
    for (int i = 0; i < nn::ngc::ProfanityFilterPatternList_Max; ++i)
    {
        NN_LOG("Check %d.txt\n", i);
        DoApiTest((1 << i));
    }
}

TEST_F(NgcAllNgWordCheckTest, MesureCheckWordTest)
{
    for (int i = 0; i < nn::ngc::ProfanityFilterPatternList_Max; ++i)
    {
        NN_LOG("Measure %d.txt\n", i);
        MeasureCheckWordTime((1 << i));
    }
}

TEST_F(NgcAllNgWordCheckTest, MesureMaskTest)
{
    for (int i = 0; i < nn::ngc::ProfanityFilterPatternList_Max; ++i)
    {
        NN_LOG("Measure %d.txt\n", i);
        MeasureMaskTime((1 << i));
    }
}
