﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <nn/mii/mii_CharInfoAccessor.h>
#include <nn/util/util_CharacterEncoding.h>

#include "testNfp_Graphics.h"
#include "testNfp_Writer.h"

#define EOS(s) ((s) + std::strlen(s))

//-----------------------------------------------------------------------------

namespace {

struct CharacterString
{
    uint16_t numberingId;
    const char* string;
};

const CharacterString CharacterStringTable[] = {
    {  0, "Mario"},
    { 52, "Mario"},
    { 60, "Mario"},
    { 61, "Mario"},
    {568, "Mario"},
    {569, "Mario"},
    { 25, "Dr. Mario"},
    { 12, "Luigi"},
    { 53, "Luigi"},
    {  1, "Peach"},
    { 54, "Peach"},
    {  2, "Yoshi"},
    { 55, "Yoshi"},
    { 65, "Green Yarn Yoshi"},
    { 66, "Pink Yarn Yoshi"},
    { 67, "Light-Blue Yarn Yoshi"},
    {574, "Mega Yarn Yoshi"},
    {610, "Rosalina"},
    { 19, "Rosalina"},
    { 20, "Bowser"},
    { 57, "Bowser"},
    { 21, "Bowser Jr."},
    { 26, "Wario"},
    {611, "Wario"},
    {  3, "Donkey Kong"},
    {612, "Donkey Kong"},
    { 13, "Diddy Kong"},
    {613, "Diddy Kong"},
    { 56, "Toad"},
    {614, "Daisy"},
    {615, "Waluigi"},
    {616, "Boo"},
    {  4, "Link"},
    {843, "Link"},
    {847, "Link"},
    { 22, "Toon Link"},
    {848, "Toon Link"},
    { 14, "Zelda"},
    {850, "Zelda"},
    { 23, "Sheik"},
    { 27, "Ganondorf"},
    {591, "Wolf Link"},
    {  8, "Villager"},
    {587, "Isabelle"},
    {575, "Isabelle"},
    {576, "K.K."},
    {578, "Tom Nook"},
    {589, "Timmy & Tommy"},
    {577, "Mabel"},
    {581, "Reese"},
    {582, "Cyrus"},
    {579, "Digby"},
    {588, "Rover"},
    {585, "Resetti"},
    {583, "Blathers"},
    {584, "Celeste"},
    {586, "Kicks"},
    {590, "Kapp'n"},
    {580, "Lottie"},
    {  5, "Fox"},
    { 28, "Falco"},
    {  6, "Samus"},
    { 29, "Zero Suit Samus"},
    { 18, "Captain Falcon"},
    { 30, "Olimar"},
    { 15, "Little Mac"},
    {  7, "Wii Fit Trainer"},
    { 16, "Pit"},
    { 32, "Dark Pit"},
    { 31, "Palutena"},
    { 45, "Mr. Game & Watch"},
    { 46, "R.O.B."},
    { 51, "R.O.B."},
    { 47, "Duck Hunt"},
    { 33, "Mii Brawler"},
    { 34, "Mii Swordfighter"},
    { 35, "Mii Gunner"},
    { 62, "Inkling Girl"},
    {607, "Inkling Girl"},
    { 63, "Inkling Boy"},
    {608, "Inkling Boy"},
    { 64, "Inkling Squid"},
    {609, "Inkling Squid"},
    {605, "Callie"},
    {606, "Marie"},
    { 36, "Charizard"},
    {  9, "Pikachu"},
    { 38, "Jigglypuff"},
    {573, "Mewtwo"},
    { 17, "Lucario"},
    { 37, "Greninja"},
    { 10, "Kirby"},
    {596, "Kirby"},
    { 39, "Meta Knight"},
    {597, "Meta Knight"},
    { 40, "King Dedede"},
    {598, "King Dedede"},
    {599, "Waddle Dee"},
    { 11, "Marth"},
    { 24, "Ike"},
    { 41, "Lucina"},
    { 42, "Robin"},
    {594, "Roy"},
    { 43, "Shulk"},
    { 44, "Ness"},
    {593, "Lucas"},
    { 58, "Chibi-Robo"},
    { 48, "Sonic the Hedgehog"},
    { 50, "PAC-MAN"},
    { 49, "Mega Man"},
    {595, "Ryu"},
};

const char BlackSquare[] = {'\xe2', '\x96', '\xa0', '\x00'}; //U+25A0 "■"

char g_Buffer[256];

struct ProcessBuffer
{
    char process[64];
    bool isSuccess;
    char result[64];
};
ProcessBuffer g_ProcessBuffer[32];

}   // namespace

//-----------------------------------------------------------------------------

//
//  情報を表示
//
int WriteCaption(int x, int y, const char* caption)
{
    std::sprintf(g_Buffer, "%s %s", BlackSquare, caption);
    WriteText(x, y, g_Buffer, Color::Orange, false);
    y++;

    return y;
}

int WriteTagInfo(int x, int y, const nn::nfp::TagInfo& tagInfo)
{
    y = WriteCaption(x, y, "Tag Info");

    std::sprintf(g_Buffer, "Tag ID(hex) : ");
    for(uint8_t i = 0; i < tagInfo.tagId.length; ++i)
    {
        std::sprintf(EOS(g_Buffer), "%02x ", static_cast<uint8_t>(tagInfo.tagId.uid[i]));
    }
    WriteText(x + 1, y, g_Buffer);
    y++;

    std::sprintf(g_Buffer, "Nfc Protocol(hex) : ");
    const char* protocol;
    switch(tagInfo.protocol)
    {
    case nn::nfp::NfcProtocol_Error:
        protocol = "Error";
        break;
    case nn::nfp::NfcProtocol_TypeA:
        protocol = "TypeA";
        break;
    case nn::nfp::NfcProtocol_TypeB:
        protocol = "TypeB";
        break;
    case nn::nfp::NfcProtocol_TypeF:
        protocol = "TypeF";
        break;
    case nn::nfp::NfcProtocol_Type15693:
        protocol = "Type15693";
        break;
    default:
        protocol = "Unknown";
        break;
    }
    std::sprintf(EOS(g_Buffer), "%s(%08x)", protocol, tagInfo.protocol);
    WriteText(x + 1, y, g_Buffer);
    y++;

    std::sprintf(g_Buffer, "Tag Type : ");
    const char* type;
    switch(tagInfo.type)
    {
    case nn::nfp::TagType_Error:
        type = "Error";
        break;
    case nn::nfp::TagType_Type1:
        type = "Type1";
        break;
    case nn::nfp::TagType_Type2:
        type = "Type2";
        break;
    case nn::nfp::TagType_Type3:
        type = "Type3";
        break;
    case nn::nfp::TagType_Type4A:
        type = "Type4A";
        break;
    case nn::nfp::TagType_Type4B:
        type = "Type4B";
        break;
    case nn::nfp::TagType_Iso15693:
        type = "Iso15693";
        break;
    default:
        type = "Unknown";
        break;
    }
    std::sprintf(EOS(g_Buffer), "%s(%08x)", type, tagInfo.type);
    WriteText(x + 1, y, g_Buffer);
    y++;

    return y;
}


int WriteModelInfo(int x, int y, const nn::nfp::ModelInfo& modelInfo)
{
    y = WriteCaption(x, y, "Model Info");

    std::sprintf(g_Buffer, "Character ID(hex) : ");
    for(size_t i = 0; i < sizeof(modelInfo.characterId); ++i)
    {
        std::sprintf(EOS(g_Buffer), "%02x ", static_cast<nn::Bit8>(modelInfo.characterId[i]));
    }
    WriteText(x + 1, y, g_Buffer);
    y++;

    const char* seriesString;
    switch(modelInfo.seriesId)
    {
    case 0:
        seriesString = "Super Smash Bros.";
        break;
    case 1:
        seriesString = "Super Mario";
        break;
    case 2:
        seriesString = "Chibi-Robo";
        break;
    case 3:
        seriesString = "Yoshi's Woolly World";
        break;
    case 4:
        seriesString = "Splatoon";
        break;
    case 5:
        seriesString = "Animal Crossing";
        break;
    case 6:
        seriesString = "SUPER MARIO BROS. 30th";
        break;
    case 9:
        seriesString = "The Legend of Zelda";
        break;
    case 12:
        seriesString = "Kirby";
        break;
    default:
        seriesString = "Unknown";
        break;
    }
    std::sprintf(g_Buffer, "Series ID(hex) : %s(%02x)", seriesString, modelInfo.seriesId);
    WriteText(x + 1, y, g_Buffer);
    y++;

    const char* character = nullptr;
    for(const auto& characterString : CharacterStringTable)
    {
        if(characterString.numberingId == modelInfo.numberingId)
        {
            character = characterString.string;
            break;
        }
    }
    std::sprintf(g_Buffer, "Numbering ID : %s(%d)", (character ? character : "Unknown"), modelInfo.numberingId);
    WriteText(x + 1, y, g_Buffer);
    y++;

    const char* nfpTypeString;
    switch(modelInfo.nfpType)
    {
    case 0:
        nfpTypeString = "Character Figure";
        break;
    case 1:
        nfpTypeString = "Card";
        break;
    case 2:
        nfpTypeString = "Knitted Figure";
        break;
    default:
        nfpTypeString = "Unknown";
        break;
    }
    std::sprintf(g_Buffer, "Nfp Type(hex) : %s(%02x)", nfpTypeString, modelInfo.nfpType);
    WriteText(x + 1, y, g_Buffer);
    y++;

    return y;
}

int WriteCommonInfo(int x, int y, const nn::nfp::CommonInfo& commonInfo)
{
    y = WriteCaption(x, y, "Common Info");

    std::sprintf(g_Buffer, "Write Counter : %d", commonInfo.writeCounter);
    WriteText(x + 1, y, g_Buffer);
    y++;

    std::sprintf(g_Buffer, "Nfp Version : %d", commonInfo.nfpVersion);
    WriteText(x + 1, y, g_Buffer);
    y++;

    std::sprintf(g_Buffer, "Application Area Size : %d", commonInfo.applicationAreaSize);
    WriteText(x + 1, y, g_Buffer);
    y++;

    return y;
}

int WriteCharInfo(int x, int y, const nn::mii::CharInfo& charInfo)
{
    WriteText(x, y, "Char Info :", Color::Gray, false);
    y++;

    nn::mii::CharInfoAccessor mii(charInfo);

    std::sprintf(g_Buffer, "Is Valid : %s", (mii.IsValid() ? "true" : "false"));
    WriteText(x + 1, y, g_Buffer, Color::Gray, false);
    y++;

    if(mii.IsValid())
    {
        std::sprintf(g_Buffer, "Nickname :");
        WriteText(x + 1, y, g_Buffer, Color::Gray, false);
        y++;

        nn::mii::Nickname name;
        mii.GetNickname(&name, nn::mii::FontRegionFlag_JpUsEu);
        std::sprintf(g_Buffer, "Is Valid : %s", (name.IsValid() ? "true" : "false"));
        WriteText(x + 2, y, g_Buffer, Color::Gray, false);
        y++;

        if(name.IsValid())
        {
            std::sprintf(g_Buffer, "Name(hex) : ");
            if(nn::util::ConvertStringUtf16NativeToUtf8(EOS(g_Buffer), sizeof(g_Buffer) - static_cast<int>(EOS(g_Buffer) - g_Buffer), name.name) != nn::util::CharacterEncodingResult_Success)
            {
                std::sprintf(g_Buffer, "Name(hex) : ");
            }
            WriteText(x + 2, y, g_Buffer, Color::Gray, false);
            y++;

            std::sprintf(g_Buffer, "");
            for(int i = 0; i < sizeof(name.name) / sizeof(name.name[0]); i++)
            {
                nn::Bit16 data = static_cast<nn::Bit16>(name.name[i]);
                std::sprintf(EOS(g_Buffer), "%04x ", data);
                if(data == 0)
                {
                    break;
                }
            }
            WriteText(x + 3, y, g_Buffer, Color::Gray, false);
            y++;
        }

        {
            std::sprintf(g_Buffer, "Font Region : ");
            const char* regionString[] = {
                "JpUsEu",
                "China",
                "Korea",
                "Taiwan",
                "Unknown",
            };
            auto region = mii.GetFontRegion();
            auto index = (nn::mii::FontRegion_Min <= region && region <= nn::mii::FontRegion_Max ? region : nn::mii::FontRegion_Max + 1);
            std::sprintf(EOS(g_Buffer), "%s(%d)", regionString[index], region);
            WriteText(x + 1, y, g_Buffer, Color::Gray, false);
            y++;
        }

        {
            std::sprintf(g_Buffer, "Favorite Color : ");
            const char* colorString[] = {
                "Red",                       ///<  0 : 赤色
                "Orange",                    ///<  1 : 橙色
                "Yellow",                    ///<  2 : 黄色
                "YellowGreen",               ///<  3 : 黄緑色
                "Green",                     ///<  4 : 緑色
                "Blue",                      ///<  5 : 青色
                "SkyBlue",                   ///<  6 : 水色
                "Pink",                      ///<  7 : ピンク色
                "Purple",                    ///<  8 : 紫色
                "Brown",                     ///<  9 : 茶色
                "White",                     ///< 10 : 白色
                "Black",                     ///< 11, 黒色
                "Unknown",
            };
            auto color = mii.GetFavoriteColor();
            auto index = (nn::mii::FavoriteColor_Min <= color && color <= nn::mii::FavoriteColor_Max ? color : nn::mii::FavoriteColor_Max + 1);
            std::sprintf(EOS(g_Buffer), "%s(%d)", colorString[index], color);
            WriteText(x + 1, y, g_Buffer, Color::Gray, false);
            y++;
        }

        {
            std::sprintf(g_Buffer, "Gender : ");
            const char* genderString[] = {
                "Male",
                "Famale",
                "Unknown",
            };
            auto gender = mii.GetGender();
            auto index  = (nn::mii::Gender_Min <= gender && gender <= nn::mii::Gender_Max ? gender : nn::mii::Gender_Max + 1);
            std::sprintf(EOS(g_Buffer), "%s(%d)", genderString[index], gender);
            WriteText(x + 1, y, g_Buffer, Color::Gray, false);
            y++;
        }

        std::sprintf(g_Buffer, "Hight : %d\n", mii.GetHeight());
        WriteText(x + 1, y, g_Buffer, Color::Gray, false);
        y++;

        std::sprintf(g_Buffer, "Build : %d\n", mii.GetBuild());
        WriteText(x + 1, y, g_Buffer, Color::Gray, false);
        y++;

        std::sprintf(g_Buffer, "Is Hair Flip : %s\n", (mii.IsHairFlip() ? "true" : "false"));
        WriteText(x + 1, y, g_Buffer, Color::Gray, false);
        y++;

        std::sprintf(g_Buffer, "Hair Type : %d\n", mii.GetHairType());
        WriteText(x + 1, y, g_Buffer, Color::Gray, false);
        y++;

        std::sprintf(g_Buffer, "Faceline Type : %d\n", mii.GetFacelineType());
        WriteText(x + 1, y, g_Buffer, Color::Gray, false);
        y++;

        std::sprintf(g_Buffer, "Skin Color : %d\n", mii.GetSkinColor());
        WriteText(x + 1, y, g_Buffer, Color::Gray, false);
        y++;
    }

    return y;
} // NOLINT(impl/function_size)

int WriteRegisterInfo(int x, int y, bool hasRegisterInfo, const nn::nfp::RegisterInfo& registerInfo)
{
    y = WriteCaption(x, y, "Register Info");

    if(!hasRegisterInfo)
    {
        WriteText(x + 1, y, "Need Register");
        y++;
        return y;
    }

    WriteText(x + 1, y, "Mii Data(hex) :");
    y++;

    for(size_t i = 0; i < sizeof(registerInfo.miiData); ++i)
    {
        nn::Bit8 data = (reinterpret_cast<const nn::Bit8*>(&registerInfo.miiData))[i];
        if(i % 16 == 0)
        {
            std::sprintf(g_Buffer, "%02x ", data);
        }
        else
        {
            std::sprintf(EOS(g_Buffer), "%02x ", data);
        }

        if((i + 1) % 16 == 0 || (i + 1) == sizeof(registerInfo.miiData))
        {
            WriteTextFixedWidth(x + 2, y, g_Buffer);
            y++;
        }
    }

    y = WriteCharInfo(x + 2, y, registerInfo.miiData);

    std::sprintf(g_Buffer, "Nickname(hex) : ");
    if(std::memchr(registerInfo.nickname, 0, sizeof(registerInfo.nickname)) != nullptr)
    {
        std::sprintf(EOS(g_Buffer), "%s", registerInfo.nickname);
    }
    WriteText(x + 1, y, g_Buffer);
    y++;

    for(size_t i = 0; i < sizeof(registerInfo.nickname); ++i)
    {
        nn::Bit8 data = static_cast<nn::Bit8>(registerInfo.nickname[i]);
        if(i % 16 == 0)
        {
            std::sprintf(g_Buffer, "%02x ", data);
        }
        else
        {
            std::sprintf(EOS(g_Buffer), "%02x ", data);
        }

        if((i + 1) % 16 == 0 || (i + 1) == sizeof(registerInfo.nickname) || data == 0)
        {
            WriteTextFixedWidth(x + 2, y, g_Buffer);
            y++;

            if(data == 0)
            {
                break;
            }
        }
    }


    {
        std::sprintf(g_Buffer, "Font Region : ");
        const char* regionString[] = {
            "JpUsEu",
            "China",
            "Korea",
            "Taiwan",
            "Unknown",
        };
        auto region = registerInfo.fontRegion;
        auto index = (nn::nfp::FontRegion_Min <= region && region <= nn::nfp::FontRegion_Max ? region : nn::nfp::FontRegion_Max + 1);
        std::sprintf(EOS(g_Buffer), "%s(%d)", regionString[index], region);
        WriteText(x + 1, y, g_Buffer);
        y++;
    }


    std::sprintf(g_Buffer, "Register Date : %d/%02d/%02d\n", registerInfo.registerDate.year, registerInfo.registerDate.month, registerInfo.registerDate.day);
    WriteText(x + 1, y, g_Buffer);
    y++;

    return y;
}

int WriteApplicationArea(int x, int y, const void* buffer, size_t bufferSize)
{
    y = WriteCaption(x, y, "Application Area(hex)");

    for(size_t i = 0; i < bufferSize; ++i)
    {
        nn::Bit8 data = (reinterpret_cast<const nn::Bit8*>(buffer))[i];
        if(i % 16 == 0)
        {
            std::sprintf(g_Buffer, "%02x ", data);
        }
        else
        {
            std::sprintf(EOS(g_Buffer), "%02x ", data);
        }

        if((i + 1) % 16 == 0 || (i + 1) == bufferSize)
        {
            WriteTextFixedWidth(x + 1, y, g_Buffer);
            y++;
        }
    }

    return y;
}

void InitializeWriter()
{
    InitializeGraphicSystem();
    std::memset(g_ProcessBuffer, 0, sizeof(g_ProcessBuffer));
}

int GetNextProcessIndex()
{
    int index;
    for(index = 0; index < sizeof(g_ProcessBuffer) / sizeof(g_ProcessBuffer[0]); index++)
    {
        if(g_ProcessBuffer[index].process[0] == 0)
        {
            break;
        }
    }
    return index;
}

int GetNextResultIndex()
{
    int index;
    for(index = 0; index < sizeof(g_ProcessBuffer) / sizeof(g_ProcessBuffer[0]); index++)
    {
        if(g_ProcessBuffer[index].result[0] == 0)
        {
            break;
        }
    }
    return index;
}

void WriteProcess()
{
    static int count = 0;
    BeginText();
    for(int y = 0; y < sizeof(g_ProcessBuffer) / sizeof(g_ProcessBuffer[0]); y++)
    {
        if(g_ProcessBuffer[y].process[0] == 0)
        {
            break;
        }
        std::sprintf(g_Buffer, "%s", g_ProcessBuffer[y].process);
        WriteText(0, y, g_Buffer);
        nn::util::Unorm8x4 color;
        bool isFixedWidthEnabled;
        if(g_ProcessBuffer[y].result[0] == 0)
        {
            const char* progress[] = { "-", "\\", "|", "/" };
            color = Color::White;
            isFixedWidthEnabled = true;
            std::sprintf(g_Buffer, "[%s]", progress[(count / 10) % 4]);
            count++;
        }
        else
        {
            color = (g_ProcessBuffer[y].isSuccess ? Color::Green : Color::Red);
            isFixedWidthEnabled = false;
            std::sprintf(g_Buffer, "[%s]", g_ProcessBuffer[y].result);
        }
        WriteText(64, y, g_Buffer, color, isFixedWidthEnabled);
    }
    EndText();
}

void WriteProcess(const char* process)
{
    int index = GetNextProcessIndex();
    std::sprintf(g_ProcessBuffer[index].process, "%s", process);

    WriteProcess();
}

void WriteProcess(nn::Result result)
{
    int index = GetNextResultIndex();

    if(result.IsSuccess())
    {
        g_ProcessBuffer[index].isSuccess = true;
        std::sprintf(g_ProcessBuffer[index].result, "OK");
    }
    else
    {
        const char* r;
        g_ProcessBuffer[index].isSuccess = false;
        if(result <= nn::nfp::ResultNfcDeviceNotFound())
        {
            r = "NfcDeviceNotFound";
        }
        else if(result <= nn::nfp::ResultNfcDisabled())
        {
            r = "NfcDisabled";
        }
        else if(result <= nn::nfp::ResultNeedRetry())
        {
            r = "NeedRetry";
        }
        else if(result <= nn::nfp::ResultNeedRestart())
        {
            r = "NeedRestart";
        }
        else if(result <= nn::nfp::ResultNeedRegister())
        {
            r = "NeedRegister";
        }
        else if(result <= nn::nfp::ResultNeedCreate())
        {
            r = "NeedCreate";
        }
        else if(result <= nn::nfp::ResultNeedRestore())
        {
            r = "NeedRestore";
        }
        else if(result <= nn::nfp::ResultNeedFormat())
        {
            r = "NeedFormat";
        }
        else if(result <= nn::nfp::ResultAccessIdMisMatch())
        {
            r = "AccessIdMisMatch";
        }
        else if(result <= nn::nfp::ResultNotBroken())
        {
            r = "NotBroken";
        }
        else if(result <= nn::nfp::ResultAlreadyCreated())
        {
            r = "AlreadyCreated";
        }
        else if(result <= nn::nfp::ResultNotSupported())
        {
            r = "NotSupported";
        }
        else if(result <= nn::nfp::ResultInvalidFormatVersion())
        {
            r = "InvalidFormatVersion";
        }
        else if(result <= nn::nfp::ResultMaxNfcDeviceActivated())
        {
            r = "MaxNfcDeviceActivated";
        }
        else if(result <= nn::nfp::ResultConflictFunction())
        {
            r = "ConflictFunction";
        }
        else if(result <= nn::nfp::ResultNotUpdated())
        {
            r = "NotUpdated";
        }
        else
        {
            r = "NG";
        }

        std::sprintf(g_ProcessBuffer[index].result, "%s(Module :%d, Description :%d)", r, result.GetModule(), result.GetDescription());
    }

    WriteProcess();
}

void WriteProcess(nn::nfp::DeviceState deviceState)
{
    int index = GetNextResultIndex();

    const char* stateString[] = {
        "Init",
        "Search",
        "Active",
        "Deactive",
        "Mount",
        "Unexpected",
    };

    g_ProcessBuffer[index].isSuccess = false;
    std::sprintf(g_ProcessBuffer[index].result, "NG(DeviceState :%s)", stateString[deviceState]);

    WriteProcess();
}

void WriteProcess(const nn::nfp::TagInfo& tagInfo,
                  const nn::nfp::ModelInfo& modelInfo,
                  const nn::nfp::CommonInfo& commonInfo,
                  bool hasRegisterInfo, const nn::nfp::RegisterInfo& registerInfo,
                  const void* applicationArea, size_t applicationAreaSize)
{
    int y = 0;
    BeginText();
    y = WriteTagInfo(0, y, tagInfo);
    y = WriteModelInfo(0, y, modelInfo);
    y = WriteCommonInfo(0, y, commonInfo);
    y = WriteApplicationArea(0, y, applicationArea, applicationAreaSize);

    y = 0;
    y = WriteRegisterInfo(64, y, hasRegisterInfo, registerInfo);

    EndText();
}
