﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nfp.h>
#include <nn/oe.h>
#include <nn/mii.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Event.h>
#include <nn/mii/mii_StoreDataAccessor.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNfp_Common.h>

//================================================================================
// このテストで使用する定義です。
//================================================================================
namespace {
    const int    MiiNameLengthMax   = nn::mii::Nickname::Length;
    const uint16_t MiiNameNx[MiiNameLengthMax + 1]     =
    {
        'M', 'i', 'i', '_', 'N', 'X', '\0'
    };
    const uint16_t MiiNameCtr[MiiNameLengthMax + 1]    =
    {
        'M', 'i', 'i', '_', 'C', 'T', 'R', '\0'
    };
    const uint16_t MiiNameSnake[MiiNameLengthMax + 1]  =
    {
        'M', 'i', 'i', '_', 'S', 'N', 'A', 'K', 'E', '\0'
    };
    const uint16_t MiiNameWiiU[MiiNameLengthMax + 1]   =
    {
        'M', 'i', 'i', '_', 'W', 'i', 'i', 'U', '\0'
    };

    enum TestTragetPlatform
    {
        TestTragetPlatform_Wiiu,
        TestTragetPlatform_Ctr,
        TestTragetPlatform_Snake,
    };

    void DoTestCaseMiiSequence2(TestTragetPlatform targetPlatform) NN_NOEXCEPT
    {
        // タグ検知まで待機する
        NN_LOG("[LOG] ========== Waiting for Tag detection ... ========== \n");
        nnt::nfp::DoSearch();
        while(NN_STATIC_CONDITION(true))
        {
            bool activated = nnt::nfp::WaitForActivate();
            nn::nfp::State state = nnt::nfp::wrapper::GetState();
            nn::nfp::DeviceState deviceState = nnt::nfp::wrapper::GetDeviceState();

            if( activated &&
                state == nn::nfp::State_Init &&
                deviceState == nn::nfp::DeviceState_Active )
            {
                NN_LOG("[LOG] ========== Tag detected !! ========== \n");
                break;
            }
        }
        // 他のプラットフォームで作成したタグをマウントする
        nnt::nfp::DoMount();

        nn::nfp::RegisterInfo regInfo;
        nn::nfp::RegisterInfoPrivate regInfoSet;

        // マウントされたタグのRegisterInfoを取得する
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));

        // RegisterInfoからmiiネームを取得する
        nn::mii::Nickname nickname = {};
        nn::mii::CharInfoAccessor charInfo(regInfo.miiData);
        charInfo.GetNickname(&nickname,nn::mii::FontRegionFlag_All);

        // miiNameが書き込まれたプラットフォームと一致することを確認する
        if(targetPlatform == TestTragetPlatform_Wiiu)
        {
            EXPECT_TRUE( std::memcmp(nickname.name, MiiNameWiiU, sizeof(MiiNameWiiU)) == 0);
        }
        else if(targetPlatform == TestTragetPlatform_Ctr)
        {
            EXPECT_TRUE( std::memcmp(nickname.name, MiiNameCtr, sizeof(MiiNameCtr)) == 0);
        }
        else if(targetPlatform == TestTragetPlatform_Snake)
        {
            EXPECT_TRUE( std::memcmp(nickname.name, MiiNameSnake, sizeof(MiiNameSnake)) == 0);
        }
        else
        {
            EXPECT_TRUE( false );
        }

        // 変更したそのままのmiiデータをRegisterInfoPrivateに書込み
        nn::nfp::RegisterInfoPrivate outRegInfo = {};

        nnt::nfp::wrapper::GetRegisterInfo(&outRegInfo);
        std::memcpy(regInfoSet.nickname, outRegInfo.nickname, sizeof(regInfoSet.nickname));
        regInfoSet.fontRegion = outRegInfo.fontRegion;
        std::memcpy(&regInfoSet.miiData, &outRegInfo.miiData, sizeof(regInfoSet.miiData));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));

        // タグを書き込む
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Flush());

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    }

    void DoTestCaseMiiSequence4() NN_NOEXCEPT
    {
        // タグ検知まで待機する
        NN_LOG("[LOG] ========== Waiting for Tag detection ... ========== \n");
        nnt::nfp::DoSearch();
        while(NN_STATIC_CONDITION(true))
        {
            bool activated = nnt::nfp::WaitForActivate();
            nn::nfp::State state = nnt::nfp::wrapper::GetState();
            nn::nfp::DeviceState deviceState = nnt::nfp::wrapper::GetDeviceState();

            if( activated &&
                state == nn::nfp::State_Init &&
                deviceState == nn::nfp::DeviceState_Active )
            {
                NN_LOG("[LOG] ========== Tag detected !! ========== \n");
                break;
            }
        }
        // 新規タグを作成する
        nnt::nfp::DoCreateZeroTag();

        //miinameを指定してタグに書き込む
        nn::nfp::RegisterInfoPrivate regInfoSet;
        nnt::nfp::BuildMiiData(regInfoSet.miiData);
        regInfoSet.fontRegion  = nn::nfp::FontRegion_China;
        std::memcpy(regInfoSet.nickname, "Nintendo", sizeof("Nintendo"));

        nn::mii::Nickname nickname;
        nickname.Set(MiiNameNx);
        nn::mii::StoreDataAccessor storeData(&regInfoSet.miiData);
        storeData.SetNickname(nickname,nn::mii::FontRegion_JpUsEu);

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));
        nnt::nfp::wrapper::Flush();

        // MiiNameが正しく書き込まれていることを確認する

        // マウントされたタグのRegisterInfoを取得する
        nn::nfp::RegisterInfo regInfo;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));

        nn::mii::CharInfoAccessor charInfo(regInfo.miiData);
        charInfo.GetNickname(&nickname,nn::mii::FontRegionFlag_All);

        EXPECT_TRUE( std::memcmp(nickname.name, MiiNameNx, sizeof(nickname.name)) == 0);

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    }

    void DoTestCaseMiiSequence6() NN_NOEXCEPT
    {
        // タグ検知まで待機する
        NN_LOG("[LOG] ========== Waiting for Tag detection ... ========== \n");
        nnt::nfp::DoSearch();
        while(NN_STATIC_CONDITION(true))
        {
            bool activated = nnt::nfp::WaitForActivate();
            nn::nfp::State state = nnt::nfp::wrapper::GetState();
            nn::nfp::DeviceState deviceState = nnt::nfp::wrapper::GetDeviceState();

            if( activated &&
                state == nn::nfp::State_Init &&
                deviceState == nn::nfp::DeviceState_Active )
            {
                NN_LOG("[LOG] ========== Tag detected !! ========== \n");
                break;
            }
        }

        // タグを読み込みする
        nnt::nfp::DoMount();

        // マウントされたタグのRegisterInfoを取得する
        nn::nfp::RegisterInfo regInfo;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));

        nn::mii::Nickname nickname;
        nn::mii::CharInfoAccessor charInfo(regInfo.miiData);
        charInfo.GetNickname(&nickname,nn::mii::FontRegionFlag_All);

        // タグのmiinameがNXで設定した値と一致することを確認する
        EXPECT_TRUE( std::memcmp(nickname.name, MiiNameNx, sizeof(nickname.name)) == 0);

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    }
}

class NfpMii : public nnt::nfp::TestFramework
{
protected:

    NfpMii() NN_NOEXCEPT
    {
        // oeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::nfp::InitializeHidController();
    }

    ~NfpMii() NN_NOEXCEPT
    {
    }


    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
    }
};

//================================================================================
// テストスイート全体で共通の処理です。
//================================================================================

//================================================================================
// テストケースの実装です。
//================================================================================

TEST_F(NfpMii, TestCaseMiiSequence2ForWiiu)
{
    DoTestCaseMiiSequence2(TestTragetPlatform_Wiiu);
}

TEST_F(NfpMii, TestCaseMiiSequence2ForCtr)
{
    DoTestCaseMiiSequence2(TestTragetPlatform_Ctr);
}

TEST_F(NfpMii, TestCaseMiiSequence2ForSnake)
{
    DoTestCaseMiiSequence2(TestTragetPlatform_Snake);
}

TEST_F(NfpMii, TestCaseMiiSequence4)
{
    DoTestCaseMiiSequence4();
}

TEST_F(NfpMii, TestCaseMiiSequence6)
{
    DoTestCaseMiiSequence6();
}
