﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nfp.h>
#include <nn/hid.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_SdkThread.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/applet/applet.h>
#include <nn/ae/ae.h>

#include <nn/hid/hid_KeyboardKey.h>
#include <nn/settings/settings_DebugPad.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNfp_Common.h>

#include "testNfp_TagAccessor.h"
#include "testNfp_MenuRender.h"

//================================================================================
// このテストで使用する定義です。
//================================================================================
namespace
{

    //================================================================================
    //  このファイル内で定義される関数の前方宣言です。
    //================================================================================

#if !defined(NNT_NFP_PLATFORM_NX) //NX では必要としない
        nn::applet::AppletQueryReply SleepQueryCallback(void* pPtr);

        void AwakeCallback(void* pPtr);
#endif //!defined(NNT_NFP_PLATFORM_NX)

    bool Update(const nn::hid::DebugPadButtonSet padButtonDown,
                const nn::nfp::State state,
                const nn::nfp::DeviceState deviceState,
                const bool nfcState) NN_NOEXCEPT;

    void DrawText(MenuRenderSystem* pRenderSystem,
                  const int x, const int y, const char* pText) NN_NOEXCEPT;

    void DrawState(MenuRenderSystem* pRenderSystem, int x, int y,
                   nn::nfp::State state) NN_NOEXCEPT;

    void DrawDeviceState(MenuRenderSystem* pRenderSystem, int x, int y,
                         nn::nfp::DeviceState deviceState) NN_NOEXCEPT;

    void DrawTagState(MenuRenderSystem* pRenderSystem,
                      int x, int y, nn::Bit32 tagState) NN_NOEXCEPT;

    void DrawLastResult(MenuRenderSystem* pRenderSystem, int x, int y) NN_NOEXCEPT;

    void DrawMenu(MenuRenderSystem* pRenderSystem, int x, int y) NN_NOEXCEPT;

    void DrawTagInfo(MenuRenderSystem* pRenderSystem, int x, int y,
                     const TagData& tagData) NN_NOEXCEPT;

    void DrawCommonInfo(MenuRenderSystem* pRenderSystem, int x, int y,
                        const TagData& tagData) NN_NOEXCEPT;

    void DrawRegisterInfo(MenuRenderSystem* pRenderSystem, int x, int y,
                          const TagData& tagData) NN_NOEXCEPT;

    void DrawAdminInfo(MenuRenderSystem* pRenderSystem, int x, int y,
                       const TagData& tagData) NN_NOEXCEPT;

    void DrawApplicationArea(MenuRenderSystem* pRenderSystem, int x, int y,
                             const TagData& tagData) NN_NOEXCEPT;

    void DrawTab(MenuRenderSystem* pRenderSystem, int x, int y,
                 const TagData& tagData) NN_NOEXCEPT;

    void DrawManual(MenuRenderSystem* pRenderSystem, int x, int y) NN_NOEXCEPT;

    void DrawUpper(MenuRenderSystem* pRenderSystem,
                   const nn::nfp::State state,
                   const nn::nfp::DeviceState deviceState,
                   const bool nfcState,
                   const TagData& tagData) NN_NOEXCEPT;

    void DrawLower(MenuRenderSystem* pRenderSystem,
                   const nn::nfp::State state,
                   const nn::nfp::DeviceState deviceState,
                   const TagData& tagData) NN_NOEXCEPT;

    void NfpManagerExMainThread(void* pArg) NN_NOEXCEPT;

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::Result MoveBackGround() NN_NOEXCEPT;

    nn::Result MoveBackGroundAndInit() NN_NOEXCEPT;

    void NfpStartMiiEdit() NN_NOEXCEPT;

    nn::applet::LibraryAppletHandle PrepareMiiEditLibraryAppletImpl(
            const char* pInData, size_t inDataSize) NN_NOEXCEPT;

#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)
// 起床用のイベントです。
#if !defined(NNT_NFP_PLATFORM_NX) //NX では必要としない
    nn::os::EventType g_AwakeEvent;
#endif //!defined(NNT_NFP_PLATFORM_NX)

//================================================================================
//  画面描画に関するパラメータの設定です。
//================================================================================

    const int      FontSize                     = 8;
    const int      LineHeight                   = 8;
    const int      LeftLength                   = 14;
    const int      MiddleLength                 = 2;
    const float    LineWidth                    = 1.0f;
    const float    BackgroundColorRed           = 0.0f;
    const float    BackgroundColorGreen         = 0.0f;
    const float    BackgroundColorBlue          = 0.0f;
    const float    FontActiveColorRed           = 1.0f;
    const float    FontActiveColorGreen         = 1.0f;
    const float    FontActiveColorBlue          = 1.0f;
    const float    FontInactiveColorRed         = 0.5f;
    const float    FontInactiveColorGreen       = 0.5f;
    const float    FontInactiveColorBlue        = 0.5f;

    const int      TextBufferSize               = 256;
    char           g_Buffer[TextBufferSize];

    const int      RenderSystemMemorySize     = 8 * 1024 * 1024;
    char           g_RenderSystemMemory[RenderSystemMemorySize];

    const int      DispatchSleepTime            = 50;

//================================================================================
//  メニュー関連のパラメータです。
//================================================================================

    typedef nn::Result (*MenuItemHandler)();

    struct MenuItem
    {
        const char*     pText;                                  // メニューに表示する文字列です。
        MenuItemHandler handler;                               // 選択されたときに実行される関数です。
    };

    const MenuItem MenuItems[] =                              // メニュー項目です。
    {
        { "Initialize",                     NfpInitialize },
        { "ListDevices",                    NfpListDevices },
        { "StartDetection",                 NfpStartDetection },
        { "Mount",                          NfpMountDirect },
#if defined(NNT_NFP_LIB_VERSION_ALPHA)
        { "Restore",                        NfpRestore },
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA)
        { "Format",                         NfpFormat },
        { "CreateApplicationArea",          NfpCreateApplicationArea },
        { "SetApplicationArea (*1)",        NfpSetApplicationArea },
#if defined(NNT_NFP_PLATFORM_NX)
        { "RecreateApplicationArea",        NfpRecreateApplicationArea },
#endif // defined(NNT_NFP_PLATFORM_NX)
        { "DeleteApplicationArea",          NfpDeleteApplicationArea },
        { "SetNfpRegisterInfo (*1)(*2)",    NfpSetRegisterInfo },
        { "DeleteNfpRegisterInfo",          NfpDeleteRegisterInfo },
        { "Flush",                          NfpFlush },
        { "StopDetection",                  NfpStopDetection },
        { "UnMount",                        NfpUnmount },
#if defined(NNT_NFP_LIB_VERSION_BETA)
        { "WiFi On/Off",                    NfpWifiSwitch },
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
        { "DeleteSystemSaveData",           NfpDeleteSystemSaveData },
        { "CheckBackup",                    CheckBackup },
        { "Finalize" ,                      NfpFinalize },
#if defined(NNT_NFP_LIB_VERSION_BETA)
        { "StartNicknameAndOwnerSettings",           NfpStartNicknameAndOwnerSettings },
        { "StartGameDataEraser",                     NfpStartGameDataEraser },
        { "StartRestorer",                           NfpStartRestorer },
        { "StartNicknameAndOwnerSettings(private)",  NfpStartNicknameAndOwnerSettingsPrivate },
        { "StartGameDataEraser(private)",            NfpStartGameDataEraserPrivate },
        { "StartRestorer(private)",                  NfpStartRestorerPrivate },
        { "StartFormatter(private)",                 NfpStartFormatterPrivate },
        { "BreakTag",                       NfpBreakTag},
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        { "MoveBackGround",                 MoveBackGround},
        { "MoveBackGround and Init",     MoveBackGroundAndInit},
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
    };
    const int      MenuCount          = sizeof(MenuItems) / sizeof(MenuItem);

    int            g_SelectedMenuIndex = 0;                    // カーソルで選択中のメニューです。
    int            g_LastMenuIndex     = -1;                   // 最後に実行したメニューです。
    nn::nfp::State g_LastState;                                // 最後に確認したStateです。
    nn::nfp::DeviceState g_LastDeviceState;                    // 最後に確認したDeviceStateです。
    bool           g_LastNfcState;                             // 最後に確認したNfcStateです。
    nn::Result     g_LastResult;                               // 最後に実行したメニューの結果です。

//================================================================================
//  タグの内容を表示するためのタブ関連のパラメータです。
//================================================================================

    typedef void (*DrawTabHandler)(MenuRenderSystem*, int, int, const TagData&);

    struct DrawTabItem
    {
        const char*     pText;                                  // タブの名前です。
        DrawTabHandler  handler;                               // タブの描画関数です。
    };

    const DrawTabItem DrawTabItems[] =                       // タブ項目です。
    {
        { "info",        DrawTagInfo },
        { "admin",       DrawAdminInfo },
        { "common",      DrawCommonInfo },
        { "register",    DrawRegisterInfo },
        { "application", DrawApplicationArea }
    };
    const int      TabCount           = sizeof(DrawTabItems) / sizeof(DrawTabItem);

    const int      TabMargin          = 1;                    // タブの間隔です。
    int            g_SelectedTabIndex  = 0;                    // 選択中のタブです

//================================================================================
//  スレッド用のパラメータです。
//================================================================================
    // スレッドスタックサイズ
    const size_t ThreadStackSize = 32 * 1024;
    // スレッドタイプ
    nn::os::ThreadType  g_Thread = {};
    // スレッドのスタック
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ ThreadStackSize ];
    // スレッド生成フラグ
    bool g_ThreadCreateFlg = false;
    // プログラム終了フラグ
    bool g_NfpManagerExEndFlg = false;

//================================================================================
//  バックグラウンドテスト用パラメータです。
//================================================================================
    enum BackGroundTest
    {
        BackGroundTest_None = 0,
        BackGroundTest_Initialize = 1,
    };
    BackGroundTest g_BackGroundTestFlg = BackGroundTest_None;

#if !defined(NNT_NFP_PLATFORM_NX) //NX では必要としない
    //================================================================================
    //! @brief      スリープ通知を受け取るコールバックです。
    //! @param[in]  arg          コールバックの指定時に設定した引数です。このツールでは利用しません。
    //! @return     スリープ要求に対する応答です。
    //================================================================================
    nn::applet::AppletQueryReply SleepQueryCallback(void* pPtr) NN_NOEXCEPT
    {
        NN_UNUSED(arg);
        if ( !nn::applet::IsActive() )
        {
            // 他のアプレットやメニューに制御が移っている場合は即座にスリープを受け入れます。
            return nn::applet::REPLY_ACCEPT;
        }
        else
        {
            // それ以外のケースでは必要な処理を終えてからスリープするため一旦保留とします。
            // タグへの書き込み中にスリープ状態に入ると、タグのデータが壊れます。
            return nn::applet::REPLY_LATER;
        }
    }

    //================================================================================
    //! @brief      スリープからの復帰通知を受け取るコールバックです。
    //! @param[in]  arg          コールバックの指定時に設定した引数です。このツールでは利用しません。
    //================================================================================
    void AwakeCallback(void* pPtr) NN_NOEXCEPT
    {
        NN_UNUSED(arg);
        g_AwakeEvent.Signal();
    }
#endif //!defined(NNT_NFP_PLATFORM_NX)

    //================================================================================
    //! @brief      キーボードのキーを DebugPad のボタンに割り当てます。
    //================================================================================
    void InitializeHidDevices() NN_NOEXCEPT
    {
        nn::hid::InitializeDebugPad();

        //キーボードのキーを DebugPad のボタンに割り当てます。
        nn::settings::DebugPadKeyboardMap map;
        nn::settings::GetDebugPadKeyboardMap(&map);
        map.buttonA     = nn::hid::KeyboardKey::A::Index;
        map.buttonB     = nn::hid::KeyboardKey::B::Index;
        map.buttonX     = nn::hid::KeyboardKey::X::Index;
        map.buttonY     = nn::hid::KeyboardKey::Y::Index;
        map.buttonL     = nn::hid::KeyboardKey::L::Index;
        map.buttonR     = nn::hid::KeyboardKey::R::Index;
        map.buttonLeft  = nn::hid::KeyboardKey::LeftArrow::Index;
        map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
        map.buttonUp    = nn::hid::KeyboardKey::UpArrow::Index;
        map.buttonDown  = nn::hid::KeyboardKey::DownArrow::Index;
        map.buttonStart = nn::hid::KeyboardKey::Return::Index;
        nn::settings::SetDebugPadKeyboardMap(map);
    }

    //================================================================================
    //! @brief      状態更新用の関数です。
    //! @param[in]  padStatus     パッドの入力状態です。
    //! @param[in]  nfpState      NFP ライブラリの状態です。
    //================================================================================
    bool Update(nn::hid::DebugPadButtonSet padButtonDown,
                const nn::nfp::State state,
                const nn::nfp::DeviceState deviceState,
                const bool nfcState) NN_NOEXCEPT
    {
        bool isUpdate = false;

        // StateおよびDeviceStateを更新します。
        if( g_LastState != state
            || g_LastDeviceState != deviceState
            || g_LastNfcState != nfcState)
        {
            g_LastState = state;
            g_LastDeviceState = deviceState;
            g_LastNfcState = nfcState;
            isUpdate = true;
        }

        // 上下キーでメニューを選択します。
        if (padButtonDown.Test<::nn::hid::DebugPadButton::Up>())
        {
            g_SelectedMenuIndex = (g_SelectedMenuIndex + MenuCount - 1) % MenuCount;
            isUpdate = true;
        }
        else if (padButtonDown.Test<::nn::hid::DebugPadButton::Down>())
        {
            g_SelectedMenuIndex = (g_SelectedMenuIndex + 1) % MenuCount;
            isUpdate = true;
        }

        // 左右キーでタブを選択します。
        if (padButtonDown.Test<::nn::hid::DebugPadButton::Left>())
        {
            g_SelectedTabIndex = (g_SelectedTabIndex + TabCount - 1) % TabCount;
            isUpdate = true;
        }
        else if (padButtonDown.Test<::nn::hid::DebugPadButton::Right>())
        {
            g_SelectedTabIndex = (g_SelectedTabIndex + 1) % TabCount;
            isUpdate = true;
        }

        // A ボタンで選択したメニューコマンドを実行します。
        if (padButtonDown.Test<::nn::hid::DebugPadButton::A>())
        {
            NNT_PRINT_LOG("RUN COMMAND :  %s \n", MenuItems[g_SelectedMenuIndex].pText);
            g_LastResult    = MenuItems[g_SelectedMenuIndex].handler();
            g_LastMenuIndex = g_SelectedMenuIndex;

            if( g_LastResult.IsFailure() )
            {
                NNT_PRINT_LOG("Last Result Failure:  Module=%d Description=%d\n",
                       g_LastResult.GetModule(),g_LastResult.GetDescription());
            }
            isUpdate = true;
        }
        return isUpdate;
    }

    //================================================================================
    //! @brief      指定された座標に文字列を描画します。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  x             X 座標です。
    //! @param[in]  y             Y 座標です。
    //! @param[in]  text          描画する文字列です。
    //================================================================================
    void DrawText(MenuRenderSystem* pRenderSystem, const int x, const int y, const char* pText) NN_NOEXCEPT
    {
        pRenderSystem->DrawText(FontSize * x, LineHeight * y, pText);
    }

    //================================================================================
    //! @brief      指定された座標に文字列のセットを描画します。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  x             X 座標です。
    //! @param[in]  y             Y 座標です。
    //! @param[in]  text1         項目名です。
    //! @param[in]  text2         その項目の値です。
    //================================================================================
    void DrawTextSet(MenuRenderSystem* pRenderSystem, int x, int y,
                     const char* pText1, const char* pText2) NN_NOEXCEPT
    {
        const int left   = x;
        const int mid    = x + LeftLength;
        const int right  = mid + MiddleLength;

        if (pText1 != nullptr)
        {
            DrawText(pRenderSystem, left,  y, pText1);
            DrawText(pRenderSystem, mid,   y, ":");
        }
        if (pText2 != nullptr)
        {
            DrawText(pRenderSystem, right, y, pText2);
        }
    }

    //================================================================================
    //! @brief      NFP ライブラリの状態を描画する関数です。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  x             X 座標です。
    //! @param[in]  y             Y 座標です。
    //! @param[in]  state      NFP ライブラリの状態です。
    //================================================================================
    void DrawState(MenuRenderSystem* pRenderSystem,
                      int x, int y,
                      nn::nfp::State state) NN_NOEXCEPT
    {
        const char* stateText = nullptr;

        switch (state)
        {
        case nn::nfp::State_None:
            {
                stateText = "None";
            }
            break;
        case nn::nfp::State_Init:
            {
                stateText = "Init";
            }
            break;
        case nn::nfp::State_Unexpected:
            {
                stateText = "Unexpected";
            }
            break;
        default:
            {
                stateText = "not support";
            }
            break;
        }
        DrawTextSet(pRenderSystem, x, y, "STATE", stateText);

    }

    //================================================================================
    //! @brief      NFPデバイスの状態を描画する関数です。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  x             X 座標です。
    //! @param[in]  y             Y 座標です。
    //! @param[in]  deviceState      NFP デバイスの状態です。
    //================================================================================
    void DrawDeviceState(MenuRenderSystem* pRenderSystem,
                      int x, int y,
                      nn::nfp::DeviceState deviceState) NN_NOEXCEPT
    {
        const char* stateText = nullptr;

        switch (deviceState)
        {
        case nn::nfp::DeviceState_Init:
            {
                stateText = "Init";
            }
            break;
        case nn::nfp::DeviceState_Search:
            {
                stateText = "Search";
            }
            break;
        case nn::nfp::DeviceState_Active:
            {
                stateText = "Active";
            }
            break;
        case nn::nfp::DeviceState_Deactive:
            {
                stateText = "Deactive";
            }
            break;
        case nn::nfp::DeviceState_Mount:
            {
                stateText = "Mount";
            }
            break;
        case nn::nfp::DeviceState_Unexpected:
            {
                stateText = "Unexpected";
            }
            break;
        default:
            {
                stateText = "not support";
            }
            break;
        }
        DrawTextSet(pRenderSystem, x, y, "DEVICE STATE", stateText);
    }

    //================================================================================
    //! @brief      NFC機能の状態を描画する関数です。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  x             X 座標です。
    //! @param[in]  y             Y 座標です。
    //! @param[in]  nfcState      NFC機能の状態です。
    //================================================================================
    void DrawNfcState(MenuRenderSystem* pRenderSystem,
                      int x, int y,
                      bool nfcState) NN_NOEXCEPT
    {
        const char* stateText = nullptr;

        if(nfcState == true)
        {
            stateText = "Enable";
        }
        else
        {
            stateText = "Disable";
        }
        DrawTextSet(pRenderSystem, x, y, "NFC STATE", stateText);
    }

    //================================================================================
    //! @brief      NFP タグの状態を描画する関数です。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  x             X 座標です。
    //! @param[in]  y             Y 座標です。
    //! @param[in]  tagState      NFP タグの状態です。
    //================================================================================
    void DrawTagState(MenuRenderSystem* pRenderSystem,
                      int x, int y, nn::Bit32 tagState) NN_NOEXCEPT
    {
        const char* stateText = nullptr;
        switch (tagState)
        {
        case TagState_None:
            {
                stateText = "NONE";
            }
            break;
        case TagState_Good:
            {
                stateText = "GOOD";
            }
            break;
        case TagState_NotSupported:
            {
                stateText = "NOT SUPPORTED";
            }
            break;
        case TagState_InvalidFormatVersion:
            {
                stateText = "INVALID FORMAT VERSION";
            }
            break;
        case TagState_NeedRestore:
            {
                stateText = "NEED RESTORE";
            }
            break;
        case TagState_NeedFormat:
            {
                stateText = "NEED FORMAT";
            }
            break;
        default:
            {
                stateText = "UNEXPECTED";
            }
            break;
        }
        DrawTextSet(pRenderSystem, x, y, "TAG STATE", stateText);
    }

    //================================================================================
    //! @brief      現在選択されているコマンドの名前を描画します。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  x             X 座標です。
    //! @param[in]  y             Y 座標です。
    //================================================================================
    void DrawSelectedCommand(MenuRenderSystem* pRenderSystem, int x, int y) NN_NOEXCEPT
    {
        DrawTextSet(pRenderSystem, x, y, "NEXT COMMAND", MenuItems[g_SelectedMenuIndex].pText);
    }

    //================================================================================
    //! @brief      最後に実行したコマンドの結果を描画します。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  x             X 座標です。
    //! @param[in]  y             Y 座標です。
    //================================================================================
    void DrawLastResult(MenuRenderSystem* pRenderSystem, int x, int y) NN_NOEXCEPT
    {
        const char* pResultString;
        if (g_LastMenuIndex < 0)
        {
            pResultString = nullptr;
        }
        else
        {
            pResultString = nnt::nfp::GetNfpResultTypeString(g_LastResult);
        }

        DrawTextSet(pRenderSystem, x, y, "LAST RESULT", pResultString);
    }

    //================================================================================
    //! @brief      メニューを描画します。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  x             X 座標です。
    //! @param[in]  y             Y 座標です。
    //================================================================================
    void DrawMenu(MenuRenderSystem* pRenderSystem, int x, int y) NN_NOEXCEPT
    {
        for (int i = 0; i < MenuCount; ++i)
        {
            // 選択中か否かによってメニューの描画色を切り替えます。
            if (i == g_SelectedMenuIndex)
            {
                pRenderSystem->SetColor(FontActiveColorRed,
                                        FontActiveColorGreen,
                                        FontActiveColorBlue,
                                        1.0f);
#ifdef NNT_NFP_MENU_CONSOLE // メニューでコンソール出力する場合は先頭に ">"をいれる
                DrawText(pRenderSystem, 0, y + i, ">");
#endif
            }
            else
            {
                pRenderSystem->SetColor(FontInactiveColorRed,
                                        FontInactiveColorGreen,
                                        FontInactiveColorBlue,
                                        1.0f);
            }

            // メニュー項目を描画します。
            DrawText(pRenderSystem, x, y + i, MenuItems[i].pText);
        }
    }

    //================================================================================
    //! @brief      nn::nfp::TagInfo の内容を画面に表示します。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  x             X 座標です。
    //! @param[in]  y             Y 座標です。
    //! @param[in]  tagData       タグに含まれる情報です。
    //================================================================================
    void DrawTagInfo(MenuRenderSystem* pRenderSystem, int x, int y,
                     const TagData& tagData) NN_NOEXCEPT
    {
        if (!tagData.hasTagInfo)
        {
            DrawText(pRenderSystem, x, y, "** ERROR **");
            return;
        }

        const nn::nfp::TagId& tagId = tagData.tagInfo.tagId;
        for (int i = 0; i < tagId.length && i < nn::nfp::UidLengthMax; ++i)
        {
            std::sprintf(g_Buffer + i * 3, "%02X ", tagId.uid[i]);
        }
        DrawTextSet(pRenderSystem, x, y++, "UID", g_Buffer);

        std::sprintf(g_Buffer, "%d", tagData.tagInfo.protocol);
        DrawTextSet(pRenderSystem, x, y++, "PROTOCOL", g_Buffer);

        std::sprintf(g_Buffer, "%d", tagData.tagInfo.type);
        DrawTextSet(pRenderSystem, x, y++, "TYPE", g_Buffer);

        std::sprintf(g_Buffer, "%d", tagData.systemWriteCounter);
        DrawTextSet(pRenderSystem, x, y++, "sysWrCounter", g_Buffer);
    }

    //================================================================================
    //! @brief      nn::nfp::CommonInfo の内容を画面に表示します。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  x             X 座標です。
    //! @param[in]  y             Y 座標です。
    //! @param[in]  tagData       タグに含まれる情報です。
    //================================================================================
    void DrawCommonInfo(MenuRenderSystem* pRenderSystem, int x, int y,
                        const TagData& tagData) NN_NOEXCEPT
    {
        if (!tagData.hasCommonInfo)
        {
            DrawText(pRenderSystem, x, y, "** ERROR **");
            return;
        }
        const nn::nfp::CommonInfo& info = tagData.commonInfo;

#if defined(NNT_NFP_LIB_VERSION_BETA)
        std::sprintf(g_Buffer, "%04d-%02d-%02d", info.lastWriteDate.year,
                                                 info.lastWriteDate.month,
                                                 info.lastWriteDate.day);
        DrawTextSet(pRenderSystem, x, y++, "LAST WRITE", g_Buffer);
#endif //defined(NNT_NFP_LIB_VERSION_BETA)
        std::sprintf(g_Buffer, "%d", info.writeCounter);
        DrawTextSet(pRenderSystem, x, y++, "WRITE COUNTER", g_Buffer);

        std::sprintf(g_Buffer, "%d", info.nfpVersion);
        DrawTextSet(pRenderSystem, x, y++, "NFP VERSION", g_Buffer);

        std::sprintf(g_Buffer, "%d Bytes", info.applicationAreaSize);
        DrawTextSet(pRenderSystem, x, y++, "APP AREA SIZE", g_Buffer);
    }

    //================================================================================
    //! @brief      nn::nfp::RegisterInfo の内容を画面に表示します。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  x             X 座標です。
    //! @param[in]  y             Y 座標です。
    //! @param[in]  tagData       タグに含まれる情報です。
    //================================================================================
    void DrawRegisterInfo(MenuRenderSystem* pRenderSystem, int x, int y,
                          const TagData& tagData) NN_NOEXCEPT
    {
        if (!tagData.hasRegisterInfo)
        {
            DrawText(pRenderSystem, x, y, "** ERROR **");
            return;
        }
        const nn::nfp::RegisterInfoPrivate& info = tagData.registerInfo;

        std::sprintf(g_Buffer, "%d", info.fontRegion);
        DrawTextSet(pRenderSystem, x, y++, "FONT REGION", g_Buffer);

#if !defined(NNT_NFP_PLATFORM_NX) //NX には国コードがない
        std::sprintf(g_Buffer, "%d", info.country);
        DrawTextSet(pRenderSystem, x, y++, "COUNTRY", g_Buffer);
#endif //!defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_LIB_VERSION_BETA)
        std::sprintf(g_Buffer, "%04d-%02d-%02d", info.registerDate.year,
                                                 info.registerDate.month,
                                                 info.registerDate.day);
        DrawTextSet(pRenderSystem, x, y++, "REGISTER DATE", g_Buffer);
#endif//defined(NNT_NFP_LIB_VERSION_BETA)

        const int NameBreakAt = (sizeof(info.nickname) / sizeof(info.nickname[0]) + 1) / 2;
        for (int i = 0; i < NameBreakAt; ++i)
        {
            std::sprintf(g_Buffer + i * 3, "%02X ", info.nickname[i]);
        }
        DrawTextSet(pRenderSystem, x, y++, "NICKNAME", g_Buffer);
        for (int i = NameBreakAt; i < sizeof(info.nickname) / sizeof(info.nickname[0]); ++i)
        {
            std::sprintf(g_Buffer + (i - NameBreakAt) * 3, "%02X ", info.nickname[i]);
        }
        DrawTextSet(pRenderSystem, x, y++, nullptr, g_Buffer);

#if defined(NNT_NFP_LIB_VERSION_BETA)//Miiデータの仕様が決まるまで除外
    #if defined(NNT_NFP_LIB_MII_ENABLE)
        const nn::Bit8* mii        = reinterpret_cast<const nn::Bit8*>(&info.miiData);
        const int   miiBreakAt = 10;
        const int   lineCount  = (sizeof(info.miiData) + miiBreakAt - 1) / miiBreakAt;
        for (int i = 0; i < lineCount; ++i)
        {
            for (int j = 0; j < miiBreakAt; ++j)
            {
                int k = i * miiBreakAt + j;
                if (k < sizeof(info.miiData))
                {
                    std::sprintf(g_Buffer + j * 3, "%02X ", mii[i * miiBreakAt + j]);
                }
            }
            if (i == 0)
            {
                DrawTextSet(pRenderSystem, x, y++, "MII DATA", g_Buffer);
            }
            else
            {
                DrawTextSet(pRenderSystem, x, y++, nullptr, g_Buffer);
            }
        }
    #endif  // defined(NNT_NFP_LIB_MII_ENABLE)
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
    }

    //================================================================================
    //! @brief      nn::nfp::AdminInfo の内容を画面に表示します。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  x             X 座標です。
    //! @param[in]  y             Y 座標です。
    //! @param[in]  tagData       タグに含まれる情報です。
    //================================================================================
    void DrawAdminInfo(MenuRenderSystem* pRenderSystem, int x, int y,
                       const TagData& tagData) NN_NOEXCEPT
    {
        if (!tagData.hasAdminInfo)
        {
            DrawText(pRenderSystem, x, y, "** ERROR **");
            return;
        }
        const nn::nfp::AdminInfo& info = tagData.adminInfo;

#if !defined(NNT_NFP_PLATFORM_NX)
        std::sprintf(g_Buffer, "%016llX", info.titleId);
        DrawTextSet(pRenderSystem, x, y++, "TITLE ID", g_Buffer);
#elif defined(NNT_NFP_LIB_VERSION_BETA)
        std::sprintf(g_Buffer, "%016llX", (info.applicationId.value & 0xFFFFFFFFFFFFFFFFULL));
        DrawTextSet(pRenderSystem, x, y++, "TITLE ID", g_Buffer);
#endif // !defined(NNT_NFP_PLATFORM_NX)

        std::sprintf(g_Buffer, "%08X", info.accessId);
        DrawTextSet(pRenderSystem, x, y++, "ACCESS ID", g_Buffer);

        std::sprintf(g_Buffer, "%d", info.moveCounter);
        DrawTextSet(pRenderSystem, x, y++, "MOVE COUNTER", g_Buffer);

        std::sprintf(g_Buffer, "%02X", info.registerInfo);
        DrawTextSet(pRenderSystem, x, y++, "REGISTER INFO", g_Buffer);

        std::sprintf(g_Buffer, "%d", info.formatVersion);
        DrawTextSet(pRenderSystem, x, y++, "FORMAT VER", g_Buffer);

        std::sprintf(g_Buffer, "%d", info.platform);
        DrawTextSet(pRenderSystem, x, y++, "PLATFORM", g_Buffer);
    }

    //================================================================================
    //! @brief      アプリケーション専用領域の内容を画面に表示します。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  x             X 座標です。
    //! @param[in]  y             Y 座標です。
    //! @param[in]  tagData       タグに含まれる情報です。
    //================================================================================
    void DrawApplicationArea(MenuRenderSystem* pRenderSystem, int x, int y,
                             const TagData& tagData) NN_NOEXCEPT
    {
        if (!tagData.hasApplicationArea)
        {
            DrawText(pRenderSystem, x, y, "** ERROR **");
            return;
        }

        const int hexPerLine = 16;
        int px = x;
        int py = y;
        for (int i = 0; i < nn::nfp::ApplicationAreaSizeV2; ++i)
        {
            std::sprintf(g_Buffer, "%02X", tagData.applicationArea[i]);
            DrawText(pRenderSystem, px, py, g_Buffer);
            if (i % hexPerLine == (hexPerLine - 1))
            {
                px = x;
                py = py + 1;
            }
            else
            {
                px = px + 3;
            }
        }
    }

    //================================================================================
    //! @brief      タグの内容を表示するためのタブを描画します。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  x             X 座標です。
    //! @param[in]  y             Y 座標です。
    //! @param[in]  tagData       タグに含まれる情報です。
    //================================================================================
    void DrawTab(MenuRenderSystem* pRenderSystem, int x, int y,
                 const TagData& tagData) NN_NOEXCEPT
    {
        NN_UNUSED(pRenderSystem);
        NN_UNUSED(x);
        NN_UNUSED(y);
        NN_UNUSED(tagData);

        // タブを描画します。
        int offset = x;
        for (int i = 0; i < TabCount; ++i)
        {
            int length = static_cast<int>(std::strlen(DrawTabItems[i].pText));

            // 選択中か否かによってタブ名の描画色を切り替えます。
            if (i == g_SelectedTabIndex)
            {
                pRenderSystem->SetColor(FontActiveColorRed,
                                        FontActiveColorGreen,
                                        FontActiveColorBlue,
                                        1.0f);
            }
            else
            {
                pRenderSystem->SetColor(FontInactiveColorRed,
                                        FontInactiveColorGreen,
                                        FontInactiveColorBlue,
                                        1.0f);
            }

#ifdef NNT_NFP_MENU_CONSOLE // メニューでコンソール出力する場合は選択中のタブを"<>"で囲む
            if (i == g_SelectedTabIndex)
            {
                DrawText(pRenderSystem, offset - 1, y, "<");
            }
#endif
            DrawText(pRenderSystem, offset, y, DrawTabItems[i].pText);
#ifdef NNT_NFP_MENU_CONSOLE // メニューでコンソール出力する場合は選択中のタブを"<>"で囲む
            if (i == g_SelectedTabIndex)
            {
                DrawText(pRenderSystem, offset + length + TabMargin - 1, y, ">");
            }
#endif

            // タブ間の区切りを描画します。
            pRenderSystem->SetColor(FontInactiveColorRed,
                                    FontInactiveColorGreen,
                                    FontInactiveColorBlue,
                                    1.0f);
            DrawText(pRenderSystem, offset + length + TabMargin, y, "|");
            offset += length + TabMargin * 2 + 1;

            // タブの内容を描画します。
            if (i == g_SelectedTabIndex)
            {
                pRenderSystem->SetColor(FontActiveColorRed,
                                        FontActiveColorGreen,
                                        FontActiveColorBlue,
                                        1.0f);
                DrawTabItems[i].handler(pRenderSystem, x, y + 2, tagData);
            }
        }
    }

    //================================================================================
    //! @brief      注意事項の描画関数です。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  x             X 座標です。
    //! @param[in]  y             Y 座標です。
    //================================================================================
    void DrawNote(MenuRenderSystem* pRenderSystem, int x, int y) NN_NOEXCEPT
    {
        pRenderSystem->SetColor(FontInactiveColorRed,
                                FontInactiveColorGreen,
                                FontInactiveColorBlue,
                                1.0f);

        DrawText(pRenderSystem, x, y++, "*1 Flush is required to write to tag");
        DrawText(pRenderSystem, x, y++, "*2 please create your Mii on ahead");
    }

    //================================================================================
    //! @brief      操作方法の描画関数です。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  x             X 座標です。
    //! @param[in]  y             Y 座標です。
    //================================================================================
    void DrawManual(MenuRenderSystem* pRenderSystem, int x, int y) NN_NOEXCEPT
    {
        pRenderSystem->SetColor(FontInactiveColorRed,
                                FontInactiveColorGreen,
                                FontInactiveColorBlue,
                                1.0f);

        DrawText(pRenderSystem, x, y++, "UP    : Previous Command");
        DrawText(pRenderSystem, x, y++, "DOWN  : Next Command");
        DrawText(pRenderSystem, x, y++, "LEFT  : Previous Tab");
        DrawText(pRenderSystem, x, y++, "RIGHT : Next Tab");
        DrawText(pRenderSystem, x, y++, "A     : Execute Command");
        DrawText(pRenderSystem, x, y++, "Start : Exit");
    }

    //================================================================================
    //! @brief      上画面描画用の関数です。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  nfpState      NFP ライブラリの状態です。
    //! @param[in]  tagData       タグから読みだしたデータです。
    //================================================================================
    void DrawUpper(MenuRenderSystem* pRenderSystem,
                   const nn::nfp::State state,
                   const nn::nfp::DeviceState deviceState,
                   const bool nfcState,
                   const TagData& tagData) NN_NOEXCEPT
    {
        // 画面をクリアします。
        pRenderSystem->SetRenderTarget(RenderDisplay_0);
        pRenderSystem->Clear();

        // 文字の色を選択します。
        pRenderSystem->SetColor(FontActiveColorRed,
                                FontActiveColorGreen,
                                FontActiveColorBlue,
                                1.0f);

        // 描画処理です。
        int x = 1;
        int y = 1;
        DrawState(pRenderSystem, x, y++, state);
        DrawDeviceState(pRenderSystem, x, y++, deviceState);
        DrawNfcState(pRenderSystem, x, y++, nfcState);
        DrawTagState(pRenderSystem, x, y++, tagData.state);
        DrawSelectedCommand(pRenderSystem, x, y++);
        DrawLastResult(pRenderSystem, x, y++);
        DrawTab(pRenderSystem, x, y + 1, tagData);

        // スワップします。
        pRenderSystem->SwapBuffers();
    }

    //================================================================================
    //! @brief      下画面描画用の関数です。
    //! @param[in]  pRenderSystem 画面描画に使用するデモ用の RenderSystem です。
    //! @param[in]  nfpState      NFP ライブラリの状態です。
    //! @param[in]  tagData       タグから読みだしたデータです。
    //================================================================================
    void DrawLower(MenuRenderSystem* pRenderSystem,
                   const nn::nfp::State state,
                   const nn::nfp::DeviceState deviceState,
                   const TagData& tagData) NN_NOEXCEPT
    {
        NN_UNUSED(state);
        NN_UNUSED(deviceState);
        NN_UNUSED(tagData);

        // 画面をクリアします。
        pRenderSystem->SetRenderTarget(RenderDisplay_1);
        pRenderSystem->Clear();

        // 描画処理です。
        DrawMenu(pRenderSystem, 1, 1);
        DrawNote(pRenderSystem, 1, MenuCount + 2);
        DrawManual(pRenderSystem, 1, MenuCount + 1);

        // スワップします。
        pRenderSystem->SwapBuffers();
    }

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    //================================================================================
    //! @brief      バックグラウンドに遷移させる関数です。
    //================================================================================
    nn::Result MoveBackGround() NN_NOEXCEPT
    {
        g_BackGroundTestFlg = BackGroundTest_None;
        NN_LOG("MoveBackGround Start\n");
        NfpStartMiiEdit();
        return nn::ResultSuccess();
    }

    //================================================================================
    //! @brief      バックグラウンドへ遷移させ、バックグラウンドでInitを実行する
    //================================================================================
    nn::Result MoveBackGroundAndInit() NN_NOEXCEPT
    {
        g_BackGroundTestFlg = BackGroundTest_Initialize;
        NN_LOG("MoveBackGroundAndInit\n");
        NfpStartMiiEdit();
        return nn::ResultSuccess();
    }

    //================================================================================
    //! @brief      MiiEditを起動する関数です。
    //================================================================================
    void NfpStartMiiEdit() NN_NOEXCEPT
    {
        NN_LOG("NfpStartMiiEditThread start\n");
        const int InDataSize = 2048;
        char inData[InDataSize] = {};
        NN_LOG("NfpStartMiiEdit start \n");
        // 呼び出し準備
        auto handle = PrepareMiiEditLibraryAppletImpl(inData, InDataSize);
        // 開始
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::StartLibraryApplet(handle));
        nn::os::WaitSystemEvent(nn::applet::GetLibraryAppletExitEvent(handle));
        NN_LOG("NfpStartMiiEditThread end \n");
        return;
    }

    //================================================================================
    //! @brief      MiiEditアプレット呼び出し準備の関数です。
    //! @param[in]  pInData     ストレージに入力するデータ
    //! @param[in]  inDataSize  ストレージに入力するデータサイズ
    //================================================================================
    nn::applet::LibraryAppletHandle PrepareMiiEditLibraryAppletImpl(
            const char* pInData, size_t inDataSize) NN_NOEXCEPT
    {
        nn::applet::LibraryAppletHandle handle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateLibraryApplet(&handle,
                nn::applet::AppletId_LibraryAppletMiiEdit,
                nn::applet::LibraryAppletMode_AllForeground));
        // ストレージの作成
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::applet::CreateStorage(&storageHandle, inDataSize));
        // ストレージへの書き込み
        NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::applet::WriteToStorage(storageHandle, 0, pInData, inDataSize));
        // 入力チャンネルへの push
        nn::applet::PushToInChannel(handle, storageHandle);
        return handle;
    }
#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)
    //================================================================================
    //! @brief      プロセスメッセージのハンドラ
    //================================================================================
    void ProcessMessage() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::SetFocusHandlingMode( nn::oe::FocusHandlingMode_Notify );
        nn::oe::Message message;
        for(;;)
        {
            if (nn::oe::TryPopNotificationMessage(&message))
            {
                // メッセージごとの処理
                switch( message )
                {
                // フォーカス状態変更通知
                case nn::oe::MessageFocusStateChanged:
                    {
                        auto state = nn::oe::GetCurrentFocusState();
                        switch (state)
                        {
                        // HOME メニューへの遷移
                        case nn::oe::FocusState_Background:
                            {
                                NN_LOG("FocusState= Background\n");
                                // バックグラウンドテスト
                                if (g_BackGroundTestFlg == BackGroundTest_Initialize)
                                {
                                    NfpInitialize();
                                }
                            }
                            break;
                        // ライブラリアプレット呼出し
                        case nn::oe::FocusState_OutOfFocus:
                            {
                                NN_LOG("FocusState= OutOfFocus\n");
                                // バックグラウンドテスト
                                if (g_BackGroundTestFlg == BackGroundTest_Initialize)
                                {
                                    NfpInitialize();
                                }
                            }
                            break;
                        //  HOME メニューや LA 呼出しから復帰した
                        case nn::oe::FocusState_InFocus:
                            {
                                NN_LOG("FocusState= InFocus\n");
                            }
                            break;
                        default: NN_UNEXPECTED_DEFAULT;
                        }
                    }
                    break;
                // 終了
                case nn::oe::MessageExitRequest:
                    {
                        NN_LOG("Received MessageExitRequest\n");
                    }
                    break;
                // 動作モード（携帯／据置）が変更
                case nn::oe::MessageOperationModeChanged:
                    {
                        nn::oe::OperationMode operationMode = nn::oe::GetOperationMode();
                        NN_LOG("Received MessageOperationModeChanged OperationMode = %d\n",
                                operationMode);
                    }
                    break;
                // 性能モード（ノーマル／ブースト）が変更
                case nn::oe::MessagePerformanceModeChanged:
                    {
                        nn::oe::PerformanceMode PerformanceMode = nn::oe::GetPerformanceMode();
                        NN_LOG("Received MessagePerformanceModeChanged PerformanceMode = %d\n",
                                PerformanceMode);
                    }
                    break;
                // 未定義メッセージ
                default:
                    {
                        NN_LOG("Received unknown message= 0x%08x", message);
                    }
                    break;
                }
            }
            if (g_NfpManagerExEndFlg == true)
            {
                // プログラム終了
                break;
            }
            nnt::nfp::Sleep(DispatchSleepTime);
        }
        return;
#else // defined(NN_BUILD_CONFIG_OS_HORIZON)
        for(;;)
        {
            if (g_NfpManagerExEndFlg == true)
            {
                // プログラム終了
                break;
            }
            nnt::nfp::Sleep(DispatchSleepTime);
        }
#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)
    }   // NOLINT(impl/function_size)

    //================================================================================
    //! @brief      プログラムのメインスレッドです。
    //================================================================================
    void NfpManagerExMainThread(void* pArg) NN_NOEXCEPT
    {
        NN_UNUSED(pArg);
        nn::Result result;

        // スリープ関連のコールバックを設定します。
#if !defined(NNT_NFP_PLATFORM_NX) //NX では必要としない
        g_AwakeEvent.Initialize(false);
        nn::applet::SetSleepQueryCallback(SleepQueryCallback, 0);
        nn::applet::SetAwakeCallback(AwakeCallback, 0);
        nn::applet::Enable();
#endif //!defined(NNT_NFP_PLATFORM_NX)

        //amiibo設定実行用にoeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // パッドの入力を取得するため hid を初期化します。
        InitializeHidDevices();
        nn::hid::DebugPadState debugPadState[nn::hid::DebugPadStateCountMax];

#if !defined(NNT_NFP_PLATFORM_NX) //NXではテスト用 Mii を使う
        // このデモでは Mii を取得するために friends ライブラリを使用します。
        result = nn::friends::Initialize();
        NN_UTIL_PANIC_IF_FAILED(result);
#endif // !defined(NNT_NFP_PLATFORM_NX)

        // 画面描画に利用するデモ用の RenderSystem を初期化します。
        MenuRenderSystem renderSystem;
        renderSystem.Initialize(g_RenderSystemMemory, RenderSystemMemorySize);
        renderSystem.SetClearColor(RenderDisplay_Both,
                                   BackgroundColorRed,
                                   BackgroundColorGreen,
                                   BackgroundColorBlue,
                                   1.0f);
        renderSystem.SetFontSize(FontSize);
        renderSystem.SetLineWidth(LineWidth);

        bool isFirstLoop = true;
        // メインループです。
        for(;;)
        {
            // NFP ライブラリの状態を取得します。
            nn::nfp::State state = nnt::nfp::wrapper::GetState();
            nn::nfp::DeviceState deviceState = nnt::nfp::wrapper::GetDeviceState();
            bool nfcState = nnt::nfp::IsNfcEnable();

            nn::hid::GetDebugPadStates(debugPadState, nn::hid::DebugPadStateCountMax);

            // パッドの入力を取得して状態を更新します。
            static int64_t s_PrevPadSamplingNumber = 0;

            // 直前から現在までの Pad 入力の更新回数を取得します。
            int64_t padSamplingCount = debugPadState[0].samplingNumber - s_PrevPadSamplingNumber;
            if(padSamplingCount >= nn::hid::DebugPadStateCountMax)
            {
                padSamplingCount = nn::hid::DebugPadStateCountMax - 1;
            }
            nn::hid::DebugPadButtonSet padButtonDown(debugPadState[0].buttons
                                                     & ~debugPadState[padSamplingCount].buttons);

            // 現在までの Pad 入力の更新回数を取得します。
            s_PrevPadSamplingNumber = debugPadState[0].samplingNumber;

            if (padButtonDown.Test<::nn::hid::DebugPadButton::Start>())
            {
                // 終了
                g_NfpManagerExEndFlg = true;
                break;
            }

            bool isUpdate = Update(padButtonDown, state, deviceState, nfcState);

            // NFP タグの状態を取得します。
            TagData tagData;
            NfpGetTagData(&tagData);

            // 画面を描画します。
            DrawUpper(&renderSystem, state, deviceState, nfcState, tagData);
            DrawLower(&renderSystem, state, deviceState, tagData);
            if(isUpdate == true || isFirstLoop == true)
            {
                renderSystem.WaitVsync(RenderDisplay_Both);
                isFirstLoop = false;
            }
            // スリープの対応です。
#if !defined(NNT_NFP_PLATFORM_NX) //NX では必要としない
            if ( nn::applet::IsExpectedToReplySleepQuery() )
            {
                // 以下の処理が無くても NFP ライブラリによって自動的に Unmount/StopDetection されます。
                // NfpStopDetection();

                nn::applet::ReplySleepQuery(nn::applet::REPLY_ACCEPT);
                g_AwakeEvent.Wait();
                nn::gx::StartLcdDisplay();
            }
#endif //!defined(NNT_NFP_PLATFORM_NX)

            // HOME ボタンの対応です。
#if !defined(NNT_NFP_PLATFORM_NX) //NX では必要としない
            if (nn::applet::IsExpectedToProcessHomeButton())
            {
                nn::applet::ProcessHomeButton();
                nn::applet::WaitForStarting();
                nngxUpdateState(NN_GX_STATE_ALL);
                nngxValidateState(NN_GX_STATE_ALL, GL_TRUE);
            }
#endif //!defined(NNT_NFP_PLATFORM_NX)

            // POWER ボタンの対応です。
#if !defined(NNT_NFP_PLATFORM_NX) //NX では必要としない
            if (nn::applet::IsExpectedToProcessPowerButton())
            {
                nn::applet::ProcessPowerButton();
                nn::applet::WaitForStarting();
                nngxUpdateState(NN_GX_STATE_ALL);
                nngxValidateState(NN_GX_STATE_ALL, GL_TRUE);
            }

            // 終了要求の処理です。
            if (nn::applet::IsExpectedToCloseApplication())
            {
                break;
            }
#endif //!defined(NNT_NFP_PLATFORM_NX)

            // 処理のディスパッチを促すスリープ
            // このスリープがないとジョイコン切断時に再接続できない問題が発生します。
            nnt::nfp::Sleep(DispatchSleepTime);
        }

        // タグの検知スレッドを終了します。
        EndTagDetectionThread();

        if(nnt::nfp::wrapper::GetState() == nn::nfp::State_Init)
        {
            NfpFinalize();
        }

        // その他のライブラリを解放します。
        renderSystem.Finalize();
#if !defined(NNT_NFP_PLATFORM_NX) //NXではテスト用 Mii を使う
        nn::friends::Finalize();
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) //NX では必要としない
        nn::applet::DisableSleep();
        nn::applet::SetSleepQueryCallback(nullptr, 0);
        nn::applet::SetAwakeCallback(nullptr, 0);
        g_AwakeEvent.Finalize();
        nn::applet::CloseApplication();
#endif //!defined(NNT_NFP_PLATFORM_NX)
    }

} // end of anonymous namespace

//================================================================================
//! @brief      プログラムのエントリポイントです。
//================================================================================
extern "C" void nnMain()
{
    //amiibo設定実行用にoeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)

    // コントローラの初期化
    nnt::nfp::InitializeHidController();

    // NfpManagerExMainThreadを生成して起動する
    NN_LOG("Create NfpManagerExMainThread\n");
    NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(
            &g_Thread, NfpManagerExMainThread, nullptr, g_ThreadStack,
            sizeof(g_ThreadStack), nn::os::DefaultThreadPriority));
    g_ThreadCreateFlg = true;
    nn::os::StartThread(&g_Thread);
    // プロセスメッセージのハンドラ
    NN_LOG("ProcessMessage start\n");
    ProcessMessage();
    NN_LOG("ProcessMessage end\n");
    // NfpManagerExMainThreadの後処理
    if (g_ThreadCreateFlg)
    {
        nn::os::WaitThread(&g_Thread);
        nn::os::DestroyThread(&g_Thread);
        g_ThreadCreateFlg = false;
    }
}
