﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nfp.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/oe.h>
#include <nn/applet/applet.h>
#include <nn/os/os_Event.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNfp_Common.h>

namespace
{
    // スレッドループのディスパッチ時間
    const int DispatchSleepTime = 50;
    // アプレット起動待ち時間
    const int WaitTimeLibraryAppletStart = 3000;
    // テスト終了フラグ
    bool g_TestEndFlg = false;
    // アプレット起動フラグ
    bool g_AppletFlg = false;

    // スレッドスタックサイズ
    const size_t AppletThreadStackSize = 32 * 1024;
    // スレッドタイプ
    nn::os::ThreadType  g_AppletThread;
    // スレッドのスタック
    NN_OS_ALIGNAS_THREAD_STACK char g_AppletThreadStack[ AppletThreadStackSize ];
    // ライブラリアプレットハンドル
    nn::applet::LibraryAppletHandle g_LibraryAppletHandle;

    // スレッドスタックサイズ
    const size_t ProcessMessageThreadSize = 32 * 1024;
    // スレッドタイプ
    nn::os::ThreadType  g_ProcessMessageThread;
    // スレッドのスタック
    NN_OS_ALIGNAS_THREAD_STACK char g_ProcessMessageThreadStack[ ProcessMessageThreadSize ];

    void ProcessMessageThread(void* pArg) NN_NOEXCEPT;
    nn::Result NfpStartMiiEdit() NN_NOEXCEPT;
    void NfpStartMiiEditThread(void* pArg) NN_NOEXCEPT;
    nn::applet::LibraryAppletHandle PrepareMiiEditLibraryAppletImpl(
            const char* pInData, size_t inDataSize) NN_NOEXCEPT;

    void ProcessMessageThread(void* pArg) NN_NOEXCEPT
    {
        NN_UNUSED(pArg);
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::SetFocusHandlingMode( nn::oe::FocusHandlingMode_Notify );
        nn::oe::Message message;
        for(;;)
        {
            if (nn::oe::TryPopNotificationMessage(&message))
            {
                // メッセージごとの処理
                switch( message )
                {
                // フォーカス状態変更通知
                case nn::oe::MessageFocusStateChanged:
                    {
                        auto state = nn::oe::GetCurrentFocusState();
                        switch (state)
                        {
                        // HOME メニューへの遷移
                        case nn::oe::FocusState_Background:
                            {
                                NN_LOG("FocusState= Background\n");
                            }
                            break;
                        // ライブラリアプレット呼出し
                        case nn::oe::FocusState_OutOfFocus:
                            {
                                NN_LOG("FocusState= OutOfFocus\n");
                            }
                            break;
                        //  HOME メニューやライブラリアプレット呼出しから復帰した
                        case nn::oe::FocusState_InFocus:
                            {
                                NN_LOG("FocusState= InFocus\n");
                                if (g_AppletFlg)
                                {
                                    nn::applet::JoinLibraryApplet(g_LibraryAppletHandle);
                                    g_AppletFlg = false;
                                }
                            }
                            break;
                        default: NN_UNEXPECTED_DEFAULT;
                        }
                    }
                    break;
                // 終了
                case nn::oe::MessageExitRequest:
                    {
                        NN_LOG("Received MessageExitRequest\n");
                    }
                    break;
                // 動作モード（携帯／据置）が変更
                case nn::oe::MessageOperationModeChanged:
                    {
                        nn::oe::OperationMode operationMode = nn::oe::GetOperationMode();
                        NN_LOG("Received MessageOperationModeChanged OperationMode = %d\n",
                                operationMode);
                    }
                    break;
                // 性能モード（ノーマル／ブースト）が変更
                case nn::oe::MessagePerformanceModeChanged:
                    {
                        nn::oe::PerformanceMode PerformanceMode = nn::oe::GetPerformanceMode();
                        NN_LOG("Received MessagePerformanceModeChanged PerformanceMode = %d\n",
                                PerformanceMode);
                    }
                    break;
                // 未定義メッセージ
                default:
                    {
                        NN_LOG("Received unknown message= 0x%08x", message);
                    }
                    break;
                }
            }
            if (g_TestEndFlg == true)
            {
                // プログラム終了
                break;
            }
            nnt::nfp::Sleep(DispatchSleepTime);
        }
        return;
#else // defined(NN_BUILD_CONFIG_OS_HORIZON)
        for(;;)
        {
            if (g_TestEndFlg == true)
            {
                // プログラム終了
                break;
            }
            nnt::nfp::Sleep(DispatchSleepTime);
        }
#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)
    }   // NOLINT(impl/function_size)

    nn::Result NfpStartMiiEdit() NN_NOEXCEPT
    {
        NN_LOG("Create NfpStartMiiEditThread\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(
                &g_AppletThread, NfpStartMiiEditThread, nullptr, g_AppletThreadStack,
                sizeof(g_AppletThreadStack), nn::os::DefaultThreadPriority));
        nn::os::StartThread(&g_AppletThread);
        return nn::ResultSuccess();
    }

    void NfpStartMiiEditThread(void* pArg) NN_NOEXCEPT
    {
        NN_UNUSED(pArg);
        const int InDataSize = 2048;
        char inData[InDataSize] = {};
        NN_LOG("NfpStartMiiEditThread MiiEdit start \n");
        // 呼び出し準備
        g_LibraryAppletHandle = PrepareMiiEditLibraryAppletImpl(inData, InDataSize);
        // MiiEdit 開始
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::StartLibraryApplet(/*handle*/ g_LibraryAppletHandle));
        g_AppletFlg = true;
        NN_LOG("NfpStartMiiEditThread end \n");
        return;
    }

    nn::applet::LibraryAppletHandle PrepareMiiEditLibraryAppletImpl(
            const char* pInData, size_t inDataSize) NN_NOEXCEPT
    {
        nn::applet::LibraryAppletHandle handle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateLibraryApplet(&handle,
                nn::applet::AppletId_LibraryAppletMiiEdit,
                nn::applet::LibraryAppletMode_AllForeground));
        // ストレージの作成
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::applet::CreateStorage(&storageHandle, inDataSize));
        // ストレージへの書き込み
        NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::applet::WriteToStorage(storageHandle, 0, pInData, inDataSize));
        // 入力チャンネルへの push
        nn::applet::PushToInChannel(handle, storageHandle);
        return handle;
    }

}
//================================================================================
// 全プラットフォームで共通のテストスイートです。
// 必ずタグを設置してからテストを開始してください。
//================================================================================

class NfpBackGround : public nnt::nfp::TestFramework
{
protected:

    NfpBackGround() NN_NOEXCEPT
    {
        // oeライブラリを初期化します。実機のみ
        nn::oe::Initialize();
        // コントローラの初期化
        nnt::nfp::InitializeHidController();
    }

    ~NfpBackGround() NN_NOEXCEPT
    {
    }

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        // 試験をバックグラウンドで実施するためアプレット起動
        NfpStartMiiEdit();
        // バックグラウンドのイベント処理
        NN_LOG("Create NfpStartMiiEditThread\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(
                &g_ProcessMessageThread, ProcessMessageThread, nullptr, g_ProcessMessageThreadStack,
                sizeof(g_ProcessMessageThreadStack), nn::os::DefaultThreadPriority));
        nn::os::StartThread(&g_ProcessMessageThread);
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 次のテストに影響が出ないようにライブラリを一旦終了しておきます。
        if(nnt::nfp::wrapper::GetState() == nn::nfp::State_Init)
        {
            nnt::nfp::WifiOn();
            nnt::nfp::FinalizeSystem();
        }
        g_TestEndFlg = true;
        nn::os::WaitThread(&g_AppletThread);
        nn::os::DestroyThread(&g_AppletThread);
        nn::os::WaitThread(&g_ProcessMessageThread);
        nn::os::DestroyThread(&g_ProcessMessageThread);
        NN_LOG("TearDown end \n");
    }
};

//================================================================================
// テストスイート全体で共通の処理です。
//================================================================================

//================================================================================
// テストケースの実装です。
//================================================================================
TEST_F(NfpBackGround, TestCaseGoodOperationBackGround)
{
    // バックグラウンド時に成功するAPIをテストします。
    nn::nfp::DeviceState deviceState;

    // アプレットが起動するまで少し待つ
    nnt::nfp::Sleep(WaitTimeLibraryAppletStart);

    NN_LOG("TestCaseGoodOperationBackGround start \n");
    // ----------------------------------------
    // テスト対象 : nn::nfp::Initialize()
    //              nn::nfp::Finalize()
    // ----------------------------------------
    EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Unexpected, nnt::nfp::wrapper::GetDeviceState());
    nnt::nfp::wrapper::Initialize();
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    deviceState = nnt::nfp::wrapper::GetDeviceState();
    if (!(deviceState == nn::nfp::DeviceState_Unexpected ||
            deviceState == nn::nfp::DeviceState_Init))
    {
        EXPECT_TRUE(false);
        NN_LOG("TestCaseGoodOperationBackGround: State Error %d", deviceState);
    }
    nnt::nfp::wrapper::Finalize();
    EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Unexpected, nnt::nfp::wrapper::GetDeviceState());
    // ----------------------------------------

    // ----------------------------------------
    // テスト対象 : nn::nfp::InitializeSystem()
    //              nn::nfp::FinalizeSystem()
    // ----------------------------------------
    EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Unexpected, nnt::nfp::wrapper::GetDeviceState());
    nnt::nfp::wrapper::InitializeSystem();
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    deviceState = nnt::nfp::wrapper::GetDeviceState();
    if (!(deviceState == nn::nfp::DeviceState_Unexpected ||
            deviceState == nn::nfp::DeviceState_Init))
    {
        EXPECT_TRUE(false);
        NN_LOG("TestCaseGoodOperationBackGround: State Error %d", deviceState);
    }
    nnt::nfp::wrapper::FinalizeSystem();
    EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Unexpected, nnt::nfp::wrapper::GetDeviceState());

    // ----------------------------------------
    // テスト対象 : nn::nfp::InitializeDebug()
    //              nn::nfp::GetState()
    //              nn::nfp::GetDeviceState()
    // ----------------------------------------
    EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Unexpected, nnt::nfp::wrapper::GetDeviceState());
    nnt::nfp::wrapper::InitializeDebug();
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    deviceState = nnt::nfp::wrapper::GetDeviceState();
    if (!(deviceState == nn::nfp::DeviceState_Unexpected ||
            deviceState == nn::nfp::DeviceState_Init))
    {
        EXPECT_TRUE(false);
        NN_LOG("TestCaseGoodOperationBackGround: State Error %d", deviceState);
    }
    nn::os::SystemEventType availabilityChangeEvent;
    nnt::nfp::wrapper::AttachAvailabilityChangeEvent(&availabilityChangeEvent);


    // ----------------------------------------
    // テスト対象 : nn::nfp::FinalizeDebug()
    // ----------------------------------------
    nnt::nfp::wrapper::FinalizeDebug();
    EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Unexpected, nnt::nfp::wrapper::GetDeviceState());

    NN_LOG("TestCaseGoodOperationBackGround end \n");
}

TEST_F(NfpBackGround, TestCaseInvalidOperationBackGround)
{
    // バックグラウンド時に失敗するAPIのテストをします。

    // アプレットが起動するまで少し待つ
    nnt::nfp::Sleep(WaitTimeLibraryAppletStart);

    NN_LOG("TestCaseInvalidOperationBackGround start \n");
    // INIT 状態に移行します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::InitializeSystemWithRetry());

    // ----------------------------------------
    // テスト対象 : nn::nfp::ListDevices()
    //              nn::nfp::GetNpadId()
    //              nn::nfp::AttachActivateEvent()
    //              nn::nfp::AttachDeactivateEvent()
    //              nn::nfp::StartDetection()
    //              nn::nfp::Restore()
    //              nn::nfp::Format()
    //              nn::nfp::Mount(...)
    //              nn::nfp::Mount(..., MountTarget mountTarget)
    //              nn::nfp::CreateApplicationArea()
    //              nn::nfp::GetAll()
    //              nn::nfp::SetAll()
    //              nn::nfp::ExistsApplicationArea()
    //              nn::nfp::OpenApplicationArea()
    //              nn::nfp::GetApplicationArea()
    //              nn::nfp::SetApplicationArea()
    //              nn::nfp::RecreateApplicationArea()
    //              nn::nfp::SetRegisterInfo()
    //              nn::nfp::GetAdminInfo()
    //              nn::nfp::GetRegisterInfo()
    //              nn::nfp::GetRegisterInfo()(非公開)
    //              nn::nfp::GetCommonInfo()
    //              nn::nfp::GetModelInfo()
    //              nn::nfp::GetTagInfo()
    //              nn::nfp::Flush()
    //              nn::nfp::FlushDebug()
    //              nn::nfp::DeleteRegisterInfo()
    //              nn::nfp::DeleteApplicationArea()
    //              nn::nfp::BreakTag()
    //              nn::nfp::Unmount()
    //              nn::nfp::StopDetection()
    // ----------------------------------------
    nn::os::SystemEventType activateEvent;
    nn::os::SystemEventType deactivateEvent;
    nn::nfp::DeviceHandle deviceHandle;
    int outCount = 0;
    nnt::nfp::SetCurrentDeviceHandle(deviceHandle);
    nn::hid::NpadIdType npadId;
    bool isExist;
    nn::nfp::AdminInfo adminInfo    = {};
    nn::nfp::RegisterInfo regInfo   = {};
    nn::nfp::RegisterInfoPrivate regInfoPrivate = {};
    nn::nfp::RegisterInfoPrivate regInfoSet = {};
    nn::nfp::CommonInfo commonInfo  = {};
    nn::nfp::ModelInfo modelInfo    = {};
    nn::nfp::TagInfo tagInfo        = {};
    nn::Bit8 readBuffer[nn::nfp::ApplicationAreaSizeV2] = {};
    nn::Bit8 randNumSeq[nn::nfp::ApplicationAreaSizeV2] = {};
    nn::Bit8 buffer[nn::nfp::ApplicationAreaSizeV2] = {};
    nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2] = {};
    nn::nfp::NfpData nfpData = {};
    nn::nfp::ApplicationAreaCreateInfo appAreaInfo = { nnt::nfp::ZeroTagId,
                                                       nn::nfp::ApplicationAreaSizeV2,
                                                       tagData };
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::ListDevices(&deviceHandle, &outCount, 1));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::GetNpadId(&npadId, nnt::nfp::GetCurrentDeviceHandle()));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::AttachDeactivateEvent(&deactivateEvent));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::StartDetection());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::Restore());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::Format(buffer, sizeof(buffer)));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::Mount());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::MountRom());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::CreateApplicationArea(appAreaInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::GetAll(&nfpData));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::SetAll(nfpData));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::ExistsApplicationArea(&isExist));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::ZeroTagId));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::GetApplicationArea(readBuffer, sizeof(readBuffer)));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::SetApplicationArea(&randNumSeq, sizeof(randNumSeq)));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::RecreateApplicationArea(appAreaInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::GetRegisterInfo(&regInfoPrivate));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::Flush());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::FlushDebug());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::DeleteRegisterInfo());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::DeleteApplicationArea());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::BreakTag(nn::nfp::BreakType_None));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::Unmount());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotForeground,
            nnt::nfp::wrapper::StopDetection());

    NN_LOG("TestCaseInvalidOperationBackGround end \n");
}

TEST_F(NfpBackGround, TestCaseGoodOperationBackGroundMiiSetting)
{
    // バックグラウンド時に成功するMiiSetting関連APIをテストします。
    // 本来の期待動作は成功だが、MiiSetting関連APIの試験はダミーアプリを使うため
    // ダミーアプリが起動することとResultNotUpdatedが返却されることを期待とする

    // アプレットが起動するまで少し待つ
    nnt::nfp::Sleep(WaitTimeLibraryAppletStart);

    NN_LOG("TestCaseGoodOperationBackGroundMiiSetting start \n");

    // INIT 状態に移行します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::InitializeSystemWithRetry());

    // 試験をバックグラウンドで実施するためアプレットはSetUp()で起動済み
    // ----------------------------------------
    // テスト対象 : nn::nfp::StartNicknameAndOwnerSettings()
    // ----------------------------------------
    nn::nfp::DeviceHandle deviceHandle;
    bool isRegistered;
    nn::nfp::RegisterInfo registerInfo = {};
    nn::nfp::AmiiboSettingsStartParam startParam = {};
    nn::nfp::TagInfo tagInfo = {};
    nn::nfp::RegisterInfo registerInfoSet = {};

    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE PUSH START BUTTON 2 TIMES TO CLOSE DUMMYMIIEDIT AND DUMMYCABINET.\n");
    NN_LOG("==================================================\n");
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotUpdated,
            nnt::nfp::wrapper::StartNicknameAndOwnerSettings(&deviceHandle,
                                                             &isRegistered,
                                                             &registerInfo,
                                                             startParam,
                                                             tagInfo,
                                                             registerInfoSet));

    // 試験をバックグラウンドで実施するためアプレット起動
    nn::os::WaitThread(&g_AppletThread);
    nn::os::DestroyThread(&g_AppletThread);
    NfpStartMiiEdit();
    // アプレットが起動するまで少し待つ
    nnt::nfp::Sleep(WaitTimeLibraryAppletStart);
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE PUSH START BUTTON 2 TIMES TO CLOSE DUMMYMIIEDIT AND DUMMYCABINET.\n");
    NN_LOG("==================================================\n");
    // ----------------------------------------
    // テスト対象 : nn::nfp::StartGameDataEraser()
    // ----------------------------------------
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotUpdated,
            nnt::nfp::wrapper::StartGameDataEraser(&deviceHandle,
                                                   startParam,
                                                   tagInfo));

    // 試験をバックグラウンドで実施するためアプレット起動
    nn::os::WaitThread(&g_AppletThread);
    nn::os::DestroyThread(&g_AppletThread);
    NfpStartMiiEdit();
    // アプレットが起動するまで少し待つ
    nnt::nfp::Sleep(WaitTimeLibraryAppletStart);
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE PUSH START BUTTON 2 TIMES TO CLOSE DUMMYMIIEDIT AND DUMMYCABINET.\n");
    NN_LOG("==================================================\n");
    // ----------------------------------------
    // テスト対象 : nn::nfp::StartRestorer()
    // ----------------------------------------
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotUpdated,
            nnt::nfp::wrapper::StartRestorer(&deviceHandle,
                                             startParam,
                                             tagInfo));

    // 試験をバックグラウンドで実施するためアプレット起動
    nn::os::WaitThread(&g_AppletThread);
    nn::os::DestroyThread(&g_AppletThread);
    NfpStartMiiEdit();
    // アプレットが起動するまで少し待つ
    nnt::nfp::Sleep(WaitTimeLibraryAppletStart);
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE PUSH START BUTTON 2 TIMES TO CLOSE DUMMYMIIEDIT AND DUMMYCABINET.\n");
    NN_LOG("==================================================\n");
    // ----------------------------------------
    // テスト対象 : nn::nfp::StartNicknameAndOwnerSettings()
    // ----------------------------------------
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotUpdated,
            nnt::nfp::wrapper::StartNicknameAndOwnerSettings(&tagInfo,
                                                             &deviceHandle,
                                                             &isRegistered,
                                                             &registerInfo,
                                                             startParam));

    // 試験をバックグラウンドで実施するためアプレット起動
    nn::os::WaitThread(&g_AppletThread);
    nn::os::DestroyThread(&g_AppletThread);
    NfpStartMiiEdit();
    // アプレットが起動するまで少し待つ
    nnt::nfp::Sleep(WaitTimeLibraryAppletStart);
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE PUSH START BUTTON 2 TIMES TO CLOSE DUMMYMIIEDIT AND DUMMYCABINET.\n");
    NN_LOG("==================================================\n");
    // ----------------------------------------
    // テスト対象 : nn::nfp::StartGameDataEraser()
    // ----------------------------------------
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotUpdated,
            nnt::nfp::wrapper::StartGameDataEraser(&tagInfo,
                                                   &deviceHandle,
                                                   startParam));

    // 試験をバックグラウンドで実施するためアプレット起動
    nn::os::WaitThread(&g_AppletThread);
    nn::os::DestroyThread(&g_AppletThread);
    NfpStartMiiEdit();
    // アプレットが起動するまで少し待つ
    nnt::nfp::Sleep(WaitTimeLibraryAppletStart);
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE PUSH START BUTTON 2 TIMES TO CLOSE DUMMYMIIEDIT AND DUMMYCABINET.\n");
    NN_LOG("==================================================\n");
    // ----------------------------------------
    // テスト対象 : nn::nfp::StartRestorer()
    // ----------------------------------------
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotUpdated,
            nnt::nfp::wrapper::StartRestorer(&tagInfo,
                                             &deviceHandle,
                                             startParam));

    // 試験をバックグラウンドで実施するためアプレット起動
    nn::os::WaitThread(&g_AppletThread);
    nn::os::DestroyThread(&g_AppletThread);
    NfpStartMiiEdit();
    // アプレットが起動するまで少し待つ
    nnt::nfp::Sleep(WaitTimeLibraryAppletStart);
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE PUSH START BUTTON 2 TIMES TO CLOSE DUMMYMIIEDIT AND DUMMYCABINET.\n");
    NN_LOG("==================================================\n");
    // ----------------------------------------
    // テスト対象 : nn::nfp::StartFormatter()
    // ----------------------------------------
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotUpdated,
            nnt::nfp::wrapper::StartFormatter(&tagInfo,
                                              &deviceHandle,
                                              startParam));

    NN_LOG("TestCaseGoodOperationBackGroundMiiSetting end \n");
}
