﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nfp.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/hid.h>
#include <nn/os/os_Event.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNfp_Common.h>
#include <nnt/nfp/testMifare_Common.h>

#include <nn/irsensor.h>
#include <nn/nn_Log.h>

namespace
{
    // #define NNT_NFP_NFPAUTONX_AMIIBO_API_TEST // Amiibo関連API試験のスイッチ
}
//================================================================================
// 全プラットフォームで共通のテストスイートです。
// 必ずタグを設置してからテストを開始してください。
//================================================================================

class NfpAutoNx : public nnt::nfp::TestFramework
{
protected:

    NfpAutoNx() NN_NOEXCEPT
    {
        // oeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::nfp::InitializeHidController();
    }

    ~NfpAutoNx() NN_NOEXCEPT
    {
    }

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 次のテストに影響が出ないようにライブラリを一旦終了しておきます。
        if(nnt::nfp::wrapper::GetState() == nn::nfp::State_Init)
        {
            nnt::nfp::WifiOn();
            nnt::nfp::FinalizeSystem();
        }
    }
};

//================================================================================
// テストスイート全体で共通の処理です。
//================================================================================

//================================================================================
// テストケースの実装です。
//================================================================================
#if defined(NNT_NFP_LIB_VERSION_BETA)
TEST_F(NfpAutoNx, TestCaseGoodOperationWifiDisable)
{
    // テスト実施前に有効なデバイスリストを取得します
    nnt::nfp::WifiOn();
    nnt::nfp::DoSearchDeviceHandle();

    // 無線通信状態を無効に設定します
    nnt::nfp::WifiOff();

    // 無線通信無効時に呼出すことができるAPIをテストします。

    // ----------------------------------------
    // テスト対象 : nn::nfp::GetState()
    //              nn::nfp::GetDeviceState()
    //              nn::nfp::AttachAvailabilityChangeEvent()
    nnt::nfp::InitializeSystem();
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());
    nn::os::SystemEventType availabilityChangeEvent;
    nnt::nfp::wrapper::AttachAvailabilityChangeEvent(&availabilityChangeEvent);
    nnt::nfp::FinalizeSystem();
    // ----------------------------------------

    // ----------------------------------------
    // テスト対象 : nn::nfp::Initialize()
    //              nn::nfp::Finalize()
    // ----------------------------------------
    EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Unexpected, nnt::nfp::wrapper::GetDeviceState());
    nnt::nfp::wrapper::Initialize();
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());
    nnt::nfp::wrapper::Finalize();
    // ----------------------------------------

    // ----------------------------------------
    // テスト対象 : nn::nfp::InitializeSystem()
    //              nn::nfp::FinalizeSystem()
    // ----------------------------------------
    EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Unexpected, nnt::nfp::wrapper::GetDeviceState());
    nnt::nfp::wrapper::InitializeSystem();
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());
    nnt::nfp::wrapper::FinalizeSystem();
    EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Unexpected, nnt::nfp::wrapper::GetDeviceState());
    // ----------------------------------------

    // 無線通信無効時 && DeviceState_Mount時に
    // 呼出すことができるAPIをテストします。

    // テスト用のタグを作成します。
    nnt::nfp::WifiOn();
    nnt::nfp::DoCreateZeroTagAndEnd();

    // DeviceState_Mount状態に遷移します
    nnt::nfp::DoMount();
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Mount, nnt::nfp::wrapper::GetDeviceState());

    // 無線通信を無効に設定します
    nnt::nfp::WifiOff();

    // 状態がDeviceState_Initに変化していることを確認します
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());

    // ----------------------------------------
}
#endif // defined(NNT_NFP_LIB_VERSION_BETA)

#if defined(NNT_NFP_LIB_VERSION_BETA)
TEST_F(NfpAutoNx, TestCaseInvalidOperationWifiDisable)
{
    // 無線通信状態を有効に設定します
    nnt::nfp::WifiOn();
    // テスト実施前に有効なデバイスリストを取得します
    nnt::nfp::DoSearchDeviceHandle();

    /* nn::nfp::ResultNfcDisabled()を返すAPIをテストします */

    // ----------------------------------------
    // テスト対象 : nn::nfp::ListDevices()
    //              nn::nfp::GetNpadId()
    //              nn::nfp::AttachActivateEvent()
    //              nn::nfp::AttachDeactivateEvent()
    //              nn::nfp::StartDetection()
    // ----------------------------------------
    // DeviceState_Init状態に遷移します
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());

    // 無線通信を無効に設定します
    nnt::nfp::WifiOff();
    nn::nfp::DeviceHandle deviceHandle;
    int outCount = 0;
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::ListDevices(&deviceHandle, &outCount, 1));
    nn::hid::NpadIdType npadId;
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::GetNpadId(&npadId, deviceHandle));
    nn::os::SystemEventType activateEvent;
    nn::os::SystemEventType deactivateEvent;
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::AttachDeactivateEvent(&deactivateEvent));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::StartDetection());
    // 状態が変化していないことを確認します
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());
    // ----------------------------------------

    // ----------------------------------------
    // テスト対象 : nn::nfp::StopDetection()
    //              nn::nfp::Format()
    //              nn::nfp::Mount()
    // ----------------------------------------
    // DeviceState_Active状態に遷移します
    nnt::nfp::WifiOn();
    nnt::nfp::DoActivate();

    // 無線通信を無効に設定します
    nnt::nfp::WifiOff();
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());

    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::StopDetection());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::Format());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::Mount());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::MountRom());

    // 状態が変化していないことを確認します
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());
    // ----------------------------------------

    // ----------------------------------------
    // テスト対象 : nn::nfp::CreateApplicationArea()
    //              nn::nfp::DeleteRegisterInfo()
    //              nn::nfp::DeleteApplicationArea()
    //              nn::nfp::Flush()
    //              nn::nfp::FlushDebug()
    //              nn::nfp::BreakTag()
    //              nn::nfp::Restore()
    //              nn::nfp::GetAll()
    //              nn::nfp::SetAll()
    //              nn::nfp::ExistsApplicationArea()
    //              nn::nfp::OpenApplicationArea()
    //              nn::nfp::GetApplicationArea()
    //              nn::nfp::SetApplicationArea()
    //              nn::nfp::RecreateApplicationArea()
    //              nn::nfp::SetRegisterInfo()
    //              nn::nfp::GetAdminInfo()
    //              nn::nfp::GetRegisterInfo()
    //              nn::nfp::GetCommonInfo()
    //              nn::nfp::GetModelInfo()
    //              nn::nfp::GetTagInfo()
    //              nn::nfp::Unmount()
    // ----------------------------------------
    // DeviceState_Mount状態に遷移します
    nnt::nfp::WifiOn();
    nnt::nfp::DoMount();

    // 無線通信を無効に設定します
    nnt::nfp::WifiOff();
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());

    nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2] = { 0 };
    nn::nfp::ApplicationAreaCreateInfo appAreaInfo = { nnt::nfp::ZeroTagId,
                                                       nn::nfp::ApplicationAreaSizeV2,
                                                       tagData };
    bool isExist;
    nn::nfp::AdminInfo adminInfo    = {};
    nn::nfp::RegisterInfo regInfo   = {};
    nn::nfp::RegisterInfoPrivate regInfoPrivate = {};
    nn::nfp::RegisterInfoPrivate regInfoSet = {};
    nn::nfp::CommonInfo commonInfo  = {};
    nn::nfp::ModelInfo modelInfo    = {};
    nn::nfp::TagInfo tagInfo        = {};
    nn::Bit8 readBuffer[nn::nfp::ApplicationAreaSizeV2] = {};
    nn::Bit8 randNumSeq[nn::nfp::ApplicationAreaSizeV2] = {};
    nn::nfp::NfpData nfpData = {};

    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::CreateApplicationArea(appAreaInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::DeleteRegisterInfo());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::DeleteApplicationArea());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::Flush());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::FlushDebug());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::BreakTag(nn::nfp::BreakType_None));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::Restore());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::ExistsApplicationArea(&isExist));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::ZeroTagId));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::GetApplicationArea(readBuffer, sizeof(readBuffer)));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::SetApplicationArea(&randNumSeq, sizeof(randNumSeq)));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::RecreateApplicationArea(appAreaInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::GetAll(&nfpData));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::SetAll(nfpData));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::GetRegisterInfo(&regInfoPrivate));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDisabled, nnt::nfp::wrapper::Unmount());

// AmiiboSetting系APIはNfcの状態によって書き込めた／書き込めなかったを
// 返却するわけではないため試験の対象から除外します。
// 例えば、Nfc無効状態でAmiiboSetting系のAPIが呼ばれ別のアプリが起動中に
// Nfc有効状態に変更された場合、成功となる。
#if defined(NNT_NFP_NFPAUTONX_AMIIBO_API_TEST)
    /* nn::nfp::ResultNotUpdated()を返すAPIをテストします */
    // ----------------------------------------
    // テスト対象 : nn::nfp::StartNicknameAndOwnerSettings()
    // ----------------------------------------
    bool isRegistered;
    nn::nfp::RegisterInfo registerInfo = {};
    nn::nfp::AmiiboSettingsStartParam startParam = {};
    nn::nfp::RegisterInfo registerInfoSet = {};
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotUpdated,
            nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &deviceHandle, &isRegistered, &registerInfo,
            startParam, tagInfo, registerInfoSet));

    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotUpdated,
            nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &deviceHandle, &isRegistered, &registerInfo,
            startParam, tagInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotUpdated,
            nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &tagInfo, &deviceHandle, &isRegistered,
            &registerInfo, startParam));

    // ----------------------------------------
    // テスト対象 : nn::nfp::StartGameDataEraser()
    // ----------------------------------------
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotUpdated,
            nnt::nfp::wrapper::StartGameDataEraser(
            &deviceHandle, startParam, tagInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotUpdated,
            nnt::nfp::wrapper::StartGameDataEraser(
            &tagInfo, &deviceHandle, startParam));

    // ----------------------------------------
    // テスト対象 : nn::nfp::StartRestorer()
    // ----------------------------------------
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotUpdated,
            nnt::nfp::wrapper::StartRestorer(
            &deviceHandle, startParam, tagInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotUpdated,
            nnt::nfp::wrapper::StartRestorer(
            &tagInfo, &deviceHandle, startParam));

    // ----------------------------------------
    // テスト対象 : nn::nfp::StartFormatter()
    // ----------------------------------------
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNotUpdated,
            nnt::nfp::wrapper::StartFormatter(
            &tagInfo, &deviceHandle, startParam));
#endif // defined(NNT_NFP_NFPAUTONX_AMIIBO_API_TEST)

    // 状態が変化していないことを確認します
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());
    // ----------------------------------------
} // NOLINT(impl/function_size)
#endif // defined(NNT_NFP_LIB_VERSION_BETA)

TEST_F(NfpAutoNx, TestCaseCheckGetRegisterInfoOfPrivateApi)
{
    // 試験用RegisterInfoPrivate宣言と初期化
    nn::nfp::RegisterInfoPrivate       regInfoSet = {};
    nn::nfp::RegisterInfoPrivate       regInfoGet = {};
    // フォントのリージョン情報
    const int FontRegion = nn::nfp::FontRegion_Count - 1;
#if defined(NNT_NFP_LIB_MII_ENABLE)
    // Miiの判定無視サイズ
    const int ExceptDataSize = 20;
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    // 年月構造体
    nn::nfp::Date testDate = {};
    nn::nfp::Date testNextDate = {};
    // テストに使用するタグをつくります。
    nn::nfp::TagInfo             tagInfo;
    nn::nfp::AdminInfo           prevAdminInfo;
    nn::nfp::CommonInfo          prevCommonInfo;
    nn::nfp::RegisterInfoPrivate prevRegInfo;
    nn::nfp::ModelInfo           prevModelInfo;
    nnt::nfp::DoCreateZeroTag();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&prevAdminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&prevCommonInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&prevModelInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRegister,
                              nnt::nfp::wrapper::GetRegisterInfo(&prevRegInfo));
    EXPECT_TRUE(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevAdminInfo._reserved, sizeof(prevAdminInfo._reserved)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevCommonInfo._reserved1, sizeof(prevCommonInfo._reserved1)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevCommonInfo._reserved2, sizeof(prevCommonInfo._reserved2)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevModelInfo._reserved, sizeof(prevModelInfo._reserved)));

    // RegisterInfoPrivateデータ設定
#if defined(NNT_NFP_LIB_MII_ENABLE)
    // Miiデータ
    nnt::nfp::BuildMiiData(regInfoSet.miiData);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    // 本日を取得
    testDate = nnt::nfp::GetCurrentDate();
    // テスト用に来年を設定
    testDate.year++;
    // 登録日を設定
    std::memcpy(&regInfoSet.registerDate, &testDate, sizeof(regInfoSet.registerDate));
    // ニックネーム
    std::memcpy(regInfoSet.nickname,
                nnt::nfp::NormalTagNickName,
                sizeof(nnt::nfp::NormalTagNickName));
    // フォントのリージョン情報
    regInfoSet.fontRegion = FontRegion;

    // ----------------------------------------
    // テスト対象 : nn::nfp::GetRegisterInfo() (非公開)
    // ----------------------------------------
    // SetRegisterInfoを実行して共用領域に情報を書き込む
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));
    // フラッシュ
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FlushWithRetry());
    // マウントしなおす
    nnt::nfp::DoMount();
    // 非公開 APIのGetRegisterInfoを実行して共用領域に情報を読み込む
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfoGet));

    // RegisterInfoPrivateのregisterDate以外のフィールドが一致すること
#if defined(NNT_NFP_LIB_MII_ENABLE)
    EXPECT_TRUE(std::memcmp(&regInfoGet.miiData, &regInfoSet.miiData,
            regInfoSet.miiData.StructureSize - ExceptDataSize) == 0);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    EXPECT_TRUE(std::memcmp(regInfoGet.nickname, regInfoSet.nickname, sizeof(regInfoSet.nickname)) == 0);
    EXPECT_EQ(FontRegion, regInfoGet.fontRegion);
#if !defined(NNT_NFP_PLATFORM_NX) //NX には国コードがない
    EXPECT_EQ(nnt::nfp::GetCountry(), regInfo.country);
#endif //!defined(NNT_NFP_PLATFORM_NX)
    EXPECT_TRUE(std::memcmp(regInfoGet._reserved, regInfoSet._reserved, sizeof(regInfoSet._reserved)) == 0);

    // registerDateが無視されてNXに設定されている日付情報と一致すること
    // (日付をまたいだときのために1日の誤差を許容する
    testDate = nnt::nfp::GetCurrentDate();
    testNextDate = testDate;
    testNextDate.day++;
    EXPECT_TRUE((std::memcmp(&regInfoGet.registerDate, &testDate, sizeof(testDate)) == 0) ||
            (std::memcmp(&regInfoGet.registerDate, &testNextDate, sizeof(testNextDate)) == 0));
}

TEST_F(NfpAutoNx, TestCaseSetNpadId)
{
    nn::hid::NpadIdType npadId;
    // ListDevicesでデバイスハンドルを取得する
    nnt::nfp::DoSearchDeviceHandle();

    // ----------------------------------------
    // テスト対象 : nn::nfp::GetNpadId()
    // ----------------------------------------
    // 取得したデバイスハンドルを指定してGetNpadIdを実行する。ResultSuccessが返ること
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId));

    // 取得したNpad IDを指定してGetNpadStyleSetを実行する
    nn::hid::NpadStyleSet style;
    style = nn::hid::GetNpadStyleSet(npadId);
    EXPECT_TRUE((style.Test<nn::hid::NpadStyleFullKey>() == true) |
            (style.Test<nn::hid::NpadStyleHandheld>() == true) |
            (style.Test<nn::hid::NpadStyleJoyDual>() == true) |
            (style.Test<nn::hid::NpadStyleJoyLeft>() == true) |
            (style.Test<nn::hid::NpadStyleJoyRight>() == true) |
            (style.Test<nn::hid::system::NpadStyleSystem>() == true) |
            (style.Test<nn::hid::system::NpadStyleSystemExt>() == true));
}
TEST_F(NfpAutoNx, TestCaseConflictIr)
{
    // IRカメラハンドル
    const int NpadIdCountMax = 5; // 試験で使用するNpadIdは5つ
    nn::irsensor::IrCameraHandle irCameraHandles[NpadIdCountMax];
    for (int i = 0; i < NpadIdCountMax; ++i)
    {
        // IR カメラのハンドルの取得
        irCameraHandles[i] = nn::irsensor::GetIrCameraHandle(nnt::nfp::NpadIds[i]);
        // IR カメラの初期化
        nn::irsensor::Initialize(irCameraHandles[i]);
        NN_LOG("NpadPlayerNumber(%d)\n",nnt::nfp::NpadIds[i]);
    }

    // モーメントプロセッサ設定と開始
    for (int i = 0; i < NpadIdCountMax; ++i)
    {
        nn::irsensor::MomentProcessorConfig config;
        // モーメントプロセッサのデフォルト設定の取得
        nn::irsensor::GetMomentProcessorDefaultConfig(&config);
        // モーメントプロセッサの開始
        nn::irsensor::RunMomentProcessor(irCameraHandles[i], config);
    }

    // Nfp初期化
    nn::nfp::State libState;
    libState = nnt::nfp::wrapper::GetState();
    if (libState == nn::nfp::State_None)
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::InitializeSystemWithRetry());
    }

    // デバイスハンドルの取得を行う
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::SearchDeviceHandle());

    // イベント取得準備
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::SetDefaultActivateEvent());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::SetDefaultDeactivateEvent());

    // 状態確認
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());

    // ----------------------------------------
    // テスト対象 : nn::nfp::StartDetection()
    // ----------------------------------------
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultConflictFunction,
            nnt::nfp::wrapper::StartDetection());

    // IRセンサー終了処理
    for (int i = 0; i < NpadIdCountMax; ++i)
    {
        //モーメントプロセッサの停止
        nn::irsensor::StopImageProcessor(irCameraHandles[i]);
        //IR カメラの終了処理
        nn::irsensor::Finalize(irCameraHandles[i]);
    }
}
TEST_F(NfpAutoNx, TestCaseNfpNfc)
{
    nn::nfp::TagInfo             tagNfpInfo;
    nn::nfc::TagInfo             tagNfcInfo;

    // -----------------------------------------------------
    // テスト対象 : Nfp Init -> Nfc Init -> Nfc Fin -> Nfp Fin
    // -----------------------------------------------------
    nnt::nfp::InitializeSystem();
    nnt::nfp::DoActivate();

    EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState());

    nnt::mifare::InitializeSystem();

    EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::SearchDeviceHandle());
    EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState());
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetTagInfo(&tagNfcInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultInvalidDeviceState, nnt::mifare::wrapper::StartDetection());

    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagNfpInfo));

    nnt::mifare::FinalizeSystem();

    EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagNfpInfo));

    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());

    // -----------------------------------------------------
    // テスト対象 : Nfp Init -> Nfc Init -> Nfp Fin -> Nfc Fin
    // -----------------------------------------------------
    nnt::nfp::InitializeSystem();
    nnt::nfp::DoActivate();

    EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState());

    nnt::mifare::InitializeSystem();

    EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::SearchDeviceHandle());
    EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState());
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetTagInfo(&tagNfcInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultInvalidDeviceState, nnt::mifare::wrapper::StartDetection());

    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagNfpInfo));

    nnt::nfp::FinalizeSystem();

    EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState());

    EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());

    nnt::mifare::DoActivate();

    EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState());
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetTagInfo(&tagNfcInfo));

    nnt::mifare::FinalizeSystem();
    EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());

    // -----------------------------------------------------
    // テスト対象 : Nfc Init -> Nfp Init -> Nfp Fin -> Nfc Fin
    // -----------------------------------------------------
    nnt::mifare::InitializeSystem();
    nnt::mifare::DoActivate();

    EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState());

    nnt::nfp::InitializeSystem();

    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::SearchDeviceHandle());
    EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagNfpInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidDeviceState, nnt::nfp::wrapper::StartDetection());

    EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState());
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetTagInfo(&tagNfcInfo));

    nnt::nfp::FinalizeSystem();

    EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState());
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetTagInfo(&tagNfcInfo));

    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::FinalizeSystem());
    EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());

    // -----------------------------------------------------
    // テスト対象 : Nfc Init -> Nfp Init -> Nfc Fin -> Nfp Fin
    // -----------------------------------------------------
    nnt::mifare::InitializeSystem();
    nnt::mifare::DoActivate();

    EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState());

    nnt::nfp::InitializeSystem();

    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::SearchDeviceHandle());
    EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagNfpInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidDeviceState, nnt::nfp::wrapper::StartDetection());

    EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState());
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetTagInfo(&tagNfcInfo));

    nnt::mifare::FinalizeSystem();

    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
    EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());

    EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());

    nnt::nfp::DoActivate();

    EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagNfpInfo));

    nnt::nfp::FinalizeSystem();
    EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
} // NOLINT(impl/function_size)
