﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nfp.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Event.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNfp_Common.h>

//================================================================================
// このテストで使用する定義です。
//================================================================================

namespace
{
    // エイジングの回数を指定します。タグの書き込み回数には最大 110,000 回の制限があります。
    const int AgingCountMax = 10000;

#if defined(NNT_NFP_LIB_MII_ENABLE)
    // Miiデータの判定フラグ
    const int MiiCheckflags =
            nn::mii::CheckFlag_Nickname |
            nn::mii::CheckFlag_Gender |
            nn::mii::CheckFlag_Height |
            nn::mii::CheckFlag_Build |
            nn::mii::CheckFlag_FavoriteColor;

    // Miiデータの比較対象外データサイズ
    const int ExceptDataSize = 20;
#endif // defined(NNT_NFP_LIB_MII_ENABLE)

    // Mount の結果を 2 重チェックします。
    nn::Result MountTest()
    {
        nn::Result result1 = nnt::nfp::MountWithRetry();
        if (result1.IsSuccess())
        {
            nnt::nfp::wrapper::Unmount();
        }

        nn::Result result2 = nnt::nfp::MountWithRetry();
        if ((result1.GetModule() != result2.GetModule()) ||
            (result1.GetDescription() != result2.GetDescription()))
        {
            NN_LOG("ASSERT: mount result is not consistent.\n");
            NN_LOG("ASSERT: first mount Module=%d Description=%d \n",
                   result1.GetModule(),result1.GetDescription());
            NN_LOG("ASSERT: second mount Module=%d Description=%d \n",
                   result2.GetModule(),result2.GetDescription());
        }
        return result2;
    }

} // end of anonymous namespace

//================================================================================
// タグの共有領域やアプリケーション専用領域の読み書きを繰り返すエイジングです。
// 必ず新しいタグを設置してからテストを開始してください。
// 使い込んだタグでは耐性を超えてしまい、書き込みの失敗が頻発する可能性があります。
//================================================================================

//================================================================================
// テスト用のユーティリティです。
//================================================================================
class ScopedDateProgressor
{
public:

    explicit ScopedDateProgressor()
    {
        // TODO: 23:59 以降であれば 0:00 まで待機する

        m_Today = nnt::nfp::GetCurrentDate();
        nnt::nfp::SetNextDate();
        m_Tomorrow = nnt::nfp::GetCurrentDate();
    }

    ~ScopedDateProgressor()
    {
        nnt::nfp::SetPreviousDate();
    }

    nn::nfp::Date GetToday() const
    {
        return m_Today;
    }

    nn::nfp::Date GetTomorrow() const
    {
        return m_Tomorrow;
    }

private:

    nn::nfp::Date m_Today;
    nn::nfp::Date m_Tomorrow;

};

//================================================================================
// エイジングテスト用のフレームワークです。
//================================================================================

class NfpAgingStableTag : public nnt::nfp::TestFramework
{
protected:

    NfpAgingStableTag() NN_NOEXCEPT
    {
        //amiibo設定実行用にoeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::nfp::InitializeHidController();
        nnt::nfp::wrapper::SetApiCallLoggingMode(nnt::nfp::wrapper::LogMode_Aging);
    }

    bool TestCreateApplicationArea() NN_NOEXCEPT;
    bool TestDeleteApplicationArea(bool needOpen) NN_NOEXCEPT;
    bool TestReadWriteApplicationArea(const void* pWriteData, int size) NN_NOEXCEPT;
#if defined(NNT_NFP_PLATFORM_NX)
    bool TestRecreateApplicationArea(nn::Bit32 accessId, const void* pWriteData, int size) NN_NOEXCEPT;
#endif //defined(NNT_NFP_PLATFORM_NX)
    bool TestReadWriteRegInfo() NN_NOEXCEPT;
#if defined(NNT_NFP_LIB_VERSION_ALPHA)
    bool TestRestore(nn::nfp::BreakType breakType) NN_NOEXCEPT;
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA)
    bool TestFormat() NN_NOEXCEPT;

};


bool NfpAgingStableTag::TestCreateApplicationArea() NN_NOEXCEPT
{
    NN_LOG("NFP AGING: - CreateApplicationArea Start\n");

    // タグをマウントします。
    nn::Result mountResult1 = MountTest();
    if (mountResult1.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(mountResult1);
        return false;
    }

    // 現時点でのタグのフィールドを記憶しておきます。
    nn::nfp::TagInfo      tagInfo;
    nn::nfp::AdminInfo    prevAdminInfo;
    nn::nfp::CommonInfo   prevCommonInfo;
    nn::nfp::RegisterInfo prevRegInfo;
    nn::nfp::ModelInfo    prevModelInfo;
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&prevAdminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&prevCommonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&prevRegInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&prevModelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_EQUAL(nn::nfp::ResultNeedCreate(),
                                      nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevAdminInfo._reserved,
                                             sizeof(prevAdminInfo._reserved)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevCommonInfo._reserved1,
                                             sizeof(prevCommonInfo._reserved1)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevCommonInfo._reserved2,
                                             sizeof(prevCommonInfo._reserved2)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevRegInfo._reserved,
                                             sizeof(prevRegInfo._reserved)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevModelInfo._reserved,
                                             sizeof(prevModelInfo._reserved)));

    // テストのため日付を進めます。
    ScopedDateProgressor date;
    nn::nfp::Date tomorrow = date.GetTomorrow();

    // アプリケーション専用領域を作成します。
    nn::nfp::ApplicationAreaCreateInfo createInfo;
#if !defined(NNT_NFP_PLATFORM_NX)// NX ではAPIが削除されている
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeCreateInfo(&createInfo));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(createInfo._reserved, sizeof(createInfo._reserved)));
#endif // !defined(NNT_NFP_PLATFORM_NX)
    nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2];
    std::memcpy(tagData,nnt::nfp::NormalTagData,sizeof(nnt::nfp::NormalTagData));
    createInfo.accessId        = nnt::nfp::NormalTagId;
    createInfo.pInitialData    = tagData;
    createInfo.initialDataSize = sizeof(nnt::nfp::NormalTagData);
    nn::Result createResult = nnt::nfp::CreateApplicationAreaWithRetry(createInfo);
    if (createResult.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(createResult);
        nnt::nfp::wrapper::Unmount();
        return false;
    }

    // タグのフィールドが仕様通り変化していることを検証します。
    nn::nfp::AdminInfo    adminInfo;
    nn::nfp::CommonInfo   commonInfo;
    nn::nfp::RegisterInfo regInfo;
    nn::nfp::ModelInfo    modelInfo;
    nn::Bit8              appArea[nn::nfp::ApplicationAreaSizeV2];
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea,
                                                                              sizeof(appArea)));
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(nnt::nfp::TITLE_ID, adminInfo.titleId);
#else
    NNT_NFP_AGING_ASSERT_EQUAL_N(nnt::nfp::TestApplicationId, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(nnt::nfp::NormalTagId, adminInfo.accessId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevAdminInfo.moveCounter, adminInfo.moveCounter);
    NNT_NFP_AGING_ASSERT_EQUAL_N(
            nn::nfp::RegisterInfoFlags_ApplicationArea | nn::nfp::RegisterInfoFlags_CommonArea,
            adminInfo.registerInfo);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nnt::nfp::TestNoftVersion, adminInfo.formatVersion);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nnt::nfp::Platform, adminInfo.platform);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&tomorrow, &commonInfo.lastWriteDate, sizeof(tomorrow)) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.writeCounter + 1, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    NNT_NFP_AGING_ASSERT(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, 3) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.seriesId, commonInfo.seriesId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.numberingId, commonInfo.numberingId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.nfpVersion, commonInfo.nfpVersion);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.applicationAreaSize, commonInfo.applicationAreaSize);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
#if defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
            prevRegInfo.miiData, regInfo.miiData) == 0);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    NNT_NFP_AGING_ASSERT(std::memcmp(prevModelInfo.characterId, commonInfo.characterID, 3) == 0);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevModelInfo.seriesId, modelInfo.seriesId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevModelInfo.numberingId, modelInfo.numberingId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevModelInfo.nfpType, modelInfo.nfpType);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));
    NNT_NFP_AGING_ASSERT(std::memcmp(nnt::nfp::NormalTagData, appArea, sizeof(appArea)) == 0);

    // アプリケーション専用領域が存在する状態でもう一度つくろうとすると失敗します。
    NNT_NFP_AGING_ASSERT_RESULT_EQUAL(nn::nfp::ResultAlreadyCreated(),
                                      nnt::nfp::CreateApplicationAreaWithRetry(createInfo));

    // タグをマウントしなおします。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    nn::Result mountResult2 = MountTest();
    if (mountResult2.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(mountResult2);
        return false;
    }

    // タグのフィールドが仕様通り変化していることをもう一度検証します。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea,
                                                                              sizeof(appArea)));
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(nnt::nfp::TITLE_ID, adminInfo.titleId);
#else
    NNT_NFP_AGING_ASSERT_EQUAL_N(nnt::nfp::TestApplicationId, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(nnt::nfp::NormalTagId, adminInfo.accessId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevAdminInfo.moveCounter, adminInfo.moveCounter);
    NNT_NFP_AGING_ASSERT_EQUAL_N(
            nn::nfp::RegisterInfoFlags_ApplicationArea | nn::nfp::RegisterInfoFlags_CommonArea,
            adminInfo.registerInfo);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nnt::nfp::TestNoftVersion, adminInfo.formatVersion);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nnt::nfp::Platform, adminInfo.platform);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&tomorrow, &commonInfo.lastWriteDate, sizeof(tomorrow)) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.writeCounter + 1, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    NNT_NFP_AGING_ASSERT(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, 3) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.seriesId, commonInfo.seriesId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.numberingId, commonInfo.numberingId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.nfpVersion, commonInfo.nfpVersion);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.applicationAreaSize, commonInfo.applicationAreaSize);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
#if defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
            prevRegInfo.miiData, regInfo.miiData) == 0);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    NNT_NFP_AGING_ASSERT(std::memcmp(prevModelInfo.characterID, commonInfo.characterID, 3) == 0);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevModelInfo.seriesId, modelInfo.seriesId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevModelInfo.numberingId, modelInfo.numberingId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevModelInfo.nfpType, modelInfo.nfpType);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));
    NNT_NFP_AGING_ASSERT(std::memcmp(nnt::nfp::NormalTagData, appArea, sizeof(appArea)) == 0);

    // アプリケーション専用領域が存在する状態でもう一度つくろうとすると失敗します。
    NNT_NFP_AGING_ASSERT_RESULT_EQUAL(nn::nfp::ResultAlreadyCreated(),
                                      nnt::nfp::CreateApplicationAreaWithRetry(createInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());

    return true;
} // NOLINT(impl/function_size)

bool NfpAgingStableTag::TestDeleteApplicationArea(bool needOpen)
{
    NN_LOG("NFP AGING: - DeleteApplicationArea Start\n");

    // タグをマウントします。
    nn::Result mountResult1 = MountTest();
    if (mountResult1.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(mountResult1);
        return false;
    }

    // 現時点でのタグのフィールドを記憶しておきます。
    nn::nfp::TagInfo      tagInfo;
    nn::nfp::AdminInfo    prevAdminInfo;
    nn::nfp::CommonInfo   prevCommonInfo;
    nn::nfp::RegisterInfo prevRegInfo;
    nn::nfp::ModelInfo    prevModelInfo;
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&prevAdminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&prevCommonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&prevRegInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&prevModelInfo));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevAdminInfo._reserved,
                                             sizeof(prevAdminInfo._reserved)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevCommonInfo._reserved1,
                                             sizeof(prevCommonInfo._reserved1)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevCommonInfo._reserved2,
                                             sizeof(prevCommonInfo._reserved2)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevRegInfo._reserved, sizeof(prevRegInfo._reserved)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevModelInfo._reserved,
                                             sizeof(prevModelInfo._reserved)));

    // テストのため日付を進めます。
    ScopedDateProgressor date;

    // 引数で指定された場合にだけアプリケーション専用領域をオープンします。
    if (needOpen)
    {
        nn::Result openResult = nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId);
        if (openResult.IsFailure())
        {
            NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(openResult);
            nnt::nfp::wrapper::Unmount();
            return false;
        }
    }

    // アプリケーション専用領域を削除します。
    nn::Result deleteResult = nnt::nfp::DeleteApplicationAreaWithRetry();
    if (deleteResult.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(deleteResult);
        nnt::nfp::wrapper::Unmount();
        return false;
    }

    // タグのフィールドが仕様通り変化していることを検証します。
    nn::nfp::AdminInfo    adminInfo;
    nn::nfp::CommonInfo   commonInfo;
    nn::nfp::RegisterInfo regInfo;
    nn::nfp::ModelInfo      modelInfo;
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_EQUAL(nn::nfp::ResultNeedCreate(),
                                      nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.titleId);
#else
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::ncm::ApplicationId{0}, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.accessId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevAdminInfo.moveCounter, adminInfo.moveCounter);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::nfp::RegisterInfoFlags_CommonArea, adminInfo.registerInfo);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nnt::nfp::TestNoftVersion, adminInfo.formatVersion);
#if !defined(NNT_NFP_PLATFORM_NX)// NX ではCTR/CAFE/NX以外であることを確認
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::nfp::PLATFORM_OTHER, adminInfo.platform);
#else
    NNT_NFP_AGING_ASSERT_NOT_EQUAL_N(nn::nfp::PlatformType_Ctr, adminInfo.platform);
    NNT_NFP_AGING_ASSERT_NOT_EQUAL_N(nn::nfp::PlatformType_Cafe, adminInfo.platform);
    NNT_NFP_AGING_ASSERT_NOT_EQUAL_N(nn::nfp::PlatformType_Nx, adminInfo.platform);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevCommonInfo.lastWriteDate, &commonInfo.lastWriteDate, sizeof(commonInfo.lastWriteDate)) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.writeCounter, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    NNT_NFP_AGING_ASSERT(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, 3) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.seriesId, commonInfo.seriesId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.numberingId, commonInfo.numberingId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, commonInfo.nfpVersion);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.applicationAreaSize, commonInfo.applicationAreaSize);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
#if defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
            prevRegInfo.miiData, regInfo.miiData) == 0);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    NNT_NFP_AGING_ASSERT(std::memcmp(prevModelInfo.characterID, commonInfo.characterID, 3) == 0);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevModelInfo.seriesId, modelInfo.seriesId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevModelInfo.numberingId, modelInfo.numberingId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevModelInfo.nfpType, modelInfo.nfpType);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));

    // アプリケーション専用領域が存在しない状態で削除を試行します。
    NNT_NFP_AGING_ASSERT_RESULT_EQUAL(nn::nfp::ResultNeedCreate(),
                                      nnt::nfp::DeleteApplicationAreaWithRetry());

    // タグをマウントしなおします。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    nn::Result mountResult2 = MountTest();
    if (mountResult2.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(mountResult2);
        return false;
    }

    // タグのフィールドが仕様通り変化していることを検証します。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_EQUAL(nn::nfp::ResultNeedCreate(),
                                      nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.titleId);
#else
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::ncm::ApplicationId{0}, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.accessId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevAdminInfo.moveCounter, adminInfo.moveCounter);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::nfp::RegisterInfoFlags_CommonArea, adminInfo.registerInfo);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nnt::nfp::TestNoftVersion, adminInfo.formatVersion);
#if !defined(NNT_NFP_PLATFORM_NX)// NX ではCTR/CAFE/NX以外であることを確認
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::nfp::PLATFORM_OTHER, adminInfo.platform);
#else
    NNT_NFP_AGING_ASSERT_NOT_EQUAL_N(nn::nfp::PlatformType_Ctr, adminInfo.platform);
    NNT_NFP_AGING_ASSERT_NOT_EQUAL_N(nn::nfp::PlatformType_Cafe, adminInfo.platform);
    NNT_NFP_AGING_ASSERT_NOT_EQUAL_N(nn::nfp::PlatformType_Nx, adminInfo.platform);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevCommonInfo.lastWriteDate, &commonInfo.lastWriteDate, sizeof(commonInfo.lastWriteDate)) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.writeCounter, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    NNT_NFP_AGING_ASSERT(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, 3) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.seriesId, commonInfo.seriesId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.numberingId, commonInfo.numberingId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, commonInfo.nfpVersion);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.applicationAreaSize, commonInfo.applicationAreaSize);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
#if defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
            prevRegInfo.miiData, regInfo.miiData) == 0);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    NNT_NFP_AGING_ASSERT(std::memcmp(prevModelInfo.characterID, commonInfo.characterID, 3) == 0);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevModelInfo.seriesId, modelInfo.seriesId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevModelInfo.numberingId, modelInfo.numberingId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevModelInfo.nfpType, modelInfo.nfpType);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));

    // アプリケーション専用領域が存在しない状態で削除を試行します。
    NNT_NFP_AGING_ASSERT_RESULT_EQUAL(nn::nfp::ResultNeedCreate(),
                                      nnt::nfp::DeleteApplicationAreaWithRetry());
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());

    return true;
} // NOLINT(impl/function_size)

bool NfpAgingStableTag::TestReadWriteApplicationArea(const void* pWriteData, int size)
{
    NN_LOG("NFP AGING: - ReadWriteApplicationArea Start\n");

    // タグをマウントします。
    nn::Result mountResult1 = MountTest();
    if (mountResult1.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(mountResult1);
        return false;
    }

    // 現時点でのタグのフィールドを記憶しておきます。
    nn::nfp::TagInfo      tagInfo;
    nn::nfp::AdminInfo    prevAdminInfo;
    nn::nfp::CommonInfo   prevCommonInfo;
    nn::nfp::RegisterInfo prevRegInfo;
    nn::nfp::ModelInfo      prevModelInfo;
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&prevAdminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&prevCommonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&prevRegInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&prevModelInfo));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevAdminInfo._reserved,
                                             sizeof(prevAdminInfo._reserved)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevCommonInfo._reserved1,
                                             sizeof(prevCommonInfo._reserved1)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevCommonInfo._reserved2,
                                             sizeof(prevCommonInfo._reserved2)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevRegInfo._reserved, sizeof(prevRegInfo._reserved)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevModelInfo._reserved,
                                             sizeof(prevModelInfo._reserved)));

    // テストのため日付を進めます。
    ScopedDateProgressor date;
    nn::nfp::Date tomorrow = date.GetTomorrow();

    // アプリケーション専用領域のキャッシュだけを書き換えます。
    nn::Result openResult1 = nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId);
    if (openResult1.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(openResult1);
        nnt::nfp::wrapper::Unmount();
        return false;
    }
    nn::Result writeResult = nnt::nfp::wrapper::SetApplicationArea(pWriteData, size, tagInfo.tagId);
    if (writeResult.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(writeResult);
        nnt::nfp::wrapper::Unmount();
        return false;
    }

    // この時点では共有領域に変化はなく、アプリケーション専用領域だけが書き換わっています。
    nn::nfp::AdminInfo    adminInfo;
    nn::nfp::CommonInfo   commonInfo;
    nn::nfp::RegisterInfo regInfo;
    nn::nfp::ModelInfo      modelInfo;
    nn::Bit8                  appArea[nn::nfp::ApplicationAreaSizeV2];
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea,
                                                                              sizeof(appArea)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevAdminInfo,  &adminInfo,  sizeof(adminInfo))  == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevCommonInfo, &commonInfo, sizeof(commonInfo)) == 0);
#if defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
            prevRegInfo.miiData, regInfo.miiData) == 0);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevModelInfo,    &modelInfo,    sizeof(modelInfo))    == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(appArea, pWriteData, size) == 0);

    // タグに書き込みます。
    nn::Result flushResult = nnt::nfp::FlushWithRetry();
    if (flushResult.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(flushResult);
        nnt::nfp::wrapper::Unmount();
        return false;
    }

    // 共有領域やアプリケーション専用領域が仕様通り変化していることを確認します。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea,
                                                                              sizeof(appArea)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevAdminInfo,  &adminInfo,  sizeof(adminInfo))  == 0);
#if defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
            prevRegInfo.miiData, regInfo.miiData) == 0);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevModelInfo,    &modelInfo,    sizeof(modelInfo))    == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(appArea, pWriteData, size) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&tomorrow, &commonInfo.lastWriteDate, sizeof(tomorrow)) == 0);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    NNT_NFP_AGING_ASSERT(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, sizeof(commonInfo) - 6) == 0);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.writeCounter + 1, commonInfo.writeCounter);

    // タグをマウントしなおします。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    nn::Result mountResult2 = MountTest();
    if (mountResult2.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(mountResult2);
        return false;
    }

    // 共有領域やアプリケーション専用領域が仕様通り変化していることを確認します。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea,
                                                                              sizeof(appArea)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevAdminInfo,  &adminInfo,  sizeof(adminInfo))  == 0);
#if defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
            prevRegInfo.miiData, regInfo.miiData) == 0);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevModelInfo,    &modelInfo,    sizeof(modelInfo))    == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(appArea, pWriteData, size) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&tomorrow, &commonInfo.lastWriteDate, sizeof(tomorrow)) == 0);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    NNT_NFP_AGING_ASSERT(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, sizeof(commonInfo) - 6) == 0);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.writeCounter + 1, commonInfo.writeCounter);

    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    return true;
} // NOLINT(impl/function_size)

#if defined(NNT_NFP_PLATFORM_NX)
bool NfpAgingStableTag::TestRecreateApplicationArea(nn::Bit32 accessId, const void* pWriteData, int size)
{
    NN_LOG("NFP AGING: - RecreateApplicationArea Start\n");

    // タグをマウントします。
    nn::Result mountResult1 = MountTest();
    if (mountResult1.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(mountResult1);
        return false;
    }

    // 現時点でのタグのフィールドを記憶しておきます。
    nn::nfp::TagInfo      tagInfo;
    nn::nfp::AdminInfo    prevAdminInfo;
    nn::nfp::CommonInfo   prevCommonInfo;
    nn::nfp::RegisterInfo prevRegInfo;
    nn::nfp::ModelInfo      prevModelInfo;
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&prevAdminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&prevCommonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&prevRegInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&prevModelInfo));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevAdminInfo._reserved,
                                             sizeof(prevAdminInfo._reserved)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevCommonInfo._reserved1,
                                             sizeof(prevCommonInfo._reserved1)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevCommonInfo._reserved2,
                                             sizeof(prevCommonInfo._reserved2)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevRegInfo._reserved, sizeof(prevRegInfo._reserved)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevModelInfo._reserved,
                                             sizeof(prevModelInfo._reserved)));

    // アプリケーション専用領域のキャッシュだけを書き換えます。
    nn::Result openResult1 = nnt::nfp::wrapper::OpenApplicationArea(accessId == nnt::nfp::NormalTagId ? nnt::nfp::NormalTagId + 1 : nnt::nfp::NormalTagId);
    if (openResult1.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(openResult1);
        nnt::nfp::wrapper::Unmount();
        return false;
    }
    nn::nfp::ApplicationAreaCreateInfo createInfo;
    createInfo.accessId        = accessId;
    createInfo.pInitialData    = const_cast<void*>(pWriteData);
    createInfo.initialDataSize = size;
    nn::Result writeResult = nnt::nfp::wrapper::RecreateApplicationArea(createInfo);
    if (writeResult.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(writeResult);
        nnt::nfp::wrapper::Unmount();
        return false;
    }

    // この時点ではアクセスIDとアプリケーション専用領域だけが書き換わっています。
    nn::nfp::AdminInfo    adminInfo;
    nn::nfp::CommonInfo   commonInfo;
    nn::nfp::RegisterInfo regInfo;
    nn::nfp::ModelInfo      modelInfo;
    nn::Bit8                  appArea[nn::nfp::ApplicationAreaSizeV2];
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea,
                                                                              sizeof(appArea)));

    prevAdminInfo.accessId = accessId;
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevAdminInfo,  &adminInfo,  sizeof(adminInfo))  == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevCommonInfo, &commonInfo, sizeof(commonInfo)) == 0);
#if defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
            prevRegInfo.miiData, regInfo.miiData) == 0);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevModelInfo,    &modelInfo,    sizeof(modelInfo))    == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(appArea, pWriteData, size) == 0);

    // タグに書き込みます。
    nn::Result flushResult = nnt::nfp::FlushWithRetry();
    if (flushResult.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(flushResult);
        nnt::nfp::wrapper::Unmount();
        return false;
    }

    // 共有領域やアプリケーション専用領域が仕様通り変化していることを確認します。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea,
                                                                              sizeof(appArea)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevAdminInfo,  &adminInfo,  sizeof(adminInfo))  == 0);
#if defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
            prevRegInfo.miiData, regInfo.miiData) == 0);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevModelInfo,    &modelInfo,    sizeof(modelInfo))    == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(appArea, pWriteData, size) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.writeCounter + 1, commonInfo.writeCounter);

    // タグをマウントしなおします。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    nn::Result mountResult2 = MountTest();
    if (mountResult2.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(mountResult2);
        return false;
    }

    // 共有領域やアプリケーション専用領域が仕様通り変化していることを確認します。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(accessId));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea,
                                                                              sizeof(appArea)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevAdminInfo,  &adminInfo,  sizeof(adminInfo))  == 0);
#if defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
            prevRegInfo.miiData, regInfo.miiData) == 0);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevModelInfo,    &modelInfo,    sizeof(modelInfo))    == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(appArea, pWriteData, size) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.writeCounter + 1, commonInfo.writeCounter);

    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    return true;
} // NOLINT(impl/function_size)
#endif //defined(NNT_NFP_PLATFORM_NX)

bool NfpAgingStableTag::TestReadWriteRegInfo()
{
    NN_LOG("NFP AGING: - ReadWriteRegInfo Start\n");

    const int fontRegion = nn::nfp::FontRegion_Count - 1;

    // タグをマウントします。
    nn::Result mountResult1 = MountTest();
    if (mountResult1.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(mountResult1);
        return false;
    }

    // 現時点でのタグのフィールドを記憶しておきます。
    nn::nfp::TagInfo      tagInfo;
    nn::nfp::AdminInfo    prevAdminInfo;
    nn::nfp::CommonInfo   prevCommonInfo;
    nn::nfp::RegisterInfo prevRegInfo;
    nn::nfp::ModelInfo    prevModelInfo;
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&prevAdminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&prevCommonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_EQUAL(nn::nfp::ResultNeedRegister(),
                                      nnt::nfp::wrapper::GetRegisterInfo(&prevRegInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&prevModelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_EQUAL(nn::nfp::ResultNeedCreate(),
                                      nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevAdminInfo._reserved, sizeof(prevAdminInfo._reserved)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevCommonInfo._reserved1,
                                             sizeof(prevCommonInfo._reserved1)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevCommonInfo._reserved2,
                                             sizeof(prevCommonInfo._reserved2)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevModelInfo._reserved,
                                             sizeof(prevModelInfo._reserved)));

    // テストのため日付を進めます。
    ScopedDateProgressor date;
    nn::nfp::Date tomorrow = date.GetTomorrow();

    // 登録情報のキャッシュだけを書き換えます。
    nn::nfp::RegisterInfoPrivate regInfoSet;
#if !defined(NNT_NFP_PLATFORM_NX)// NX ではAPIが削除されている
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeRegisterInfoSet(&regInfoSet));
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if defined(NNT_NFP_LIB_VERSION_BETA)
    #if defined(NNT_NFP_LIB_MII_ENABLE)
    nnt::nfp::BuildMiiData(regInfoSet.miiData);
    #endif // defined(NNT_NFP_LIB_MII_ENABLE)
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
    std::memcpy(regInfoSet.nickname, nnt::nfp::NormalTagNickName, sizeof(nnt::nfp::NormalTagNickName));
    regInfoSet.fontRegion = fontRegion;
    nn::Result registerResult = nnt::nfp::wrapper::SetRegisterInfo(regInfoSet);
    if (registerResult.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(registerResult);
        nnt::nfp::wrapper::Unmount();
        return false;
    }

    // 登録情報を取得して仕様通りの値が設定されていることを確認します。
    nn::nfp::Date         defaultDate = { 2000, 1, 1 };
    nn::nfp::AdminInfo    adminInfo;
    nn::nfp::CommonInfo   commonInfo;
    nn::nfp::RegisterInfo regInfo;
    nn::nfp::ModelInfo      modelInfo;
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_EQUAL(nn::nfp::ResultNeedCreate(),
                                      nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.titleId);
#else
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::ncm::ApplicationId{0}, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.accessId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevAdminInfo.moveCounter, adminInfo.moveCounter);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::nfp::RegisterInfoFlags_CommonArea, adminInfo.registerInfo);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevAdminInfo.platform, adminInfo.platform);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&defaultDate, &commonInfo.lastWriteDate, sizeof(commonInfo.lastWriteDate)) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.writeCounter, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    NNT_NFP_AGING_ASSERT(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, sizeof(commonInfo.characterID)) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.seriesId, commonInfo.seriesId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.numberingId, commonInfo.numberingId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.nfpVersion, commonInfo.nfpVersion);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::nfp::ApplicationAreaSizeV2, commonInfo.applicationAreaSize);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevModelInfo, &modelInfo, sizeof(modelInfo)) == 0);
#if defined(NNT_NFP_LIB_VERSION_BETA)
    nn::nfp::RegisterInfoPrivate regInfoGet;
    nnt::nfp::wrapper::GetRegisterInfo(&regInfoGet);
    #if defined(NNT_NFP_LIB_MII_ENABLE)
    EXPECT_TRUE(std::memcmp(&regInfoGet.miiData, &regInfoSet.miiData,
            regInfoSet.miiData.StructureSize - ExceptDataSize) == 0);
    #endif // defined(NNT_NFP_LIB_MII_ENABLE)
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
    NNT_NFP_AGING_ASSERT(std::memcmp(nnt::nfp::NormalTagNickName,
                                     regInfo.nickname,
                                     sizeof(nnt::nfp::NormalTagNickName)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&tomorrow, &regInfo.registerDate, sizeof(tomorrow))  == 0);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(regInfo._reserved, sizeof(regInfo._reserved)));
    NNT_NFP_AGING_ASSERT_EQUAL_N(fontRegion, regInfo.fontRegion);
#if !defined(NNT_NFP_PLATFORM_NX) //NX には国コードがない
    NNT_NFP_AGING_ASSERT_EQUAL_N(nnt::nfp::GetCountry(), regInfo.country);
#endif //!defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(regInfo._reserved, sizeof(regInfo._reserved)));

    // タグに書き込みます。
    nn::Result flushResult = nnt::nfp::FlushWithRetry();
    if (flushResult.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(flushResult);
        nnt::nfp::wrapper::Unmount();
        return false;
    }

    // 共有領域やアプリケーション専用領域が仕様通り変化していることを確認します。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_EQUAL(nn::nfp::ResultNeedCreate(),
                                      nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.titleId);
#else
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::ncm::ApplicationId{0}, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.accessId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevAdminInfo.moveCounter + 1, adminInfo.moveCounter);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::nfp::RegisterInfoFlags_CommonArea, adminInfo.registerInfo);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevAdminInfo.platform, adminInfo.platform);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&tomorrow, &commonInfo.lastWriteDate, sizeof(tomorrow)) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.writeCounter + 1, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    NNT_NFP_AGING_ASSERT(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, sizeof(commonInfo.characterID)) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.seriesId, commonInfo.seriesId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.numberingId, commonInfo.numberingId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.nfpVersion, commonInfo.nfpVersion);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::nfp::ApplicationAreaSizeV2, commonInfo.applicationAreaSize);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
#if defined(NNT_NFP_LIB_VERSION_BETA)
    std::memset(&regInfoGet, 0x00, sizeof(regInfoGet));
    nnt::nfp::wrapper::GetRegisterInfo(&regInfoGet);
    #if defined(NNT_NFP_LIB_MII_ENABLE)
    EXPECT_TRUE(std::memcmp(&regInfoGet.miiData, &regInfoSet.miiData,
            regInfoSet.miiData.StructureSize - ExceptDataSize) == 0);
    #endif // defined(NNT_NFP_LIB_MII_ENABLE)
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
    NNT_NFP_AGING_ASSERT(std::memcmp(nnt::nfp::NormalTagNickName,
                                     regInfo.nickname,
                                     sizeof(nnt::nfp::NormalTagNickName)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&tomorrow, &regInfo.registerDate, sizeof(tomorrow))  == 0);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(regInfo._reserved, sizeof(regInfo._reserved)));
    NNT_NFP_AGING_ASSERT_EQUAL_N(fontRegion, regInfo.fontRegion);
#if !defined(NNT_NFP_PLATFORM_NX) //NX には国コードがない
    NNT_NFP_AGING_ASSERT_EQUAL_N(nnt::nfp::GetCountry(), regInfo.country);
#endif //!defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(regInfo._reserved, sizeof(regInfo._reserved)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevModelInfo, &modelInfo, sizeof(modelInfo)) == 0);

    // タグをマウントしなおします。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    nn::Result mountResult2 = MountTest();
    if (mountResult2.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(mountResult2);
        nnt::nfp::wrapper::Unmount();
        return false;
    }

    // 共有領域やアプリケーション専用領域が仕様通り変化していることを確認します。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_EQUAL(nn::nfp::ResultNeedCreate(),
                                      nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.titleId);
#else
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::ncm::ApplicationId{0}, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.accessId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevAdminInfo.moveCounter + 1, adminInfo.moveCounter);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::nfp::RegisterInfoFlags_CommonArea, adminInfo.registerInfo);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevAdminInfo.platform, adminInfo.platform);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&tomorrow, &commonInfo.lastWriteDate, sizeof(tomorrow)) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.writeCounter + 1, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    NNT_NFP_AGING_ASSERT(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, sizeof(commonInfo.characterID)) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.seriesId, commonInfo.seriesId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.numberingId, commonInfo.numberingId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.nfpVersion, commonInfo.nfpVersion);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::nfp::ApplicationAreaSizeV2, commonInfo.applicationAreaSize);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
#if defined(NNT_NFP_LIB_VERSION_BETA)
    std::memset(&regInfoGet, 0x00, sizeof(regInfoGet));
    nnt::nfp::wrapper::GetRegisterInfo(&regInfoGet);
    #if defined(NNT_NFP_LIB_MII_ENABLE)
    EXPECT_TRUE(std::memcmp(&regInfoGet.miiData, &regInfoSet.miiData,
            regInfoSet.miiData.StructureSize - ExceptDataSize) == 0);
    #endif // defined(NNT_NFP_LIB_MII_ENABLE)
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
    NNT_NFP_AGING_ASSERT(std::memcmp(nnt::nfp::NormalTagNickName,
                                     regInfo.nickname,
                                     sizeof(nnt::nfp::NormalTagNickName))  == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&tomorrow, &regInfo.registerDate, sizeof(tomorrow))  == 0);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(regInfo._reserved, sizeof(regInfo._reserved)));
    NNT_NFP_AGING_ASSERT_EQUAL_N(fontRegion, regInfo.fontRegion);
#if !defined(NNT_NFP_PLATFORM_NX) //NX には国コードがない
    NNT_NFP_AGING_ASSERT_EQUAL_N(nnt::nfp::GetCountry(), regInfo.country);
#endif //!defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(regInfo._reserved, sizeof(regInfo._reserved)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevModelInfo, &modelInfo, sizeof(modelInfo)) == 0);

    // もう一度登録情報をタグに書き込みます。
    registerResult = nnt::nfp::wrapper::SetRegisterInfo(regInfoSet);
    if (registerResult.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(registerResult);
        nnt::nfp::wrapper::Unmount();
        return false;
    }
    flushResult = nnt::nfp::FlushWithRetry();
    if (flushResult.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(flushResult);
        nnt::nfp::wrapper::Unmount();
        return false;
    }

    // 今度は新規登録では無いので writeCounter や registerDate が変動しません。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_EQUAL_N(prevCommonInfo.writeCounter + 1, commonInfo.writeCounter);
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT(std::memcmp(&tomorrow, &regInfo.registerDate, sizeof(tomorrow)) == 0);

    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    return true;
} // NOLINT(impl/function_size)

#if defined(NNT_NFP_LIB_VERSION_ALPHA)
bool NfpAgingStableTag::TestRestore(nn::nfp::BreakType breakType)
{
    NN_LOG("NFP AGING: - Restore Start\n");

    // タグをマウントします。
    nn::Result mountResult1 = MountTest();
    if (mountResult1.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(mountResult1);
        return false;
    }

    // 現時点でのタグのフィールドを記憶しておきます。
    nn::nfp::TagInfo      tagInfo;
    nn::nfp::AdminInfo    prevAdminInfo;
    nn::nfp::CommonInfo   prevCommonInfo;
    nn::nfp::RegisterInfo prevRegInfo;
    nn::nfp::ModelInfo    prevModelInfo;
    nn::Bit8              prevAppArea[nn::nfp::ApplicationAreaSizeV2];
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&prevAdminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&prevCommonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&prevRegInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&prevModelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(prevAppArea,
                                                                              sizeof(prevAppArea)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevAdminInfo._reserved, sizeof(prevAdminInfo._reserved)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevCommonInfo._reserved1,
                                             sizeof(prevCommonInfo._reserved1)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevCommonInfo._reserved2,
                                             sizeof(prevCommonInfo._reserved2)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevRegInfo._reserved, sizeof(prevRegInfo._reserved)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(prevModelInfo._reserved,
                                             sizeof(prevModelInfo._reserved)));

    // タグを破壊します。
    nn::Result breakResult = nnt::nfp::BreakTagWithRetry(breakType);
    if (breakResult.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(breakResult);
        nnt::nfp::wrapper::Unmount();
        return false;
    }

    // タグが壊れていることを確認します。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    nn::Result mountResult2 = MountTest();
    if (!nn::nfp::ResultNeedRestore().Includes(mountResult2))
    {
        NNT_NFP_AGING_ASSERT_RESULT_EQUAL(nn::nfp::ResultNeedRestore(), mountResult2);
        return false;
    }

    // タグを復旧します。
    nn::Result restoreResult = nnt::nfp::RestoreWithRetry();
    if (restoreResult.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(restoreResult);
        return false;
    }

#if defined(NNT_NFP_PLATFORM_NX) //NXではRestore()実行後にマウントされないのでマウントする
    nnt::nfp::DoMount();
#endif // defined(NNT_NFP_PLATFORM_NX)

    // 復旧結果に問題が無いことを確認します。
    nn::nfp::CommonInfo   commonInfo;
    nn::nfp::RegisterInfo regInfo;
    nn::nfp::AdminInfo    adminInfo;
    nn::nfp::ModelInfo    modelInfo;
    nn::Bit8              appArea[nn::nfp::ApplicationAreaSizeV2];
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea,
                                                                              sizeof(appArea)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevCommonInfo, &commonInfo, sizeof(commonInfo)) == 0);
#if defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
            prevRegInfo.miiData, regInfo.miiData) == 0);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevAdminInfo, &adminInfo, sizeof(adminInfo)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevModelInfo, &modelInfo, sizeof(modelInfo)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevAppArea, appArea, sizeof(appArea)) == 0);

    // タグをマウントしなおします。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    nn::Result mountResult3 = MountTest();
    if (mountResult3.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(mountResult3);
        return false;
    }

    // 復旧結果に問題が無いことを確認します。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea,
                                                                              sizeof(appArea)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevCommonInfo, &commonInfo, sizeof(commonInfo)) == 0);
#if defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
            prevRegInfo.miiData, regInfo.miiData) == 0);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevAdminInfo, &adminInfo, sizeof(adminInfo)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(&prevModelInfo, &modelInfo, sizeof(modelInfo)) == 0);
    NNT_NFP_AGING_ASSERT(std::memcmp(prevAppArea, appArea, sizeof(appArea)) == 0);

    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    return true;
}
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA)

bool NfpAgingStableTag::TestFormat()
{
    NN_LOG("NFP AGING: - Format Start\n");

    // タグを初期化します。
    nn::Result formatResult = nnt::nfp::FormatWithRetry(nnt::nfp::ZeroTagData,
                                                        sizeof(nnt::nfp::ZeroTagData));
    if (formatResult.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(formatResult);
        return false;
    }

    // タグをマウントします。
    nn::Result mountResult1 = MountTest();
    if (mountResult1.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(mountResult1);
        return false;
    }

    // 想定通りに値が初期化されていることを確認します。
    nn::nfp::Date         defaultDate = { 2000, 1, 1 };
    nn::nfp::AdminInfo    adminInfo;
    nn::nfp::CommonInfo   commonInfo;
    nn::nfp::RegisterInfo regInfo;
    nn::nfp::ModelInfo      modelInfo;
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_EQUAL(nn::nfp::ResultNeedRegister(),
                                      nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_EQUAL(nn::nfp::ResultNeedCreate(),
                                      nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.titleId);
#else
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::ncm::ApplicationId{0}, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.accessId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.moveCounter);
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.registerInfo);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nnt::nfp::TestNoftVersion, adminInfo.formatVersion);
#if !defined(NNT_NFP_PLATFORM_NX)// NX ではCTR/CAFE/NX以外であることを確認
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::nfp::PLATFORM_OTHER, adminInfo.platform);
#else
    NNT_NFP_AGING_ASSERT_NOT_EQUAL_N(nn::nfp::PlatformType_Ctr, adminInfo.platform);
    NNT_NFP_AGING_ASSERT_NOT_EQUAL_N(nn::nfp::PlatformType_Cafe, adminInfo.platform);
    NNT_NFP_AGING_ASSERT_NOT_EQUAL_N(nn::nfp::PlatformType_Nx, adminInfo.platform);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&defaultDate, &commonInfo.lastWriteDate, sizeof(defaultDate)) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, commonInfo.writeCounter);
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, commonInfo.nfpVersion);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::nfp::ApplicationAreaSizeV2, commonInfo.applicationAreaSize);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));

    // タグをマウントしなおします。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    nn::Result mountResult2 = MountTest();
    if (mountResult2.IsFailure())
    {
        NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(mountResult2);
        return false;
    }

    // 想定通りに値が初期化されていることを確認します。
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_AGING_ASSERT_RESULT_EQUAL(nn::nfp::ResultNeedRegister(),
                                      nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_NFP_AGING_ASSERT_RESULT_EQUAL(nn::nfp::ResultNeedCreate(),
                                      nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.titleId);
#else
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::ncm::ApplicationId{0}, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.accessId);
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.moveCounter);
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, adminInfo.registerInfo);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nnt::nfp::TestNoftVersion, adminInfo.formatVersion);
#if !defined(NNT_NFP_PLATFORM_NX)// NX ではCTR/CAFE/NX以外であることを確認
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::nfp::PLATFORM_OTHER, adminInfo.platform);
#else
    NNT_NFP_AGING_ASSERT_NOT_EQUAL_N(nn::nfp::PlatformType_Ctr, adminInfo.platform);
    NNT_NFP_AGING_ASSERT_NOT_EQUAL_N(nn::nfp::PlatformType_Cafe, adminInfo.platform);
    NNT_NFP_AGING_ASSERT_NOT_EQUAL_N(nn::nfp::PlatformType_Nx, adminInfo.platform);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    NNT_NFP_AGING_ASSERT(std::memcmp(&defaultDate, &commonInfo.lastWriteDate, sizeof(defaultDate)) == 0);
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, commonInfo.writeCounter);
    NNT_NFP_AGING_ASSERT_EQUAL_N(0, commonInfo.nfpVersion);
    NNT_NFP_AGING_ASSERT_EQUAL_N(nn::nfp::ApplicationAreaSizeV2, commonInfo.applicationAreaSize);
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
    NNT_NFP_AGING_ASSERT(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));

    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    return true;
}

//================================================================================
// エイジングの実装です。
//================================================================================

TEST_F(NfpAgingStableTag,TestCaseAgingStableTag)
{
    NNT_NFP_ASSERT_RESULT_SUCCESS_FATAL(nnt::nfp::InitializeSystemWithRetry());
#if defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(nnt::nfp::SetDefaultAvailabilityChangeEvent());
#endif // defined(NNT_NFP_PLATFORM_NX)
    nnt::nfp::DoSearchDeviceHandle();

    // タグの耐用回数までエイジングを行います。
    for (int i = 0; i < AgingCountMax; ++i)
    {
        NN_LOG("NFP AGING: @COUNT = %d/%d\n", i + 1, AgingCountMax);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectに対応しないのでStopDetectionで代用
        // 前回のエイジングの結果が今回に影響するのを防ぐため、一旦切断からやり直します。
        nnt::nfp::wrapper::Disconnect();

        // FANGATE への接続を開始します。
        nn::Result connectionResult;
        connectionResult = nnt::nfp::wrapper::Connect();
        if (connectionResult.IsFailure())
        {
            NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(connectionResult);
            continue;
        }

        // 接続が完了するまで待機します。
        nn::nfp::TargetConnectionStatus connectionStatus;
        for (int t = 0; t < 50; ++t)
        {
            nn::Result getTargetResult = nnt::nfp::wrapper::GetTargetConnectionStatus(&connectionStatus);
            NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(getTargetResult);
            if (connectionStatus != nn::nfp::TARGET_CONNECTING)
            {
                break;
            }
            nnt::nfp::Sleep(100);
        }
        if (connectionStatus != nn::nfp::TARGET_CONNECTED)
        {
            NNT_NFP_AGING_ASSERT_EQUAL_N(nn::nfp::TARGET_CONNECTING, connectionStatus);
            continue;
        }
#else
        NNT_NFP_AGING_ASSERT(!nnt::nfp::WaitForAvailabilityChange(1));

        //StopDetection()が実行可能な状態の時のみStopDetection()を実行する
        nn::nfp::DeviceState deviceState = nnt::nfp::wrapper::GetDeviceState();
        if((deviceState != nn::nfp::DeviceState_Init) &&
           (deviceState != nn::nfp::DeviceState_Unexpected))
        {
            nn::Result stopResult = nnt::nfp::wrapper::StopDetection();
            if (stopResult.IsFailure())
            {
                NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(stopResult);
                continue;
            }
        }
#endif // !defined(NNT_NFP_PLATFORM_NX)

        // タグの検出を開始します。
        nn::Result startResult = nnt::nfp::wrapper::StartDetection();
        if (startResult.IsFailure())
        {
            NNT_NFP_AGING_ASSERT_RESULT_SUCCESS(startResult);
            continue;
        }

        // タグが見つかるまで待機します。
        bool isActivated = nnt::nfp::WaitForActivate();
        if (!isActivated)
        {
            NNT_NFP_AGING_ASSERT_EQUAL_N(true, isActivated);
            continue;
        }

        // タグを初期化します。
        if (!TestFormat())
        {
            continue;
        }

        // 初期登録情報の作成に関するテストです。
        if (!TestReadWriteRegInfo())
        {
            continue;
        }

        // アプリケーション専用領域の読み書きに関するテストです。
        if (!TestCreateApplicationArea())
        {
            continue;
        }
        if (!TestReadWriteApplicationArea(nnt::nfp::ZeroTagData, sizeof(nnt::nfp::ZeroTagData) / 2))
        {
            continue;
        }
        if (!TestReadWriteApplicationArea(nnt::nfp::NormalTagData, sizeof(nnt::nfp::NormalTagData)))
        {
            continue;
        }
#if defined(NNT_NFP_PLATFORM_NX)
        if (!TestRecreateApplicationArea(nnt::nfp::NormalTagId + 1, nnt::nfp::ZeroTagData, sizeof(nnt::nfp::ZeroTagData) / 2))
        {
            continue;
        }
        if (!TestRecreateApplicationArea(nnt::nfp::NormalTagId, nnt::nfp::NormalTagData, sizeof(nnt::nfp::NormalTagData)))
        {
            continue;
        }
#endif
        if (!TestDeleteApplicationArea(false))
        {
            continue;
        }

        // タグの復旧に関するテストです。
#if defined(NNT_NFP_LIB_VERSION_ALPHA)
        if (!TestCreateApplicationArea())
        {
            continue;
        }
        if (!TestRestore(nn::nfp::BreakType_Hmac))
        {
            continue;
        }
        if (!TestRestore(nn::nfp::BreakType_Activation))
        {
            continue;
        }
        if (!TestDeleteApplicationArea(true))
        {
            continue;
        }
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA)
    }

    NNT_NFP_ASSERT_RESULT_SUCCESS_FATAL(nnt::nfp::FinalizeSystem());
} // NOLINT(impl/function_size)

