﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/hid.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Event.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testMifare_Common.h>

namespace
{
//================================================================================
// このテストで使用する定義です。
//================================================================================
    enum WaitTime
    {
        WaitTime_TagSwitch              = 3000,
        WaitTime_CountDown              = 1000,
        WaitTime_AttachTag              = 10000,
        WaitTime_NextStep               = 1000,
        WaitTime_BluetoothSignalRestore = 10000,
    };

    // スレッド関連の定義
    const size_t StackSize = 32 * 1024;//4096;
    nn::os::ThreadType g_RunApiThread;
    NN_ALIGNAS( 4096 ) char g_Stack[ StackSize ];
    nn::os::EventType g_BootUpEvent;
    nn::os::EventType g_RunApiEvent;
    nn::Result g_RunApiThreadResult;

    // 要求API設定
    enum RunMifareApi
    {
        RunMifareApi_Read = 0,
        RunMifareApi_Write,
    };
    // 初期値はFlushの試験に設定する
    RunMifareApi g_RunMifareApi = RunMifareApi_Read;

    // マルチデバイス関連定義
    // Mifareタグは2枚で試験を実施する
#if defined(NNT_MIFARE_MULTI_2DEVICES)
    const int DeviceCountMax = 2;
#else // defined(NNT_MIFARE_MULTI_2DEVICES)
    const int DeviceCountMax = 4;
#endif // defined(NNT_MIFARE_MULTI_2DEVICES)
    const int MaxCore = 3;
    struct TagData
    {
        nn::Bit8 *NormalTagData;
        unsigned char *NormalTagNickName;
    };
    struct threadinfo
    {
        int deviceNum;
    };
#if defined(NNT_MIFARE_MULTI_2DEVICES)
    threadinfo g_threadParam[DeviceCountMax] = {{0},{1}};
#else // defined(NNT_MIFARE_MULTI_2DEVICES)
    threadinfo g_threadParam[DeviceCountMax] = {{0},{1},{2},{3}};
#endif // defined(NNT_MIFARE_MULTI_2DEVICES)
    void DeviceTestThread(void* pArg) NN_NOEXCEPT;
    const size_t DeviceThreadStackSize = 32 * 1024;
    nn::os::ThreadType g_DeviceThread[DeviceCountMax];
    NN_OS_ALIGNAS_THREAD_STACK char g_DeviceThreadStack[DeviceCountMax][DeviceThreadStackSize];
    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle g_CurrentDeviceHandleMulti[DeviceCountMax];
    const int WaitTimeout = 5000;
    const int TestLoop = 10;

//================================================================================
// テストスイート全体で共通の処理です。
//================================================================================

    void ThreadRunMifareApi(void* pPtr)
    {
        NN_UNUSED(pPtr);

        g_RunApiThreadResult = NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNfcDeviceError();

        NN_LOG("### %s Activated !!\n", __FUNCTION__);

        if(nn::os::TryWaitEvent(&g_BootUpEvent) == true)
        {
            NN_LOG("### Illeagal Event State !!\n");
            nn::os::SignalEvent(&g_BootUpEvent);
            nn::os::ClearEvent(&g_BootUpEvent);
            EXPECT_TRUE(false);
            return;
        }

        NN_LOG("### Indicate Thread Boot!!\n");
        nn::os::SignalEvent(&g_BootUpEvent);

        NN_LOG("### Waiting for RunMifareApi Signal...\n");
        nn::os::WaitEvent(&g_RunApiEvent);

        switch (g_RunMifareApi)
        {
        case RunMifareApi_Write:
            {
                NN_LOG("### Run Write\n");
                const int WriteFullBlockCount = 47;
                NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) writeBlockParameter[WriteFullBlockCount] = {};
                nnt::mifare::PrepareWriteParam(writeBlockParameter, WriteFullBlockCount);
                nnt::mifare::PrintWriteBlockParameter(writeBlockParameter, WriteFullBlockCount);
                g_RunApiThreadResult = nnt::mifare::wrapper::Write(
                        writeBlockParameter, WriteFullBlockCount);
            }
            break;
        case RunMifareApi_Read:
            {
                NN_LOG("### Run Read\n");
                const int ReadFullBlockCount = 47;
                NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockData) readBlockData[ReadFullBlockCount] = {};
                NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockParameter) readBlockParameter[ReadFullBlockCount] = {};
                nnt::mifare::PrepareReadParam(readBlockParameter, ReadFullBlockCount);
                g_RunApiThreadResult = nnt::mifare::wrapper::Read(
                        readBlockData, readBlockParameter, ReadFullBlockCount);
                nnt::mifare::PrintReadBlockData(readBlockData, ReadFullBlockCount);
            }
            break;
        default:
            {
                NN_LOG("### Nothing\n");
            }
            break;
        }
        NN_LOG("### Indicate End RunApi Signal RunApi: %d, Result: %s !!\n",
                g_RunMifareApi, nnt::mifare::GetMifareResultTypeString(g_RunApiThreadResult));
        nn::os::ClearEvent(&g_RunApiEvent);

        if(nn::os::TryWaitEvent(&g_BootUpEvent) == true)
        {
            nn::os::SignalEvent(&g_BootUpEvent);
        }

        nn::os::ClearEvent(&g_BootUpEvent);
        return;
    }

    inline void CheckDeactive(bool isDeactive)
    {
        EXPECT_EQ(nnt::mifare::WaitForDeactivate(),isDeactive);
    }

    void TestInitRunApiThreadWait()
    {
        // スタック準備
        nn::os::InitializeEvent(&g_BootUpEvent, false,  nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&g_RunApiEvent, false,  nn::os::EventClearMode_ManualClear);

        nn::os::ClearEvent(&g_BootUpEvent);
        nn::os::ClearEvent(&g_RunApiEvent);

        // RunMifareApi thread Start
        NNT_MIFARE_EXPECT_RESULT_SUCCESS(
                nn::os::CreateThread(&g_RunApiThread,
                                     ThreadRunMifareApi,
                                     nullptr,
                                     g_Stack,
                                     sizeof(g_Stack),
                                     nn::os::DefaultThreadPriority));
        nn::os::StartThread(&g_RunApiThread);

        NN_LOG("### Wainting for Thraed Start \n");
        nn::os::WaitEvent(&g_BootUpEvent);
        NN_LOG("### Thread Start Signal Detected !! \n");
        return;
    }

    void TestMainTagDropInProgress()
    {
        NN_LOG("# TestMainTagDropInProgress \n");

        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE DROP TAG Ready... 5 ");
        nnt::mifare::Sleep(WaitTime_CountDown);
        NN_LOG(" 4 ");
        nnt::mifare::Sleep(WaitTime_CountDown);
        NN_LOG(" 3 ");
        nnt::mifare::Sleep(WaitTime_CountDown);
        NN_LOG(" 2 ");
        nnt::mifare::Sleep(WaitTime_CountDown);
        NN_LOG(" 1 ");
        nnt::mifare::Sleep(WaitTime_CountDown);
        NN_LOG(" NOW!! \n");
        NN_LOG("==================================================\n");

        nn::os::SignalEvent(&g_RunApiEvent);

        // DeactivateEventが発生する
        CheckDeactive(true);

        EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState() );
        EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init, nnt::mifare::wrapper::GetState() );

        // Flushスレッドの終了まで待つ
        nn::os::WaitThread(&g_RunApiThread);
        nn::os::DestroyThread(&g_RunApiThread);

        NNT_MIFARE_EXPECT_RESULT_EQUAL(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultTagNotFound(), g_RunApiThreadResult);

        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE ATTACH a TAG!!\n");
        NN_LOG("==================================================\n");
        nnt::mifare::DoActivate();
        NN_LOG(" TAG Attached !!\n");
        EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState());

        return;
    }

    void TestMainBluetoothCutInProgress()
    {
        NN_LOG("# TestMainBluetoothCutInProgress \n");

        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE CUT BT SIGNAL Ready... 5 ");
        nnt::mifare::Sleep(WaitTime_CountDown);
        NN_LOG(" 4 ");
        nnt::mifare::Sleep(WaitTime_CountDown);
        NN_LOG(" 3 ");
        nnt::mifare::Sleep(WaitTime_CountDown);
        NN_LOG(" 2 ");
        nnt::mifare::Sleep(WaitTime_CountDown);
        NN_LOG(" 1 ");
        nnt::mifare::Sleep(WaitTime_CountDown);
        NN_LOG(" NOW!! \n");
        NN_LOG("==================================================\n");

        nn::os::SignalEvent(&g_RunApiEvent);

        // DeactivateEventが発生する
        CheckDeactive(true);

        EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Unexpected, nnt::mifare::wrapper::GetDeviceState() );
        EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init, nnt::mifare::wrapper::GetState() );

        // Flushスレッドの終了まで待つ
        nn::os::WaitThread(&g_RunApiThread);
        nn::os::DestroyThread(&g_RunApiThread);

        NNT_MIFARE_EXPECT_RESULT_EQUAL(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNfcDeviceNotFound(), g_RunApiThreadResult);

        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE RESTORE BT MODULE (Within %dms) ...       \n",WaitTime_BluetoothSignalRestore);
        NN_LOG("==================================================\n");
        nnt::mifare::Sleep(WaitTime_BluetoothSignalRestore);

        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE ATTACH a TAG!!\n");
        NN_LOG("==================================================\n");
        nnt::mifare::DoActivate();
        NN_LOG(" TAG Attached !!\n");
        EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState());

        return;
    }

    void TestFinalizeRunMifareApiThread()
    {
        if(g_RunApiThread._state == nn::os::ThreadType::State_Started)
        {
            nn::os::DestroyThread(&g_RunApiThread);
        }

        NNT_MIFARE_EXPECT_RESULT_SUCCESS( nnt::mifare::FinalizeSystem() );
        nn::os::FinalizeEvent(&g_BootUpEvent);
        nn::os::FinalizeEvent(&g_RunApiEvent);
    }

    void TestInitTransPreRead()
    {
        NN_LOG("TestInitTransPreRead\n");
        // TestAPI設定
        g_RunMifareApi = RunMifareApi_Read;
        // Active状態に遷移
        nnt::mifare::DoActivate();
        // TestAPI準備
        TestInitRunApiThreadWait();
    }

    void TestInitTransPreWrite()
    {
        NN_LOG("TestInitTransPreWrite\n");
        // TestAPI設定
        g_RunMifareApi = RunMifareApi_Write;
        // Active状態に遷移
        nnt::mifare::DoActivate();
        // TestAPI準備
        TestInitRunApiThreadWait();
    }

    bool WaitForActivate(int ms, nn::os::SystemEventType activateEvent) NN_NOEXCEPT
    {
        if((activateEvent._state != nn::os::SystemEventType::State_NotInitialized) &&
                (activateEvent._state != nn::os::SystemEventType::State_NotInitialized))
        {
            return nn::os::TimedWaitSystemEvent(&activateEvent, nn::TimeSpan::FromMilliSeconds(ms));
        }
        else
        {
            NN_LOG("WaitForActivate() ActiveEvent DeactivateEvent is NotInitialized \n");
            return false;
        }
    }

    bool WaitForDeactivate(int ms, nn::os::SystemEventType deactivateEvent) NN_NOEXCEPT
    {
        return nn::os::TimedWaitSystemEvent(&deactivateEvent, nn::TimeSpan::FromMilliSeconds(ms));
    }

    void DestroyDefaultActivateEvent(nn::os::SystemEventType activateEvent) NN_NOEXCEPT
    {
        nn::os::DestroySystemEvent(&activateEvent);
        return;
    }

    void DestroyDefaultDeactivateEvent(nn::os::SystemEventType deactivateEvent) NN_NOEXCEPT
    {
        nn::os::DestroySystemEvent(&deactivateEvent);
        return;
    }

    void DeviceTestThread(void* pArg) NN_NOEXCEPT
    {
        threadinfo* threaditem = static_cast<threadinfo*>(pArg);
        int deviceNum = threaditem->deviceNum;
        for(int i = 0; i < TestLoop; i++)
        {
            nn::os::SystemEventType activateEvent[DeviceCountMax] = {};
            nn::os::SystemEventType deactivateEvent[DeviceCountMax] = {};
            NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachActivateEvent(
                    &activateEvent[deviceNum], g_CurrentDeviceHandleMulti[deviceNum]));
            NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachDeactivateEvent(
                    &deactivateEvent[deviceNum], g_CurrentDeviceHandleMulti[deviceNum]));

            NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StartDetection(g_CurrentDeviceHandleMulti[deviceNum]));
            if (!WaitForActivate(WaitTimeout, activateEvent[deviceNum]))
            {
                NN_LOG("StartDetection err deviceNum=%d Loop=%d\n", deviceNum, i);
                EXPECT_TRUE(false);
                break;
            }

            EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Active,nnt::mifare::wrapper::GetDeviceState(g_CurrentDeviceHandleMulti[deviceNum]));

            // 1ブロックにランダムのデータを書き込む
            // データを書き込む
            const int OneBlockCount = 1;
            // 読み込みデータ
            NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockData) readBlockData[OneBlockCount] = {};
            // 読み込みブロックパラメータ
            NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockParameter) readBlockParameter[OneBlockCount] = {};
            // 書き込みブロックパラメータ
            NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) writeBlockParameter[OneBlockCount] = {};

            nnt::mifare::PrepareWriteParam(writeBlockParameter, OneBlockCount);
            nnt::mifare::PrintWriteBlockParameter(writeBlockParameter, OneBlockCount);
            NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::Write(
                    g_CurrentDeviceHandleMulti[deviceNum], writeBlockParameter, OneBlockCount));
            nnt::mifare::PrepareReadParam(readBlockParameter, OneBlockCount);
            NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::Read(
                    readBlockData, g_CurrentDeviceHandleMulti[deviceNum], readBlockParameter, OneBlockCount));
            nnt::mifare::PrintReadBlockData(readBlockData, OneBlockCount);
            // 書き込んだデータを読み込めたか確認
            nnt::mifare::CheckReadData(readBlockData, writeBlockParameter, OneBlockCount);

            NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StopDetection(g_CurrentDeviceHandleMulti[deviceNum]));
            if (!WaitForDeactivate(WaitTimeout, deactivateEvent[deviceNum]))
            {
                NN_LOG("StopDetection err deviceNum=%d Loop=%d\n", deviceNum, i);
                EXPECT_TRUE(false);
                break;
            }
            DestroyDefaultActivateEvent(activateEvent[deviceNum]);
            DestroyDefaultDeactivateEvent(deactivateEvent[deviceNum]);
        }
    }

} // end of anonymous namespace

//================================================================================
// 全プラットフォームで共通のテストスイートです。
// 必ずタグを設置してからテストを開始してください。
//================================================================================

class MifareManual : public nnt::mifare::TestFramework
{
protected:

    MifareManual() NN_NOEXCEPT
    {
        // oeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::mifare::InitializeHidController();
        // API呼び出しログ出力フラグ
        //nnt::mifare::wrapper::SetApiCallLoggingMode(nnt::mifare::wrapper::LogMode_Aging);
    }

    ~MifareManual() NN_NOEXCEPT
    {
    }

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 次のテストに影響が出ないようにライブラリを一旦終了しておきます。
        if(nnt::mifare::wrapper::GetState() == NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init)
        {
            nnt::mifare::FinalizeSystem();
        }
    }
};

//================================================================================
// テストケースの実装です。
//================================================================================
TEST_F(MifareManual, TestCaseGoodOperationDeviceDeactive)
{
    // デバイス状態がDeactive時に呼び出すことができるAPIのテストをします。
    NN_LOG("TestCaseGoodOperationDeviceDeactive start\n");

    // 状態をDeviceDeactiveに遷移させる
    NN_LOG("========================================================================\n");
    NN_LOG(" PLEASE ATTACH MIFARE TAG (Within %dms) ...    \n",
           WaitTime_TagSwitch);
    NN_LOG("========================================================================\n");
    nnt::mifare::Sleep(WaitTime_TagSwitch);
    nnt::mifare::DoActivate();
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE REMOVE a TAG!!\n");
    NN_LOG("==================================================\n");
    nnt::mifare::WaitForDeactivate();

    // ----------------------------------------
    // テスト対象 : nn::nfc::mifare::StartDetection()
    //              nn::nfc::mifare::GetDeviceState()
    //              nn::nfc::mifare::GetNpadId()
    //              nn::nfc::mifare::ListDevices()
    //              nn::nfc::mifare::AttachActivateEvent()
    //              nn::nfc::mifare::AttachDeactivateEvent()
    //              nn::nfc::mifare::StopDetection()
    // ----------------------------------------
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState());
    nn::hid::NpadIdType npadId;
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&npadId));
    nnt::mifare::CheckNPadId(npadId);
    int outCount;
    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle currentDeviceHandle;
    NNT_EXPECT_RESULT_SUCCESS(
            nnt::mifare::wrapper::ListDevices(&currentDeviceHandle, &outCount, 1));
    nn::os::SystemEventType activateEvent;
    nn::os::SystemEventType deactivateEvent;
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachDeactivateEvent(&deactivateEvent));
    nn::os::DestroySystemEvent(&activateEvent);
    nn::os::DestroySystemEvent(&deactivateEvent);
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StartDetection());

    // 状態をDeviceDeactiveに遷移させる
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE ATTACH a TAG!!\n");
    NN_LOG("==================================================\n");
    nnt::mifare::WaitForActivate();
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState());
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE REMOVE a TAG!!\n");
    NN_LOG("==================================================\n");
    nnt::mifare::WaitForDeactivate();

    // ----------------------------------------
    // テスト対象 : nn::nfc::mifare::StopDetection()
    // ----------------------------------------
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StopDetection());
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState());
}
TEST_F(MifareManual, TestCaseInvalidOperationDeviceDeactive)
{
    // デバイス状態がDeactive時に呼び出すことができないAPIのテストをします。
    NN_LOG("TestCaseInvalidOperationDeviceDeactive start\n");

    // 状態をDeviceDeactiveに遷移させる
    NN_LOG("========================================================================\n");
    NN_LOG(" PLEASE ATTACH MIFARE TAG (Within %dms) ...    \n",
           WaitTime_TagSwitch);
    NN_LOG("========================================================================\n");
    nnt::mifare::Sleep(WaitTime_TagSwitch);
    nnt::mifare::DoActivate();
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE REMOVE a TAG!!\n");
    NN_LOG("==================================================\n");
    nnt::mifare::WaitForDeactivate();
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState());

    // ----------------------------------------
    // テスト対象 : nn::nfc::mifare::Read()
    //              nn::nfc::mifare::Write()
    //              nn::nfc::mifare::GetTagInfo()
    // ----------------------------------------
    const int blockCount = 2;
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockData) readBlockData[blockCount] = {};
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockParameter) readBlockParameter[blockCount] = {};
    NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) writeBlockParameter[blockCount] = {};
    nnt::mifare::PrepareWriteParam(writeBlockParameter, blockCount);
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultTagNotFound,
            nnt::mifare::wrapper::Write(writeBlockParameter, blockCount));
    nnt::mifare::PrepareReadParam(readBlockParameter, blockCount);
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultTagNotFound,
            nnt::mifare::wrapper::Read(readBlockData, readBlockParameter, blockCount));
    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::TagInfo tagInfo;
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultTagNotFound,
            nnt::mifare::wrapper::GetTagInfo(&tagInfo));
}

TEST_F(MifareManual, TestCaseInvalidKeyTag)
{
    // 鍵の違うMifareタグの読み書きのテストをします。
    NN_LOG("TestCaseInvalidKeyTag start\n");

    // 鍵の違うMifareタグをアタッチする
    NN_LOG("========================================================================\n");
    NN_LOG(" PLEASE ATTACH <<INVALID KEY>> MIFARE TAG (Within %dms) ...    \n",
           WaitTime_TagSwitch);
    NN_LOG("========================================================================\n");
    nnt::mifare::Sleep(WaitTime_TagSwitch);
    nnt::mifare::DoActivate();

    // 鍵の違うMifareタグを読み書きしてResultAccessErrorが返却されることを確認する
    const int blockCount = 2;
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockData) readBlockData[blockCount] = {};
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockParameter) readBlockParameter[blockCount] = {};
    NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) writeBlockParameter[blockCount] = {};
    nnt::mifare::PrepareWriteParam(writeBlockParameter, blockCount);
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultAccessError,
            nnt::mifare::wrapper::Write(writeBlockParameter, blockCount));
    nnt::mifare::PrepareReadParam(readBlockParameter, blockCount);
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultAccessError,
            nnt::mifare::wrapper::Read(readBlockData, readBlockParameter, blockCount));

    // 鍵の違うMifareタグをデタッチする
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE REMOVE a TAG!!\n");
    NN_LOG("==================================================\n");
    nnt::mifare::WaitForDeactivate();
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState());
}

TEST_F(MifareManual, TestCaseStateTransWhenReadWrite)
{
    // 書き込み／読み込み中にMifareタグを離すテストをします。
    NN_LOG("TestCaseInvalidKeyTag start\n");

    // スレッド準備
    struct TestCaseProc
    {
        const char* pStateName;
        const char* pEventName;
        void  (*pFuncTestInitial)();
        void  (*pFuncTestMain)();
        void  (*pFuncTestFinal)();
    };

    enum StateKind
    {
        StateKind_Active  = 0,
        StateKind_Max
    };

    enum EventKind
    {
        EventKind_TagDropInProgressRead  = 0,
        EventKind_TagDropInProgressWrite,
        EventKind_BluetoothCutInProgressRead,
        EventKind_BluetoothCutInProgressWrite,
        EventKind_Max
    };

    static const TestCaseProc testCaseTbl[StateKind_Max][EventKind_Max] =
    {
        /************************************************** ACTIVE ***********************************************************/
        {
            // StateName EventName   Before Test            Main                            After
            {  "ACTIVE", "Tag Drop", TestInitTransPreRead,  TestMainTagDropInProgress,      TestFinalizeRunMifareApiThread },
            {  "ACTIVE", "Tag Drop", TestInitTransPreWrite, TestMainTagDropInProgress,      TestFinalizeRunMifareApiThread },
            {  "ACTIVE", "BT Off",   TestInitTransPreRead,  TestMainBluetoothCutInProgress, TestFinalizeRunMifareApiThread },
            {  "ACTIVE", "BT Off",   TestInitTransPreWrite, TestMainBluetoothCutInProgress, TestFinalizeRunMifareApiThread },
        }
    };

    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE ATTACH a TAG (Within %dms) !! \n",WaitTime_AttachTag);
    NN_LOG("==================================================\n");

    nnt::mifare::Sleep(WaitTime_AttachTag);

    for(uint8_t stateCount = 0; stateCount < StateKind_Max; stateCount++)
    {
        for(uint8_t eventCount = 0; eventCount < EventKind_Max; eventCount++)
        {
            const TestCaseProc* pTestCaseTbl =
                    &testCaseTbl[stateCount][eventCount];

            if( pTestCaseTbl->pFuncTestInitial == nullptr )
            {
                continue;
            }

            NN_LOG("**************************************************\n");
            NN_LOG("*   TestState : %s \n", pTestCaseTbl->pStateName );
            NN_LOG("*   TestEvent : %s \n", pTestCaseTbl->pEventName );
            NN_LOG("**************************************************\n");

            // 事前処理
            pTestCaseTbl->pFuncTestInitial();
            nnt::mifare::Sleep(WaitTime_NextStep);

            // テストケース
            pTestCaseTbl->pFuncTestMain();
            nnt::mifare::Sleep(WaitTime_NextStep);

            // あとしまつ
            pTestCaseTbl->pFuncTestFinal();
            nnt::mifare::Sleep(WaitTime_NextStep);
        }
    }
}

TEST_F(MifareManual, TestCaseMultiDeviceAccess)
{
    NN_LOG("========================================================================\n");
    NN_LOG(" Please Pairing Multi Devices and Setting Mifare Tags on All Device \n");
    NN_LOG("========================================================================\n");
    nnt::mifare::Sleep(WaitTimeout);

    // Mifare初期化
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::InitializeSystem());

    // 4つのデバイスを取得
    int outCount;
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(
            g_CurrentDeviceHandleMulti, &outCount, DeviceCountMax));
    EXPECT_TRUE( outCount == DeviceCountMax);
    if(outCount != DeviceCountMax)
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" Please Check to Connect Devices\n");
        NN_LOG("========================================================================\n");
        // 4台のデバイスが取得できなかったのでアサートさせる
        ASSERT_TRUE(false);
    }

    for(int i = 0; i < outCount; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(
                &g_DeviceThread[i], DeviceTestThread, &g_threadParam[i], g_DeviceThreadStack[i],
                sizeof(g_DeviceThreadStack[i]), nn::os::DefaultThreadPriority));
        // コアを指定する
        // コアは0～2までの3つが指定できるので4つ目のスレッドには0を指定する
        int idealCore = i;
        if (idealCore >= MaxCore)
        {
            idealCore = 0;
        }
        nn::os::SetThreadCoreMask(&g_DeviceThread[i], idealCore, 0x1LL << idealCore);
    }

    for(int i = 0; i < outCount; i++)
    {
        nn::os::StartThread(&g_DeviceThread[i]);
    }

    for(int i = 0; i < outCount; i++)
    {
        nn::os::WaitThread(&g_DeviceThread[i]);
        nn::os::DestroyThread(&g_DeviceThread[i]);
    }
}

TEST_F(MifareManual,TestCaseErrorVisible)
{
    NN_LOG("========================================================================\n");
    NN_LOG(" Please ready for NFC device with invalid FW\n");
    NN_LOG("========================================================================\n");
    nnt::mifare::wrapper::Initialize();
    EXPECT_EQ(true, nnt::mifare::wrapper::IsErrorVisible());
    nnt::mifare::wrapper::SetErrorVisible(false);
    EXPECT_EQ(false, nnt::mifare::wrapper::IsErrorVisible());
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::SearchDeviceHandle());
    NN_LOG("========================================================================\n");
    NN_LOG(" THIS TEST IS EXPECTED NOT TO DISPLAY ERROR\n");
    NN_LOG("========================================================================\n");
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultNfcDeviceNotFound, nnt::mifare::wrapper::StartDetection());
    nnt::mifare::wrapper::SetErrorVisible(true);
    EXPECT_EQ(true, nnt::mifare::wrapper::IsErrorVisible());
    int waitTime = 60;
    NN_LOG("========================================================================\n");
    NN_LOG(" Wait %dsec\n", waitTime);
    NN_LOG("========================================================================\n");
    for(int i = 0; i < waitTime; ++i)
    {
        nnt::mifare::Sleep(1000);
        NN_LOG(" %d ", waitTime - 1 - i);
        if((waitTime - 1 - i) % 10 == 0)
        {
            NN_LOG("\n");
        }
    }
    NN_LOG("========================================================================\n");
    NN_LOG(" THIS TEST IS EXPECTED TO DISPLAY ERROR\n");
    NN_LOG("========================================================================\n");
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultNfcDeviceNotFound, nnt::mifare::wrapper::StartDetection());
    nnt::mifare::wrapper::Finalize();
}
