﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_SdkText.h>
#include <nn/hid.h>
#include <nn/mii.h>
#include <nn/nfc.h>
#include <nn/nfc/nfc_PrivateApi.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/mii/mii_StoreDataAccessor.h>

#include <nnt/result/testResult_Assert.h>
#include <nnt/nfp/testMifare_Common.h>


namespace nnt { namespace mifare {

    //======================================================
    // テストで使用する定数です。
    //======================================================

    //======================================================
    // ユーティリティ関数で使用する使用する変数です。
    //======================================================

    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle      g_CurrentDeviceHandle;

    nn::os::SystemEventType            g_ActivateEvent = {};
    nn::os::SystemEventType            g_DeactivateEvent = {};

    nn::TimeSpan g_Span;

    //======================================================
    // テストで使用するデータです。
    //======================================================

    // Mifare Standard 1k のデータブロック数は
    // 16byte のブロックが 64 ブロック
    // そのうちデータブロックで使用できないブロックを引いた数
    // 使用できないのは 0 と 4の倍数+3 のブロック
    // Mifare Standard 1k では 47ブロック を使用できる
    // ここでは、試験用に 95ブロック のアドレスを作っておく
    const int DataBlockCount = 95;
    uint8_t g_DataBlockAddr[DataBlockCount] = {};

    //======================================================
    // コントローラ初期化に関連する変数です。
    //======================================================
    const nn::hid::NpadIdType NpadIds[] = {nn::hid::NpadId::No1,
                                       nn::hid::NpadId::No2,
                                       nn::hid::NpadId::No3,
                                       nn::hid::NpadId::No4,
                                       nn::hid::NpadId::Handheld,
                                       };
    const int NpadIdCountMax = sizeof(NpadIds) / sizeof(nn::hid::NpadIdType);

    //======================================================
    // プラットフォーム毎に異なる処理をラップする関数です。
    //======================================================

    nn::Result InitializeSystem() NN_NOEXCEPT
    {

        if( nnt::mifare::wrapper::GetState() == NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_None )
        {
#if defined(NNT_NFP_BUILD_CONFIG_OLD_LIBRARY) || !defined(NNT_MIFARE_PLATFORM_NX)
            return nnt::mifare::wrapper::Initialize();
#else
            return nnt::mifare::wrapper::InitializeDebug();
#endif
        }
        else
        {
            NN_LOG("* Duplicate function call : InitializeDebug() \n");
            return nn::ResultSuccess();
        }
    }

    nn::Result FinalizeSystem() NN_NOEXCEPT
    {
        if( nnt::mifare::wrapper::GetState() != NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_None )
        {
#if defined(NNT_NFP_BUILD_CONFIG_OLD_LIBRARY) || !defined(NNT_MIFARE_PLATFORM_NX)
            nnt::mifare::wrapper::Finalize();
#else
            nnt::mifare::wrapper::FinalizeDebug();
#endif
        }
        else
        {
            NN_LOG("* Duplicate function call : FinalizeDebug()");
        }
        return nn::ResultSuccess();
    }

    nn::Result SetDefaultActivateEvent() NN_NOEXCEPT
    {
        if(g_ActivateEvent._state != nn::os::SystemEventType::State_NotInitialized)
        {
            DestroyDefaultActivateEvent();
        }
        nn::Result result;
        result = nnt::mifare::wrapper::AttachActivateEvent(&g_ActivateEvent);
        if(!result.IsSuccess())
        {
            std::memset(&g_ActivateEvent, 0x00, sizeof(g_ActivateEvent));
        }
        return result;
    }

    nn::Result SetDefaultDeactivateEvent() NN_NOEXCEPT
    {
        if(g_DeactivateEvent._state != nn::os::SystemEventType::State_NotInitialized)
        {
            DestroyDefaultDeactivateEvent();
        }
        nn::Result result;
        result = nnt::mifare::wrapper::AttachDeactivateEvent(&g_DeactivateEvent);
        if(!result.IsSuccess())
        {
            std::memset(&g_DeactivateEvent, 0x00, sizeof(g_DeactivateEvent));
        }
        return result;
    }

    void DestroyDefaultActivateEvent() NN_NOEXCEPT
    {
        nn::os::DestroySystemEvent(&g_ActivateEvent);
        return;
    }

    void DestroyDefaultDeactivateEvent() NN_NOEXCEPT
    {
        nn::os::DestroySystemEvent(&g_DeactivateEvent);
        return;
    }

    bool WaitForActivate(int ms) NN_NOEXCEPT
    {
        if((g_ActivateEvent._state != nn::os::SystemEventType::State_NotInitialized) &&
                (g_DeactivateEvent._state != nn::os::SystemEventType::State_NotInitialized))
        {
            return nn::os::TimedWaitSystemEvent(&g_ActivateEvent, nn::TimeSpan::FromMilliSeconds(ms));
        }
        else
        {
            NN_LOG("WaitForActivate() ActiveEvent DeactivateEvent is NotInitialized \n");
            return false;
        }
    }

    bool WaitForActivate() NN_NOEXCEPT
    {
        return WaitForActivate(ActivateTimeout);
    }

    bool WaitForDeactivate(int ms) NN_NOEXCEPT
    {
        return nn::os::TimedWaitSystemEvent(&g_DeactivateEvent, nn::TimeSpan::FromMilliSeconds(ms));
    }

    bool WaitForDeactivate() NN_NOEXCEPT
    {
        return WaitForDeactivate(DeactivateTimeout);
    }

    //==================================================
    // MIFARE ライブラリの API をラップしてリトライに対応した関数です。
    //==================================================

    nn::Result InitializeWithRetry() NN_NOEXCEPT
    {
        nn::Result result;

        if( nnt::mifare::wrapper::GetState() == NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init )
        {
            return nn::ResultSuccess();
        }

        result = nnt::mifare::wrapper::Initialize();

        return result;
    }

    nn::Result InitializeSystemWithRetry() NN_NOEXCEPT
    {
        nn::Result result;

        if( nnt::mifare::wrapper::GetState() == NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init )
        {
            return nn::ResultSuccess();
        }

        result = InitializeSystem();

        return result;
    }

    //==================================================
    // 頻繁に利用されるシーケンスをまとめた関数です。
    //==================================================

    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle GetCurrentDeviceHandle() NN_NOEXCEPT
    {
        return g_CurrentDeviceHandle;
    }

    void SetCurrentDeviceHandle(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle deviceHandle) NN_NOEXCEPT
    {
        g_CurrentDeviceHandle = deviceHandle;
    }

    nn::Result SearchDeviceHandle() NN_NOEXCEPT
    {
        int outCount;
        NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State state = nnt::mifare::wrapper::GetState();

        //デバイスリストが取得できない状態でこの関数が実行された場合は、テスト手順が不正
        EXPECT_NE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_None,state);

        return nnt::mifare::wrapper::ListDevices(&g_CurrentDeviceHandle, &outCount, 1);
    }

    void DoSearchDeviceHandle() NN_NOEXCEPT
    {
        NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State state;
        state = nnt::mifare::wrapper::GetState();
        if (state == NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_None)
        {
            NNT_EXPECT_RESULT_SUCCESS(InitializeSystemWithRetry());
        }

        //NXではConnectの代わりにデバイスハンドルの取得を行う
        NNT_EXPECT_RESULT_SUCCESS(SearchDeviceHandle());

        //イベント取得準備
        NNT_EXPECT_RESULT_SUCCESS(SetDefaultActivateEvent());
        NNT_EXPECT_RESULT_SUCCESS(SetDefaultDeactivateEvent());

        return;
    }

    void DoSearch() NN_NOEXCEPT
    {
        DoSearchDeviceHandle();
        NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState state = nnt::mifare::wrapper::GetDeviceState();

        NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State libState = nnt::mifare::wrapper::GetState();
        if ((libState == NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init)
            && (state == NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Init || state == NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Deactive))
        {
            NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StartDetection());
        }
    }

    void DoActivate() NN_NOEXCEPT
    {
        DoSearch();
        NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState state = nnt::mifare::wrapper::GetDeviceState();
        NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State libState = nnt::mifare::wrapper::GetState();
        if ((libState == NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init)
            && (state == NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Search))
        {
            EXPECT_TRUE(WaitForActivate());
            EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Active,nnt::mifare::wrapper::GetDeviceState());
        }
    }

    //==================================================
    // その他のユーティリティです。
    //==================================================

    void Sleep(int ms) NN_NOEXCEPT
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ms));
    }

    void CreateRandomNumberSequence(nn::Bit16* pOut, int size) NN_NOEXCEPT
    {
        std::srand(nn::os::GetSystemTick().GetInt64Value() & 0xFFFFFFFF);
        for(int i = 0; i < size; i++)
        {
            pOut[i] = std::rand() & 0xFFFF;
        }
    }

    void CreateRandomNumberSequence(char* pOut, int size) NN_NOEXCEPT
    {
        std::srand(nn::os::GetSystemTick().GetInt64Value() & 0xFFFFFFFF);
        for(int i = 0; i < size; i++)
        {
            pOut[i] = std::rand() & 0xFF;
        }
    }

    void NfcOn() NN_NOEXCEPT
    {
#if defined(NNT_NFP_BUILD_CONFIG_OLD_LIBRARY) || !defined(NNT_MIFARE_PLATFORM_NX)
        nn::nfc::InitializeSystem();
        nn::nfc::SetNfcEnabled(true);
        nn::nfc::FinalizeSystem();
#else
        if( nnt::mifare::wrapper::GetState() == nn::nfc::State_None )
        {
            nn::nfc::InitializeSystem();
            nn::nfc::SetNfcEnabled(true);
            nn::nfc::FinalizeSystem();
        }
        else
        {
            nn::nfc::SetNfcEnabled(true);
        }
#endif
    }

    void NfcOff() NN_NOEXCEPT
    {
#if defined(NNT_NFP_BUILD_CONFIG_OLD_LIBRARY) || !defined(NNT_MIFARE_PLATFORM_NX)
        nn::nfc::InitializeSystem();
        nn::nfc::SetNfcEnabled(false);
        nn::nfc::FinalizeSystem();
#else
        if( nnt::mifare::wrapper::GetState() == nn::nfc::State_None )
        {
            nn::nfc::InitializeSystem();
            nn::nfc::SetNfcEnabled(false);
            nn::nfc::FinalizeSystem();
        }
        else
        {
            nn::nfc::SetNfcEnabled(false);
        }
#endif
    }

    const char* GetMifareResultTypeString(nn::Result result) NN_NOEXCEPT
    {
        const char* pString = nullptr;
        #define MIFARE_GET_RESULT_STRING(resultType,result,pOut)                               \
            else if ((NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::resultType().GetModule() == (result).GetModule()) &&         \
                     (NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::resultType().GetDescription() == (result).GetDescription())) \
            { \
                pOut = #resultType;\
            }

        if (result.IsSuccess())
        {
            pString = "ResultSuccess";
        }
        //private
#if defined(NNT_NFP_BUILD_CONFIG_OLD_LIBRARY) || !defined(NNT_MIFARE_PLATFORM_NX)
        MIFARE_GET_RESULT_STRING(ResultBackupError,result,pString)
        MIFARE_GET_RESULT_STRING(ResultBackupReadFailed,result,pString)
        MIFARE_GET_RESULT_STRING(ResultBackupWriteFailed,result,pString)
        MIFARE_GET_RESULT_STRING(ResultBackupCrcBroken,result,pString)
#endif
        MIFARE_GET_RESULT_STRING(ResultBadRequest,result,pString)
        MIFARE_GET_RESULT_STRING(ResultInvalidDeviceState,result,pString)
#if defined(NNT_NFP_BUILD_CONFIG_OLD_LIBRARY) || !defined(NNT_MIFARE_PLATFORM_NX)
        MIFARE_GET_RESULT_STRING(ResultTimeOutError,result,pString)
        MIFARE_GET_RESULT_STRING(ResultOperationFailed,result,pString)
#endif
        MIFARE_GET_RESULT_STRING(ResultTagNotFound,result,pString)
        MIFARE_GET_RESULT_STRING(ResultUidMisMatch,result,pString)
        MIFARE_GET_RESULT_STRING(ResultSleep,result,pString)
        MIFARE_GET_RESULT_STRING(ResultNotForeground,result,pString)
        MIFARE_GET_RESULT_STRING(ResultNfcDeviceError,result,pString)
        MIFARE_GET_RESULT_STRING(ResultBackupSystemError,result,pString)
        MIFARE_GET_RESULT_STRING(ResultNeedUpdate,result,pString)
        MIFARE_GET_RESULT_STRING(ResultNeedCharge,result,pString)
#if defined(NNT_NFP_BUILD_CONFIG_OLD_LIBRARY) || !defined(NNT_MIFARE_PLATFORM_NX)
        MIFARE_GET_RESULT_STRING(ResultInvalidFormat,result,pString)
#endif
        MIFARE_GET_RESULT_STRING(ResultInvalidTag,result,pString)
        MIFARE_GET_RESULT_STRING(ResultAuthenticationError,result,pString)
#if defined(NNT_NFP_BUILD_CONFIG_OLD_LIBRARY) || !defined(NNT_MIFARE_PLATFORM_NX)
        MIFARE_GET_RESULT_STRING(ResultInvalidRomArea,result,pString)
#endif
        MIFARE_GET_RESULT_STRING(ResultAccessTimeOutError,result,pString)
        MIFARE_GET_RESULT_STRING(ResultAccessOperationFailed,result,pString)
        //public
        MIFARE_GET_RESULT_STRING(ResultNfcDeviceNotFound,result,pString)
        MIFARE_GET_RESULT_STRING(ResultNfcDisabled,result,pString)
#if defined(NNT_NFP_BUILD_CONFIG_OLD_LIBRARY) || !defined(NNT_MIFARE_PLATFORM_NX)
        MIFARE_GET_RESULT_STRING(ResultNeedRetry,result,pString)
#endif
        MIFARE_GET_RESULT_STRING(ResultNeedRestart,result,pString)
        MIFARE_GET_RESULT_STRING(ResultNotSupported,result,pString)
        MIFARE_GET_RESULT_STRING(ResultMaxNfcDeviceActivated,result,pString)
        MIFARE_GET_RESULT_STRING(ResultConflictFunction,result,pString)
        MIFARE_GET_RESULT_STRING(ResultAccessError,result,pString)
        else
        {
            //不明な場合はエラーコードを表示する
            static char unknownErrorStringBuffer[64];
            std::memset(unknownErrorStringBuffer,0x00,sizeof(unknownErrorStringBuffer));
            std::sprintf(unknownErrorStringBuffer,
                         "Unknown(Module=%d,Description=%d)",result.GetModule(),
                                                             result.GetDescription());
            pString = unknownErrorStringBuffer;
        }
        #undef MIFARE_GET_RESULT_STRING

        return pString;
    }

    bool CheckNpadStyle(nn::hid::NpadIdType npadId) NN_NOEXCEPT
    {
        bool ret = false;

        // 取得したNpad IDを指定してGetNpadStyleSetを実行する
        nn::hid::NpadStyleSet style;
        style = nn::hid::GetNpadStyleSet(npadId);
        if ((style.Test<nn::hid::NpadStyleFullKey>() == true) |
                (style.Test<nn::hid::NpadStyleHandheld>() == true) |
                (style.Test<nn::hid::NpadStyleJoyDual>() == true) |
                (style.Test<nn::hid::NpadStyleJoyLeft>() == true) |
                (style.Test<nn::hid::NpadStyleJoyRight>() == true) |
                (style.Test<nn::hid::system::NpadStyleSystem>() == true) |
                (style.Test<nn::hid::system::NpadStyleSystemExt>() == true))
        {
            ret = true;
        }
        return ret;
    }

    bool IsNfcEnable() NN_NOEXCEPT
    {
        bool ret;

#if defined(NNT_NFP_BUILD_CONFIG_OLD_LIBRARY) || !defined(NNT_MIFARE_PLATFORM_NX)
        nn::nfc::Initialize();
        ret = nn::nfc::IsNfcEnabled();
        nn::nfc::Finalize();
#else
        if( nnt::mifare::wrapper::GetState() == nn::nfc::State_None )
        {
            nn::nfc::Initialize();
            ret = nn::nfc::IsNfcEnabled();
            nn::nfc::Finalize();
        }
        else
        {
            ret = nn::nfc::IsNfcEnabled();
        }
#endif

        return ret;
    }

    void InitializeHidController() NN_NOEXCEPT
    {
        nn::hid::InitializeNpad();

        //使用する操作形態を設定
        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask |
                nn::hid::NpadStyleJoyDual::Mask | nn::hid::NpadStyleHandheld::Mask);

        // 使用する Npad を設定
        nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);
    }

    void CheckNPadId(nn::hid::NpadIdType npadId) NN_NOEXCEPT
    {
        // 取得したNpad IDを指定してGetNpadStyleSetを実行する
        nn::hid::NpadStyleSet style;
        style = nn::hid::GetNpadStyleSet(npadId);
        EXPECT_TRUE((style.Test<nn::hid::NpadStyleFullKey>() == true) |
                (style.Test<nn::hid::NpadStyleHandheld>() == true) |
                (style.Test<nn::hid::NpadStyleJoyDual>() == true) |
                (style.Test<nn::hid::NpadStyleJoyLeft>() == true) |
                (style.Test<nn::hid::NpadStyleJoyRight>() == true) |
                (style.Test<nn::hid::system::NpadStyleSystem>() == true) |
                (style.Test<nn::hid::system::NpadStyleSystemExt>() == true));
    }

    void PrintWriteBlockParameter(NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) * writeBlockParameter,
            int blockCount) NN_NOEXCEPT
    {
        NN_LOG("-WriteData-\n");
        for(int i = 0; i < blockCount; i++)
        {
            NN_LOG("--BlockAddr %d ---------------------\n", writeBlockParameter[i].blockAddr);
            for(int j = 0; j < NNT_NFC_COMMON_IF_TYPE_MIFARE(BlockDataSize); j++)
            {
                NN_LOG("0x%02x ", writeBlockParameter[i].data[j]);
            }
            NN_LOG("\n-----------------------------------\n");
        }
        return;
    }

    void PrepareWriteParam(NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) * writeBlockParameter,
            int blockCount) NN_NOEXCEPT
    {
        CreateBlockAddress(g_DataBlockAddr);
        for(int i = 0; i < blockCount; i++)
        {
            // ブロックアドレス設定
            writeBlockParameter[i].blockAddr = g_DataBlockAddr[i];
            // キータイプ設定
            writeBlockParameter[i].key.type = NNT_NFC_COMMON_IF_TYPE_MIFARE(KeyType_A);
            // キーフォーマット設定
            writeBlockParameter[i].key.valueFormat = NNT_NFC_COMMON_IF_TYPE_MIFARE(KeyValueFormat_Raw);
            // キー設定
            std::memset(writeBlockParameter[i].key.raw.value, 0xff,
                    sizeof(writeBlockParameter[i].key.raw.value));
            // ランダムのデータ設定
            char writeData[NNT_NFC_COMMON_IF_TYPE_MIFARE(BlockDataSize)] = {};
            CreateRandomNumberSequence(writeData, sizeof(writeData));
            std::memcpy(writeBlockParameter[i].data, writeData,
                    sizeof(writeBlockParameter[i].data));
        }
    }

    void PrintReadBlockData(NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockData) * readBlockData,
            int blockCount) NN_NOEXCEPT
    {
        NN_LOG("-ReadData-\n");
        for(int i = 0; i < blockCount; i++)
        {
            NN_LOG("--BlockAddr %d ---------------------\n", readBlockData[i].blockAddr);
            for(int j = 0; j < NNT_NFC_COMMON_IF_TYPE_MIFARE(BlockDataSize); j++)
            {
                NN_LOG("0x%02x ", readBlockData[i].buffer[j]);
            }
            NN_LOG("\n-----------------------------------\n");
        }
        return;
    }

    void PrepareReadParam(NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockParameter) * readBlockParameter,
            int blockCount) NN_NOEXCEPT
    {
        CreateBlockAddress(g_DataBlockAddr);
        for(int i = 0; i < blockCount; i++)
        {
            // ブロックアドレス設定
            readBlockParameter[i].blockAddr = g_DataBlockAddr[i];
            // キータイプ設定
            readBlockParameter[i].key.type = NNT_NFC_COMMON_IF_TYPE_MIFARE(KeyType_A);
            // キーフォーマット設定
            readBlockParameter[i].key.valueFormat = NNT_NFC_COMMON_IF_TYPE_MIFARE(KeyValueFormat_Raw);
            // キー設定
            std::memset(readBlockParameter[i].key.raw.value, 0xff,
                    sizeof(readBlockParameter[i].key.raw.value));
        }
    }

    void CheckReadData(NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockData) * readBlockData,
            NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) * writeBlockParameter,
            int blockCount) NN_NOEXCEPT
    {
        for(int i = 0; i < blockCount; i++)
        {
            EXPECT_TRUE(std::memcmp(readBlockData[i].buffer,
                    writeBlockParameter[i].data, sizeof(writeBlockParameter[i].data)) == 0);
        }
    }

    void CreateBlockAddress(uint8_t* pBlockAddr)
    {
        // エラーを考慮していないので、pBlockAddrは95バイトのメモリを確保しておくこと

        // 書き込み不可能ブロックアドレス
        const int ManufacturerBlockAddress = 0;
        // 書き込みしてはいけない各セクタの終端ブロックアドレス
        int sectorTrailerAjust = 0;
        #define SECTOR_TRAILER_ADDRESSS (sectorTrailerAjust * 4 + 3)
        // MIFARE Standard 1k のブロックアドレス数は64だが
        // 試験用に倍のアドレスを作っておく
        const int BlockCount = 128;
        for (int address = 0; address < BlockCount; address++)
        {
            if(address == ManufacturerBlockAddress)
            {
                // 一番初めのアドレスは書き込み禁止
                continue;
            }
            else if (address == SECTOR_TRAILER_ADDRESSS)
            {
                // 4の倍数+3のアドレスは書き込み禁止
                sectorTrailerAjust++;
                continue;
            }
            std::memcpy(pBlockAddr++, &address, sizeof(pBlockAddr));
        }
    }
}} // end of namespace nnt::mifare
