﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nnt/nfp/testMifare_Common.h>

namespace {

    uint32_t g_LoggingMode = 0;

    const char* RemoveNameSpaceString(const char* pApiName) NN_NOEXCEPT
    {
        const char* NameSpaceString = "nnt::mifare::wrapper::";
        if((std::strlen(pApiName) > std::strlen(NameSpaceString)) &&
           (std::memcmp(pApiName,NameSpaceString,std::strlen(NameSpaceString)) == 0))
        {
            // namespace名分の文字列をスキップ
            return pApiName + std::strlen(NameSpaceString);
        }
        return pApiName;
    }


    void PrintApiName(const char* pApiName) NN_NOEXCEPT
    {
        if(g_LoggingMode & nnt::mifare::wrapper::LogMode_Aging)
        {
            NN_LOG("[MIFARE_API_CALL]nn::nfc::mifare::%s() ,",RemoveNameSpaceString(pApiName));
        }
    }

    void PrintApiCallLog(const char* pApiName, nn::Result result, uint64_t count) NN_NOEXCEPT
    {
        NN_UNUSED(pApiName);
        if(g_LoggingMode & nnt::mifare::wrapper::LogMode_Aging)
        {
            NN_LOG("Description = %s ,call count = %lld\n",
                   nnt::mifare::GetMifareResultTypeString(result),++count);
        }
    }

    void PrintApiCallLog(const char* pApiName,
            NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State state, uint64_t count) NN_NOEXCEPT
    {
        NN_UNUSED(pApiName);
        if(g_LoggingMode & nnt::mifare::wrapper::LogMode_Aging)
        {
            const char* pString;
            switch(state)
            {
            case NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init:
                {
                    pString = "State_Init";
                }
                break;
            case NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_None:
                {
                    pString = "State_None";
                }
                break;
#if defined(NNT_NFP_BUILD_CONFIG_OLD_LIBRARY) || !defined(NNT_MIFARE_PLATFORM_NX)
            case NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Unexpected:
                {
                    pString = "State_Unexpected";
                }
                break;
#endif
            default:
                NN_UNEXPECTED_DEFAULT;
            }
            NN_LOG("State = %s ,call count = %lld\n",
                   pString,++count);
        }
    }

    void PrintApiCallLog(const char* pApiName,
            NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState deviceState, uint64_t count) NN_NOEXCEPT
    {
        NN_UNUSED(pApiName);
        if(g_LoggingMode & nnt::mifare::wrapper::LogMode_Aging)
        {
            const char* pString;
            switch(deviceState)
            {
            case NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Init:
                {
                    pString = "DeviceState_Init";
                }
                break;
            case NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Search:
                {
                    pString = "DeviceState_Search";
                }
                break;
            case NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Active:
                {
                    pString = "DeviceState_Active";
                }
                break;
            case NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Deactive:
                {
                    pString = "DeviceState_Deactive";
                }
                break;
#if !defined(NNT_NFP_BUILD_CONFIG_OLD_LIBRARY) && defined(NNT_NFP_PLATFORM_NX)
            case NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Mount:
                {
                    pString = "DeviceState_Mount";
                }
                break;
            case NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Keep:
                {
                    pString = "DeviceState_Keep";
                }
                break;
#endif
            case NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Unexpected:
                {
                    pString = "DeviceState_Unexpected";
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
            NN_LOG("DeviceState = %s ,call count = %lld\n",
                   pString,++count);
        }
    }

    void PrintApiCallLog(const char* pApiName, bool result, uint64_t count) NN_NOEXCEPT
    {
        NN_UNUSED(pApiName);
        if(g_LoggingMode & nnt::mifare::wrapper::LogMode_Aging)
        {
            NN_LOG("bool = %s ,call count = %lld\n",
                   result ? "true" : "false",++count);
        }
    }

    void PrintApiCallLog(const char* pApiName, uint64_t count) NN_NOEXCEPT
    {
        NN_UNUSED(pApiName);
        if(g_LoggingMode & nnt::mifare::wrapper::LogMode_Aging)
        {
            NN_LOG("call count = %lld\n",++count);
        }
    }
} // namespace

namespace nnt { namespace mifare { namespace wrapper {
    //==================================================
    // API実行時のログ設定
    //==================================================

    void SetApiCallLoggingMode(const uint32_t mode) NN_NOEXCEPT
    {
        g_LoggingMode = mode;
    }

    //MifareライブラリAPI実行用ラッパー関数
    nn::Result Initialize() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::Initialize();
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        /* テストプログラムの互換性の為、nn::ResultSuccess()を返す */

        return nn::ResultSuccess();
    }

    void Finalize() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::Finalize();
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
    }

    nn::Result ListDevices(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle* pOutBuffer,
            int* pOutCount, int bufferCount) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ListDevices(pOutBuffer, pOutCount, bufferCount);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result StartDetection(const NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::StartDetection(deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result StartDetection() NN_NOEXCEPT
    {
        return nnt::mifare::wrapper::StartDetection(GetCurrentDeviceHandle());
    }

    nn::Result StopDetection(const NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::StopDetection(deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result StopDetection() NN_NOEXCEPT
    {
        return nnt::mifare::wrapper::StopDetection(GetCurrentDeviceHandle());
    }

    nn::Result GetTagInfo(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::TagInfo* pOutTagInfo,
            const NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::GetTagInfo(pOutTagInfo, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result GetTagInfo(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::TagInfo* pOutTagInfo) NN_NOEXCEPT
    {
        return nnt::mifare::wrapper::GetTagInfo(pOutTagInfo, GetCurrentDeviceHandle());
    }

    nn::Result AttachActivateEvent(nn::os::SystemEventType* pEvent,
            const NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::AttachActivateEvent(pEvent, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result AttachActivateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        return nnt::mifare::wrapper::AttachActivateEvent(pEvent, GetCurrentDeviceHandle());
    }

    nn::Result AttachDeactivateEvent(nn::os::SystemEventType* pEvent,
            const NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::AttachDeactivateEvent(pEvent, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result AttachDeactivateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        return nnt::mifare::wrapper::AttachDeactivateEvent(pEvent, GetCurrentDeviceHandle());
    }

    nn::Result AttachAvailabilityChangeEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::AttachAvailabilityChangeEvent(pEvent);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        return nn::ResultSuccess();
    }

    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State GetState() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State state = NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::GetState();
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,state,++count);
        return state;
    }

    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState GetDeviceState(
            const NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState deviceState = NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::GetDeviceState(deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,deviceState,++count);
        return deviceState;
    }

    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState GetDeviceState() NN_NOEXCEPT
    {
        return nnt::mifare::wrapper::GetDeviceState(GetCurrentDeviceHandle());
    }

    nn::Result GetNpadId(nn::hid::NpadIdType* pOutNpadId,
            const NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::GetNpadId(pOutNpadId, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result GetNpadId(nn::hid::NpadIdType* pOutNpadId) NN_NOEXCEPT
    {
        return nnt::mifare::wrapper::GetNpadId(pOutNpadId, GetCurrentDeviceHandle());
    }

    nn::Result Read(NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockData) * pOutBlockData,
            const NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle& deviceHandle,
            const NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockParameter) * pBlockParameter,
            size_t blockCount) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
#if !defined(NNT_NFP_BUILD_CONFIG_OLD_LIBRARY) && defined(NNT_MIFARE_PLATFORM_NX)
        nn::Result result = nn::nfc::ReadMifare(pOutBlockData,
                deviceHandle, pBlockParameter, blockCount);
#else
        nn::Result result = nn::nfc::mifare::Read(pOutBlockData,
                deviceHandle, pBlockParameter, blockCount);
#endif
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result Read(NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockData) * pOutBlockData,
            const NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockParameter) * pBlockParameter,
            size_t blockCount) NN_NOEXCEPT
    {
        return nnt::mifare::wrapper::Read(pOutBlockData,
                GetCurrentDeviceHandle(), pBlockParameter, blockCount);
    }

    nn::Result Write(const NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle& deviceHandle,
            const NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) * pBlockParameter,
            size_t blockCount) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
#if !defined(NNT_NFP_BUILD_CONFIG_OLD_LIBRARY) && defined(NNT_MIFARE_PLATFORM_NX)
        nn::Result result = nn::nfc::WriteMifare(deviceHandle, pBlockParameter, blockCount);
#else
        nn::Result result = nn::nfc::mifare::Write(deviceHandle, pBlockParameter, blockCount);
#endif
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result Write(const NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) * pBlockParameter,
            size_t blockCount) NN_NOEXCEPT
    {
        return nnt::mifare::wrapper::Write(GetCurrentDeviceHandle(), pBlockParameter, blockCount);
    }

#if !defined(NNT_NFP_BUILD_CONFIG_OLD_LIBRARY) && defined(NNT_MIFARE_PLATFORM_NX)
    nn::Result InitializeSystem() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfc::InitializeSystem();
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        return nn::ResultSuccess();
    }

    void FinalizeSystem() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        static uint64_t count = 0;
        nn::nfc::FinalizeSystem();
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        return;
    }

    nn::Result InitializeDebug() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfc::InitializeSystem();
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        /* テストプログラムの互換性の為、nn::ResultSuccess()を返す */
        return nn::ResultSuccess();
    }

    void FinalizeDebug() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        static uint64_t count = 0;
        nn::nfc::FinalizeSystem();
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        return;
    }
#endif

    void SetErrorVisible(bool visible) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfc::SetErrorVisible(visible);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,++count);
    }

    bool IsErrorVisible() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        bool result = nn::nfc::IsErrorVisible();
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }
}}} //// end of namespace nnt::mifare::wrapper
