﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testNews_Common.h"
#include <nn/hid.h>

static void PostLocalNews(const char* path) NN_NOEXCEPT
{
    static nn::Bit8 g_MountRomCacheBuffer[4 * 1024];

    size_t mountRomCacheUseSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheUseSize));
    NN_ABORT_UNLESS(mountRomCacheUseSize <= sizeof (g_MountRomCacheBuffer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom("rom", g_MountRomCacheBuffer, sizeof (g_MountRomCacheBuffer)));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("rom");
    };

    nn::fs::FileHandle handle = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    static char buffer[2 * 1024 * 1024];
    size_t read = 0;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(&read, handle, 0, buffer, sizeof (buffer)));

    for (size_t i = 0; i < read - sizeof ("deletion_priority") + 1; i++)
    {
        if (std::memcmp(&buffer[i], "deletion_priority", sizeof ("deletion_priority") - 1) == 0)
        {
            char deletionPriority = static_cast<char>(rand() % 128);

            NN_LOG("deletion_priority: %d -> %d\n", buffer[i + sizeof ("deletion_priority") - 1], deletionPriority);

            // positive fixnum (0 ~ 127) の値が格納されているため、0 ~ 127 の範囲で適当に書き換える。
            buffer[i + sizeof ("deletion_priority") - 1] = deletionPriority;
            break;
        }
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::news::PostLocalNews(buffer, read));
}

static void ShowUsage() NN_NOEXCEPT
{
    NN_LOG("--------------------------------------------------\n");
    NN_LOG("A:      Post(Small)\n");
    NN_LOG("B:      Post(Large)\n");
    NN_LOG("X:      ShowList\n");
    NN_LOG("Y:      ClearStorage\n");
    NN_LOG("L:      Post300(Small)\n");
    NN_LOG("Select: ShowUsage\n");
    NN_LOG("Start:  Exit\n");
    NN_LOG("--------------------------------------------------\n");
}

TEST(StorageFull, Initialize)
{
    std::srand(static_cast<unsigned int>(nn::os::GetSystemTick().GetInt64Value()));

    nn::nifm::Initialize();
    nn::hid::InitializeDebugPad();
}

TEST(StorageFull, MainLoop)
{
    ShowUsage();

    nn::hid::DebugPadState prevPadState = {};

    while (NN_STATIC_CONDITION(1))
    {
        nn::hid::DebugPadState padState = {};
        nn::hid::GetDebugPadState(&padState);

        nn::hid::DebugPadButtonSet trigger = (~prevPadState.buttons) & padState.buttons;
        prevPadState = padState;

        if (trigger.Test<nn::hid::DebugPadButton::A>())
        {
            NN_LOG("Post(Small).\n");
            PostLocalNews("rom:/small.msgpack");
        }
        else if (trigger.Test<nn::hid::DebugPadButton::B>())
        {
            NN_LOG("Post(Large).\n");
            PostLocalNews("rom:/large.msgpack");
        }
        else if (trigger.Test<nn::hid::DebugPadButton::X>())
        {
            nn::news::Database db;
            static nn::news::NewsRecord records[300];
            int count;

            NN_LOG("ShowList.\n");
            ASSERT_RESULT_SUCCESS(db.Open());
            ASSERT_RESULT_SUCCESS(db.GetList(&count, records, "", "received_at DESC", 0, 300));

            NN_LOG("--------------------------------------------------\n");
            NN_LOG("news.db\n");
            NN_LOG("--------------------------------------------------\n");
            NN_LOG("Count = %d\n", count);

            for (int i = 0; i < count; i++)
            {
                NN_LOG("--------------------------------------------------\n");
                NN_LOG("NewsRecord[%d]\n", i);
                NN_LOG("    - NewsId: %s\n", records[i].newsId.value);
                NN_LOG("    - UserId: %s\n", records[i].userId.value);
                NN_LOG("    - ReceivedTime: %lld\n", records[i].receivedTime.value);
                NN_LOG("    - Read: %d\n", records[i].read);
                NN_LOG("    - Newly: %d\n", records[i].newly);
                NN_LOG("    - Displayed: %d\n", records[i].displayed);

                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
            }

            NN_LOG("--------------------------------------------------\n");
        }
        else if (trigger.Test<nn::hid::DebugPadButton::Y>())
        {
            NN_LOG("ClearStorage.\n");
            EXPECT_RESULT_SUCCESS(nn::news::ClearStorage());
        }
        else if (trigger.Test<nn::hid::DebugPadButton::L>())
        {
            NN_LOG("Post300(Small).\n");

            for (int i = 0; i < 300; i++)
            {
                NN_LOG("Post(%d)\n", i);
                PostLocalNews("rom:/small.msgpack");
            }
        }
        else if (trigger.Test<nn::hid::DebugPadButton::Select>())
        {
            ShowUsage();
        }
        else if (trigger.Test<nn::hid::DebugPadButton::Start>())
        {
            NN_LOG("Exit.\n");
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }
}

TEST(StorageFull, Finalize)
{
#if defined (NN_BUILD_CONFIG_OS_WIN)
    nn::news::service::StopServer();
#endif
}
