﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testNews_Common.h"
#include <nn/news/detail/service/core/news_NewsListReader.h>
#include <nn/news/detail/service/msgpack/news_FileInputStream.h>

using namespace nn::news::detail::service::core;

namespace
{
    nn::Bit8 g_ListData[4 * 1024 * 1024];
    size_t g_ListDataSize = 0;
}

void ReadList() NN_NOEXCEPT
{
    static nn::Bit8 g_MountRomCacheBuffer[4 * 1024];

    size_t mountRomCacheUseSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheUseSize));
    NN_ABORT_UNLESS(mountRomCacheUseSize <= sizeof (g_MountRomCacheBuffer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom("rom", g_MountRomCacheBuffer, sizeof (g_MountRomCacheBuffer)));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("rom");
    };

    nn::fs::FileHandle handle = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, "rom:/list.msgpack", nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(&g_ListDataSize, handle, 0, g_ListData, sizeof (g_ListData)));
}

TEST(ListParser, Initialize)
{
    nn::time::Initialize();

    ReadList();
}

void DumpListInfo(const NewsListReader::ListInfo& info)
{
    NN_SDK_LOG("--------------------------------------------------\n");
    NN_SDK_LOG("ListInfo\n");
    NN_SDK_LOG("--------------------------------------------------\n");
    NN_SDK_LOG("topicId:\n");
    NN_SDK_LOG("    %s\n", info.topicId.value);
    NN_SDK_LOG("isInService:\n");
    NN_SDK_LOG("    %d\n", info.isInService);
    NN_SDK_LOG("isNintendoAccountRequired:\n");
    NN_SDK_LOG("    %d\n", info.isNintendoAccountRequired);
}

void DumpDataEntry(const NewsListReader::DataEntry* entries, int count)
{
    NN_SDK_LOG("--------------------------------------------------\n");
    NN_SDK_LOG("Count = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        NN_SDK_LOG("--------------------------------------------------\n");
        NN_SDK_LOG("DataEntry[%d]\n", i);
        NN_SDK_LOG("--------------------------------------------------\n");
        NN_SDK_LOG("version:\n");
        NN_SDK_LOG("    %d.%d\n", entries[i].version.format, entries[i].version.semantics);
        NN_SDK_LOG("newsId:\n");
        NN_SDK_LOG("    %llu\n", entries[i].newsId);
        NN_SDK_LOG("dataId:\n");
        NN_SDK_LOG("    %llu\n", entries[i].dataId);
        NN_SDK_LOG("url:\n");
        NN_SDK_LOG("    %s\n", entries[i].url.value);
        NN_SDK_LOG("size:\n");
        NN_SDK_LOG("    %llu\n", entries[i].size);
        NN_SDK_LOG("isOverwrite:\n");
        NN_SDK_LOG("    %d\n", entries[i].isOverwrite);
    }
}

TEST(ListParser, ListParser)
{
    nne::nlib::MemoryInputStream stream;
    stream.Init(g_ListData, g_ListDataSize);

    NewsListReader::DataEntry entries[30];
    int count = 0;

    NewsListReader reader;

    reader.SetLanguage("ja");
    ASSERT_RESULT_SUCCESS(reader.Read(&count, stream, entries, 0, 30));

    DumpListInfo(reader.GetListInfo());
    DumpDataEntry(entries, count);
}

TEST(ListParser, Finalize)
{
    nn::time::Finalize();
}
