﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 Test process for Network
 *---------------------------------------------------------------------------*/

#include "NetTest_Port.h"
#include "Api/testNet_ApiCommon.h" // ERROR_IF

#include "Tests/ResolverGetAddrInfoTest.h"
#include "Utils/CommandLineParser.h"

#include <nnt/nntest.h>

#include <nn/init.h>
#include <nn/nn_Assert.h>
#include <cstring> // strcpy

namespace {

    // ParseOptions
    bool ParseOptions(char* testName, nn::util::Uuid* pNetProfile) NN_NOEXCEPT
    {
        bool isSuccess = true;
        nn::Result rval;

        NATF::Utils::ParserGroup parser;
        nn::util::Uuid netProfileLocal;
        char testNameLocal[NATF::BaseTest::NameBufferLen];

        int argC;
        const char * const * argV;

        ERROR_IF(nullptr == testName, "TestSetup: testName nullptr");
        ERROR_IF(nullptr == pNetProfile, "TestSetup: pNetProfile nullptr");

        NETTEST_GET_ARGS(argC, argV);

        parser.AddParser(NATF::Utils::StringParser  ("--Name", nullptr, testNameLocal, sizeof(testNameLocal)));
        parser.AddParser(NATF::Utils::UuidParser    ("--NetProfile", &nn::util::InvalidUuid, netProfileLocal));

        ERROR_IF( !parser.Parse(argC, argV), "Failed to parse command line arguments");

        strcpy(testName, testNameLocal);
        *pNetProfile = netProfileLocal;

out:
        return isSuccess;
    }

    // ResolverGetAddrInfo_CheckNoError
    TEST(natf, ResolverGetAddrInfo_CheckNoError)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetAddrInfoTest *pResolverGetAddrInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetAddrInfo_CheckNoError: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetAddrInfoTest = new NATF::Tests::ResolverGetAddrInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetAddrInfoTest, "NATF::Tests::ResolverGetAddrInfoTest constructor failed");

        pResolverGetAddrInfoTest->CheckNoError();
        isSuccess = pResolverGetAddrInfoTest->Run();

out:
        if( nullptr != pResolverGetAddrInfoTest )
        {
            delete pResolverGetAddrInfoTest;
            pResolverGetAddrInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetAddrInfo_CheckEaiAddrfamily
    TEST(natf, ResolverGetAddrInfo_CheckEaiAddrfamily)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetAddrInfoTest *pResolverGetAddrInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetAddrInfo_CheckEaiAddrfamily: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetAddrInfoTest = new NATF::Tests::ResolverGetAddrInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetAddrInfoTest, "NATF::Tests::ResolverGetAddrInfoTest constructor failed");

        pResolverGetAddrInfoTest->CheckEaiAddrfamily();
        isSuccess = pResolverGetAddrInfoTest->Run();

out:
        if( nullptr != pResolverGetAddrInfoTest )
        {
            delete pResolverGetAddrInfoTest;
            pResolverGetAddrInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetAddrInfo_CheckEaiAgain
    TEST(natf, ResolverGetAddrInfo_CheckEaiAgain)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetAddrInfoTest *pResolverGetAddrInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetAddrInfo_CheckEaiAgain: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetAddrInfoTest = new NATF::Tests::ResolverGetAddrInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetAddrInfoTest, "NATF::Tests::ResolverGetAddrInfoTest constructor failed");

        pResolverGetAddrInfoTest->CheckEaiAgain();
        isSuccess = pResolverGetAddrInfoTest->Run();

out:
        if( nullptr != pResolverGetAddrInfoTest )
        {
            delete pResolverGetAddrInfoTest;
            pResolverGetAddrInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetAddrInfo_CheckEaiBadflagsWithInvalidFlags
    TEST(natf, ResolverGetAddrInfo_CheckEaiBadflagsWithInvalidFlags)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetAddrInfoTest *pResolverGetAddrInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetAddrInfo_CheckEaiBadflagsWithInvalidFlags: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetAddrInfoTest = new NATF::Tests::ResolverGetAddrInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetAddrInfoTest, "NATF::Tests::ResolverGetAddrInfoTest constructor failed");

        pResolverGetAddrInfoTest->CheckEaiBadflagsWithInvalidFlags();
        isSuccess = pResolverGetAddrInfoTest->Run();

out:
        if( nullptr != pResolverGetAddrInfoTest )
        {
            delete pResolverGetAddrInfoTest;
            pResolverGetAddrInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetAddrInfo_CheckEaiBadflagsWithAiCanonname
    TEST(natf, ResolverGetAddrInfo_CheckEaiBadflagsWithAiCanonname)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetAddrInfoTest *pResolverGetAddrInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetAddrInfo_CheckEaiBadflagsWithAiCanonname: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetAddrInfoTest = new NATF::Tests::ResolverGetAddrInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetAddrInfoTest, "NATF::Tests::ResolverGetAddrInfoTest constructor failed");

        pResolverGetAddrInfoTest->CheckEaiBadflagsWithAiCanonname();
        isSuccess = pResolverGetAddrInfoTest->Run();

out:
        if( nullptr != pResolverGetAddrInfoTest )
        {
            delete pResolverGetAddrInfoTest;
            pResolverGetAddrInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetAddrInfo_CheckEaiFail
    TEST(natf, ResolverGetAddrInfo_CheckEaiFail)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetAddrInfoTest *pResolverGetAddrInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetAddrInfo_CheckEaiFail: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetAddrInfoTest = new NATF::Tests::ResolverGetAddrInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetAddrInfoTest, "NATF::Tests::ResolverGetAddrInfoTest constructor failed");

        pResolverGetAddrInfoTest->CheckEaiFail();
        isSuccess = pResolverGetAddrInfoTest->Run();

out:
        if( nullptr != pResolverGetAddrInfoTest )
        {
            delete pResolverGetAddrInfoTest;
            pResolverGetAddrInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetAddrInfo_CheckEaiFamily
    TEST(natf, ResolverGetAddrInfo_CheckEaiFamily)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetAddrInfoTest *pResolverGetAddrInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetAddrInfo_CheckEaiFamily: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetAddrInfoTest = new NATF::Tests::ResolverGetAddrInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetAddrInfoTest, "NATF::Tests::ResolverGetAddrInfoTest constructor failed");

        pResolverGetAddrInfoTest->CheckEaiFamily();
        isSuccess = pResolverGetAddrInfoTest->Run();

out:
        if( nullptr != pResolverGetAddrInfoTest )
        {
            delete pResolverGetAddrInfoTest;
            pResolverGetAddrInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetAddrInfo_CheckEaiMemory
    TEST(natf, ResolverGetAddrInfo_CheckEaiMemory)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetAddrInfoTest *pResolverGetAddrInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetAddrInfo_CheckEaiMemory: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetAddrInfoTest = new NATF::Tests::ResolverGetAddrInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetAddrInfoTest, "NATF::Tests::ResolverGetAddrInfoTest constructor failed");

        pResolverGetAddrInfoTest->CheckEaiMemory();
        isSuccess = pResolverGetAddrInfoTest->Run();

out:
        if( nullptr != pResolverGetAddrInfoTest )
        {
            delete pResolverGetAddrInfoTest;
            pResolverGetAddrInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetAddrInfo_CheckEaiNodata
    TEST(natf, ResolverGetAddrInfo_CheckEaiNodata)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetAddrInfoTest *pResolverGetAddrInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetAddrInfo_CheckEaiNodata: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetAddrInfoTest = new NATF::Tests::ResolverGetAddrInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetAddrInfoTest, "NATF::Tests::ResolverGetAddrInfoTest constructor failed");

        pResolverGetAddrInfoTest->CheckEaiNodata();
        isSuccess = pResolverGetAddrInfoTest->Run();

out:
        if( nullptr != pResolverGetAddrInfoTest )
        {
            delete pResolverGetAddrInfoTest;
            pResolverGetAddrInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetAddrInfo_CheckEaiNonameWithUnknownArgs
    TEST(natf, ResolverGetAddrInfo_CheckEaiNonameWithUnknownArgs)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetAddrInfoTest *pResolverGetAddrInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetAddrInfo_CheckEaiNonameWithUnknownArgs: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetAddrInfoTest = new NATF::Tests::ResolverGetAddrInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetAddrInfoTest, "NATF::Tests::ResolverGetAddrInfoTest constructor failed");

        pResolverGetAddrInfoTest->CheckEaiNonameWithUnknownArgs();
        isSuccess = pResolverGetAddrInfoTest->Run();

out:
        if( nullptr != pResolverGetAddrInfoTest )
        {
            delete pResolverGetAddrInfoTest;
            pResolverGetAddrInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetAddrInfo_CheckEaiNonameWithNullArgs
    TEST(natf, ResolverGetAddrInfo_CheckEaiNonameWithNullArgs)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetAddrInfoTest *pResolverGetAddrInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetAddrInfo_CheckEaiNonameWithNullArgs: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetAddrInfoTest = new NATF::Tests::ResolverGetAddrInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetAddrInfoTest, "NATF::Tests::ResolverGetAddrInfoTest constructor failed");

        pResolverGetAddrInfoTest->CheckEaiNonameWithNullArgs();
        isSuccess = pResolverGetAddrInfoTest->Run();

out:
        if( nullptr != pResolverGetAddrInfoTest )
        {
            delete pResolverGetAddrInfoTest;
            pResolverGetAddrInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetAddrInfo_CheckEaiNonameWithAiNumericserv
    TEST(natf, ResolverGetAddrInfo_CheckEaiNonameWithAiNumericserv)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetAddrInfoTest *pResolverGetAddrInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetAddrInfo_CheckEaiNonameWithAiNumericserv: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetAddrInfoTest = new NATF::Tests::ResolverGetAddrInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetAddrInfoTest, "NATF::Tests::ResolverGetAddrInfoTest constructor failed");

        pResolverGetAddrInfoTest->CheckEaiNonameWithAiNumericserv();
        isSuccess = pResolverGetAddrInfoTest->Run();

out:
        if( nullptr != pResolverGetAddrInfoTest )
        {
            delete pResolverGetAddrInfoTest;
            pResolverGetAddrInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetAddrInfo_CheckEaiService
    TEST(natf, ResolverGetAddrInfo_CheckEaiService)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetAddrInfoTest *pResolverGetAddrInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetAddrInfo_CheckEaiService: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetAddrInfoTest = new NATF::Tests::ResolverGetAddrInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetAddrInfoTest, "NATF::Tests::ResolverGetAddrInfoTest constructor failed");

        pResolverGetAddrInfoTest->CheckEaiService();
        isSuccess = pResolverGetAddrInfoTest->Run();

out:
        if( nullptr != pResolverGetAddrInfoTest )
        {
            delete pResolverGetAddrInfoTest;
            pResolverGetAddrInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetAddrInfo_CheckEaiSocktype
    TEST(natf, ResolverGetAddrInfo_CheckEaiSocktype)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetAddrInfoTest *pResolverGetAddrInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetAddrInfo_CheckEaiSocktype: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetAddrInfoTest = new NATF::Tests::ResolverGetAddrInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetAddrInfoTest, "NATF::Tests::ResolverGetAddrInfoTest constructor failed");

        pResolverGetAddrInfoTest->CheckEaiSocktype();
        isSuccess = pResolverGetAddrInfoTest->Run();

out:
        if( nullptr != pResolverGetAddrInfoTest )
        {
            delete pResolverGetAddrInfoTest;
            pResolverGetAddrInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetAddrInfo_CheckEaiSystem
    TEST(natf, ResolverGetAddrInfo_CheckEaiSystem)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetAddrInfoTest *pResolverGetAddrInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetAddrInfo_CheckEaiSystem: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetAddrInfoTest = new NATF::Tests::ResolverGetAddrInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetAddrInfoTest, "NATF::Tests::ResolverGetAddrInfoTest constructor failed");

        pResolverGetAddrInfoTest->CheckEaiSystem();
        isSuccess = pResolverGetAddrInfoTest->Run();

out:
        if( nullptr != pResolverGetAddrInfoTest )
        {
            delete pResolverGetAddrInfoTest;
            pResolverGetAddrInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

}
