﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 Test process for Network
 *---------------------------------------------------------------------------*/

#include "NetTest_Port.h"

#include "Tests/CurlHttpUploadTest.h"
#include "Utils/CommandLineParser.h"
#include "../Common/ParseHelpers.h"

#include <nnt/nntest.h>
#include <curl/curl.h>
#include <cstring>

namespace
{
    const uint32_t WaitTypeBufLen = 16;
    const uint32_t ProxyAuthBufLen = 32;
    const uint32_t UploadDataBufLen = 128;
    const uint32_t CacheModeBufLen = 8;
}

// CurlHttpUpload
TEST(natf, CurlHttpUpload)
{
    bool isSuccess;
    NATF::Utils::ParserGroup parser;
    NATF::Modules::LibCurl::Params params;
    int argc;
    int32_t defaultHttpResponse = 200;
    uint32_t timeoutMs = 0;
    int32_t defaultUploadSize = -1;
    int32_t uploadSizeMb = 0;
    int32_t uploadSizeKb = 0;
    int32_t uploadSizeB = 0;
    uint32_t defaultConnectionCount = 1;
    uint32_t connectionCount = defaultConnectionCount;
    uint32_t defaultMultiApiConnectionCount = 0;
    int32_t defaultCurlReturn = 0;
    uint16_t defaultProxyPort = 0;
    int32_t defaultSendBuf = 32 * 1024;
    int32_t defaultRecvBuf = 256 * 1024;
    int defaultAuthError = 0;
    bool defaultDoVerifyPeer = true;
    bool defaultDoVerifyHostName = true;
    bool defaultDoVerifyDate = true;
    bool defaultUseInternalPki = false;
    bool defaultTeamCityUp = false;
    nn::util::Uuid netProfile;
    char pTestName[NATF::BaseTest::NameBufferLen];
    char pWaitType[WaitTypeBufLen];
    char pProxyAuth[ProxyAuthBufLen];
    char pCacheModeStr[CacheModeBufLen];
    char pUploadDataBuf[UploadDataBufLen];
    const char* pDefaultWaitType = "Select";
    const char* pDefaultProxyServer = "";
    const char* pDefaultProxyCredentials = "";
    const char* pDefaultProxyAuth = "NEGOTIATE";
    const char* pDefaultCacheMode = "NONE";
    const char * const * pArgv;

    NN_NETTEST_LOG("\nEnter NATF Process\n\n");

    NETTEST_GET_ARGS(argc, pArgv);

    parser.AddParser(NATF::Utils::StringParser ("--Name", nullptr, pTestName, sizeof(pTestName)));
    parser.AddParser(NATF::Utils::UInt32Parser ("--ConnectionCount", &defaultConnectionCount, connectionCount));
    parser.AddParser(NATF::Utils::UInt32Parser ("--MultiApiConnectionCount", &defaultMultiApiConnectionCount, params.multiApiConnectionCount));
    parser.AddParser(NATF::Utils::UuidParser   ("--NetProfile", &nn::util::InvalidUuid, netProfile));
    parser.AddParser(NATF::Utils::StringParser ("--Url", nullptr, params.pUrl, sizeof(params.pUrl)));
    parser.AddParser(NATF::Utils::Int32Parser  ("--UploadSize", &defaultUploadSize, uploadSizeB));
    parser.AddParser(NATF::Utils::Int32Parser  ("--UploadSizeMb", &defaultUploadSize, uploadSizeMb));
    parser.AddParser(NATF::Utils::Int32Parser  ("--UploadSizeKb", &defaultUploadSize, uploadSizeKb));
    parser.AddParser(NATF::Utils::Int32Parser  ("--HttpResponse", &defaultHttpResponse, params.expectedHttpResponse));
    parser.AddParser(NATF::Utils::Int32Parser  ("--SslAuthError", &defaultAuthError, params.expectedAuthReturn));
    parser.AddParser(NATF::Utils::Int32Parser  ("--SendBuffer", &defaultSendBuf, params.sendBuffer));
    parser.AddParser(NATF::Utils::Int32Parser  ("--RecvBuffer", &defaultRecvBuf, params.recvBuffer));
    parser.AddParser(NATF::Utils::UInt32Parser ("--TimeoutMs", nullptr, timeoutMs));
    parser.AddParser(NATF::Utils::StringParser ("--WaitType", pDefaultWaitType, pWaitType, sizeof(pWaitType)));
    parser.AddParser(NATF::Utils::BoolParser   ("--VerifyPeer", &defaultDoVerifyPeer, params.doVerifyPeer));
    parser.AddParser(NATF::Utils::BoolParser   ("--VerifyHostName", &defaultDoVerifyHostName, params.doVerifyHostname));
    parser.AddParser(NATF::Utils::BoolParser   ("--VerifyDate", &defaultDoVerifyDate, params.doVerifyDate));
    parser.AddParser(NATF::Utils::StringParser ("--SessionCacheMode", pDefaultCacheMode, pCacheModeStr, sizeof(pCacheModeStr)));
    parser.AddParser(NATF::Utils::BoolParser   ("--UseInternalPki", &defaultUseInternalPki, params.useInternalPki));
    parser.AddParser(NATF::Utils::BoolParser   ("--LogTeamCityUpSpeed", &defaultTeamCityUp, params.publishUpSpeedToTeamCity));
    parser.AddParser(NATF::Utils::StringParser ("--ProxyServer", pDefaultProxyServer, params.pProxyServer, sizeof(params.pProxyServer)));
    parser.AddParser(NATF::Utils::StringParser ("--ProxyCredentials", pDefaultProxyCredentials, params.pProxyUserPwd, sizeof(params.pProxyUserPwd)));
    parser.AddParser(NATF::Utils::StringParser ("--ClientCert", "", params.pClientCertPath, sizeof(params.pClientCertPath)));
    parser.AddParser(NATF::Utils::StringParser ("--ServerCert", "", params.pServerCertPath, sizeof(params.pServerCertPath)));
    parser.AddParser(NATF::Utils::StringParser ("--HttpHeaderOption", "", params.pHttpHeaderOption, sizeof(params.pHttpHeaderOption)));
    parser.AddParser(NATF::Utils::UInt16Parser ("--ProxyPort", &defaultProxyPort, params.proxyPort));
    parser.AddParser(NATF::Utils::StringParser ("--ProxyAuth", pDefaultProxyAuth, pProxyAuth, sizeof(pProxyAuth)));
    parser.AddParser(NATF::Utils::Int32Parser  ("--CurlReturn", &defaultCurlReturn, params.expectedCurlReturn));
    parser.AddParser(NATF::Utils::StringParser ("--UploadData", "", pUploadDataBuf, sizeof(pUploadDataBuf)));

    if (!parser.Parse(argc, pArgv))
    {
        NN_NETTEST_LOG(" * Failed to parse command line arguements!\n\n");
        EXPECT_EQ(false, true);
        return;
    }

    NetTest::StrUpr(pWaitType);
    if(nnt::ParserHelpers::GetWaitTypeFromString(pWaitType, WaitTypeBufLen, params.useSelect) == false)
    {
        FAIL();
        return;
    }

    if( strncmp(pProxyAuth, "", sizeof(pProxyAuth)) != 0 )
    {
        NetTest::StrUpr(pProxyAuth);
        if(nnt::ParserHelpers::GetProxyAuthFromString(pProxyAuth, sizeof(pProxyAuth), params.proxyAuthMethod) == false)
        {
            FAIL();
            return;
        }
    }

    NetTest::StrUpr(pCacheModeStr);
    if(nnt::ParserHelpers::GetCacheModeFromString(pCacheModeStr, sizeof(pCacheModeStr), params.sessionCacheMode) == false)
    {
        FAIL();
        return;
    }

    if(pUploadDataBuf[0] != '\0')
    {
        params.pUploadData = pUploadDataBuf;
    }

    if(uploadSizeMb == 0 || uploadSizeKb == 0 || uploadSizeB == 0)
    {
        NN_NETTEST_LOG("\n * Error: UploadSize must be more than '0'.\n\n");
        FAIL();
        return;
    }

    if(uploadSizeMb > -1)
    {
        params.uploadSize = 1024 * 1024 * static_cast<uint32_t>(uploadSizeMb);
    }
    else if(uploadSizeKb > -1)
    {
        params.uploadSize = 1024 * static_cast<uint32_t>(uploadSizeKb);
    }
    else if(uploadSizeB > -1)
    {
        params.uploadSize = uploadSizeB;
    }
    else
    {
        NN_NETTEST_LOG("\n * Error: --UploadSize, --UploadSizeMb or --UploadSizeKb must be specified!\n");
        FAIL();
        return;
    }

    NATF::Tests::CurlHttpUpload curlHttpUpload(pTestName, netProfile, timeoutMs, connectionCount, params);

    isSuccess = curlHttpUpload.Run();
    EXPECT_EQ(isSuccess, true);

    NN_NETTEST_LOG("\nExit NATF Process\n\n");
} // NOLINT(impl/function_size)
