﻿
/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdio>
#include <cstdlib>
#include <cinttypes>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nnt/nnt_Argument.h>

#include <unistd.h>
#include <sys/sysctl.h>
#include <sys/ctype.h>

const int            MaxMibLength   = 22;
const uint32_t       MemoryPoolSize = (128 * 1024);

nn::lmem::HeapHandle g_HeapHandle;
uint8_t              g_HeapMemoryPool[MemoryPoolSize] NN_ALIGNAS(nn::os::ThreadStackAlignment);
uint32_t             g_PrintMode;
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

enum
{
    PrintMode_aflag = (1<<0),   // print all
    PrintMode_dflag = (1<<1),   // print description
    PrintMode_oflag = (1<<7),   // print opaque
    PrintMode_Tflag = (1<<9),   // print variables settable by loader only
    PrintMode_Wflag = (1<<10),  // print writable non stat variables
    PrintMode_xflag = (1<<11),  // as -o but print all data as hex instead of first few bytes
};

void* Allocate(size_t size)
NN_NOEXCEPT
{
    return nn::lmem::AllocateFromExpHeap(g_HeapHandle, size);
}

void Free(void* p)
NN_NOEXCEPT
{
    nn::lmem::FreeToExpHeap(g_HeapHandle, p);
}

void* operator new(size_t size)
NN_NOEXCEPT
{
    return Allocate(size);
}

void operator delete(void* p)
NN_NOEXCEPT
{
    Free(p);
}

void ShowHelp()
NN_NOEXCEPT
{
    NN_LOG("%s\n",
        "usage: sysctl [-doTWx] name[=value] ...\n"
        "       sysctl [-doTWx] -a\n"
        "       -a      List all currently available non-opaque values.  This option\n"
        "               is ignored if one or more variable names are specified on the\n"
        "               command line.\n"
        "       -d      Print the description of the variable instead of its value.\n"
        "       -o      Show opaque variables (which are normally suppressed).  The format\n"
        "               and length are printed, as well as a hex dump of the first sixteen\n"
        "               bytes of the value.\n"
        "       -T      Display only variables that are setable via loader (CTLFLAG_TUN).\n"
        "       -W      Display only stats unrelated writable variables.  Useful\n"
        "               for determining the set of runtime tunable sysctls.\n"
        "       -x      The same as -o, but prints a hex dump of the entire value instead of\n"
        "               the first few bytes.\n"
    );
}

class MibContainer
{
private:
    struct
    {
        uint32_t type;
        uint8_t  formatData[100];
    } m_Format;

    uint8_t  m_Name[100];
    uint8_t  m_Description[100];
    uint8_t* m_DataPointer;
    size_t   m_DataLength;
    int      m_Mib[MaxMibLength];
    size_t   m_MibLength;

    size_t m_TypeLength[CTLTYPE + 1] =
    {
        [CTLTYPE_INT]   = sizeof(int),
        [CTLTYPE_UINT]  = sizeof(unsigned int),
        [CTLTYPE_LONG]  = sizeof(long),
        [CTLTYPE_ULONG] = sizeof(unsigned long),
        [CTLTYPE_S64]   = sizeof(int64_t),
        [CTLTYPE_U64]   = sizeof(uint64_t)
    };

    int m_TypeSign[CTLTYPE + 1] =
    {
        [CTLTYPE_INT]   = 1,
        [CTLTYPE_UINT]  = 0,
        [CTLTYPE_LONG]  = 1,
        [CTLTYPE_ULONG] = 0,
        [CTLTYPE_S64]   = 1,
        [CTLTYPE_U64]   = 0
    };

public:
    MibContainer()
    NN_NOEXCEPT
        : m_DataPointer(nullptr)
    {

    }

    void PrintValue()
    NN_NOEXCEPT
    {
        uint32_t  type = m_Format.type & CTLTYPE;
        uint8_t*  dataPointer = m_DataPointer;
        size_t    dataLength  = m_DataLength;
        int       valueSign;
        size_t    valueLength;
        size_t    hexLength;
        uintmax_t value;

        if ((g_PrintMode & PrintMode_Tflag) && !(m_Format.type & CTLFLAG_TUN))
        {
            return;
        }
        if ((g_PrintMode & PrintMode_Wflag) && (!(m_Format.type & CTLFLAG_WR) || (m_Format.type & CTLFLAG_STATS)))
        {
            return;
        }
        if (g_PrintMode & PrintMode_dflag)
        {
            NN_LOG("%s:%s\n", m_Name, m_Description);
            return;
        }
        switch (type)
        {
        case CTLTYPE_STRING:
            NN_LOG("%s:%.*s\n", m_Name, (int)m_DataLength, m_DataPointer);
            break;
        case CTLTYPE_INT:
        case CTLTYPE_UINT:
        case CTLTYPE_LONG:
        case CTLTYPE_ULONG:
        case CTLTYPE_S64:
        case CTLTYPE_U64:
            valueSign   = m_TypeSign[type];
            valueLength = m_TypeLength[type];
            hexLength   = 2 + (valueLength * 8 + 3) / 4;
            NN_LOG("%s:", m_Name);
            while (dataLength >= valueLength)
            {
                switch (type)
                {
                case CTLTYPE_INT:
                case CTLTYPE_UINT:
                    value = *(unsigned int*)m_DataPointer;
                    break;
                case CTLTYPE_LONG:
                case CTLTYPE_ULONG:
                    value = *(uint32_t*)m_DataPointer;
                    break;
                case CTLTYPE_S64:
                case CTLTYPE_U64:
                    value = *(uint64_t*)m_DataPointer;
                    break;
                default:
                    NN_LOG("unknown type\n");
                    return;
                }
                if (g_PrintMode & PrintMode_xflag)
                {
                    NN_LOG("%#0*jx", hexLength, value);
                }
                NN_LOG(valueSign ? "%jd" : "%ju", value);
                dataLength  -= valueLength;
                dataPointer += valueLength;
            }
            NN_LOG("\n");
            break;
        case CTLTYPE_OPAQUE:
        default:
            if (!(g_PrintMode & (PrintMode_oflag | PrintMode_xflag)))
            {
                return;
            }
            NN_LOG("%s: Format:%s Length:%zu Dump:0x", m_Name, m_Format.formatData, m_DataLength);
            while (dataLength-- &&
                  ((g_PrintMode & PrintMode_xflag) ||
                  (dataPointer < m_DataPointer + 16)))
            {
                NN_LOG("%02x", *dataPointer++);
            }
            if (!(g_PrintMode & PrintMode_xflag) && dataLength > 16)
            {
                NN_LOG("...");
            }
            NN_LOG("\n");
        }
    }

    void SetValue(char* valueString)
    NN_NOEXCEPT
    {
        int           integerValue;
        unsigned int  uintegerValue;
        long          longValue;
        unsigned long ulongValue;
        int64_t       longlongValue;
        uint64_t      ulonglongValue;
        void*         valuePointer;
        size_t        valueLength;
        char*         valueStringEndPointer;

        uint32_t type = m_Format.type & CTLTYPE;

        if (m_Format.type & CTLTYPE_NODE)
        {
            NN_LOG("%s is not a leaf node\n", m_Name);
            return;
        }
        if (!(m_Format.type & CTLFLAG_WR))
        {
            NN_LOG("%s is read only\n", m_Name);
            return;
        }
        if (type == CTLTYPE_INT   ||
            type == CTLTYPE_UINT  ||
            type == CTLTYPE_LONG  ||
            type == CTLTYPE_ULONG ||
            type == CTLTYPE_S64   ||
            type == CTLTYPE_U64)
        {
            if (strlen(valueString) == 0)
            {
                NN_LOG("empty numberic value\n");
                return;
            }
        }

        switch (type)
        {
        case CTLTYPE_INT:
            integerValue   = (int)strtol((const char*)valueString, (char**)&valueStringEndPointer, 0);
            valuePointer   = &integerValue;
            valueLength    = sizeof(int);
            break;
        case CTLTYPE_UINT:
            uintegerValue  = (unsigned int)strtoul((const char*)valueString, (char**)&valueStringEndPointer, 0);
            valuePointer   = &uintegerValue;
            valueLength    = sizeof(unsigned int);
            break;
        case CTLTYPE_LONG:
            longValue      = strtol((const char*)valueString, (char**)&valueStringEndPointer, 0);
            valuePointer   = &ulongValue;
            valueLength    = sizeof(long);
            break;
        case CTLTYPE_ULONG:
            ulongValue     = strtoul((const char*)valueString, (char**)&valueStringEndPointer, 0);
            valuePointer   = &ulongValue;
            valueLength    = sizeof(unsigned long);
            break;
        case CTLTYPE_S64:
            longlongValue  = strtoimax((const char*)valueString, (char**)&valueStringEndPointer, 0);
            valuePointer   = &longlongValue;
            valueLength    = sizeof(int64_t);
            break;
        case CTLTYPE_U64:
            ulonglongValue = strtoumax((const char*)valueString, (char**)&valueStringEndPointer, 0);
            valuePointer   = &ulonglongValue;
            valueLength    = sizeof(uint64_t);
            break;
        case CTLTYPE_STRING:
        case CTLTYPE_OPAQUE:
        default:
            NN_LOG("can't set this type...\n");
            return;
        }

        if (valueStringEndPointer == valueString || *valueStringEndPointer != '\0')
        {
            NN_LOG("invalid numberic argument %s\n", valueString);
            return;
        }

        PrintValue();

        if (nn::socket::Sysctl(m_Mib, m_MibLength, 0, 0, valuePointer, valueLength) < 0)
        {
            switch (nn::socket::GetLastError())
            {
                case nn::socket::Errno::EOpNotSupp:
                    NN_LOG("%s: value is not available\n", m_Name);
                    break;
                case nn::socket::Errno::ENotDir:
                    NN_LOG("%s: specification is incomplete\n", m_Name);
                    break;
                case nn::socket::Errno::ENoMem:
                    NN_LOG("%s: type is unknown\n", m_Name);
                    break;
                default:
                    NN_LOG("%s: unknown error...\n", m_Name);
                    break;
            }
        }

        NN_LOG(" -> ");

        if (nn::socket::Sysctl(m_Mib, m_MibLength, m_DataPointer, &m_DataLength, 0, 0) < 0)
        {
            NN_LOG("%s: failed to read value back\n");
            return;
        }

        PrintValue();
    } // NOLINT (impl/function_size)

    int Initialize(int* pMibEntries, size_t mibEntryCount)
    NN_NOEXCEPT
    {
        int    mib[MaxMibLength];
        size_t len;
        size_t mibLength = mibEntryCount + 2;

        memcpy(&m_Mib, pMibEntries, mibEntryCount * sizeof(int));
        m_MibLength = mibEntryCount;

        memset(&mib[0], 0x0, sizeof(mib));
        memcpy(&mib[2], pMibEntries, mibEntryCount * sizeof(int));

        mib[1] = 1;
        len    = sizeof(m_Name);
        if (nn::socket::Sysctl(mib, mibLength, m_Name, &len, 0, 0) < 0)
        {
            return -1;
        }
        mib[1] = 4;
        len    =  sizeof(m_Format);
        if (nn::socket::Sysctl(mib, mibLength, (void*)&m_Format, &len, 0, 0) < 0)
        {
            return -1;
        }
        mib[1] = 5;
        len    = sizeof(m_Description);
        if (nn::socket::Sysctl(mib, mibLength, m_Description, &len, 0, 0) < 0)
        {
            return -1;
        }
        len = 0;
        if (nn::socket::Sysctl(m_Mib, m_MibLength, NULL, &len, 0, 0) < 0)
        {
            return -1;
        }
        if ((m_DataPointer = (uint8_t*)Allocate(len + 1)) == nullptr)
        {
            return -1;
        }
        m_DataLength = len;
        if (nn::socket::Sysctl(m_Mib, m_MibLength, m_DataPointer, &m_DataLength, 0, 0) < 0)
        {
            return -1;
        }
        m_DataPointer[m_DataLength] = '\0';

        return 0;
    }

    void Finalize()
    NN_NOEXCEPT
    {
        if (m_DataPointer != nullptr)
        {
            Free(m_DataPointer);
            m_DataPointer = nullptr;
        }
    }
};

int MibByName(const char* name, int* pMibEntries, size_t mibEntryCount)
NN_NOEXCEPT
{
    int    mib1[]    = {0, 3};
    size_t mibLength = mibEntryCount * sizeof(int);
    if (nn::socket::Sysctl(mib1, sizeof(mib1) / sizeof(mib1[0]), pMibEntries, &mibLength, (void*)name, strlen(name)) < 0)
    {
        return -1;
    }
    return mibLength / sizeof(int);
}

void Execute(char* argv)
NN_NOEXCEPT
{
    MibContainer* mibContainer;

    int      mib[MaxMibLength];
    int      mibLength;
    uint8_t  commandBuffer[100];
    uint8_t* valueBufferPointer = commandBuffer;
    uint8_t* mibStringPointer;

    if (snprintf((char*)commandBuffer, sizeof(commandBuffer), "%s", argv) >= (int)(sizeof(commandBuffer)))
    {
        NN_LOG("sysctl id %s too long\n", argv);
        return;
    }

    mibStringPointer = (uint8_t*)strsep((char**)&valueBufferPointer, "=:");

    if (valueBufferPointer != nullptr)
    {
        if (g_PrintMode & (PrintMode_Tflag | PrintMode_Wflag))
        {
            NN_LOG("can't set variables when using -T or -W\n");
            ShowHelp();
            return;
        }
        while (isspace(*valueBufferPointer))
        {
            valueBufferPointer++;
        }
        switch(*valueBufferPointer)
        {
            case '\"':
            case '\'':
                if (valueBufferPointer[strlen((const char*)valueBufferPointer) - 1] == *valueBufferPointer)
                {
                    valueBufferPointer[strlen((const char*)valueBufferPointer) - 1] = '\0';
                }
                valueBufferPointer++;
                break;
            default:
                break;
        }
    }

    if ((mibLength =  MibByName((const char*)commandBuffer, mib, sizeof(mib) / sizeof(mib[0]))) < 0)
    {
        NN_LOG("unknown sysctl id %s\n", commandBuffer);
        return;
    }

    if ((mibContainer = new MibContainer()) == nullptr)
    {
        NN_LOG("out of memory\n");
        return;
    }

    if (mibContainer->Initialize(mib, (size_t)mibLength) == 0)
    {
        if (valueBufferPointer == nullptr)
        {
            mibContainer->PrintValue();
        }
        else
        {
            mibContainer->SetValue((char*)valueBufferPointer);
        }
    }

    delete(mibContainer);

    return;
}

int ShowAll(int* pRootMib, size_t rootMibLength)
NN_NOEXCEPT
{
    int           mib1[MaxMibLength];
    int           mib2[MaxMibLength];
    size_t        mib1Length;
    size_t        mib2Length;
    int           result;
    MibContainer* mibContainer;

    mib1[0]    = 0;
    mib1[1]    = 2;
    mib1Length = 2;

    if ((mibContainer = new MibContainer()) == nullptr)
    {
        return -1;
    }

    if (rootMibLength)
    {
        memcpy(mib1 + 2, pRootMib, rootMibLength * sizeof(int));
        mib1Length += rootMibLength;
    } else {
        mib1[2] = 1;
        mib1Length++;
    }

    for (;;)
    {
        mib2Length = sizeof(mib2);

        result = nn::socket::Sysctl(mib1, mib1Length, (void*)mib2, &mib2Length, 0, 0);

        if (result < 0)
        {
            if (nn::socket::GetLastError() == nn::socket::Errno::ENoEnt)
            {
                goto out;
            }
            else
            {
                NN_LOG("sysctl(getnext) %d %zu", nn::socket::GetLastError(), mib2Length);
            }
        }

        mib2Length /= sizeof(int);

        if (mib2Length < rootMibLength)
        {
            goto out;
        }

        for (size_t i = 0; i < rootMibLength; i++)
        {
            if (mib2[i] != pRootMib[i])
            {
                goto out;
            }
        }

        if (mibContainer->Initialize(mib2, mib2Length) == 0)
        {
            mibContainer->PrintValue();
            mibContainer->Finalize();
        }

        memcpy(mib1 + 2, mib2, mib2Length * sizeof(int));

        mib1Length = 2 + mib2Length;
    }

out:
    delete(mibContainer);

    return 0;
}

extern "C" void nninitStartup()
{
    nn::os::SetMemoryHeapSize(nn::socket::DefaultSocketMemoryPoolSize);
    g_HeapHandle = nn::lmem::CreateExpHeap(g_HeapMemoryPool, sizeof(g_HeapMemoryPool), nn::lmem::CreationOption_NoOption);
}

extern "C" void nnMain()
{
    int    argc;
    char** argv;
    int    ch;

    g_PrintMode = 0;

    argc = nnt::GetHostArgc();
    argv = nnt::GetHostArgv();

    while ((ch = getopt(argc, argv, "Aabdef:hiNnoqTWxX")) != -1) {
        switch (ch) {
        case 'a':
            g_PrintMode |= PrintMode_aflag;
            break;
        case 'd':
            g_PrintMode |= PrintMode_dflag;
            break;
        case 'o':
            g_PrintMode |= PrintMode_oflag;
            break;
        case 'T':
            g_PrintMode |= PrintMode_Tflag;
            break;
        case 'W':
            g_PrintMode |= PrintMode_Wflag;
            break;
        case 'x':
            g_PrintMode |= PrintMode_xflag;
            break;
        case 'A':
        case 'b':
        case 'e':
        case 'f':
        case 'h':
        case 'i':
        case 'N':
        case 'n':
        case 'q':
        case 'X':
        case '?':
        default:
            NN_LOG("option '%c' not supported\n", ch);
            ShowHelp();
            return;
        }
    }
    argc -= optind;
    argv += optind;

    nn::socket::Initialize(reinterpret_cast<void*>(g_SocketMemoryPoolBuffer),
                           nn::socket::DefaultSocketMemoryPoolSize,
                           nn::socket::DefaultSocketAllocatorSize,
                           nn::socket::DefaultConcurrencyLimit);

    if ((g_PrintMode & PrintMode_aflag) && argc == 0)
    {
        ShowAll(0, 0);
        goto out;
    }

    while (argc-- > 0)
    {
        Execute(*argv++);
    }

out:
    nn::socket::Finalize();

    return;
}
