﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nnt.h>

#include <nn/socket/socket_Api.h>
#include <nn/tsc.h>
#include <nn/tsc/tsc_DebugLibrary.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/init.h>

#ifdef NN_NETWORK_USE_BSDSOCKET
// Debug flag -------------------------------------------------------------------------------------
#define ENABLE_MYDBG_REPORT
// ------------------------------------------------------------------------------------------------
#ifdef ENABLE_MYDBG_REPORT
#define MYDBG_REPORT(format, ...) NN_SDK_LOG(format, ##__VA_ARGS__)
#define MYDBG_ERROR_REPORT(result, format, ...)                                                    \
    NN_SDK_LOG(format " (Module:%d Desc:%d)\n",                                                    \
               ##__VA_ARGS__, result.GetModule(), result.GetDescription());
#else
#define MYDBG_REPORT(format, ...)
#define MYDBG_ERROR_REPORT(result, format, ...)
#endif

extern "C"
void nndiagStartup()
{
}

namespace {

    const uint32_t       g_MemoryPoolSize = (512 * 1024);
    nn::lmem::HeapHandle g_HeapHandle;
    uint8_t              g_HeapMemoryPool[g_MemoryPoolSize] NN_ALIGNAS(nn::os::ThreadStackAlignment);

    void* AllocAligned(size_t size, size_t align)
    {
        return nn::lmem::AllocateFromExpHeap(g_HeapHandle, size, align);
    }

    void FreeAligned(void* p)
    {
        nn::lmem::FreeToExpHeap(g_HeapHandle, p);
    }
}

namespace nnt { namespace tsc {

namespace
{
    const uint32_t g_MtuValue                 = 1500;
    const uint32_t g_ApplyTimeoutMsec         = 1000 * 10;
    const char*    g_InterfaceName            = "usb0";

    const char*    g_ipStringInterfaceAddress = "192.168.11.202";
    const char*    g_ipStringSubnetMask       = "255.255.255.0";
    const char*    g_ipStringDefaultGateway   = "192.168.11.1";
    const char*    g_ipStringPreferredDns     = "192.168.11.1";
    const char*    g_ipStringAlternativeDns   = "192.168.11.2";

    bool                       g_IsConfigurationDone = false;
    nn::tsc::Ipv4ConfigContext g_IpConfigCtx(g_InterfaceName, nn::os::EventClearMode_AutoClear);



    class SimpleLocalTimer
    {
        typedef void (*CallbackFunction)(); // Callback function when timer is done
        NN_ALIGNAS(nn::os::StackRegionAlignment) char m_TimerThreadStack[4096];
        nn::os::ThreadType                            m_TimerThread;
        nn::os::TimerEventType                        m_TimerEvent;
        nn::os::EventType                             m_TimerDone;
        bool                                          m_IsTimerRunning;
        int                                           m_TimeoutMsec;
        CallbackFunction                              m_Callback;

        static void ExecuteThread(void* args)
        {
            reinterpret_cast<SimpleLocalTimer*>(args)->TimerThreadFunction(args);
        }
        int CreateTimerThread()
        {
            nn::Result result = nn::ResultSuccess();
            result = nn::os::CreateThread(&m_TimerThread, ExecuteThread, this, m_TimerThreadStack,
                                          4096,
                                          nn::os::GetThreadPriority(nn::os::GetCurrentThread()) - 1);
            if(result.IsFailure())
            {
                return -1;
            }
            nn::TimeSpan maxWaitTime = nn::TimeSpan::FromMilliSeconds(m_TimeoutMsec);
            nn::os::InitializeTimerEvent(&m_TimerEvent, nn::os::EventClearMode_AutoClear);
            nn::os::StartOneShotTimerEvent(&m_TimerEvent, maxWaitTime);
            nn::os::StartThread(&m_TimerThread);
            m_IsTimerRunning = true;
            return 0;
        }

        void TimerThreadFunction(void* pParam)
        {
            #define pThis static_cast<SimpleLocalTimer*>(pParam)
            nn::os::WaitTimerEvent(&pThis->m_TimerEvent);
            if(m_Callback)
            {
                m_Callback();
            }
            nn::os::StopTimerEvent(&m_TimerEvent);
            nn::os::FinalizeTimerEvent(&m_TimerEvent);
            m_IsTimerRunning = false;
            nn::os::SignalEvent(&m_TimerDone);
        }

    public:
        SimpleLocalTimer()
        {
            m_IsTimerRunning = false;
            m_TimeoutMsec    = 0;
            nn::os::InitializeEvent(&m_TimerDone, false, nn::os::EventClearMode_AutoClear);
        }
        ~SimpleLocalTimer()
        {
            if(m_IsTimerRunning)
            {
                Cancel();
                nn::os::WaitEvent(&m_TimerDone);
            }

            nn::os::FinalizeEvent(&m_TimerDone);
            nn::os::WaitThread(&m_TimerThread);
            nn::os::DestroyThread(&m_TimerThread);
        }
        void Cancel()
        {
            if(m_IsTimerRunning)
            {
                nn::os::SignalTimerEvent(&m_TimerEvent);
            }
        }
        int Start(uint32_t msecTimeout, CallbackFunction callback)
        {
            if(m_IsTimerRunning)
            {
                return -1;
            }
            m_TimeoutMsec = msecTimeout;
            m_Callback    = callback;
            return CreateTimerThread();
        }
    };

    void LocalTimerCallback()
    {
        MYDBG_REPORT("Timer done.\n");
        if(g_IsConfigurationDone != true)
        {
            nn::Result result = g_IpConfigCtx.CancelApplyConfig();
            MYDBG_ERROR_REPORT(result, "CancelApplyConfig()");
            NN_UNUSED(result);
        }
    }

    void SetIpv4Config(nn::tsc::Ipv4ConfigMethod method, nn::tsc::Ipv4ConfigContext* pOutConfig)
    {
        nn::tsc::Ipv4Config ipConfig;
        memset(&ipConfig, 0x00, sizeof(ipConfig));

        switch (method)
        {
        case nn::tsc::Ipv4ConfigMethod_Dhcp:
        case nn::tsc::Ipv4ConfigMethod_DhcpHybrid:
            {
                ipConfig.method = nn::tsc::Ipv4ConfigMethod_Dhcp;
            }
            break;
        case nn::tsc::Ipv4ConfigMethod_Static:
            {
                ipConfig.method = nn::tsc::Ipv4ConfigMethod_Static;
                nn::socket::InetAton(g_ipStringInterfaceAddress, reinterpret_cast<nn::socket::InAddr *>(&ipConfig.interfaceAddress));
                nn::socket::InetAton(g_ipStringSubnetMask,       reinterpret_cast<nn::socket::InAddr *>(&ipConfig.subnetMask));
                nn::socket::InetAton(g_ipStringDefaultGateway,   reinterpret_cast<nn::socket::InAddr *>(&ipConfig.defaultGateway));
                nn::socket::InetAton(g_ipStringPreferredDns,     reinterpret_cast<nn::socket::InAddr *>(&ipConfig.preferredDns));
                nn::socket::InetAton(g_ipStringAlternativeDns,   reinterpret_cast<nn::socket::InAddr *>(&ipConfig.alternativeDns));
            }
            break;
        default:
            FAIL();
            break;
        }

        ASSERT_TRUE(pOutConfig->SetMtu(g_MtuValue).IsSuccess());
        ASSERT_TRUE(pOutConfig->SetConfig(&ipConfig).IsSuccess());
    }

    void DumpActiveIpInformation()
    {
        nn::tsc::ActiveConfigContext activeConfig(g_InterfaceName);
        nn::tsc::Ipv4Config          ipInfo;
        do
        {
            ASSERT_TRUE(activeConfig.GetInterfaceAddress(&ipInfo.interfaceAddress).IsSuccess());
            ASSERT_TRUE(activeConfig.GetSubnetMask(&ipInfo.subnetMask).IsSuccess());
            ASSERT_TRUE(activeConfig.GetDefaultGateway(&ipInfo.defaultGateway).IsSuccess());
            ASSERT_TRUE(activeConfig.GetPreferredDns(&ipInfo.preferredDns).IsSuccess());
            ASSERT_TRUE(activeConfig.GetAlternativeDns(&ipInfo.alternativeDns).IsSuccess());
            nn::tsc::debug::DumpIpAddressInfo(&ipInfo);
            return;
        } while (0);

        NN_SDK_LOG("[socket]Failed to obtain current address information\n");
    }
}

//-----------------------------------------------------------------------------

TEST(BsdsocketCfgOverTsc, DhcpTest)
{
    nn::Result                 result = nn::ResultSuccess();
    SimpleLocalTimer*          pTimerObj = nullptr;

    result = nn::tsc::Initialize();
    MYDBG_REPORT("Initialized TSC library.\n");

    ASSERT_TRUE(result.IsSuccess());
    ASSERT_TRUE(g_IpConfigCtx.SetInterfaceName(g_InterfaceName).IsSuccess());
    SetIpv4Config(nn::tsc::Ipv4ConfigMethod_Dhcp, &g_IpConfigCtx);

    pTimerObj = (SimpleLocalTimer *)AllocAligned( sizeof(SimpleLocalTimer), nn::os::ThreadStackAlignment);
    if(pTimerObj == nullptr)
    {
        FAIL();
    }
    pTimerObj = new (pTimerObj) SimpleLocalTimer();

    MYDBG_REPORT("Starting configuration.\n");
    pTimerObj->Start(g_ApplyTimeoutMsec, LocalTimerCallback);
    result = g_IpConfigCtx.ApplyConfig(nn::tsc::IpApplyModeMask_None);
    ASSERT_TRUE(result.IsSuccess());

    MYDBG_REPORT("Waiting for ApplyConfig process gets done...\n");
    g_IpConfigCtx.GetEventPointer()->Wait();
    result = g_IpConfigCtx.GetApplyResult();
    if(result.IsFailure())
    {
        MYDBG_ERROR_REPORT(result, "GetApplyResult indicates error");
    }
    else
    {
        DumpActiveIpInformation();
    }

    result = g_IpConfigCtx.CancelApplyConfig();
    MYDBG_ERROR_REPORT(result, "test- CancelApplyConfig()");

    MYDBG_REPORT("Wiat for 10 seconds...\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    ASSERT_TRUE(g_IpConfigCtx.NotifyInterfaceDown().IsSuccess());

    g_IsConfigurationDone = true;
    pTimerObj->~SimpleLocalTimer();
    FreeAligned(pTimerObj);
    result = nn::tsc::Finalize();
    ASSERT_TRUE(result.IsSuccess());

    result.IsSuccess() ? SUCCEED() : ADD_FAILURE();
}

TEST(BsdsocketCfgOverTscWithSystemEvent, DhcpTest)
{
    nn::Result                 result = nn::ResultSuccess();
    SimpleLocalTimer*          pTimerObj = nullptr;
    nn::os::SystemEvent*       pSystemEvent = nullptr;

    result = nn::tsc::Initialize();
    MYDBG_REPORT("Initialized TSC library.\n");

    ASSERT_TRUE(result.IsSuccess());
    ASSERT_TRUE(g_IpConfigCtx.SetInterfaceName(g_InterfaceName).IsSuccess());
    SetIpv4Config(nn::tsc::Ipv4ConfigMethod_Dhcp, &g_IpConfigCtx);

    pTimerObj = (SimpleLocalTimer *)AllocAligned( sizeof(SimpleLocalTimer), nn::os::ThreadStackAlignment);
    if(pTimerObj == nullptr)
    {
        FAIL();
    }
    pTimerObj = new (pTimerObj) SimpleLocalTimer();


    // Get system event pointer
    pSystemEvent = g_IpConfigCtx.GetSystemEventPointer();

    MYDBG_REPORT("Starting configuration.\n");
    pTimerObj->Start(g_ApplyTimeoutMsec, LocalTimerCallback);
    result = g_IpConfigCtx.ApplyConfig(nn::tsc::IpApplyModeMask_None);
    ASSERT_TRUE(result.IsSuccess());

    MYDBG_REPORT("Waiting for ApplyConfig process gets done...\n");
    g_IpConfigCtx.GetEventPointer()->Wait();
    result = g_IpConfigCtx.GetApplyResult();
    if(result.IsFailure())
    {
        MYDBG_ERROR_REPORT(result, "GetApplyResult indicates error");
    }
    else
    {
        DumpActiveIpInformation();
    }

    int retry = 0;
    do
    {
        // Get the interface statistics
        nn::tsc::ActiveConfigContext::InterfaceStatistics stats;
        nn::tsc::ActiveConfigContext                      activeConfig(g_InterfaceName);

        memset(&stats, 0x00, sizeof(stats));
        result = activeConfig.GetInterfaceStatistics(&stats);
        if(result.IsFailure())
        {
            MYDBG_ERROR_REPORT(result, "GetInterfaceStatistics indicates error");
        }
        else
        {
            NN_LOG("Interface statistics - Status:%d renewedLeaseCount:%d reboundLeaseCount:%d\n",
                stats.interfaceStatus.GetDescription(),
                stats.dhcp.renewedLeaseCount,
                stats.dhcp.reboundLeaseCount);
        }

        pSystemEvent->Wait();
        NN_LOG("System event signaled!\n");
    } while (retry++ < 5);

    ASSERT_TRUE(g_IpConfigCtx.NotifyInterfaceDown().IsSuccess());

    g_IsConfigurationDone = true;
    pTimerObj->~SimpleLocalTimer();
    FreeAligned(pTimerObj);
    result = nn::tsc::Finalize();
    ASSERT_TRUE(result.IsSuccess());

    result.IsSuccess() ? SUCCEED() : ADD_FAILURE();
}

//-----------------------------------------------------------------------------
TEST(BsdsocketCfgOverTsc, StaticTest)
{
    nn::Result                 result = nn::ResultSuccess();
    SimpleLocalTimer*          pTimerObj = nullptr;

    result = nn::tsc::Initialize();

    ASSERT_TRUE(result.IsSuccess());
    ASSERT_TRUE(g_IpConfigCtx.SetInterfaceName(g_InterfaceName).IsSuccess());
    SetIpv4Config(nn::tsc::Ipv4ConfigMethod_Static, &g_IpConfigCtx);

    pTimerObj = (SimpleLocalTimer *)AllocAligned(sizeof(SimpleLocalTimer), nn::os::ThreadStackAlignment);
    if (pTimerObj == nullptr)
    {
        FAIL();
    }
    pTimerObj = new (pTimerObj) SimpleLocalTimer();


    pTimerObj->Start(g_ApplyTimeoutMsec, LocalTimerCallback);
    result = g_IpConfigCtx.ApplyConfig(nn::tsc::IpApplyModeMask_None);
    ASSERT_TRUE(result.IsSuccess());

    g_IpConfigCtx.GetEventPointer()->Wait();
    result = g_IpConfigCtx.GetApplyResult();
    if(result.IsFailure())
    {
        MYDBG_ERROR_REPORT(result, "GetApplyResult indicates error");
    }
    else
    {
        DumpActiveIpInformation();
    }

    result = g_IpConfigCtx.CancelApplyConfig();
    MYDBG_ERROR_REPORT(result, "test- CancelApplyConfig()");

    MYDBG_REPORT("Wiat for 10 seconds...\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    ASSERT_TRUE(g_IpConfigCtx.NotifyInterfaceDown().IsSuccess());

    g_IsConfigurationDone = true;
    pTimerObj->~SimpleLocalTimer();
    FreeAligned(pTimerObj);
    result = nn::tsc::Finalize();
    ASSERT_TRUE(result.IsSuccess());

    result.IsSuccess() ? SUCCEED() : ADD_FAILURE();
}

}} // namespace nnt { namespace tsc {
#endif //NN_NETWORK_USE_BSDSOCKET

extern "C"
void nninitStartup()
{
    const size_t MemoryHeapSize = 8 * 1024 * 1024;
    nn::os::SetMemoryHeapSize(MemoryHeapSize);

    uintptr_t address;
    nn::os::AllocateMemoryBlock(&address, MemoryHeapSize);

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), MemoryHeapSize);

    g_HeapHandle =
        nn::lmem::CreateExpHeap(
            g_HeapMemoryPool,
            sizeof(g_HeapMemoryPool),
            nn::lmem::CreationOption_NoOption
        );
}

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C"
void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char **argv = nnt::GetHostArgv();

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);

    int result = RUN_ALL_TESTS();

    // テスト終了
    NN_LOG("\n=== End Test of bsdsocketCfgOverTcp test\n");

    nnt::Exit(result);
}
