﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testNet_P2P.h"
#include "natf/Utils/InitApis.h"

namespace nnt { namespace net { namespace p2p {

uint32_t PacketRateTest::g_pPeerIndeces[MaxNodes]; // Maps 'peer' index to 'node' index.

NETTEST_DECLARE_THREAD_FUNC(PrintThread, pArg);

char PacketRateTest::g_pPrintBuffer[PrintBufferLen];
BufferedLogger PacketRateTest::g_Logger(g_pPrintBuffer, PrintBufferLen);

char PacketRateTest::g_pCsvBuffer[PrintBufferLen];
BufferedLogger PacketRateTest::g_CsvLogger(g_pCsvBuffer, PrintBufferLen);

NN_ALIGNAS(4096) unsigned char PacketRateTest::g_pRecvThreadStack[StackLen];
NN_ALIGNAS(4096) unsigned char PacketRateTest::g_pSendThreadStack[StackLen];
NN_ALIGNAS(4096) unsigned char PacketRateTest::g_pPrintThreadStack[StackLen];

NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

Params PacketRateTest::g_Params; // Parsed params
NodeInfo PacketRateTest::g_pNodes[MaxNodes]; // All nodes including self.
GlobalStats PacketRateTest::g_Stats;
NetMessage PacketRateTest::g_RecvNetMessage;

ssize_t (*PacketRateTest::g_pRecvFromFn)(int socket, NetMessage& message, nn::socket::SockAddrIn* pAddr, nn::socket::SockLenT* pAddrLen, NetTest::Tick& recvdTick) NN_NOEXCEPT = PacketRateTest::NormalRecvFrom;
ssize_t (*PacketRateTest::g_pSendToFn)(int socket, NetMessage& message, NodeInfo& peer, uint32_t packetSize) NN_NOEXCEPT = PacketRateTest::NormalSendTo;

bool PacketRateTest::ProcessPacket(const NetMessage& packet, int recvSocket, const NetTest::Tick& recvdTick) NN_NOEXCEPT
{
    NodeInfo& peer = g_pNodes[packet.GetNodeIndex()];
    uint32_t seqNum = packet.GetSequenceNum();

    if( packet.GetFlags() != NetHeaderFlags::Ack )
    {
        ++peer.packetStats.packetsRecved;
        ++peer.packetStats.frameRecvd;

        // Is sequence number in current frame range?
        if( seqNum >= peer.baseSequenceNumber )
        {
            ++peer.framePacketCount;

            // Keep track of highest SeqNum.
            if( seqNum > peer.highSeqNum )
            {
                peer.highSeqNum = seqNum;
            }
        }
        else
        {
            ++peer.packetStats.packetsOutOfRange;
        }
    }

    if( packet.GetFlags() == NetHeaderFlags::Rtt )
    {
        NetMessage ackMessage;
        ackMessage.SetHeader(g_Params.nodeIndex, seqNum, NetHeaderFlags::Ack, sizeof(NetHeader));
        size_t rval = PacketRateTest::NormalSendTo(recvSocket, ackMessage, peer, sizeof(NetHeader));
        if( rval != sizeof(NetHeader) )
        {
            nn::socket::Errno errNo = NetTest::GetLastError();
            if( errNo == nn::socket::Errno::EAgain || errNo == nn::socket::Errno::EWouldBlock || errNo == nn::socket::Errno::ENetUnreach )
            {
                return true;
            }

            NN_NETTEST_LOG("\n Sendto failed while trying to reply to an RTT packet. rval: %d, errno: %d\n\n", (int)rval, errNo);
            return false;
        }
    }
    else if( packet.GetFlags() == NetHeaderFlags::Ack )
    {
        if( peer.packetStats.rttSeqNum == seqNum )
        {
            ++peer.packetStats.ackRecvCount;
            peer.packetStats.totalRttTicks += recvdTick - peer.packetStats.rttStart;
        }
    }

    return true;
}

int PacketRateTest::InitRecvSocket() NN_NOEXCEPT
{
    nn::socket::SockAddrIn localAddr;
    int recvSocket = NetTest::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    if( recvSocket < 0 )
    {
        NN_NETTEST_LOG("\n * Failed to create recv socket. rval: %d errno: %d\n\n", recvSocket, NetTest::GetLastError());
        m_RecvThreadData.error = ThreadError_Unknown;
        return -1;
    }

    int rval = NetTest::SetSockOpt(recvSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &SockRecvBufLen, sizeof(SockRecvBufLen));
    if( rval != 0 )
    {
        NN_NETTEST_LOG("WARNING: setsockopt(nn::socket::Option::So_RcvBuf) failed. rval: %d, errno: %d\n\n", rval, NetTest::GetLastError());
        NetTest::SleepMs(3000);
    }

    memset(&localAddr, 0, sizeof(localAddr));
    localAddr.sin_family = nn::socket::Family::Af_Inet;
    localAddr.sin_port = NetTest::Htons(g_Params.pAddrs[g_Params.nodeIndex].port);
    localAddr.sin_addr.S_addr = nn::socket::InAddr_Any;

    rval = NetTest::Bind(recvSocket, (nn::socket::SockAddr*)&localAddr, sizeof(localAddr));
    if( rval != 0 )
    {
        NN_NETTEST_LOG("\n Failed to bind to local port: %d, rval: %d, errno: %d\n\n", (int)g_Params.pAddrs[g_Params.nodeIndex].port, rval, NetTest::GetLastError());
        m_RecvThreadData.error = ThreadError_Unknown;
        return -1;
    }

    // Set socket to non-blocking
    rval = NetTest::Fcntl(recvSocket, static_cast<int>(nn::socket::FcntlCommand::F_SetFl), nn::socket::FcntlFlag::O_NonBlock);
    if( rval != 0 )
    {
        NN_NETTEST_LOG("\n Failed to set socket to non-blocking. rval: %d\n\n", rval);
        m_RecvThreadData.error = ThreadError_Unknown;
        return -1;
    }

    return recvSocket;
}

RecvError PacketRateTest::CheckRecvError(int recvSocket, ssize_t recvLen, NetTest::Tick startFrame) NN_NOEXCEPT
{
    if( recvLen <= 0 )
    {
        // If function would have blocked, then wait on read ready.
        if( NetTest::GetLastError() == nn::socket::Errno::EAgain || NetTest::GetLastError() == nn::socket::Errno::EWouldBlock )
        {
            NetTest::Tick now = NetTest::GetTick();
            float recvDurationMicroSec = (float)NetTest::TickToTime(now - startFrame).GetMicroSeconds();
            float diff = g_Params.frameIntervalMs - (recvDurationMicroSec / 1000.0f);
            if(  diff >= 1.0f + RecvFrameReserveMs )
            {
                nn::socket::PollFd pollFD = {};
                pollFD.fd      = recvSocket;
                pollFD.events  = nn::socket::PollEvent::PollRdNorm;
                pollFD.revents  = nn::socket::PollEvent::PollNone;

                int timeoutMs = (int)diff;

                int rval = NetTest::Poll(&pollFD, 1, timeoutMs);
                if( rval == 0 )
                {
                    return RecvError_NotReady;
                }
                else if( rval < 0 )
                {
                    NN_NETTEST_LOG("\n ERROR: poll() failed. rval: %d, errno: %d\n\n", rval, NetTest::GetLastError());
                    m_RecvThreadData.error = ThreadError_Recv;
                    return RecvError_Error;
                }
                else
                {
                    return RecvError_Ready;
                }
            }

            return RecvError_NotReady;
        }

        NN_NETTEST_LOG("\n Recvfrom failed. rval: %d, errno: %d\n\n", (int)recvLen, NetTest::GetLastError());
        m_RecvThreadData.error = ThreadError_Recv;
        return RecvError_Error;
    }

    return RecvError_None;
}

// Receive Thread
NETTEST_DECLARE_THREAD_FUNC(RecvThread, pArg)
{
    int recvSocket;
    nn::socket::SockLenT addrLen;
    nn::socket::SockAddrIn remoteAddr;
    NetTest::Tick recvdTick;
    bool processSuccess = true;

    PacketRateTest* pTest = (PacketRateTest*)pArg;
    ThreadData& params = pTest->m_RecvThreadData;

    recvSocket = pTest->InitRecvSocket();
    if( recvSocket < 0 )
    {
        goto cleanup;
    }

    // Frame Loop
    while( !params.doShutdown )
    {
        NetTest::Tick startFrame;
        NetTest::Time waitFrameTimeout = NetTest::TimeFromMs(FrameTimeoutMS);

        NetTest::SignalEvent(&params.isReady);

        for(unsigned iPeer = 0; iPeer < pTest->m_PeerCount; ++iPeer)
        {
            NodeInfo& peer = pTest->GetPeerByHandle(iPeer);
            peer.baseSequenceNumber = peer.highSeqNum + 1;
            peer.framePacketCount = 0;
            peer.packetStats.frameRecvd = 0;
        }

        bool isSuccess = NetTest::TimedWaitEvent(&params.doFrame, waitFrameTimeout);
        startFrame = NetTest::GetTick(); // Get start time as soon as possible!

        // If waiting for event timed out,
        //  continue and check for shutdown command.
        if( !isSuccess )
        {
            continue;
        }

        NetTest::ClearEvent(&params.doFrame);

        do // Receive loop
        {
            addrLen = sizeof(remoteAddr);
            ssize_t recvLen = PacketRateTest::g_pRecvFromFn(recvSocket, PacketRateTest::g_RecvNetMessage, &remoteAddr, &addrLen, recvdTick);

            RecvError recvError = pTest->CheckRecvError(recvSocket, recvLen, startFrame);
            if( recvError != RecvError_None )
            {
                if( recvError == RecvError_Ready )
                {
                    continue;
                }
                else if( recvError == RecvError_NotReady )
                {
                    goto check_frame_time;
                }
                else if( recvError == RecvError_Error )
                {
                    goto cleanup;
                }
            }

            // Make sure header and packet are in expected format and from expected address.
            if( !PacketRateTest::g_RecvNetMessage.ValidateHeader((uint32_t)recvLen, remoteAddr) )
            {
                params.error = ThreadError_Recv;
                goto cleanup;
            }

            // Verify checksum shows no errors.
            if( PacketRateTest::GetParams().doVerifyData && !PacketRateTest::g_RecvNetMessage.VerifyCheckSum((uint32_t)recvLen) )
            {
                if( PacketRateTest::GetParams().doTermOnBadData )
                {
                    NN_NETTEST_LOG("\nERROR: Checksum failed!\n\n");
                    params.error = ThreadError_Recv;
                    goto cleanup;
                }
                else
                {
                    ++PacketRateTest::g_pNodes[PacketRateTest::g_RecvNetMessage.GetNodeIndex()].packetStats.packetsBadData;
                }
            }

            processSuccess = pTest->ProcessPacket(PacketRateTest::g_RecvNetMessage, recvSocket, recvdTick);
            if( !processSuccess )
            {
                params.error = ThreadError_Unknown;
                goto cleanup;
            }

check_frame_time:
            float recvDuration = NetTest::TickToTime(NetTest::GetTick() - startFrame).GetMicroSeconds() / 1000.0f;
            if( recvDuration > PacketRateTest::GetParams().frameIntervalMs - RecvFrameReserveMs )
            {
                break;
            }

            NetTest::YieldThread();
        } while( processSuccess );

        for(unsigned iNode = 0; iNode < pTest->m_PeerCount; ++iNode)
        {
            NodeInfo& peer = pTest->GetPeerByHandle(iNode);
            peer.packetStats.packetsDropped += ((peer.highSeqNum + 1) - peer.baseSequenceNumber) - peer.framePacketCount;
        }
    }

cleanup:
    if( recvSocket >= 0 )
    {
        NetTest::Close(recvSocket);
    }

    NetTest::SignalEvent(&params.isReady);
    NetTest::SignalEvent(&params.isExited);
    NETTEST_THREAD_RETURN;
}

// Print Thread
NETTEST_DECLARE_THREAD_FUNC(PrintThread, pArg)
{
    PrintParam* pParams = (PrintParam*)pArg;

    while( NN_STATIC_CONDITION(true) )
    {
        if( pParams->doShutdown )
        {
            PacketRateTest::g_Logger.Flush(0);
            goto out;
        }
        else
        {
            PacketRateTest::g_Logger.Flush(32);
        }

        NetTest::SleepMs(16);
    }

out:
    NetTest::SignalEvent(&pParams->isExitedEvent);
    NETTEST_THREAD_RETURN;
}

bool PacketRateTest::SendRttToPeers(PacketRateTest* pTest, ThreadData& params, int sendSocket, uint32_t sequenceNumber, uint32_t sentPackets) NN_NOEXCEPT
{
    NetMessage netMessage;
    netMessage.SetHeader(PacketRateTest::GetParams().nodeIndex, sequenceNumber + sentPackets, NetHeaderFlags::Rtt, PacketRateTest::GetParams().packetSize);

    for(unsigned iPeer = 0; iPeer < pTest->m_PeerCount; ++iPeer)
    {
        NodeInfo& peer = pTest->GetPeerByHandle(iPeer);

        ssize_t rval = PacketRateTest::g_pSendToFn(sendSocket, netMessage, peer, PacketRateTest::GetParams().packetSize);
        peer.packetStats.rttStart = NetTest::GetTick();
        peer.packetStats.rttSeqNum = sequenceNumber + sentPackets;
        if( rval <= 0 )
        {
            nn::socket::Errno errNo = NetTest::GetLastError();
            if( errNo == nn::socket::Errno::EAgain || errNo == nn::socket::Errno::EWouldBlock || errNo == nn::socket::Errno::ENetUnreach )
            {
                continue;
            }

            NN_NETTEST_LOG("\n sendto failed. rval: %d, errno: %d\n\n", (int)rval, errNo);
            params.error = ThreadError_Recv;
            return false;
        }

        ++peer.packetStats.frameSent;
        ++peer.packetStats.packetsSent;
        NetTest::YieldThread();
    }

    return true;
}

bool PacketRateTest::SendToPeers(PacketRateTest* pTest, ThreadData& params, int sendSocket, uint32_t sequenceNumber, uint32_t sentPackets) NN_NOEXCEPT
{
    NetMessage netMessage;
    netMessage.SetHeader(PacketRateTest::GetParams().nodeIndex, sequenceNumber + sentPackets, NetHeaderFlags::None, PacketRateTest::GetParams().packetSize);

    for(unsigned iPeer = 0; iPeer < pTest->m_PeerCount; ++iPeer)
    {
        NodeInfo& peer = pTest->GetPeerByHandle(iPeer);

        ssize_t rval = PacketRateTest::g_pSendToFn(sendSocket, netMessage, peer, PacketRateTest::GetParams().packetSize);
        if( rval <= 0 )
        {
            nn::socket::Errno errNo = NetTest::GetLastError();
            if( errNo == nn::socket::Errno::EAgain || errNo == nn::socket::Errno::EWouldBlock || errNo == nn::socket::Errno::ENetUnreach )
            {
                continue;
            }

            NN_NETTEST_LOG("\n sendto failed. rval: %d, errno: %d\n\n", (int)rval, NetTest::GetLastError());
            params.error = ThreadError_Recv;
            return false;
        }

        ++peer.packetStats.frameSent;
        ++peer.packetStats.packetsSent;
        NetTest::YieldThread();
    }

    return true;
}

// Send Thread
NETTEST_DECLARE_THREAD_FUNC(SendThread, pArg)
{
    int rval;
    int sendSocket;
    bool isSuccess;
    uint32_t sequenceNumber = 1;
    PacketRateTest* pTest = (PacketRateTest*)pArg;
    ThreadData& params = pTest->m_SendThreadData;
    bool (*pSendPeers)(PacketRateTest* pTest, ThreadData& params, int sendSocket, uint32_t sequenceNumber, uint32_t sentPackets) = PacketRateTest::SendToPeers;

    sendSocket = NetTest::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    if( sendSocket < 0 )
    {
        NN_NETTEST_LOG("\n * Failed to create recv socket. rval: %d errno: %d\n\n", sendSocket, NetTest::GetLastError());
        params.error = ThreadError_Unknown;
        goto cleanup;
    }

    // Set socket to non-blocking
    rval = NetTest::Fcntl(sendSocket, static_cast<int>(nn::socket::FcntlCommand::F_SetFl), nn::socket::FcntlFlag::O_NonBlock);
    if( rval != 0 )
    {
        NN_NETTEST_LOG("\n Failed to set socket to non-blocking. rval: %d\n\n", rval);
        params.error = ThreadError_Unknown;
        goto cleanup;
    }

    // Frame loop
    while( !params.doShutdown )
    {
        NetTest::Tick startFrame;
        NetTest::Time waitFrameTimeout = NetTest::TimeFromMs(FrameTimeoutMS);
        uint32_t sentPackets = 0;

        NetTest::SignalEvent(&params.isReady);

        isSuccess = NetTest::TimedWaitEvent(&params.doFrame, waitFrameTimeout);

        startFrame = NetTest::GetTick();
        if( !isSuccess )
        {
            continue;
        }

        NetTest::ClearEvent(&params.doFrame);

        for(unsigned iPeer = 0; iPeer < pTest->m_PeerCount; ++iPeer)
        {
            NodeInfo& peer = pTest->GetPeerByHandle(iPeer);
            peer.packetStats.frameSent = 0;
        }

        ++PacketRateTest::g_Stats.rttFrameCounter;
        if( PacketRateTest::g_Stats.rttFrameCounter > PacketRateTest::g_Params.rttFrameInterval )
        {
            PacketRateTest::g_Stats.rttFrameCounter = 1;
        }

        if( PacketRateTest::g_Stats.rttFrameCounter == PacketRateTest::g_Params.rttFrameInterval )
        {
            pSendPeers  = PacketRateTest::SendRttToPeers;
        }

        do
        {
            if( !pSendPeers(pTest, params, sendSocket, sequenceNumber, sentPackets) )
            {
                goto cleanup;
            }

            pSendPeers = PacketRateTest::SendToPeers;

            ++sentPackets;
        } while( (sentPackets < PacketRateTest::GetParams().packetsPerFrame) &&
                 ((NetTest::TickToTime(NetTest::GetTick() - startFrame).GetMicroSeconds() / 1000.0f) < PacketRateTest::GetParams().frameIntervalMs) );

        sequenceNumber += sentPackets;
    }

cleanup:
    if( sendSocket >= 0 )
    {
        NetTest::Close(sendSocket);
    }

    NetTest::SignalEvent(&params.isExited);
    NETTEST_THREAD_RETURN;
}

bool PacketRateTest::Init() NN_NOEXCEPT
{
    NetTest::Time initThreadTimeout = NetTest::TimeFromSec(InitThreadTimeoutSec);

    m_PeerCount = 0;
    for( unsigned iPeer = 0; iPeer < g_Params.nodeCount; ++iPeer )
    {
        if( g_Params.nodeIndex != iPeer )
        {
            g_pPeerIndeces[m_PeerCount] = iPeer;
            ++m_PeerCount;
        }
    }

    for(uint32_t i = 0; i < sizeof(g_RecvNetMessage.payload) / 4; ++i)
    {
        ((uint32_t*)g_RecvNetMessage.payload)[i] = NetTest::GetRandom();
    }

    memset(g_pNodes, 0, sizeof(g_pNodes));
    for(unsigned iPeer = 0; iPeer < m_PeerCount; ++iPeer)
    {
        uint32_t nodeIndex = g_pPeerIndeces[iPeer];
        NodeInfo& peer = GetPeerByHandle(iPeer);
        peer.nodeIndex = nodeIndex;
        peer.baseSequenceNumber = 1;
        peer.highSeqNum = 0;
        peer.addr.sin_family = nn::socket::Family::Af_Inet;
        peer.addr.sin_port = NetTest::Htons(g_Params.pAddrs[nodeIndex].port);

        int rval = NetTest::InetAddrPton(nn::socket::Family::Af_Inet, g_Params.pAddrs[nodeIndex].pIp, &peer.addr.sin_addr);
        if( rval < 0 )
        {
            NN_NETTEST_LOG("\n Failed to convert string ip. string: %s rval: %d\n\n", g_Params.pAddrs[nodeIndex].pIp, rval);
            return false;
        }
    }

    bool isSuccess = NetTest::CreateThread(&m_RecvThreadData.hThread, RecvThread, this, g_pRecvThreadStack, StackLen);
    if( !isSuccess )
    {
        NN_NETTEST_LOG("\n Failed to create recv thread!\n\n");
        return false;
    }

    isSuccess = NetTest::CreateThread(&m_SendThreadData.hThread, SendThread, this, g_pSendThreadStack, StackLen);
    if( !isSuccess )
    {
        NN_NETTEST_LOG("\n Failed to create send thread!\n\n");
        return false;
    }

    if( !g_Params.noReport )
    {
        isSuccess = NetTest::CreateThread(&m_PrintParam.thread, PrintThread, &m_PrintParam, g_pPrintThreadStack, StackLen);
        if( !isSuccess )
        {
            NN_NETTEST_LOG("\n Failed to create print thread!\n\n");
            return false;
        }

        NetTest::StartThread(&m_PrintParam.thread);
    }

    NetTest::StartThread(&m_RecvThreadData.hThread);
    NetTest::StartThread(&m_SendThreadData.hThread);

    isSuccess = NetTest::TimedWaitEvent(&m_RecvThreadData.isReady, initThreadTimeout);
    if( !isSuccess )
    {
        NN_NETTEST_LOG("\n Recv thread timed out during init phase! ThreadError: %d\n\n", m_RecvThreadData.error);
        return false;
    }

    isSuccess = NetTest::TimedWaitEvent(&m_SendThreadData.isReady, initThreadTimeout);
    if( !isSuccess )
    {
        NN_NETTEST_LOG("\n Send thread timed out during init phase! ThreadError: %d\n\n", m_SendThreadData.error);
        return false;
    }

    if( g_Params.reportCsv )
    {
        CSV_LOG("Node Index,Node count,Packets per frame,Frame duration(MS),Packet length,Sample duration(MS)\n");
        CSV_LOG("%d,%d,%d,%d.%d,%d,%d\n", (int)g_Params.nodeIndex, (int)g_Params.nodeCount, (int)g_Params.packetsPerFrame, (int)g_Params.frameIntervalMs, int(g_Params.frameIntervalMs * 100.0f) % 100, (int)g_Params.packetSize, (int)g_Params.reportIntervalMs);
        CSV_LOG("Node Index,Recv Rate(PPS),Send Rate(PPS),Recv Drop Rate(%%),Average Rtt(MicroSec)\n");
    }

    return true;
}

void PacketRateTest::CleanUp() NN_NOEXCEPT
{
    NetTest::Time threadExitTimeout = NetTest::TimeFromSec(ThreadExitTimeoutSec);

    NN_NETTEST_LOG(" Sending shutdown commands...\n");

    m_SendThreadData.doShutdown = true;
    m_RecvThreadData.doShutdown = true;

    NetTest::SignalEvent(&m_SendThreadData.doFrame);
    NetTest::SignalEvent(&m_RecvThreadData.doFrame);

    NN_NETTEST_LOG(" Waiting for send thread...\n");
    bool isSuccess = NetTest::TimedWaitEvent(&m_SendThreadData.isExited, threadExitTimeout);
    if( isSuccess )
    {
        NN_NETTEST_LOG("Exit signaled from send thread. Joining thread...");
        NetTest::WaitThread(&m_SendThreadData.hThread);
        NN_NETTEST_LOG("Thread joined.\n");
    }
    else
    {
        NN_NETTEST_LOG("\n Send thread timed out during exit phase! ThreadError: %d\n\n", m_SendThreadData.error);
    }

    NN_NETTEST_LOG(" Waiting for recv thread...\n");
    isSuccess = NetTest::TimedWaitEvent(&m_RecvThreadData.isExited, threadExitTimeout);
    if( isSuccess )
    {
        NN_NETTEST_LOG("Exit signaled from recv thread. Joining thread...");
        NetTest::WaitThread(&m_RecvThreadData.hThread);
        NN_NETTEST_LOG("Thread joined.\n");
    }
    else
    {
        NN_NETTEST_LOG("\n Recv thread timed out during exit phase! ThreadError: %d\n\n", m_RecvThreadData.error);
    }

    if( !g_Params.noReport )
    {
        NN_NETTEST_LOG(" Sending shutdown commands to PrintThread...\n");
        m_PrintParam.doShutdown = true;
        NetTest::TimedWaitEvent(&m_PrintParam.isExitedEvent, threadExitTimeout);
        NN_NETTEST_LOG(" PrintThread joined.\n\n");
    }

    g_CsvLogger.Flush(0);
    NN_LOG("\n\n");
}

bool PacketRateTest::RunTestPrivate(int argC, const char * const * argV) NN_NOEXCEPT
{
    bool isSuccess;
    bool retSuccess = true;
    NetTest::Time frameTimeout = NetTest::TimeFromMs(FrameTimeoutMS);

    if( !g_Params.ParseCommandLine(argC, argV) )
    {
        NN_NETTEST_LOG(" WARNING: Failed to parse command line arguments!\n\n");
        return true;
    }

    // Initialize data and start threads.
    if( !Init() )
    {
        retSuccess = false;
        goto cleanup;
    }

    // Start the frame loop
    m_TestStart = NetTest::GetTick();
    m_StatsStart = m_TestStart;
    do
    {
        NetTest::Tick frameStart = NetTest::GetTick();

        NetTest::SignalEvent(&m_SendThreadData.doFrame);
        NetTest::SignalEvent(&m_RecvThreadData.doFrame);

        isSuccess = NetTest::TimedWaitEvent(&m_SendThreadData.isReady, frameTimeout);
        if( !isSuccess )
        {
            NN_NETTEST_LOG("\n WARNING!: Send thread timed out during Frame phase! ThreadError: %d\n\n", m_SendThreadData.error);
        }

        NetTest::ClearEvent(&m_SendThreadData.isReady);

        isSuccess = NetTest::TimedWaitEvent(&m_RecvThreadData.isReady, frameTimeout);
        if( !isSuccess )
        {
            NN_NETTEST_LOG("\n WARNING: Recv thread timed out during frame phase! ThreadError: %d\n\n", m_RecvThreadData.error);
        }

        NetTest::ClearEvent(&m_RecvThreadData.isReady);

        if( m_SendThreadData.error != ThreadError_None )
        {
            NN_NETTEST_LOG("Send thread returned error: %d\n", m_SendThreadData.error);
            retSuccess = false;
        }

        if( m_RecvThreadData.error != ThreadError_None )
        {
            NN_NETTEST_LOG("Recv thread returned error: %d\n", m_RecvThreadData.error);
            retSuccess = false;
        }

        ++g_Stats.frameCount;

        float frameDurationMs = NetTest::TickToTime(NetTest::GetTick() - frameStart).GetMicroSeconds() / 1000.0f;
        while( frameDurationMs < g_Params.frameIntervalMs )
        {
            NetTest::YieldThread();
            frameDurationMs = NetTest::TickToTime(NetTest::GetTick() - frameStart).GetMicroSeconds() / 1000.0f;
        }

        g_Stats.frameDurationSumMs += frameDurationMs;

        NetTest::Tick now = NetTest::GetTick();
        if(NetTest::TickToTime(now - m_StatsStart).GetMilliSeconds() >= INT_MAX)
        {
            NN_NETTEST_LOG("ERROR: duration overflows 32-bit value\n\n");
            retSuccess = false;
            goto cleanup;
        }

        uint32_t statsDurationMs = (uint32_t)NetTest::TickToTime(now - m_StatsStart).GetMilliSeconds();
        if( statsDurationMs > g_Params.reportIntervalMs )
        {
            if(NetTest::TickToTime(now - frameStart).GetMilliSeconds() >= INT_MAX)
            {
                NN_NETTEST_LOG("ERROR: duration overflows 32-bit value\n\n");
                retSuccess = false;
                goto cleanup;
            }

            PrintStats(statsDurationMs, (uint32_t)frameDurationMs);
            m_StatsStart = NetTest::GetTick();
        }

    } while( NetTest::TickToTime(NetTest::GetTick() - m_TestStart).GetMicroSeconds() < g_Params.testDurationSec * 1000000 && retSuccess );

cleanup:

    CleanUp();
    return retSuccess;
}

bool PacketRateTest::RunTest(int argC, const char * const * argV) NN_NOEXCEPT
{
    static bool s_IsRunning = false;

    if( s_IsRunning )
    {
        NN_NETTEST_LOG("ERROR: PacketRateTest already running!\n\n");
        return false;
    }

    s_IsRunning = true;

    PacketRateTest test;
    bool isSuccess = test.RunTestPrivate(argC, argV);;

    s_IsRunning = false;
    return isSuccess;
}

}}} // Namespaces

// Main Entry point
TEST(Net, P2P)
{
    int argC = 0;
    const char * const * argV = NULL;
    NATF::Utils::InitApi initApi(NATF::Utils::InitApiFlags::InitApiFlags_Nifm | NATF::Utils::InitApiFlags::InitApiFlags_Network | NATF::Utils::InitApiFlags::InitApiFlags_Socket);

    bool isSuccess = initApi.Init(nn::util::InvalidUuid);
    if( !isSuccess )
    {
        NN_NETTEST_LOG("ERROR: Failed to initialize something!\n\n");
        FAIL();
        return;
    }

    NETTEST_GET_ARGS(argC, argV);
    isSuccess = nnt::net::p2p::PacketRateTest::RunTest(argC, argV);

    initApi.Finalize();
    EXPECT_EQ(isSuccess, true);
    NETTEST_MAIN_RETURN();
}
