﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testNet_P2P.h"
#include "CommandLineParser.h"

namespace nnt { namespace net { namespace p2p {

// Parser Definitions
NodeIndexParser       g_NodeIndexParser;
PacketSizeParser      g_PacketSizeParser;
PacketCountParser     g_PacketCountParser;
FrameIntParser        g_FrameIntParser;
ReportIntParser       g_ReportIntParser;
TestDurationParser    g_TestDurationParser;
AddrInfoParser        g_AddrInfoParser;
VerifyDataParser      g_VerifyDataParser;
CallTimeParser        g_CallTimeParser;
TermBadDataParser     g_TermBadDataParser;
ReportNodeStatsParser g_ReportNodeStatsParser;
ReportCsvParser       g_ReportCsvParser;
NoReportParser        g_NoReport;
RttFrameIntParser     g_RttFrameIntParser;

// List of parsers
const Parser* const Parser::g_pParsers[] = {&g_NodeIndexParser, &g_PacketSizeParser,   &g_PacketCountParser,     &g_FrameIntParser,
                                            &g_ReportIntParser, &g_TestDurationParser, &g_AddrInfoParser,        &g_VerifyDataParser,
                                            &g_CallTimeParser,  &g_TermBadDataParser,  &g_ReportNodeStatsParser, &g_ReportCsvParser,
                                            &g_NoReport,        &g_RttFrameIntParser};
const uint32_t Parser::ParserCount = sizeof(g_pParsers) / sizeof(g_pParsers[0]);

// Parse one float from command line args.
bool Parser::ParseFloat(float& fltOut, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    float value = -1;

    int rval = sscanf(&argV[iArg][strlen(GetName())], "=%f", &value);
    if( rval != 1 )
    {
        NN_NETTEST_LOG("Failed to parse %s. rval: %d\n\n", GetName(), rval);
        return false;
    }

    fltOut = value;
    NN_NETTEST_LOG("Parsed %s: %d.%d\n", GetName(), (int)value, int(value * 100.0f) % 100);

    return true;
}

// Parse one int from command line args.
bool Parser::ParseInt(int& intOut, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    int value = -1;

    int rval = sscanf(&argV[iArg][strlen(GetName())], "=%d", &value);
    if( rval != 1 )
    {
        NN_NETTEST_LOG("Failed to parse %s. rval: %d\n\n", GetName(), rval);
        return false;
    }

    intOut = value;
    NN_NETTEST_LOG("Parsed %s: %d\n", GetName(), value);
    return true;
}

// Parse --NODE_INDEX
bool NodeIndexParser::Parse(Params& param, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    int value = -1;
    if( !ParseInt(value, argC, argV, iArg) )
    {
        return false;
    }

    if( value < 0 )
    {
        NN_NETTEST_LOG("\nERROR: %s must be greater or equal to zero!\n\n", GetName());
        return false;
    }
    else if( value >= static_cast<int32_t>(MaxNodes) )
    {
        NN_NETTEST_LOG("\nERROR: %s must be less than %d\n\n", GetName(), MaxNodes);
        return false;
    }

    param.nodeIndex = (uint8_t) value;
    return true;
}

// Parse --PACKET_SIZE
bool PacketSizeParser::Parse(Params& param, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    int value = -1;
    if( !ParseInt(value, argC, argV, iArg) )
    {
        return false;
    }

    if( static_cast<size_t>(value) < sizeof(NetHeader) || value > static_cast<int>(MaxPacketLen) )
    {
        NN_NETTEST_LOG("\nERROR: Invalid value. Valid range: %d <= %s <= %d!\n\n", (int)sizeof(NetHeader), GetName(), (int)MaxPacketLen);
        return false;
    }

    param.packetSize = value;
    return true;
}

// Parse --PACKET_COUNT
bool PacketCountParser::Parse(Params& param, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    int value = -1;
    if( !ParseInt(value, argC, argV, iArg) )
    {
        return false;
    }

    if( value <= 0 )
    {
        NN_NETTEST_LOG("\nERROR: %s must be greater than zero!\n\n", GetName());
        return false;
    }

    param.packetsPerFrame = value;
    return true;
}

// Parse --FRAME_INT
bool FrameIntParser::Parse(Params& param, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    float value = -1.0f;
    if( !ParseFloat(value, argC, argV, iArg) )
    {
        return false;
    }

    if( value <= 0.0f )
    {
        NN_NETTEST_LOG("\nERROR: %s must be greater than zero!\n\n", GetName());
        return false;
    }

    param.frameIntervalMs = value;
    return true;
}

// Parse --REPORT_INT
bool ReportIntParser::Parse(Params& param, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    int value = -1;
    if( !ParseInt(value, argC, argV, iArg) )
    {
        return false;
    }

    if( value <= 0 )
    {
        NN_NETTEST_LOG("\nERROR: %s must be greater than zero!\n\n", GetName());
        return false;
    }

    param.reportIntervalMs = value;
    return true;
}

// Parse --TEST_DURATION
bool TestDurationParser::Parse(Params& param, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    int value = -1;
    if( !ParseInt(value, argC, argV, iArg) )
        return false;

    if( value <= 0 )
    {
        NN_NETTEST_LOG("\nERROR: %s must be greater than zero!\n\n", GetName());
        return false;
    }

    param.testDurationSec = value;
    return true;
}

// Parse --VERIFY_DATA
bool VerifyDataParser::Parse(Params& param, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    NN_NETTEST_LOG("Parsed %s\n", GetName());
    param.doVerifyData = true;
    return true;
}

// Parse --REPORT_CALLTIME
bool CallTimeParser::Parse(Params& param, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    NN_NETTEST_LOG("Parsed %s\n", GetName());
    PacketRateTest::g_pSendToFn = PacketRateTest::TimedSendTo;
    PacketRateTest::g_pRecvFromFn = PacketRateTest::TimedRecvFrom;
    param.reportCallTime = true;
    return true;
}

// Parse --TERM_BAD_DATA
bool TermBadDataParser::Parse(Params& param, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    NN_NETTEST_LOG("Parsed %s\n", GetName());
    param.doTermOnBadData = true;
    return true;
}

// Parse --REPORT_NODE_STATS
bool ReportNodeStatsParser::Parse(Params& param, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    NN_NETTEST_LOG("Parsed %s\n", GetName());
    param.reportNodeStats = true;
    return true;
}

// Parse --REPORT_CSV
bool ReportCsvParser::Parse(Params& param, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    NN_NETTEST_LOG("Parsed %s\n", GetName());
    param.reportCsv = true;
    return true;
}

// Parse --NO_REPORT
bool NoReportParser::Parse(Params& param, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    NN_NETTEST_LOG("Parsed %s\n", GetName());
    param.noReport = true;
    return true;
}

// Parse --NODES_ADDR_INFO
// Several IP:PORT for the nodes.
bool AddrInfoParser::Parse(Params& param, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    int value = -1;
    char pIpPort[32];

    param.nodeCount = 0;

    uint32_t argLen = (uint32_t)strlen(argV[iArg]);
    uint32_t charPos = (uint32_t)strlen(GetName()) + 1; // Plus one for '='

    do // Loop until we reach the null terminator.
    {
        int ip1, ip2, ip3, ip4;

        // Skip seperators
        if( argV[iArg][charPos] == ',' )
        {
            ++charPos;
        }

        int rval = sscanf(&argV[iArg][charPos], "%d.%d.%d.%d:%d", &ip1, &ip2, &ip3, &ip4, &value);
        if( rval != 5 ) // Did we successfully parse all expected elements?
        {
            NN_NETTEST_LOG("Failed to parse %s. rval: %d\n\n", GetName(), rval);
            return false;
        }
        else // Successfully parsed an IP:PORT combination.
        {
            if( value <= 0 )
            {
                NN_NETTEST_LOG("\nERROR: port numbers in %s must be greater than zero!\n\n", GetName());
                return false;
            }

            if(value > 65536)
            {
                NN_NETTEST_LOG("\nERROR: Value overflows u16 size \n\n");
                return false;
            }

            NETTEST_SNPRINTF(param.pAddrs[param.nodeCount].pIp, sizeof(param.pAddrs[param.nodeCount].pIp), "%d.%d.%d.%d", ip1, ip2, ip3, ip4);
            param.pAddrs[param.nodeCount].port = (uint16_t) value;

            NN_NETTEST_LOG("IP: %s, Port: %d\n", param.pAddrs[param.nodeCount].pIp, value);

            ++param.nodeCount;

            NETTEST_SNPRINTF(pIpPort, sizeof(pIpPort), "%d.%d.%d.%d:%d", ip1, ip2, ip3, ip4, value);
            uint32_t readLen = (uint32_t)strlen(pIpPort);
            charPos += readLen;

            if( charPos > argLen )
            {
                NN_NETTEST_LOG("\nERROR: Failed to parse %s\n\n", GetName());
                return false;
            }
        }
    } while( argV[iArg][charPos] != '\0' );

    if( param.nodeCount > MaxNodes )
    {
        NN_NETTEST_LOG("\nERROR: Failed to parse %s. Max Peers: %d\n\n", GetName(), MaxNodes);
        return false;
    }

    return true;
}

// Parse --RTT_FRAME_INT
bool RttFrameIntParser::Parse(Params& param, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    int value = -1;
    if( !ParseInt(value, argC, argV, iArg) )
    {
        return false;
    }

    if( value < 0 )
    {
        NN_NETTEST_LOG("\nERROR: %s must be a non-negative number!\n\n", GetName());
        return false;
    }

    param.rttFrameInterval = value;
    return true;
}

}}} // Namespaces
