﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>

#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiIpAddress.h>          // nn::nifm::GetCurrentIpConfigInfo
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions


/* udplite.h */

#ifndef UDPLITE_SEND_CSCOV
static const int UDPLITE_SEND_CSCOV = 2;  /* Sender checksum coverage. */
#endif

#ifndef UDPLITE_RECV_CSCOV
static const int UDPLITE_RECV_CSCOV = 4;  /* Receiver checksum coverage. */
#endif


namespace NATF {
namespace API {

//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32


//*********************
//*  G L O B A L S
//*********************

static const int            kThreadPriority = 10;       // Thread Priority
static const int            kMaxNumOfSockets = 3;       // Max test sockets

static ThreadedController * g_pThreadedTest = NULL;
static int                  g_echoServerThreadId = -1;
static EchoServer *         g_echoServer = NULL;


static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}


static bool
InitializeTesting()
{
    bool        isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumOfSockets );
    // Create a thread to run the Server in

    g_echoServerThreadId = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, kThreadPriority );
    if (  g_echoServerThreadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // If the loopback DNS server is allocated
    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait for Echo Server thread to die
    g_pThreadedTest->WaitThread( g_echoServerThreadId );

    // Destroy Echo Server Thread
    g_pThreadedTest->DestroyThread( g_echoServerThreadId );

    // Delete Echo Server
    delete g_echoServer;
    g_echoServer = NULL;

    // Delete Threaded Test
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;
    g_echoServerThreadId = -1;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );

out:

    return( false );
}



////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,Socket_Initialize)
{
    InitializeTesting();
}


//***************************
// Socket Types
//
//#define SOCK_STREAM      1
//#define SOCK_DGRAM       2
//#define SOCK_RAW         3
//#define SOCK_RDM         4
//#define SOCK_SEQPACKET   5
//#define SOCK_PACKET      10
//
// Socket Domains
//
//#define AF_UNSPEC 0
//#define AF_UNIX 1
//#define AF_LOCAL 1
//#define AF_INET 2
//#define AF_AX25 3
//#define AF_IPX 4
//#define AF_APPLETALK 5
//#define AF_NETROM 6
//#define AF_BRIDGE 7
//#define AF_ATMPVC 8
//#define AF_X25 9
//#define AF_INET6 10
//#define AF_ROSE 11
//#define AF_DECnet 12
//#define AF_NETBEUI 13
//#define AF_SECURITY 14
//#define AF_KEY 15
//#define AF_NETLINK 16
//#define AF_ROUTE AF_NETLINK
//#define AF_PACKET 17
//#define AF_ASH 18
//#define AF_ECONET 19
//#define AF_ATMSVC 20
//#define AF_RDS 21
//#define AF_SNA 22
//#define AF_IRDA 23
//#define AF_PPPOX 24
//#define AF_WANPIPE 25
//#define AF_LLC 26
//#define AF_CAN 29
//#define AF_TIPC 30
//#define AF_BLUETOOTH 31
//#define AF_IUCV 32
//#define AF_RXRPC 33
//#define AF_ISDN 34
//#define AF_PHONET 35
//#define AF_IEEE802154 36
//#define AF_CAIF 37
//#define AF_ALG 38
//#define AF_MAX 39
//
// Socket Protocol
//
//IPPROTO_IPV6            = 41,          /* IP6 header */
//IPPROTO_RAW             = 255,         /* raw IP packet */
//IPPROTO_HOPOPTS         = 0,           /* IP6 hop-by-hop options */
//IPPROTO_ICMP            = 1,           /* Internet Control Message Protocol */
//IPPROTO_IGMP            = 2,           /* group mgmt protocol */
//IPPROTO_GGP             = 3,           /* gateway^2 (deprecated) */
//IPPROTO_IPV4            = 4,           /* IPv4 encapsulation */
//IPPROTO_IPIP            = IPPROTO_IPV4,    /* for compatibility */
//IPPROTO_ST              = 7,           /* Stream protocol II */
//IPPROTO_EGP             = 8,           /* exterior gateway protocol */
//IPPROTO_PIGP            = 9,           /* private interior gateway */
//IPPROTO_RCCMON          = 10,          /* BBN RCC Monitoring */
//IPPROTO_NVPII           = 11,          /* network voice protocol*/
//IPPROTO_PUP             = 12,          /* pup */
//IPPROTO_ARGUS           = 13,          /* Argus */
//IPPROTO_EMCON           = 14,          /* EMCON */
//IPPROTO_XNET            = 15,          /* Cross Net Debugger */
//IPPROTO_CHAOS           = 16,          /* Chaos*/
//IPPROTO_MUX             = 18,          /* Multiplexing */
//IPPROTO_MEAS            = 19,          /* DCN Measurement Subsystems */
//IPPROTO_HMP             = 20,          /* Host Monitoring */
//IPPROTO_PRM             = 21,          /* Packet Radio Measurement */
//IPPROTO_IDP             = 22,          /* xns idp */
//IPPROTO_TRUNK1          = 23,          /* Trunk-1 */
//IPPROTO_TRUNK2          = 24,          /* Trunk-2 */
//IPPROTO_LEAF1           = 25,          /* Leaf-1 */
//IPPROTO_LEAF2           = 26,          /* Leaf-2 */
//IPPROTO_RDP             = 27,          /* Reliable Data */
//IPPROTO_IRTP            = 28,          /* Reliable Transaction */
//IPPROTO_TP              = 29,          /* tp-4 w/ class negotiation */
//IPPROTO_BLT             = 30,          /* Bulk Data Transfer */
//IPPROTO_NSP             = 31,          /* Network Services */
//IPPROTO_INP             = 32,          /* Merit Internodal */
//IPPROTO_SEP             = 33,          /* Sequential Exchange */
//IPPROTO_3PC             = 34,          /* Third Party Connect */
//IPPROTO_IDPR            = 35,          /* InterDomain Policy Routing */
//IPPROTO_XTP             = 36,          /* XTP */
//IPPROTO_DDP             = 37,          /* Datagram Delivery */
//IPPROTO_CMTP            = 38,          /* Control Message Transport */
//IPPROTO_TPXX            = 39,          /* TP++ Transport */
//IPPROTO_IL              = 40,          /* IL transport protocol */
//IPPROTO_SDRP            = 42,          /* Source Demand Routing */
//IPPROTO_ROUTING         = 43,          /* IP6 routing header */
//IPPROTO_FRAGMENT        = 44,          /* IP6 fragmentation header */
//IPPROTO_IDRP            = 45,          /* InterDomain Routing*/
//IPPROTO_RSVP            = 46,          /* resource reservation */
//IPPROTO_GRE             = 47,          /* General Routing Encap. */
//IPPROTO_MHRP            = 48,          /* Mobile Host Routing */
//IPPROTO_BHA             = 49,          /* BHA */
//IPPROTO_ESP             = 50,          /* IP6 Encap Sec. Payload */
//IPPROTO_AH              = 51,          /* IP6 Auth Header */
//IPPROTO_INLSP           = 52,          /* Integ. Net Layer Security */
//IPPROTO_SWIPE           = 53,          /* IP with encryption */
//IPPROTO_NHRP            = 54,          /* Next Hop Resolution */
//IPPROTO_MOBILE          = 55,          /* IP Mobility */
//IPPROTO_TLSP            = 56,          /* Transport Layer Security */
//IPPROTO_SKIP            = 57,          /* SKIP */
//IPPROTO_ICMPV6          = 58,          /* ICMP6 */
//IPPROTO_NONE            = 59,          /* IP6 no next header */
//IPPROTO_DSTOPTS         = 60,          /* IP6 destination option */
//IPPROTO_AHIP            = 61,          /* any host internal protocol */
//IPPROTO_CFTP            = 62,          /* CFTP */
//IPPROTO_HELLO           = 63,          /* "hello" routing protocol */
//IPPROTO_SATEXPAK        = 64,          /* SATNET/Backroom EXPAK */
//IPPROTO_KRYPTOLAN       = 65,          /* Kryptolan */
//IPPROTO_RVD             = 66,          /* Remote Virtual Disk */
//IPPROTO_IPPC            = 67,          /* Pluribus Packet Core */
//IPPROTO_ADFS            = 68,          /* Any distributed FS */
//IPPROTO_SATMON          = 69,          /* Satnet Monitoring */
//IPPROTO_VISA            = 70,          /* VISA Protocol */
//IPPROTO_IPCV            = 71,          /* Packet Core Utility */
//IPPROTO_CPNX            = 72,          /* Comp. Prot. Net. Executive */
//IPPROTO_CPHB            = 73,          /* Comp. Prot. HeartBeat */
//IPPROTO_WSN             = 74,          /* Wang Span Network */
//IPPROTO_PVP             = 75,          /* Packet Video Protocol */
//IPPROTO_BRSATMON        = 76,          /* BackRoom SATNET Monitoring */
//IPPROTO_ND              = 77,          /* Sun net disk proto (temp.) */
//IPPROTO_WBMON           = 78,          /* WIDEBAND Monitoring */
//IPPROTO_WBEXPAK         = 79,          /* WIDEBAND EXPAK */
//IPPROTO_EON             = 80,          /* ISO cnlp */
//IPPROTO_VMTP            = 81,          /* VMTP */
//IPPROTO_SVMTP           = 82,          /* Secure VMTP */
//IPPROTO_VINES           = 83,          /* Banyon VINES */
//IPPROTO_DGP             = 86,          /* dissimilar gateway prot. */
//IPPROTO_TCF             = 87,          /* TCF */
//IPPROTO_IGRP            = 88,          /* Cisco/GXS IGRP */
//IPPROTO_OSPFIGP         = 89,          /* OSPFIGP */
//IPPROTO_SRPC            = 90,          /* Strite RPC protocol */
//IPPROTO_LARP            = 91,          /* Locus Address Resoloution */
//IPPROTO_MTP             = 92,          /* Multicast Transport */
//IPPROTO_AX25            = 93,          /* AX.25 Frames */
//IPPROTO_IPEIP           = 94,          /* IP encapsulated in IP */
//IPPROTO_MICP            = 95,          /* Mobile Int.ing control */
//IPPROTO_SCCSP           = 96,          /* Semaphore Comm. security */
//IPPROTO_ETHERIP         = 97,          /* Ethernet IP encapsulation */
//IPPROTO_ENCAP           = 98,          /* encapsulation header */
//IPPROTO_APES            = 99,          /* any private encr. scheme */
//IPPROTO_GMTP            = 100,         /* GMTP*/
//IPPROTO_IPCOMP          = 108,         /* payload compression (IPComp) */
//IPPROTO_SCTP            = 132,         /* SCTP */
//IPPROTO_MH              = 135,         /* IPv6 Mobility Header */
//IPPROTO_UDPLITE         = 136,         /* UDP-Lite */
//IPPROTO_HIP             = 139,         /* IP6 Host Identity Protocol */
//IPPROTO_SHIM6           = 140,         /* IP6 Shim6 Protocol */
///* 101-254: Partly Unassigned */
//IPPROTO_PIM             = 103,         /* Protocol Independent Mcast */
//IPPROTO_CARP            = 112,         /* CARP */
//IPPROTO_PGM             = 113,         /* PGM */
//IPPROTO_MPLS            = 137,         /* MPLS-in-IP */
//IPPROTO_PFSYNC          = 240,         /* PFSYNC */
//IPPROTO_RESERVED_253    = 253,         /* Reserved */
//IPPROTO_RESERVED_254    = 254,         /* Reserved */
///* 255: Reserved */
//IPPROTO_OLD_DIVERT      = 254,         /* OLD divert pseudo-proto */
//IPPROTO_MAX             = 256,
//***************************


TEST(ApiUnit,Socket_Full_Range)
{
    nn::socket::Family      domainIdx;
    nn::socket::Type        typeIdx;
    nn::socket::Protocol    protoIdx;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    int                     cliSock = -1;
    int                     eperm = 0, eafnosupport = 0, eprotonosupport = 0, eprototype = 0, unknown = 0;
    bool                    isSuccess = false;

    // socket(int domain, int type, int protocol);


    NN_LOG( "Processing Domain, Type and Protocol:");

    for( domainIdx = nn::socket::Family::Af_Unspec; domainIdx < nn::socket::Family::Af_Max; domainIdx = static_cast<nn::socket::Family>(static_cast<int>(domainIdx) + 1) )  // nn::socket::Family::Af_Unspec (to) nn::socket::Family::Af_Max
    {
        for( typeIdx = nn::socket::Type::Sock_Default; typeIdx <= static_cast<nn::socket::Type>(10); typeIdx = static_cast<nn::socket::Type>(static_cast<int>(typeIdx) + 1) )  // nn::socket::Type::Sock_Stream (to) SOCK_PACKET
        {
            for( protoIdx = nn::socket::Protocol::IpProto_Ip; protoIdx < nn::socket::Protocol::IpProto_Max; protoIdx = static_cast<nn::socket::Protocol>(static_cast<int>(protoIdx) + 1) )  // IPPROTO_HOPOPTS (to) IPPROTO_MAX
            {
               // TCP + Protocol = 0, default protocol for nn::socket::Family::Af_Inet / nn::socket::Type::Sock_Stream == TCP - Skip
               if ( domainIdx == nn::socket::Family::Af_Inet && typeIdx == nn::socket::Type::Sock_Stream && protoIdx == nn::socket::Protocol::IpProto_Ip ) continue;

               // TCP + nn::socket::Protocol::IpProto_Tcp - Skip
               if ( domainIdx == nn::socket::Family::Af_Inet && typeIdx == nn::socket::Type::Sock_Stream && protoIdx == nn::socket::Protocol::IpProto_Tcp ) continue;

               // UPD + Protocol = 0, default protocol for nn::socket::Family::Af_Inet / nn::socket::Type::Sock_Dgram == UDP - Skip
               if ( domainIdx == nn::socket::Family::Af_Inet && typeIdx == nn::socket::Type::Sock_Dgram && protoIdx == nn::socket::Protocol::IpProto_Ip ) continue;

               // UDP + nn::socket::Protocol::IpProto_Udp - Skip
               if ( domainIdx == nn::socket::Family::Af_Inet && typeIdx == nn::socket::Type::Sock_Dgram && protoIdx == nn::socket::Protocol::IpProto_Udp ) continue;

               // UDP + nn::socket::Protocol::IpProto_UdpLite - Skip
               if ( domainIdx == nn::socket::Family::Af_Inet && typeIdx == nn::socket::Type::Sock_Dgram && protoIdx == nn::socket::Protocol::IpProto_UdpLite ) continue;

               // RAW + ICMP - Skip
               if ( domainIdx == nn::socket::Family::Af_Inet && typeIdx == nn::socket::Type::Sock_Raw && protoIdx == nn::socket::Protocol::IpProto_Icmp ) continue;

               cliSock = nn::socket::Socket( domainIdx, typeIdx, protoIdx );
               if ( cliSock > -1 )
               {
                   NN_LOG( "nn::socket::Socket() Successeded for Domain: %d, Type: %d, Proto: %d\n", domainIdx, typeIdx, protoIdx );
                   nn::socket::Close( cliSock );
                   unknown++;
                   continue;
               }

               myError = nn::socket::GetLastError();
               switch( myError )
               {
               case nn::socket::Errno::EPerm:            eperm++; break;
               case nn::socket::Errno::EAfNoSupport:     eafnosupport++; break;
               case nn::socket::Errno::EProtoNoSupport:  eprotonosupport++; break;
               case nn::socket::Errno::EPrototype:       eprototype++; break;
               default:
                    NN_LOG( "I expected errno nn::socket::Errno::EAfNoSupport, but I got: %d - Domain: %d, Type: %d, Proto: %d\n", myError, domainIdx, typeIdx, protoIdx );
                    unknown++;
                    break;
               }
           }
        }
    }

    NN_LOG( "\nResults:\n" );
    NN_LOG( "Unknown [%d]\n", unknown );
    NN_LOG( "nn::socket::Errno::EPerm [%d]\n", eperm );
    NN_LOG( "nn::socket::Errno::EAfNoSupport [%d]\n", eafnosupport );
    NN_LOG( "nn::socket::Errno::EProtoNoSupport [%d]\n", eprotonosupport );
    NN_LOG( "nn::socket::Errno::EPrototype [%d]\n", eprototype );
    NN_LOG( "Total Invalid Protocols: %d\n",
            unknown + eperm + eafnosupport + eprotonosupport + eprototype );

    ERROR_IF_AND_COUNT(unknown > 0,"There were [%d] Uknown combinations of socket (family, type, IP Protocol) found!  This needs to be checked!\n", unknown);
    nTestsPassing++;

out:

    return;
}


TEST(ApiUnit,Socket_UDPLITE)
{
    nn::socket::SockAddrIn localAddr = { 0 }, targetAddr = { 0 };
    nn::socket::TimeVal myTime = { 0 };
    bool       isSuccess = true;
    int        udpSock = -1, udpServSock = -1, rc = 0;
    int        optVal = 0;
    nn::socket::SockLenT  optLen;
    char       buf1[1024];
    size_t     buf1Len = 0;

    /////////////////
    // Create Client UDPLITE Socket
    /////////////////

    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_UdpLite );
    ERROR_IF_AND_COUNT( udpSock < 0, "Failed to create a UDPLITE nn::socket::Type::Sock_Dgram socket!  Errno: %d\n", nn::socket::GetLastError() );

    // Client - Local Address
    memset( &localAddr, 0, sizeof(localAddr));
    localAddr.sin_port        = nn::socket::InetHtons( 9013 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client - Local Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Client - Local Bind
    rc = nn::socket::Bind( udpSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

    // Client - Tagrget Address
    memset( &targetAddr, 0, sizeof(targetAddr));
    targetAddr.sin_port        = nn::socket::InetHtons( 9012 );
    targetAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client - Target Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &targetAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Client - Connect
    rc = nn::socket::Connect( udpSock, (nn::socket::SockAddr *) &targetAddr, sizeof( targetAddr ) );
    ERROR_IF( rc != 0, "Connect failed for local socket, but should have succeeded!\n" );

    // Client - Checksum covers 8 octets of header + 12 octets of the application protocol.
    optVal = 20;
    rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_UdpLite, static_cast<nn::socket::Option>(UDPLITE_SEND_CSCOV), (void *) &optVal, sizeof( optVal ) );
    ERROR_IF( rc != 0, "SetSockOpt failed for nn::socket::Level::Sol_UdpLite, errno: %d!\n", nn::socket::GetLastError() );

    optLen = sizeof( optVal );
    rc = nn::socket::GetSockOpt( udpSock, nn::socket::Level::Sol_UdpLite, static_cast<nn::socket::Option>(UDPLITE_SEND_CSCOV), (void *) &optVal, &optLen );
    ERROR_IF_AND_COUNT( rc != 0, "GetSockOpt failed for nn::socket::Level::Sol_UdpLite, errno: %d!\n", nn::socket::GetLastError() );
    NN_LOG( "Client [UDPLITE_SEND_CSCOV] = %d\n", optVal );


    /////////////////
    // Create Server UDPLITE Socket
    /////////////////

    udpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_UdpLite );
    ERROR_IF_AND_COUNT( udpSock < 0, "Failed to create a UDPLITE nn::socket::Type::Sock_Dgram socket!  Errno: %d\n", nn::socket::GetLastError() );

    // Server - Local Address
    memset( &localAddr, 0, sizeof(localAddr));
    localAddr.sin_port        = nn::socket::InetHtons( 9012 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server - Local Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server - Local Bind
    rc = nn::socket::Bind( udpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

    // Server - Tagrget Address
    memset( &targetAddr, 0, sizeof(targetAddr));
    targetAddr.sin_port        = nn::socket::InetHtons( 9013 );
    targetAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server - Target Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &targetAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server - Connect
    rc = nn::socket::Connect( udpServSock, (nn::socket::SockAddr *) &targetAddr, sizeof( targetAddr ) );
    ERROR_IF( rc != 0, "Connect failed for local socket, but should have succeeded!\n" );

    // Server - Checksum covers 8 octets of header + 12 octets of the application protocol.
    optVal = 20;
    rc = nn::socket::SetSockOpt( udpServSock, nn::socket::Level::Sol_UdpLite, static_cast<nn::socket::Option>(UDPLITE_RECV_CSCOV), (void *) &optVal, sizeof( optVal ) );
    ERROR_IF( rc != 0, "SetSockOpt failed for nn::socket::Level::Sol_UdpLite, errno: %d!\n", nn::socket::GetLastError() );

    optLen = sizeof( optVal );
    rc = nn::socket::GetSockOpt( udpServSock, nn::socket::Level::Sol_UdpLite, static_cast<nn::socket::Option>(UDPLITE_RECV_CSCOV), (void *) &optVal, &optLen );
    ERROR_IF_AND_COUNT( rc != 0, "GetSockOpt failed for nn::socket::Level::Sol_UdpLite, errno: %d!\n", nn::socket::GetLastError() );
    NN_LOG( "Server [UDPLITE_RECV_CSCOV] = %d\n", optVal );


    /////////////////
    // Client Sends Data / Server receives Data
    /////////////////

    NN_LOG( "Client Writes: [%s]\n",  "12345678901234567890" );
    rc = nn::socket::Send( udpSock, "12345678901234567890", 20, nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc < 0, "Client send of Data failed!  Errno: %d\n", nn::socket::GetLastError() );


    buf1Len = sizeof( buf1 );
    rc = nn::socket::Recv( udpServSock, buf1, buf1Len, nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc < 0, "Server recv of Data failed!  Errno: %d\n", nn::socket::GetLastError() );
    ERROR_IF( rc != 20, "I expected 20 bytes, received: %d\n", rc )
    NN_LOG( "Server Receives: [%.*s]\n", rc, buf1 );


    /////////////////
    // Client Checksum drops to 15
    /////////////////

    // Client - Checksum covers 8 octets of header + 7 octets of the application protocol.
    optVal = 15;
    rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_UdpLite, static_cast<nn::socket::Option>(UDPLITE_SEND_CSCOV), (void *) &optVal, sizeof( optVal ) );
    ERROR_IF( rc != 0, "SetSockOpt failed for nn::socket::Level::Sol_UdpLite, errno: %d!\n", nn::socket::GetLastError() );


    optLen = sizeof( optVal );
    rc = nn::socket::GetSockOpt( udpSock, nn::socket::Level::Sol_UdpLite, static_cast<nn::socket::Option>(UDPLITE_SEND_CSCOV), (void *) &optVal, &optLen );
    ERROR_IF( rc != 0, "GetSockOpt failed for nn::socket::Level::Sol_UdpLite, errno: %d!\n", nn::socket::GetLastError() );
    NN_LOG( "(Upated) Client [UDPLITE_SEND_CSCOV] = %d\n", optVal );


    /////////////////
    // Server configures a Recieve Timeout
    /////////////////

    // Server - Checksum covers 8 octets of header + 11 octets of the application protocol.
    myTime.tv_sec  = 1;
    myTime.tv_usec = 0;

    rc = nn::socket::SetSockOpt( udpServSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, (void *) &myTime, sizeof( myTime ) );
    ERROR_IF( rc != 0, "SetSockOpt failed for nn::socket::Option::So_RcvTimeo, errno: %d!\n", nn::socket::GetLastError() );
    NN_LOG( "(Upated) Server [nn::socket::Option::So_RcvTimeo] = 1s\n" );


    /////////////////
    // Client Sends Data / Server times out (UDP packet discarded due to failed checksum)
    /////////////////

    NN_LOG( "Client Writes: [%s]\n",  "12345678901234567890" );
    rc = nn::socket::Send( udpSock, "12345678901234567890", 20, nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc < 0, "Client send of Data failed!  Errno: %d\n", nn::socket::GetLastError() );

    buf1Len = sizeof( buf1 );
    rc = nn::socket::Recv( udpServSock, buf1, buf1Len, nn::socket::MsgFlag::Msg_None );
    if ( rc < 0 )
    {
        if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
        {
            NN_LOG( "*************************************\n" );
            NN_LOG( "PASSED!  UDPLite packet was discarded due to failed checksum\n" );
            NN_LOG( "*************************************\n" );

            nTestsPassing++;

            // Fall Thru
        }
        else
        {
            ERROR_IF( rc < 0, "Server recv failed!  Errno: %d\n", nn::socket::GetLastError() );
        }
    }
    else
    {
        NN_LOG( "FAILED!  Recieved 20 bytes of data but checksum should have failed!\n" );

        nTestsFailing++;
    }


out:

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    if ( udpServSock > -1 )
    {
        nn::socket::Close( udpServSock );
        udpServSock = -1;
    }

}   // NOLINT(impl/function_size)

#ifdef COMMENTED_OUT

TEST(ApiUnit,Socket_NIFM_UDPLITE)
{
    nn::socket::SockAddrIn   localAddr = { 0 }, targetAddr = { 0 };
    nn::socket::TimeVal       myTime = { 0 };
    bool       isSuccess = true;
    int        udpSock = -1, rc = 0;
    int        optVal = 0;
    nn::socket::SockLenT  optLen;
    char       buf1[1024];
    size_t     buf1Len = 0;

    nn::socket::InAddr myIp = { 0 };
    nn::socket::InAddr myMask = { 0 };
    nn::socket::InAddr myGw = { 0 };
    nn::socket::InAddr myDns1 = { 0 };
    nn::socket::InAddr myDns2 = { 0 };

    // Ask NIFM for our current (primary interface) IP Address
    nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

    /////////////////
    // Create Client UDPLITE Socket
    /////////////////

    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_UdpLite );
    ERROR_IF_AND_COUNT( udpSock < 0, "Failed to create a UDPLITE nn::socket::Type::Sock_Dgram socket!  Errno: %d\n", nn::socket::GetLastError() );

    // Client - Local Address
    memset( &localAddr, 0, sizeof(localAddr));
    localAddr.sin_port        = nn::socket::InetHtons( 9013 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client - Local Address
    NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
    localAddr.sin_addr.S_addr = myIp.S_addr;

    // Client - Local Bind
    rc = nn::socket::Bind( udpSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

    // Client - Tagrget Address
    memset( &targetAddr, 0, sizeof(targetAddr));
    targetAddr.sin_port        = nn::socket::InetHtons( 8053 );
    targetAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client - Target Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "10.46.194.177", &targetAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Client - Connect
    rc = nn::socket::Connect( udpSock, (nn::socket::SockAddr *) &targetAddr, sizeof( targetAddr ) );
    ERROR_IF( rc != 0, "Connect failed for local socket, but should have succeeded!\n" );

    // Client - Checksum covers 8 octets of header + 12 octets of the application protocol.
    optVal = 20;
    rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_UdpLite, static_cast<nn::socket::Option>(UDPLITE_SEND_CSCOV), (void *) &optVal, sizeof( optVal ) );
    ERROR_IF( rc != 0, "SetSockOpt failed for nn::socket::Level::Sol_UdpLite, errno: %d!\n", nn::socket::GetLastError() );

    optLen = sizeof( optVal );
    rc = nn::socket::GetSockOpt( udpSock, nn::socket::Level::Sol_UdpLite, static_cast<nn::socket::Option>(UDPLITE_SEND_CSCOV), (void *) &optVal, &optLen );
    ERROR_IF_AND_COUNT( rc != 0, "GetSockOpt failed for nn::socket::Level::Sol_UdpLite, errno: %d!\n", nn::socket::GetLastError() );
    NN_LOG( "Client [UDPLITE_SEND_CSCOV] = %d\n", optVal );


    /////////////////
    // Client Sends Data / Server receives Data
    /////////////////

    NN_LOG( "Client Writes: [%s]\n",  "12345678901234567890" );
    rc = nn::socket::Send( udpSock, "12345678901234567890", 20, 0 );
    ERROR_IF( rc < 0, "Client send of Data failed!  Errno: %d\n", nn::socket::GetLastError() );

out:

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }
}

#endif

TEST(ApiUnit,Socket_Teardown)
{
    TeardownTesting();
}

#else // Windows

TEST(ApiUnit,Socket_Windows)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

}}  // Namespace: NATF::API
