﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"
#include "Unit/testNet_CommonFunctions.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc

#include <nn/socket/socket_ConstantsPrivate.h>
#include <nn/socket.h>

#include <nnt/nntest.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiIpAddress.h>
#include <nn/nifm/nifm_ApiNetworkProfile.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_TemporaryNetworkProfile.h>

#include <nn/os.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_TimerEvent.h>

#include <nn/nn_Log.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server


namespace NATF {
namespace API {

//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32


//*********************
//*  G L O B A L S
//*********************

static ThreadedController * g_pThreadedTest = NULL;
static EchoServer *         g_echoServer = NULL;

static const int            kIPHeaderLen = 20;        // IP Header Len
static const int            kICMPHeaderLen = 8;       // ICMP Header Len

// Anonymous Namespace
namespace {

// Integer Value Results
typedef struct {

    int   value;
    bool  result;

} SockOptValueResult;

static SockOptValueResult g_SockOptValueResults[] =
{
    {-2147483648, true},
    {-1, true},
    {0, false},
    {1, true},
    {INT_MAX, true}

};

// Struct Timeval Results
typedef struct {

    int      tv_sec;
    int      tv_usec;
    bool     result;

} SockOptTimeoutResult;


} // Anonymous Namespace

static void
StartEchoServer( void * inArg )
{
    NN_UNUSED(kICMPHeaderLen);
    NN_UNUSED(kIPHeaderLen);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}


static bool
InitializeTesting()
{
    nn::Result                result;
    int                       rc;
    bool                      isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );

    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( 32 );

    // Create a thread to run the Server in
    rc = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, 10 );
    if ( rc < 0 )
    {
        NN_LOG( "Failed to create and start DNS Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // Free Threaded Tester
    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait DNS Server threads to die
    g_pThreadedTest->WaitAllThreads();

    // Destroy all Threads
    g_pThreadedTest->DestroyAllThreads();

    // Delete Object Pointer
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;

    delete g_echoServer;
    g_echoServer = NULL;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

out:

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );
}


TEST(ApiUnit,SetSockOpt_TCP_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,SetSockOpt_TCP_INFO)
{
    nn::socket::TcpInfo    myTCPInfo = {};
    int                    tcpsock = -1, listensock = -1, acceptedsock = -1;
    int                    rc = -1;
    bool                   isSuccess = true, rval = false;
    nn::socket::SockLenT   optLen = 0;

    ///////////////////
    // Make TCP Client and Server
    ///////////////////

    rval = NATF::API::COMMON::MakeListenSocket( listensock, 9020, "127.0.0.1" );
    ERROR_IF_AND_COUNT( rval == false, "Failed calling Makeilistensocket!  Errno:%d\n", nn::socket::GetLastError() );

    tcpsock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 9020 );
    ERROR_IF_AND_COUNT( tcpsock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    acceptedsock = nn::socket::Accept( listensock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT( acceptedsock < 0, "Failed calling Accept!  Errno:%d\n", nn::socket::GetLastError() );

    /////////////
    // Get Information on Socket Connection
    /////////////

    optLen = sizeof(myTCPInfo );
    rc = nn::socket::GetSockOpt(tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_Info, (void *) &myTCPInfo, &optLen);
    ERROR_IF_AND_COUNT(rc < 0, "GetSockOpt failed for option nn::socket::Option::Tcp_Info - errno: %d\n", nn::socket::GetLastError());

    NN_LOG( "[nn::socket::TcpInfo:\n" );
    NN_LOG( "state:           %d\n", myTCPInfo.tcpi_state );          // TCP FSM state.
    NN_LOG( "ca state:        %d\n", myTCPInfo._tcpi_ca_state );
    NN_LOG( "retransmits:     %d\n", myTCPInfo._tcpi_retransmits );
    NN_LOG( "probes:          %d\n", myTCPInfo._tcpi_probes );
    NN_LOG( "backoff:         %d\n", myTCPInfo._tcpi_backoff );
    NN_LOG( "options:         %d\n", myTCPInfo.tcpi_options );        // Options enabled on conn.
    NN_LOG( "snd_wscale:      %d\n", myTCPInfo.tcpi_snd_wscale );     // RFC1323 send shift value
    NN_LOG( "rcv_wscale:      %d\n", myTCPInfo.tcpi_rcv_wscale );     // RFC1323 recv shitf value
    NN_LOG( "rto:             %d\n", myTCPInfo.tcpi_rto );            // Retransmition timeout (usec)
    NN_LOG( "ato:             %d\n", myTCPInfo._tcpi_ato );
    NN_LOG( "snd_mss:         %d\n", myTCPInfo.tcpi_snd_mss );        // Max Segment Size for Send
    NN_LOG( "rcv_mss:         %d\n", myTCPInfo.tcpi_rcv_mss );        // Max Segment Size for Recv
    NN_LOG( "unacked:         %d\n", myTCPInfo._tcpi_unacked );
    NN_LOG( "sacked:          %d\n", myTCPInfo._tcpi_sacked );
    NN_LOG( "lost:            %d\n", myTCPInfo._tcpi_lost );
    NN_LOG( "retrans:         %d\n", myTCPInfo._tcpi_retrans );
    NN_LOG( "fackets:         %d\n", myTCPInfo._tcpi_fackets );
    NN_LOG( "last_data_sent:  %d\n", myTCPInfo._tcpi_last_data_sent );
    NN_LOG( "last_ack_sent:   %d\n", myTCPInfo._tcpi_last_ack_sent );
    NN_LOG( "last_data_recv:  %d\n", myTCPInfo.tcpi_last_data_recv );   // Time since last received data
    NN_LOG( "last_ack_recv:   %d\n", myTCPInfo._tcpi_last_ack_recv );
    NN_LOG( "pmtu:            %d\n", myTCPInfo._tcpi_pmtu );
    NN_LOG( "rcv_ssthresh:    %d\n", myTCPInfo._tcpi_rcv_ssthresh );
    NN_LOG( "rtt:             %d\n", myTCPInfo.tcpi_rtt );              // Smoothed RTT in usecs
    NN_LOG( "rttvar:          %d\n", myTCPInfo.tcpi_rttvar );           // RTT variance in usecs
    NN_LOG( "snd_ssthresh:    %d\n", myTCPInfo.tcpi_snd_ssthresh );     // Slow start threshhold
    NN_LOG( "snd_cwnd:        %d\n", myTCPInfo.tcpi_snd_cwnd );         // Send congestion window
    NN_LOG( "advmss:          %d\n", myTCPInfo._tcpi_advmss );
    NN_LOG( "reordering:      %d\n", myTCPInfo._tcpi_reordering );
    NN_LOG( "rcv_rtt:         %d\n", myTCPInfo._tcpi_rcv_rtt );
    NN_LOG( "rcv_space:       %d\n", myTCPInfo.tcpi_rcv_space );        // Advertised recv window
    NN_LOG( "snd_wnd:         %d\n", myTCPInfo.tcpi_snd_wnd );          // Advertised send window
    NN_LOG( "snd_bwnd:        %d\n", myTCPInfo.tcpi_snd_bwnd );         // Not used
    NN_LOG( "snd_nxt:         %d\n", myTCPInfo.tcpi_snd_nxt );          // Next egress seqno
    NN_LOG( "rcv_nxt:         %d\n", myTCPInfo.tcpi_rcv_nxt );          // Next ingress seqno
    NN_LOG( "toe_tid:         %d\n", myTCPInfo.tcpi_toe_tid );          // HWTID for TOE endpoints
    NN_LOG( "snd_rexmitpack   %d\n", myTCPInfo.tcpi_snd_rexmitpack );   // Retransmitted Packets
    NN_LOG( "rcv_ooopack      %d\n", myTCPInfo.tcpi_rcv_ooopack );      // Out-Of-Order Packets
    NN_LOG( "snd_zerowin      %d\n", myTCPInfo.tcpi_snd_zerowin );      // Zero-sized windows sent

out:

    if ( tcpsock > -1 )
    {
        nn::socket::Close( tcpsock );
        tcpsock = -1;
    }

    if ( listensock > -1 )
    {
        nn::socket::Close( listensock );
        listensock = -1;
    }

    if ( acceptedsock > -1 )
    {
        nn::socket::Close( acceptedsock );
        acceptedsock = -1;
    }
}


TEST(ApiUnit,SetSockOpt_TCP_MAXSEG)
{
    nn::socket::SockAddrIn remoteAddr = { 0 };
    nn::socket::TcpInfo    myTCPInfo;
    int                    tcpsock = -1, listensock = -1, acceptedsock = -1;
    int                    rc = -1, idx = 0;
    bool                   isSuccess = true;
    nn::socket::SockLenT   optLen = 0;
    int                    optVal = 0;

    ////////////////
    // GetSockOpt Tests for: TCP_MAXSEG
    ///////////////

    static int sizeArrayMaxSegBad[] = {-65535, -32767, -16384, -9000, -1, 1600, 2500, 5000, INT_MAX};
    static int sizeArraySizeMaxSegBad = sizeof(sizeArrayMaxSegBad) / sizeof(int);

    //static int sizeArrayMaxSegGood[] = {128, 256, 512, 786, 1019, 1300, 1500, 1512};
    static int sizeArrayMaxSegGood[] = {1460};
    static int sizeArraySizeMaxSegGood = sizeof(sizeArrayMaxSegGood) / sizeof(int);

    ////////////////
    // TCP Client tests - BAD
    ///////////////

    for(idx = 0; idx < sizeArraySizeMaxSegBad; idx++)
    {
        // Client: Socket
        tcpsock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
        ERROR_IF_AND_COUNT(tcpsock < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError());

        // Client: Socket Option
        optVal = sizeArrayMaxSegBad[idx];
        optLen = sizeof(int);
        NN_LOG( "Testing: (SOL_TCP, TCP_MAXSEG, Value: %d)\n", optVal);
        PRINT_AND_CALL(rc = nn::socket::SetSockOpt(tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_MaxSeg, &optVal, optLen));
        ERROR_IF_AND_COUNT(rc > -1,"SetSockOpt Failed (SOL_TCP, TCP_MAXSEG, Value: %d) - Errno=<%d>\n", sizeArrayMaxSegBad[idx], nn::socket::GetLastError());
        NN_LOG("==> (SetSockOpt - Failed - Errno=<%d>) - This is correct, TEST PASSED!\n", nn::socket::GetLastError());

        nn::socket::Close(tcpsock);
        tcpsock = -1;
    }

WRAP_FAILING_TEST( "SIGLO-76934", "[NX] TCP Max Segment Size (MSS) refuses to set" )
{

    ////////////////
    // TCP Client tests - Good
    ///////////////

    for(idx = 0; idx < sizeArraySizeMaxSegGood; idx++)
    {
        // Client: Socket
        tcpsock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
        ERROR_IF_AND_COUNT(tcpsock < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError());

        // Client: SetSockOpt() - BEFORE Connect!
        optVal = sizeArrayMaxSegGood[idx];
        optLen = sizeof(int);
        NN_LOG( "Testing: (SOL_TCP, TCP_MAXSEG, Value: %d)\n", optVal);
        PRINT_AND_CALL(rc = nn::socket::SetSockOpt(tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_MaxSeg, &optVal, optLen));
        ERROR_IF_AND_COUNT(rc < 0,"SetSockOpt Failed (SOL_TCP, TCP_MAXSEG, Value: %d) - Errno=<%d>\n", sizeArrayMaxSegGood[idx], nn::socket::GetLastError());

        // Client: GetSockOpt() - BEFORE Connect!
        optVal = 0;
        optLen = sizeof(int);
        PRINT_AND_CALL(rc = nn::socket::GetSockOpt(tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_MaxSeg, &optVal, &optLen));
        ERROR_IF_AND_COUNT(rc < 0,"GetSockOpt Failed (SOL_TCP, TCP_MAXSEG) - Errno=<%d>\n", nn::socket::GetLastError());
        NN_LOG("GetSockOpt Returns: intoptval: %d, intoptlen: %d\n", optVal, optLen);

        // Client: Address
        memset(&remoteAddr, 0, sizeof(remoteAddr));
        remoteAddr.sin_port        = nn::socket::InetHtons(g_echoServer->kEchoServerTCPPort);
        remoteAddr.sin_family      = nn::socket::Family::Af_Inet;

        // Client: Turn IP Address into Network Address
        rc = nn::socket::InetPton(nn::socket::Family::Af_Inet, "127.0.0.1", &remoteAddr.sin_addr.S_addr );
        ERROR_IF_AND_COUNT(rc != 1, "InetPton() Failed but Success was expected!" );

        // Tell Tester what we are doing
        NN_LOG( "Establishing a TCP Connection to [%s:%d]\n", "127.0.0.1", g_echoServer->kEchoServerTCPPort);

        // Client: Connect
        rc = nn::socket::Connect(tcpsock, reinterpret_cast<nn::socket::SockAddr *>(&remoteAddr), sizeof(remoteAddr) );
        ERROR_IF_AND_COUNT(rc < 0, "Connect failed to loopback echo server!  Testing MSS\n");

        optLen = sizeof( myTCPInfo );
        rc = nn::socket::GetSockOpt( tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_Info, (void *) &myTCPInfo, &optLen);
        ERROR_IF_AND_COUNT(rc < 0, "GetSockOpt failed for option nn::socket::Option::Tcp_Info - errno: %d\n", nn::socket::GetLastError());
        ERROR_IF_AND_COUNT(myTCPInfo.tcpi_snd_mss != sizeArrayMaxSegGood[idx], "Set Max Segment Size to %d, but got size: %d back!\n",
                           sizeArrayMaxSegGood[idx], myTCPInfo.tcpi_snd_mss );
        NN_LOG( "GetSockOpt() - TCPInfo: %d - TEST PASSED!\n", myTCPInfo.tcpi_snd_mss );

        nn::socket::Close(tcpsock);
        tcpsock = -1;
    }

}   // END - WRAP FAILING TEST

out:

    if ( tcpsock > -1 )
    {
        nn::socket::Close( tcpsock );
        tcpsock = -1;
    }

    if ( listensock > -1 )
    {
        nn::socket::Close( listensock );
        listensock = -1;
    }

    if ( acceptedsock > -1 )
    {
        nn::socket::Close( acceptedsock );
        acceptedsock = -1;
    }
}


TEST(ApiUnit,SetSockOpt_TCP_CONGESTION)
{
    int                    tcpsock = -1, listensock = -1, acceptedsock = -1;
    int                    rc = -1;
    bool                   isSuccess = true, rval = false;
    nn::socket::SockLenT   optLen = 0;
    char                   buf1[nn::socket::Tcp_Ca_Name_Max + 1];

    ///////////////////
    // Make TCP Client and Server
    ///////////////////

    rval = NATF::API::COMMON::MakeListenSocket( listensock, 9020, "127.0.0.1" );
    ERROR_IF_AND_COUNT( rval == false, "Failed calling MakeListensocket!  Errno:%d\n", nn::socket::GetLastError() );

    tcpsock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 9020 );
    ERROR_IF_AND_COUNT( tcpsock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    acceptedsock = nn::socket::Accept( listensock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT( acceptedsock < 0, "Failed calling Accept!  Errno:%d\n", nn::socket::GetLastError() );

    /////////////
    // GetSockOpt Information on Socket Connection
    /////////////

    memset( buf1, 0, sizeof( buf1 ) );
    optLen = nn::socket::Tcp_Ca_Name_Max;

    rc = nn::socket::GetSockOpt( tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_Congestion, (void *) buf1, &optLen );
    ERROR_IF( rc < 0, "GetSockOpt failed for option nn::socket::Option::Tcp_Congestion - errno: %d\n", nn::socket::GetLastError() );
    NN_LOG( "Congestion Algorithum len: %d, Val: [%.*s] returned!\n", optLen, optLen, buf1 );

    ERROR_IF_AND_COUNT(optLen < strlen( "newreno") || memcmp( "newreno", buf1, strlen( "newreno" ) ) != 0,
    "Expected TCP CONGESTION value 'newreno', but got algorithum: [%.*s] back!\n", optLen, buf1 );

    /////////////
    // Set cubic Congestion Algorithum
    /////////////

    memset( buf1, 0, sizeof( buf1 ) );
    memcpy( buf1, "cubic", strlen( "cubic" ) );
    optLen = strlen( buf1 );


    /////////////
    // Set TCP Congestion on Socket Connection
    /////////////

    rc = nn::socket::SetSockOpt( tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_Congestion, (void *) buf1, optLen );
    ERROR_IF( rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_MaxSeg - errno: %d\n", nn::socket::GetLastError() );

    /////////////
    // GetSockOpt Information on Socket Connection
    /////////////

    memset( buf1, 0, sizeof( buf1 ) );
    optLen = nn::socket::Tcp_Ca_Name_Max;

    rc = nn::socket::GetSockOpt( tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_Congestion, (void *) buf1, &optLen );
    ERROR_IF( rc < 0, "GetSockOpt failed for option nn::socket::Option::Tcp_Congestion - errno: %d\n", nn::socket::GetLastError() );
    NN_LOG( "Congestion Algorithum len: %d, Val: %.*s returned!\n", optLen, optLen, buf1 );

    ERROR_IF_AND_COUNT( optLen < strlen( "cubic" ) || memcmp( "cubic", buf1, strlen( "cubic" ) ) != 0,
    "Set TCP CONGESTION to 'cubic', but got algorithum: %.*s back!\n", optLen, buf1 );

out:
    if ( tcpsock > -1 )
    {
        nn::socket::Close( tcpsock );
        tcpsock = -1;
    }

    if ( listensock > -1 )
    {
        nn::socket::Close( listensock );
        listensock = -1;
    }

    if ( acceptedsock > -1 )
    {
        nn::socket::Close( acceptedsock );
        acceptedsock = -1;
    }
}


TEST(ApiUnit,SetSockOpt_TCP_NOPUSH)
{
    nn::socket::SockLenT    optLen = 0;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    size_t                  myResultSize = 0;
    int                     tcpsock = -1, listensock = -1, acceptedsock = -1;
    int                     rc = -1, optVal = -1, idx = 0;
    bool                    isSuccess = true, rval = false;
    char                    buf1[1024] = { 0 };

    ////////////////
    // GetSockOpt Tests for: TCP_NOPUSH
    ///////////////

    /// TCP
    tcpsock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, g_echoServer->kEchoServerTCPPort);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed to call MakeTCPConnection()\n");

    myResultSize = sizeof(g_SockOptValueResults) / sizeof(SockOptValueResult);
    for(idx = 0; idx < myResultSize; idx++)
    {
        NN_LOG("GetSockOpt:  [TCP Client]: TCP_NOPUSH - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoPush, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckBooleanSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: TCP_NOPUSH - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoPush, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckBooleanSetSockOpt()\n");
    }

    // Close TCP Socket
    nn::socket::Close(tcpsock);
    tcpsock = -1;

    ///////////////////
    // Make TCP Client and Server
    ///////////////////

    rval = NATF::API::COMMON::MakeListenSocket( listensock, 9020, "127.0.0.1" );
    ERROR_IF_AND_COUNT( rval == false, "Failed calling MakeListensocket!  Errno: %d\n", nn::socket::GetLastError() );

    tcpsock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 9020 );
    ERROR_IF_AND_COUNT( tcpsock < 0, "Failed calling MakeTCPConnection!  Errno: %d\n", nn::socket::GetLastError() );

    acceptedsock = nn::socket::Accept( listensock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT( acceptedsock < 0, "Failed calling Accept!  Errno: %d\n", nn::socket::GetLastError() );

    rc = nn::socket::Fcntl( acceptedsock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    ////////////
    // Client: Write a byte
    ////////////

    rc = nn::socket::Write( tcpsock, "X", 1 );
    ERROR_IF( rc != 1, "Client failed to Write 1 byte!" );

    ////////////
    // Server:  Receives a byte!
    ////////////

    rc = nn::socket::Read( acceptedsock, buf1, sizeof( buf1 ) ) ;
    ERROR_IF_AND_COUNT( rc != 1, "Server should have Received 1 byte, but got rc: %d\n", rc );
    NN_LOG( "(Before) NOPUSH - Received 1 byte\n" );

    /////////////
    // Set NO_PUSH on Client Socket
    /////////////

    optVal = 1;
    optLen = sizeof(int);
    rc = nn::socket::SetSockOpt( tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoPush, (void *) &optVal, optLen );
    ERROR_IF( rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_MaxSeg - errno: %d\n", nn::socket::GetLastError() );

    ////////////
    // Client: Write a byte
    ////////////

    rc = nn::socket::Write( tcpsock, "X", 1 );
    ERROR_IF( rc != 1, "Client failed to Write 1 byte!" );

    ////////////
    // Server:  (DO NOT) receive a byte!
    ////////////

    rc = nn::socket::Read( acceptedsock, buf1, sizeof( buf1 ) ) ;
    ERROR_IF_AND_COUNT(rc > -1, "Received 1 byte, but no bytes were expected!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT(myError != nn::socket::Errno::EAgain && myError != nn::socket::Errno::EWouldBlock,
                       "Server was expecting to BLOCK but instead received errno: %d!\n", myError );

    NN_LOG( "(After) NOPUSH - Received (no bytes)!\n" );

out:
    if ( tcpsock > -1 )
    {
        nn::socket::Close( tcpsock );
        tcpsock = -1;
    }

    if ( listensock > -1 )
    {
        nn::socket::Close( listensock );
        listensock = -1;
    }

    if ( acceptedsock > -1 )
    {
        nn::socket::Close( acceptedsock );
        acceptedsock = -1;
    }
}


TEST(ApiUnit,SetSockOpt_TCP_KEEPINIT)
{
    nn::socket::SockAddrIn      targetAddr = { 0 };
    nn::socket::SockLenT        optLen = 0;
    uint64_t                    startTime = 0, endTime = 0, deltaTime = 0;
    size_t                      myResultSize = 0;
    int                         cliSock = -1, tcpsock = -1, optVal = -1, rc = -1, idx = 0;
    bool                        isSuccess = true, rval = false;

    static SockOptValueResult SockOptKeepInitResults[] =
    {
        {-2147483648, false},
        {-1, false},
        {0, true},
        {1, true},
        {75000, true},
        {225000, true},
        {900000, true},
        {3600000, true},
        {4294966, true},
    };

    ////////////////
    // GetSockOpt Tests for: TCP_KEEPINIT
    ///////////////

    /// TCP
    tcpsock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, g_echoServer->kEchoServerTCPPort);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed to call MakeTCPConnection()\n");


    myResultSize = sizeof(SockOptKeepInitResults) / sizeof(SockOptValueResult);
    for(idx = 0; idx < myResultSize; idx++)
    {
        NN_LOG("GetSockOpt:  [TCP Client]: TCP_KEEPINIT - Test [Value: %d]\n", SockOptKeepInitResults[idx].value);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepInit, SockOptKeepInitResults[idx].result, SockOptKeepInitResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: TCP_KEEPINIT - Test [Value: %d]\n", SockOptKeepInitResults[idx].value);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepInit, SockOptKeepInitResults[idx].result, SockOptKeepInitResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");
    }

    // Close TCP Socket
    nn::socket::Close(tcpsock);
    tcpsock = -1;


    ///////////////////
    // Make TCP Client
    ///////////////////

    cliSock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);

    // Client: Target Address
    memset( &targetAddr, 0, sizeof( targetAddr ) );
    targetAddr.sin_port     = nn::socket::InetHtons( 8053 );
    targetAddr.sin_family   = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "172.16.1.1", &targetAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );


    ///////////
    // Set Client TCP Connect Time
    //
    // nn::socket::Option::Tcp_KeepInit - This setsockopt(2) option accepts a per-socket timeout argument of u_int
    // in seconds, for new, non-established TCP connections.
    ///////////

    optVal = 5;  // 5 seconds
    optLen = sizeof( int );
    rc = nn::socket::SetSockOpt( cliSock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepInit, (void *) &optVal, optLen );
    ERROR_IF( rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_KeepInit - errno: %d\n", nn::socket::GetLastError() );


    /////////////
    // GetSockOpt Information on Socket Connection
    /////////////

    optVal = 0;
    optLen = sizeof( int );
    rc = nn::socket::GetSockOpt( cliSock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepInit, (void *) &optVal, &optLen );
    ERROR_IF( rc < 0, "GetSockOpt failed for option nn::socket::Option::Tcp_KeepInit - errno: %d\n", nn::socket::GetLastError() );
    ERROR_IF_AND_COUNT( optVal != 5, "Set TCP KEEPINIT to 5, but got timeout: %d back!\n", optVal );


    /////////////
    // Perform the Connect!
    /////////////

    startTime = NATF::API::COMMON::GetTicksInSeconds();
    rc = nn::socket::Connect( cliSock, (nn::socket::SockAddr *) &targetAddr, sizeof( targetAddr ) );
    endTime = NATF::API::COMMON::GetTicksInSeconds();
    deltaTime = (endTime - startTime );

    NN_LOG( "Start Time: %ld, End Time: %ld, Elapsed Time: %ld\n", startTime, endTime, deltaTime );
    if ( deltaTime > 6 )
    {
        NN_LOG( "FAILED!  Expected a timeout time of 5 seconds, got: %d\n", deltaTime );
        goto out;
    }

    NN_LOG( "********************\n" );
    NN_LOG( "PASSED - Connect timed out in: %d seconds!\n", deltaTime );
    NN_LOG( "********************\n" );

    if ( rc > 0 )
    {
        NN_LOG( "FAILED!  Connect succeeded instead of failed!  Expected Connect failure\n" );
        goto out;
    }

out:
    if (tcpsock > -1 )
    {
        nn::socket::Close(tcpsock);
        tcpsock = -1;
    }

    if ( cliSock > -1 )
    {
        nn::socket::Close( cliSock );
        cliSock = -1;
    }
}


TEST(ApiUnit,SetSockOpt_TCP_KEEPINTVL)
{
    size_t                myResultSize = 0;
    int                   tcpsock = -1, listensock = -1, acceptedsock = -1;
    int                   rc = -1, optVal = -1, idx = 0;
    bool                  isSuccess = true, rval = false;
    nn::socket::SockLenT  optLen = 0;
    char                  buf1[1024];

    static SockOptValueResult SockOptKeepIntvlResults[] =
    {
        {-2147483648, false},
        {-1, false},
        {0, true},
        {1, true},
        {75000, true},
        {225000, true},
        {900000, true},
        {3600000, true},
        {4294966, true},
    };

    ////////////////
    // GetSockOpt Tests for: TCP_KEEPINTVL
    ///////////////

    /// TCP
    tcpsock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, g_echoServer->kEchoServerTCPPort);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed to call MakeTCPConnection()\n");

    myResultSize = sizeof(SockOptKeepIntvlResults) / sizeof(SockOptValueResult);
    for(idx = 0; idx < myResultSize; idx++)
    {
        NN_LOG("GetSockOpt:  [TCP Client]: TCP_KEEPINTVL - Test [Value: %d]\n", SockOptKeepIntvlResults[idx].value);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepIntvl, SockOptKeepIntvlResults[idx].result, SockOptKeepIntvlResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: TCP_KEEPINTVL - Test [Value: %d]\n", SockOptKeepIntvlResults[idx].value);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepIntvl, SockOptKeepIntvlResults[idx].result, SockOptKeepIntvlResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");
    }

    // Close TCP Socket
    nn::socket::Close(tcpsock);
    tcpsock = -1;

    ///////////////////
    // Make TCP Client and Server
    ///////////////////

    rval = NATF::API::COMMON::MakeListenSocket(listensock, 9020, "127.0.0.1" );
    ERROR_IF_AND_COUNT(rval == false, "Failed calling Makeilistensocket!  Errno:%d\n", nn::socket::GetLastError());

    tcpsock = NATF::API::COMMON::MakeTCPConnection("127.0.0.1", 9020);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError());

    acceptedsock = nn::socket::Accept(listensock, static_cast<nn::socket::SockAddr*>(nullptr), 0);
    ERROR_IF_AND_COUNT(acceptedsock < 0, "Failed calling Accept!  Errno:%d\n", nn::socket::GetLastError());

    rc = nn::socket::Fcntl(acceptedsock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock);
    ERROR_IF_AND_COUNT(rc < 0, "Failed to NON-BLOCKING I/O on accepted socket\n");

    ///////////////////
    // Establish Keep Interval
    ///////////////////

    optVal = 0;  // 0 idle seconds before probes are sent
    optLen = sizeof( int );
    rc = nn::socket::SetSockOpt( acceptedsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepIdle, (void *) &optVal, optLen );
    ERROR_IF( rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_KeepInit - errno: %d\n", nn::socket::GetLastError() );

    optVal = 3;  // 3 seconds between keep alive probes
    optLen = sizeof( int );
    rc = nn::socket::SetSockOpt( acceptedsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepIntvl, (void *) &optVal, optLen );
    ERROR_IF( rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_KeepIntvl - errno: %d\n", nn::socket::GetLastError() );

    optVal = 1;  // Drop the connection if even 1 probe is missed
    optLen = sizeof( int );
    rc = nn::socket::SetSockOpt( acceptedsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepCnt, (void *) &optVal, optLen );
    ERROR_IF( rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_KeepCnt - errno: %d\n", nn::socket::GetLastError() );


    ///////////////////
    // Test for 2 minutes
    ///////////////////

    NN_LOG( "Testing for two minutes....\n" );
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(120));


    /////////////////
    // Send Data on the Connection
    /////////////////

    rc = nn::socket::Write( tcpsock, "X", 1 );
    ERROR_IF( rc != 1, "Client failed to Write 1 byte!" );

    rc = nn::socket::Read( acceptedsock, buf1, 1 );
    ERROR_IF( rc != 1, "Server failed to Read 1 byte!" );


    NN_LOG( "*********************\n" );
    NN_LOG( "PASSED - nn::socket::Option::Tcp_KeepIntvl Succeeded\n" );
    NN_LOG( "*********************\n" );

out:

    if ( tcpsock > -1 )
    {
        nn::socket::Close( tcpsock );
        tcpsock = -1;
    }

    if ( listensock > -1 )
    {
        nn::socket::Close( listensock );
        listensock = -1;
    }

    if ( acceptedsock > -1 )
    {
        nn::socket::Close( acceptedsock );
        acceptedsock = -1;
    }
}


TEST(ApiUnit,SetSockOpt_TCP_KEEPIDLE)
{
    size_t                myResultSize = 0;
    int                   tcpsock = -1, listensock = -1, acceptedsock = -1;
    int                   rc = -1, optVal = -1, idx = 0;
    bool                  isSuccess = true, rval = false;
    nn::socket::SockLenT  optLen = 0;
    char                  buf1[1024];

    static SockOptValueResult SockOptKeepIdleResults[] =
    {
        {-2147483648, false},
        {-1, false},
        {0, true},
        {1, true},
        {75000, true},
        {225000, true},
        {900000, true},
        {3600000, true},
        {4294966, true}
    };

    ////////////////
    // GetSockOpt Tests for: TCP_KEEPIDLE
    ///////////////

    /// TCP
    tcpsock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, g_echoServer->kEchoServerTCPPort);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed to call MakeTCPConnection()\n");

    myResultSize = sizeof(SockOptKeepIdleResults) / sizeof(SockOptValueResult);
    for(idx = 0; idx < myResultSize; idx++)
    {
        NN_LOG("GetSockOpt:  [TCP Client]: TCP_KEEPIDLE - Test [Value: %d]\n", SockOptKeepIdleResults[idx].value);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepIdle, SockOptKeepIdleResults[idx].result, SockOptKeepIdleResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: TCP_KEEPIDLE - Test [Value: %d]\n", SockOptKeepIdleResults[idx].value);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepIdle, SockOptKeepIdleResults[idx].result, SockOptKeepIdleResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");
    }

    // Close TCP Socket
    nn::socket::Close(tcpsock);
    tcpsock = -1;

    ///////////////////
    // Make TCP Client and Server
    ///////////////////

    rval = NATF::API::COMMON::MakeListenSocket(listensock, 9020, "127.0.0.1" );
    ERROR_IF_AND_COUNT(rval == false, "Failed calling MakeListenSocket!  Errno:%d\n", nn::socket::GetLastError());

    tcpsock = NATF::API::COMMON::MakeTCPConnection("127.0.0.1", 9020);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError());

    acceptedsock = nn::socket::Accept(listensock, static_cast<nn::socket::SockAddr*>(nullptr), 0);
    ERROR_IF_AND_COUNT(acceptedsock < 0, "Failed calling Accept!  Errno:%d\n", nn::socket::GetLastError());

    rc = nn::socket::Fcntl(acceptedsock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock);
    ERROR_IF_AND_COUNT(rc < 0, "Failed to NON-BLOCKING I/O on accepted socket\n");

    ///////////////////
    // Establish Keep Interval
    ///////////////////

    optVal = 2;  // 2 idle seconds before probes are sent
    optLen = sizeof( int );
    rc = nn::socket::SetSockOpt( acceptedsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepIdle, (void *) &optVal, optLen );
    ERROR_IF( rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_KeepInit - errno: %d\n", nn::socket::GetLastError() );

    optVal = 3;  // 3 seconds between keep alive probes
    optLen = sizeof( int );
    rc = nn::socket::SetSockOpt( acceptedsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepIntvl, (void *) &optVal, optLen );
    ERROR_IF( rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_KeepIntvl - errno: %d\n", nn::socket::GetLastError() );

    optVal = 1;  // Drop the connection if even 1 probe is missed
    optLen = sizeof( int );
    rc = nn::socket::SetSockOpt( acceptedsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepCnt, (void *) &optVal, optLen );
    ERROR_IF( rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_KeepCnt - errno: %d\n", nn::socket::GetLastError() );


    ///////////////////
    // Test for 2 minutes
    ///////////////////

    NN_LOG( "Testing for two minutes....\n" );
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(120));

    /////////////////
    // Send Data on the Connection
    /////////////////

    rc = nn::socket::Write(tcpsock, "X", 1);
    ERROR_IF( rc != 1, "Client failed to Write 1 byte!");

    rc = nn::socket::Read(acceptedsock, buf1, 1);
    ERROR_IF( rc != 1, "Server failed to Read 1 byte!");


    NN_LOG( "*********************\n");
    NN_LOG( "PASSED - nn::socket::Option::Tcp_KeepIdle Succeeded\n");
    NN_LOG( "*********************\n");

out:

    if ( tcpsock > -1 )
    {
        nn::socket::Close( tcpsock );
        tcpsock = -1;
    }

    if ( listensock > -1 )
    {
        nn::socket::Close( listensock );
        listensock = -1;
    }

    if ( acceptedsock > -1 )
    {
        nn::socket::Close( acceptedsock );
        acceptedsock = -1;
    }
}


TEST(ApiUnit,SetSockOpt_TCP_KEEPCNT)
{
    size_t                myResultSize = 0;
    int                   tcpsock = -1, listensock = -1, acceptedsock = -1;
    int                   rc = -1, optVal = -1, idx = 0;
    bool                  isSuccess = true, rval = false;
    nn::socket::SockLenT  optLen = 0;
    char                  buf1[1024];

    static SockOptValueResult SockOptKeepCntResults[] =
    {
        {-2147483648, true},
        {-1, true},
        {0, true},
        {1, true},
        {75000, true},
        {INT_MAX, true},
        {static_cast<int>(UINT_MAX), true}
    };

    ////////////////
    // GetSockOpt Tests for: TCP_KEEPCNT
    ///////////////

    /// TCP
    tcpsock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, g_echoServer->kEchoServerTCPPort);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed to call MakeTCPConnection()\n");

    myResultSize = sizeof(SockOptKeepCntResults) / sizeof(SockOptValueResult);
    for(idx = 0; idx < myResultSize; idx++)
    {
        NN_LOG("GetSockOpt:  [TCP Client]: TCP_KEEPCNT - Test [Value: %d]\n", SockOptKeepCntResults[idx].value);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepCnt, SockOptKeepCntResults[idx].result, SockOptKeepCntResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: TCP_KEEPCNT - Test [Value: %d]\n", SockOptKeepCntResults[idx].value);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepCnt, SockOptKeepCntResults[idx].result, SockOptKeepCntResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");
    }

    // Close TCP Socket
    nn::socket::Close(tcpsock);
    tcpsock = -1;

    ///////////////////
    // Make TCP Client and Server
    ///////////////////

    rval = NATF::API::COMMON::MakeListenSocket(listensock, 9020, "127.0.0.1");
    ERROR_IF_AND_COUNT(rval == false, "Failed calling MakeListenSocket!  Errno:%d\n", nn::socket::GetLastError());

    tcpsock = NATF::API::COMMON::MakeTCPConnection("127.0.0.1", 9020);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError());

    acceptedsock = nn::socket::Accept(listensock, static_cast<nn::socket::SockAddr*>(nullptr), 0);
    ERROR_IF_AND_COUNT(acceptedsock < 0, "Failed calling Accept!  Errno:%d\n", nn::socket::GetLastError());

    rc = nn::socket::Fcntl(acceptedsock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock);
    ERROR_IF_AND_COUNT(rc < 0, "Failed to NON-BLOCKING I/O on accepted socket\n");

    ///////////////////
    // Establish Keep Interval
    ///////////////////

    optVal = 2;  // 2 idle seconds before probes are sent
    optLen = sizeof( int );
    rc = nn::socket::SetSockOpt(acceptedsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepIdle, (void *) &optVal, optLen);
    ERROR_IF( rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_KeepInit - errno: %d\n", nn::socket::GetLastError());

    optVal = 3;  // 3 seconds between keep alive probes
    optLen = sizeof( int );
    rc = nn::socket::SetSockOpt(acceptedsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepIntvl, (void *) &optVal, optLen);
    ERROR_IF( rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_KeepIntvl - errno: %d\n", nn::socket::GetLastError());

    optVal = 1;  // Drop the connection if even 1 probe is missed
    optLen = sizeof( int );
    rc = nn::socket::SetSockOpt(acceptedsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepCnt, (void *) &optVal, optLen );
    ERROR_IF( rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_KeepCnt - errno: %d\n", nn::socket::GetLastError());


    ///////////////////
    // Test for 2 minutes
    ///////////////////

    NN_LOG( "Testing for two minutes....\n" );
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(120));

    /////////////////
    // Send Data on the Connection
    /////////////////

    rc = nn::socket::Write(tcpsock, "X", 1);
    ERROR_IF( rc != 1, "Client failed to Write 1 byte!");

    rc = nn::socket::Read(acceptedsock, buf1, 1);
    ERROR_IF( rc != 1, "Server failed to Read 1 byte!");


    NN_LOG( "*********************\n");
    NN_LOG( "PASSED - nn::socket::Option::Tcp_KeepCnt Succeeded\n");
    NN_LOG( "*********************\n");

out:

    if (tcpsock > -1)
    {
        nn::socket::Close(tcpsock);
        tcpsock = -1;
    }

    if (listensock > -1)
    {
        nn::socket::Close(listensock);
        listensock = -1;
    }

    if (acceptedsock > -1)
    {
        nn::socket::Close(acceptedsock);
        acceptedsock = -1;
    }

}


TEST(ApiUnit,SetSockOpt_TCP_NOOPT)
{
    nn::socket::TcpInfo    myTCPInfo = {};
    size_t                 myResultSize = 0;
    int                    tcpsock = -1, listensock = -1, acceptedsock = -1;
    int                    rc = -1, optVal = -1, idx = 0;
    bool                   isSuccess = true, rval = false;
    nn::socket::SockLenT   optLen = 0;

    ////////////////
    // GetSockOpt Tests for: TCP_NOOPT
    ///////////////

    /// TCP
    tcpsock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, g_echoServer->kEchoServerTCPPort);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed to call MakeTCPConnection()\n");

    myResultSize = sizeof(g_SockOptValueResults) / sizeof(SockOptValueResult);
    for(idx = 0; idx < myResultSize; idx++)
    {
        NN_LOG("GetSockOpt:  [TCP Client]: TCP_NOOPT - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoOpt, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckBooleanSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: TCP_NOOPT - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoOpt, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckBooleanSetSockOpt()\n");
    }

    // Close TCP Socket
    nn::socket::Close(tcpsock);
    tcpsock = -1;

WRAP_FAILING_TEST( "SIGLO-76934", "[NX] TCP Max Segment Size (MSS) refuses to set" )
{
    // This test needs to pass the TCP max segment size (MSS) value in the TCP header options.
    // However a bug in this setting prevents the updated (MSS) from being set into the TCP
    // header options.  When this bug is fixed - re-enable this test.


    ///////////////////
    // Make TCP Server with Max Segment Size (MAXGEG)
    ///////////////////

    rval = NATF::API::COMMON::MakeListenSocket(listensock, 9020, "127.0.0.1");
    ERROR_IF_AND_COUNT(rval == false, "Failed calling MakeListenSocket!  Errno:%d\n", nn::socket::GetLastError());

    tcpsock = NATF::API::COMMON::MakeTCPConnection("127.0.0.1", 9020);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError());

    optVal = 1024;
    optLen = sizeof( int );
    rc = nn::socket::SetSockOpt(tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_MaxSeg, (void *) &optVal, optLen);
    ERROR_IF(rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_MaxSeg - errno: %d\n", nn::socket::GetLastError());


    /////////////
    // SetSockOpt nn::socket::Option::Tcp_NoOpt
    /////////////

    optVal = 1;
    optLen = sizeof( int );
    rc = nn::socket::SetSockOpt(tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoOpt, (void *) &optVal, optLen);
    ERROR_IF(rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_NoOpt - errno: %d\n", nn::socket::GetLastError());


    /////////////
    // GetSockOpt nn::socket::Option::Tcp_NoOpt
    /////////////

    optVal = 0;
    optLen = sizeof( int );
    rc = nn::socket::GetSockOpt( tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoOpt, (void *) &optVal, &optLen);
    ERROR_IF(rc < 0, "GetSockOpt failed for option nn::socket::Option::Tcp_NoOpt - errno: %d\n", nn::socket::GetLastError());
    ERROR_IF_AND_COUNT(optVal == 0, "Set nn::socket::Option::Tcp_NoOpt to 1, but got setting: %d back!\n", optVal);


    ///////////////////
    // Establish Connection
    ///////////////////

    acceptedsock = nn::socket::Accept(listensock, static_cast<nn::socket::SockAddr*>(nullptr), 0);
    ERROR_IF_AND_COUNT(acceptedsock < 0, "Failed calling Accept!  Errno:%d\n", nn::socket::GetLastError());

    /////////////
    // nn::socket::Option::Tcp_Info Information on Socket Connection
    /////////////

    optLen = sizeof( myTCPInfo );
    rc = nn::socket::GetSockOpt(acceptedsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_Info, (void *) &myTCPInfo, &optLen);
    ERROR_IF_AND_COUNT(rc < 0, "GetSockOpt failed for option nn::socket::Option::Tcp_Info - errno: %d\n", nn::socket::GetLastError());
    ERROR_IF_AND_COUNT(myTCPInfo.tcpi_snd_mss != 16332, "Expected TCP default MAXSEG size, got size: %d back!\n", myTCPInfo.tcpi_snd_mss);
    NN_LOG( "[Default]: Max Segment Size: [%d]\n", myTCPInfo.tcpi_snd_mss );
}

out:

    if ( tcpsock > -1 )
    {
        nn::socket::Close( tcpsock );
        tcpsock = -1;
    }

    if ( listensock > -1 )
    {
        nn::socket::Close( listensock );
        listensock = -1;
    }

    if ( acceptedsock > -1 )
    {
        nn::socket::Close( acceptedsock );
        acceptedsock = -1;
    }
}


TEST(ApiUnit,SetSockOpt_TCP_NODELAY)
{
    nn::socket::SockLenT    optLen = 0;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    size_t                  myResultSize = 0;
    int                     tcpsock = -1, listensock = -1, acceptedsock = -1;
    int                     rc = -1, optVal = -1, idx = 0;
    bool                    isSuccess = true, rval = false;
    char                    buf1[1024] = { 0 };

    ////////////////
    // GetSockOpt Tests for: TCP_NODELAY
    ///////////////

    /// TCP
    tcpsock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, g_echoServer->kEchoServerTCPPort);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed to call MakeTCPConnection()\n");

    myResultSize = sizeof(g_SockOptValueResults) / sizeof(SockOptValueResult);
    for(idx = 0; idx < myResultSize; idx++)
    {
        NN_LOG("GetSockOpt:  [TCP Client]: TCP_NODELAY - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoDelay, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckBooleanSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: TCP_NODELAY - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoDelay, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckBooleanSetSockOpt()\n");
    }

    // Close TCP Socket
    nn::socket::Close(tcpsock);
    tcpsock = -1;

    ////////////////
    // Make Client / Server
    ////////////////

    rval = NATF::API::COMMON::MakeListenSocket(listensock, 9020, "127.0.0.1");
    ERROR_IF_AND_COUNT(rval == false, "Failed calling MakeListenSocket!  Errno: %d\n", nn::socket::GetLastError());

    tcpsock = NATF::API::COMMON::MakeTCPConnection("127.0.0.1", 9020);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed calling MakeTCPConnection!  Errno: %d\n", nn::socket::GetLastError());

    acceptedsock = nn::socket::Accept(listensock, static_cast<nn::socket::SockAddr*>(nullptr), 0);
    ERROR_IF_AND_COUNT(acceptedsock < 0, "Failed calling Accept!  Errno: %d\n", nn::socket::GetLastError());

    rc = nn::socket::Fcntl(acceptedsock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock);
    ERROR_IF_AND_COUNT(rc < 0, "Failed to NON-BLOCKING I/O on accepted socket\n");

    /////////////
    // TCP CLIENT:  Test TCP_NODELAY - ON / ENABLED
    /////////////

    optVal = 1;  // Enable: NO DELAY
    optLen = sizeof( int );
    rc = nn::socket::SetSockOpt(tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoDelay, (void *) &optVal, optLen);
    ERROR_IF(rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_NoDelay - errno: %d\n", nn::socket::GetLastError());

    NN_LOG("TCP CLIENT: Read [");
    for(idx = 0; idx < 100; idx++)
    {
        rc = nn::socket::Write(tcpsock, "X", 1);
        ERROR_IF_AND_COUNT(rc != 1, "Failed calling (TCP Client Write)!  rc: %d, Errno: %d\n", rc, nn::socket::GetLastError());

        rc = nn::socket::Read(acceptedsock, buf1, 1);
        ERROR_IF_AND_COUNT(rc != 1, "Failed calling (TCP Server Read)!  rc: %d, Errno: %d\n", rc, nn::socket::GetLastError());
        NN_LOG( "%c", buf1[0]);
    }
    NN_LOG("]\n");

    /////////////
    // TCP CLIENT:  Test TCP_NODELAY - OFF / DISABLED
    /////////////

    optVal = 0;  // Disable: NO DELAY
    optLen = sizeof( int );
    rc = nn::socket::SetSockOpt(tcpsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoDelay, (void *) &optVal, optLen);
    ERROR_IF(rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_NoDelay - errno: %d\n", nn::socket::GetLastError());

    NN_LOG("TCP CLIENT: Read [");
    for(idx = 0; idx < 100; idx++)
    {
        rc = nn::socket::Write(tcpsock, "X", 1);
        ERROR_IF_AND_COUNT(rc != 1, "Failed calling (TCP Client Write)!  rc: %d, Errno: %d\n", rc, nn::socket::GetLastError());

        rc = nn::socket::Read(acceptedsock, buf1, 1);
        if (rc < 0)
        {
            myError = nn::socket::GetLastError();
            if (myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock)
            {
                NN_LOG( "====> PASSED!  TCP CLIENT: Sender (batch sent) more than 1 queued byte!\n");
                ++NATF::API::nTestsPassing;
                break;
            }
            ERROR_IF_AND_COUNT(rc < 0, "Failed calling (TCP Server Read)!  rc: %d, Errno: %d\n", rc, nn::socket::GetLastError());
        }
        NN_LOG( "%c", buf1[0]);
    }
    NN_LOG("]\n");

    /////////////
    // TCP SERVER:  Test TCP_NODELAY - ON / ENABLED
    /////////////

    optVal = 1;  // Enable: NO DELAY
    optLen = sizeof( int );
    rc = nn::socket::SetSockOpt(acceptedsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoDelay, (void *) &optVal, optLen);
    ERROR_IF(rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_NoDelay - errno: %d\n", nn::socket::GetLastError());

    NN_LOG("TCP SERVER: Read [");
    for(idx = 0; idx < 100; idx++)
    {
        rc = nn::socket::Write(acceptedsock, "X", 1);
        ERROR_IF_AND_COUNT(rc != 1, "Failed calling (TCP Server Write)!  rc: %d, Errno: %d\n", rc, nn::socket::GetLastError());

        rc = nn::socket::Read(tcpsock, buf1, 1);
        ERROR_IF_AND_COUNT(rc != 1, "Failed calling (TCP Client Read)!  rc: %d, Errno: %d\n", rc, nn::socket::GetLastError());
        NN_LOG( "%c", buf1[0]);
    }
    NN_LOG("]\n");

    /////////////
    // TCP SERVER:  Test TCP_NODELAY - OFF / DISABLED
    /////////////

    optVal = 0;  // Disable: NO DELAY
    optLen = sizeof( int );
    rc = nn::socket::SetSockOpt(acceptedsock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoDelay, (void *) &optVal, optLen);
    ERROR_IF(rc < 0, "SetSockOpt failed for option nn::socket::Option::Tcp_NoDelay - errno: %d\n", nn::socket::GetLastError());

    NN_LOG("TCP SERVER: Read [");
    for(idx = 0; idx < 100; idx++)
    {
        rc = nn::socket::Write(acceptedsock, "X", 1);
        ERROR_IF_AND_COUNT(rc != 1, "Failed calling (TCP Server Write)!  rc: %d, Errno: %d\n", rc, nn::socket::GetLastError());

        rc = nn::socket::Read(tcpsock, buf1, 1);
        if (rc < 0)
        {
            myError = nn::socket::GetLastError();
            if (myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock)
            {
                NN_LOG( "====> PASSED!  TCP SERVER: (batch sent) more than 1 queued byte!\n");
                ++NATF::API::nTestsPassing;
                break;
            }
            ERROR_IF_AND_COUNT(rc < 0, "Failed calling (TCP Client Read)!  rc: %d, Errno: %d\n", rc, nn::socket::GetLastError());
        }
        NN_LOG( "%c", buf1[0]);
    }
    NN_LOG("]\n");


out:
    if (tcpsock > -1)
    {
        nn::socket::Close(tcpsock);
        tcpsock = -1;
    }

    if (listensock > -1)
    {
        nn::socket::Close(listensock);
        listensock = -1;
    }

    if (acceptedsock > -1)
    {
        nn::socket::Close(acceptedsock);
        acceptedsock = -1;
    }

}      // NOLINT(impl/function_size)


TEST(ApiUnit,SetSockOpt_TCP_Teardown)
{
    TeardownTesting();
}

#else // Windows

TEST(ApiUnit,SetSockOpt_TCP_Windows)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}
#endif

}}  // Namespace: NATF::API
