﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>

#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiIpAddress.h>          // nn::nifm::GetCurrentIpConfigInfo
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions

namespace NATF {
namespace API {

//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32

//*********************
//*  G L O B A L S
//*********************

static const int            kThreadPriority = 10;       // Thread Priority
static const int            kMaxNumOfSockets = 3;       // Max test sockets

static ThreadedController * g_pThreadedTest = NULL;
static int                  g_echoServerThreadId = -1;
static EchoServer *         g_echoServer = NULL;


static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}


static bool
InitializeTesting()
{
    bool        isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumOfSockets );
    // Create a thread to run the Server in

    g_echoServerThreadId = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, kThreadPriority );
    if (  g_echoServerThreadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // If the loopback DNS server is allocated
    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait for Echo Server thread to die
    g_pThreadedTest->WaitThread( g_echoServerThreadId );

    // Destroy Echo Server Thread
    g_pThreadedTest->DestroyThread( g_echoServerThreadId );

    // Delete Echo Server
    delete g_echoServer;
    g_echoServer = NULL;

    // Delete Threaded Test
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;
    g_echoServerThreadId = -1;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );

out:

    return( false );
}



////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,Send_Initialize)
{
    InitializeTesting();
}


// Anonymous namespace
namespace {

//////////////////////////////////////////
// $ dig google.com
//
// ; <<>> DiG 9.10.3-P4 <<>> google.com
// ;; global options: +cmd
// ;; Got answer:
// ;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 43551
// ;; flags: qr rd ra; QUERY: 1, ANSWER: 1, AUTHORITY: 0, ADDITIONAL: 1
//
// ;; OPT PSEUDOSECTION:
// ; EDNS: version: 0, flags:; udp: 4096
// ;; QUESTION SECTION:
// ;google.com.                    IN      A
//
// ;; ANSWER SECTION:
// google.com.             255     IN      A       172.217.3.174
//
// ;; Query time: 4 msec
// ;; SERVER: 10.1.19.29#53(10.1.19.29)
// ;; WHEN: Wed Oct 26 09:05:02 PDT 2016
// ;; MSG SIZE  rcvd: 55
//
//////////////////////////////////////////

static const char GoogleDNSRequest[] =
{
    0xaa, 0x1f, 0x01, 0x20, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x06, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x03, 0x63, 0x6f, 0x6d, 0x00,
    0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x29, 0x10, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00
};

}; // End - Anonymous namespace


TEST(ApiUnit,Send_Various_Tests)
{
    nn::socket::SockAddrIn  localAddr = { 0 }, outAddr = { 0 }, targetAddr = { 0 };
    nn::socket::SockLenT    outAddrLen = sizeof( outAddr );
    int                     tcpSock = -1, udpSock = -1, udpServSock = -1;
    int                     rc = -1;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    int                     idx;
    bool                    isSuccess = true;

    ////////////
    //  Establish UDP Target
    ///////////

    // Client: Target Address
    memset(&outAddr, 0, sizeof(outAddr));
    outAddr.sin_port        = nn::socket::InetHtons( 8053 );
    outAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &outAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    ////////////
    // Test nn::socket::Errno::EBadf
    ///////////

    NN_LOG( "Testing descriptor:" );
    for ( idx = -2; idx < 40; idx++ )
    {
        if  ( ( idx == g_echoServer->serverTCP )  ||
              ( idx == g_echoServer->serverUDP )  )
        {
            NN_LOG( "(Skip: %d) ", idx );
            continue;
        }

        NN_LOG( "%d, ", idx );

        outAddrLen = sizeof( nn::socket::SockAddr );
        rc = nn::socket::Send( idx, "X", 1, nn::socket::MsgFlag::Msg_None );
        if ( rc > 0 )
        {
            NN_LOG( "\nSend() succeeded but should have failed processing descriptor: %d\n", idx );
            goto out;
        }

        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError != nn::socket::Errno::EBadf )
            {
                NN_LOG( "\nSend() bad fds: %d should have failed with nn::socket::Errno::EBadf, but actually failed with: %d\n", idx, myError );
                goto out;
            }

            continue;  // Jump to top
        }

        if ( rc == 0 )
        {
            NN_LOG( "\nSend() fds: %d:  Succeeded but sent (no bytes)!\n", idx );
            goto out;
        }
    }

    NN_LOG( "\n" );


    ////////////
    // Make TCP Connection
    ///////////

    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( tcpSock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // Make UDP Connection
    ///////////

    udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9000, "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( udpSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // [TCP] - Test NULL Send Buffer
    ///////////

    NN_LOG( "Testing: [TCP] - NULL Send Buffer - nn::socket::Errno::EInval\n" );

    outAddrLen = sizeof( outAddr );
    rc = nn::socket::Send( tcpSock, nullptr, 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1, "Send() succeeded but should have failed! [TCP]\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "Send() expected Errno: nn::socket::Errno::EInval. actually got: %d\n", myError );


    ////////////
    // [UDP] - Test NULL Send Buffer
    ///////////

    NN_LOG( "Testing: [UDP] - NULL Send Buffer - nn::socket::Errno::EInval\n" );

    outAddrLen = sizeof( outAddr );
    rc = nn::socket::Send( udpSock, nullptr, 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP]: Send() succeeded but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "Send() expected Errno: nn::socket::Errno::EInval. actually got: %d\n", myError );


    ////////////
    // [UDP] - Test NULL No Address Length
    ///////////

    NN_LOG( "Testing: [UDP] - (No Address Len)\n" );

    outAddrLen = 0;
    rc = nn::socket::Send( udpSock, "X", 0, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc != 0, "[UDP] Send() failed but should have succeeded! Errno: %d\n", nn::socket::GetLastError() );



    ////////////
    // [UDP] - Test NULL (Negative) Address Length
    ///////////

    NN_LOG( "Testing: [UDP] - (Negative Len == -1)\n" );

    outAddrLen = 0;
    rc = nn::socket::Send( udpSock, "X", -1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc >= 0, "[UDP] Send() Succeeded but should have failed!\n" );


    ////////////
    // [UDP] - Connect established destination - Same Address
    ///////////

    NN_LOG( "Testing: [UDP]- Connect() establishes destination - Same Address\n" );

    outAddrLen = 0;
    rc = nn::socket::Send( udpSock, "X", 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc < 0, "[UDP] Send() failed but should have Suceeded!  Errno: %d\n", nn::socket::GetLastError() );


    ////////////
    // [UDP] - Exceed UDP Maximum Transmit size - nn::socket::Errno::EMsgSize
    ////////////

    NN_LOG( "Testing: [UDP]- Exceed Maximum Transmit Size - nn::socket::Errno::EMsgSize\n" );

    outAddrLen = 65600;  // Something greater than 65535
    rc = nn::socket::Send( udpSock, "X", outAddrLen, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP] Send() Suceeded but should have failed!\n" );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EMsgSize, "[UDP] Send() expected errno: nn::socket::Errno::EMsgSize, actually got Errno: %d\n", myError );


    ////////////
    // [UDP]
    //
    // ECONNNREFUSED - The socket received an ICMP destination unreachable from the last
    // message sent.
    ////////////

    NN_LOG( "Testing: [UDP]- Disconnected UDP Server - nn::socket::Errno::EConnRefused\n" );

    {
        // UDP Server Sock
        udpServSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9050, "127.0.0.1", 9060 );
        ERROR_IF_AND_COUNT( udpServSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );

        // UDP Client - Close
        nn::socket::Close( udpSock );

        // UDP Client - Connect to Server
        udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9060, "127.0.0.1", 9050 );
        ERROR_IF_AND_COUNT( udpSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );

        // Write from Client to Server
        nn::socket::Send( udpSock, "x", 1, nn::socket::MsgFlag::Msg_None);

        // Close UDP Server
        nn::socket::Close( udpServSock );
        udpServSock = -1;

        // Client keeps writing - but expected to FAIL
        for( idx = 0; idx < 10; idx++ )
        {
            rc = nn::socket::Send( udpSock, "X", 1, nn::socket::MsgFlag::Msg_None );
            if ( rc < 0 )
            {
                myError = nn::socket::GetLastError();
                if ( myError == nn::socket::Errno::EConnRefused )
                {
                    NN_LOG( "PASSED!  Send failed with nn::socket::Errno::EConnRefused!\n" );
                    break;
                }

                NN_LOG( "Send failed with errno: %d\n", myError );
            }

            NN_LOG( "Good Send to closed UDP socket...\n" );
        }

        if ( idx >= 10 )
        {
            NN_LOG( "FAILED!  Send succeeded 10 times but should have failed with nn::socket::Errno::EConnRefused!\n" );
            goto out;
        }
    }


    ////////////
    // [UDP]
    //
    // nn::socket::MsgFlag::Msg_DontRoute - If an interface is not directly connected to the NX don't follow
    // the default route.
    ////////////

    {
        nn::socket::InAddr myIp = { 0 };
        nn::socket::InAddr myMask = { 0 };
        nn::socket::InAddr myGw = { 0 };
        nn::socket::InAddr myDns1 = { 0 };
        nn::socket::InAddr myDns2 = { 0 };


        NN_LOG( "Testing: [UDP] - nn::socket::MsgFlag::Msg_DontRoute - (Errno: nn::socket::Errno::EHostDown, nn::socket::Errno::ENetDown, etc)\n" );

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

        // UDP Client - Close
        nn::socket::Close( udpSock );

        // UDP Client - Create
        udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

        // UDP Client - Local Address
        memset( &localAddr, 0, sizeof(localAddr));
        localAddr.sin_port     = nn::socket::InetHtons( 9004 );
        localAddr.sin_family   = nn::socket::Family::Af_Inet;

        // UDP - Client - IP
        NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
        localAddr.sin_addr.S_addr = myIp.S_addr;

        // UDP Client - Local Bind
        rc = nn::socket::Bind( udpSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
        ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

        // UDP - Target Address
        memset( &targetAddr, 0, sizeof(targetAddr));
        targetAddr.sin_port     = nn::socket::InetHtons( 53 );   // DNS == domain == 53
        targetAddr.sin_family   = nn::socket::Family::Af_Inet;

        // UDP - Target IP: (E-ROOT)
        rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "192.203.230.10", &targetAddr.sin_addr.S_addr );
        ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

        // Call Connect to Bind Servers IP as Peer
        rc = nn::socket::Connect( udpSock,(nn::socket::SockAddr *) &targetAddr, sizeof( targetAddr ) );
        ERROR_IF( rc != 0, "[UDP] Connect failed for UDP Peer socket!  Errno: %d\n", nn::socket::GetLastError() );

        // Send test data which is not (directly) routable from NX
        rc = nn::socket::Send( udpSock, "Test Data", strlen( "Test Data" ), nn::socket::MsgFlag::Msg_DontRoute );
        ERROR_IF_AND_COUNT( rc > 0, "[UDP] Send() succeeded but should have failed!\n" );

        myError = nn::socket::GetLastError();
        switch( myError )
        {
        case nn::socket::Errno::EHostUnreach:
        case nn::socket::Errno::EHostDown:
        case nn::socket::Errno::ENetDown:
        case nn::socket::Errno::EConnRefused:

            break;   // All is well

        default:
            NN_LOG( "Unexpected Errno: %d received, expected one of the 4x legal network ones\n", myError );
            goto out;

        }
    }

    ////////////
    // [UDP]
    //
    // Client to Server on NIFM LAN Interface
    ////////////

    NN_LOG( "Testing: [UDP] - Send Direct Send to Peer on Lan Iface - Valid)\n" );

    {
        nn::socket::InAddr myIp = { 0 };
        nn::socket::InAddr myMask = { 0 };
        nn::socket::InAddr myGw = { 0 };
        nn::socket::InAddr myDns1 = { 0 };
        nn::socket::InAddr myDns2 = { 0 };
        nn::socket::SockAddrIn remoteHost = { 0 }, servAddr = { 0 };
        nn::socket::TimeVal myTime = { 0 };
        nn::socket::SockLenT optlen;
        char dnsRequest[512];

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

        // UDP Server Sock
        udpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

        // UDP Server - Local Address
        memset( &servAddr, 0, sizeof(servAddr));
        servAddr.sin_port        = nn::socket::InetHtons( 9056 );
        servAddr.sin_family      = nn::socket::Family::Af_Inet;

        NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
        servAddr.sin_addr.S_addr = myIp.S_addr;

        // UDP Server - Don't wait forever..
        myTime.tv_sec  = 5;
        myTime.tv_usec = 0;
        optlen         = sizeof( myTime );

        rc = nn::socket::SetSockOpt( udpServSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo,(void *) &myTime, optlen );
        ERROR_IF( rc < 0, "SetSockOpt for nn::socket::Option::So_RcvTimeo failed - Errno: %d\n", nn::socket::GetLastError() );

        // UDP Server - Local Bind
        rc = nn::socket::Bind( udpServSock, (nn::socket::SockAddr *) &servAddr, sizeof( servAddr ) );
        ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

        // UDP Client - Close
        nn::socket::Close( udpSock );

        // UDP Client - Create
        udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

        // UDP Client - Local Address
        memset( &localAddr, 0, sizeof(localAddr));
        localAddr.sin_port        = nn::socket::InetHtons( 9057 );
        localAddr.sin_family      = nn::socket::Family::Af_Inet;

        NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
        localAddr.sin_addr.S_addr = myIp.S_addr;

        // UDP Client - Local Bind
        rc = nn::socket::Bind( udpSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
        ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

        // UDP - Target Address
        memset( &targetAddr, 0, sizeof(targetAddr));
        targetAddr.sin_port     = nn::socket::InetHtons( 9056 );   // Point to UDP Server
        targetAddr.sin_family   = nn::socket::Family::Af_Inet;

        // UDP - Target IP: LAN DIRECTED BROADCAST ADDRESS
        targetAddr.sin_addr.S_addr = myIp.S_addr;
        NN_LOG( "Connected Peer Address: %08x\n", targetAddr.sin_addr.S_addr );

        // Call Connect to Bind Servers IP as Peer
        rc = nn::socket::Connect( udpSock,(nn::socket::SockAddr *) &targetAddr, sizeof( targetAddr ) );
        ERROR_IF( rc != 0, "[UDP] Connect failed for UDP Peer socket!  Errno: %d\n", nn::socket::GetLastError() );

        // Send data from NX (LAN Iface) to NX (Lan Iface)
        rc = nn::socket::Send( udpSock, GoogleDNSRequest, sizeof( GoogleDNSRequest ), nn::socket::MsgFlag::Msg_None );
        ERROR_IF_AND_COUNT( rc < 0, "[UDP] Send() failed but should have succeeded!  Errno: %d\n", nn::socket::GetLastError());

        NN_LOG( "Send - Sent %d bytes\n", rc );
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

        // For()ever
        for( ; ; )
        {
            // Recieve DNS Response Messages
            optlen = sizeof( remoteHost );
            rc = nn::socket::RecvFrom( udpServSock, dnsRequest, sizeof( dnsRequest ), nn::socket::MsgFlag::Msg_None,
                                       (nn::socket::SockAddr *) &remoteHost, &optlen );
            if ( rc < 0 )
            {
                myError = nn::socket::GetLastError();
                if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
                {
                    NN_LOG( "Finished waiting for DNS Responses!\n" );
                    break;
                }

                NN_LOG( "Failed calling RecvFrom() - Errno: %d\n", myError );
            }

            NN_LOG( "DNS Request from Host: [%s:%d]\n", nn::socket::InetNtoa(  remoteHost.sin_addr ),
                                                        nn::socket::InetNtohs( remoteHost.sin_port ) );
        }
    }


    ////////////
    // [UDP]
    //
    // Send Directed Broadcast Address - No SetSockOpt - nn::socket::Errno::EAcces
    // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=177019016
    ////////////

    NN_LOG( "Testing: [UDP] - Send Directed Broadcast Address - No SetSockOpt - nn::socket::Errno::EAcces)\n" );

    {
        nn::socket::InAddr myIp = { 0 };
        nn::socket::InAddr myMask = { 0 };
        nn::socket::InAddr myGw = { 0 };
        nn::socket::InAddr myDns1 = { 0 };
        nn::socket::InAddr myDns2 = { 0 };

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

        // UDP Client - Close
        nn::socket::Close( udpSock );

        // UDP Client - Create
        udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

        // UDP Client - Local Address
        memset( &localAddr, 0, sizeof(localAddr));
        localAddr.sin_port     = nn::socket::InetHtons( 9005 );
        localAddr.sin_family   = nn::socket::Family::Af_Inet;

        NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
        localAddr.sin_addr.S_addr = myIp.S_addr;

        // UDP Client - Local Bind
        rc = nn::socket::Bind( udpSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
        ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

        // UDP - Target Address
        memset( &targetAddr, 0, sizeof(targetAddr));
        targetAddr.sin_port     = nn::socket::InetHtons( 9006 );   // Somewhere harmless
        targetAddr.sin_family   = nn::socket::Family::Af_Inet;

        // UDP - Target IP: LAN DIRECTED BROADCAST ADDRESS
        targetAddr.sin_addr.S_addr = myIp.S_addr |= ~myMask.S_addr;
        NN_LOG( "Directed Broadcast Address: %08x\n", targetAddr.sin_addr.S_addr );

        // Call Connect to Bind Servers IP as Peer
        rc = nn::socket::Connect( udpSock,(nn::socket::SockAddr *) &targetAddr, sizeof( targetAddr ) );
        ERROR_IF( rc != 0, "[UDP] Connect failed for UDP Peer socket!  Errno: %d\n", nn::socket::GetLastError() );

        // Send test data from NX
        rc = nn::socket::Send( udpSock, "Test Data", strlen( "Test Data" ), nn::socket::MsgFlag::Msg_None);

        ERROR_IF_AND_COUNT( rc > 0, "[UDP] Send() Directed Broadcast Address succeeded but should have failed!\n" );

        myError = nn::socket::GetLastError();
        ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAcces, "[UDP] Send() Directed Broadcast Address returns Errno: %s!\n", myError );
    }

    ////////////
    // [UDP]
    //
    // Send Directed Broadcast Address - SetSockOpt - Valid Response
    // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=177019016
    ////////////

    NN_LOG( "Testing: [UDP] - Send Directed Broadcast Address - SockOpt - Valid)\n" );

    {
        nn::socket::InAddr myIp = { 0 };
        nn::socket::InAddr myMask = { 0 };
        nn::socket::InAddr myGw = { 0 };
        nn::socket::InAddr myDns1 = { 0 };
        nn::socket::InAddr myDns2 = { 0 };
        nn::socket::SockAddrIn remoteHost = { 0 }, servAddr = { 0 };
        nn::socket::TimeVal myTime = { 0 };
        nn::socket::SockLenT optlen;
        int enableFlag = 1;
        char dnsRequest[512];

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

        // UDP Server Sock - Close
        nn::socket::Close( udpServSock );

        // UDP Server Sock
        udpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

        // UDP Server - Local Address
        memset( &servAddr, 0, sizeof(servAddr));
        servAddr.sin_port        = nn::socket::InetHtons( 9058 );
        servAddr.sin_family      = nn::socket::Family::Af_Inet;

        NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
        servAddr.sin_addr.S_addr = myIp.S_addr;

        // UDP Server - Don't wait forever..
        myTime.tv_sec  = 5;
        myTime.tv_usec = 0;
        optlen         = sizeof( myTime );

        rc = nn::socket::SetSockOpt( udpServSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo,(void *) &myTime, optlen );
        ERROR_IF( rc < 0, "SetSockOpt for nn::socket::Option::So_RcvTimeo failed - Errno: %d\n", nn::socket::GetLastError() );

        // UDP Server - Local Bind
        rc = nn::socket::Bind( udpServSock, (nn::socket::SockAddr *) &servAddr, sizeof( servAddr ) );
        ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

        // UDP Client - Close
        nn::socket::Close( udpSock );

        // UDP Client - Create
        udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

        // UDP Client - May send Broadcast Packets
        rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Broadcast, &enableFlag, sizeof(enableFlag));

        // This converts a directed broadcast to a limited broadcast.
        // See "nn::socket::Option::Ip_OnesBcast option" on the following page for details
        // https://www.freebsd.org/cgi/man.cgi?query=ip&apropos=0&sektion=0&manpath=FreeBSD+10.1-RELEASE&arch=default&format=html
        rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_OnesBcast, &enableFlag, sizeof(enableFlag));
        ERROR_IF( rc < 0, "SetSockOpt for nn::socket::Option::Ip_OnesBcast failed - Errno: %d\n", nn::socket::GetLastError() );

        // UDP Client - Local Address
        memset( &localAddr, 0, sizeof(localAddr));
        localAddr.sin_port        = nn::socket::InetHtons( 9057 );
        localAddr.sin_family      = nn::socket::Family::Af_Inet;

        NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
        localAddr.sin_addr.S_addr = myIp.S_addr;

        // UDP Client - Local Bind
        rc = nn::socket::Bind( udpSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
        ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

        // UDP - Target Address
        memset( &targetAddr, 0, sizeof(targetAddr));
        targetAddr.sin_port     = nn::socket::InetHtons( 9058 );   // Point to UDP Server
        targetAddr.sin_family   = nn::socket::Family::Af_Inet;

        // UDP - Target IP: LAN LIMITED BROADCAST ADDRESS
        targetAddr.sin_addr.S_addr = myIp.S_addr |= ~myMask.S_addr;
        NN_LOG( "Directed Broadcast Address: %08x\n", targetAddr.sin_addr.S_addr );

        // Call Connect to Bind Servers IP as Peer
        rc = nn::socket::Connect( udpSock,(nn::socket::SockAddr *) &targetAddr, sizeof( targetAddr ) );
        ERROR_IF( rc != 0, "[UDP] Connect failed for UDP Peer socket!  Errno: %d\n", nn::socket::GetLastError() );

        // Send Directed data from NX
        rc = nn::socket::Send( udpSock, GoogleDNSRequest, sizeof( GoogleDNSRequest ), nn::socket::MsgFlag::Msg_None );

        ERROR_IF_AND_COUNT( rc < 0,
        "[UDP] Send() Directed Broadcast Address failed but should have succeeded!  Errno:\n", nn::socket::GetLastError());

        NN_LOG( "Send - Sent %d bytes\n", rc );

        // For()ever
        for( ; ; )
        {
            // Recieve DNS Response Messages
            optlen = sizeof( remoteHost );
            rc = nn::socket::RecvFrom( udpServSock, dnsRequest, sizeof( dnsRequest ), nn::socket::MsgFlag::Msg_None,
                                       (nn::socket::SockAddr *) &remoteHost, &optlen );
            if ( rc < 0 )
            {
                myError = nn::socket::GetLastError();
                if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
                {
                    NN_LOG( "Finished waiting for DNS Responses!\n" );
                    break;
                }

                NN_LOG( "Failed calling RecvFrom() - Errno: %d\n", myError );
            }

            NN_LOG( "DNS Request from Host: [%s:%d]\n", nn::socket::InetNtoa( remoteHost.sin_addr ),
                                                        nn::socket::InetNtohs( remoteHost.sin_port ) );
        }
    }

    ////////////
    // [UDP]
    //
    // Send Limited Broadcast Address Address - nn::socket::Errno::EAcces
    // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=177019016
    ////////////

    NN_LOG( "Testing: [UDP] - Send Limited Broadcast Address - No SetSockOpt - nn::socket::Errno::EAcces)\n" );

    {
        nn::socket::InAddr myIp = { 0 };
        nn::socket::InAddr myMask = { 0 };
        nn::socket::InAddr myGw = { 0 };
        nn::socket::InAddr myDns1 = { 0 };
        nn::socket::InAddr myDns2 = { 0 };
        int enableFlag = 1;

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

        // UDP Client - Close
        nn::socket::Close( udpSock );

        // UDP Client - Create
        udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

        // UDP Client - Local Address
        memset( &localAddr, 0, sizeof(localAddr));
        localAddr.sin_port     = nn::socket::InetHtons( 9010 );
        localAddr.sin_family   = nn::socket::Family::Af_Inet;

        // UDP Client - Local Bind
        rc = nn::socket::Bind( udpSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
        ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

        // UDP - Target Address
        memset( &targetAddr, 0, sizeof(targetAddr));
        targetAddr.sin_port     = nn::socket::InetHtons( 9011 );   // Somewhere harmless
        targetAddr.sin_family   = nn::socket::Family::Af_Inet;

        // UDP - Target IP: LAN LIMITED BROADCAST ADDRESS
        targetAddr.sin_addr.S_addr = myIp.S_addr |= ~myMask.S_addr;

        NN_LOG( "Limited Broadcast Address: %08x\n", targetAddr.sin_addr.S_addr );

        // This converts a directed broadcast to a limited broadcast.
        // See "nn::socket::Option::Ip_OnesBcast option" on the following page for details
        // https://www.freebsd.org/cgi/man.cgi?query=ip&apropos=0&sektion=0&manpath=FreeBSD+10.1-RELEASE&arch=default&format=html
        nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_OnesBcast, &enableFlag, sizeof(enableFlag));


        // Call Connect to Bind Servers IP as Peer
        rc = nn::socket::Connect( udpSock, (nn::socket::SockAddr *) &targetAddr, sizeof( targetAddr ) );
        ERROR_IF( rc != 0, "[UDP] Connect failed for UDP Peer socket!  Errno: %d\n", nn::socket::GetLastError() );

        // Send test data which is not (directly) routable from NX
        rc = nn::socket::Send( udpSock, "Test Data", strlen( "Test Data" ), nn::socket::MsgFlag::Msg_None );

        ERROR_IF_AND_COUNT( rc > 0, "[UDP] Send() Directed Broadcast Address succeeded but should have failed!\n" );

        myError = nn::socket::GetLastError();
        ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAcces, "[UDP] Send() Directed Broadcast Address returns Errno: %s!\n", myError );
    }


#if 0

    ////////////
    // [TCP]
    //
    // MSG_EOR - Indicate that this is a complete transmission (like "over" on a walkie-talkie)
    // MSG_EOF - Instructs remote to shutdown it's end of connection (like "out" on a walkie-talkie)
    ///////////

    ////////////
    // [UDP] - Create UDP Server and Client
    ///////////

    NN_LOG( "[UDP]: Create Client/Server Socket\n" );

    // Server:  Create Socket
    udpServSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9000,
                                                        "127.0.0.1", 9001 );
    ERROR_IF_AND_COUNT( udpServSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    // Client: Close Client Sock
    nn::socket::Close( udpSock );

    // Clienr: Create UDP Socket
    udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9001,
                                                    "127.0.0.1", 9000);
    ERROR_IF_AND_COUNT( udpSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );
    {
        int testFlags = -1;
        const char * msgs[] = { "The", "long", "and", "winding", "road", NULL };

        for( idx = 0; msgs[idx] != NULL, idx++ )
        {
            if ( msgs[idx + 1] == NULL )
                intFlags = MSG_EOF;
            else
                intFlags = MSG_EOR;

            NN_LOG( "Write: [%.*s]\n", strlen( msgs[idx] ), msgs[idx] );

            rc = nn::socket::Send( udpSock, msgs[0], strlen( msgs[0] ), MSG_EOR,
                                     (nn::socket::SockAddr *) &outAddr, sizeof( outAddr ) );

            ERROR_IF( rc != strlen( msgs[idx] ). "Failed to send msg: [%.*s]\n",
                                                     strlen( msgs[idx] ), msgs[idx] );
        }

        rc = nn::socket::RecvFrom( udpServSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_Peek, nullptr, 0 );
        if ( rc < 0 )
        {
            break;
        }

    ////////////
    // Test TCP Peer
    ///////////

    NN_LOG( "Testing: [TCP] Peer (Echo Server)\n" );

    outAddrLen = sizeof( outAddr );
    rc = nn::socket::GetPeerName( tcpSock, (nn::socket::SockAddr *) &outAddr, &outAddrLen );
    ERROR_IF_AND_COUNT( rc != 0, "GetPeerAddr() failed, but should have succeded [TCP]\n" );

    asciiAddr = nn::socket::InetNtoa( outAddr.sin_addr );
    NN_LOG( "===> Returned Ascii Address: %s\n", asciiAddr );

    ERROR_IF_AND_COUNT( strlen( asciiAddr ) != strlen( "127.0.0.1" ), "Returned Address is not the right length!\n" );
    ERROR_IF_AND_COUNT( memcmp( asciiAddr, "127.0.0.1", strlen( "127.0.0.1" ) ) != 0,
                        "Returned Address does not equal [127.0.0.1] - Actual returned address: %s\n", asciiAddr );

    ////////////
    // Test UDP Peer
    ///////////

    NN_LOG( "Testing: [UDP] Peer (Echo Server)\n" );

    outAddrLen = sizeof( outAddr );
    rc = nn::socket::GetPeerName( udpSock, (nn::socket::SockAddr *) &outAddr, &outAddrLen );
    ERROR_IF_AND_COUNT( rc != 0, "GetPeerAddr() failed, but should have succeded [UDP]\n" );

    asciiAddr = nn::socket::InetNtoa( outAddr.sin_addr );
    NN_LOG( "===> Returned Ascii Address: %s\n", asciiAddr );

    ERROR_IF_AND_COUNT( strlen( asciiAddr ) != strlen( "127.0.0.1" ), "Returned Address is not the right length!\n" );
    ERROR_IF_AND_COUNT( memcmp( asciiAddr, "127.0.0.1", strlen( "127.0.0.1" ) ) != 0,
                        "Returned Address does not equal [127.0.0.1] - Actual returned address: %s\n", asciiAddr );


    ////////////
    // Test UDP (NO Peer) - nn::socket::Errno::ENotConn
    ///////////

    NN_LOG( "Testing: [UDP] No Peer\n" );

    // Close UDP Socket
    nn::socket::Close( udpSock );
    udpSock = -1;

    // Create UDP Socket
    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpSock < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Client: Target Address
    memset(&localAddr, 0, sizeof(localAddr));
    localAddr.sin_port        = nn::socket::InetHtons( 9000 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Client: Bind (self)
    rc = nn::socket::Bind( udpSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind() Failed but Success was expected!" );

    outAddrLen = sizeof( outAddr );
    rc = nn::socket::GetPeerName( udpSock, (nn::socket::SockAddr *) &outAddr, &outAddrLen );
    ERROR_IF_AND_COUNT( rc == 0, "GetPeerAddr() succeded but should have failed [UDP]\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::ENotConn, "Expected errno: nn::socket::Errno::ENotConn, but actually got errno: %d [UDP]\n", myError );

    // Close UDP Socket
    nn::socket::Close( udpSock );
    udpSock = -1;


    ////////////
    // Test TCP (NO Peer) - nn::socket::Errno::ENotConn
    ///////////

    NN_LOG( "Testing: [TCP] No Peer\n" );

    // Close TCP Socket
    nn::socket::Close( tcpSock );
    tcpSock = -1;

    // Create TCP Socket
    tcpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, 0 );
    ERROR_IF( tcpSock < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Call GetPeerName
    rc = nn::socket::GetPeerName( tcpSock, (nn::socket::SockAddr *) &outAddr, &outAddrLen );
    ERROR_IF_AND_COUNT( rc == 0, "GetPeerAddr() succeded but should have failed [UDP]\n" );

    // Check Result
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::ENotConn, "Expected errno: nn::socket::Errno::ENotConn, but actually got errno: %d [UDP]\n", myError );

    // Close TCP Socket
    nn::socket::Close( tcpSock );
    tcpSock = -1;

#endif

out:

    if ( tcpSock > -1 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    if ( udpServSock > -1 )
    {
        nn::socket::Close( udpServSock );
        udpServSock = -1;
    }
}



TEST(ApiUnit,Send_Blocking_Test)
{
    nn::socket::TimeVal myTime = { 0 };
    int                 cliSock = -1, listenSock = -1, acceptedSock = -1, rc = 0;
    nn::socket::Errno   myError = nn::socket::Errno::ESuccess;

    bool                isSuccess = true, rval;
    nn::socket::SockLenT    optlen;
    uint8_t             noiseVal = 1;
    unsigned char *     buf1 = NULL;

    ////////////
    // Create TCP connection
    ///////////

    rval = NATF::API::COMMON::MakeListenSocket( listenSock, 9060, "127.0.0.1" );
    ERROR_IF( rval == false, "Failed to create Listen Socket - Errno: %d\n", nn::socket::GetLastError() );

    cliSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 9060 );
    ERROR_IF( cliSock < 0, "Failed to create Client Socket - Errno: %d\n", nn::socket::GetLastError() );

    acceptedSock = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF( acceptedSock < 0, "Failed calling Accept on Listen Socket - Errno: %d\n", nn::socket::GetLastError() );

    ////////////
    // Make Noise Buffer
    ///////////

    rc = NATF::API::COMMON::MakeRandomNoise( &buf1, 32767, noiseVal );
    ERROR_IF( rc < 0, "Failed calling MakeRandomNoise() for size 32767\n" );


    ////////////
    // Establish Send Timeout
    ///////////

    myTime.tv_sec  = 5;
    myTime.tv_usec = 0;
    optlen         = sizeof( myTime );

    rc = nn::socket::SetSockOpt( cliSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndTimeo,(void *) &myTime, optlen );
    ERROR_IF( rc < 0, "SetSockOpt() failed - Errno: %d\n", nn::socket::GetLastError() );

    for( ; ; )
    {
        NN_LOG( "[TCP]: Sending 32767\n" );

        rc = nn::socket::Send( cliSock, buf1, 32767, nn::socket::MsgFlag::Msg_None );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                NN_LOG( "================================\n" );
                NN_LOG( "PASSED -- SNDTIMEO unlocked after 5 seconds\n" );
                NN_LOG( "================================\n" );

                break;
            }

            NN_LOG( "Send() failed with Errno: %d\n", myError );
        }
    }

out:

    if ( buf1 != NULL )
    {
        free( buf1 );
        buf1 = NULL;
    }

    if ( cliSock > -1 )
    {
        nn::socket::Close( cliSock );
        cliSock = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

    if ( listenSock > -1 )
    {
        nn::socket::Close( listenSock );
        listenSock = -1;
    }

}


TEST( ApiUnit,Send_SO_OOB)
{

    int                     cliSock = -1, listenSock = -1, acceptedSock = -1;
    int                     rc = -1;
    bool                    isSuccess = true, isOOB = false;
    int                     disableFlag = 0, count = 0, atMark = 0;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    nn::socket::SockLenT    optlen = sizeof( disableFlag );
    char                    buf1[1024] = {};

    const char * msgs[]
    {
         "The rain in Spain...",
         "Flows",
         " mainly on the plain.",
         "The rain in Spain...Flows mainly on the plain."
    };

    ////////////////
    // Create Server
    ////////////////

    rc = NATF::API::COMMON::MakeListenSocket( listenSock, 9001, g_echoServer->kInterfaceIPv4Addr );
    ERROR_IF_AND_COUNT( listenSock < 0, "Failed to creat Server socket\n" );
    NN_LOG( "Server established at: loopback:9001\n" );


    ////////////////
    // Create Client
    ////////////////

    cliSock = NATF::API::COMMON::MakeTCPConnection( g_echoServer->kInterfaceIPv4Addr, 9001 );
    ERROR_IF_AND_COUNT( cliSock < 0, "Failed to create Server socket\n" );
    NN_LOG( "Client established at: loopback:9001\n" );


    ////////////////
    // Server Accepts New Client
    ////////////////

    acceptedSock = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT( acceptedSock < 0, "Failed calling Accept on Listen Socket - Errno: %d\n", nn::socket::GetLastError() );


    ////////////////
    // Server Turns off OOBINLINE
    ////////////////

    rc = nn::socket::SetSockOpt( acceptedSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_OobInline,(void *) &disableFlag, optlen );
    ERROR_IF( rc < 0, "SetSockOpt() failed for nn::socket::Option::So_OobInline - Errno: %d\n", nn::socket::GetLastError() );


    ////////////////
    // Server turns on NON-BLOCKING
    ////////////////

    rc = nn::socket::Fcntl( acceptedSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed for nn::socket::FcntlFlag::O_NonBlock - Errno: %d\n", nn::socket::GetLastError() );


    //////////////
    // Client Writes (regular) and (OOB) data to server..
    //////////////

    NN_LOG( "Send(): %.*s\n", strlen( msgs[0] ), msgs[0] );
    rc = nn::socket::Send( cliSock, (unsigned char *) msgs[0], strlen( msgs[0] ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc != strlen( msgs[0] ), "Client failed to write; %s\n", msgs[0] );

    NN_LOG( "Send() nn::socket::MsgFlag::Msg_Oob: %.*s\n", strlen( msgs[1] ), msgs[1] );
    rc = nn::socket::Send( cliSock, (unsigned char *) msgs[1], strlen( msgs[1] ), nn::socket::MsgFlag::Msg_Oob );
    ERROR_IF_AND_COUNT( rc != strlen( msgs[1] ), "Client failed to Send (OOB); %s\n", msgs[1] );

    NN_LOG( "Send(): %.*s\n", strlen( msgs[2] ), msgs[2] );
    rc = nn::socket::Send( cliSock, (unsigned char *) msgs[2], strlen( msgs[2] ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc != strlen( msgs[2] ), "Client failed to write; %s\n", msgs[2] );


    ///////////////
    // Server Reads (Regular) data BEFORE the OOB data (aka mark)
    ///////////////

    NN_LOG( "Read before Mark: [" );
    isOOB = false;
    for( ; ; )
    {
        atMark = nn::socket::SockAtMark( acceptedSock );
        if (atMark < 0 )
        {
            NN_LOG( "SockAtMark() failed - errno: %d\n", nn::socket::GetLastError() );
        }
        if ( atMark == 1 )
        {
            NN_LOG("[At Mark!]\n" );
            isOOB = true;
        }
        else
        {
            isOOB = false;
        }

        rc = nn::socket::Recv( acceptedSock, (unsigned char *) &buf1[count], 1, (isOOB == true ? nn::socket::MsgFlag::Msg_Oob : nn::socket::MsgFlag::Msg_None ) );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();

            // TRICKY: If nn::socket::Errno::EInval is returned it means "read again" without the OOB flag
            if ( myError == nn::socket::Errno::EInval )
            {
                rc = nn::socket::Recv( acceptedSock, (unsigned char *) &buf1[count], 1, nn::socket::MsgFlag::Msg_None );
                ERROR_IF( rc < 0, "Recv() failed without nn::socket::MsgFlag::Msg_Oob - Errno: %d\n", myError );
                NN_LOG( "%c", (char) buf1[count] );
                count++;
                continue;
            }
            // If it's not nn::socket::Errno::EInval then see if we are done reading
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                break;
            }

            NN_LOG( "Recv() failed with errno: %d\n", nn::socket::GetLastError() );
            goto out;
        }
        NN_LOG( "%c", (char) buf1[count] );
        count++;
        continue;
    }
    NN_LOG( "]\n" );

    printf( "Actual Received Data: [%.*s]\n", count, buf1 );

    // Validate returned data:
    if ( count == strlen( msgs[3] )  &&
         memcmp( msgs[3], buf1, count ) == 0 )
    {
        NN_LOG( "====================\n" );
        NN_LOG( "PASSED!  OOB and Normal Data returned correctly!\n" );
        NN_LOG( "====================\n" );
    }
    else
    {
        NN_LOG( "====================\n" );
        NN_LOG( "FAILED!  OOB Data [%.*s] and Actually Recieved Data [%.*s] do not match!\n",
                               count, buf1, strlen( msgs[3] ), msgs[3] );
        NN_LOG( "====================\n" );
    }


out:

    if ( cliSock > -1 )
    {
        nn::socket::Close( cliSock );
        cliSock = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

    if ( listenSock > -1 )
    {
        nn::socket::Close( listenSock );
        listenSock = -1;
    }
}


TEST( ApiUnit,Send_SO_OOBINLINE)
{

    int                     cliSock = -1, listenSock = -1, acceptedSock = -1;
    int                     rc = -1;
    bool                    isSuccess = true, isOOB = false;
    int                     enableFlag = 1, count = 0, atMark = 0;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    nn::socket::SockLenT    optlen = sizeof( enableFlag );
    char                    buf1[1024] = {};

    const char * msgs[]
    {
         "The rain in Spain...",
         "Flows",
         " mainly on the plain.",
         "The rain in Spain...Flows mainly on the plain."
    };

    ////////////////
    // Create Server
    ////////////////

    rc = NATF::API::COMMON::MakeListenSocket( listenSock, 9001, g_echoServer->kInterfaceIPv4Addr );
    ERROR_IF_AND_COUNT( listenSock < 0, "Failed to creat Server socket\n" );
    NN_LOG( "Server established at: loopback:9001\n" );


    ////////////////
    // Create Client
    ////////////////

    cliSock = NATF::API::COMMON::MakeTCPConnection( g_echoServer->kInterfaceIPv4Addr, 9001 );
    ERROR_IF_AND_COUNT( cliSock < 0, "Failed to create Server socket\n" );
    NN_LOG( "Client established at: loopback:9001\n" );


    ////////////////
    // Server Accepts New Client
    ////////////////

    acceptedSock = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT( acceptedSock < 0, "Failed calling Accept on Listen Socket - Errno: %d\n", nn::socket::GetLastError() );


    ////////////////
    // Server Turns ON OOBINLINE
    ////////////////

    rc = nn::socket::SetSockOpt( acceptedSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_OobInline,(void *) &enableFlag, optlen );
    ERROR_IF( rc < 0, "SetSockOpt() failed for nn::socket::Option::So_OobInline - Errno: %d\n", nn::socket::GetLastError() );


    ////////////////
    // Server turns ON NON-BLOCKING
    ////////////////

    rc = nn::socket::Fcntl( acceptedSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed for nn::socket::FcntlFlag::O_NonBlock - Errno: %d\n", nn::socket::GetLastError() );


    //////////////
    // Client Writes (regular) and (OOB) data to server..
    //////////////

    NN_LOG( "Send(): %.*s\n", strlen( msgs[0] ), msgs[0] );
    rc = nn::socket::Send( cliSock, (unsigned char *) msgs[0], strlen( msgs[0] ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc != strlen( msgs[0] ), "Client failed to write; %s\n", msgs[0] );

    NN_LOG( "Send() nn::socket::MsgFlag::Msg_Oob: %.*s\n", strlen( msgs[1] ), msgs[1] );
    rc = nn::socket::Send( cliSock, (unsigned char *) msgs[1], strlen( msgs[1] ), nn::socket::MsgFlag::Msg_Oob );
    ERROR_IF_AND_COUNT( rc != strlen( msgs[1] ), "Client failed to Send (OOB); %s\n", msgs[1] );

    NN_LOG( "Send(): %.*s\n", strlen( msgs[2] ), msgs[2] );
    rc = nn::socket::Send( cliSock, (unsigned char *) msgs[2], strlen( msgs[2] ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc != strlen( msgs[2] ), "Client failed to write; %s\n", msgs[2] );


    ///////////////
    // Server Reads (Regular) data BEFORE the OOB data (aka mark)
    ///////////////

    NN_LOG( "Read before Mark: [" );
    isOOB = false;
    for( ; ; )
    {
        atMark = nn::socket::SockAtMark( acceptedSock );
        if (atMark < 0 )
        {
            NN_LOG( "SockAtMark() failed - errno: %d\n", nn::socket::GetLastError() );
        }
        if ( atMark == 1 )
        {
            NN_LOG("[At Mark!]\n" );
            isOOB = true;
        }
        else
        {
            isOOB = false;
        }

        rc = nn::socket::Recv( acceptedSock, (unsigned char *) &buf1[count], 1, (isOOB == true ? nn::socket::MsgFlag::Msg_Oob : nn::socket::MsgFlag::Msg_None ) );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();

            // TRICKY: If nn::socket::Errno::EInval is returned it means "read again" without the OOB flag
            if ( myError == nn::socket::Errno::EInval )
            {
                rc = nn::socket::Recv( acceptedSock, (unsigned char *) &buf1[count], 1, nn::socket::MsgFlag::Msg_None );
                ERROR_IF( rc < 0, "Recv() failed without nn::socket::MsgFlag::Msg_Oob - Errno: %d\n", myError );
                NN_LOG( "%c", (char) buf1[count] );
                count++;
                continue;
            }
            // If it's not nn::socket::Errno::EInval then see if we are done reading
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                break;
            }

            NN_LOG( "Recv() failed with errno: %d\n", nn::socket::GetLastError() );
            goto out;
        }
        NN_LOG( "%c", (char) buf1[count] );
        count++;
        continue;
    }
    NN_LOG( "]\n" );

    printf( "Actual Received Data: [%.*s]\n", count, buf1 );

    // Validate returned data:
    if ( count == strlen( msgs[3] )  &&
         memcmp( msgs[3], buf1, count ) == 0 )
    {
        NN_LOG( "====================\n" );
        NN_LOG( "PASSED!  OOB and Normal Data returned correctly!\n" );
        NN_LOG( "====================\n" );
    }
    else
    {
        NN_LOG( "====================\n" );
        NN_LOG( "FAILED!  OOB Data [%.*s] and Actually Recieved Data [%.*s] do not match!\n",
                               count, buf1, strlen( msgs[3] ), msgs[3] );
        NN_LOG( "====================\n" );
    }


out:

    if ( cliSock > -1 )
    {
        nn::socket::Close( cliSock );
        cliSock = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

    if ( listenSock > -1 )
    {
        nn::socket::Close( listenSock );
        listenSock = -1;
    }
}

TEST(ApiUnit,Send_Teardown)
{
    TeardownTesting();
}

#else // Windows

TEST(ApiUnit,Send_Windows)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

}}  // Namespace: NATF::API
