﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_LoopbackDnsServer.h"     // Loopback DNS
#include "Unit/testNet_CommonFunctions.h"       // Common Functions

#include <nn/socket/socket_ApiPrivate.h>

#ifndef NN_BUILD_CONFIG_OS_WIN32
#include <nn/socket/resolver/resolver_Client.h>
#include <nn/socket/socket_ResolverOptionsPrivate.h>
#include <nn/socket/resolver/private/resolver_PrivateApi.h>
#endif


namespace NATF {
namespace API {


//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32

//*********************
//*  G L O B A L S
//*********************

static ThreadedController * g_pThreadedTest      = NULL;
static LoopbackDnsServer *  g_LoopbackDnsServer  = NULL;
static LoopbackDnsServer *  g_LoopbackDnsServer2 = NULL;

static int           g_WorkVaruint32_t  = 0;
static int           g_WorkVaruint32_t2 = 0;
static int           g_WorkVaruint32_t3 = 0;
static int           g_WorkVarint_t = 0;
static unsigned char g_WorkVaruchar = 0;
static bool          g_workVarBool = false;

static int           kMaxNumberOfThreads = 10;
static int           kThreadPriority = 10;

static int           g_OffsetDnsResp = 0;
static int           g_SharedChar  = '0';
static int           g_SharedChar2 = '0';
static int           g_DnsRecordCount = 0;
static size_t        g_NewDnsRespLen = 0;
static uint8_t       g_ReturnCode = 0;
static uint8_t       g_OpCode = 0;
static bool          g_MyResults[5];

static bool         g_ServerHit = true;

static const int    g_ServerA = 1;
static const int    g_ServerB = 2;


// Anonymous
namespace {

// Standard Google Lookup
//
// ; <<>> DiG 9.10.3-P4 <<>> +noedns=0 google.com
// ;; global options: +cmd
// ;; Got answer:
// ;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 35193
// ;; flags: qr rd ra; QUERY: 1, ANSWER: 1, AUTHORITY: 0, ADDITIONAL: 0
//
// ;; QUESTION SECTION:
// ;; google.com.                    IN      A
//
// ;; ANSWER SECTION:
// ;; google.com.            150     IN      A       216.58.193.78
//
// ;; Query time: 31 msefalsec
// ;; SERVER: 10.1.19.29#53(10.1.19.29)
// ;; WHEN: Tue Aug 02 12:59:44 PDT 2016
// ;; MSG SIZE  rcvd: 44

static const unsigned char DnsCacheGoogleComSimple[] = {

0x96, 0xac, 0x81, 0x80, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
0x06, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x03, 0x63, 0x6f, 0x6d, 0x00,
0x00, 0x01, 0x00, 0x01, 0xc0, 0x0c, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
0x00, 0xcd, 0x00, 0x04, 0xd8, 0x3a, 0xc1, 0x4e

};

// Simple Google Lookup (with TC bit) - (Pos 3) - Set From: 0x81 To: 0x83
//
// ; <<>> DiG 9.10.3-P4 <<>> +noedns=0 google.com
// ;; global options: +cmd
// ;; Got answer:
// ;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 35193
// ;; flags: qr rd ra tc; QUERY: 1, ANSWER: 1, AUTHORITY: 0, ADDITIONAL: 0
//
// ;; QUESTION SECTION:
// ;google.com.                    IN      A
//
// ;; ANSWER SECTION:
// google.com.             150     IN      A       216.58.193.78
//
// ;; Query time: 31 msefalsec
// ;; SERVER: 10.1.19.29#53(10.1.19.29)
// ;; WHEN: Tue Aug 02 12:59:44 PDT 2016
// ;; MSG SIZE  rcvd: 44

static const unsigned char DnsCacheGoogleComSimpleWithTC[] = {

0x96, 0xac, 0x83, 0x80, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
0x06, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x03, 0x63, 0x6f, 0x6d, 0x00,
0x00, 0x01, 0x00, 0x01, 0xc0, 0x0c, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
0x00, 0xcd, 0x00, 0x04, 0xd8, 0x3a, 0xc1, 0x4e

};

// Google Lookup with EDNS0 + Payload Size = 4K
//
// ; <<>> DiG 9.10.3-P4 <<>> google.com
// ;; global options: +cmd
// ;; Got answer:
// ;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 7008
// ;; flags: qr rd ra; QUERY: 1, ANSWER: 1, AUTHORITY: 0, ADDITIONAL: 1
//
// ;; OPT PSEUDOSECTION:
// ; EDNS: version: 0, flags:; udp: 4096
// ;; QUESTION SECTION:
// ;google.com.                    IN      A
//
// ;; ANSWER SECTION:
// google.com.             90      IN      A       216.58.193.78
//
// ;; Query time: 3 msec
// ;; SERVER: 10.1.19.29#53(10.1.19.29)
// ;; WHEN: Tue Aug 02 12:33:48 PDT 2016
// ;; MSG SIZE  rcvd: 55

static const unsigned char DnsCacheGoogleComSimpleWithEDNS04k[] = {

0x1b, 0x60, 0x81, 0x80, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
0x06, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x03, 0x63, 0x6f, 0x6d, 0x00,
0x00, 0x01, 0x00, 0x01, 0xc0, 0x0c, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
0x00, 0x5a, 0x00, 0x04, 0xd8, 0x3a, 0xc1, 0x4e, 0x00, 0x00, 0x29, 0x10,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00

};
//; <<>> DiG 9.10.4-P4 <<>> alias.landsberger.com
//;; global options: +cmd
//;; Got answer:
//;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 10242
//;; flags: qr rd ra; QUERY: 1, ANSWER: 2, AUTHORITY: 0, ADDITIONAL: 1
//
//;; OPT PSEUDOSECTION:
//; EDNS: version: 0, flags:; udp: 4096
//;; QUESTION SECTION:
//;alias.landsberger.com. IN   A
//
//;; ANSWER SECTION:
//alias.landsberger.com.    2228    IN  CNAME landsberger.com.
//landsberger.com.  2228    IN  A   138.128.120.101
//
//;; Query time: 7 msec
//;; SERVER: 10.1.19.29#53(10.1.19.29)
//;; WHEN: Wed May 03 11:35:13 PST 2017
//;; MSG SIZE  rcvd: 80

static const unsigned char DnsCacheAliasLandsbergberComCNAME[] = {

0x77, 0x91, 0x81, 0x80, 0x00, 0x01, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01,
0x05, 0x61, 0x6c, 0x69, 0x61, 0x73, 0x0b, 0x6c, 0x61, 0x6e, 0x64, 0x73,
0x62, 0x65, 0x72, 0x67, 0x65, 0x72, 0x03, 0x63, 0x6f, 0x6d, 0x00, 0x00,
0x01, 0x00, 0x01, 0xc0, 0x0c, 0x00, 0x05, 0x00, 0x01, 0x00, 0x00, 0x0d,
0x6a, 0x00, 0x02, 0xc0, 0x12, 0xc0, 0x12, 0x00, 0x01, 0x00, 0x01, 0x00,
0x00, 0x0d, 0x6a, 0x00, 0x04, 0x8a, 0x80, 0x78, 0x65, 0x00, 0x00, 0x29,
0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00

};

// Standard: E-Root lookup
//
// $ dig -x 192.203.230.10 +noedns
//
// ; <<>> DiG 9.10.3-P4 <<>> -x 192.203.230.10 +noedns
// ;; global options: +cmd
// ;; Got answer:
// ;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 30073
// ;; flags: qr rd ra; QUERY: 1, ANSWER: 1, AUTHORITY: 0, ADDITIONAL: 0
//
// ;; QUESTION SECTION:
// ;10.230.203.192.in-addr.arpa.   IN      PTR
//
// ;; ANSWER SECTION:
// 10.230.203.192.in-addr.arpa. 10066 IN   PTR     e.root-servers.net.
//
// ;; Query time: 2 msec
// ;; SERVER: 10.1.19.29#53(10.1.19.29)
// ;; WHEN: Tue Aug 23 12:27:05 PDT 2016
// ;; MSG SIZE  rcvd: 77

static const unsigned char DnsCacheERootComSimple[] = {

0xf3, 0x78, 0x81, 0x80, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
0x02, 0x31, 0x30, 0x03, 0x32, 0x33, 0x30, 0x03, 0x32, 0x30, 0x33, 0x03,
0x31, 0x39, 0x32, 0x07, 0x69, 0x6e, 0x2d, 0x61, 0x64, 0x64, 0x72, 0x04,
0x61, 0x72, 0x70, 0x61, 0x00, 0x00, 0x0c, 0x00, 0x01, 0xc0, 0x0c, 0x00,
0x0c, 0x00, 0x01, 0x00, 0x00, 0x26, 0xaa, 0x00, 0x14, 0x01, 0x65, 0x0c,
0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72, 0x73,
0x03, 0x6e, 0x65, 0x74, 0x00

};

// Standard: E-Root lookup with Truncate Bit (TC) - (Pos 3) - Set From: 0x81 To: 0x83
//
// $ dig -x 192.203.230.10 +noedns
//
// ; <<>> DiG 9.10.3-P4 <<>> -x 192.203.230.10 +noedns
// ;; global options: +cmd
// ;; Got answer:
// ;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 30073
// ;; flags: qr rd ra; QUERY: 1, ANSWER: 1, AUTHORITY: 0, ADDITIONAL: 0
//
// ;; QUESTION SECTION:
// ;10.230.203.192.in-addr.arpa.   IN      PTR
//
// ;; ANSWER SECTION:
// 10.230.203.192.in-addr.arpa. 10066 IN   PTR     e.root-servers.net.
//
// ;; Query time: 2 msec
// ;; SERVER: 10.1.19.29#53(10.1.19.29)
// ;; WHEN: Tue Aug 23 12:27:05 PDT 2016
// ;; MSG SIZE  rcvd: 77

static const unsigned char DnsCacheERootComSimpleWithTC[] = {

0xf3, 0x78, 0x83, 0x80, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
0x02, 0x31, 0x30, 0x03, 0x32, 0x33, 0x30, 0x03, 0x32, 0x30, 0x33, 0x03,
0x31, 0x39, 0x32, 0x07, 0x69, 0x6e, 0x2d, 0x61, 0x64, 0x64, 0x72, 0x04,
0x61, 0x72, 0x70, 0x61, 0x00, 0x00, 0x0c, 0x00, 0x01, 0xc0, 0x0c, 0x00,
0x0c, 0x00, 0x01, 0x00, 0x00, 0x26, 0xaa, 0x00, 0x14, 0x01, 0x65, 0x0c,
0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72, 0x73,
0x03, 0x6e, 0x65, 0x74, 0x00

};


// 60 Character Label + EDNS0
//
// $ dig 123456789012345678901234567890123456789012345678901234567890.com
//
//; <<>> DiG 9.10.4-P4 <<>> 123456789012345678901234567890123456789012345678901234567890.com
//;; global options: +cmd
//;; Got answer:
//;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 36135
//;; flags: qr rd ra; QUERY: 1, ANSWER: 1, AUTHORITY: 0, ADDITIONAL: 1
//
//;; OPT PSEUDOSECTION:
//; EDNS: version: 0, flags:; udp: 4096
//;; QUESTION SECTION:
//;123456789012345678901234567890123456789012345678901234567890.com. IN A
//
//;; ANSWER SECTION:
//123456789012345678901234567890123456789012345678901234567890.com. 431 IN A 72.52.4.121
//
//;; Query time: 2 msec
//;; SERVER: 10.1.19.29#53(10.1.19.29)
//;; WHEN: Tue Jun 13 10:31:04 PST 2017
//;; MSG SIZE  rcvd: 109

static const unsigned char DnsCache123456ComSimple[] = {

0x8d, 0x27, 0x81, 0x80, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
0x3c, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x30, 0x31,
0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x30, 0x31, 0x32, 0x33,
0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35,
0x36, 0x37, 0x38, 0x39, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
0x38, 0x39, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
0x30, 0x03, 0x63, 0x6f, 0x6d, 0x00, 0x00, 0x01, 0x00, 0x01, 0xc0, 0x0c,
0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x01, 0xaf, 0x00, 0x04, 0x48, 0x34,
0x04, 0x79, 0x00, 0x00, 0x29, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00

};

// Google: Pointer to (SELF)
//
// Pos (30 == 0x1E ) points to it's own offset
//
// ; <<>> DiG 9.10.3-P4 <<>> +noedns=0 google.com
// ;; global options: +cmd
// ;; Got answer:
// ;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 35193
// ;; flags: qr rd ra; QUERY: 1, ANSWER: 1, AUTHORITY: 0, ADDITIONAL: 0
//
// ;; QUESTION SECTION:
// ;google.com.                    IN      A
//
// ;; ANSWER SECTION:
// google.com.             150     IN      A       216.58.193.78
//
// ;; Query time: 31 msefalsec
// ;; SERVER: 10.1.19.29#53(10.1.19.29)
// ;; WHEN: Tue Aug 02 12:59:44 PDT 2016
// ;; MSG SIZE  rcvd: 44

static const unsigned char GoogleComLabelPointerToSelfBug[] = {

0x96, 0xac, 0x81, 0x80, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
0x06, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x03, 0x63, 0x6f, 0x6d, 0x00,  // google.com + 0x0
0x00, 0x01, 0x00, 0x01, 0xc0, 0x1e, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
0x00, 0xcd, 0x00, 0x04, 0xd8, 0x3a, 0xc1, 0x4e

};

// Google: Pointer to Pointer to (Self)
//
// Pos (29 == 0x1D ) Pointer to Pointer to (Self)
//
// ; <<>> DiG 9.10.3-P4 <<>> +noedns=0 google.com
// ;; global options: +cmd
// ;; Got answer:
// ;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 35193
// ;; flags: qr rd ra; QUERY: 1, ANSWER: 1, AUTHORITY: 0, ADDITIONAL: 0
//
// ;; QUESTION SECTION:
// ;google.com.                    IN      A
//
// ;; ANSWER SECTION:
// google.com.             150     IN      A       216.58.193.78
//
// ;; Query time: 31 msefalsec
// ;; SERVER: 10.1.19.29#53(10.1.19.29)
// ;; WHEN: Tue Aug 02 12:59:44 PDT 2016
// ;; MSG SIZE  rcvd: 44

static const unsigned char GoogleComLabelPointerToPointerToSelfBug[] = {

0x96, 0xac, 0x81, 0x80, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
0x06, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x03, 0x63, 0x6f, 0x6d, 0x00,  // google.com + 0x0
0x00, 0x01, 0x00, 0x01, 0xc0, 0x1d, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
0x00, 0xcd, 0x00, 0x04, 0xd8, 0x3a, 0xc1, 0x4e

};


// Google: Pointer to Pointer
//
// ; <<>> DiG 9.10.3-P4 <<>> +noedns=0 google.com
// ;; global options: +cmd
// ;; Got answer:
// ;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 35193
// ;; flags: qr rd ra; QUERY: 1, ANSWER: 1, AUTHORITY: 0, ADDITIONAL: 0
//
// ;; QUESTION SECTION:
// ;google.com.                    IN      A
//
// ;; ANSWER SECTION:
// google.com.             150     IN      A       216.58.193.78
//
// ;; Query time: 31 msefalsec
// ;; SERVER: 10.1.19.29#53(10.1.19.29)
// ;; WHEN: Tue Aug 02 12:59:44 PDT 2016
// ;; MSG SIZE  rcvd: 44

static const unsigned char GoogleComLabelPointerToPointerBug[] = {

0x96, 0xac, 0x81, 0x80, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
0x06, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x03, 0x63, 0x6f, 0x6d, 0x00,  // google.com + 0x0
0x00, 0x01, 0x00, 0x01, 0xc0, 0x1f, 0xc0, 0x1d, 0x00, 0x01, 0x00, 0x01,
0x00, 0x00, 0x00, 0xcd, 0x00, 0x04, 0xd8, 0x3a, 0xc1, 0x4e

};



static
int SetDnsResolver( const char * pIpAddressA, uint16_t portA,
                    const char * pIpAddressB, uint16_t portB  )
{
    nn::socket::ResolverOption                 myOption;
    nn::socket::PrivateDnsAddressArrayArgument arg;
    int                                        result = -1;

    NN_LOG("SetDnsResolver() - Begin\n");
    NN_LOG("HostA: %s, PortA: %d, HostB: %s, PortB: %d\n", pIpAddressA, portA, pIpAddressB, portB);
    arg.count = 0;
    if ( pIpAddressA != nullptr )
    {
        NN_LOG( "Saving HostA\n" );
        arg.count++;
        arg.addresses[0].sin_port   = nn::socket::InetHtons(portA);
        arg.addresses[0].sin_family = nn::socket::Family::Af_Inet;
        nn::socket::InetPton(nn::socket::Family::Af_Inet, pIpAddressA, &arg.addresses[0].sin_addr.S_addr );
    }
    if ( pIpAddressB != nullptr )
    {
        NN_LOG( "Saving HostB\n" );
        arg.count++;
        arg.addresses[1].sin_port   = nn::socket::InetHtons(portB);
        arg.addresses[1].sin_family = nn::socket::Family::Af_Inet;
        nn::socket::InetPton(nn::socket::Family::Af_Inet, pIpAddressB, &arg.addresses[1].sin_addr.S_addr );
    }

    myOption.key = static_cast<nn::socket::ResolverOptionKey>(static_cast<uint32_t>(
                                                                  nn::socket::ResolverOptionPrivateKey::SetDnsServerAddressesPointer));
    myOption.type = nn::socket::ResolverOptionType::Pointer;
    myOption.size = sizeof(arg);
    myOption.data.pointerValue = (char*)&arg;
    result = nn::socket::ResolverSetOption(myOption);
    NN_LOG("SetDnsResolver() - End\n");

    return result;
}

static
void GetDnsResolver()
{
    nn::socket::ResolverOption                 myOption;
    nn::socket::PrivateDnsAddressArrayArgument arg;
    int                                        result = -1;

    NN_LOG("GetDnsResolver() - Begin\n");
    myOption.key = static_cast<nn::socket::ResolverOptionKey>(static_cast<uint32_t>(
                                                                  nn::socket::ResolverOptionPrivateKey::GetDnsServerAddressesPointer));
    myOption.type = nn::socket::ResolverOptionType::Pointer;
    myOption.size = sizeof(arg);
    myOption.data.pointerValue = (char*)&arg;
    NN_LOG( "(Before) nn::socket::ResolverGetOption\n" );
    result = nn::socket::ResolverGetOption(&myOption,
                                           static_cast<nn::socket::ResolverOptionKey>(
                                               static_cast<uint32_t>(nn::socket::ResolverOptionPrivateKey::GetDnsServerAddressesPointer)));
    NN_LOG( "(Before) nn::socket::ResolverGetOption.  Result: %d\n", result );
    if (result == -1)
    {
        NN_LOG( "Failed calling: 'ResolverOptionKey::PrivateGetDnsServerAddressesPointer'\n" );
        return;
    }

    NN_LOG( "Arg count: %d\n", arg.count );

    if ( arg.count > 0 )
    {
        NN_LOG( "Host A: %s, Port A: %d\n",
                nn::socket::InetNtoa( (nn::socket::InAddr) arg.addresses[0].sin_addr ),
                nn::socket::InetNtohs( arg.addresses[0].sin_port ) );
    }

    if ( arg.count > 1 )
    {
        NN_LOG( "Host B: %s, Port B: %d\n",
                nn::socket::InetNtoa( (nn::socket::InAddr) arg.addresses[1].sin_addr ),
                nn::socket::InetNtohs( arg.addresses[1].sin_port ) );
    }

    NN_LOG("GetDnsResolver() - End\n");
}

static
bool IsDomainNameInCache( const char *  inDomainName )
{
    nn::socket::ResolverOption   myOption;
    int result;
    bool isInCache = false;

    NN_LOG("IsDomainNameInCache() - Begin\n");
    myOption.key = static_cast<nn::socket::ResolverOptionKey>(static_cast<uint32_t>(
                                                                  nn::socket::ResolverOptionPrivateKey::GetCacheEntryCountForDomainnamePointer));
    myOption.type = nn::socket::ResolverOptionType::Pointer;
    myOption.size = strlen(inDomainName) + 1;
    myOption.data.pointerValue = inDomainName;
    result = nn::socket::ResolverGetOption(&myOption, myOption.key);
    if ( result > 0 )
    {
        isInCache = true;
    }
    else
    {
        isInCache = false;
    }

    NN_LOG("IsDomainNameInCache() - End (%s)\n", (isInCache == true ? "true" : "false" ));
    return( isInCache );
}

static
bool IsIPInCache( const char * inIPv4 )
{
    nn::socket::ResolverOption   myOption;
    int result;
    bool isInCache = false;

    NN_LOG("IsIPInCache() - Begin\n");
    myOption.key = static_cast<nn::socket::ResolverOptionKey>(static_cast<uint32_t>(
                                                                  nn::socket::ResolverOptionPrivateKey::GetCacheEntryCountForIpUnsigned32));
    myOption.type = nn::socket::ResolverOptionType::Unsigned32;
    myOption.size = sizeof( uint32_t );
    nn::socket::InetPton(nn::socket::Family::Af_Inet, inIPv4, &myOption.data.unsigned32Value);
    result = nn::socket::ResolverGetOption(&myOption, myOption.key);
    if ( result > 0 )
    {
        isInCache = true;
    }
    else
    {
        isInCache = false;
    }

    NN_LOG("IsIPInCache() - End (%s)\n", (isInCache == true ? "true" : "false" ) );
    return( isInCache );
}

static
void CacheFlush()
{
    nn::socket::ResolverOption   myOption;

    NN_LOG("CacheFlush() - Begin\n");
    myOption.key = static_cast<nn::socket::ResolverOptionKey>(static_cast<uint32_t>(
                                                                  nn::socket::ResolverOptionPrivateKey::SetFlushCacheBoolean ));
    myOption.type = nn::socket::ResolverOptionType::Boolean;
    myOption.size = sizeof(bool);
    myOption.data.booleanValue = true;
    nn::socket::ResolverSetOption( myOption );
    NN_LOG("CacheFlush() - End\n");
}

static
int CacheRemoveHostname( const char * pArgument )
{
    nn::socket::ResolverOption   myOption;
    int                          result = -1;

    NN_LOG("CacheRemoveHostname() - Begin\n");
    NN_LOG("Removing Hostname: %s\n", pArgument );
    myOption.key = static_cast<nn::socket::ResolverOptionKey>(nn::socket::ResolverOptionKey::SetRemoveDomainnameFromCachePointer);
    myOption.type = nn::socket::ResolverOptionType::Pointer;
    myOption.size = strlen(pArgument) + 1;
    myOption.data.pointerValue = pArgument;
    result = nn::socket::ResolverSetOption(myOption);
    NN_LOG( "result: %d\n", result );
    NN_LOG("CacheRemoveHostname() - End\n");

    return result;
}

static
int CacheRemoveIpAddress( const char * pArgument )
{
    nn::socket::ResolverOption   myOption;
    uint32_t                     ip = 0;
    int                          result = -1;

    NN_LOG("CacheRemoveIpAddress() - Begin\n");
    nn::socket::InetPton(nn::socket::Family::Af_Inet, pArgument, &ip);
    NN_LOG("Removing IP Address: %s\n", pArgument);
    myOption.key = static_cast<nn::socket::ResolverOptionKey>(nn::socket::ResolverOptionKey::SetRemoveIpAddressFromCacheUnsigned32);
    myOption.type = nn::socket::ResolverOptionType::Unsigned32;
    myOption.size = sizeof(uint32_t);
    myOption.data.unsigned32Value = ip;
    result = nn::socket::ResolverSetOption(myOption);
    NN_LOG("CacheRemoveIpAddress() - End\n");

    return result;
}

static
int CacheUpdateTimeToLiveForDomainname( const char * pDomainname, int ttl )
{
    nn::socket::ResolverOption                            myOption;
    nn::socket::PrivateSetTimeToLiveForDomainnameArgument arg;
    int                                                   result = -1;

    NN_LOG("CacheUpdateTimeToLiveForDomainname() - Begin\n");
    NN_LOG("Updating TTL for Hostname: %s, to TTL: %d\n", pDomainname, ttl );

    arg.ttl = ttl;
    memset(arg.domainname, 0, sizeof( arg.domainname ) );
    memcpy(arg.domainname, pDomainname, strlen( pDomainname ) + 1 );

    myOption.key = static_cast<nn::socket::ResolverOptionKey>(static_cast<uint32_t>(
                                                                  nn::socket::ResolverOptionPrivateKey::SetTimeToLiveForDomainnamePointer));
    myOption.type = nn::socket::ResolverOptionType::Pointer;
    myOption.size = sizeof(arg);
    myOption.data.pointerValue = (char*)&arg;
    result = nn::socket::ResolverSetOption(myOption);
    NN_LOG( "result: %d\n", result);
    NN_LOG("CacheUpdateTimeToLiveForDomainname() - End\n");

    return result;
}

static
nn::socket::ResolverOption DisableDnsCache()
{
    nn::socket::ResolverOption   myOption;

    NN_LOG( "DisableDnsCache() - Begin\n" );
    myOption.key = static_cast<nn::socket::ResolverOptionKey>( nn::socket::ResolverOptionKey::RequestEnableDnsCacheBoolean );
    myOption.type = nn::socket::ResolverOptionType::Boolean;
    myOption.size = sizeof(bool);
    myOption.data.booleanValue = false;
    NN_LOG( "DisableDnsCache() - End\n" );

    return myOption;
}

static
nn::socket::ResolverOption EnableDnsCache()
{
    nn::socket::ResolverOption   myOption;

    NN_LOG( "EnableDnsCache() - Begin\n" );
    myOption.key = static_cast<nn::socket::ResolverOptionKey>( nn::socket::ResolverOptionKey::RequestEnableDnsCacheBoolean );
    myOption.type = nn::socket::ResolverOptionType::Boolean;
    myOption.size = sizeof(bool);
    myOption.data.booleanValue = true;
    NN_LOG( "EnableDnsCache() - End\n" );

    return myOption;
}

static
nn::socket::ResolverOption SetCancelHandleInteger( int inCancelHandle )
{
    nn::socket::ResolverOption   myOption;

    NN_LOG( "SetCancelHandleInteger() - Begin\n" );
    myOption.key = nn::socket::ResolverOptionKey::SetCancelHandleInteger;
    myOption.type = nn::socket::ResolverOptionType::Integer;
    myOption.size = sizeof(int);
    myOption.data.integerValue = inCancelHandle;
    NN_LOG( "SetCancelHandleInteger() - End\n" );

    return myOption;
}

static
nn::socket::ResolverOption GetCancelHandleInteger( int inCancelHandle )
{
    nn::socket::ResolverOption   myOption;

    NN_LOG( "GetCancelHandleInteger() - Begin\n" );
    myOption.key = nn::socket::ResolverOptionKey::RequestCancelHandleInteger;
    myOption.type = nn::socket::ResolverOptionType::Integer;
    myOption.size = sizeof(int);
    myOption.data.integerValue = inCancelHandle;
    NN_LOG( "GetCancelHandleInteger() - End\n" );

    return myOption;
}

void
StartLookbackDnsServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Loopback DNS server]: Thread starting\n" );

    LoopbackDnsServer * loopbackDnsServer = reinterpret_cast<LoopbackDnsServer *>( inArg );

    // Start the DNS Server Main Loop
    loopbackDnsServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Loopback DNS server]: Thread exiting\n" );
}

bool
InitializeTesting()
{
    nn::Result                result;
    int                       rc;
    bool                      isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate DNS Server
    ///////////////////////////

    g_LoopbackDnsServer = new LoopbackDnsServer();
    if ( g_LoopbackDnsServer == NULL )
    {
        NN_LOG( "Failed to allocate a LoopbackDnsServer.  Can't start Loopback port DNS Server\n" );
        goto out;
    }

    ///////////////////////////
    ////  Change DNS Server Address - to (loopback)
    ///////////////////////////

    rc = SetDnsResolver( "127.0.0.1", 8053, nullptr, 0 );
    ERROR_IF( rc == -1, "Failed calling SetDnsResolver()" );

    GetDnsResolver();

    ///////////////////////////
    //// Threaded DNS Server
    ///////////////////////////

    NN_LOG( "===> Starting [loopback] DNS Server\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumberOfThreads );

    // Create a thread to run the Server in
    rc = g_pThreadedTest->CreateThread( &StartLookbackDnsServer, reinterpret_cast<void *>( g_LoopbackDnsServer ), kThreadPriority );
    if ( rc < 0 )
    {
        NN_LOG( "Failed to create and start DNS Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started DNS Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_LoopbackDnsServer != NULL )
    {
        delete g_LoopbackDnsServer;
        g_LoopbackDnsServer = NULL;
    }

    return( false );
}

bool
CallGetHostByName( const char * in_domainname, nn::socket::ResolverOption *myOption, int optionCount )
{
    nn::socket::HostEnt *   respHosts;
    nn::socket::InAddr **   addr_list;
    int                     idx;

    if ( optionCount < 1 )
    {
        PRINT_AND_CALL(respHosts = nn::socket::GetHostEntByName( in_domainname ));
    }
    else
    {
        PRINT_AND_CALL(respHosts = nn::socket::GetHostEntByName( in_domainname, myOption, optionCount ));
    }


    if ( respHosts == NULL )
    {
        NN_LOG( "nn::socket::GetHostEntByName() Failed with: errno: %d, h_errno: %d, hstrerror: %s\n",
                 nn::socket::GetLastError(), *(nn::socket::GetHError() ),
                 nn::socket::HStrError(  *(nn::socket::GetHError() ) ) );
        goto out;
    }
    else
    {
        NN_LOG( "Host name       : [%s]\n", respHosts->h_name );
        NN_LOG( "Address Type    : [%d]\n", respHosts->h_addrtype );
        NN_LOG( "Len of Addr     : [%d]\n", respHosts->h_length );

        for(idx = 0; respHosts->h_aliases[idx] != NULL; idx++)
        {

           NN_LOG( "Alias: [%s]\n", respHosts->h_aliases[idx] );
        }

        addr_list = (nn::socket::InAddr **) respHosts->h_addr_list;
        for(idx = 0; addr_list[idx] != NULL; idx++)
        {

           NN_LOG( "IP Address: [%s]\n", nn::socket::InetNtoa( (*addr_list[idx] ) ) );
        }

        NN_LOG( "There are a total of [%d] IPv4 Addresses\n", idx );
    }

    return( true );

out:

    return( false );
}

bool
CallGetAddrInfo( const char *in_domainname, nn::socket::ResolverOption *myOption, int optionCount )
{
    nn::socket::SockAddrIn  *ipv4;
    nn::socket::AddrInfo    *respIdx, hints;
    nn::socket::AddrInfo    *out_resp;
    nn::socket::AiErrno      out_error;

    // Set GAI Hints
    memset( (char *) &hints, 0, sizeof( hints ) );
    hints.ai_family   = nn::socket::Family::Af_Inet;

    hints.ai_family   = nn::socket::Family::Af_Inet;
    hints.ai_socktype = nn::socket::Type::Sock_Stream;
    hints.ai_flags    = nn::socket::AddrInfoFlag::Ai_CanonName;

    if ( optionCount < 1 )
    {
        PRINT_AND_CALL(out_error = nn::socket::GetAddrInfo( in_domainname, 0, &hints, &out_resp ));
    }
    else
    {
        PRINT_AND_CALL(out_error = nn::socket::GetAddrInfo( in_domainname, 0, &hints, &out_resp, myOption, optionCount ));
    }

    if ( out_error != nn::socket::AiErrno::EAi_Success )
    {
        NN_LOG( "nn::socket::GetAddrInfo() Failed with: errno: %d, h_errno: %d, hstrerror: %s\n",
        nn::socket::GetLastError(), *(nn::socket::GetHError() ),
        nn::socket::HStrError(  *(nn::socket::GetHError() ) ) );
        goto out;
    }

    for ( respIdx = out_resp; respIdx; respIdx = respIdx->ai_next)
    {
        ipv4 = (nn::socket::SockAddrIn *) (respIdx->ai_addr);

        NN_LOG( "Family: %d, Socktype: %d, Protocol: %d, Flags: 0x%0x, IPAddr: %s, Port: %d, Name: %s\n",
                 respIdx->ai_family, respIdx->ai_socktype, respIdx->ai_protocol,
                 respIdx->ai_flags, nn::socket::InetNtoa( ipv4->sin_addr ),
                 nn::socket::InetNtohs( ipv4->sin_port ), respIdx->ai_canonname );
    }

    // Free Response
    nn::socket::FreeAddrInfo( out_resp );

    return( true );

out:

    return( false );

}

bool
CallGetHostByAddr( const char *  dottedIP, nn::socket::SockLenT addrLen, nn::socket::Family family,
                   nn::socket::ResolverOption *myOption, int optionCount )
{
    nn::socket::InAddr      ipv4Addr;
    nn::socket::HostEnt *   respHosts;
    nn::socket::InAddr **   addr_list;
    int                     idx, rc;

    // Turn English IP Address into octets
    if ( addrLen > 3 )
    {
        rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, dottedIP, &ipv4Addr );
        if ( rc != 1 )
        {
            NN_LOG( "InetPton() failed - rc: %d errno: %d\n", rc, nn::socket::GetLastError() );
            goto out;
        }
    }

    NN_LOG( "Loopup IP Address: %d.%d.%d.%d, Len: %d, family: %d\n",
               (uint8_t) ( ipv4Addr.S_addr ), (uint8_t) ( ipv4Addr.S_addr >> 8 ),
               (uint8_t) ( ipv4Addr.S_addr >> 16 ), (uint8_t) ( ipv4Addr.S_addr >> 24 ),
               addrLen, family );

    if ( optionCount < 1 )
    {
        PRINT_AND_CALL(respHosts = nn::socket::GetHostEntByAddr( &ipv4Addr, addrLen, family ));
    }
    else
    {
        PRINT_AND_CALL(respHosts = nn::socket::GetHostEntByAddr( &ipv4Addr, addrLen, family, myOption, optionCount ));
    }

    if ( respHosts == NULL )
    {
        NN_LOG( "nn::socket::GetHostEntByAddr() Failed with: errno: %d, h_errno: %d, hstrerror: %s\n",
        nn::socket::GetLastError(), *(nn::socket::GetHError() ),
        nn::socket::HStrError(  *(nn::socket::GetHError() ) ) );
        goto out;
    }
    else
    {
        NN_LOG( "Host name       : [%s]\n", respHosts->h_name );
        NN_LOG( "Address Type    : [%d]\n", respHosts->h_addrtype );
        NN_LOG( "Len of Addr     : [%d]\n", respHosts->h_length );

        for(idx = 0; respHosts->h_aliases[idx] != NULL; idx++)
        {
            NN_LOG( "Alias: [%s]\n", respHosts->h_aliases[idx] );
        }

        addr_list = (nn::socket::InAddr **) respHosts->h_addr_list;
        for(idx = 0; addr_list[idx] != NULL; idx++)
        {
            NN_LOG( "IP Address: [%s]\n", nn::socket::InetNtoa( (*addr_list[idx] ) ) );
        }

        NN_LOG( "There are a total of [%d] IPv4 Addresses\n", idx );
    }

    return( true );

out:

    return( false );
}

bool
CallGetNameInfo( const char *ipAddr, nn::socket::ResolverOption *myOption, int optionCount )
{
    nn::socket::SockAddrIn      sa;
    nn::socket::SockLenT        saLen = 0;
    int                         rc = 0;
    char                        rtrnHost[80], rtrnService[80];
    size_t                      hostLen = 80, serviceLen = 80;
    nn::socket::NameInfoFlag    flags = nn::socket::NameInfoFlag::Ni_NameReqd;
    nn::socket::AiErrno         aiErrno = nn::socket::AiErrno::EAi_Success;

    // Initialize IP Address to lookup
    sa.sin_family = nn::socket::Family::Af_Inet;
    sa.sin_port   = nn::socket::InetHtons( 80 );

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, ipAddr, &(sa.sin_addr) );
    if ( rc != 1 )
    {
       NN_LOG( "CallGetNameInfo() - InetPton failed to convert IP Address %s!", ipAddr );
       goto out;
    }

    // Initialize Return Values
    memset( rtrnHost, 0, sizeof( rtrnHost ) );
    memset( rtrnService, 0, sizeof( rtrnService ) );

    // Call GetNameInfo()
    saLen = sizeof( sa );

    if ( optionCount < 1 )
    {
        PRINT_AND_CALL(aiErrno = nn::socket::GetNameInfo( reinterpret_cast<nn::socket::SockAddr *>(&sa), saLen, rtrnHost, hostLen,
                                                          rtrnService, serviceLen, flags ));
    }
    else
    {
        PRINT_AND_CALL(aiErrno = nn::socket::GetNameInfo( reinterpret_cast<nn::socket::SockAddr *>(&sa), saLen, rtrnHost, hostLen,
                                                          rtrnService, serviceLen, flags, myOption, optionCount ));
    }

    if ( aiErrno == nn::socket::AiErrno::EAi_Success )
    {
        NN_LOG( "GNI: Good return Code, Returned, rc: %d\n", rc );
        NN_LOG( "Rtrn Host    : [%.*s]\n", hostLen, rtrnHost );
        NN_LOG( "Rtrn Service : [%.*s]\n", serviceLen, rtrnService );

        // Fall thru
    }
    else
    {
        NN_LOG( "nn::socket::CallGetNameInfo() Failed with: aiErrno: %d, errno: %d, h_errno: %d, hstrerror: %s\n",
                 aiErrno, nn::socket::GetLastError(), *(nn::socket::GetHError() ),
                 nn::socket::HStrError(  *(nn::socket::GetHError() ) ) );
        goto out;
    }

    return( true );

out:

    return( false );

}

static bool
TeardownTesting()
{
     bool    isSuccess = true;

    /////////////////////////
    // Stop 2nd DNS Server
    /////////////////////////

    // If the (Second) loopback DNS server is allocated
    if ( g_LoopbackDnsServer2 != NULL )
    {
        NN_LOG( "Stopping 2nd DNS Server\n" );
        g_LoopbackDnsServer2->Stop();
        delete g_LoopbackDnsServer2;
        g_LoopbackDnsServer2 = NULL;
    }

    ///////////////////
    // Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping 1st DNS Server\n" );

    // If the loopback DNS server is allocated
    if ( g_LoopbackDnsServer != NULL )
    {
        g_LoopbackDnsServer->Stop();
        delete g_LoopbackDnsServer;
        g_LoopbackDnsServer = NULL;
    }

    // Wait DNS Server threads to die
    g_pThreadedTest->WaitAllThreads();

    // Destroy all Threads
    g_pThreadedTest->DestroyAllThreads();

    // Delete Object Pointer
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;

    NN_LOG( "DNS Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

out:

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );
}

} // Anonymous

TEST(ApiUnit,DnsCache_Initialize)
{
    InitializeTesting();
}

static int
GetLabelLength( unsigned char *pStartDnsLabel )
{
    int size = 0;
    unsigned char *c_ptr = pStartDnsLabel;

    while( *c_ptr != 0x00 )
    {
        size++;
        c_ptr++;
    }

    return( size );
}


static int
DnsCache_Simple_Caching( void *                  inArg,
                         unsigned char *         dnsRqst,
                         size_t &                dnsRqstLen,
                         unsigned char *         dnsResp,
                         size_t &                dnsRespLen,
                         bool *                  isUDP,
                         bool *                  shouldContinue )
{
    LoopbackDnsServer *   thisObj = (LoopbackDnsServer *) inArg;

    // Server HIT!
    g_ServerHit = true;

    // Process UDP normally
    return( thisObj->MakeDnsResponseCB( thisObj, dnsRqst, dnsRqstLen, dnsResp, dnsRespLen, isUDP, shouldContinue )  );
}


TEST(ApiUnit,DnsCache_GoodResponse_UDP)
{
    bool              isSuccess = true, rval;

    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB(  DnsCache_Simple_Caching  );

    // Set alias.landsberger.com Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheAliasLandsbergberComCNAME;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheAliasLandsbergberComCNAME );

    // Flush Cache
    CacheFlush();

    //*********************
    //* GHBN
    //*********************

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetHostByName][alias.landsberger.com]\n" );
    rval = CallGetHostByName( "alias.landsberger.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GHBN failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBN (did not) query the DNS server, but should have!" );

    g_ServerHit = false;
    NN_LOG( "====> Get from CACHE\n" );
    NN_LOG( "Query: [GetHostByName][alias.landsberger.com]\n" );
    rval = CallGetHostByName( "alias.landsberger.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GHBN failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == true, "GHBN queried the DNS server, but (should not) have!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "alias.landsberger.com" ) == false, "Domain name should be cached, but it's not!\n" );
    ERROR_IF_AND_COUNT( CacheRemoveHostname( "alias.landsberger.com" ) == -1, "Failed calling CacheRemoveHostname()\n" );

    //*********************
    //* GETADDRINFO
    //*********************

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetAddrInfo][alias.landsbergber.com]\n" );
    rval = CallGetAddrInfo( "alias.landsberger.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GAI failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GAI (did not) query the DNS server, but should have!" );

    g_ServerHit = false;
    NN_LOG( "====> Get from CACHE\n" );
    NN_LOG( "Query: [GetAddrInfo][alias.landsbergber.com]\n" );
    rval = CallGetAddrInfo( "alias.landsberger.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GAI failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == true, "GAI queried the DNS server, but (should not) have!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "alias.landsberger.com" ) == false, "Domain name should be cached, but it's not!\n" );
    ERROR_IF_AND_COUNT( CacheRemoveHostname( "alias.landsberger.com" ) == -1, "Failed calling CacheRemoveHostname()\n" );

    //*********************
    //* GHBA
    //*********************

    // Set ERoot DNS Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheERootComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheERootComSimple );

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GHBA failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBA (did not) query the DNS server, but should have!" );

    g_ServerHit = false;
    NN_LOG( "====> Get from CACHE\n" );
    rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GHBA failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == true, "GHBA queried the DNS server, but (should not) have!" );
    ERROR_IF_AND_COUNT( IsIPInCache( "192.203.230.10" ) == false, "IP Address should be cached, but it's not!\n" );
    ERROR_IF_AND_COUNT( CacheRemoveIpAddress( "192.203.230.10" ) == -1, "Failed CacheRemoveIpAddress\n" );

    //*********************
    //* GNI
    //*********************

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetNameInfo][ERoot] Returns wrong answer\n" );
    rval = CallGetNameInfo( "192.203.230.10", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GNI failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GNI (did not) query the DNS server, but should have!" );

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetNameInfo][ERoot] Returns wrong answer\n" );
    rval = CallGetNameInfo( "192.203.230.10", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GNI failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == true, "GNI queried the DNS server, but (should not) have!" );
    ERROR_IF_AND_COUNT( IsIPInCache( "192.203.230.10" ) == false, "IP Address should be cached, but it's not!\n" );
    ERROR_IF_AND_COUNT( CacheRemoveIpAddress( "192.203.230.10" ) == -1, "Failed CacheRemoveIpAddress\n" );

    //*********************
    //* GHBN - EDNS0 - 4K
    //*********************

    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheGoogleComSimpleWithEDNS04k;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheGoogleComSimpleWithEDNS04k );

    CacheRemoveHostname( "google.com" );
    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetHostByName][google.com - EDNS0/4k]\n" );
    rval = CallGetHostByName( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GHBN failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBN (did not) query the DNS server, but should have!" );

    g_ServerHit = false;
    NN_LOG( "====> Get from CACHE\n" );
    NN_LOG( "Query: [GetHostByName][google.com - EDNS0/4k]\n" );
    rval = CallGetHostByName( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GHBN failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == true, "GHBN queried the DNS server, but (should not) have!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == false, "Domain name should be cached, but it's not!\n" );
    ERROR_IF_AND_COUNT( CacheRemoveHostname( "google.com" ) == -1, "Failed calling CacheRemoveHostname()\n" );

    //*********************
    //* GETADDRINFO
    //*********************

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetAddrInfo][google.com - EDNS0/4k]\n" );
    rval = CallGetAddrInfo( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GAI failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GAI (did not) query the DNS server, but should have!" );

    g_ServerHit = false;
    NN_LOG( "====> Get from CACHE\n" );
    NN_LOG( "Query: [GetAddrInfo][google.com - EDNS0/4k]\n" );
    rval = CallGetAddrInfo( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GAI failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == true, "GAI queried the DNS server, but (should not) have!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == false, "Domain name should be cached, but it's not!\n" );
    ERROR_IF_AND_COUNT( CacheRemoveHostname( "google.com" ) == -1, "Failed calling CacheRemoveHostname()\n" );

out:

    return;
}


TEST(ApiUnit,DnsCache_BadResponse_UDP)
{
    bool              isSuccess = true, rval;

    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB(  DnsCache_Simple_Caching  );

    // Set UDP Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheGoogleComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheGoogleComSimple );
    g_LoopbackDnsServer->UDPResponseLen = g_LoopbackDnsServer->UDPResponseLen - 1;

    // Flush Cache
    CacheFlush();

    //*********************
    //* GHBN
    //*********************

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetHostByName][google.com]\n" );
    rval = CallGetHostByName( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GHBN succeeded but should have failed!" );

    g_ServerHit = false;
    NN_LOG( "====> Get from CACHE\n" );
    NN_LOG( "Query: [GetHostByName][google.com]\n" );
    rval = CallGetHostByName( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GHBN succeeded but should have failed!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBN accessed the DNS Cache, but (should not) have!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true, "Domainname is in the DNS cache, but should not be!\n" );

    //*********************
    //* GETADDRINFO
    //*********************

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetAddrInfo][google,com]\n" );
    rval = CallGetAddrInfo( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GAI succeeded but should have failed!" );

    g_ServerHit = false;
    NN_LOG( "====> Get from CACHE\n" );
    NN_LOG( "Query: [GetAddrInfo][google.com]\n" );
    rval = CallGetAddrInfo( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GAI succeeded but should have failed!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GAI accessed the DNS Cache, but (should not) have!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true, "Domainname is in the DNS cache, but should not be!\n" );

    //*********************
    //* GHBA
    //*********************

    // Set ERoot DNS Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheERootComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheERootComSimple );
    g_LoopbackDnsServer->UDPResponseLen = g_LoopbackDnsServer->UDPResponseLen - 1;

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GHBA succeeded but should have failed!" );

    g_ServerHit = false;
    NN_LOG( "====> Get from CACHE\n" );
    rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GHBA succeeded but should have failed!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBA accessed the DNS Cache, but (should not) have!" );
    ERROR_IF_AND_COUNT( IsIPInCache( "192.203.230.10" ) == true, "IP Address is in cache, but it should not be!\n" );

    //*********************
    //* GNI
    //*********************

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetNameInfo][ERoot] Returns wrong answer\n" );
    rval = CallGetNameInfo( "192.203.230.10", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GNI succeeded but should have failed!" );

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetNameInfo][ERoot] Returns wrong answer\n" );
    rval = CallGetNameInfo( "192.203.230.10", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GNI succeeded but should have failed!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GNI accessed the DNS Cache, but (should not) have!" );
    ERROR_IF_AND_COUNT( IsIPInCache( "192.203.230.10" ) == true, "IP Address is in cache, but it should not be!\n" );


WRAP_FAILING_TEST( "SIGLO-67348", "[NX] - DNS parser pointer to (self) causes bad cache entry" )
{
    ///////////////////
    // Pointer To Self Bug Test
    ///////////////////

    g_LoopbackDnsServer->UDPResponse = (unsigned char *) GoogleComLabelPointerToSelfBug;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( GoogleComLabelPointerToSelfBug );

    // GetAddrInfo()
    NN_LOG( "Query: [GetAddrInfo][Google] Pointer to (Self)\n" );
    rval = CallGetAddrInfo( "google.com", nullptr, 0);
    ERROR_IF_AND_COUNT( rval == true, "GAI should have failed testing [Pointer to (Self)] - but actually passed!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true, "Domainname is in the DNS cache, but should not be!\n" );

    // GetHostByName()
    NN_LOG( "Query: [GetHostByName][Google] Pointer to (Self)\n" );
    rval = CallGetHostByName( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GHBN should have failed testing [Pointer to (Self)] - but actually passed!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true, "Domainname is in the DNS cache, but should not be!\n" );
}


    ///////////////////
    // Pointer To Pointer To Pointer to Self Bug Test
    ///////////////////

    g_LoopbackDnsServer->UDPResponse = (unsigned char *) GoogleComLabelPointerToPointerToSelfBug;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( GoogleComLabelPointerToPointerToSelfBug );

    // GetAddrInfo()
    NN_LOG( "Query: [GetAddrInfo][Google] Pointer to Pointer to (Self)\n" );
    rval = CallGetAddrInfo( "google.com", nullptr, 0);
    ERROR_IF_AND_COUNT( rval == true, "GAI should have failed testing [Pointer to Pointer to (Self)] - but actually passed!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true, "Domainname is in the DNS cache, but should not be!\n" );

    // GetHostByName()
    NN_LOG( "Query: [GetHostByName][Google] Pointer to Pointer to (Self)\n" );
    rval = CallGetHostByName( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GHBN should have failed testing [Pointer to Pointer to (Self)] - but actually passed!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true, "Domainname is in the DNS cache, but should not be!\n" );


    ///////////////////
    // Pointer To Pointer To Pointer
    ///////////////////

    g_LoopbackDnsServer->UDPResponse = (unsigned char *) GoogleComLabelPointerToPointerBug;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( GoogleComLabelPointerToPointerBug );

    // GetAddrInfo()
    NN_LOG( "Query: [GetAddrInfo][Google] Pointer to Pointer\n" );
    rval = CallGetAddrInfo( "google.com", nullptr, 0);
    ERROR_IF_AND_COUNT( rval == true, "GAI should have failed testing [Pointer to Pointer] - but actually passed!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true, "Domainname is in the DNS cache, but should not be!\n" );

    // GetHostByName()
    NN_LOG( "Query: [GetHostByName][Google] Pointer to Pointer\n" );
    rval = CallGetHostByName( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GHBN should have failed testing [Pointer to Pointer] - but actually passed!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true, "Domainname is in the DNS cache, but should not be!\n" );


    ///////////////////////////////////////
    //// Return WRONG response via UDP
    ///////////////////////////////////////

    // UDP: Request Google, Response: 123456.com
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCache123456ComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCache123456ComSimple );

    // GetAddrInfo()
    NN_LOG( "Query: [GetAddrInfo][Google] wrong response from UDP\n" );
    rval = CallGetAddrInfo( "google.com", nullptr, 0);
    ERROR_IF_AND_COUNT( rval == true, "GAI should have failed testing [Wrong response from UDP] - but actually passed!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true, "Domainname is in the DNS cache, but should not be!\n" );

    // GetHostByName()
    NN_LOG( "Query: [GetHostByName][Google] wrong response from UDP\n" );
    rval = CallGetHostByName( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GHBN should have failed testing [Wrong response from UDP] - but actually passed!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true, "Domainname is in the DNS cache, but should not be!\n" );

out:

    return;

}  // NOLINT(impl/function_size)


TEST(ApiUnit,DnsCache_BadResponse_TCP)
{
    bool              isSuccess = true, rval;

    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB(  DnsCache_Simple_Caching  );

    // UDP: google.com + truncate bit
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheGoogleComSimpleWithTC;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheGoogleComSimpleWithTC );

    // TCP: google.com - but bad response len
    g_LoopbackDnsServer->TCPResponse = (unsigned char *) DnsCacheGoogleComSimple;
    g_LoopbackDnsServer->TCPResponseLen = sizeof( DnsCacheGoogleComSimple );
    g_LoopbackDnsServer->TCPResponseLen = g_LoopbackDnsServer->TCPResponseLen - 1;

    // Flush Cache
    CacheFlush();

    //*********************
    //* GHBN
    //*********************

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetHostByName][google.com]\n" );
    rval = CallGetHostByName( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GHBN succeeded but should have failed!" );

    g_ServerHit = false;
    NN_LOG( "====> Get from CACHE\n" );
    NN_LOG( "Query: [GetHostByName][google.com]\n" );
    rval = CallGetHostByName( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GHBN succeeded but should have failed!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBN accessed the DNS Cache, but (should not) have!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true, "Domainname is in the DNS cache, but should not be!\n" );

    //*********************
    //* GETADDRINFO
    //*********************

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetAddrInfo][google,com]\n" );
    rval = CallGetAddrInfo( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GAI succeeded but should have failed!" );

    g_ServerHit = false;
    NN_LOG( "====> Get from CACHE\n" );
    NN_LOG( "Query: [GetAddrInfo][google.com]\n" );
    rval = CallGetAddrInfo( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GAI succeeded but should have failed!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GAI accessed the DNS Cache, but (should not) have!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true, "Domainname is in the DNS cache, but should not be!\n" );

    //*********************
    //* GHBA
    //*********************

    // UDP: ERoot with TC bit
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheERootComSimpleWithTC;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheERootComSimpleWithTC );

    // TCP: ERoot - but bad response len
    g_LoopbackDnsServer->TCPResponse = (unsigned char *) DnsCacheERootComSimple;
    g_LoopbackDnsServer->TCPResponseLen = sizeof( DnsCacheERootComSimple );
    g_LoopbackDnsServer->TCPResponseLen = g_LoopbackDnsServer->TCPResponseLen - 1;


    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GHBA succeeded but should have failed!" );

    g_ServerHit = false;
    NN_LOG( "====> Get from CACHE\n" );
    rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GHBA succeeded but should have failed!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBA accessed the DNS Cache, but (should not) have!" );
    ERROR_IF_AND_COUNT( IsIPInCache( "192.203.230.10" ) == true, "IP Address is in cache, but it should not be!\n" );

    //*********************
    //* GNI
    //*********************

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetNameInfo][ERoot] Returns wrong answer\n" );
    rval = CallGetNameInfo( "192.203.230.10", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GNI succeeded but should have failed!" );

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetNameInfo][ERoot] Returns wrong answer\n" );
    rval = CallGetNameInfo( "192.203.230.10", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GNI succeeded but should have failed!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GNI accessed the DNS Cache, but (should not) have!" );
    ERROR_IF_AND_COUNT( IsIPInCache( "192.203.230.10" ) == true, "IP Address is in cache, but it should not be!\n" );

    ///////////////////////////////////////
    //// Return WRONG response via TCP
    ///////////////////////////////////////

    // UDP: Google with the TC bit set
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheGoogleComSimpleWithTC;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheGoogleComSimpleWithTC );

    // TCP: 123456.com (without) TC bit set
    g_LoopbackDnsServer->TCPResponse = (unsigned char *) DnsCache123456ComSimple;
    g_LoopbackDnsServer->TCPResponseLen = sizeof( DnsCache123456ComSimple );

    // GetAddrInfo()
    NN_LOG( "Query: [GetAddrInfo][Google] wrong response from TCP\n" );
    rval = CallGetAddrInfo( "google.com", nullptr, 0);
    ERROR_IF_AND_COUNT( rval == true, "GAI should have failed testing [Wrong response from TCP] - but actually passed!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true, "Domainname is in the DNS cache, but should not be!\n" );

    // GetHostByName()
    NN_LOG( "Query: [GetHostByName][Google] wrong response from TCP\n" );
    rval = CallGetHostByName( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == true, "GHBN should have failed testing [Wrong response from TCP] - but actually passed!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true, "Domainname is in the DNS cache, but should not be!\n" );

out:

    return;
}


TEST(ApiUnit,DnsCache_GoodResponse_TCP)
{
    bool              isSuccess = true, rval;

    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB(  DnsCache_Simple_Caching  );

    // UDP: google.com + truncate bit
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheGoogleComSimpleWithTC;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheGoogleComSimpleWithTC );

    // TCP: google.com - but bad response len
    g_LoopbackDnsServer->TCPResponse = (unsigned char *) DnsCacheGoogleComSimple;
    g_LoopbackDnsServer->TCPResponseLen = sizeof( DnsCacheGoogleComSimple );

    // Flush Cache
    CacheFlush();

    //*********************
    //* GHBN
    //*********************

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetHostByName][google.com]\n" );
    rval = CallGetHostByName( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GHBN failed but should have succeeded!" );

    g_ServerHit = false;
    NN_LOG( "====> Get from CACHE\n" );
    NN_LOG( "Query: [GetHostByName][google.com]\n" );
    rval = CallGetHostByName( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GHBN DNS lookup failed - but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == true, "GHBN queried (a DNS Server), but (should not) have!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == false,
                        "Domainname is NOT in the DNS cache, but it should be!\n" );
    ERROR_IF_AND_COUNT( CacheRemoveHostname( "google.com" ) == -1, "Failed calling CacheRemoveHostname()\n" );

    //*********************
    //* GETADDRINFO
    //*********************

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetAddrInfo][google,com]\n" );
    rval = CallGetAddrInfo( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GAI failed but should have succeeded!" );

    g_ServerHit = false;
    NN_LOG( "====> Get from CACHE\n" );
    NN_LOG( "Query: [GetAddrInfo][google.com]\n" );
    rval = CallGetAddrInfo( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GAI DNS lookup failed - but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == true, "GAI queried (a DNS Server), but (should not) have!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == false,
                        "Domainname is NOT in the DNS cache, but it should be!\n" );
    ERROR_IF_AND_COUNT( CacheRemoveHostname( "google.com" ) == -1, "Failed calling CacheRemoveHostname()\n" );

    //*********************
    //* GHBA
    //*********************

    // UDP: ERoot with TC bit
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheERootComSimpleWithTC;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheERootComSimpleWithTC );

    // TCP: ERoot - but bad response len
    g_LoopbackDnsServer->TCPResponse = (unsigned char *) DnsCacheERootComSimple;
    g_LoopbackDnsServer->TCPResponseLen = sizeof( DnsCacheERootComSimple );

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetHostByAddr][ERoot]\n" );
    rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GHBA failed but should have succeeded!" );

    g_ServerHit = false;
    NN_LOG( "====> Get from CACHE\n" );
    NN_LOG( "Query: [GetHostByAddr][ERoot]\n" );
    rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GHBA DNS lookup failed - but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == true, "GHBA queried (a DNS Server), but (should not) have!" );
    ERROR_IF_AND_COUNT( IsIPInCache( "192.203.230.10" ) == false,
                        "IP Address is NOT in cache, but it should be!\n" );
    ERROR_IF_AND_COUNT( CacheRemoveIpAddress( "192.203.230.10" ) == -1, "Failed CacheRemoveIpAddress\n" );

    //*********************
    //* GNI
    //*********************

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetNameInfo][ERoot]\n" );
    rval = CallGetNameInfo( "192.203.230.10", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GNI failed but should have succeeded!" );

    g_ServerHit = false;
    NN_LOG( "====> Add to CACHE\n" );
    NN_LOG( "Query: [GetNameInfo][ERoot]\n" );
    rval = CallGetNameInfo( "192.203.230.10", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GNI DNS lookup failed - but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == true, "GNI queried (a DNS Server), but (should not) have!" );
    ERROR_IF_AND_COUNT( IsIPInCache( "192.203.230.10" ) == false,
                        "IP Address is NOT in cache, but should be!\n" );
    ERROR_IF_AND_COUNT( CacheRemoveIpAddress( "192.203.230.10" ) == -1, "Failed CacheRemoveIpAddress\n" );

out:

    return;
}


static int
DnsCache_Bad_Response_Len( void *              inArg,
                           unsigned char *     dnsRqst,
                           size_t &            dnsRqstLen,
                           unsigned char *     dnsResp,
                           size_t &            dnsRespLen,
                           bool *              isUDP,
                           bool *              shouldContinue )
{
    struct DNS_HEADER_REC *    pDnsHeader;
    uint16_t                   saveId;
    LoopbackDnsServer *        thisObj = (LoopbackDnsServer *) inArg;

    // Save DNS Request Transaction Id
    pDnsHeader = (struct DNS_HEADER_REC *) &dnsRqst[0];
    saveId = nn::socket::InetNtohs( pDnsHeader->Id );

    // Fill in the right Response
    memset( dnsResp, 0, dnsRespLen );

    // Work var initialized in (other) thread
    g_WorkVaruint32_t++;
    if (  g_WorkVaruint32_t < thisObj->UDPResponseLen )
    {
        memcpy( dnsResp, thisObj->UDPResponse, g_WorkVaruint32_t );
        dnsRespLen = g_WorkVaruint32_t;
    }
    else
    {
        memcpy( dnsResp, thisObj->UDPResponse, thisObj->UDPResponseLen );
        dnsRespLen    = thisObj->UDPResponseLen;
        g_workVarBool = false;
    }

    // Fix DNS Response
    pDnsHeader = (struct DNS_HEADER_REC *) dnsResp;
    pDnsHeader->Id = nn::socket::InetHtons( saveId );

    return( 0 );
}


TEST(ApiUnit,DnsCache_Bad_Response_Len)
{
    bool              rval, isSuccess = true;

// WRAP_FAILING_TEST( "SIGLO-67474", "[NX] - Resolver Crash: Test: DNS Response truncated" )
{
    // Bad Response Len
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB( DnsCache_Bad_Response_Len );

    // Simple Google
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheGoogleComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheGoogleComSimple );

    // Initialize Work Vars
    g_WorkVaruint32_t = 2;
    g_workVarBool     = true;

    // Flush Cache
    CacheFlush();

    // For Idx = 2 to End of DNS Request
    while( g_workVarBool == true )
    {
        NN_LOG( "====> Add to CACHE\n" );
        NN_LOG( "Query: [GetHostByName][google.com]\n" );
        rval = CallGetHostByName( "google.com", nullptr, 0 );

        if ( rval == true )
        {
            ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == false,
                                "GHBN: received a GOOD response, but response was NOT saved into the Cache\n" );

            ERROR_IF_AND_COUNT( CacheRemoveHostname( "google.com" ) == -1, "Failed calling CacheRemoveHostname()\n" );
        }
        else
        {
            ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true,
                                "GHBN: received a FAILED response, but response was saved into the Cache!\n" );
        }
    }

    NN_LOG( "Successfully tested Bad Response Length\n" );

    // Test Passed
    nTestsPassing++;

out:

    return;
}

}


static int
DnsCache_Bad_Response_Data( void *              inArg,
                            unsigned char *     dnsRqst,
                            size_t &            dnsRqstLen,
                            unsigned char *     dnsResp,
                            size_t &            dnsRespLen,
                            bool *              isUDP,
                            bool *              shouldContinue )
{
    struct DNS_HEADER_REC *    pDnsHeader;
    uint16_t                   saveId;
    LoopbackDnsServer *        thisObj = (LoopbackDnsServer *) inArg;

    // Save DNS Request Transaction Id
    pDnsHeader = (struct DNS_HEADER_REC *) dnsRqst;
    saveId = nn::socket::InetNtohs( pDnsHeader->Id );

    // Fill in the right Response
    memset( dnsResp, 0, dnsRespLen );

    // Copy and Place (bad value) into Buffer
    g_WorkVaruint32_t++;
    if ( g_WorkVaruint32_t < thisObj->UDPResponseLen )
    {
        memcpy( dnsResp, thisObj->UDPResponse, thisObj->UDPResponseLen );
        NN_LOG( "Value: %d written at positon: %d\n", g_WorkVaruchar, g_WorkVaruint32_t );
        dnsResp[g_WorkVaruint32_t] = g_WorkVaruchar;
        dnsRespLen                 = thisObj->UDPResponseLen;
    }
    else
    {
        memcpy( dnsResp, thisObj->UDPResponse, thisObj->UDPResponseLen );
        dnsRespLen = thisObj->UDPResponseLen;
        g_workVarBool = false;
    }

    // Fix DNS Response
    pDnsHeader = (struct DNS_HEADER_REC *) dnsResp;
    pDnsHeader->Id = nn::socket::InetHtons( saveId );

    return( 0 );
}


TEST(ApiUnit,DnsCache_Bad_Response_Data)
{
    bool              rval, isSuccess = false;
    int               idx = 0;
    static unsigned char testValues[] = { 0x0, 0x1, 0x0c, 0xff };

    // Normal DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB( DnsCache_Bad_Response_Data );

    // Simple Google
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheGoogleComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheGoogleComSimple );

    // Initialize Work Vars
    g_WorkVaruint32_t = 2;
    g_workVarBool     = true;
    g_WorkVaruchar    = 0xFF;

    // Flush Cache
    CacheFlush();

    //////////////////
    // High Values = 0xFF
    //////////////////

    for( idx = 0; idx < sizeof( testValues) ; idx++ )
    {
        NN_LOG( "==> Testing: %d/0x%02x\n", testValues[ idx ], testValues[ idx ] );

        // Initialize Work Vars
        g_WorkVaruint32_t = 11;   // 2
        g_workVarBool     = true;
        g_WorkVaruchar    = (char) testValues[ idx ];

        // For Idx = 2 to End of DNS Request
        while( g_workVarBool == true )
        {
            ERROR_IF_AND_COUNT( CacheRemoveHostname( "google.com" ) == -1, "Failed to remove [google.com] from the Cache!\n" );

            NN_LOG( "====> Add to CACHE\n" );
            NN_LOG( "Query: [GetHostByName][google.com]\n" );
            rval = CallGetHostByName( "google.com", nullptr, 0 );

            if ( rval == false )
            {
                ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true,
                                    "GHBN: received a FAILED response, but response was saved into the Cache!\n" );
            }
        }
    }

    // Test Passed
    nTestsPassing++;

out:

    return;
}


static int
DnsCache_Fill_Cache_GHBN( void *                  inArg,
                          unsigned char *         dnsRqst,
                          size_t &                dnsRqstLen,
                          unsigned char *         dnsResp,
                          size_t &                dnsRespLen,
                          bool *                  isUDP,
                          bool *                  shouldContinue )
{
    struct DNS_HEADER_REC * pDnsHeader;
    uint16_t                saveId = 0x00;
    LoopbackDnsServer * thisPtr = (LoopbackDnsServer *) inArg;

    // Save DNS Request Transaction Id
    pDnsHeader = (struct DNS_HEADER_REC *) dnsRqst;
    saveId = nn::socket::InetNtohs( pDnsHeader->Id );

    // Fill in the right Response
    memset( dnsResp, 0, dnsRespLen );

    memcpy( dnsResp, thisPtr->UDPResponse, thisPtr->UDPResponseLen );
    dnsRespLen = thisPtr->UDPResponseLen;

    // Fix DNS Response
    pDnsHeader = (struct DNS_HEADER_REC *) dnsResp;
    pDnsHeader->Id    = nn::socket::InetHtons( saveId );

    // Server HIT!
    g_ServerHit = true;

    // Save new Domainname
    dnsResp[ g_OffsetDnsResp ] = g_SharedChar;

    // Process UDP normally
    return( 0 );
}


TEST(ApiUnit,DnsCache_Fill_Cache_GHBN)
{
    int             offsetDnsRqst;
    char            domainname[256];
    const char *    domainnametmpl = "123456789012345678901234567890123456789012345678901234567890.com";
    bool            rval = false, isSuccess = false;
    uint32_t        totalEntries = 0;
    int             domainCount;

    // Flush the Cache
    CacheFlush();

    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB(  DnsCache_Fill_Cache_GHBN );

    // Set 123456789012345678901234567890123456789012345678901234567890.com
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCache123456ComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCache123456ComSimple );

    // Initialize DNS request/response
    memset( domainname, 0, sizeof( domainname ) );

    g_SharedChar    = 'A';
    offsetDnsRqst = 0;
    g_OffsetDnsResp = 13;
    domainCount   = 0;

    // Forever
    for(;;)
    {
        // If we are out of alphabet
        g_SharedChar++;
        if ( g_SharedChar < 'A' || g_SharedChar > 'Z' )
        {
            domainCount++;
            if ( domainCount >= 60 )
            {
                NN_LOG( "60 character length Label processed - Finished processing!  Total Entries: %d\n", totalEntries );
                break;
            }

            g_SharedChar = 'A';
            offsetDnsRqst++;
            g_OffsetDnsResp++;
        }

        memcpy( domainname, domainnametmpl, strlen( domainnametmpl ) );
        domainname[ offsetDnsRqst ] = g_SharedChar;

        g_ServerHit = false;
        NN_LOG( "Query: [GetHostByName][%s]\n", domainname );
        rval = CallGetHostByName( domainname, nullptr, 0);
        ERROR_IF_AND_COUNT( rval == false, "GHBN failed but should have succeeded!" );
        ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBN (did not) query the DNS server, but should have!" );
        ERROR_IF_AND_COUNT( IsDomainNameInCache( domainname ) == false,
                            "GHBN: received a GOOD response, but response was NOT saved into the Cache\n" );

        totalEntries++;
    }

out:

    return;
}

static int
DnsCache_Fill_Cache_GHBA( void *                  inArg,
                          unsigned char *         dnsRqst,
                          size_t &                dnsRqstLen,
                          unsigned char *         dnsResp,
                          size_t &                dnsRespLen,
                          bool *                  isUDP,
                          bool *                  shouldContinue )
{
    struct DNS_HEADER_REC * pDnsHeader;
    uint16_t                saveId = 0x00;
    LoopbackDnsServer * thisPtr = (LoopbackDnsServer *) inArg;

    // Save DNS Request Transaction Id
    pDnsHeader = (struct DNS_HEADER_REC *) dnsRqst;
    saveId = nn::socket::InetNtohs( pDnsHeader->Id );

    // Fill in the right Response
    memset( dnsResp, 0, dnsRespLen );

    memcpy( dnsResp, thisPtr->UDPResponse, thisPtr->UDPResponseLen );
    dnsRespLen = thisPtr->UDPResponseLen;

    // Fix DNS Response
    pDnsHeader = (struct DNS_HEADER_REC *) dnsResp;
    pDnsHeader->Id    = nn::socket::InetHtons( saveId );

    // Server HIT!
    g_ServerHit = true;

    // Save new Domainname
    dnsResp[ g_OffsetDnsResp ]     = g_SharedChar;
    dnsResp[ g_OffsetDnsResp + 1 ] = g_SharedChar2;

    // Process UDP normally
    return( 0 );
}

TEST(ApiUnit,DnsCache_Fill_Cache_GHBA)
{
    int             offsetDnsRqst = -1;
    char            domainname[256];
    const char *    domainnametmpl = "192.203.230.10";
    bool            rval = false, isSuccess = false;
    uint32_t        totalEntries = 0;
    nn::socket::SockLenT    domainnamelen = 0;

    // Clear the DNS Cache
    CacheFlush();

    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB(  DnsCache_Fill_Cache_GHBA );


    // Set alias.landsberger.com Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheERootComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheERootComSimple );

    // Initialize DNS request/response
    memset( domainname, 0, sizeof( domainname ) );
    memcpy( domainname, domainnametmpl, strlen( domainnametmpl ) );


    g_SharedChar    = '1';
    g_SharedChar2   = '0';
    offsetDnsRqst   = 12;
    g_OffsetDnsResp = 13;

    // Forever
    for(;;)
    {

        // If we are out of alphabet
        g_SharedChar2++;
        if ( g_SharedChar2 < '0' || g_SharedChar2 > '9' )
        {
            g_SharedChar2 = '0';
            g_SharedChar++;

            if ( g_SharedChar > '9' )
            {
                NN_LOG( "Finished subnet\n" );
                break;
            }
        }

        domainname[ offsetDnsRqst ]     = g_SharedChar;
        domainname[ offsetDnsRqst  + 1] = g_SharedChar2;

        domainnamelen = strlen( domainname );

        g_ServerHit = false;
        NN_LOG( "Query: [GetHostByAddr][%s]\n", domainname );
        rval = CallGetHostByAddr( domainname, 4, nn::socket::Family::Af_Inet, nullptr, 0 );
        ERROR_IF_AND_COUNT( rval == false, "GHBA failed but should have succeeded!" );
        ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBN (did not) query the DNS server, but should have!" );
        ERROR_IF_AND_COUNT( IsIPInCache( domainname ) == false,
                            "GHBA: received a GOOD response, but response was NOT saved into the Cache\n" );

        totalEntries++;
    }

out:

    return;
}


static int
DnsCache_Timeout( void *                  inArg,
                  unsigned char *         dnsRqst,
                  size_t &                dnsRqstLen,
                  unsigned char *         dnsResp,
                  size_t &                dnsRespLen,
                  bool *                  isUDP,
                  bool *                  shouldContinue )
{
    LoopbackDnsServer *   thisObj = (LoopbackDnsServer *) inArg;

    // Server HIT!
    g_ServerHit = true;

    // Process UDP normally
    return( thisObj->MakeDnsResponseCB( thisObj, dnsRqst, dnsRqstLen, dnsResp, dnsRespLen, isUDP, shouldContinue )  );
}

TEST(ApiUnit,DnsCache_Timeout)
{
    int                idx = 0;
    bool               isSuccess = false, rval = false;
    nn::os::Tick       startTick, stopTick;

    // Flush the Cache
    CacheFlush();

    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB(  DnsCache_Timeout );

    // Set alias.landsberger.com Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheGoogleComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheGoogleComSimple );

    //*********************
    //* GHBN
    //*********************

    g_ServerHit = false;
    NN_LOG( "Query: [GetHostByName][google.com]\n" );
    rval = CallGetHostByName( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GHBN failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBN (did not) query the DNS server, but should have!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == false,
                        "GHBN: received a GOOD response, but response was NOT saved into the Cache\n" );
    ERROR_IF_AND_COUNT( CacheUpdateTimeToLiveForDomainname( "google.com", 15 ) == -1, "Failed to updated TTL for DNS Cache Entry\n" );

    startTick = nn::os::GetSystemTick();
    for( idx = 0; idx < 20; idx++ )
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        g_ServerHit = false;
        NN_LOG( "Query: [GetHostByName][google.com]\n" );
        rval = CallGetHostByName( "google.com", nullptr, 0 );
        stopTick = nn::os::GetSystemTick();
        ERROR_IF_AND_COUNT( rval == false, "GHBN failed but should have succeeded!" );

        auto diffsec = (stopTick.ToTimeSpan() - startTick.ToTimeSpan()).GetSeconds();
        NN_LOG("seconds: %ld\n", diffsec);
        if ( g_ServerHit == true )
        {
            ERROR_IF_AND_COUNT( diffsec < 14, "GHBN: Cache TTL set to 15 seconds, but DNS server was HIT in: %d seconds\n", diffsec );
            NN_LOG( "GHBN: DNS Server was hit - DNS Cache entry timed out after: %d seconds\n", diffsec );
            break;
        }
    }

    ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBN: DNS Cache record NEVER timed out after 20 seconds\n" );
    ERROR_IF_AND_COUNT( CacheRemoveHostname( "google.com" ) == -1, "Failed calling CacheRemoveHostname()\n" );

    //*********************
    //* GETADDRINFO
    //*********************

    g_ServerHit = false;
    NN_LOG( "Query: [GetAddrInfo][google,com]\n" );
    rval = CallGetAddrInfo( "google.com", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GAI failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GAI (did not) query the DNS server, but should have!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == false,
                        "GAI: received a GOOD response, but response was NOT saved into the Cache\n" );
    ERROR_IF_AND_COUNT( CacheUpdateTimeToLiveForDomainname( "google.com", 15 ) == -1, "Failed to updated TTL for DNS Cache Entry\n" );

    startTick = nn::os::GetSystemTick();
    for( idx = 0; idx < 20; idx++ )
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        g_ServerHit = false;
        NN_LOG( "Query: [GetAddrInfo][google,com]\n" );
        rval = CallGetAddrInfo( "google.com", nullptr, 0 );
        stopTick = nn::os::GetSystemTick();
        ERROR_IF_AND_COUNT( rval == false, "GAI failed but should have succeeded!" );

        auto diffsec = (stopTick.ToTimeSpan() - startTick.ToTimeSpan()).GetSeconds();
        NN_LOG("seconds: %ld\n", diffsec);
        if ( g_ServerHit == true )
        {
            ERROR_IF_AND_COUNT( diffsec < 14, "GAI: Cache TTL set to 15 seconds, but DNS server was HIT in: %d seconds\n", diffsec );
            NN_LOG( "GAI: DNS Server was hit - DNS Cache entry timed out after: %d seconds\n", diffsec );
            break;
        }
    }

    ERROR_IF_AND_COUNT( g_ServerHit == false, "GAI: DNS Cache record NEVER timed out after 20 seconds\n" );
    ERROR_IF_AND_COUNT( CacheRemoveHostname( "google.com" ) == -1, "Failed calling CacheRemoveHostname()\n" );


    //*********************
    //* GHBA
    //*********************

    // UDP: ERoot
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheERootComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheERootComSimple );

    g_ServerHit = false;
    NN_LOG( "Query: [GetHostByAddr][e.root-servers.net: 192.203.230.10]\n" );
    rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GHBA failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBA (did not) query the DNS server, but should have!" );
    ERROR_IF_AND_COUNT( IsIPInCache( "192.203.230.10" ) == false,
                        "GHBA: received a GOOD response, but response was NOT saved into the Cache\n" );
    ERROR_IF_AND_COUNT( CacheUpdateTimeToLiveForDomainname( "e.root-servers.net", 15 ) == -1, "Failed to updated TTL for DNS Cache Entry\n" );

    startTick = nn::os::GetSystemTick();
    for( idx = 0; idx < 20; idx++ )
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        g_ServerHit = false;
        NN_LOG( "Query: [GetHostByAddr][e.root-servers.net: 192.203.230.10]\n" );
        rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, nullptr, 0 );
        stopTick = nn::os::GetSystemTick();
        ERROR_IF_AND_COUNT( rval == false, "GHBA: failed but should have succeeded!" );

        auto diffsec = (stopTick.ToTimeSpan() - startTick.ToTimeSpan()).GetSeconds();
        NN_LOG("seconds: %ld\n", diffsec);
        if ( g_ServerHit == true )
        {
            ERROR_IF_AND_COUNT( diffsec < 14, "GHBA: Cache TTL set to 15 seconds, but DNS server was HIT in: %d seconds\n", diffsec );
            NN_LOG( "GHBA: DNS Server was hit - DNS Cache entry timed out after: %d seconds\n", diffsec );
            break;
        }
    }

    ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBA: DNS Cache record NEVER timed out after 20 seconds\n" );
    ERROR_IF_AND_COUNT( CacheRemoveIpAddress( "192.203.230.10" ) == -1, "Failed CacheRemoveIpAddress\n" );

    //*********************
    //* GNI
    //*********************

    g_ServerHit = false;
    NN_LOG( "Query: [GetNameInfo][ERoot]\n" );
    rval = CallGetNameInfo( "192.203.230.10", nullptr, 0 );
    ERROR_IF_AND_COUNT( rval == false, "GNI failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GNI: (did not) query the DNS server, but should have!" );
    ERROR_IF_AND_COUNT( IsIPInCache( "192.203.230.10" ) == false,
                        "GNI: received a GOOD response, but response was NOT saved into the Cache\n" );
    ERROR_IF_AND_COUNT( CacheUpdateTimeToLiveForDomainname( "e.root-servers.net", 15 ) == -1, "Failed to updated TTL for DNS Cache Entry\n" );

    startTick = nn::os::GetSystemTick();
    for( idx = 0; idx < 20; idx++ )
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        g_ServerHit = false;
        NN_LOG( "Query: [GetNameInfo][ERoot]\n" );
        rval = CallGetNameInfo( "192.203.230.10", nullptr, 0 );
        stopTick = nn::os::GetSystemTick();
        ERROR_IF_AND_COUNT( rval == false, "GNI: failed but should have succeeded!" );

        auto diffsec = (stopTick.ToTimeSpan() - startTick.ToTimeSpan()).GetSeconds();
        NN_LOG("seconds: %ld\n", diffsec);
        if ( g_ServerHit == true )
        {
            ERROR_IF_AND_COUNT( diffsec < 14, "GNI: Cache TTL set to 15 seconds, but DNS server was HIT in: %d seconds\n", diffsec );
            NN_LOG( "GNI: DNS Server was hit - DNS Cache entry timed out after: %d seconds\n", diffsec );
            break;
        }
    }

    ERROR_IF_AND_COUNT( g_ServerHit == false, "GNI: DNS Cache record NEVER timed out after 20 seconds\n" );
    ERROR_IF_AND_COUNT( CacheRemoveIpAddress( "192.203.230.10" ) == -1, "Failed CacheRemoveIpAddress\n" );

out:

    return;

}    // NOLINT(impl/function_size)


static int
DnsCache_Cache_Size_GHBN( void *                  inArg,
                          unsigned char *         dnsRqst,
                          size_t &                dnsRqstLen,
                          unsigned char *         dnsResp,
                          size_t &                dnsRespLen,
                          bool *                  isUDP,
                          bool *                  shouldContinue )
{
    struct DNS_HEADER_REC * pDnsHeader;
    uint16_t                saveId = 0x00;
    int                     idx = 0;
    LoopbackDnsServer *     thisPtr = (LoopbackDnsServer *) inArg;
    unsigned char           newDnsResp[1024];
    const unsigned char     addlRecord[] = {  0xc0, 0x0c, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
                                              0x00, 0xcd, 0x00, 0x04, 0xd8, 0x3a, 0xc1, 0x4e };
    int                     rc;
    bool                    isReq = false;

    // Save DNS Header in Request
    pDnsHeader = (struct DNS_HEADER_REC *) dnsRqst;
    saveId = nn::socket::InetNtohs( pDnsHeader->Id );

    // Copy Starting part of record
    memset( newDnsResp, 0, sizeof( newDnsResp ) );
    memcpy( newDnsResp, thisPtr->UDPResponse, thisPtr->UDPResponseLen );
    g_NewDnsRespLen = thisPtr->UDPResponseLen;

    // Copy Records up to DNS Record Count
    NN_LOG( "Dns Record Count: %d\n", g_DnsRecordCount );
    for( idx = 0; idx < g_DnsRecordCount; idx++ )
    {
        memcpy( &newDnsResp[ g_NewDnsRespLen ], (char *) &addlRecord, sizeof( addlRecord ) );
        g_NewDnsRespLen = g_NewDnsRespLen + sizeof( addlRecord );
    }
    g_DnsRecordCount++;   // There is already 1x answer in the original buffer

    // Fix DNS Header in Response
    pDnsHeader = (struct DNS_HEADER_REC *) newDnsResp;
    pDnsHeader->Id      = nn::socket::InetHtons( saveId );
    pDnsHeader->ancount = nn::socket::InetHtons( g_DnsRecordCount );

    //
    thisPtr->PrintHex( newDnsResp, g_NewDnsRespLen, &isReq, isUDP );

    // Server HIT!
    g_ServerHit = true;

    // Send UDP Response
    NN_LOG( "DNS Response Size: %d\n", g_NewDnsRespLen );
    rc = nn::socket::SendTo( thisPtr->serverUDP, newDnsResp, g_NewDnsRespLen, nn::socket::MsgFlag::Msg_None,
                             (nn::socket::SockAddr *) &thisPtr->recvAddr, thisPtr->recvAddrSize );

    NN_LOG( "UDP: SendTo rc = %lld\n", rc );

    if ( rc < 0 )
    {
        NN_LOG( "DNS UDP: SendTo(): Failed - errno: %d\n", nn::socket::GetLastError() );
    }

    // Stop
    *shouldContinue = false;

    return( 0 );
}


TEST(ApiUnit,DnsCache_Cache_Size_GHBN)
{
    const char *    domainname = "google.com";
    bool            rval = false, isSuccess = false;

    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB(  DnsCache_Cache_Size_GHBN  );

    // Set alias.landsberger.com Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheGoogleComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheGoogleComSimple );

    // Forever
    g_DnsRecordCount = 0;

    for(;;)
    {
        // Flush the Cache
        NN_LOG( "Calling Cache Flush\n" );
        CacheFlush();

        g_ServerHit = false;
        NN_LOG( "Query: [GetHostByName][%s]\n", domainname );
        rval = CallGetHostByName( domainname, nullptr, 0 );
        ERROR_IF_AND_COUNT( rval == false, "GHBN failed but should have succeeded!\n" );
        ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBN (did not) query the DNS server, but should have!\n" );

        if ( g_NewDnsRespLen > 500 )
        {
            ERROR_IF_AND_COUNT( IsDomainNameInCache( domainname ) == true,
                                "Domain name: %s is in the DNS Cache, but size: %d is larger than 500 bytes!\n",
                                domainname, g_NewDnsRespLen );
        }

        if ( g_NewDnsRespLen + 30 >= 1024 )
        {
            NN_LOG( "===> MAXIMUM BUFFER SIZED reached at: %ld\n", g_NewDnsRespLen );
            break;
        }

        g_DnsRecordCount = g_DnsRecordCount + 2;
    }

out:

    return;
}

static int
DnsCache_Cache_LabelLength( void *                  inArg,
                            unsigned char *         dnsRqst,
                            size_t &                dnsRqstLen,
                            unsigned char *         dnsResp,
                            size_t &                dnsRespLen,
                            bool *                  isUDP,
                            bool *                  shouldContinue )
{
    struct DNS_HEADER_REC * pDnsHeader;
    uint16_t                saveId = 0x00;
    LoopbackDnsServer *     thisPtr = (LoopbackDnsServer *) inArg;
    unsigned char           newDnsResp[1024];
    int                     rc;
    int                     labelSize = 0;
    int                     offset = 0;
    bool                    isReq = false;
    const unsigned char     questRecord[] = { 0x00, 0x01, 0x00, 0x01 };
    const unsigned char     addlRecord[] = {  0xc0, 0x0c, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
                                              0x00, 0xcd, 0x00, 0x04, 0xd8, 0x3a, 0xc1, 0x4e };

    // Save DNS Request Transaction Id
    pDnsHeader = (struct DNS_HEADER_REC *) dnsRqst;
    saveId = nn::socket::InetNtohs( pDnsHeader->Id );

    // Copy DNS Header (Only)
    memcpy( newDnsResp, thisPtr->UDPResponse, 12);
    g_NewDnsRespLen = thisPtr->UDPResponseLen;

    // Copy DNS Question from (DNS Request) to (DNS Response)
    offset = 12;
    labelSize = GetLabelLength( &dnsRqst[offset] );
    memcpy( &newDnsResp[offset], &dnsRqst[offset], labelSize );
    offset = offset + labelSize;

    // Null Terminate Question Label
    newDnsResp[offset] = 0x00;
    offset++;

    // Add Question: TYPE and CLASS
    memcpy( &newDnsResp[offset], questRecord, sizeof( questRecord ) );
    offset = offset + sizeof( questRecord );

    // Add Additional Record
    memcpy( &newDnsResp[offset], addlRecord, sizeof( addlRecord ) );
    offset = offset + sizeof( addlRecord );

    // Save DNSLength
    g_NewDnsRespLen = offset;

    // Fix DNS Response
    pDnsHeader = (struct DNS_HEADER_REC *) newDnsResp;
    pDnsHeader->Id    = nn::socket::InetHtons( saveId );

    // Display Response
    isReq = false;
    thisPtr->PrintHex( newDnsResp, g_NewDnsRespLen, &isReq, isUDP );

    // Send Response back to caller
    rc = nn::socket::SendTo( thisPtr->serverUDP, newDnsResp, g_NewDnsRespLen, nn::socket::MsgFlag::Msg_None,
                             (nn::socket::SockAddr *) &thisPtr->recvAddr, thisPtr->recvAddrSize );

    NN_LOG( "UDP: SendTo rc = %lld\n", rc );

    if ( rc < 0 )
    {
        NN_LOG( "DNS UDP: SendTo(): Failed - errno: %d\n", nn::socket::GetLastError() );
    }

    // Stop
    *shouldContinue = false;

    // Server was HIT
    g_ServerHit = true;

    return( 0 );
}

TEST(ApiUnit,DnsCache_Cache_LabelLength)
{
    char               domainname[270];
    const char *       oneLabel = "012345678901234567890123456789012345678901234567890123456789012";
    int                labelOffset  = 0;
    int                domainOffset = 0;
    int                domainSize = 0;
    int                idx = 0;
    bool               rval = false, isSuccess = false;

    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB(  DnsCache_Cache_LabelLength  );

    // Set alias.landsberger.com Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheGoogleComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheGoogleComSimple );

    domainSize = 0;
    while( domainSize < 246 )
    {
        memset( domainname, 0, sizeof( domainname ) );
        domainSize++;

        labelOffset  = -1;
        domainOffset = 0;

        NN_LOG( "Building Domain Name of Length: %d\n", domainSize );
        for( idx = 0; idx < domainSize; idx++ )
        {
            labelOffset++;
            if ( labelOffset > 62 )
            {
                labelOffset = 0;
                domainname[domainOffset] = '.';
                domainOffset++;
            }

            domainname[domainOffset] = oneLabel[labelOffset];
            domainOffset++;
        }

        memcpy( &domainname[domainOffset], ".com", 4 );
        domainOffset = domainOffset + 4;

        //*********************
        //* GHBN
        //*********************

        g_ServerHit = false;
        NN_LOG( "Query: [GetHostByName][Len: %d, Val: %s]\n", strlen( domainname), domainname );
        rval = CallGetHostByName( domainname, nullptr, 0 );
        ERROR_IF_AND_COUNT( rval == false, "GHBN failed but should have succeeded!\n" );
        ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBN (did not) query the DNS server, but should have!\n" );

        g_ServerHit = false;
        NN_LOG( "Query: [GetHostByName][%s]\n", domainname );
        rval = CallGetHostByName( domainname, nullptr, 0 );
        ERROR_IF_AND_COUNT( rval == false, "GHBN failed but should have succeeded!\n" );
        ERROR_IF_AND_COUNT( g_ServerHit == true, "GHBN queried the DNS server, but (SHOULD NOT) have!\n" );
        ERROR_IF_AND_COUNT( IsDomainNameInCache( domainname ) == false,
                            "GHBN: Should be in the cache, but it is not!\n", domainname );
        ERROR_IF_AND_COUNT( CacheRemoveHostname( domainname ) == -1, "Failed calling CacheRemoveHostname()\n" );

        //*********************
        //* GETADDRINFO
        //*********************

        g_ServerHit = false;
        NN_LOG( "Query: [GetAddrInfo][Len: %d, Val: %s]\n", strlen( domainname), domainname );
        rval = CallGetAddrInfo( domainname, nullptr, 0 );
        ERROR_IF_AND_COUNT( rval == false, "GAI failed but should have succeeded!" );

        g_ServerHit = false;
        NN_LOG( "Query: [GetAddrInfo][%s]\n", domainname );
        rval = CallGetAddrInfo( domainname, nullptr, 0 );
        ERROR_IF_AND_COUNT( rval == false, "GAI DNS lookup failed - but should have succeeded!" );
        ERROR_IF_AND_COUNT( g_ServerHit == true, "GAI queried (a DNS Server), but (should not) have!" );
        ERROR_IF_AND_COUNT( IsDomainNameInCache( domainname ) == false,
                            "GAI: Domain Name: %s should be in the cache, but it is not!\n", domainname );
    }

out:

    return;
}


const uint8_t NsrNoError = 0;   /* No error occurred. */
const uint8_t NsrFormerr = 1;   /* Format error. */
const uint8_t NsrServfail = 2;  /* Server failure. */
const uint8_t NsrNxdomain = 3;  /* Name error. */
const uint8_t NsrNotimpl = 4;   /* Unimplemented. */
const uint8_t NsrRefused = 5;   /* Operation refused. */
/* these are for BIND_UPDATE */
const uint8_t NsrYxdomain = 6;  /* Name exists */
const uint8_t NsrYxrrset = 7;   /* RRset exists */
const uint8_t NsrNxrrset = 8;   /* RRset does not exist */
const uint8_t NsrNotauth = 9;   /* Not authoritative for zone */
const uint8_t NsrNotzone = 10;  /* Zone of record different from zone section */

#ifdef COMMENTED_OUT
/* The following are EDNS extended rcodes */
const uint8_t NsrBadvers = 16;
/* The following are TSIG errors */
const uint8_t NsrBadsig = 16;
const uint8_t NsrBadkey = 17;
const uint8_t NsrBadtime = 18;
#endif


static int
DnsCache_Bad_RCODE( void *                  inArg,
                    unsigned char *         dnsRqst,
                    size_t &                dnsRqstLen,
                    unsigned char *         dnsResp,
                    size_t &                dnsRespLen,
                    bool *                  isUDP,
                    bool *                  shouldContinue )
{
    struct DNS_HEADER_REC *pDnsHeader;
    uint16_t              saveId = 0x00;
    LoopbackDnsServer     *thisPtr = (LoopbackDnsServer *) inArg;
    uint8_t               rcode = 0;

    // Save DNS Request Transaction Id
    pDnsHeader = (struct DNS_HEADER_REC *) dnsRqst;
    saveId = nn::socket::InetNtohs( pDnsHeader->Id );

    // Fill in the right Response
    memset( dnsResp, 0, dnsRespLen );

    memcpy( dnsResp, thisPtr->UDPResponse, thisPtr->UDPResponseLen );
    dnsRespLen = thisPtr->UDPResponseLen;

    // Fix DNS Response
    pDnsHeader = (struct DNS_HEADER_REC *) dnsResp;
    pDnsHeader->Id    = nn::socket::InetHtons( saveId );

    switch( g_ReturnCode )
    {
    case 0:   rcode = NsrNoError;   /* No error occurred. */
              break;
    case 1:   rcode = NsrFormerr;   /* Format error. */
              break;
    case 2:   rcode = NsrServfail;  /* Server failure. */
              break;
    case 3:   rcode = NsrNxdomain;  /* Name error. */
              break;
    case 4:   rcode = NsrNotimpl;   /* Unimplemented. */
              break;
    case 5:   rcode = NsrRefused;   /* Operation refused. */
              break;
    case 6:   rcode = NsrYxdomain;  /* Name exists */
              break;
    case 7:   rcode = NsrYxrrset;   /* RRset exists */
              break;
    case 8:   rcode = NsrNxrrset;   /* RRset does not exist */
              break;
    case 9:   rcode = NsrNotauth;   /* Not authoritative for zone */
              break;
    case 10:  rcode = NsrNotzone;   /* Zone of record different from zone section */
              break;
    default:
              break;
    }

    NN_LOG( "Saving: RCODE: %d\n", rcode );
    pDnsHeader->rcode = rcode & 0x0F;


    // Server HIT!
    g_ServerHit = true;

    // Process UDP normally
    return( 0 );
}

TEST(ApiUnit,DnsCache_Bad_RCODE)
{
    const char *    domainname = "google.com";
    bool            rval = false, isSuccess = false;

    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB(  DnsCache_Bad_RCODE  );

    // Set alias.landsberger.com Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheGoogleComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheGoogleComSimple );


    g_ReturnCode = 0;
    for(;;)
    {
        // Next RCODE (OK to skip 0 == good result)
        g_ReturnCode++;
        if ( g_ReturnCode > 10 )
        {
            NN_LOG( "Finished with RCODE testing\n" );
            break;
        }

        // Flush the Cache
        NN_LOG( "Calling Cache Flush\n" );
        CacheFlush();

        NN_LOG( "Try to Add to Cache\n" );
        g_ServerHit = false;
        NN_LOG( "Query: [GetHostByName][%s]\n", domainname );
        rval = CallGetHostByName( domainname, nullptr, 0 );
        ERROR_IF_AND_COUNT( rval == true, "GHBN succeeded but should have failed!\n" );
        ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBN (did not) query the DNS server, but should have!\n" );
        ERROR_IF_AND_COUNT( IsDomainNameInCache( domainname ) == true,
                            "Domainame: %s is in the cache, but domainname has a bad RCODE!\n", domainname );
    }

out:

    return;
}


const uint8_t DnsOpcodeQuery = 0;        /* Query Response */
const uint8_t DnsOpcodeIquery = 1;       /* Inverse Query Response */
const uint8_t DnsOpcodeStatus = 2;       /* Status */
const uint8_t DnsOpcodeUnassigned1 = 3;  /* Unassigned */
const uint8_t DnsOpcodeNotify = 4;       /* Notify */
const uint8_t DnsOpcodeUpdate = 5;       /* Update */
const uint8_t DnsOpcodeUnassigned2 = 6;  /* Unassigned */

static int
DnsCache_Bad_Opcode( void *                  inArg,
                     unsigned char *         dnsRqst,
                     size_t &                dnsRqstLen,
                     unsigned char *         dnsResp,
                     size_t &                dnsRespLen,
                     bool *                  isUDP,
                     bool *                  shouldContinue )
{
    struct DNS_HEADER_REC *pDnsHeader;
    uint16_t              saveId = 0x00;
    LoopbackDnsServer     *thisPtr = (LoopbackDnsServer *) inArg;
    uint8_t               opcode = 0;

    // Save DNS Request Transaction Id
    pDnsHeader = (struct DNS_HEADER_REC *) dnsRqst;
    saveId = nn::socket::InetNtohs( pDnsHeader->Id );

    // Fill in the right Response
    memset( dnsResp, 0, dnsRespLen );

    memcpy( dnsResp, thisPtr->UDPResponse, thisPtr->UDPResponseLen );
    dnsRespLen = thisPtr->UDPResponseLen;

    // Fix DNS Response
    pDnsHeader = (struct DNS_HEADER_REC *) dnsResp;
    pDnsHeader->Id    = nn::socket::InetHtons( saveId );

    switch( g_OpCode )
    {
    case 0:   opcode = DnsOpcodeQuery;
              break;
    case 1:   opcode = DnsOpcodeIquery;
              break;
    case 2:   opcode = DnsOpcodeStatus;
              break;
    case 3:   opcode = DnsOpcodeUnassigned1;
              break;
    case 4:   opcode = DnsOpcodeNotify;
              break;
    case 5:   opcode = DnsOpcodeUpdate;
              break;
    case 6:   opcode = DnsOpcodeUnassigned2;
              break;
    default:
              break;
    }

    NN_LOG( "Saving: OPCODE: %d\n", opcode );
    pDnsHeader->opcode = opcode & 0x0F;

    // Server HIT!
    g_ServerHit = true;

    // Process UDP normally
    return( 0 );
}

TEST(ApiUnit,DnsCache_Bad_Opcode)
{
    const char *    domainname = "google.com";
    bool            rval = false, isSuccess = false;

    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB(  DnsCache_Bad_Opcode  );

    // Set alias.landsberger.com Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheGoogleComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheGoogleComSimple );


    g_OpCode = 0;
    for(;;)
    {
        // Next OPCODE (OK to skip 0 == normal query)
        g_OpCode++;
        if ( g_OpCode > 6 )
        {
            NN_LOG( "Finished with OpCode testing\n" );
            break;
        }

        // Flush the Cache
        NN_LOG( "Calling Cache Flush\n" );
        CacheFlush();

        NN_LOG( "Try to Add to Cache\n" );
        g_ServerHit = false;
        NN_LOG( "Query: [GetHostByName][%s]\n", domainname );
        rval = CallGetHostByName( domainname, nullptr, 0 );
        // ERROR_IF_AND_COUNT( rval == true, "GHBN succeeded but should have failed!\n" );
        ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBN (did not) query the DNS server, but should have!\n" );
        ERROR_IF_AND_COUNT( IsDomainNameInCache( domainname ) == true,
                            "Domainame: %s is in the cache, but domainname has a bad OPCODE!\n", domainname );
    }

out:

    return;
}


TEST(ApiUnit,DnsCache_ResolverOption_CacheDisabled)
{
    bool                         isSuccess = true, rval;
    nn::socket::ResolverOption   myOption;


    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB(  DnsCache_Simple_Caching  );

    // UDP: google.com Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheGoogleComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheGoogleComSimple );

    // Flush Cache
    CacheFlush();

    //*********************
    //* GHBN
    //*********************

    myOption = DisableDnsCache();

    g_ServerHit = false;
    NN_LOG( "Query: [GetHostByName][google.com]\n" );
    rval = CallGetHostByName( "google.com", &myOption, 1 );
    ERROR_IF_AND_COUNT( rval == false, "GHBN failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBN (did not) query the DNS server, but should have!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true,
                        "Domainname is in cache, but it should not be!" );

    //*********************
    //* GETADDRINFO
    //*********************

    myOption = DisableDnsCache();

    g_ServerHit = false;
    NN_LOG( "Query: [GetAddrInfo][google,com]\n" );
    rval = CallGetAddrInfo( "google.com", &myOption, 1 );
    ERROR_IF_AND_COUNT( rval == false, "GAI failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GAI (did not) query the DNS server, but should have!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true,
                        "Domainname is in cache, but it should not be!" );


    //*********************
    //* GHBA
    //*********************

    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheERootComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheERootComSimple );

    myOption = DisableDnsCache();

    g_ServerHit = false;
    NN_LOG( "Query: [GetHostByAddr][ERoot: 192.203.230.10]\n" );
    rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, &myOption, 1 );
    ERROR_IF_AND_COUNT( rval == false, "GHBA failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBA (did not) query the DNS server, but should have!" );
    ERROR_IF_AND_COUNT( IsIPInCache( "192.203.230.10" ) == true,
                        "IP Address is in cache, but it should not be\n" );

    //*********************
    //* GNI
    //*********************

    myOption = DisableDnsCache();

    g_ServerHit = false;
    NN_LOG( "Query: [GetNameInfo][ERoot: 192.203.230.10]\n" );
    rval = CallGetNameInfo( "192.203.230.10", &myOption, 1 );
    ERROR_IF_AND_COUNT( rval == false, "GNI failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GNI (did not) query the DNS server, but should have!" );
    ERROR_IF_AND_COUNT( IsIPInCache( "192.203.230.10" ) == true,
                        "IP Address is in cache, but it should not be\n" );

out:

    return;
}

TEST(ApiUnit,DnsCache_ResolverOption_CacheEnabled)
{
    bool                         isSuccess = true, rval;
    nn::socket::ResolverOption   myOption;

    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB(  DnsCache_Simple_Caching  );

    // UDP: google.com Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheGoogleComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheGoogleComSimple );

    // Flush Cache
    CacheFlush();

    //*********************
    //* GHBN
    //*********************

    myOption = EnableDnsCache();

    g_ServerHit = false;
    NN_LOG( "Query: [GetHostByName][google.com]\n" );
    rval = CallGetHostByName( "google.com", &myOption, 1 );
    ERROR_IF_AND_COUNT( rval == false, "GHBN failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBN (did not) query the DNS server, but should have!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == false,
                        "Domainname is NOT in cache, but it should  be!" );
    ERROR_IF_AND_COUNT( CacheRemoveHostname( "google.com" ) == -1, "Failed calling CacheRemoveHostname()\n" );


    //*********************
    //* GETADDRINFO
    //*********************

    myOption = EnableDnsCache();

    g_ServerHit = false;
    NN_LOG( "Query: [GetAddrInfo][google,com]\n" );
    rval = CallGetAddrInfo( "google.com", &myOption, 1 );
    ERROR_IF_AND_COUNT( rval == false, "GAI failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GAI (did not) query the DNS server, but should have!" );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == false,
                        "Domainname is NOT in cache, but it should  be!" );
    ERROR_IF_AND_COUNT( CacheRemoveHostname( "google.com" ) == -1, "Failed calling CacheRemoveHostname()\n" );

    //*********************
    //* GHBA
    //*********************

    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheERootComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheERootComSimple );

    myOption = EnableDnsCache();

    g_ServerHit = false;
    NN_LOG( "Query: [GetHostByAddr][ERoot: 192.203.230.10]\n" );
    rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, &myOption, 1 );
    ERROR_IF_AND_COUNT( rval == false, "GHBA failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GHBA (did not) query the DNS server, but should have!" );
    ERROR_IF_AND_COUNT( IsIPInCache( "192.203.230.10" ) == false, "IP Address is NOT in cache, but it should be\n" );
    ERROR_IF_AND_COUNT( CacheRemoveIpAddress( "192.203.230.10" ) == -1, "Failed CacheRemoveIpAddress\n" );

    //*********************
    //* GNI
    //*********************

    myOption = EnableDnsCache();

    g_ServerHit = false;
    NN_LOG( "Query: [GetNameInfo][ERoot: 192.203.230.10]\n" );
    rval = CallGetNameInfo( "192.203.230.10", &myOption, 1 );
    ERROR_IF_AND_COUNT( rval == false, "GNI failed but should have succeeded!" );
    ERROR_IF_AND_COUNT( g_ServerHit == false, "GNI (did not) query the DNS server, but should have!" );
    ERROR_IF_AND_COUNT( IsIPInCache( "192.203.230.10" ) == false, "IP Address is NOT in cache, but it should be\n" );

out:

    return;
}


//****************************************************************
//*
//*   T E S T S   W I T H   T H R E A D s
//*
//****************************************************************

int
DnsCache_Cancel_Handle_TCP( void *                  inArg,
                            unsigned char *         dnsRqst,
                            size_t &                dnsRqstLen,
                            unsigned char *         dnsResp,
                            size_t &                dnsRespLen,
                            bool *                  isUDP,
                            bool *                  shouldContinue )
{
    int                   rc;
    uint16_t              dnsLen;
    LoopbackDnsServer *   thisObj = (LoopbackDnsServer *) inArg;

    // Process UDP normally
    if ( *isUDP == true )
    {
        return( thisObj->MakeDnsResponseCB( thisObj, dnsRqst, dnsRqstLen, dnsResp, dnsRespLen, isUDP, shouldContinue )  );
    }

    // Else.. TCP

    // Don't continue.. This routine is handling the response
    *shouldContinue = false;

    // Write (Only) Length Bytes
    NN_LOG( "[TCP_DNS]: Writing bad TCP DNS Response Length Bytes (Value: %d)\n", 0xFFFF );
    dnsLen = nn::socket::InetHtons( (uint16_t) 0xFFFF );
    rc = nn::socket::Write( thisObj->acceptedTCP, (char *) &dnsLen, sizeof(uint16_t) );
    if ( rc != sizeof( uint16_t) )
    {
        NN_LOG( "[TCP_DNS]: Tried to Write 2x length bytes failed!" );
        g_workVarBool = true;
        return( -1 );
    }

    // Wakeup Cancel Thread
    g_workVarBool = true;

    // Nothing more TODO - Client expected to close connection

    return( 0 );
}

void
DnsCache_Cancel_Handle_Thread( void *    inArg )
{
    NN_LOG( "Cancel Thread: Startup..\n" );

    while( g_workVarBool == false )
    {
        NN_LOG( "Cancel Thread: Waiting for TCP response to go back to caller\n" );
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    // Cancel thread wake up!
    NN_LOG( "Cancel Thread: Cancel thread has woken up!  Waiting [1] more seconds\n" );
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // Issue Cancel
    NN_LOG( "Cancel Thread: Issuing Cancel using Cancel Handle: %ld\n", g_WorkVarint_t );
    nn::socket::ResolverOption myOption = SetCancelHandleInteger( g_WorkVarint_t );
    nn::socket::ResolverSetOption(myOption);

    // Exit Thead
    NN_LOG( "Cancel Thread: exiting..\n" );
}

TEST(ApiUnit,DnsCache_Cancel_Handle_Test)
{
    int                        cancelThread;
    bool                       isSuccess = true, rval;
    nn::socket::ResolverOption option;
    nn::socket::ResolverOption cancelOption;

    // Normal DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB( DnsCache_Cancel_Handle_TCP );

    // UDP: google.com + truncate bit
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheGoogleComSimpleWithTC;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheGoogleComSimpleWithTC );

    // TCP: google.com - but bad response len
    g_LoopbackDnsServer->TCPResponse = (unsigned char *) DnsCacheGoogleComSimple;
    g_LoopbackDnsServer->TCPResponseLen = sizeof( DnsCacheGoogleComSimple );

    // Flush Cache
    CacheFlush();

    //*********************
    //* GHBN
    //*********************

    // Get a Socket API - Cancel Handle
    nn::socket::ResolverGetOption(&option, nn::socket::ResolverOptionKey::GetCancelHandleInteger);
    g_WorkVarint_t = option.data.integerValue;
    NN_LOG( "I have been assigned Cancel Handle: %ld\n", g_WorkVarint_t );

    // Signal for Cancel Thread..
    g_workVarBool = false;

    // Create Cancel Thread
    cancelThread = g_pThreadedTest->CreateThread( &DnsCache_Cancel_Handle_Thread, (void *) NULL, 10 );
    ERROR_IF( cancelThread < 0, "Failed calling: CreateThread for Cancel Api\n" );

    // Give Cancel Thread time to spin up..
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // Get Cancel Handle
    cancelOption = GetCancelHandleInteger(  g_WorkVarint_t );

    // Call GetHostByName - using Cancel Handle
    NN_LOG( "Query: [GetHostByName][google,com]\n" );
    rval = nn::socket::GetHostEntByName( "google.com", &cancelOption, 1 );

    // Cleanup Testing Thread
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    g_pThreadedTest->WaitThread( cancelThread );
    g_pThreadedTest->DestroyThread( cancelThread );

    // Process GHBN Result
    ERROR_IF_AND_COUNT( rval == true, "GHBN: PASSED but failure was expected!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::ECanceled,
                        "GHBN should have Failed with errno: [ECANCELED], but actually failed with errno: %d\n",
                        nn::socket::GetLastError() );
    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true, "Domainname is in cache, but it should NOT be!" );

    NN_LOG( "GetHostByName: Test PASSED!\n" );

    //*********************
    //* GETADDRINFO
    //*********************

    // Get a Socket API - Cancel Handle
    nn::socket::ResolverGetOption(&option, nn::socket::ResolverOptionKey::GetCancelHandleInteger);
    g_WorkVarint_t = option.data.integerValue;
    NN_LOG( "I have been assigned Cancel Handle: %ld\n", g_WorkVarint_t );

    // Signal for Cancel Thread..
    g_workVarBool = false;

    // Create Cancel Thread
    cancelThread = g_pThreadedTest->CreateThread( &DnsCache_Cancel_Handle_Thread, (void *) NULL, 10 );
    ERROR_IF( cancelThread < 0, "Failed calling: CreateThread for Cancel Api\n" );

    // Give Cancel Thread time to spin up..
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // Get Cancel Handle
    cancelOption = GetCancelHandleInteger(  g_WorkVarint_t );

    // Call GetAddrInfo - using Cancel Handle
    NN_LOG( "Query: [GetAddrInfo][google,com]\n" );
    rval = CallGetAddrInfo( "google.com", &cancelOption, 1 );

    // Cleanup Testing Thread
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    g_pThreadedTest->WaitThread( cancelThread );
    g_pThreadedTest->DestroyThread( cancelThread );

    // Process GAI Result
    ERROR_IF_AND_COUNT( rval == true, "GAI: PASSED but failure was expected!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::ECanceled,
                        "GAI should have Failed with errno: [ECANCELED], but actually failed with errno: %d\n",
                        nn::socket::GetLastError() );

    ERROR_IF_AND_COUNT( IsDomainNameInCache( "google.com" ) == true, "Domainname is in cache, but it should NOT be!" );

    NN_LOG( "GetAddrInfo: Test PASSED!\n" );

    //*********************
    //* GHBA
    //*********************

    // UDP: ERoot with TC bit
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheERootComSimpleWithTC;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheERootComSimpleWithTC );

    // TCP: ERoot - but bad response len
    g_LoopbackDnsServer->TCPResponse = (unsigned char *) DnsCacheERootComSimple;
    g_LoopbackDnsServer->TCPResponseLen = sizeof( DnsCacheERootComSimple );

    // Get a Socket API - Cancel Handle
    nn::socket::ResolverGetOption(&option, nn::socket::ResolverOptionKey::GetCancelHandleInteger);
    g_WorkVarint_t = option.data.integerValue;
    NN_LOG( "I have been assigned Cancel Handle: %ld\n", g_WorkVarint_t );

    // Signal for Cancel Thread..
    g_workVarBool = false;

    // Create Cancel Thread
    cancelThread = g_pThreadedTest->CreateThread( &DnsCache_Cancel_Handle_Thread, (void *) NULL, 10 );
    ERROR_IF( cancelThread < 0, "Failed calling: CreateThread for Cancel Api\n" );

    // Give Cancel Thread time to spin up..
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // Get Cancel Handle
    cancelOption = GetCancelHandleInteger(  g_WorkVarint_t );

    // Call GetHostByAddr - using Cancel Handle
    NN_LOG( "Query: [GetHostByAddr][ERoot: 192.203.230.10]\n" );
    rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, &cancelOption, 1 );

    // Cleanup Testing Thread
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    g_pThreadedTest->WaitThread( cancelThread );
    g_pThreadedTest->DestroyThread( cancelThread );

    // Process GHBA Result
    ERROR_IF_AND_COUNT( rval == true, "GHBA: PASSED but failure was expected!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::ECanceled,
                        "GHBA should have Failed with errno: [ECANCELED], but actually failed with errno: %d\n",
                        nn::socket::GetLastError() );

    ERROR_IF_AND_COUNT( IsIPInCache( "192.203.230.10" ) == true, "IP Address is cached, but it should not be!\n" );

    NN_LOG( "GetHostByAddr: Test PASSED!\n" );

    //*********************
    //* GNI
    //*********************

    // Get a Socket API - Cancel Handle
    nn::socket::ResolverGetOption(&option, nn::socket::ResolverOptionKey::GetCancelHandleInteger);
    g_WorkVarint_t = option.data.integerValue;
    NN_LOG( "I have been assigned Cancel Handle: %ld\n", g_WorkVarint_t );

    // Signal for Cancel Thread..
    g_workVarBool = false;

    // Create Cancel Thread
    cancelThread = g_pThreadedTest->CreateThread( &DnsCache_Cancel_Handle_Thread, (void *) NULL, 10 );
    ERROR_IF( cancelThread < 0, "Failed calling: CreateThread for Cancel Api\n" );

    // Give Cancel Thread time to spin up..
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // Get Cancel Handle
    cancelOption = GetCancelHandleInteger(  g_WorkVarint_t );

    // Call GetNameInfo - using Cancel Handle
    NN_LOG( "Query: [GetNameInfo][ERoot: 192.203.230.10]\n" );
    rval = CallGetNameInfo( "192.203.230.10", &cancelOption, 1 );

    // Cleanup Testing Thread
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    g_pThreadedTest->WaitThread( cancelThread );
    g_pThreadedTest->DestroyThread( cancelThread );

    // Process GNI Result
    ERROR_IF_AND_COUNT( rval == true, "GNI: PASSED but failure was expected!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::ECanceled,
                        "GNI: Should have Failed with errno: [ECANCELED], but actually failed with errno: %d\n",
                        nn::socket::GetLastError() );

    ERROR_IF_AND_COUNT( IsIPInCache( "192.203.230.10" ) == true, "IP Address is cached, but it should not be!\n" );

    NN_LOG( "GetNameInfo: Test PASSED!\n" );

out:

    return;
}


static int
DnsCache_Thread_Host_DnsCB( void *                  inArg,
                            unsigned char *         dnsRqst,
                            size_t &                dnsRqstLen,
                            unsigned char *         dnsResp,
                            size_t &                dnsRespLen,
                            bool *                  isUDP,
                            bool *                  shouldContinue )
{
    struct DNS_HEADER_REC * pDnsHeader;
    uint16_t                saveId = 0x00;
    LoopbackDnsServer *     thisPtr = (LoopbackDnsServer *) inArg;
    unsigned char           newDnsResp[1024];
    int                     rc;
    int                     labelSize = 0;
    int                     offset = 0;
    bool                    isReq = false;
    const unsigned char     questRecord[] = { 0x00, 0x01, 0x00, 0x01 };
    const unsigned char     addlRecord[] = {  0xc0, 0x0c, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
                                              0x00, 0xcd, 0x00, 0x04, 0xd8, 0x3a, 0xc1, 0x4e };

    // Save DNS Request Transaction Id
    pDnsHeader = (struct DNS_HEADER_REC *) dnsRqst;
    saveId = nn::socket::InetNtohs( pDnsHeader->Id );

    // Copy DNS Header (Only)
    memcpy( newDnsResp, thisPtr->UDPResponse, 12);
    g_NewDnsRespLen = thisPtr->UDPResponseLen;

    // Copy DNS Question from (DNS Request) to (DNS Response)
    offset = 12;
    labelSize = GetLabelLength( &dnsRqst[offset] );
    memcpy( &newDnsResp[offset], &dnsRqst[offset], labelSize );
    offset = offset + labelSize;

    // Null Terminate Question Label
    newDnsResp[offset] = 0x00;
    offset++;

    // Add Question: TYPE and CLASS
    memcpy( &newDnsResp[offset], questRecord, sizeof( questRecord ) );
    offset = offset + sizeof( questRecord );

    // Add Additional Record
    memcpy( &newDnsResp[offset], addlRecord, sizeof( addlRecord ) );
    offset = offset + sizeof( addlRecord );

    // Save DNSLength
    g_NewDnsRespLen = offset;

    // Fix DNS Response
    pDnsHeader = (struct DNS_HEADER_REC *) newDnsResp;
    pDnsHeader->Id       = nn::socket::InetHtons( saveId );

    /* pDnsHeader->ancount  = nn::socket::InetHtons( 1 );
    pDnsHeader->nscount  = nn::socket::InetHtons( 0 );
    pDnsHeader->arcount  = nn::socket::InetHtons( 0 ); */

    NN_LOG(
    "Resp Dns: Id: %d, OPCODE: %d, QR: %d, AA: %d, TC: %d, RD: %d, RA: %d, AD: %d, CD: %d, rcode: %d, totQ: %d, totA %d, totAut: %d, totAdd: %d\n",
    nn::socket::InetNtohs( pDnsHeader->Id ), pDnsHeader->opcode, pDnsHeader->QR, pDnsHeader->AA, pDnsHeader->TC,
    pDnsHeader->RD, pDnsHeader->RA,  pDnsHeader->AD, pDnsHeader->CD,  pDnsHeader->rcode,
    nn::socket::InetNtohs( pDnsHeader->qdcount ),  nn::socket::InetNtohs( pDnsHeader->ancount ),
    nn::socket::InetNtohs( pDnsHeader->nscount ),  nn::socket::InetNtohs( pDnsHeader->arcount ) );

    // Display Response
    isReq = false;
    thisPtr->PrintHex( newDnsResp, g_NewDnsRespLen, &isReq, isUDP );

    // Send Response back to caller
    rc = nn::socket::SendTo( thisPtr->serverUDP, newDnsResp, g_NewDnsRespLen, nn::socket::MsgFlag::Msg_None,
                             (nn::socket::SockAddr *) &thisPtr->recvAddr, thisPtr->recvAddrSize );

    NN_LOG( "UDP: SendTo rc = %lld\n", rc );

    if ( rc < 0 )
    {
        NN_LOG( "DNS UDP: SendTo(): Failed - errno: %d\n", nn::socket::GetLastError() );
    }

    // Stop
    *shouldContinue = false;

    // Server was HIT
    g_ServerHit = true;

    return( 0 );
}

void
DnsCache_Thread_Host_Forward( void * inArg )
{
    char             myChar;
    int              offsetDnsRqst;
    char             domainname[256];
    const char *     domainnametmpl = "123456789012345678901234567890123456789012345678901234567890.com";
    bool             rval = false, isSuccess = false;
    uint32_t         totalEntries = 0;
    int              domainCount;
    bool             *myResult = (bool *) inArg;

    // Initialize DNS request/response
    memset( domainname, 0, sizeof( domainname ) );

    myChar        = 'A';
    offsetDnsRqst = 0;
    domainCount   = 0;

    // Forever
    for(;;)
    {
        // If we are out of alphabet
        myChar++;
        if ( myChar < 'A' || myChar > 'Z' )
        {
            domainCount++;
            if ( domainCount >= 60 )
            {
                NN_LOG( "[Thread] 60 character length Label processed - Finished processing!  Total Entries: %d\n", totalEntries );
                break;
            }

            myChar = 'A';
            offsetDnsRqst++;
        }

        memcpy( domainname, domainnametmpl, strlen( domainnametmpl ) );
        domainname[ offsetDnsRqst ] = myChar;

        //*********************
        //* GHBN
        //*********************

        rval = CallGetHostByName( domainname, nullptr, 0);
        ERROR_IF_AND_COUNT( rval == false, "GHBN failed but should have succeeded!" );
        ERROR_IF_AND_COUNT( CacheRemoveHostname(  domainname ) == -1, "Failed calling CacheRemoveHostname()\n" );

        //*********************
        //* GETADDRINFO
        //*********************

        rval = CallGetAddrInfo( domainname, nullptr, 0 );
        ERROR_IF_AND_COUNT( rval == false, "GAI failed but should have succeeded!" );

        totalEntries++;
    }

    *myResult = true;
    NN_LOG( "Thread Passed!\n" );

    return;

out:

    *myResult = false;
    NN_LOG( "Thread Failed!\n" );
}

void
DnsCache_Thread_Host_Backward( void * inArg )
{
    char             myChar;
    int              offsetDnsRqst;
    char             domainname[256];
    const char *     domainnametmpl = "123456789012345678901234567890123456789012345678901234567890.com";
    bool             rval = false, isSuccess = false;
    uint32_t         totalEntries = 0;
    int              domainCount;
    bool             *myResult = (bool *) inArg;

    // Initialize DNS request/response
    memset( domainname, 0, sizeof( domainname ) );

    myChar        = 'A';
    offsetDnsRqst = 59;
    domainCount   = 59;

    // Forever
    for(;;)
    {
        // If we are out of alphabet
        myChar++;
        if ( myChar < 'A' || myChar > 'Z' )
        {
            domainCount--;
            if ( domainCount < 0 )
            {
                NN_LOG( "[Thread] 60 character length Label processed - Finished processing!  Total Entries: %d\n", totalEntries );
                break;
            }

            myChar = 'A';
            offsetDnsRqst--;
        }

        memcpy( domainname, domainnametmpl, strlen( domainnametmpl ) );
        domainname[ offsetDnsRqst ] = myChar;

        //*********************
        //* GHBN
        //*********************

        // NN_LOG( "Query: [GetHostByName][%s]\n", domainname );
        rval = CallGetHostByName( domainname, nullptr, 0);
        ERROR_IF_AND_COUNT( rval == false, "GHBN failed but should have succeeded!" );
        CacheRemoveHostname(  domainname );

        //*********************
        //* GETADDRINFO
        //*********************

        //NN_LOG( "Query: [GetAddrInfo][%s]\n", domainname );
        rval = CallGetAddrInfo( domainname, nullptr, 0 );
        ERROR_IF_AND_COUNT( rval == false, "GAI failed but should have succeeded!" );

        totalEntries++;
    }

    *myResult = true;
    NN_LOG( "Thread Passed!\n" );

    return;

out:

    *myResult = false;
    NN_LOG( "Thread Failed!\n" );
}

TEST(ApiUnit,DnsCache_Thread_Host)
{
    int                        testThread[5], idx = 0;;
    bool                       isSuccess = true, rval;

    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB(  DnsCache_Thread_Host_DnsCB  );

    // Set 123456789012345678901234567890123456789012345678901234567890.com
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCache123456ComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCache123456ComSimple );

    // Flush Cache
    CacheFlush();

    //*********************
    // Create Testing Treads
    rval = false;
    for( idx = 0; idx < 5; idx++ )
    {
        g_MyResults[idx] = false;
        if ( rval == false )
        {
            testThread[idx] = g_pThreadedTest->CreateThread( &DnsCache_Thread_Host_Forward, (void *) &g_MyResults[idx], 10 );
            rval = true;
        }
        else
        {
            testThread[idx] = g_pThreadedTest->CreateThread( &DnsCache_Thread_Host_Backward, (void *) &g_MyResults[idx], 10 );
            rval = false;
        }
        ERROR_IF( testThread[idx] < 0, "Failed calling: CreateThread for GetHostname\n" );
        NN_LOG( "Started Working Thread: %d\n", testThread[idx] );
    }

    NN_LOG( "Waiting for Worker Threads\n" );

    for( idx = 0; idx < 5; idx++ )
    {
        NN_LOG( "Waiting for Worker Thread: %d\n", testThread[idx] );
        g_pThreadedTest->WaitThread( testThread[idx] );

        NN_LOG( "Destroying Worker Thread: %d\n",  testThread[idx] );
        g_pThreadedTest->DestroyThread( testThread[idx] );
    }

    for( idx = 0; idx < 5; idx++ )
    {
        ERROR_IF_AND_COUNT( g_MyResults[idx] == false, "Failed!  Worker Thread: %d reports Failure!\n", idx );
        NN_LOG( "Passed!  Worker Thread: %d reports Success!\n", idx );
    }

    NN_LOG( "Test Complete\n" );

out:

    return;
}

static int
DnsCache_Thread_IP_DnsCB( void *                  inArg,
                          unsigned char *         dnsRqst,
                          size_t &                dnsRqstLen,
                          unsigned char *         dnsResp,
                          size_t &                dnsRespLen,
                          bool *                  isUDP,
                          bool *                  shouldContinue )
{
    struct DNS_HEADER_REC * pDnsHeader;
    uint16_t                saveId = 0x00;
    LoopbackDnsServer *     thisPtr = (LoopbackDnsServer *) inArg;
    unsigned char           newDnsResp[1024];
    int                     rc;
    int                     labelSize = 0;
    int                     offset = 0;
    bool                    isReq = false;
    const unsigned char     questRecord[] = { 0x00, 0x0c, 0x00, 0x01 };
    const unsigned char     addlRecord[] = {  0xc0, 0x0c, 0x00, 0x0c, 0x00, 0x01, 0x00, 0x00,
                                              0x26, 0xaa, 0x00, 0x14, 0x01, 0x65, 0x0c, 0x72,
                                              0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76,
                                              0x65, 0x72, 0x73, 0x03, 0x6e, 0x65, 0x74, 0x00  };

    // Save DNS Request Transaction Id
    pDnsHeader = (struct DNS_HEADER_REC *) dnsRqst;
    saveId = nn::socket::InetNtohs( pDnsHeader->Id );

    // Copy DNS Header (Only)
    memcpy( newDnsResp, thisPtr->UDPResponse, 12);
    g_NewDnsRespLen = thisPtr->UDPResponseLen;

    // Copy DNS Question from (DNS Request) to (DNS Response)
    offset = 12;
    labelSize = GetLabelLength( &dnsRqst[offset] );
    memcpy( &newDnsResp[offset], &dnsRqst[offset], labelSize );
    offset = offset + labelSize;

    // Null Terminate Question Label
    newDnsResp[offset] = 0x00;
    offset++;

    // Add Question: TYPE and CLASS
    memcpy( &newDnsResp[offset], questRecord, sizeof( questRecord ) );
    offset = offset + sizeof( questRecord );

    // Add Additional Record
    memcpy( &newDnsResp[offset], addlRecord, sizeof( addlRecord ) );
    offset = offset + sizeof( addlRecord );

    // Save DNSLength
    g_NewDnsRespLen = offset;

    // Fix DNS Response
    pDnsHeader = (struct DNS_HEADER_REC *) newDnsResp;
    pDnsHeader->Id    = nn::socket::InetHtons( saveId );
    pDnsHeader->ancount  = nn::socket::InetHtons( 1 );
    pDnsHeader->nscount  = nn::socket::InetHtons( 0 );
    pDnsHeader->arcount  = nn::socket::InetHtons( 0 );

    NN_LOG(
    "Resp Dns: Id: %d, OPCODE: %d, QR: %d, AA: %d, TC: %d, RD: %d, RA: %d, AD: %d, CD: %d, rcode: %d, totQ: %d, totA %d, totAut: %d, totAdd: %d\n",
    nn::socket::InetNtohs( pDnsHeader->Id ), pDnsHeader->opcode, pDnsHeader->QR, pDnsHeader->AA, pDnsHeader->TC,
    pDnsHeader->RD, pDnsHeader->RA,  pDnsHeader->AD, pDnsHeader->CD,  pDnsHeader->rcode,
    nn::socket::InetNtohs( pDnsHeader->qdcount ),  nn::socket::InetNtohs( pDnsHeader->ancount ),
    nn::socket::InetNtohs( pDnsHeader->nscount ),  nn::socket::InetNtohs( pDnsHeader->arcount ) );

    // Display Response
    isReq = false;
    thisPtr->PrintHex( newDnsResp, g_NewDnsRespLen, &isReq, isUDP );

    // Send Response back to caller
    rc = nn::socket::SendTo( thisPtr->serverUDP, newDnsResp, g_NewDnsRespLen, nn::socket::MsgFlag::Msg_None,
                             (nn::socket::SockAddr *) &thisPtr->recvAddr, thisPtr->recvAddrSize );

    NN_LOG( "UDP: SendTo rc = %lld\n", rc );

    if ( rc < 0 )
    {
        NN_LOG( "DNS UDP: SendTo(): Failed - errno: %d\n", nn::socket::GetLastError() );
    }

    // Stop
    *shouldContinue = false;

    return( 0 );
}


void
DnsCache_Thread_IP_Forward( void * inArg )
{
    char             domainname[256];
    bool             rval = false, isSuccess = false;
    uint32_t         totalEntries = 0;
    bool             *myResult = (bool *) inArg;
    int              octet1, octet2;

    // Initialize DNS request/response
    memset( domainname, 0, sizeof( domainname ) );

    octet1 = 0;
    octet2 = 0;

    // Forever
    for(;;)
    {
        octet1++;
        if ( octet1 > 32 )
        {
            octet1 = 0;
            octet2++;
        }
        if ( octet2 > 32 )
        {
           NN_LOG( "PASSED - Tested 1K of unique IPv4 Addresses\n" );
           break;
        }

        memset(  domainname, 0, sizeof( domainname ) );
        sprintf( domainname, "192.168.%d.%d", octet1, octet2 );

        //*********************
        //* GHBA
        //*********************

        rval = CallGetHostByAddr( domainname, 4, nn::socket::Family::Af_Inet, nullptr, 0 );
        ERROR_IF_AND_COUNT( rval == false, "GHBA: failed but should have succeeded!" );
        ERROR_IF_AND_COUNT( CacheRemoveIpAddress( domainname ) == -1, "Failed CacheRemoveIpAddress\n" );


        //*********************
        //* GNI
        //*********************

        rval = CallGetNameInfo( domainname, nullptr, 0 );
        ERROR_IF_AND_COUNT( rval == false, "GNI failed but should have succeeded!" );

        totalEntries++;
    }

    *myResult = true;
    NN_LOG( "Thread Passed!\n" );

    return;

out:

    *myResult = false;
    NN_LOG( "Thread Failed!\n" );
}

void
DnsCache_Thread_IP_Backward( void * inArg )
{
    char             domainname[256];
    bool             rval = false, isSuccess = false;
    uint32_t         totalEntries = 0;
    bool             *myResult = (bool *) inArg;
    int              octet1, octet2;

    // Initialize DNS request/response
    memset( domainname, 0, sizeof( domainname ) );

    octet1 = 32;
    octet2 = 32;

    // Forever
    for(;;)
    {
        octet1--;
        if ( octet1 > -1 )
        {
            octet1 = 32;
            octet2--;
        }
        if ( octet2 > -1 )
        {
           NN_LOG( "PASSED - Tested 1K of unique IPv4 Addresses\n" );
           break;
        }

        memset(  domainname, 0, sizeof( domainname ) );
        sprintf( domainname, "192.168.%d.%d", octet1, octet2 );

        //*********************
        //* GHBA
        //*********************

        rval = CallGetHostByAddr( domainname, 4, nn::socket::Family::Af_Inet, nullptr, 0 );
        ERROR_IF_AND_COUNT( rval == false, "GHBA: failed but should have succeeded!" );
        ERROR_IF_AND_COUNT( CacheRemoveIpAddress( domainname ) == -1, "Failed CacheRemoveIpAddress\n" );


        //*********************
        //* GNI
        //*********************

        rval = CallGetNameInfo( domainname, nullptr, 0 );
        ERROR_IF_AND_COUNT( rval == false, "GNI failed but should have succeeded!" );

        totalEntries++;
    }

    *myResult = true;
    NN_LOG( "Thread Passed!\n" );

    return;

out:

    *myResult = false;
    NN_LOG( "Thread Failed!\n" );
}

TEST(ApiUnit,DnsCache_Thread_IP)
{
    int                        testThread[5], idx = 0;;
    bool                       isSuccess = true, rval;

    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB(  DnsCache_Thread_IP_DnsCB  );

    // Set 123456789012345678901234567890123456789012345678901234567890.com
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCache123456ComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCache123456ComSimple );

    // Flush Cache
    CacheFlush();

    //*********************
    // Create Testing Treads
    rval = false;
    for( idx = 0; idx < 5; idx++ )
    {
        g_MyResults[idx] = false;
        if ( rval == false )
        {
            testThread[idx] = g_pThreadedTest->CreateThread( &DnsCache_Thread_IP_Forward, (void *) &g_MyResults[idx], 10 );
            rval = true;
        }
        else
        {
            testThread[idx] = g_pThreadedTest->CreateThread( &DnsCache_Thread_IP_Backward, (void *) &g_MyResults[idx], 10 );
            rval = false;
        }
        ERROR_IF( testThread[idx] < 0, "Failed calling: CreateThread for IP Address Loopkups\n" );
        NN_LOG( "Started Working Thread: %d\n", testThread[idx] );
    }

    NN_LOG( "Waiting for Worker Threads\n" );

    for( idx = 0; idx < 5; idx++ )
    {
        NN_LOG( "Waiting for Worker Thread: %d\n", testThread[idx] );
        g_pThreadedTest->WaitThread( testThread[idx] );

        NN_LOG( "Destroying Worker Thread: %d\n",  testThread[idx] );
        g_pThreadedTest->DestroyThread( testThread[idx] );
    }

    for( idx = 0; idx < 5; idx++ )
    {
        ERROR_IF_AND_COUNT( g_MyResults[idx] == false, "Failed!  Worker Thread: %d reports Failure!\n", idx );
        NN_LOG( "Passed!  Worker Thread: %d reports Success!\n", idx );
    }

    NN_LOG( "Test Complete\n" );

out:

    return;
}

//****************************************************************
//*
//*   T E S T   W I T H   T W O    D N S   S E R V E R s
//*
//****************************************************************

TEST(ApiUnit,DnsCache_Create_Two_DNS_Servers)
{
    int    threadId = 0;
    bool   isSuccess = true;
    int    rc;

    // Make sure Primary DNS is available
    ERROR_IF( g_LoopbackDnsServer == NULL, "Primary DNS Server is required!\n" );


    //////////////////////////////
    ////  Allocate DNS Server 2
    //////////////////////////////

    NN_LOG( "==============================\n" );
    NN_LOG( "==\n" );
    NN_LOG( "== STARTING SECOND DNS SERVER\n" );
    NN_LOG( "==\n" );
    NN_LOG( "== 127.0.0.1:9000\n" );
    NN_LOG( "==============================\n" );


    g_LoopbackDnsServer2 = new LoopbackDnsServer( "127.0.0.1", 9000, "127.0.0.1", 9000 );
    if ( g_LoopbackDnsServer2 == NULL )
    {
        NN_LOG( "Failed to allocate a server2 / LoopbackDnsServer\n" );
        goto out;
    }

    // Create a thread to run the Server in
    threadId = g_pThreadedTest->CreateThread( &StartLookbackDnsServer, (void *) g_LoopbackDnsServer2, kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and start DNS Server (thread)!\n" );
        goto out;
    }
    NN_LOG( "Successfully started [DNS-2] Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    // Tell Log what we are doing!
    NN_LOG( "=========================================\n" );
    NN_LOG( "==  Setting NIFM DNS Servers List to:\n" );
    NN_LOG( "==\n" );
    NN_LOG( "== 127.0.0.1:8053\n" );
    NN_LOG( "== 127.0.0.1:9000\n" );
    NN_LOG( "=========================================\n" );

    // Set new DNS Server Addresses
    rc = SetDnsResolver( "127.0.0.1", 8053, "127.0.0.1", 9000 );
    ERROR_IF( rc == -1, "SetDnsResolver() Failed but Success was expected!  rc = %d\n", rc );

    return;

out:

    // If the (Second) loopback DNS server is allocated
    if ( g_LoopbackDnsServer2 != NULL )
    {
        g_LoopbackDnsServer2->Stop();
        delete g_LoopbackDnsServer2;
        g_LoopbackDnsServer2 = NULL;
    }

    return;
}

namespace {   // Anonymous

int
DNS_Retry_Callback_A( void *                  inArg,
                      unsigned char *         dnsRqst,
                      size_t &                dnsRqstLen,
                      unsigned char *         dnsResp,
                      size_t &                dnsRespLen,
                      bool *                  isUDP,
                      bool *                  shouldContinue )
{
    LoopbackDnsServer *   thisObj = reinterpret_cast<LoopbackDnsServer *>( inArg );

    // Server HIT!
    g_ServerHit = true;

    // Keep a running total
    g_WorkVaruint32_t++;
    g_WorkVaruint32_t3 = g_ServerA;

    NN_LOG( "========> IN Callback A!  W1=%d, W2=%d <===========\n", g_WorkVaruint32_t, g_WorkVaruint32_t2);

    if ( g_WorkVaruint32_t == g_WorkVaruint32_t2 )
    {
        return( g_LoopbackDnsServer->MakeDnsResponseCB( thisObj, dnsRqst, dnsRqstLen, dnsResp, dnsRespLen, isUDP, shouldContinue )  );
    }
    else
    {
        *shouldContinue = false;
    }

    return( 0 );
}

static int
DNS_Retry_Callback_B( void *                  inArg,
                      unsigned char *         dnsRqst,
                      size_t &                dnsRqstLen,
                      unsigned char *         dnsResp,
                      size_t &                dnsRespLen,
                      bool *                  isUDP,
                      bool *                  shouldContinue )
{
    LoopbackDnsServer *        thisObj = (LoopbackDnsServer *) inArg;

    // Server HIT!
    g_ServerHit = true;

    // Keep a running total
    g_WorkVaruint32_t++;
    g_WorkVaruint32_t3 = g_ServerB;

    NN_LOG( "========> IN Callback B!  W1=%d, W2=%d <===========\n", g_WorkVaruint32_t, g_WorkVaruint32_t2);

    if ( g_WorkVaruint32_t == g_WorkVaruint32_t2 )
    {
        return( g_LoopbackDnsServer2->MakeDnsResponseCB( thisObj, dnsRqst, dnsRqstLen, dnsResp, dnsRespLen, isUDP, shouldContinue )  );
    }
    else
    {
        *shouldContinue = false;
    }

    return( 0 );
}

} // Anonymous


TEST(ApiUnit,DnsCache_Two_Server_Retry)
{
    bool              isSuccess = true, rval = false;

    // We require 2x DNS servers;
    ERROR_IF( g_LoopbackDnsServer2 == NULL, "This test requires 2x DNS Servers!  Failing\n" );

    // [DNS1] Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB( DNS_Retry_Callback_A );

    // [DNS1] Set DNS Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *)  DnsCacheGoogleComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheGoogleComSimple );

    // [DNS2] Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer2->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer2->SetDNSResponseCB( DNS_Retry_Callback_B );

    // [DNS2] Set DNS Response
    g_LoopbackDnsServer2->UDPResponse = (unsigned char *)  DnsCacheGoogleComSimple;
    g_LoopbackDnsServer2->UDPResponseLen = sizeof(  DnsCacheGoogleComSimple );

    //*********************
    //* GHBN
    //*********************/

    // Initialize running total
    g_WorkVaruint32_t  = 0;
    g_WorkVaruint32_t2 = 0;

    // For I = 1 to 7
    for( g_WorkVaruint32_t2 = 1; g_WorkVaruint32_t2 < 9; g_WorkVaruint32_t2++ )
    {
        NN_LOG( "Testing Iteration: [%d]\n", g_WorkVaruint32_t2 );

        // Flush the Cache
        NN_LOG( "Calling Cache Flush\n" );
        CacheFlush();

        // Clear Counter
        g_WorkVaruint32_t = 0;

        // Make Call
        NN_LOG( "Query: [GetHostByName][Google]\n" );
        rval = CallGetHostByName( "google.com", nullptr, 0 );
        if ( rval == true )
        {
            if ( g_WorkVaruint32_t == g_WorkVaruint32_t2 )
            {
                if (  g_WorkVaruint32_t3 == g_ServerA )
                {
                    switch( g_WorkVaruint32_t2 )
                    {
                    case 1: break;
                    case 3: break;
                    case 5: break;
                    case 7: break;
                    case 9: break;
                    default:     NN_LOG( "====== FAILED:  Server DNS Server A processed a DNS record for Server B! W1=%d, W1=%d, w3=%d\n",
                                         g_WorkVaruint32_t,  g_WorkVaruint32_t2,  g_WorkVaruint32_t3 );

                                 ERROR_IF_AND_COUNT( rval == true, "Failed while processing!\n" );
                                 break;
                   }
                }
                else if ( g_WorkVaruint32_t3 == g_ServerB )
                {
                    switch( g_WorkVaruint32_t2 )
                    {
                    case 2: break;
                    case 4: break;
                    case 6: break;
                    case 8: break;
                    default:     NN_LOG( "====== FAILED:  Server DNS Server B processed a DNS record for Server A! W1=%d, W1=%d, w3=%d\n",
                                        g_WorkVaruint32_t,  g_WorkVaruint32_t2,  g_WorkVaruint32_t3 );

                                 ERROR_IF_AND_COUNT( rval == true, "Failed while processing!\n" );
                                 break;
                    }
                }

                NN_LOG( "=============> MATCHED <=============\n" );
                g_ServerHit = false;
                rval = CallGetHostByName( "google.com", nullptr, 0 );
                ERROR_IF_AND_COUNT( rval == false, "GHBN should have successfully read a (cached) DNS response!\n" );
                ERROR_IF_AND_COUNT( g_ServerHit == true, "GHBN queried a DNS Server - but it was supposed to read the (cache)!\n" );
            }
            else
            {
                ERROR_IF_AND_COUNT( rval == true, "GHBN received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );
            }
        }
    }

    NN_LOG( "There were: [%d] GHBN calls to the DNS server\n", g_WorkVaruint32_t );

    //*********************
    //* GETADDRINFO
    //*********************/

    // Initialize running total
    g_WorkVaruint32_t  = 0;
    g_WorkVaruint32_t2 = 0;

    // For I = 1 to 7
    for( g_WorkVaruint32_t2 = 1; g_WorkVaruint32_t2 < 9; g_WorkVaruint32_t2++ )
    {
        // Flush the Cache
        NN_LOG( "Calling Cache Flush\n" );
        CacheFlush();

        NN_LOG( "Testing Iteration: [%d]\n", g_WorkVaruint32_t2 );

        // Clear Counter
        g_WorkVaruint32_t = 0;

        // Make Call
        NN_LOG( "Query: [GetHostByName][Google]\n" );
        rval = CallGetAddrInfo( "google.com", nullptr, 0 );
        if ( rval == true )
        {
            if ( g_WorkVaruint32_t == g_WorkVaruint32_t2 )
            {
                if (  g_WorkVaruint32_t3 == g_ServerA )
                {
                    switch( g_WorkVaruint32_t2 )
                    {
                    case 1: break;
                    case 3: break;
                    case 5: break;
                    case 7: break;
                    case 9: break;
                    default:     NN_LOG( "====== FAILED:  Server DNS Server A processed a DNS record for Server B! W1=%d, W1=%d, w3=%d\n",
                                         g_WorkVaruint32_t,  g_WorkVaruint32_t2,  g_WorkVaruint32_t3 );

                                 ERROR_IF_AND_COUNT( rval == true, "Failed while processing!\n" );
                                 break;
                   }
                }
                else if ( g_WorkVaruint32_t3 == g_ServerB )
                {
                    switch( g_WorkVaruint32_t2 )
                    {
                    case 2: break;
                    case 4: break;
                    case 6: break;
                    case 8: break;
                    default:     NN_LOG( "====== FAILED:  Server DNS Server B processed a DNS record for Server A! W1=%d, W1=%d, w3=%d\n",
                                        g_WorkVaruint32_t,  g_WorkVaruint32_t2,  g_WorkVaruint32_t3 );

                                 ERROR_IF_AND_COUNT( rval == true, "Failed while processing!\n" );
                                 break;
                    }
                }

                NN_LOG( "=============> MATCHED <=============\n" );
                g_ServerHit = false;
                rval = CallGetAddrInfo( "google.com", nullptr, 0 );
                ERROR_IF_AND_COUNT( rval == false, "GAI should have successfully read a (cached) DNS response!\n" );
                ERROR_IF_AND_COUNT( g_ServerHit == true, "GAI queried a DNS Server - but it was supposed to read the (cache)!\n" );
            }
            else
            {
                ERROR_IF_AND_COUNT( rval == true, "GHBN received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );
            }
        }
    }

    NN_LOG( "There were: [%d] GAI calls to the DNS server\n", g_WorkVaruint32_t );

    //*********************
    //* GHBA
    //*********************/

    // [DNS1] Set DNS Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsCacheERootComSimple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsCacheERootComSimple );

    // [DNS2] Set DNS Response
    g_LoopbackDnsServer2->UDPResponse = (unsigned char *) DnsCacheERootComSimple;
    g_LoopbackDnsServer2->UDPResponseLen = sizeof( DnsCacheERootComSimple );

    // Initialize running total
    g_WorkVaruint32_t  = 0;
    g_WorkVaruint32_t2 = 0;

    // For I = 1 to 7
    for( g_WorkVaruint32_t2 = 1; g_WorkVaruint32_t2 < 9; g_WorkVaruint32_t2++ )
    {
        // Flush the Cache
        NN_LOG( "Calling Cache Flush\n" );
        CacheFlush();

        NN_LOG( "Testing Iteration: [%d]\n", g_WorkVaruint32_t2 );

        // Clear Counter
        g_WorkVaruint32_t = 0;

        // Make Call
        NN_LOG( "Query: [GetHostByAddr][ERoot]\n" );
        rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, nullptr, 0 );
        if ( rval == true )
        {
            if ( g_WorkVaruint32_t == g_WorkVaruint32_t2 )
            {
                if (  g_WorkVaruint32_t3 == g_ServerA )
                {
                    switch( g_WorkVaruint32_t2 )
                    {
                    case 1: break;
                    case 3: break;
                    case 5: break;
                    case 7: break;
                    case 9: break;
                    default:     NN_LOG( "====== FAILED:  Server DNS Server A processed a DNS record for Server B! W1=%d, W1=%d, w3=%d\n",
                                         g_WorkVaruint32_t,  g_WorkVaruint32_t2,  g_WorkVaruint32_t3 );

                                 ERROR_IF_AND_COUNT( rval == true, "Failed while processing!\n" );
                                 break;
                   }
                }
                else if ( g_WorkVaruint32_t3 == g_ServerB )
                {
                    switch( g_WorkVaruint32_t2 )
                    {
                    case 2: break;
                    case 4: break;
                    case 6: break;
                    case 8: break;
                    default:     NN_LOG( "====== FAILED:  Server DNS Server B processed a DNS record for Server A! W1=%d, W1=%d, w3=%d\n",
                                        g_WorkVaruint32_t,  g_WorkVaruint32_t2,  g_WorkVaruint32_t3 );

                                 ERROR_IF_AND_COUNT( rval == true, "Failed while processing!\n" );
                                 break;
                    }
                }

                NN_LOG( "=============> MATCHED <=============\n" );
                g_ServerHit = false;
                rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, nullptr, 0 );
                ERROR_IF_AND_COUNT( rval == false, "GHBA should have successfully read a (cached) DNS response!\n" );
                ERROR_IF_AND_COUNT( g_ServerHit == true, "GHBA queried a DNS Server - but it was supposed to read the (cache)!\n" );
            }
            else
            {
                ERROR_IF_AND_COUNT( rval == true, "GHBA received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );
            }
        }
    }

    NN_LOG( "There were: [%d] GHBA calls to the DNS server\n", g_WorkVaruint32_t );

    //*********************
    //* GNI
    //*********************/

    // Initialize running total
    g_WorkVaruint32_t  = 0;
    g_WorkVaruint32_t2 = 0;

    // For I = 1 to 7
    for( g_WorkVaruint32_t2 = 1; g_WorkVaruint32_t2 < 9; g_WorkVaruint32_t2++ )
    {
        // Flush the Cache
        NN_LOG( "Calling Cache Flush\n" );
        CacheFlush();

        NN_LOG( "Testing Iteration: [%d]\n", g_WorkVaruint32_t2 );

        // Clear Counter
        g_WorkVaruint32_t = 0;

        // Make Call
        NN_LOG( "Query: [GetNameInfo][ERoot]\n" );
        rval = CallGetNameInfo( "192.203.230.10", nullptr, 0 );
        if ( rval == true )
        {
            if ( g_WorkVaruint32_t == g_WorkVaruint32_t2 )
            {
                if (  g_WorkVaruint32_t3 == g_ServerA )
                {
                    switch( g_WorkVaruint32_t2 )
                    {
                    case 1: break;
                    case 3: break;
                    case 5: break;
                    case 7: break;
                    case 9: break;
                    default:     NN_LOG( "====== FAILED:  Server DNS Server A processed a DNS record for Server B! W1=%d, W1=%d, w3=%d\n",
                                         g_WorkVaruint32_t,  g_WorkVaruint32_t2,  g_WorkVaruint32_t3 );

                                 ERROR_IF_AND_COUNT( rval == true, "Failed while processing!\n" );
                                 break;
                   }
                }
                else if ( g_WorkVaruint32_t3 == g_ServerB )
                {
                    switch( g_WorkVaruint32_t2 )
                    {
                    case 2: break;
                    case 4: break;
                    case 6: break;
                    case 8: break;
                    default:     NN_LOG( "====== FAILED:  Server DNS Server B processed a DNS record for Server A! W1=%d, W1=%d, w3=%d\n",
                                        g_WorkVaruint32_t,  g_WorkVaruint32_t2,  g_WorkVaruint32_t3 );

                                 ERROR_IF_AND_COUNT( rval == true, "Failed while processing!\n" );
                                 break;
                    }
                }

                NN_LOG( "=============> MATCHED <=============\n" );
                g_ServerHit = false;
                rval = CallGetNameInfo( "192.203.230.10", nullptr, 0 );
                ERROR_IF_AND_COUNT( rval == false, "GNI should have successfully read a (cached) DNS response!\n" );
                ERROR_IF_AND_COUNT( g_ServerHit == true, "GNI queried a DNS Server - but it was supposed to read the (cache)!\n" );
            }
        }
    }

    NN_LOG( "There were: [%d] GNI calls to the DNS server\n", g_WorkVaruint32_t );

out:

    return;

}  // NOLINT(impl/function_size)

TEST(ApiUnit,DnsCache_Teardown)
{
    TeardownTesting();
}

#else // Windows

TEST(ApiUnit,DnsCache_Window)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

}}  // Namespace: NATF::API
