﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/socket.h>
#include <nnt/nntest.h>

#include "Complex/testNet_SelectUnitNetworkCommon.h"

namespace NATF {
namespace API {

NN_ALIGNAS(4096) uint8_t g_SelectTimeoutSimpleSocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

uint64_t getTicksNow()
{
    nn::os::Tick now_in_ticks = nn::os::GetSystemTick();
    return( nn::os::ConvertToTimeSpan( now_in_ticks ).GetMicroSeconds() );
}

void testTCPSelectTimeout(nn::socket::TimeVal * pTimeout, int in_flags )
{
    nn::Result result;
    int rc = -1;
    int sockfd = -1;
    int flags = -1;
    nn::socket::FdSet readfds, writefds, exceptionfds;
    uint64_t start_usec, stop_usec, delta_usec, timeout_usec;
    uint64_t kTimeoutValue = 0;

    // retain result to finalize the socket layer below
    result = nn::socket::Initialize(g_SelectTimeoutSimpleSocketMemoryPoolBuffer,
                                    nn::socket::DefaultSocketMemoryPoolSize,
                                    nn::socket::MinSocketAllocatorSize,
                                    nn::socket::DefaultConcurrencyLimit);

    if( result.IsFailure() )
    {
        NN_LOG("Error: nn::socket::Initialize() failed. Err Desc: %d\n", result.GetDescription());
        ADD_FAILURE();
        goto bail;
    }
    // create the socket
    else if (-1 == (sockfd = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp)))
    {
        NN_LOG("Unable to get socket, rc: %d, errno: %d\n", rc, sockfd, nn::socket::GetLastError());
        ADD_FAILURE();
        goto bail;
    }

    if ( SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING == in_flags )
    {
        if ( -1 == (rc = nn::socket::Fcntl(sockfd, nn::socket::FcntlCommand::F_SetFl, 0)))
        {
            NN_LOG("Error: Fcntl(%d, nn::socket::FcntlCommand::F_SetFl, 0) returned -1\n", sockfd);
            ADD_FAILURE();
            goto bail;
        }
        else if (-1 == (rc = nn::socket::Fcntl(sockfd, nn::socket::FcntlCommand::F_GetFl, flags)))
        {
            NN_LOG("Error: Fcntl(%d, nn::socket::FcntlCommand::F_GetFl, 0) returned -1\n", sockfd);
            ADD_FAILURE();
            goto bail;
        }
        else if ( 0 != ( static_cast<int>(nn::socket::FcntlFlag::O_NonBlock) & rc))
        {
            NN_LOG("Error:  flags(%x) contains nn::socket::FcntlFlag::O_NonBlock, but should not, because we set the initial flags value.\n", rc);
            ADD_FAILURE();
            goto bail;
        }
    }

    // Top level Test
    for( int idx = 0; idx < 3; idx++ )
    {

       nn::socket::FdSetZero(&readfds);
       nn::socket::FdSetZero(&writefds);
       nn::socket::FdSetZero(&exceptionfds);

       // Not used
       nn::socket::FdSetSet(sockfd, &readfds);
       nn::socket::FdSetSet(sockfd, &exceptionfds);

       // Start time
       start_usec = getTicksNow();

       // Select
       rc = nn::socket::Select(sockfd + 1, &readfds, &writefds, &exceptionfds, pTimeout );
       if ( 0 != rc )
       {
           NN_LOG( "FAILED: select() return cocde is <%ld>, but I expected it to be (ZERO == 0)!\n" );
           ADD_FAILURE();
           goto bail;
       }

       // End time
       stop_usec = getTicksNow();

       // Delta time
       delta_usec = stop_usec - start_usec;

       // Convert timeout MS
       timeout_usec =  pTimeout->tv_sec * 1000000;
       timeout_usec += pTimeout->tv_usec;

       NN_LOG( "pTimeout->sec %d, pTimeout->tv_usec %d us!\n",pTimeout->tv_sec,  pTimeout->tv_usec);

       // Display result
       NN_LOG( "Start_usec <%llu>, Stop_usec <%llu>, Elasped_usec <%llu>, Timeout_usec <%llu>\n",
                    start_usec, stop_usec, delta_usec, timeout_usec );
       // RESULTS:

       // We don't allow select() to timeout QUICKER/FASTER than the timeout
       if ( delta_usec < timeout_usec ) {
          NN_LOG( "FAILED: select() timed out after waiting <%llu>usec, but timeout is set to <%llu>usec!\n",
                           delta_usec, timeout_usec );
          ADD_FAILURE();
          continue;
//        goto bail;
       }

       kTimeoutValue = timeout_usec + 3000000;

       // system call should unblock before 3s delta
       if ( delta_usec > kTimeoutValue )
       {
           NN_LOG( "FAILED: select() timed out after waiting <%llu>usec's\n", delta_usec );
           NN_LOG( "but expected timeout to UNBLOCK within 3 seconds, or <%llu>usec's!\n", kTimeoutValue );
           ADD_FAILURE();
           continue;
//         goto bail;
       }

    };  // for()

bail:
    if ( -1 != sockfd )
    {
        nn::socket::Close(sockfd);
        sockfd = -1;
    };

    if (result.IsSuccess())
    {
        result = nn::socket::Finalize();
        if( result.IsFailure() )
        {
            NN_LOG("Error: nn::socket::Finalize() failed. Err Desc: %d\n", result.GetDescription());
        };
    };
};

TEST(SelectTimeoutSimple, notimeout)
{
    nn::socket::TimeVal  testtime = { 0, 0};
    testTCPSelectTimeout( &testtime, 0 );
};

TEST(SelectTimeoutSimple, 1usec)
{
    nn::socket::TimeVal  testtime = { 0, 1 };
    testTCPSelectTimeout( &testtime, 0 );
};

TEST(SelectTimeoutSimple, 5usecs)
{
    nn::socket::TimeVal  testtime = { 0, 5 };
    testTCPSelectTimeout( &testtime, 0 );
};

TEST(SelectTimeoutSimple, 10usecs)
{
    nn::socket::TimeVal  testtime = { 0, 10 };
    testTCPSelectTimeout( &testtime, 0 );
};

TEST(SelectTimeoutSimple, 100usecs)
{
    nn::socket::TimeVal  testtime = { 0, 100 };
    testTCPSelectTimeout( &testtime, 0 );
};

TEST(SelectTimeoutSimple, 1ms)
{
    nn::socket::TimeVal  testtime = { 0, (1 * 1000 ) };
    testTCPSelectTimeout( &testtime, 0 );
};

TEST(SelectTimeoutSimple, 5ms)
{
    nn::socket::TimeVal  testtime = { 0, (5 * 1000 ) };
    testTCPSelectTimeout( &testtime, 0 );
};

TEST(SelectTimeoutSimple, 10ms)
{
    nn::socket::TimeVal  testtime = { 0, (10 * 1000 ) };
    testTCPSelectTimeout( &testtime, 0 );
};

TEST(SelectTimeoutSimple, 100ms)
{
    nn::socket::TimeVal  testtime = { 0, (100 * 1000 ) };
    testTCPSelectTimeout( &testtime, 0 );
};

TEST(SelectTimeoutSimple, 1s)
{
    nn::socket::TimeVal  testtime = { 1, 0 };
    testTCPSelectTimeout( &testtime, 0 );
};

TEST(SelectTimeoutSimple, 1_1s)
{
    nn::socket::TimeVal  testtime = { 1, 1 };
    testTCPSelectTimeout( &testtime, 0 );
};

TEST(SelectTimeoutSimple, 1_5s)
{
    nn::socket::TimeVal  testtime = { 1, ( 500 * 1000 ) };
    testTCPSelectTimeout( &testtime, 0 );
};

TEST(SelectTimeoutSimple, 1_999999s)
{
    nn::socket::TimeVal  testtime = { 1, 999999 };
    testTCPSelectTimeout( &testtime, 0 );
};

TEST(SelectTimeoutSimple, 5s)
{
    nn::socket::TimeVal  testtime = { 5, 0 };
    testTCPSelectTimeout( &testtime, 0 );
};

TEST(SelectTimeoutSimple, 5_1s)
{
    nn::socket::TimeVal  testtime = { 5, (1 * 1000) };
    testTCPSelectTimeout( &testtime, 0 );
};


} // namespace NATF
} // namespace API
