﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 Test process for Network
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/socket.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Thread.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/init.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/socket/socket_ApiPrivate.h>

#include "testNet_ApiCommon.h"
#include "natf/Utils/md5.h"
#include "natf/Utils/CommandLineParser.h"
#include "natf/Utils/InitApis.h"
#include <nn/socket/resolver/resolver_Client.h>

namespace
{
static const int kNumTestThreads =  32;

// Socket memory
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
volatile bool g_WorkerThreadsTerminated = false;
volatile bool g_WorkerThreadComplete = false;

const int threadPriority        = 30;

NN_OS_ALIGNAS_THREAD_STACK uint8_t threadStack[kNumTestThreads][16384];
volatile int g_cancelHandle[kNumTestThreads];

nn::os::ThreadType myworkerThread[kNumTestThreads];

bool ExpectedError(nn::socket::Errno lastError)
{
    return lastError == nn::socket::Errno::ESuccess ||
           lastError == nn::socket::Errno::EAgain ||
           lastError == nn::socket::Errno::EInval ||
           lastError == nn::socket::Errno::EChild ||
           lastError == nn::socket::Errno::ETimedOut ||
           lastError == nn::socket::Errno::ECanceled;
}


void workerThread(void* arg)
{
    bool                GetHostByNameComplete = false;
    bool                isSuccess = true;
    nn::socket::Errno   lastError = nn::socket::Errno::ESuccess;

    int threadIndex = ((intptr_t)(arg)) & 0xFFFFFFFF;
    NN_LOG("Thread %d start\n", threadIndex);

    nn::socket::SetLastError(nn::socket::Errno::ESuccess);
    nn::socket::ResolverOption option;
    nn::socket::ResolverGetOption(&option, nn::socket::ResolverOptionKey::GetCancelHandleInteger);
    g_cancelHandle[threadIndex] = option.data.integerValue;

    while((lastError = nn::socket::GetLastError()) == nn::socket::Errno::EAgain)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        nn::socket::SetLastError(nn::socket::Errno::ESuccess);
        nn::socket::ResolverGetOption(&option, nn::socket::ResolverOptionKey::GetCancelHandleInteger);
        g_cancelHandle[threadIndex] = option.data.integerValue;
    }

    ERROR_IF(!ExpectedError(lastError), "Unexpected GetLastError code: %d\n", lastError);

    // No handle given, we already report an error if bad so just bail if it was an ok error.
    if( g_cancelHandle[threadIndex] != 0)
    {
        do
        {
            nn::socket::SetLastError(nn::socket::Errno::ESuccess);

            nn::socket::HostEnt* pHostEntry = NULL;
            const char* hostname = "www.nintendo.co.jp";
            uint32_t g_ResolvedAddress = 0;

            nn::socket::ResolverOption cancelOption;
            cancelOption.key = nn::socket::ResolverOptionKey::RequestCancelHandleInteger;
            cancelOption.type = nn::socket::ResolverOptionType::Integer;
            cancelOption.size = sizeof(int);
            cancelOption.data.integerValue = g_cancelHandle[threadIndex];

            if ((pHostEntry = nn::socket::GetHostEntByName(hostname, &cancelOption, 1)) != NULL)
            {
                nn::socket::InAddr **addr_list;

                addr_list = (nn::socket::InAddr **)pHostEntry->h_addr_list;

                NN_LOG("Host name: %s\n", pHostEntry->h_name);

                g_ResolvedAddress = *(uint32_t*)((nn::socket::SockAddrIn*) * pHostEntry->h_addr_list);
                NN_LOG("Resolved! %s %08X \n", nn::socket::InetNtoa(**addr_list), g_ResolvedAddress);

                GetHostByNameComplete = true;
                break;
            }

            lastError = nn::socket::GetLastError();

            if(lastError != nn::socket::Errno::EAgain)
            {
                ERROR_IF(!ExpectedError(lastError), "Unexpected GetLastError code: %d\n", lastError);

                NN_LOG("Proper Resolver Failure: %d \n", lastError);
                GetHostByNameComplete = true;
            }
            else
            {
                NN_LOG("nn::socket::Errno::EAgain returned (system busy) from nn::socket::GetHostByName() - Retry in 10ms\n");
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
        }
        while(!GetHostByNameComplete);
    }

    g_WorkerThreadComplete = true;
out:
    while (!g_WorkerThreadsTerminated)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    NN_LOG("Return from Thread %d\n", threadIndex);

    return;
}

void CancelResolverCall(int threadId)
{
    bool isSuccess = true;

    while(!g_cancelHandle[threadId])
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
    }

    nn::socket::SetLastError(nn::socket::Errno::ESuccess);

    const int cancelHandle = g_cancelHandle[threadId];
    NN_LOG("Cancel thread %d with handle %08x\n", threadId, cancelHandle);
    nn::socket::ResolverOption option;
    option.key = nn::socket::ResolverOptionKey::SetCancelHandleInteger;
    option.type = nn::socket::ResolverOptionType::Integer;
    option.size = sizeof(int);
    option.data.integerValue = cancelHandle;

    nn::socket::Errno lastError = nn::socket::GetLastError();
    ERROR_IF(!ExpectedError(lastError), "Unexpected GetLastError() errocode reported: %s (%d)\n",
             strerror(static_cast<int>(lastError)), lastError);

    while(lastError == nn::socket::Errno::EAgain)
    {
        ERROR_IF(!ExpectedError(lastError), "Unexpected GetLastError() errocode reported: %s (%d)\n",
                 strerror(static_cast<int>(lastError)), lastError);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));

        nn::socket::SetLastError(nn::socket::Errno::ESuccess);

        option.key = nn::socket::ResolverOptionKey::SetCancelHandleInteger;
        option.type = nn::socket::ResolverOptionType::Integer;
        option.size = sizeof(int);
        option.data.integerValue = cancelHandle;
        nn::socket::ResolverSetOption(option);
        lastError = nn::socket::GetLastError();
    }

out:
    g_cancelHandle[threadId] = 0;
    return;
}

bool RunCancelTest()
{
    bool isSuccess = true;
    nn::Result result;
    int conditionalExpressionSuppression = kNumTestThreads;

    for(int i=0;i<kNumTestThreads; i++)
        g_cancelHandle[i] = 0;

    NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm);

    NN_LOG("*** Generation 1 (Early finalize with cancelations test) ***\n\n");
    /* Pass large enough buffer for:
        * - transfer memory to pass to the socket process (used for socket buffers, ~100kB per socket), must be >= nn::socket::MinTransferMemorySize
        * - allocator in the socket library, must be >= nn::socket::MinSocketAllocatorSize
        */
    result = nn::socket::Initialize(
                    reinterpret_cast<void*>(g_SocketMemoryPoolBuffer),
                    nn::socket::DefaultSocketMemoryPoolSize,
                    nn::socket::MinSocketAllocatorSize,
                    nn::socket::DefaultConcurrencyLimit);

    ERROR_IF(result.IsFailure(), "Failed to initialize socket Api. Desc: %d\n\n", result.GetDescription());

    /* one server, two client threads */
    for(long long i = 0; i < kNumTestThreads / 2; i++)
    {
        NN_LOG("*** Create Thread %i\n", i);
        nn::os::CreateThread(&myworkerThread[i],  &workerThread, (void*)i, threadStack[i], sizeof(threadStack[i]), threadPriority);

        NN_LOG("*** Start Thread %i\n", i);
        nn::os::StartThread(&myworkerThread[i]);
    }

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));

    CancelResolverCall(2);

    if(conditionalExpressionSuppression > 12)
    {
        CancelResolverCall(4);
    }

    if(conditionalExpressionSuppression > 18)
    {
        CancelResolverCall(8);
    }

    //nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(8000));

    NN_LOG("*** Finalize Generation 1\n");
    nn::socket::Finalize();

    /* give server time to startup */
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

    NN_LOG("*** Generation 2 (Multi-thread completion test) ***\n\n");

    result = nn::socket::Initialize(
                    reinterpret_cast<void*>(g_SocketMemoryPoolBuffer),
                    nn::socket::DefaultSocketMemoryPoolSize,
                    nn::socket::MinSocketAllocatorSize,
                    nn::socket::DefaultConcurrencyLimit);
    ERROR_IF(result.IsFailure(), "Failed to initialize socket Api. Desc: %d\n\n", result.GetDescription());

    for(long long i = kNumTestThreads / 2; i < kNumTestThreads; i++)
    {
        NN_LOG("*** Create Thread %i\n", i);
        nn::os::CreateThread(&myworkerThread[i],  &workerThread, (void*)i, threadStack[i], sizeof(threadStack[i]), threadPriority);

        NN_LOG("*** Start Thread %i\n", i);
        nn::os::StartThread(&myworkerThread[i]);
    }


    g_WorkerThreadsTerminated = true;
    /* give server time to startup */

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5000));

    NN_LOG("*** Finalize Generation 2 ***\n");

    for(int i=0; i<kNumTestThreads;i++)
    {
        NN_LOG("Wait Thread %i\n", i);
        nn::os::WaitThread(&myworkerThread[i]);
        NN_LOG("Destroy Thread %i\n",i);
        nn::os::DestroyThread(&myworkerThread[i]);
    }

    nn::socket::Finalize();
    /* this application no longer requires use of network interface */
    NN_LOG("*** Nifm Cleanup");
    NATF::API::TestTeardown();

out:
    return isSuccess;
}

bool RunSweepTest(int delay, bool raceTest)
{
    bool isSuccess = true;
    nn::Result result;
    int numThreads = 1;

    for(int i=0;i<numThreads; i++)
        g_cancelHandle[i] = 0;

    /* Pass large enough buffer for:
        * - transfer memory to pass to the socket process (used for socket buffers, ~100kB per socket), must be >= nn::socket::MinTransferMemorySize
        * - allocator in the socket library, must be >= nn::socket::MinSocketAllocatorSize
        */
    result = nn::socket::Initialize(
                    reinterpret_cast<void*>(g_SocketMemoryPoolBuffer),
                    nn::socket::DefaultSocketMemoryPoolSize,
                    nn::socket::MinSocketAllocatorSize,
                    nn::socket::DefaultConcurrencyLimit);

    ERROR_IF(result.IsFailure(), "Failed to initialize socket Api. Desc: %d\n\n", result.GetDescription());

    /* one server, two client threads */
    for(long long i=0; i<numThreads;i++)
    {
        nn::os::CreateThread(&myworkerThread[i],  &workerThread, (void*)i, threadStack[i], sizeof(threadStack[i]), threadPriority);
        nn::os::StartThread(&myworkerThread[i]);
    }

    if(delay == 0)
    {
        g_WorkerThreadComplete = false;

        // 0 delay means spin until thread is ready
        while(!g_WorkerThreadComplete)
        {
             nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));
        }
    }
    else
    {
        if(raceTest)
        {
            CancelResolverCall(0);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(delay));
        }
        else
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(delay));
            CancelResolverCall(0);
        }
    }

    nn::socket::Finalize();

    g_WorkerThreadsTerminated = true;
    /* give server time to startup */

    for(int i=0; i<numThreads;i++)
    {
        nn::os::WaitThread(&myworkerThread[i]);
        nn::os::DestroyThread(&myworkerThread[i]);
    }

out:
    return isSuccess;
}



} // namespace Anonymous


namespace NATF {
namespace API {

TEST(CancelApi, Timing)
{
    bool isSuccess = true;
    NN_LOG("In\n\n");

    NN_LOG("Heavy Thread Test.  Running many threads, canceling a few, and finalizing at the same time across 2 initialize/finalize cycles.\n");
    isSuccess = RunCancelTest();

    EXPECT_EQ(isSuccess, true);
    NN_LOG("Out\n\n");
}

TEST(CancelApi, Sweep)
{
    bool isSuccess = true;
    NN_LOG("In\n\n");

    NN_LOG("*** Finalize the API at different times while canceling an operation.\n");

    if(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm))
    {
        isSuccess = false;
    }
    else
    {
        for(int i=1; i<100; i++)
        {
            NN_LOG("*** Delay sweep test - Generation %d, delay %d msec\n", i, i * 1);
            isSuccess = RunSweepTest(i * 1, false);
            NN_LOG("\n");

            if(!isSuccess)
            {
                break;
            }
        }

        if(isSuccess)
        {
            NN_LOG("*** Delay sweep test - Final Generation - make sure a resolve can occur.\n");
            isSuccess = RunSweepTest(0, false);
        }

        NN_LOG("*** Nifm Cleanup");
        NATF::API::TestTeardown();
    }

    EXPECT_EQ(isSuccess, true);
    NN_LOG("Out\n\n");
}

TEST(CancelApi, CancelGetHostByNameRace)
{
    bool isSuccess = true;
    NN_LOG("In\n\n");

    NN_LOG("*** Cancel the GetHostByName call as soon as possible to induce any Bionic/Siglo races.\n");

    if(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm))
    {
        isSuccess = false;
    }
    else
    {
        for(int i=1; i<40; i++)
        {
            NN_LOG("*** Cancel Race Test - Generation %d, delay %d msec\n", i, i * 3);
            isSuccess = RunSweepTest(i * 3, true);
            NN_LOG("\n");

            if(!isSuccess)
            {
                break;
            }
        }

        NN_LOG("*** Nifm Cleanup");
        NATF::API::TestTeardown();
    }

    EXPECT_EQ(isSuccess, true);
    NN_LOG("Out\n\n");
}

}} // namespace NATF::API
