﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Utils/ExchangeInfo.h"
#include <nn/htcs.h>
#include <Utils/GTestUtil.h>
#include <nn/nifm/nifm_ApiIpAddress.h>

#include <cstdlib>
#include <memory>

#include "NetTest_Port.h"

namespace // unnamed, for constants accessible only to this file
{

    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }

    const uint32_t OperationWaitDurationMs = 100;
    const uint32_t AcceptTimeoutMs         = 20000;
}

namespace NATF {
namespace Utils {

    bool ExchangeInfo::Exchange() NN_NOEXCEPT
    {
        nn::Result result;
        bool isSuccess = true;
        int rval;
        int sock;
        int serviceSocket;
        uint32_t acceptWaitMs = 0;
        uint8_t expectedBytes = 0;
        uint8_t ipStrLen = 0;
        ssize_t byteCount = 0;
        ssize_t totalReceived = 0;
        ssize_t totalSent = 0;
        nn::socket::InAddr localAddress;

        NATF_LOG("\n Starting: ExchangeInfo\n");

        memset(&localAddress, 0, sizeof(localAddress));
        result = nn::nifm::GetCurrentPrimaryIpAddress( &localAddress );
        EXCHANGE_INFO_FAIL_IF(result.IsFailure(), "Failed to get address.\n\n");

        NETTEST_SNPRINTF(m_pLocalIpStr, sizeof(m_pLocalIpStr), "%s", nn::socket::InetNtoa(localAddress));
        NATF_LOG("Local Ip Address: %s\n\n", m_pLocalIpStr);
        ipStrLen = static_cast<uint8_t>(strlen(m_pLocalIpStr));

        nn::htcs::Initialize(&Allocate, &Deallocate);

        // register the service
        nn::htcs::SockAddrHtcs serviceAddr;
        serviceAddr.family = nn::htcs::HTCS_AF_HTCS;
        serviceAddr.peerName = nn::htcs::GetPeerNameAny();
        std::strcpy(serviceAddr.portName.name, SdevServerName);

        serviceSocket = nn::htcs::Socket();
        EXCHANGE_INFO_FAIL_IF( serviceSocket < 0, "ERROR: Socket create failed, err=%d\n",  nn::htcs::GetLastError() );

        rval = nn::htcs::Bind(serviceSocket, &serviceAddr);
        EXCHANGE_INFO_FAIL_IF( rval != 0, "ERROR: Bind failed, err=%d\n",  nn::htcs::GetLastError() );

        rval = nn::htcs::Listen(serviceSocket, 1);
        EXCHANGE_INFO_FAIL_IF( rval != 0, "ERROR: Listen failed, err=%d\n",  nn::htcs::GetLastError() );

        rval = Fcntl(serviceSocket, nn::htcs::HTCS_F_SETFL, nn::htcs::HTCS_O_NONBLOCK);
        EXCHANGE_INFO_FAIL_IF( rval != 0, "Failed to set socket to non-blocking.\n\n");

        // wait for a connection
        NATF_LOG("Waiting for connection from host.\n");
        acceptWaitMs = 0;
        do
        {
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(OperationWaitDurationMs) );
            acceptWaitMs += OperationWaitDurationMs;

            sock = nn::htcs::Accept(serviceSocket, nullptr);
            EXCHANGE_INFO_FAIL_IF( sock < 0 && nn::htcs::GetLastError() != nn::htcs::HTCS_EAGAIN, "ERROR: Accept failed, err=%d\n",  nn::htcs::GetLastError() );
            EXCHANGE_INFO_FAIL_IF( acceptWaitMs >= AcceptTimeoutMs, "ERROR: Accept timeout\n");
        } while( sock < 0 );

        // got a connection!!
        NATF_LOG("Accepted client on host.\n");

        rval = Fcntl(sock, nn::htcs::HTCS_F_SETFL, nn::htcs::HTCS_O_NONBLOCK);
        EXCHANGE_INFO_FAIL_IF( rval != 0, "Failed to set socket to non-blocking.\n\n");

        do
        {
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(OperationWaitDurationMs) );
            byteCount = nn::htcs::Recv(sock, &expectedBytes, 1, 0);
        } while( byteCount != 1 && nn::htcs::GetLastError() == nn::htcs::HTCS_EAGAIN );

        EXCHANGE_INFO_FAIL_IF( byteCount != 1, "Failed to receive! htcs errno: %d\n\n", nn::htcs::GetLastError() );
        EXCHANGE_INFO_FAIL_IF( expectedBytes > sizeof(m_pRemoteIpStr), "Ip buffer not big enough for expected receive!\n\n" );

        do
        {
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(OperationWaitDurationMs) );
            byteCount = nn::htcs::Recv(sock, &m_pRemoteIpStr[totalReceived], expectedBytes - totalReceived, 0);
            if( byteCount > 0 )
            {
                totalReceived += byteCount;
            }
            else if( byteCount < 0 && nn::htcs::GetLastError() != nn::htcs::HTCS_EAGAIN )
            {
                NATF_LOG("Failed to receive! htcs errno: %d\n\n", nn::htcs::GetLastError());
                EXCHANGE_INFO_FAIL();
            }
            else // byteCount == 0
            {
                NATF_LOG("Failed to receive! Connection was closed\n\n");
                EXCHANGE_INFO_FAIL();
            }
        } while( totalReceived < expectedBytes);

        m_pRemoteIpStr[totalReceived] = '\0';
        NATF_LOG("Received IP: %s\n\n", m_pRemoteIpStr);

        byteCount = nn::htcs::Send(sock, &ipStrLen, 1, 0);
        if( byteCount != 1 )
        {
            NATF_LOG("Failed to send! rval: %d errno: %d\n\n", byteCount, nn::htcs::GetLastError());
            EXCHANGE_INFO_FAIL();
        }

        do
        {
            byteCount = nn::htcs::Send(sock, &m_pLocalIpStr[totalSent], ipStrLen - totalSent, 0);
            if( byteCount > 0 )
            {
                totalSent += byteCount;
            }
            else if( byteCount <= 0 && nn::htcs::GetLastError() != nn::htcs::HTCS_EAGAIN )
            {
                NATF_LOG("Failed to send! htcs errno: %d\n\n", nn::htcs::GetLastError());
                EXCHANGE_INFO_FAIL();
            }
        } while( totalSent < ipStrLen);

        // break the connection
        rval = nn::htcs::Close(sock);
        EXCHANGE_INFO_FAIL_IF( rval != 0, "ERROR: Close data socket failed, err=%d\n",  nn::htcs::GetLastError() );

        rval = nn::htcs::Close(serviceSocket);
        EXCHANGE_INFO_FAIL_IF( rval != 0, "ERROR: Close listen socket failed, err=%d\n",  nn::htcs::GetLastError() );

out:
        NATF_LOG("\n Finished: ExchangeInfo\n");
        return isSuccess;
    }

}} // Namespaces NATF::Utils
