﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Tests/SslHttpsDownloadMultiTest.h"
#include <nn/ssl.h>
#include <nn/nifm.h>

#include <new> // For placement new
#include <cstring>
#include <cstdlib> // For Malloc

namespace NATF {
namespace Tests {

    // Constructor
    SslHttpsDownloadMulti::SslHttpsDownloadMulti(const char* pTestName, const nn::util::Uuid& netProfile, uint32_t timeoutMs, uint32_t connectionCount, const char* pHostName, uint16_t portNum, const char* pResource, nn::ssl::Connection::VerifyOption verifyOption, const MD5Hash::Result& hashResult) NN_NOEXCEPT
        : BaseTest(pTestName, false, Utils::InitApiFlags::InitApiFlags_Nifm | Utils::InitApiFlags::InitApiFlags_Network | Utils::InitApiFlags::InitApiFlags_Socket | Utils::InitApiFlags::InitApiFlags_Ssl, netProfile),
          m_connectionCount(connectionCount),
          m_pModules(nullptr),
          m_portNum(portNum),
          m_timeoutMs(timeoutMs),
          m_verifyOption(verifyOption),
          m_expectedHash(hashResult)
    {
        if( pHostName )
        {
            NETTEST_SNPRINTF(m_pHostName, sizeof(m_pHostName), "%s", pHostName);
        }
        else
        {
            m_pHostName[0] = '\0';
        }

        if( pResource )
        {
            NETTEST_SNPRINTF(m_pResource, sizeof(m_pResource), "%s", pResource);
        }
        else
        {
            m_pResource[0] = '\0';
        }
    }

    // Destructor
    SslHttpsDownloadMulti::~SslHttpsDownloadMulti() NN_NOEXCEPT
    {
        FreeModules();
    }

    // Cleanup
    bool SslHttpsDownloadMulti::Cleanup() NN_NOEXCEPT
    {
        FreeModules();
        return true;
    }

    // Config
    bool SslHttpsDownloadMulti::Config() NN_NOEXCEPT
    {
        if( m_connectionCount > 999 )
        {
            LogError(" * Number of connections must be less than 999.\n\n");
            return false;
        }

        if( !AllocateModules() )
        {
            return false;
        }

        for(unsigned i = 0; i < m_connectionCount; ++i)
        {
            char pThreadName[] = "SslHttpsDown_000";
            size_t nameLength = strlen(pThreadName);
            NETTEST_SNPRINTF(&pThreadName[nameLength - 3], 3, "%d", i + 1);

            TestThread* pServerThread = CreateTestThread(pThreadName, m_timeoutMs);
            if( !pServerThread )
            {
                return false;
            }

            pServerThread->AddModule(m_pModules[i]);
        }

        return true;
    }

    // AllocateModules
    bool SslHttpsDownloadMulti::AllocateModules() NN_NOEXCEPT
    {
        // Allocate Server modules
        m_pModules = (Modules::SslHttpsDownload*)malloc(sizeof(Modules::SslHttpsDownload) * m_connectionCount);
        if( !m_pModules )
        {
            Log(" Failed to allocate memory!\n\n");
            FreeModules();
            return false;
        }
        else
        {
            // Initialize all server modules with placement new.
            for(unsigned i = 0; i < m_connectionCount; ++i)
            {
                new (&m_pModules[i]) Modules::SslHttpsDownload(false, m_pHostName, m_portNum, m_pResource, false, m_expectedHash, true, m_verifyOption);
            }
        }

        return true;
    }

    // FreeModules
    void SslHttpsDownloadMulti::FreeModules() NN_NOEXCEPT
    {
        if( m_pModules )
        {
            // Call destructors for modules.
            Log("Calling destructors on modules...\n");
            for(unsigned i = 0; i < m_connectionCount; ++i)
            {
                (&m_pModules[i])->~SslHttpsDownload();
            }

            Log("Deallocating modules...\n");
            free(m_pModules);
            m_pModules = nullptr;
        }
    }

} // namespace Tests
} // namespace NATF
