﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Tests/CurlHttpUploadTest.h"
#include "curl/curl.h"
#include <nn/nifm.h>
#include <nn/fs.h>

#include <new> // For placement new
#include "cstdlib" // For malloc

namespace NATF {
namespace Tests {

    // Constructor
    CurlHttpUpload::CurlHttpUpload(const char* pTestName, const nn::util::Uuid& netProfile, uint32_t timeoutMs, uint32_t connectionCount, const NATF::Modules::LibCurl::Params& params) NN_NOEXCEPT
        :
        BaseTest(pTestName, false, Utils::InitApiFlags::InitApiFlags_Nifm | Utils::InitApiFlags::InitApiFlags_Network | Utils::InitApiFlags::InitApiFlags_Socket | Utils::InitApiFlags::InitApiFlags_Curl, netProfile),
        m_pModules(nullptr),
        m_timeoutMs(timeoutMs),
        m_connectionCount(connectionCount),
        m_params(params)
    {
        m_params.httpMethod = Modules::LibCurl::HttpMethod_Post;
    }

    // Destructor
    CurlHttpUpload::~CurlHttpUpload() NN_NOEXCEPT
    {
        FreeModules();
    }

    // Init
    bool CurlHttpUpload::Init() NN_NOEXCEPT
    {
        nn::Result result = nn::fs::MountHostRoot();
        if( result.IsFailure() )
        {
            Log(" * Warning: Failed to mount host file system!\n\n");
        }

        return true;
    }

    // Cleanup
    bool CurlHttpUpload::Cleanup() NN_NOEXCEPT
    {
        FreeModules();
        nn::fs::UnmountHostRoot();

        return true;
    }

    // AllocateModules
    bool CurlHttpUpload::AllocateModules() NN_NOEXCEPT
    {
        // Allocate Server modules
        m_pModules = (Modules::LibCurl*)malloc(sizeof(Modules::LibCurl) * m_connectionCount);
        if( !m_pModules )
        {
            Log(" Failed to allocate memory!\n\n");
            return false;
        }
        else
        {
            // Initialize all server modules with placement new.
            for(uint32_t i = 0; i < m_connectionCount; ++i)
            {
                new (&m_pModules[i]) Modules::LibCurl(m_params);
            }
        }

        return true;
    }

    // FreeModules
    void CurlHttpUpload::FreeModules() NN_NOEXCEPT
    {
        if( m_pModules )
        {
            // Call destructors for modules.
            Log("Calling destructors on modules...\n");
            for(uint32_t i = 0; i < m_connectionCount; ++i)
            {
                (&m_pModules[i])->~LibCurl();
            }

            Log("Deallocating modules...\n");
            free(m_pModules);
            m_pModules = nullptr;
        }
    }

    // Config
    bool CurlHttpUpload::Config() NN_NOEXCEPT
    {
        if( !AllocateModules() )
        {
            return false;
        }

        for(uint32_t i = 0; i < m_connectionCount; ++i)
        {
            char pThreadName[] = "CurlUpload_000";
            size_t nameLength = strlen(pThreadName);
            NETTEST_SNPRINTF(&pThreadName[nameLength - 3], 3, "%d", i + 1);

            TestThread* pThread = CreateTestThread(pThreadName, m_timeoutMs);
            if( !pThread )
            {
                return false;
            }

            pThread->AddModule(m_pModules[i]);
        }

        return true;
    }

} // namespace Tests
} // namespace NATF
