﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "NetTest_Port.h"
#include "Modules/SslAbuseModule.h"
#include "Utils/InternetTestHosts.h"
#include <nn/ssl.h>
#include <nn/fs.h>

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <string>
#include <memory> // For std::unique_ptr<>

#define DESTROY_SSL_OBJ(OBJ, IS_SUCCESS, LOGGER)                      \
do                                                                    \
{                                                                     \
    if(OBJ != nullptr)                                                \
    {                                                                 \
        if((OBJ)->Destroy().IsFailure())                              \
        {                                                             \
            LOGGER(" WARNING: Failed to SSL object!\n\n");            \
            IS_SUCCESS = false;                                       \
        }                                                             \
                                                                      \
        delete OBJ;                                                   \
        OBJ = nullptr;                                                \
    }                                                                 \
} while(NN_STATIC_CONDITION(false))

#define CHECK_AND_CLOSE_SOCKET(SOCKET, IS_SUCCESS, LOGGER) \
do                                             \
{                                              \
    if( SOCKET >= 0 )                          \
    {                                          \
        int RVAL = NetTest::Close(SOCKET);     \
        if( RVAL != 0 )                        \
        {                                      \
            LOGGER("Error: Failed to close socket. rval: %d, SOErr: %d\n", RVAL, NetTest::GetLastError()); \
            IS_SUCCESS = false;                \
        }                                      \
        else                                   \
        {                                      \
            SOCKET = -1;                       \
        }                                      \
    }                                          \
} while(NN_STATIC_CONDITION(false))

namespace
{
    const uint32_t         BufferSize      = 1024 * 2;
    const char* const      g_pHttpGetCmd   = "GET";
    const char* const      g_pHttpPostCmd  = "POST";
    const char* const      g_pHttpVers     = "HTTP/1.1";
    const char* const      g_pHttpPostContentType = "application/x-www-form-urlencoded";
    const char* const      g_pPostDataPrefix  = "name=file&value=";
    const char* const      g_pPostDataPattern = "0123456789qwertyuiopasdfghjklzxcvbnmQWERTYUIOPASDFGHJKLZXCVBNM";
    const char* const      g_pHttpCodePrefix  = "HTTP/1.";
    const char* const      g_pHashFor1MegPost = "6a,a9,cb,9b,7b,29,2a,ac,b6,3d,bf,d6,40,91,69,79";
    const char* const      g_pHashFor1KPost   = "b9,4d,30,11,f4,8d,28,3e,75,e3,28,f2,f0,16,77,6b";
    const uint32_t         PollTimeoutMs   = 10000;
    bool                   g_threadSuccess = true;
    const uint32_t         CertBuffLen     = 1024 * 4;
    char                   g_pServerCertBuff[CertBuffLen] = {0};
    const uint32_t         MaxTryHostNameResolverCount = 10;
    const uint32_t         TimeBetweenResolverTrysMs = 50;
    const uint32_t         MaxContextCount = 8;
    const uint32_t         ThreadCount     = 8;
    const uint32_t         ClientCertCount = 8;
    const uint32_t         StackSize       = 16 * 1024;
    const uint32_t         ServerCertCount = 32;
    const uint32_t         CertChainCount  = 8;
    const uint32_t         CertChainBufLen = 64 * 1024;
    NetTest::Thread        g_pThreads[ThreadCount];
    uint32_t               g_pIndeces[ThreadCount];
    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_pThreadStack[ThreadCount][StackSize];
    bool                   g_doHandshakeOnThread = true;
    uint32_t               g_importCrlLoopOnThreadCount = 64;
    const uint32_t         NatfHostnameBufLen = 16;

    const uint32_t CreateDestroyCtxCount               = 5000;
    const uint32_t CreateConnectionCount               = 5000;
    const uint32_t HandshakeDiffHostsCount             = 2048;
    const uint32_t BlockingIDCacheDownloadCount        = 2048;
    const uint32_t NonBlockingTicketCacheDownloadCount = 2048;
    const uint32_t DownloadSameConnectionCount         = 2048;

    const uint32_t DownloadWithServerCertDetailsBlockingCount      = 512;
    const uint32_t DownloadWithServerCertDetailsNonBlockingCount   = 512;
    const uint32_t DownloadWithServerCertDetailsSessionIdCount     = 512;
    const uint32_t DownloadWithServerCertDetailsSessionTicketCount = 512;

    const uint32_t ImportCrlSingleContextNoHandshakeCount   = 512;
    const uint32_t ImportCrlMultipleContextNoHandshakeCount = 512;
    const uint32_t ImportCrlMultipleThreadNoHandshakeCount  = 512;

    const uint32_t ImportCrlSingleContextWithHandshakeCount   = 64;
    const uint32_t ImportCrlMultipleContextWithHandshakeCount = 64;
    const uint32_t ImportCrlMultipleThreadWithHandshakeCount  = 64;

    struct DownloadThreadParams
    {
        DownloadThreadParams() :
            pTest(nullptr),
            isGetCertChainEnabled(false),
            isPostEnabled(false),
            isUseLocalServerEnabled(false),
            verifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_None),
            pHostName(nullptr),
            portNum(0)
        {}

        NATF::Modules::SslAbuse* pTest;
        bool isGetCertChainEnabled;
        bool isPostEnabled;
        bool isUseLocalServerEnabled;
        nn::ssl::Connection::VerifyOption verifyOption;
        const char* pHostName;
        uint16_t portNum;
    };

    struct CrlThreadParams
    {
        CrlThreadParams() NN_NOEXCEPT
            : pThis(nullptr),
              index(0) {}

        NATF::Modules::SslAbuse* pThis;
        uint8_t index;
    };

    class File
    {
    public:

        File() :
            m_pData(nullptr),
            m_isFileOpen(false),
            m_cachedSize(0) {}

        char* OpenAndRead(const std::string& path, uint32_t& outFileSize)
        {
            outFileSize = 0;
            int64_t totalBytesRead = 0;

            if(m_isFileOpen)
            {
                nn::fs::CloseFile(m_Handle);
                m_isFileOpen = false;
            }

            if( m_pData )
            {
                delete [] m_pData;
                m_pData = nullptr;
            }

            nn::Result result = nn::fs::OpenFile(&m_Handle, path.c_str(), nn::fs::OpenMode_Read);
            if( result.IsFailure() )
            {
                NATF_LOG("Failed to open file! Desc: %d\n\n", result.GetDescription());
                return nullptr;
            }

            m_isFileOpen = true;

            int64_t fileSize = 0;
            result = GetSize(&fileSize);
            if( result.IsFailure() || fileSize <= 0 )
            {
                goto out;
            }

            if( fileSize > (0xFFFFFFFFFFFFFFFF >> ((8 - sizeof(uint32_t)) * 8)) )
            {
                NATF_LOG("Error! File size too large to pre-allocate!\n\n");
                goto out;
            }

            m_pData = new char[static_cast<uint32_t>(fileSize)];
            if( !m_pData )
            {
                goto out;
            }

            while( totalBytesRead < fileSize )
            {
                size_t bytesRead = 0;
                size_t maxRead = static_cast<size_t>(fileSize - totalBytesRead); // Truncation is ok.
                result = Read(&bytesRead, m_pData, totalBytesRead, maxRead);
                if( result.IsFailure() )
                {
                    delete [] m_pData;
                    m_pData = nullptr;

                    goto out;
                }

                totalBytesRead += bytesRead;
            }

            // Overflow was already checked above
            outFileSize = static_cast<uint32_t>(fileSize);
            m_cachedSize = fileSize;

out:
            if(m_isFileOpen)
            {
                nn::fs::CloseFile(m_Handle);
                m_isFileOpen = false;
            }

            return m_pData;
        }

        nn::Result Open(const std::string& path, int mode)
        {
            Close();

            nn::Result result = nn::fs::OpenFile(&m_Handle, path.c_str(), mode);

            if(result.IsSuccess())
            {
                m_isFileOpen = true;
            }

            return result;
        }

        nn::Result Read(void* buffer, int64_t offset, size_t length)
        {
            return nn::fs::ReadFile(m_Handle, offset, buffer, length, nn::fs::ReadOption());
        }

        nn::Result Read(size_t* outValue, void* buffer, int64_t offset, size_t length)
        {
            return nn::fs::ReadFile(outValue, m_Handle, offset, buffer, length, nn::fs::ReadOption());
        }

        nn::Result Write(const void* buffer, int64_t offset, size_t length)
        {
            return nn::fs::WriteFile(m_Handle, offset, buffer, length, nn::fs::WriteOption());
        }

        nn::Result SetSize(int64_t size)
        {
            return nn::fs::SetFileSize(m_Handle, size);
        }

        nn::Result GetSize(int64_t* outValue)
        {
            nn::Result result = nn::fs::GetFileSize(outValue, m_Handle);
            if(result.IsSuccess())
            {
                m_cachedSize = *outValue;
            }

            return result;
        }

        int64_t GetCachedSize()
        {
            return m_cachedSize;
        }

        nn::Result Flush()
        {
            return nn::fs::FlushFile(m_Handle);
        }

        void Close()
        {
            if(m_isFileOpen)
            {
                nn::fs::CloseFile(m_Handle);
                m_isFileOpen = false;
            }

            if( m_pData )
            {
                delete [] m_pData;
                m_pData = nullptr;
            }
        }

        char* GetData()
        {
            return m_pData;
        }

        ~File()
        {
           Close();
        }

    private:
        nn::fs::FileHandle m_Handle;
        char* m_pData;
        bool m_isFileOpen;
        int64_t m_cachedSize;
    };

    File g_pServerCertFiles[ServerCertCount];
    File g_pClientCertFiles[ClientCertCount];
    File g_pRootCertFiles[CertChainCount];
    File g_pIntermediateCertFiles[CertChainCount];
    File g_pRootCrlFiles[CertChainCount];
} // un-named namespace

namespace NATF {
namespace Modules {

    SslAbuse::HeaderData::HeaderData() :
        isHeaderComplete(false),
        isLastNewLine(false),
        contentLength(0),
        httpResponse(0),
        headerPos(0)
    {
        memset(pHeaderBuf, 0, sizeof(pHeaderBuf));
    }

    // Constructor
    SslAbuse::SslAbuse(const char* pHostName, unsigned short port, const char* pServerCertPath) NN_NOEXCEPT
        :
        m_pHostName(pHostName),
        m_port(port),
        m_pServerCertPath(pServerCertPath),
        m_pSslContext(nullptr)
        { }

    // SendHttpRequest
    bool SslAbuse::SendHttpRequest(nn::ssl::Connection& sslConnection, const char* pResource, const char* pHostName) const NN_NOEXCEPT
    {
        bool isSuccess = true;
        unsigned totalWritten;
        static const unsigned MaxRequestSize = 1024;
        char pGetRequest[MaxRequestSize];

        // Copy http request into write buffer.
        NETTEST_SNPRINTF(pGetRequest, MaxRequestSize, "%s /%s %s\r\nHost: %s\r\n\r\n", g_pHttpGetCmd, pResource, g_pHttpVers, pHostName);
        size_t requestSize = strlen(pGetRequest);
        Log("HTTP Request: %s", pGetRequest);

        // While sending a buffer, only a portion may have successfully sent.
        // You may have to call the write funtion a few times to send any remaining data.
        totalWritten = 0;
        do
        {
            Log("Writing data...\n");
            int nBytesWritten = 0;
            nn::Result result = sslConnection.Write(&pGetRequest[totalWritten], &nBytesWritten, (unsigned)requestSize - totalWritten);
            if( result.IsFailure() )
            {
                LogError("Error: SSL Write failure! rval: %d desc: %d\n\n", nBytesWritten, result.GetDescription());
                isSuccess = false;
                goto out;
            }
            else // Write was successful(or at least part), keep track of how much has been sent.
            {
                totalWritten += nBytesWritten;
            }
        } while( totalWritten < requestSize );

    out:

        return isSuccess;
    }

    // SendHttpPost
    bool SslAbuse::SendHttpPost(nn::ssl::Connection& sslConnection, const char* pResource, const char* pHostName, int32_t dataLen) const NN_NOEXCEPT
    {
        bool isSuccess = true;
        unsigned totalWritten;
        const char* const pHttpPostFormat = "%s /%s %s\r\nHost: %s\r\nContent-Type: %s\r\nContent-Length: %d\r\nConnection: Keep-Alive\r\n\r\n";

        // Copy http request into write buffer.
        int headerLen = NETTEST_SNPRINTF(nullptr, 0, pHttpPostFormat, g_pHttpPostCmd, pResource, g_pHttpVers, pHostName, g_pHttpPostContentType, dataLen);

        int requestBufferLen = dataLen + headerLen;

        std::unique_ptr<char[]> pRequest(new char[requestBufferLen]);
        if(nullptr == pRequest)
        {
            LogError("Failed to allocate buffer for HTTP POST\n\n");
            return false;
        }

        NETTEST_SNPRINTF(pRequest.get(), requestBufferLen, pHttpPostFormat, g_pHttpPostCmd, pResource, g_pHttpVers, pHostName, g_pHttpPostContentType, dataLen);

        size_t requestSize = headerLen;
        Log("HTTP Request: %s", pRequest.get());

        int prefixLen = NETTEST_SNPRINTF(&pRequest[requestSize], requestBufferLen - requestSize, g_pPostDataPrefix);
        requestSize += prefixLen;

        size_t dataPatternLen = strlen(g_pPostDataPattern);
        for(int32_t iByte = 0; iByte < dataLen - prefixLen; ++iByte)
        {
            // Fill data bytes with a looping pattern.
            pRequest[requestSize] = g_pPostDataPattern[iByte % dataPatternLen];
            ++requestSize;
        }

        // While sending a buffer, only a portion may have successfully sent.
        // You may have to call the write funtion a few times to send any remaining data.
        totalWritten = 0;
        do
        {
            Log("Writing data...\n");
            int nBytesWritten = 0;
            nn::Result result = sslConnection.Write(&pRequest[totalWritten], &nBytesWritten, (unsigned)requestSize - totalWritten);
            if( result.IsFailure() )
            {
                LogError("Error: SSL Write failure! rval: %d desc: %d\n\n", nBytesWritten, result.GetDescription());
                isSuccess = false;
                goto out;
            }
            else // Write was successful(or at least part), keep track of how much has been sent.
            {
                totalWritten += nBytesWritten;
            }
        } while( totalWritten < requestSize );

    out:

        return isSuccess;
    }

    // CreateSocketWithOptions
    int SslAbuse::CreateSocketWithOptions() const NN_NOEXCEPT
    {
        int  socketFd  = -1;
        int  rval      = -1;
        bool doCleanup = false;
        int  value     = -1;
        NetTest::SockLen paramLen;

        // Create socket for SSL connection.
        socketFd = NetTest::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
        if (socketFd < 0)
        {
            LogError("Error: Failed to create client socket. rval: %d, SOErr:%d\n", socketFd, NetTest::GetLastError());
            doCleanup = true;
            goto out;
        }

        value = 1024 * 64 - 1;
        rval = NetTest::SetSockOpt(socketFd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &value, sizeof(value));
        if (rval < 0)
        {
            Log("WARNING: Failed to set recv buffer! rval: %d, errno: %d\n\n", rval, NetTest::GetLastError());
        }

        paramLen = sizeof(int);
        rval = NetTest::GetSockOpt(socketFd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &value, &paramLen);
        if (rval < 0)
        {
            Log("WARNING: Failed to get recv buffer! rval: %d, errno: %d\n\n", rval, NetTest::GetLastError());
        }
        else
        {
            Log("nn::socket::Option::So_RcvBuf: %d\n", value);
        }

    out:

        if( doCleanup )
        {
            bool isCloseSuccess = true;
            CHECK_AND_CLOSE_SOCKET(socketFd, isCloseSuccess, LogError);
        }

        return socketFd;
    }

    void SslAbuse::InetNtoP(int32_t ipAddr, char pOutIp[MaxIpStrLen]) const
    {
        uint8_t* pData = reinterpret_cast<uint8_t*>(&ipAddr);

        NETTEST_SNPRINTF(pOutIp, MaxIpStrLen, "%d.%d.%d.%d", static_cast<int>(pData[0]),
                                                             static_cast<int>(pData[1]),
                                                             static_cast<int>(pData[2]),
                                                             static_cast<int>(pData[3]));
    }

    // ConnectToServer
    bool SslAbuse::ConnectToServer(int socketFd, const char* pHostName, unsigned short portNum) const NN_NOEXCEPT
    {
        bool isSuccess   = true;
        int  rval        = -1;
        NetTest::HostEnt* pHostEnt = nullptr;
        nn::socket::InAddr inetAddr;
        char pIp[MaxIpStrLen];
        NetTest::SockAddrIn serverAddr;
        uint32_t hostNameResolverCount = 0;

        memset(&serverAddr, 0, sizeof(serverAddr));

        if( !pHostName )
        {
            LogError("Error: pHostName is NULL\n");
            isSuccess = false;
            goto out;
        }

        Log("Resolving %s\n", pHostName);

        do
        {
            pHostEnt = NetTest::GetHostByName(pHostName);
            ++hostNameResolverCount;
            NetTest::SleepMs(TimeBetweenResolverTrysMs);
        } while(pHostEnt == nullptr && hostNameResolverCount < MaxTryHostNameResolverCount);

        if(pHostEnt == nullptr)
        {
            LogError("Failed to resolve host name.\n");
            isSuccess = false;
            goto out;
        }

        // Just pick the first one
        memcpy(&inetAddr, pHostEnt->h_addr_list[0], sizeof(inetAddr));
        InetNtoP(inetAddr.S_addr, pIp);
        Log("Server IP: %s\n", pIp);

        // Initialize server address struct
        serverAddr.sin_family      = nn::socket::Family::Af_Inet;
        serverAddr.sin_port        = NetTest::Htons(portNum);
        serverAddr.sin_addr.S_addr = inetAddr.S_addr;

        // Print server ip.
        Log("Connecting to %s\n", pHostName);

        rval = NetTest::Connect(socketFd, (NetTest::SockAddr *)&serverAddr, sizeof(serverAddr));
        if( rval < 0 )
        {
            LogError("Error: Connect: rval: %d SOErr: %d\n", rval, NetTest::GetLastError());
            isSuccess = false;
            goto out;
        }

    out:

        return isSuccess;
    }

    // ParseHeader
    char* SslAbuse::ParseHeader(char* pBuffer, unsigned bufSize, HeaderData& headerData) NN_NOEXCEPT
    {
        if( headerData.isLastNewLine && pBuffer[0] == '\n' )
        {
            headerData.isHeaderComplete = true;

            char* pFound = strstr(headerData.pHeaderBuf, "Content-Length: ");
            if(pFound)
            {
                sscanf(pFound, "Content-Length: %d", &headerData.contentLength);
            }

            pFound = strstr(headerData.pHeaderBuf, g_pHttpCodePrefix);
            if(pFound)
            {
                const char* pResponsePosition = pFound + strlen(g_pHttpCodePrefix) + 2; // +2 is for the rest of the http version and a space.
                if((pResponsePosition - headerData.pHeaderBuf) < HeaderBufLen)
                {
                    sscanf(pResponsePosition, "%d", &headerData.httpResponse);
                }
            }

            if( bufSize > 1 )
            {
                return &pBuffer[1];
            }
            else
            {
                return nullptr;
            }
        }

        unsigned i = 0;
        unsigned newLinesCount = 0;
        for(; i < bufSize; ++i)
        {
            if( pBuffer[i] == '\n' || pBuffer[i] == '\r' )
            {
                ++newLinesCount;
                if( newLinesCount >= 4 )
                {
                    int copySize = 0;
                    char* pReturn = nullptr;

                    headerData.isHeaderComplete = true;
                    if( i >= bufSize - 1 )
                    {
                        copySize = (bufSize > sizeof(headerData.pHeaderBuf) - headerData.headerPos - 1) ? sizeof(headerData.pHeaderBuf) - headerData.headerPos - 1 : bufSize;

                        pReturn = nullptr;
                    }
                    else
                    {
                        copySize = (i > sizeof(headerData.pHeaderBuf) - headerData.headerPos - 1) ? sizeof(headerData.pHeaderBuf) - headerData.headerPos - 1 : i;

                        pReturn = &pBuffer[i + 1];
                    }

                    memcpy(headerData.pHeaderBuf, pBuffer, copySize);
                    headerData.headerPos += copySize;

                    char* pFound = strstr(headerData.pHeaderBuf, "Content-Length: ");
                    if(pFound)
                    {
                        sscanf(pFound, "Content-Length: %d", &headerData.contentLength);
                    }

                    pFound = strstr(headerData.pHeaderBuf, g_pHttpCodePrefix);
                    if(pFound)
                    {
                        const char* pResponsePosition = pFound + strlen(g_pHttpCodePrefix) + 2; // +2 is for the rest of the http version and a space.
                        if((pResponsePosition - headerData.pHeaderBuf) < HeaderBufLen)
                        {
                            sscanf(pResponsePosition, "%d", &headerData.httpResponse);
                        }
                    }

                    return pReturn;

                }
            }
            else
            {
                newLinesCount = 0;
            }
        }

        if( newLinesCount )
        {
            headerData.isLastNewLine = true;
        }

        int copySize = (bufSize > sizeof(headerData.pHeaderBuf) - headerData.headerPos - 1) ? sizeof(headerData.pHeaderBuf) - headerData.headerPos - 1 : bufSize;
        memcpy(headerData.pHeaderBuf, pBuffer, copySize);
        headerData.headerPos += copySize;

        return nullptr;
    }

    // ReceiveResponse
    bool SslAbuse::ReceiveResponse(nn::ssl::Connection& sslConnection, bool isBlocking, uint32_t readChunk, int expectedHttpResponse) NN_NOEXCEPT
    {
        int bytesRead         = 0;
        int contentRead       = 0;
        bool isSuccess        = true;
        unsigned totalRead    = 0;
        bool hasPollTimedout  = false;
        char pBuffer[BufferSize + 1];
        nn::Result result = nn::ResultSuccess();
        uint32_t readSize = (readChunk > BufferSize) ? BufferSize : readChunk;
        HeaderData headerData;

        // Read the response from server
        Log("Reading header...\n\n");
        do
        {
            nn::ssl::Connection::PollEvent pollInEvent  = nn::ssl::Connection::PollEvent::PollEvent_Read;
            nn::ssl::Connection::PollEvent pollOutEvent = nn::ssl::Connection::PollEvent::PollEvent_None;

            Log("Polling...\n");
            result = sslConnection.Poll(&pollOutEvent, &pollInEvent, PollTimeoutMs);

            if( nn::ssl::ResultIoTimeout::Includes(result) )
            {
                Log("Poll: Timed out!\n");
                hasPollTimedout = true;
                break;
            }
            else if( result.IsFailure() )
            {
                LogError("Error: Poll: Desc: %d\n", result.GetDescription());
                isSuccess = false;
                goto out;
            }
            else if( (pollOutEvent & nn::ssl::Connection::PollEvent::PollEvent_Read) !=
                    nn::ssl::Connection::PollEvent::PollEvent_Read )
            {
                Log("Warning: Ssl connection not readable after poll!\n");
                break;
            }

            // Socket is now readable, read the data.
            bytesRead = 0;
            result = sslConnection.Read(pBuffer, &bytesRead, readSize);
            if(result.IsFailure())
            {
                if(!nn::ssl::ResultIoWouldBlock::Includes(result))
                {
                    LogError("Error: Recv: rval: %d Desc: %d\n", bytesRead, result.GetDescription());
                    isSuccess = false;
                    goto out;
                }
            }
            else if( bytesRead == 0 )
            {
                NN_LOG("\n");
                Log("Recv: Connection has been closed. rval: %d\n", bytesRead);
                break;
            }

            // If bytes were read
            if( bytesRead > 0 )
            {
                totalRead += bytesRead;
                char* pData = pBuffer;
                if( headerData.isHeaderComplete == false )
                {
                    pData = ParseHeader(pBuffer, bytesRead, headerData);

                    // If there was also data in this recv buf,
                    //  only print header portion.
                    if( pData )
                    {
                        NN_LOG("%.*s", pData - pBuffer, pBuffer);
                        contentRead = bytesRead - static_cast<int>(pData - pBuffer);
                    }
                    else
                    {
                        NN_LOG("%.*s", bytesRead, pBuffer);
                    }

                    if( headerData.isHeaderComplete )
                    {
                        Log("Header download complete! Content-Length: %d\n", headerData.contentLength);

                        if(expectedHttpResponse > 0 && headerData.httpResponse != expectedHttpResponse)
                        {
                            LogError("Error: Unexpected http response: %d. Expected: %d\n\n",  headerData.httpResponse, expectedHttpResponse);
                            isSuccess = false;
                            goto out;
                        }

                        Log("Reading data...\n\n");
                    }
                }
                else
                {
                    contentRead += bytesRead;
                    Log("Content Read: %d\n", contentRead);
                }
            }
        } while( (bytesRead > 0 || nn::ssl::ResultIoWouldBlock::Includes(result)) && (contentRead < headerData.contentLength || headerData.isHeaderComplete == false) );
        NN_LOG("\n");

    out:

        return isSuccess;
    }

    bool SslAbuse::ConfigSslContext(nn::ssl::Context* pOutContext) const NN_NOEXCEPT
    {
        bool isSuccess = true;

        nn::Result result = pOutContext->Create(nn::ssl::Context::SslVersion_Auto);
        if(result.IsFailure())
        {
            LogError("Failed to create context. Desc: %d\n", result.GetDescription());
            return false;
        }

        nn::ssl::SslConnectionId contextId;
        result = pOutContext->GetContextId(&contextId);
        if(result.IsFailure())
        {
            LogError("Failed to get context ID\n");
            DESTROY_SSL_OBJ(pOutContext, isSuccess, Log);

            return false;
        }

        Log("Created SSL context (ID:%d).\n", contextId);
        return isSuccess;
    }

    bool SslAbuse::ConfigSslConnection(nn::ssl::Connection* pOutConnection, nn::ssl::Context* pInContext, int socketFd, bool isBlocking, const char* pHostName, char* pServerCertBuff, uint32_t serverCertBuffLen, nn::ssl::Connection::VerifyOption verifyOption, nn::ssl::Connection::SessionCacheMode cacheMode, bool isGetCertChainEnable) const NN_NOEXCEPT
    {
        bool isSuccess = false;
        bool getCertChainOption = !isGetCertChainEnable; // Init to oposite to ensure the value is changed by GetOption()
        size_t hostNameLen = 0;
        bool isCloseSocketNeeded = true;

        nn::Result result = pOutConnection->Create(pInContext);
        if(result.IsFailure())
        {
            LogError("pOutConnection->Create failed. Desc: %d\n", result.GetDescription());
            goto cleanup;
        }

        result = pOutConnection->SetSocketDescriptor(socketFd);
        if(result.IsFailure())
        {
            LogError("pOutConnection->SetSocketDescriptor failed. Desc: %d\n", result.GetDescription());
            goto cleanup;
        }

        isCloseSocketNeeded = false;

        hostNameLen = strlen(pHostName);
        result = pOutConnection->SetHostName(pHostName, (uint32_t)hostNameLen);
        if(result.IsFailure())
        {
            LogError("pOutConnection->SetHostName failed. Desc: %d\n", result.GetDescription());
            goto cleanup;
        }

        result = pOutConnection->SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true);
        if(result.IsFailure())
        {
            LogError("pOutConnection->SetOption (OptionType_SkipDefaultVerify) failed. Desc: %d\n",
                result.GetDescription());
            goto cleanup;
        }

        result = pOutConnection->SetVerifyOption(verifyOption);
        if(result.IsFailure())
        {
            LogError("pOutConnection->SetVerifyOption failed. Desc: %d\n", result.GetDescription());
            goto cleanup;
        }

        // Set buffer to get peer certificate
        if(pServerCertBuff != nullptr)
        {
            result = pOutConnection->SetServerCertBuffer(pServerCertBuff, serverCertBuffLen);
            if(result.IsFailure())
            {
                LogError("SetServerCertBuffer failed. Desc: %d\n", result.GetDescription());
                goto cleanup;
            }
        }

        result = pOutConnection->SetSessionCacheMode(cacheMode);
        if(result.IsFailure())
        {
            LogError("pOutConnection->SetSessionCacheMode failed. Desc: %d\n", result.GetDescription());
            goto cleanup;
        }

        nn::ssl::Connection::IoMode ioMode;
        if( isBlocking )
        {
            ioMode = nn::ssl::Connection::IoMode_Blocking;
        }
        else
        {
            ioMode = nn::ssl::Connection::IoMode_NonBlocking;
        }

        result = pOutConnection->SetIoMode(ioMode);
        if(result.IsFailure())
        {
            LogError("SetIoMode failed.\n");
            goto cleanup;
        }

        result = pOutConnection->GetIoMode(&ioMode);
        if(result.IsFailure())
        {
            LogError("GetIoMode failed.\n");
            goto cleanup;
        }
        Log("IoMode: %d\n", ioMode);

        result = pOutConnection->SetOption(nn::ssl::Connection::OptionType_GetServerCertChain, isGetCertChainEnable);
        if(result.IsFailure())
        {
            LogError("SetOption failed.\n");
            goto cleanup;
        }

        result = pOutConnection->GetOption(&getCertChainOption, nn::ssl::Connection::OptionType_GetServerCertChain);
        if(result.IsFailure())
        {
            LogError("GetOption failed.\n");
            goto cleanup;
        }
        Log("GetServerCertChain: %d\n", getCertChainOption);

        if(getCertChainOption != isGetCertChainEnable)
        {
            LogError("GetOption() OptionType_GetServerCertChain does not match as expected.\n");
            goto cleanup;
        }

        isSuccess = true;

    cleanup:
        if( !isSuccess )
        {
            if(pOutConnection->Destroy().IsFailure())
            {
                LogError(" WARNING: Failed to SSL object!\n\n");
            }

            if(isCloseSocketNeeded)
            {
                // Close socket
                int rval = NetTest::Close(socketFd);
                if( rval != 0 )
                {
                    LogError("Warning: Failed to close socket. rval: %d, errno: %d\n\n", rval, NetTest::GetLastError());
                }
            }
        }

        return isSuccess;
    } // NOLINT(impl/function_size)

    void SslAbuse::DownloadThreadFn(void* pParam) NN_NOEXCEPT
    {
        DownloadThreadParams* pThreadParams = reinterpret_cast<DownloadThreadParams*>(pParam);
        SslAbuse* pThis = reinterpret_cast<SslAbuse*>(pThreadParams->pTest);
        nn::ssl::Connection* pSslConnection = nullptr;
        int expectedHttpResponse = 0;

        pThis->Log("******** DownloadThreadFn ********\n\n");

        // Create socket with custom options
        int socketFd = pThis->CreateSocketWithOptions();
        if( socketFd < 0 )
        {
            g_threadSuccess = false;
            goto out;
        }

        // Connect socket to server
        if( !pThis->ConnectToServer(socketFd, pThreadParams->pHostName, pThreadParams->portNum) )
        {
            g_threadSuccess = false;
            goto out;
        }

        pSslConnection = new nn::ssl::Connection();
        if(pSslConnection == nullptr)
        {
            pThis->LogError("Failed to allocate ssl connection\n");
            g_threadSuccess = false;
            goto out;
        }

        // Blocking connection
        if( !pThis->ConfigSslConnection(pSslConnection, pThis->m_pSslContext, socketFd, true, pThreadParams->pHostName, nullptr, 0, pThreadParams->verifyOption, nn::ssl::Connection::SessionCacheMode::SessionCacheMode_SessionId, pThreadParams->isGetCertChainEnabled) )
        {
            socketFd = -1;
            g_threadSuccess = false;
            goto out;
        }

        socketFd = -1;

        if(pThreadParams->isGetCertChainEnabled)
        {
            g_threadSuccess = pThis->CertChainHandshake(pSslConnection);
            if(!g_threadSuccess)
            {
                goto out;
            }
        }
        else
        {
            g_threadSuccess = pThis->NormalHandshake(pSslConnection, nn::ResultSuccess(), nn::ResultSuccess());
            if(!g_threadSuccess)
            {
                goto out;
            }
        }

        if(pThreadParams->isPostEnabled)
        {
            // Send HTTP POST request through SSL connection
            if( !pThis->SendHttpPost(*pSslConnection, g_pHashFor1MegPost, pThreadParams->pHostName, 1024 * 1024) )
            {
                g_threadSuccess = false;
                goto out;
            }

            expectedHttpResponse = 200;
        }
        else
        {
            const char* pResource = (pThreadParams->isUseLocalServerEnabled) ? "mb1" : "";

            // Send HTTP GET request through SSL connection
            if( !pThis->SendHttpRequest(*pSslConnection, pResource, pThreadParams->pHostName) )
            {
                g_threadSuccess = false;
                goto out;
            }

            expectedHttpResponse = 0;
        }

        // Receive response from our HTTP request
        if( !pThis->ReceiveResponse(*pSslConnection, true, 1024, expectedHttpResponse) )
        {
            g_threadSuccess = false;
            goto out;
        }


out:

        CHECK_AND_CLOSE_SOCKET(socketFd, g_threadSuccess, pThis->LogError);
        DESTROY_SSL_OBJ(pSslConnection, g_threadSuccess, pThis->LogError);
    }


    void SslAbuse::ImportThreadFn(void* pParam) NN_NOEXCEPT
    {
        uint32_t index = *reinterpret_cast<uint32_t*>(pParam);
        int64_t fileSize = 0;
        nn::Result result = nn::ResultSuccess();

        NN_LOG("\n\n ******** ImportThreadFn %d ********\n\n", index);

        nn::ssl::Context* pSslContext = new nn::ssl::Context();
        if(pSslContext == nullptr)
        {
            NN_LOG("Failed to allocate ssl context\n");
            g_threadSuccess = false;
            goto out;
        }

        result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
        if(result.IsFailure())
        {
            NN_LOG("Failed to create context.\n");
            g_threadSuccess = false;
            goto out;
        }

        for(uint32_t i = 0; i < ServerCertCount; ++i)
        {
            NN_LOG("Thread import: %d\n", i);

            fileSize = g_pServerCertFiles[i].GetCachedSize();

            nn::ssl::CertStoreId serverCertId;
            result = pSslContext->ImportServerPki( &serverCertId,
                                    g_pServerCertFiles[i].GetData(),
                                    static_cast<uint32_t>(fileSize),
                                    nn::ssl::CertificateFormat_Pem);

            if( result.IsFailure() )
            {
                NN_LOG(" * Error: Failed to import server cert! Desc: %d\n\n", result.GetDescription());
                g_threadSuccess = false;
                goto out;
            }
        }

        fileSize = g_pClientCertFiles[index].GetCachedSize();

        nn::ssl::CertStoreId clientCertId;
        result = pSslContext->ImportClientPki( &clientCertId,
                                            g_pClientCertFiles[index].GetData(),
                                            nullptr,
                                            static_cast<uint32_t>(fileSize),
                                            0);

        if( result.IsFailure() )
        {
            NN_LOG(" * Error: Failed to import client cert! Desc: %d\n\n", result.GetDescription());
            g_threadSuccess = false;
            goto out;
        }

out:

        DESTROY_SSL_OBJ(pSslContext, g_threadSuccess, NN_LOG);
    }

    bool SslAbuse::NormalHandshake(nn::ssl::Connection* pSslConnection, nn::Result expectedHandshakeResult, nn::Result expectedVerifyResult) NN_NOEXCEPT
    {
        bool isSuccess = true;
        nn::Result result;

        do
        {
            result = pSslConnection->DoHandshake();
        } while(nn::ssl::ResultIoWouldBlock::Includes(result));

        if(expectedHandshakeResult.GetInnerValueForDebug() != result.GetInnerValueForDebug())
        {
            LogError("Unexpected DoHandshake Result: %d\n\n", result.GetDescription());
            isSuccess = false;
        }

        if(nn::ssl::ResultVerifyCertFailed::Includes(result))
        {
            nn::Result verifyResult;
            nn::Result tempResult = pSslConnection->GetVerifyCertError(&verifyResult);
            if(tempResult.IsSuccess())
            {
                if(expectedVerifyResult.GetInnerValueForDebug() != verifyResult.GetInnerValueForDebug())
                {
                    LogError("Unexpected Verify Result: %d\n\n", verifyResult.GetDescription());
                    isSuccess = false;
                }
            }
            else
            {
                LogError("Failed to get Verify Result! Error Desc: %d\n\n", tempResult.GetDescription());
                isSuccess = false;
            }
        }

        return isSuccess;
    }

    bool SslAbuse::CertChainHandshake(nn::ssl::Connection* pSslConnection) NN_NOEXCEPT
    {
        bool isSuccess = true;
        nn::Result result;
        uint32_t certCount     = 0;
        uint32_t certChainSize = 0;

        char* pCertChainBuf = new char[CertChainBufLen];
        if(pCertChainBuf == nullptr)
        {
            LogError("Failed to allocate cert chain buffer!\n\n");
            isSuccess = false;
            goto out;
        }

        do
        {
            result = pSslConnection->DoHandshake(&certChainSize, &certCount, pCertChainBuf, CertChainBufLen);
            if( result.IsFailure() && !nn::ssl::ResultIoWouldBlock::Includes(result) )
            {
                LogError("Ssl hand shake failed! Desc: %d\n\n", result.GetDescription());

                nn::Result verifyResult;
                nn::Result tempResult = pSslConnection->GetVerifyCertError(&verifyResult);
                if(tempResult.IsSuccess())
                {
                    LogError("Verify Result: %d\n\n", verifyResult.GetDescription());
                }

                isSuccess = false;
                goto out;
            }
        } while(nn::ssl::ResultIoWouldBlock::Includes(result));

        Log("Cert count: %d, Cert chain size: %d\n", static_cast<int>(certCount), static_cast<int>(certChainSize));

        for(uint32_t iCert = 0; iCert < certCount; ++iCert)
        {
            nn::ssl::Connection::ServerCertDetail certDetail;
            result = pSslConnection->GetServerCertDetail(&certDetail, pCertChainBuf, iCert);
            if(result.IsFailure())
            {
                LogError("Failed to GetServerCertDetail() Index: %d Desc: %d\n\n", iCert, result.GetDescription());
                isSuccess = false;
                goto out;
            }

            Log("Cert Index: %d, Cert size: %d\n", iCert, certDetail.dataSize);
        }

out:
        if(pCertChainBuf != nullptr)
        {
            delete [] pCertChainBuf;
            pCertChainBuf = nullptr;
        }

        return isSuccess;
    }

    // Helper function that several test-cases use
    //  - Creates SSL connection
    //  - Does Ssl Handshake
    //  - Does GET request and receives data 'downloadCount' times
    bool SslAbuse::Download(nn::ssl::Context* pContext, uint32_t downloadCount, bool isBlocking, const char* pHostName, unsigned short serverPort, const char* pResource, nn::ssl::Connection::SessionCacheMode cacheMode, nn::ssl::Connection::VerifyOption verifyOption, bool isGetCertChainEnable, bool isDoPostEnabled) NN_NOEXCEPT
    {
        bool isSuccess = true;
        nn::ssl::Connection* pSslConnection = nullptr;
        int expectedHttpResponse = 0;

        // Create socket with custom options
        int socketFd = CreateSocketWithOptions();
        if( socketFd < 0 )
        {
            isSuccess = false;
            goto out;
        }

        // Connect socket to server
        if( !ConnectToServer(socketFd, pHostName, serverPort) )
        {
            isSuccess = false;
            goto out;
        }

        pSslConnection = new nn::ssl::Connection();
        if(pSslConnection == nullptr)
        {
            LogError("Failed to allocate ssl connection\n");
            isSuccess = false;
            goto out;
        }

        if( !ConfigSslConnection(pSslConnection, pContext, socketFd, isBlocking, pHostName, g_pServerCertBuff, CertBuffLen, verifyOption, cacheMode, isGetCertChainEnable) )
        {
            socketFd = -1;
            isSuccess = false;
            goto out;
        }

        socketFd = -1;

        if(isGetCertChainEnable)
        {
            if(CertChainHandshake(pSslConnection) == false)
            {
                isSuccess = false;
                goto out;
            }
        }
        else
        {
            if(NormalHandshake(pSslConnection, nn::ResultSuccess(), nn::ResultSuccess()) == false)
            {
                isSuccess = false;
                goto out;
            }
        }

        for(uint32_t iDownload = 0; iDownload < downloadCount; ++iDownload)
        {
            if(isDoPostEnabled)
            {
                // Send HTTP POST through SSL connection
                if( !SendHttpPost(*pSslConnection, g_pHashFor1KPost, pHostName, 1024) )
                {
                    isSuccess = false;
                    goto out;
                }

                expectedHttpResponse = 200;
            }
            else
            {
                // Send HTTP GET request through SSL connection
                if( !SendHttpRequest(*pSslConnection, pResource, pHostName) )
                {
                    isSuccess = false;
                    goto out;
                }

                expectedHttpResponse = 0;
            }

            // Receive response from our HTTP request
            if( !ReceiveResponse(*pSslConnection, isBlocking, 512, expectedHttpResponse) )
            {
                isSuccess = false;
                goto out;
            }
        }

out:

        CHECK_AND_CLOSE_SOCKET(socketFd, isSuccess, LogError);
        DESTROY_SSL_OBJ(pSslConnection, isSuccess, LogError);

        return isSuccess;
    }

    // Creates 8 threads. Each thread does the following:
    //  - Creates an ssl Context
    //  - Imports 'ServerCertCount' server certs('ServerCertCount' unique certs but each thread uses the same 'ServerCertCount' certs.)
    //  - Imports 1 client cert
    bool SslAbuse::ImportCertsOnDiffThreads() const NN_NOEXCEPT
    {
        g_threadSuccess = true;

        NN_LOG("\n\n");
        Log("********** ImportCertsOnDiffThreads **********\n\n");

        Log("Creating threads...");
        for(uint32_t i = 0; i < ThreadCount; ++i)
        {
            g_pIndeces[i] = i;
            bool isSuccess = NetTest::CreateThread(&g_pThreads[i], SslAbuse::ImportThreadFn, &g_pIndeces[i], g_pThreadStack[i], StackSize);
            if( !isSuccess )
            {
                LogError("Failed to create thread!\n\n");

                return false;
            }
        }
        NN_LOG("Done\n");

        Log("Starting threads...");
        for(uint32_t i = 0; i < ThreadCount; ++i)
        {
            NetTest::StartThread(&g_pThreads[i]);
        }
        NN_LOG("Done\n");

        Log("Waiting on threads...");
        for(uint32_t i = 0; i < ThreadCount; ++i)
        {
            NetTest::WaitThread(&g_pThreads[i]);
        }
        NN_LOG("Done\n");

        Log("Destroying on threads...");
        for(uint32_t i = 0; i < ThreadCount; ++i)
        {
            NetTest::DestroyThread(&g_pThreads[i]);
        }
        NN_LOG("Done\n");

        if(g_threadSuccess == false)
        {
            LogError(" * Error in download thread\n\n");
            return false;
        }

        return true;
    }

    // Create a context, Creates 8 threads. Each thread does the following:
    //  - Creates ssl connection(each thread from the same context)
    //  - Does http GET to download 1mb
    bool SslAbuse::DownloadOnDiffThreads(bool isGetCertChainEnabled, nn::ssl::Connection::VerifyOption verifyOption, bool isUseLocalServerEnabled, bool isPostEnabled) NN_NOEXCEPT
    {
        bool isSuccess = true;
        DownloadThreadParams threadParams[ThreadCount];

        for(uint32_t iThread = 0; iThread < ThreadCount; ++iThread)
        {
            threadParams[iThread].pTest = this;
            threadParams[iThread].isGetCertChainEnabled = isGetCertChainEnabled;
            threadParams[iThread].verifyOption = verifyOption;
            threadParams[iThread].isPostEnabled = isPostEnabled;
            threadParams[iThread].isUseLocalServerEnabled = isUseLocalServerEnabled;

            if(isUseLocalServerEnabled)
            {
                threadParams[iThread].pHostName = m_pHostName;
                threadParams[iThread].portNum = m_port;
            }
            else
            {
                uint32_t hostIndex = iThread % (sizeof(Utils::g_pInternetTestHosts) / sizeof(Utils::g_pInternetTestHosts[0]));
                threadParams[iThread].pHostName = Utils::g_pInternetTestHosts[hostIndex];
                threadParams[iThread].portNum = 443;
            }
        }

        NN_LOG("\n\n");
        Log("********** DownloadOnDiffThreads **********\n\n");

        // Downloads with same context, different threads
        m_pSslContext = new nn::ssl::Context();
        if(m_pSslContext == nullptr)
        {
            LogError("Failed to allocate ssl context\n");
            return false;
        }

        do
        {
            Log("Configuring SSL context...\n");
            if( !ConfigSslContext(m_pSslContext) )
            {
                isSuccess = false;
                break;
            }

            g_threadSuccess = true;

            Log("Creating threads...");
            for(uint32_t i = 0; i < ThreadCount; ++i)
            {
                g_pIndeces[i] = i;
                isSuccess = NetTest::CreateThread(&g_pThreads[i], SslAbuse::DownloadThreadFn, &threadParams[i], g_pThreadStack[i], StackSize);
                if( !isSuccess )
                {
                    LogError("Failed to create thread!\n\n");
                    isSuccess = false;
                    break;
                }
            }
            NN_LOG("Done\n");

            Log("Starting threads...");
            for(uint32_t i = 0; i < ThreadCount; ++i)
            {
                NetTest::StartThread(&g_pThreads[i]);
            }
            NN_LOG("Done\n");

            Log("Waiting on threads...");
            for(uint32_t i = 0; i < ThreadCount; ++i)
            {
                NetTest::WaitThread(&g_pThreads[i]);
            }
            NN_LOG("Done\n");

            Log("Destroying threads...");
            for(uint32_t i = 0; i < ThreadCount; ++i)
            {
                NetTest::DestroyThread(&g_pThreads[i]);
            }

            NN_LOG("Done\n");
        } while(NN_STATIC_CONDITION(false));

        DESTROY_SSL_OBJ(m_pSslContext, isSuccess, Log);

        if(g_threadSuccess == false)
        {
            LogError(" * Error in download thread\n\n");
            isSuccess = false;
        }

        return isSuccess;
    }

    // Read certificate files into buffers
    bool SslAbuse::ReadCertsIntoBuffers() const NN_NOEXCEPT
    {
        NN_LOG("\n\n");
        Log("********** ReadCertsIntoBuffers **********\n\n");

        // Read in all server certs
        for(uint32_t i = 0; i < ServerCertCount; ++i)
        {
            uint32_t fileSize = 0;
            std::string certPath = m_pServerCertPath;
            char pId[] = "00";
            certPath += "Server\\cert_";
            NETTEST_SNPRINTF(pId, sizeof(pId), "%u", i + 1);
            certPath += pId;
            certPath += ".pem";

            Log("FilePath: %s\n", certPath.c_str());
            char* pServerCert = g_pServerCertFiles[i].OpenAndRead(certPath, fileSize);
            if( !pServerCert )
            {
                LogError(" * Error: Failed to open and/or read cert file data!\n\n");
                return false;
            }
        }

        // Read in all client certs
        for(uint32_t iCert = 0; iCert < ClientCertCount; ++iCert)
        {
            uint32_t fileSize = 0;

            std::string certPath = m_pServerCertPath;
            char pId[] = "00";
            certPath += "Client\\cert_";
            NETTEST_SNPRINTF(pId, sizeof(pId), "%u", iCert + 1);
            certPath += pId;
            certPath += ".p12";

            Log("FilePath: %s\n", certPath.c_str());
            char* pClientCert = g_pClientCertFiles[iCert].OpenAndRead(certPath, fileSize);
            if( !pClientCert )
            {
                LogError(" * Error: Failed to open and/or read cert file data!\n\n");
                return false;
            }
        }

        // Read in all root CAs
        for(uint32_t i = 0; i < CertChainCount; ++i)
        {
            uint32_t fileSize = 0;
            std::string certPath = m_pServerCertPath;
            char pId[] = "00";
            certPath += "Chains\\Chain_";
            NETTEST_SNPRINTF(pId, sizeof(pId), "%u", i + 1);
            certPath += pId;
            certPath += "\\root_ca.pem.crt";

            Log("FilePath: %s\n", certPath.c_str());
            char* pServerCert = g_pRootCertFiles[i].OpenAndRead(certPath, fileSize);
            if( !pServerCert )
            {
                LogError(" * Error: Failed to open and/or read cert file data!\n\n");
                return false;
            }
        }

        // Read in all intermediate certs
        for(uint32_t iCert = 0; iCert < CertChainCount; ++iCert)
        {
            uint32_t fileSize = 0;

            std::string certPath = m_pServerCertPath;
            char pId[] = "00";
            certPath += "Chains\\Chain_";
            NETTEST_SNPRINTF(pId, sizeof(pId), "%u", iCert + 1);
            certPath += pId;
            certPath += "\\Intermediate_1\\intermediate_ca.pem.crt";

            Log("FilePath: %s\n", certPath.c_str());
            char* pServerCert = g_pIntermediateCertFiles[iCert].OpenAndRead(certPath, fileSize);
            if( !pServerCert )
            {
                LogError(" * Error: Failed to open and/or read cert file data!\n\n");
                return false;
            }
        }

        // Read in all server crls
        for(uint32_t iCrl = 0; iCrl < CertChainCount; ++iCrl)
        {
            uint32_t fileSize = 0;

            std::string crlPath = m_pServerCertPath;
            char pId[] = "00";
            crlPath += "Chains\\Chain_";
            NETTEST_SNPRINTF(pId, sizeof(pId), "%u", iCrl + 1);
            crlPath += pId;
            crlPath += "\\root.der.crl";

            Log("FilePath: %s\n", crlPath.c_str());
            char* pServerCrl = g_pRootCrlFiles[iCrl].OpenAndRead(crlPath, fileSize);
            if( !pServerCrl )
            {
                LogError(" * Error: Failed to open and/or read crl file data!\n\n");
                return false;
            }
        }

        return true;
    }

    //  - Creates Max num ssl Contexts. For each context:
    //    - Imports server certs ('ServerCertCount' unique certs but each context uses the same ServerCertCount'' certs.)
    //    - Imports 1 client cert
    bool SslAbuse::ImportCertsOnSameThread() const NN_NOEXCEPT
    {
        nn::ssl::Context pContextArray[MaxContextCount];

        NN_LOG("\n\n");
        Log("********** ImportCertsOnSameThread **********\n\n");

        for(uint32_t iContext = 0; iContext < MaxContextCount; ++iContext)
        {
            if( !ConfigSslContext(&pContextArray[iContext]) )
            {
                return false;
            }

            {
                int64_t fileSize = g_pClientCertFiles[iContext].GetCachedSize();

                nn::ssl::CertStoreId clientCertId;
                nn::Result result = pContextArray[iContext].ImportClientPki( &clientCertId,
                                                    g_pClientCertFiles[iContext].GetData(),
                                                    nullptr,
                                                    static_cast<uint32_t>(fileSize),
                                                    0);

                if( result.IsFailure() )
                {
                    LogError(" * Error: Failed to import client cert! Desc: %d\n\n", result.GetDescription());
                    return false;
                }
            }

            for(uint32_t i = 0; i < ServerCertCount; ++i)
            {
                int64_t fileSize = g_pServerCertFiles[i].GetCachedSize();

                nn::ssl::CertStoreId serverCertId;
                nn::Result result = pContextArray[iContext].ImportServerPki( &serverCertId,
                                        g_pServerCertFiles[i].GetData(),
                                        static_cast<uint32_t>(fileSize),
                                        nn::ssl::CertificateFormat_Pem);

                if( result.IsFailure() )
                {
                    LogError(" * Error: Failed to import server cert! Desc: %d\n\n", result.GetDescription());
                    return false;
                }
            }
        }

        for(uint32_t iContext = 0; iContext < MaxContextCount; ++iContext)
        {
            pContextArray[iContext].Destroy();
        }

        return true;
    }

    // Create and Destroy ssl connections a bunch of times.
    bool SslAbuse::CreateConnections(uint32_t connectionCount) const NN_NOEXCEPT
    {
        bool isSuccess = true;
        int socketFd = -1;
        nn::ssl::Connection* pSslConnection = nullptr;

        NN_LOG("\n\n");
        Log("********** CreateConnections **********\n\n");

        // Create connection loop
        for(uint32_t i = 0; i < connectionCount; ++i)
        {
            // Create socket with custom options
            socketFd = CreateSocketWithOptions();
            if( socketFd < 0 )
            {
                isSuccess = false;
                break;
            }

            // Connect socket to server
            if( !ConnectToServer(socketFd, m_pHostName, m_port) )
            {
                isSuccess = false;
                break;
            }

            pSslConnection = new nn::ssl::Connection();
            if(pSslConnection == nullptr)
            {
                LogError("Failed to allocate ssl connection\n");
                isSuccess = false;
                break;
            }

            if( !ConfigSslConnection(pSslConnection, m_pSslContext, socketFd, true, m_pHostName, g_pServerCertBuff, CertBuffLen, nn::ssl::Connection::VerifyOption::VerifyOption_None, nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None, false) )
            {
                socketFd = -1;
                isSuccess = false;
                break;
            }

            socketFd = -1;
            DESTROY_SSL_OBJ(pSslConnection, isSuccess, Log);
        }

        CHECK_AND_CLOSE_SOCKET(socketFd, isSuccess, LogError);
        DESTROY_SSL_OBJ(pSslConnection, isSuccess, LogError);

        return isSuccess;
    }

    // Does HTTP GET and download several times with same SSL connection
    bool SslAbuse::RepeatDownload(uint32_t downloadCount, bool isDoPostEnabled) NN_NOEXCEPT
    {
        NN_LOG("\n\n");
        Log("********** RepeatDownload **********\n\n");

        return Download(m_pSslContext, downloadCount, true, m_pHostName, m_port, "kb1", nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None, nn::ssl::Connection::VerifyOption::VerifyOption_None, false, isDoPostEnabled);
    }

    // Create and Destroy ssl context a bunch of times.
    bool SslAbuse::CreateDestroyContext(uint32_t loopCount) NN_NOEXCEPT
    {
        bool isSuccess = true;
        nn::ssl::Context* pSslContext = nullptr;

        NN_LOG("\n\n");
        Log("********** CreateDestroyContext **********\n\n");

        // Create context loop
        for(uint32_t i = 0; i < loopCount; ++i)
        {
            pSslContext = new nn::ssl::Context();
            if(pSslContext == nullptr)
            {
                LogError("Failed to allocate ssl context\n");
                isSuccess = false;
                goto out;
            }

            if( !ConfigSslContext(pSslContext) )
            {
                isSuccess = false;
                goto out;
            }

            if(pSslContext->Destroy().IsFailure())
            {
                LogError(" ERROR: Failed to Destroy SSL context!\n\n");
                isSuccess = false;
            }

            delete pSslContext;
            pSslContext = nullptr;
        }

out:
        DESTROY_SSL_OBJ(pSslContext, isSuccess, Log);

        return isSuccess;
    }

    // Download several times with the below:
    //  - Non-Bocking ssl connection
    //  - Session ticket cache enabled
    bool SslAbuse::NonBlockDownloadWithTicketCache(uint32_t downloadCount) NN_NOEXCEPT
    {
        NN_LOG("\n\n");
        Log("********** NonBlockDownloadWithTicketCache **********\n\n");

        // Non-Blocking download with session Ticket
        for(uint32_t i = 0; i < downloadCount; ++i)
        {
            bool isSuccess = Download(m_pSslContext, 1, false, m_pHostName, m_port, "kb2", nn::ssl::Connection::SessionCacheMode::SessionCacheMode_SessionTicket, nn::ssl::Connection::VerifyOption::VerifyOption_None, false, false);
            if(isSuccess == false)
            {
                return false;
            }
        }

        return true;
    }

    // Download several times with the below:
    //  - Bocking ssl connection
    //  - Session ID cache enabled
    bool SslAbuse::BlockingDownloadWithIDCache(uint32_t downloadCount) NN_NOEXCEPT
    {
        NN_LOG("\n\n");
        Log("********** BlockingDownloadWithIDCache **********\n\n");

        // Non-Blocking download with session Id
        for(uint32_t i = 0; i < downloadCount; ++i)
        {
            bool isSuccess = Download(m_pSslContext, 1, true, m_pHostName, m_port, "kb2", nn::ssl::Connection::SessionCacheMode::SessionCacheMode_SessionId, nn::ssl::Connection::VerifyOption::VerifyOption_None, false, false);
            if(isSuccess == false)
            {
                return false;
            }
        }

        return true;
    }

    // Blocking, with handshake, no download, different hosts.
    bool SslAbuse::HandshakeWithDiffHosts(uint32_t handshakeCount) NN_NOEXCEPT
    {
        bool isSuccess = true;

        NN_LOG("\n\n");
        Log("********** HandshakeWithDiffHosts **********\n\n");

        // Blocking, with handshake, no download, different hosts.
        for(uint32_t i = 0; i < handshakeCount; ++i)
        {
            uint32_t hostIndex = i % (sizeof(Utils::g_pInternetTestHosts) / sizeof(Utils::g_pInternetTestHosts[0]));
            isSuccess &= Download(m_pSslContext, 0, true, Utils::g_pInternetTestHosts[hostIndex], 443, "", nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None, nn::ssl::Connection::VerifyOption::VerifyOption_All, false, false);
        }

        return isSuccess;
    }

    bool SslAbuse::HandshakeWithServerCertDetailsBlocking(uint32_t handshakeCount) NN_NOEXCEPT
    {
        bool isSuccess = true;

        NN_LOG("\n\n");
        Log("********** HandshakeWithServerCertDetailsBlocking **********\n\n");

        // Blocking, with handshake, no download, different hosts.
        for(uint32_t i = 0; i < handshakeCount; ++i)
        {
            uint32_t hostIndex = i % (sizeof(Utils::g_pInternetTestHosts) / sizeof(Utils::g_pInternetTestHosts[0]));
            isSuccess &= Download(m_pSslContext, 0, true, Utils::g_pInternetTestHosts[hostIndex], 443, "", nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None, nn::ssl::Connection::VerifyOption::VerifyOption_All, true, false);
        }

        return isSuccess;
    }

    bool SslAbuse::HandshakeWithServerCertDetailsNonBlocking(uint32_t handshakeCount) NN_NOEXCEPT
    {
        bool isSuccess = true;

        NN_LOG("\n\n");
        Log("********** HandshakeWithServerCertDetailsNonBlocking **********\n\n");

        // Non-Blocking, with handshake, no download, different hosts.
        for(uint32_t i = 0; i < handshakeCount; ++i)
        {
            uint32_t hostIndex = i % (sizeof(Utils::g_pInternetTestHosts) / sizeof(Utils::g_pInternetTestHosts[0]));
            isSuccess &= Download(m_pSslContext, 0, false, Utils::g_pInternetTestHosts[hostIndex], 443, "", nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None, nn::ssl::Connection::VerifyOption::VerifyOption_All, true, false);
        }

        return isSuccess;
    }

    bool SslAbuse::HandshakeWithServerCertDetailsSessionId(uint32_t handshakeCount) NN_NOEXCEPT
    {
        bool isSuccess = true;

        NN_LOG("\n\n");
        Log("********** HandshakeWithServerCertDetailsSessionId **********\n\n");

        // Non-Blocking, with handshake, no download, different hosts.
        for(uint32_t i = 0; i < handshakeCount; ++i)
        {
            uint32_t hostIndex = i % (sizeof(Utils::g_pInternetTestHosts) / sizeof(Utils::g_pInternetTestHosts[0]));
            isSuccess &= Download(m_pSslContext, 0, false, Utils::g_pInternetTestHosts[hostIndex], 443, "", nn::ssl::Connection::SessionCacheMode::SessionCacheMode_SessionId, nn::ssl::Connection::VerifyOption::VerifyOption_All, true, false);
        }

        return isSuccess;
    }

    bool SslAbuse::HandshakeWithServerCertDetailsSessionTicket(uint32_t handshakeCount) NN_NOEXCEPT
    {
        bool isSuccess = true;

        NN_LOG("\n\n");
        Log("********** HandshakeWithServerCertDetailsSessionTicket **********\n\n");

        // Non-Blocking, with handshake, no download, different hosts.
        for(uint32_t i = 0; i < handshakeCount; ++i)
        {
            uint32_t hostIndex = i % (sizeof(Utils::g_pInternetTestHosts) / sizeof(Utils::g_pInternetTestHosts[0]));
            isSuccess &= Download(m_pSslContext, 0, false, Utils::g_pInternetTestHosts[hostIndex], 443, "", nn::ssl::Connection::SessionCacheMode::SessionCacheMode_SessionTicket, nn::ssl::Connection::VerifyOption::VerifyOption_All, true, false);
        }

        return isSuccess;
    }

    bool SslAbuse::ImportChainAndCrl(nn::ssl::Context* pSslContext, uint32_t chainIndex) NN_NOEXCEPT
    {
        if(chainIndex >= CertChainCount)
        {
            LogError(" * Error: Cert chain index out of range!\n\n");
            return false;
        }

        Log("Cert: %d\n", chainIndex);
        int64_t certSize = g_pRootCertFiles[chainIndex].GetCachedSize();

        nn::ssl::CertStoreId serverCertId;
        nn::Result result = pSslContext->ImportServerPki( &serverCertId,
                                g_pRootCertFiles[chainIndex].GetData(),
                                static_cast<uint32_t>(certSize),
                                nn::ssl::CertificateFormat_Pem);

        if( result.IsFailure() )
        {
            LogError(" * Error: Failed to import server cert! Desc: %d\n\n", result.GetDescription());
            return false;
        }

        certSize = g_pIntermediateCertFiles[chainIndex].GetCachedSize();

        nn::ssl::CertStoreId intermediateCertId;
        result = pSslContext->ImportServerPki( &intermediateCertId,
                                g_pIntermediateCertFiles[chainIndex].GetData(),
                                static_cast<uint32_t>(certSize),
                                nn::ssl::CertificateFormat_Pem);

        if( result.IsFailure() )
        {
            LogError(" * Error: Failed to import intermediate cert! Desc: %d\n\n", result.GetDescription());
            return false;
        }

        int64_t crlSize = g_pRootCrlFiles[chainIndex].GetCachedSize();

        nn::ssl::CertStoreId crlId;
        result = pSslContext->ImportCrl(&crlId,
                                        g_pRootCrlFiles[chainIndex].GetData(),
                                        static_cast<uint32_t>(crlSize));
        if(result.IsFailure())
        {
            LogError("Failed to import crl data! Desc: %d\n\n", result.GetDescription());
            return false;
        }

        return true;
    }

    // Does the following in a loop 'importCount' times
    //  - Creates an ssl context
    //  - Imports different root CAs, intermediate CAs and root CRLs starting at 'chainIndex' and incrimenting 'chainCount' times
    //  - Creates an ssl connection.
    //  - DoHandshake() is performed.
    //  - Destroy ssl connection.
    //  - Destroy ssl context.
    bool SslAbuse::ImportCrl(uint32_t importCount, uint32_t chainIndex, uint32_t chainCount, bool doHandshake) NN_NOEXCEPT
    {
        bool isSuccess = true;
        int socketFd = -1;
        nn::ssl::Context* pSslContext = nullptr;
        nn::ssl::Connection* pSslConnection = nullptr;

        if(chainIndex + chainCount > CertChainCount)
        {
            LogError("\n * Error: chainIndex out of range!\n\n");
            return false;
        }

        for(uint32_t i = 0; i < importCount && isSuccess == true; ++i)
        {
            char pHostName[NatfHostnameBufLen] = {0};
            Log("ImportCrl: %d\n", i);

            pSslContext = new nn::ssl::Context();
            if(pSslContext == nullptr)
            {
                LogError("Failed to allocate ssl context\n");
                isSuccess = false;
                break;
            }

            if( !ConfigSslContext(pSslContext) )
            {
                isSuccess = false;
                break;
            }

            for(uint32_t iCert = chainIndex; iCert < chainIndex + chainCount && isSuccess == true; ++iCert)
            {
                if( false == ImportChainAndCrl(pSslContext, iCert) )
                {
                    isSuccess = false;
                    break;
                }
            }

            if(doHandshake)
            {
                // Create socket with custom options
                socketFd = CreateSocketWithOptions();
                if( socketFd < 0 )
                {
                    isSuccess = false;
                    break;
                }

                uint32_t chainId = (i % CertChainCount) + 1;
                NETTEST_SNPRINTF(pHostName, sizeof(pHostName), "natf_%d.com", chainId);

                Log("HostName: %s, Port: %d\n", pHostName, m_port + static_cast<uint16_t>(chainId));

                // Connect socket to server
                //  Use m_pHostName because it is already resolved to an IP
                if( !ConnectToServer(socketFd, m_pHostName, m_port + static_cast<uint16_t>(chainId)) )
                {
                    isSuccess = false;
                    break;
                }

                pSslConnection = new nn::ssl::Connection();
                if(pSslConnection == nullptr)
                {
                    LogError("Failed to allocate ssl connection\n");
                    isSuccess = false;
                    break;
                }

                if( ConfigSslConnection(pSslConnection, pSslContext, socketFd, true, pHostName, nullptr, 0, nn::ssl::Connection::VerifyOption::VerifyOption_All, nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None, false) == false )
                {
                    socketFd = -1;
                    isSuccess = false;
                    break;
                }

                socketFd = -1;

                isSuccess = NormalHandshake(pSslConnection, nn::ssl::ResultVerifyCertFailed(), nn::ssl::ResultSslErrorUnkownCa());
                if( false == isSuccess )
                {
                    LogError(" * ERROR: Handshake failure!\n\n");
                    break;
                }

                CHECK_AND_CLOSE_SOCKET(socketFd, isSuccess, LogError);
                DESTROY_SSL_OBJ(pSslConnection, isSuccess, LogError);
            }

            DESTROY_SSL_OBJ(pSslContext, isSuccess, LogError);
        }

        CHECK_AND_CLOSE_SOCKET(socketFd, isSuccess, LogError);
        DESTROY_SSL_OBJ(pSslConnection, isSuccess, LogError);
        DESTROY_SSL_OBJ(pSslContext, isSuccess, LogError);

        return isSuccess;
    }

    //  - Creates 8 different contexts
    //  - On each context, imports a different root CA, intermediate CA and root CRL
    //  - If doHandshake is true, the following is also performed:
    //      - Creates an ssl connection.
    //      - DoHandshake() is performed.
    //      - Destroy ssl connection.
    //  - Destroy ssl context.
    bool SslAbuse::ImportCrlOnDiffCtx(uint32_t importCount, bool doHandshake) NN_NOEXCEPT
    {
        bool isSuccess = true;

        NN_LOG("\n\n");
        Log("********** ImportCrlOnDiffCtx **********\n\n");

        for(uint32_t i = 0; i < importCount && isSuccess == true; ++i)
        {
            nn::ssl::Context* pSslContexts[CertChainCount] = {nullptr};
            nn::ssl::Connection* pSslConnections[CertChainCount] = {nullptr};
            int socketFds[CertChainCount];

            // Init
            for(uint32_t iCtx = 0; iCtx < CertChainCount; ++iCtx)
            {
                socketFds[iCtx] = -1;
            }

            Log("ImportCrl: %d\n", i);

            // Create Contexts and import certs
            for(uint32_t iCtx = 0; iCtx < CertChainCount; ++iCtx)
            {
                pSslContexts[iCtx] = new nn::ssl::Context();
                if(pSslContexts[iCtx] == nullptr)
                {
                    LogError("Failed to allocate ssl context\n");
                    isSuccess = false;
                    break;
                }

                if( !ConfigSslContext(pSslContexts[iCtx]) )
                {
                    isSuccess = false;
                    break;
                }

                if( false == ImportChainAndCrl(pSslContexts[iCtx], iCtx) )
                {
                    isSuccess = false;
                    break;
                }
            }

            if(doHandshake)
            {
                // Create connections
                for(uint32_t iCtx = 0; iCtx < CertChainCount && isSuccess == true; ++iCtx)
                {
                    char pHostName[NatfHostnameBufLen] = {0};

                    // Create socket with custom options
                    socketFds[iCtx] = CreateSocketWithOptions();
                    if( socketFds[iCtx] < 0 )
                    {
                        isSuccess = false;
                        break;
                    }

                    uint32_t chainId = iCtx + 1;
                    NETTEST_SNPRINTF(pHostName, sizeof(pHostName), "natf_%d.com", chainId);

                    // Connect socket to server
                    //  Use m_pHostName because it is already resolved to an IP
                    if( !ConnectToServer(socketFds[iCtx], m_pHostName, m_port + static_cast<uint16_t>(chainId)) )
                    {
                        isSuccess = false;
                        break;
                    }

                    pSslConnections[iCtx] = new nn::ssl::Connection();
                    if(pSslConnections[iCtx] == nullptr)
                    {
                        LogError("Failed to allocate ssl connection\n");
                        isSuccess = false;
                        break;
                    }

                    if( ConfigSslConnection(pSslConnections[iCtx], pSslContexts[iCtx], socketFds[iCtx], true, pHostName, nullptr, 0, nn::ssl::Connection::VerifyOption::VerifyOption_All, nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None, false) == false )
                    {
                        isSuccess = false;
                        socketFds[iCtx] = -1;
                        break;
                    }

                    socketFds[iCtx] = -1;
                }

                // DoHandshake
                for(uint32_t iCtx = 0; iCtx < CertChainCount && isSuccess == true; ++iCtx)
                {
                    isSuccess = NormalHandshake(pSslConnections[iCtx], nn::ssl::ResultVerifyCertFailed(), nn::ssl::ResultSslErrorUnkownCa());
                }
            }

            // Cleanup
            for(uint32_t iCtx = 0; iCtx < CertChainCount; ++iCtx)
            {
                CHECK_AND_CLOSE_SOCKET(socketFds[iCtx], isSuccess, LogError);
                DESTROY_SSL_OBJ(pSslConnections[iCtx], isSuccess, LogError);
                DESTROY_SSL_OBJ(pSslContexts[iCtx], isSuccess, LogError);
            }
        }

        return isSuccess;
    }

    void SslAbuse::ImportCrlThreadFn(void* pParam) NN_NOEXCEPT
    {
        CrlThreadParams* pParams = reinterpret_cast<CrlThreadParams*>(pParam);
        uint32_t chainIndex = pParams->index % CertChainCount;

        if(pParams->pThis->ImportCrl(g_importCrlLoopOnThreadCount, chainIndex, 1, g_doHandshakeOnThread) == false)
        {
            g_threadSuccess = false;
        }
    }

    // Creates 8 different threads which do the following:
    //  - Creates an ssl context
    //  - On each context, imports a different root CA, intermediate CA and root CRL
    //  - If g_doHandshakeOnThread is true, the following is also performed:
    //      - Creates an ssl connection.
    //      - DoHandshake() is performed.
    //      - Destroy ssl connection.
    //  - Destroy ssl context.
    bool SslAbuse::ImportCrlOnDiffThreads(uint32_t loopCount) NN_NOEXCEPT
    {
        g_threadSuccess = true;
        g_importCrlLoopOnThreadCount = loopCount;

        CrlThreadParams threadParams[ThreadCount];

        NN_LOG("\n\n");
        Log("********** ImportCrlOnDiffThreads **********\n\n");

        for(uint32_t iThread = 0; iThread < ThreadCount; ++iThread)
        {
            threadParams[iThread].pThis = this;
            threadParams[iThread].index = static_cast<uint8_t>(iThread);
        }

        Log("Creating threads...");
        for(uint32_t i = 0; i < ThreadCount; ++i)
        {
            g_pIndeces[i] = i;
            bool isSuccess = NetTest::CreateThread(&g_pThreads[i], SslAbuse::ImportCrlThreadFn, &threadParams[i], g_pThreadStack[i], StackSize);
            if( !isSuccess )
            {
                LogError("Failed to create thread!\n\n");

                return false;
            }
        }
        NN_LOG("Done\n");

        Log("Starting threads...");
        for(uint32_t i = 0; i < ThreadCount; ++i)
        {
            NetTest::StartThread(&g_pThreads[i]);
        }
        NN_LOG("Done\n");

        Log("Waiting on threads...");
        for(uint32_t i = 0; i < ThreadCount; ++i)
        {
            NetTest::WaitThread(&g_pThreads[i]);
        }
        NN_LOG("Done\n");

        Log("Destroying on threads...");
        for(uint32_t i = 0; i < ThreadCount; ++i)
        {
            NetTest::DestroyThread(&g_pThreads[i]);
        }
        NN_LOG("Done\n");

        if(g_threadSuccess == false)
        {
            LogError(" * Error in download thread\n\n");
            return false;
        }

        return true;
    }

    // Run
    bool SslAbuse::Run() NN_NOEXCEPT
    {
        bool isSuccess = true;
        nn::Result result;

        // Create and Destroy ssl context a bunch of times.
        isSuccess &= CreateDestroyContext(CreateDestroyCtxCount);

        m_pSslContext = new nn::ssl::Context();
        if(m_pSslContext == nullptr)
        {
            LogError("Failed to allocate ssl context\n");
            isSuccess = false;
            goto out;
        }

        if( !ConfigSslContext(m_pSslContext) )
        {
            isSuccess = false;
            goto out;
        }

        // Create and Destroy ssl connections a bunch of times.
        isSuccess &= CreateConnections(CreateConnectionCount);

        // Blocking, with handshake, no download, different hosts.
        isSuccess &= HandshakeWithDiffHosts(HandshakeDiffHostsCount);

        // Download several times with the below:
        //  - Bocking ssl connection
        //  - Session ID cache enabled
        isSuccess &= BlockingDownloadWithIDCache(BlockingIDCacheDownloadCount);

        // Download several times with the below:
        //  - Non-Bocking ssl connection
        //  - Session ticket cache enabled
        isSuccess &= NonBlockDownloadWithTicketCache(NonBlockingTicketCacheDownloadCount);

        DESTROY_SSL_OBJ(m_pSslContext, isSuccess, Log);

        // Read certificate files into buffers
        isSuccess = ReadCertsIntoBuffers();
        if(isSuccess == false)
        {
            goto out;
        }

        //  - Creates 8 ssl Contexts. For each context:
        //    - Imports 'ServerCertCount' server certs('ServerCertCount' unique certs but each context uses the same 'ServerCertCount' certs.)
        //    - Imports 1 client cert
        isSuccess &= ImportCertsOnSameThread();

        // Creates 8 threads. Each thread does the following:
        //  - Creates an ssl Context
        //  - Imports 'ServerCertCount' server certs('ServerCertCount' unique certs but each thread uses the same 'ServerCertCount' certs.)
        //  - Imports 1 client cert
        isSuccess &= ImportCertsOnDiffThreads();

        // Create a context, Creates 8 threads. Each thread does the following:
        //  - Creates ssl connection(each thread from the same context)
        //  - Does http GET to download 1mb
        isSuccess &= DownloadOnDiffThreads(false, nn::ssl::Connection::VerifyOption::VerifyOption_None, true, false);

        // Create a context, Creates 8 threads. Each thread does the following:
        //  - Creates ssl connection(each thread from the same context)
        //  - Does http POST to download 1mb
        isSuccess &= DownloadOnDiffThreads(false, nn::ssl::Connection::VerifyOption::VerifyOption_None, true, true);

        m_pSslContext = new nn::ssl::Context();
        if(m_pSslContext == nullptr)
        {
            LogError("Failed to allocate ssl context\n");
            isSuccess = false;
            goto out;
        }

        if( !ConfigSslContext(m_pSslContext) )
        {
            isSuccess = false;
            goto out;
        }

        // Does HTTP GET and download several times with same SSL connection
        isSuccess &= RepeatDownload(DownloadSameConnectionCount, false);

        // Does HTTP POST and several times with same SSL connection
        isSuccess &= RepeatDownload(DownloadSameConnectionCount, true);

        // Download several times with the below:
        //  - Bocking ssl connection
        //  - Session cache disabled
        //  - Get cert chain enabled
        isSuccess &= HandshakeWithServerCertDetailsBlocking(DownloadWithServerCertDetailsBlockingCount);

        // Download several times with the below:
        //  - Non-bocking ssl connection
        //  - Session cache disabled
        //  - Get cert chain enabled
        isSuccess &= HandshakeWithServerCertDetailsNonBlocking(DownloadWithServerCertDetailsNonBlockingCount);

        // Download several times with the below:
        //  - Non-bocking ssl connection
        //  - Session Id enabled
        //  - Get cert chain enabled
        isSuccess &= HandshakeWithServerCertDetailsSessionId(DownloadWithServerCertDetailsSessionIdCount);

        // Download several times with the below:
        //  - Non-bocking ssl connection
        //  - Session Ticket enabled
        //  - Get cert chain enabled
        isSuccess &= HandshakeWithServerCertDetailsSessionTicket(DownloadWithServerCertDetailsSessionTicketCount);

        DESTROY_SSL_OBJ(m_pSslContext, isSuccess, Log);

        // Create a context, Creates 8 threads. Each thread does the following:
        //  - Creates ssl connection(each thread from the same context)
        //  - Does handshake with get cert chain enabled
        //  - Does http GET to download 1mb
        isSuccess &= DownloadOnDiffThreads(true, nn::ssl::Connection::VerifyOption::VerifyOption_All, false, false);

        // Does the following in a loop:
        //  - Creates an ssl context
        //  - Imports 8 different root CAs, intermediate CAs and root CRLs
        //  - Destroy ssl context.
        isSuccess &= ImportCrl(ImportCrlSingleContextNoHandshakeCount, 0, CertChainCount, false);

        // Does the following in a loop:
        //  - Creates 8 ssl contexts
        //  - On each context, imports a different root CA, intermediate CA and root CRL
        //  - Destroy ssl context.
        isSuccess &= ImportCrlOnDiffCtx(ImportCrlMultipleContextNoHandshakeCount, false);

        // Creates 8 different threads which do the following:
        //  - Creates an ssl context
        //  - On each context, imports a different root CA, intermediate CA and root CRL
        //  - Destroy ssl context.
        g_doHandshakeOnThread = false;
        isSuccess &= ImportCrlOnDiffThreads(ImportCrlMultipleThreadNoHandshakeCount);

        // Does the following in a loop:
        //  - Creates an ssl context
        //  - Imports 8 different root CAs, intermediate CAs and root CRLs
        //  - Creates an ssl connection.
        //  - DoHandshake() is performed.
        //  - Destroy ssl connection.
        //  - Destroy ssl context.
        isSuccess &= ImportCrl(ImportCrlSingleContextWithHandshakeCount, 0, CertChainCount, true);

        // Does the following in a loop:
        //  - Creates 8 ssl contexts
        //  - On each context, imports a different root CA, intermediate CA and root CRL
        //  - Creates an ssl connection.
        //  - DoHandshake() is performed.
        //  - Destroy ssl connection.
        //  - Destroy ssl context.
        isSuccess &= ImportCrlOnDiffCtx(ImportCrlMultipleContextWithHandshakeCount, true);

        // Creates 8 different threads which do the following:
        //  - Creates an ssl context
        //  - On each context, imports a different root CA, intermediate CA and root CRL
        //  - Creates an ssl connection.
        //  - DoHandshake() is performed.
        //  - Destroy ssl connection.
        //  - Destroy ssl context.
        g_doHandshakeOnThread = true;
        isSuccess &= ImportCrlOnDiffThreads(ImportCrlMultipleThreadWithHandshakeCount);


    out:

        DESTROY_SSL_OBJ(m_pSslContext, isSuccess, Log);

        for(uint32_t i = 0; i < ServerCertCount; ++i)
        {
            g_pServerCertFiles[i].Close();
        }

        for(uint32_t i = 0; i < ClientCertCount; ++i)
        {
            g_pClientCertFiles[i].Close();
        }

        for(uint32_t i = 0; i < CertChainCount; ++i)
        {
            g_pRootCertFiles[i].Close();
            g_pRootCrlFiles[i].Close();
            g_pIntermediateCertFiles[i].Close();
        }

        return isSuccess;
    } // NOLINT(impl/function_size)

    // GetName
    const char* SslAbuse::GetName() const NN_NOEXCEPT
    { return "SslAbuse"; }
}
}
