﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <stdint.h>
#include "Utils/md5.h"
#include "NetTest_Port.h"

namespace nn {
namespace util {
struct Uuid;
}}

namespace NATF {
namespace Utils {

class Parser
{
public:
    explicit Parser(const char* pName) NN_NOEXCEPT;

    virtual ~Parser() NN_NOEXCEPT {}

    virtual bool Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT = 0;

    virtual bool SetDefaultValue() NN_NOEXCEPT
    { return false; };

    const char* GetName() const NN_NOEXCEPT
    { return m_pName; }

protected:
    bool ParseInt32(int& intOut, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT;
    bool ParseFloat(float& fltOut, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT;
    bool ParseIpAddress(char* pBufferOut, uint32_t bufferSize, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT;
    bool ParseMd5Hash(MD5Hash::Result& hashOut, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT;

private:
    friend class ParserGroup;

    bool m_isUsed;
    Parser* m_pNext;
    const char* m_pName;
};

class ParserGroup
{
public:
    ParserGroup() NN_NOEXCEPT;
    ~ParserGroup() NN_NOEXCEPT;

    template<class PARSER>
    void AddParser(const PARSER& parser) NN_NOEXCEPT
    {
        Parser* pNewParser = new PARSER(parser);
        if( nullptr == pNewParser )
        {
            NATF_LOG(" * FATAL ERROR: Failed to allocate memory in arserGroup<Params>::AddParser\n\n");
            NETTEST_ABORT();
        }

        pNewParser->m_pNext = m_pHead;
        m_pHead = pNewParser;
    }

    bool Parse(int argC, const char * const * argV) NN_NOEXCEPT;

private:

    Parser* m_pHead;
};

class IpParser : public Utils::Parser
{
public:
    IpParser(const char* pParamName, const char* pDefaultIp, char* pIpStringBuf, uint32_t bufferLen) NN_NOEXCEPT;

    virtual bool Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT NN_OVERRIDE;
    virtual bool SetDefaultValue() NN_NOEXCEPT NN_OVERRIDE;

private:
    const char* m_pDefaultIp;
    char* m_pIpStringBuf;
    uint32_t m_bufferLen;
};

class Int32Parser : public Utils::Parser
{
public:
    Int32Parser(const char* pParamName, const int32_t* pDefaultVal, int32_t& outVal) NN_NOEXCEPT;

    virtual bool Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT NN_OVERRIDE;
    virtual bool SetDefaultValue() NN_NOEXCEPT NN_OVERRIDE;

private:
    const int32_t* m_pDefaultVal;
    int32_t& m_outVal;
};

class UInt32Parser : public Utils::Parser
{
public:
    UInt32Parser(const char* pParamName, const uint32_t* pDefaultVal, uint32_t& outVal) NN_NOEXCEPT;

    virtual bool Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT NN_OVERRIDE;
    virtual bool SetDefaultValue() NN_NOEXCEPT NN_OVERRIDE;

private:
    const uint32_t* m_pDefaultVal;
    uint32_t& m_outVal;
};

class Int16Parser : public Utils::Parser
{
public:
    Int16Parser(const char* pParamName, const int16_t* pDefaultVal, int16_t& outVal) NN_NOEXCEPT;

    virtual bool Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT NN_OVERRIDE;
    virtual bool SetDefaultValue() NN_NOEXCEPT NN_OVERRIDE;

private:
    const int16_t* m_pDefaultVal;
    int16_t& m_outVal;
};

class UInt16Parser : public Utils::Parser
{
public:
    UInt16Parser(const char* pParamName, const uint16_t* pDefaultVal, uint16_t& outVal) NN_NOEXCEPT;

    virtual bool Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT NN_OVERRIDE;
    virtual bool SetDefaultValue() NN_NOEXCEPT NN_OVERRIDE;

private:
    const uint16_t* m_pDefaultVal;
    uint16_t& m_outVal;
};

class Md5HashParser : public Utils::Parser
{
public:
    Md5HashParser(const char* pParamName, const MD5Hash::Result* pDefaultVal, MD5Hash::Result& outVal) NN_NOEXCEPT;

    virtual bool Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT NN_OVERRIDE;
    virtual bool SetDefaultValue() NN_NOEXCEPT NN_OVERRIDE;

private:
    const MD5Hash::Result* m_pDefaultVal;
    MD5Hash::Result& m_outVal;
};

class StringParser : public Utils::Parser
{
public:
    StringParser(const char* pParamName, const char* pDefaultVal, char* pOutBuffer, uint32_t bufferLen) NN_NOEXCEPT;

    virtual bool Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT NN_OVERRIDE;
    virtual bool SetDefaultValue() NN_NOEXCEPT NN_OVERRIDE;

private:
    const char* m_pDefaultVal;
    char* m_pOutBuffer;
    uint32_t m_bufferLen;
};

class UuidParser : public Utils::Parser
{
public:
    UuidParser(const char* pParamName, const nn::util::Uuid* pDefaultVal, nn::util::Uuid& outVal) NN_NOEXCEPT;

    virtual bool Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT NN_OVERRIDE;
    virtual bool SetDefaultValue() NN_NOEXCEPT NN_OVERRIDE;

private:
    const nn::util::Uuid* m_pDefaultVal;
    nn::util::Uuid& m_outVal;
};

class BoolParser : public Utils::Parser
{
public:
    BoolParser(const char* pParamName, const bool* pDefaultVal, bool& outVal) NN_NOEXCEPT;

    virtual bool Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT NN_OVERRIDE;
    virtual bool SetDefaultValue() NN_NOEXCEPT NN_OVERRIDE;

private:
    const bool* m_pDefaultVal;
    bool& m_outVal;
};

}} // Namespaces NATF::Utils
