﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/os/os_Event.h>
#include <cctype>
#include <nnt/nnt_Argument.h>

#include <nn/TargetConfigs/build_Base.h>

namespace NetTest {

#define NETTEST_DECLARE_THREAD_FUNC(THREAD_NAME, PARAM_NAME) void THREAD_NAME(void* PARAM_NAME)
#define NETTEST_THREAD_RETURN return
#define NETTEST_MAIN_RETURN() return

/* Types */
typedef nn::os::ThreadType            Thread;
typedef nn::os::MutexType             Mutex;
typedef nn::os::ConditionVariableType Condition;
typedef nn::os::EventType             Event;
typedef nn::TimeSpan                  Time;
typedef nn::os::Tick                  Tick;

typedef void (*ThreadFn)(void*);

/* APIs */
inline void InitEvent( Event* pEvent )
{ nn::os::InitializeEvent(pEvent, false, nn::os::EventClearMode::EventClearMode_AutoClear); }

inline void SignalEvent(Event* pEvent)
{ nn::os::SignalEvent(pEvent); }

inline void ClearEvent(Event* pEvent)
{ nn::os::ClearEvent(pEvent); }

inline void FinalizeEvent(Event* pEvent)
{ nn::os::FinalizeEvent(pEvent); }

inline void WaitEvent(Event* pEvent)
{ nn::os::WaitEvent(pEvent); }

inline bool TimedWaitEvent(Event* pEvent, Time time)
{ return nn::os::TimedWaitEvent(pEvent, time); }

inline void InitMutex( Mutex* pMutex )
{ nn::os::InitializeMutex(pMutex, false, 1); }

inline void LockMutex(Mutex* pMutex)
{ nn::os::LockMutex(pMutex); }

inline void UnlockMutex(Mutex* pMutex)
{ nn::os::UnlockMutex(pMutex); }

inline bool CreateThread( Thread* pThread, ThreadFn pThreadFn, void* pParam, unsigned char* pStack, uint32_t stackSize)
{ return nn::os::CreateThread(pThread, pThreadFn, pParam, pStack, stackSize, nn::os::DefaultThreadPriority, 1).IsSuccess(); }

inline void YieldThread()
{ nn::os::YieldThread(); }

inline void StartThread(Thread* pThread)
{ nn::os::StartThread(pThread); }

inline void WaitThread(Thread* pThread)
{ nn::os::WaitThread(pThread); }

inline void DestroyThread(Thread* pThread)
{ nn::os::DestroyThread(pThread); }

inline Tick GetTick()
{ return nn::os::GetSystemTick(); }

inline Time TickToTime(Tick tick)
{ return nn::os::ConvertToTimeSpan(tick); }

inline Tick TimeToTick(Time time)
{ return nn::os::ConvertToTick(time); }

inline Time TimeFromMs(uint32_t timeMs)
{ return nn::TimeSpan::FromMilliSeconds(timeMs); }

inline Time TimeFromSec(uint32_t timeSec)
{ return nn::TimeSpan::FromSeconds(timeSec); }

inline void SleepMs(uint32_t timeMs)
{ nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(timeMs) ); }

inline uint32_t GetRandom()
{
    static uint32_t prev = 0;
    return (prev = (prev ^ (uint32_t)(nn::os::GetSystemTick().GetInt64Value())));
}

inline void StrUpr(char* pStr)
{
    char* pCur = pStr;
    while( *pCur )
    {
        *pCur = (char)toupper(*pCur);
        ++pCur;
    }
}

/* Workaround while SIGLONTD-4741 is WIP*/
#define NETTEST_GET_ARGS(ARGC, ARGV) \
do { \
    ARGV = (const char * const *)nnt::GetHostArgv(); \
    ARGC = 0; \
    { \
        char lastArg[] = "--NULL"; \
        for(int iArg = 0; nullptr != ARGV[iArg]; ++iArg) \
        { \
            if(0 == strncmp(ARGV[iArg], lastArg, strlen(lastArg))) \
            { \
                break; \
            } \
            ++ARGC; \
        } \
    } \
} while( NN_STATIC_CONDITION(false) )

} // namespace NetTest
