﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/os/os_ThreadCommon.h>
#include <nn/os/os_MutexApi.h>

#include <list>
#include <map>

#include "Complex/testNet_UnitCommon.h"
#include "Complex/testNet_UnitCommonSocket.h"
#include "Complex/testNet_UnitCommonInterfaces.h"

static const int API_UNIT_STRING_MAX = 128;

namespace NATF {
namespace API {

// TODO: use interfaces
class SelectUnitData;
class SelectUnitServer;
class SelectUnitClient;

/**
 * @brief a simple C++ trace logger
 */
class TraceLogger
{
public:
    TraceLogger(const char* function, const char* file, unsigned line, const char* fmt, ...);
    ~TraceLogger();
};
#define UNIT_TEST_TRACE(fmt, ...) { NATF::API::TraceLogger ___traceLogger(__FUNCTION__, __FILE__, __LINE__, fmt, ## __VA_ARGS__); }

/**
 *
 */
template <typename ReferenceCountObjectType=IReferenceCountObject>
class AutoReleaseObject
{
private:
    ReferenceCountObjectType* m_ReferenceCountObject;

public:
    AutoReleaseObject() :
        m_ReferenceCountObject(NULL)
    {
    };

    AutoReleaseObject(ReferenceCountObjectType* object, bool initialAdd=true) :
        m_ReferenceCountObject(object)
    {
        if (true == initialAdd && NULL != m_ReferenceCountObject)
        {
            m_ReferenceCountObject->addReference();
        };
    };

    AutoReleaseObject(const AutoReleaseObject& o)
    {
        if (NULL != m_ReferenceCountObject)
        {
            m_ReferenceCountObject->releaseReference();
            m_ReferenceCountObject = NULL;
        };

        if (NULL != (m_ReferenceCountObject = o.m_ReferenceCountObject))
        {
            m_ReferenceCountObject->addReference();
        };
    };

    AutoReleaseObject& operator=(const AutoReleaseObject& o)
    {
        if (NULL != m_ReferenceCountObject)
        {
            m_ReferenceCountObject->releaseReference();
            m_ReferenceCountObject = NULL;
        };

        if (NULL != (m_ReferenceCountObject = o.m_ReferenceCountObject))
        {
            m_ReferenceCountObject->addReference();
        };

        return *this;
    };

    AutoReleaseObject& operator=(const ReferenceCountObjectType* o)
    {
        if (NULL != m_ReferenceCountObject)
        {
            m_ReferenceCountObject->releaseReference();
            m_ReferenceCountObject = NULL;
        };

        m_ReferenceCountObject = const_cast<ReferenceCountObjectType*>(o);
        return *this;
    };

    virtual ~AutoReleaseObject()
    {
        if (NULL != m_ReferenceCountObject)
        {
            m_ReferenceCountObject->releaseReference();
            m_ReferenceCountObject = NULL;
        };
    };

    ReferenceCountObjectType& operator*()
    {
        NN_ASSERT(NULL != m_ReferenceCountObject);
        return *m_ReferenceCountObject;
    };
};

/**
 * @brief a simple reference count object class
 */
class LockedReferenceCountObjectImpl :
        public virtual IReferenceCountObject
{
public:
    /** @brief add a reference to this object */
    virtual int addReference();

    /** @brief release a reference from this object */
    virtual int releaseReference();

protected:
    /** @brief ctor */
    NN_IMPLICIT LockedReferenceCountObjectImpl(const char* nameIn) NN_NOEXCEPT;

    /** @brief dtor */
    virtual ~LockedReferenceCountObjectImpl();

    /** @brief the object name */
    char m_LockedReferenceCountObjectName[API_UNIT_STRING_MAX];

private:
    /** @brief the access lock (for Finish & Run) */
    nn::os::MutexType m_lockedReferenceCountAccessLock;

    /** @brief reference count */
    unsigned int m_lockedReferenceCount;

};

/**
 * @brief a simple validator class
 */
class SimpleValidator :
        public virtual IValidator,
        public virtual LockedReferenceCountObjectImpl
{
public:
    /**
     * @brief ctor
     */
    SimpleValidator();

    /**
     * @brief fail this test
     * @param fileName the name of the file
     * @param file number where failure happened
     * @param failureCondition a condition that produces the failure on true i.e.  (!(index >= RANGE_MIN && index <= RANGE_MAX) )
     * @param functionName the function where the failure happened
     * @param message the message to accompany the failure
     */
    virtual bool TryFail(const char* fileName,
                         unsigned int lineNumber,
                         const char* functionName,
                         const bool failureCondition,
                         const char* messageFormat,
                         ... );

    /**
     * @brief this validator succeed or fail
     */
    virtual bool DidSucceed() const;

    /**
     * @brief this validator succeed or fail
     */
    virtual bool DidFail() const;

    /**
     * @brief the filename
     */
    const char* FileName() const;

    /**
     * @brief the file number
     */
    const int  Line() const;

    /**
     * @brief the file name
     */
    const char* FunctionName() const;

    /**
     * @brief the message
     */
    const char* Message() const;

protected:
    virtual ~SimpleValidator();

private:
    /** @brief the access lock (for Finish & Run) */
    mutable nn::os::MutexType m_DataAccessLock;

    /** @brief the reverse of the condition provided*/
    bool m_DidFail;

    /** @brief filename */
    char* m_FileName;

    /** @brie fline number */
    unsigned int m_LineNumber;

    /** @brief function name */
    char* m_FunctionName;

    /** @brief function name */
    char* m_Message;
};

#define SVALIDATE_PASS(v, c, m, ...) (false == v->TryFail(__FILE__, __LINE__, __FUNCTION__,( c ), m, ## __VA_ARGS__))
#define SVALIDATE_FAIL(v, c, m, ...) (true == v->TryFail(__FILE__, __LINE__, __FUNCTION__,( c ), m, ## __VA_ARGS__))

/** @brief the stack size of the thread */
static const int UnitTestThreadStackSize = 65536;

/**
 * @brief a base class for threads used in api unit tests
 */
class UnitTestThreadBase :
        public virtual IUnitTestThread,
        public virtual LockedReferenceCountObjectImpl
{
    NN_DISALLOW_COPY(UnitTestThreadBase);
public:
    /** @brief start the thread */
    virtual bool Start();

    /** @brief finish the thread */
    virtual bool Stop();

    /**
     * @brief wait for done
     * @return true
     */
    virtual bool WaitForDone();

    /**
    * @brief wait for done with timeout
    * @return true if actually done, false otherwise
    */
    virtual bool WaitForDoneWithTimeout();


    /** @brief get the state of the thread */
    virtual IUnitTestThread::State GetState();

protected:
    /** the argument has protected access for EarlyBlockingInitialize */
    friend class UnitTestThreadBaseArgument;

    /** @brief EarlyBlockingInitialize this function can be overwritten by
     * implementing classes that need to block on a condition prior to calling run
     * this allows for serialization of threads but can also lead to lock contention
     * if used improperly
     *
     * @return true if semaphore was handled, false otherwise
     */
    virtual bool EarlyBlockingInitialize(nn::os::SemaphoreType& blockingSemaphore);

    /** @brief base constructor, used only by implementing classes */
    UnitTestThreadBase(const char* nameIn, uint64_t waitForDoneTime) NN_NOEXCEPT;

    /** @brief the run function, must be implemented by implementing classes */
    virtual void Run() = 0;

    /** @brief destructor, used only by implementing classes */
    virtual ~UnitTestThreadBase();

    /** @brief static trampoline function, calls Run() */
    static void s_Run(void* pArgs);

    /** @brief housekeeping */
    IUnitTestThread::State m_State;

    /** @brief The thread that runs this monitor */
    nn::os::ThreadType m_Thread;

    /** @brief The thread stack */
    uint8_t m_ThreadStack[UnitTestThreadStackSize];

    /** @brief pointer to the aligned stack */
    uint8_t* m_ThreadStackAlignedPointer;

    /** @brief the access lock (for Finish & Run) */
    nn::os::MutexType m_BaseAccessLock;

    /** @brief Finished Event */
    nn::os::EventType m_finishedEvent;

    /** @brief timeout for condition variable */
    uint64_t m_WaitForDoneTimeout;

private:
    /** @brief the name of this thread */
    char m_UnitTestThreadBaseName[API_UNIT_STRING_MAX];
};


/**
 * @brief the test monitor thread
 * also calls close on sockets passed to it
 */
class TestManagerThread :
        public virtual ITestManager,
        public virtual UnitTestThreadBase
{
public:
    /**
     * @brief ctor
     * <TODO> use interfaces rather than concrete classes
     */
    TestManagerThread(SelectUnitData* pUnitData, SelectUnitServer* pServer, SelectUnitClient* pClient);

    /** @brief use a timed wait event rather than a wait event */
    virtual bool WaitForDone();

protected:
    /** @brief dtor */
    virtual ~TestManagerThread();

    /** @brief run function, waits for testTimeout or signal */
    virtual void Run();


private:
    /** @brief the unit data */
    SelectUnitData* m_pUnitData;

    /** @brief the server */
    SelectUnitServer *m_pServer;

    /** @brief the client */
    SelectUnitClient* m_pClient;

    /** @brief access lock for manager */
    nn::os::MutexType m_managerAccessLock;
};

}} //NATF::API
