﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/socket.h>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>

#include <mutex>

#include "testNet_ApiCommon.h"
#include <nn/nn_Assert.h>
#include <nnt.h>

#include "testNet_UnitCommon.h"

namespace NATF {
namespace API {

/** test data */

extern const char* g_Host;

extern unsigned char* g_DnsRawData;
extern size_t g_UdpDnsRawDataSize;

extern unsigned char* g_DnsRawDataWithLength;
extern size_t g_DnsRawDataWithLengthSize;

const unsigned MAX_HOST_LENGTH = 128;

void InitializeServerDataResponse();

void FinalizeServerDataResponse();

void DnsBurnTestGetServerSocketAddress(nn::socket::SockAddrIn& addr);

void AddCancelHandleToList(int cancelHandle);

void RemoveCancelHandleFromList(int cancelHandle);

void AttemptCancel(int & cancelHandleInOut, const char* why);

enum
{
    LENGTH_POSITION_RAW_ONLY = 0,
    TRANSACTION_ID_POSITION = 0,
    FLAGS_POSITION = 2,
    NUM_ANSWERS_POSITION = 6,
    NUM_UDP_ANSWERS = 28,
    NUM_TCP_ANSWERS = 4092,
    TC_BIT_SHIFT = 9
};

/**
 * @brief the test thread
 */
class DnsBurnTestThread :
        public virtual UnitTestThreadBase
{
public:
    /** @brief ctor */
    NN_IMPLICIT DnsBurnTestThread(bool isServer=false, bool shouldAcquireCancelHandle=false);

    /**
     * @brief initialize the thread
     */
    virtual void InitializeServer(SimpleValidator* pSimpleValidator,
                                  unsigned serverDelay,
                                  unsigned numberOfListeners,
                                  bool forceServerTcp);

    virtual void InitializeClient(unsigned workerThreadNumber,
                                  SimpleValidator* pSimpleValidator,
                                  const char* hostName,
                                  bool shouldUseGetAddrInfo,
                                  unsigned listenBacklog);

   /**
    * @brief Interrupt thread by closing the socket
    * @return 0 if able to interrupt -1 if the thread was already done
    */
    virtual void Interrupt();

    /** @brief operator new */
    void* operator new( std::size_t count );

    /** @brief get the final statistics when a worker / client thread completes */
    void GetFinalStatistics(unsigned & total, unsigned & success, unsigned & failure) const;

protected:
    /** @brief dtor */
    virtual ~DnsBurnTestThread();

    /** @brief run function, waits for testTimeout or signal */
    virtual void RunClient();

    /** @brief run function, waits for testTimeout or signal */
    virtual void RunServer();

    /** @brief run function, waits for testTimeout or signal */
    virtual void Run();

    /** @brief blocks the calling thread until the server is started */
    bool EarlyBlockingInitialize(nn::os::SemaphoreType& blockingSemaphore);

    int WasInterrupted(); //OrAcquireClientCancelHandle();

private:
    /** @brief the validator */
    SimpleValidator* m_pValidator;

    bool m_isInitialized;

    /** @brief access lock for manager */
    bool m_isServer;

    /** @brief access lock for manager */
    char m_HostName[MAX_HOST_LENGTH];

    /** @brief client should use getaddrinfo */
    bool m_ClientShouldUseGetaddrinfo;

    /** @brief access lock for manager */
    mutable std::mutex m_AccessLock;

    /** was thread was interrupted */
    bool m_WasInterrupted;

    /** worker thread number */
    unsigned m_WorkerThreadNumber;

    /** number of seconds to delay responses from the server */
    unsigned m_ServerDelay;

    /** the udp server socket */
    int m_UdpServerSocket;

    /** the tcp listening socket */
    int m_TcpServerSocket;

    unsigned m_NumberOfListeners;

    bool m_ForceServerTcp;

    /** total number of attempts to call a resolver function */
    unsigned m_TotalResolveAttempts;

    /** number successful attempts */
    unsigned m_NumberOfSuccessfulResolves;

    /** number of listen backlog */
    unsigned m_ListenBacklog;

    /** should I acquire a cancel handle */
    bool m_ShouldAcquireCancelHandle;

    /** by default this is zero, but if m_ShouldAcquireCancelHandle
        true then BurnXXX functions acquire and release a handle */
    int m_CancelHandle;

    /** handle udp path for server */
    int HandleServerUdp();

    /** handle tcp path for server */
    int HandleServerTcp();

    /** burn get addrinfo */
    nn::socket::AiErrno BurnGetAddrInfo();

    /** burn gethostbyname */
    int BurnGetHostByName();
};

}};
