﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>

#include <nn/ndd.h>
#include "../Common/testNdd_Common.h"

#define TEST_LOG(format, ...) NN_LOG("[testNdd_Receive]" format, ##__VA_ARGS__)

namespace {
//スレッド
const size_t ThreadStackSize = 4096;
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack[ThreadStackSize];
nn::os::ThreadType  g_DummyDeviceThread;
}

bool IsValidData(const void* pData, size_t size)
{
    static PayloadUtil payloadUtil;
    return payloadUtil.Import(pData, size);
}

void ScanThread(void* arg)
{
    static const int RddCountMax = 50;
    static nn::ndd::ReceiveDataDescription g_Rdd[RddCountMax];

    nn::os::SystemEventType deviceScanEvent;
    nn::ndd::AcquireDeviceScanEvent(&deviceScanEvent);

    for(;;)
    {
        nn::ndd::StartDeviceScan();
        nn::os::WaitSystemEvent(&deviceScanEvent);

        auto resultCount = nn::ndd::GetDeviceScanResult(&g_Rdd[0], RddCountMax);
        for(int i=0;i<resultCount;++i)
        {
            auto isValid = IsValidData(&g_Rdd[i].data[0], g_Rdd[i].dataSize);
            TEST_LOG("ScanResult(%d/%d) dataSize = %d, data[0] = %d, receiveTimePoint = %lld, isValidData = %d\n",
                    i + 1, resultCount, g_Rdd[i].dataSize, g_Rdd[i].data[0], g_Rdd[i].receiveTimePoint.value, isValid);
        }
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    }
}

extern "C" void nnMain()
{
    nn::ndd::Initialize();

    //Scanスレッドを生成
    auto result = nn::os::CreateThread( &g_DummyDeviceThread, ScanThread, nullptr, g_ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::StartThread(&g_DummyDeviceThread);

    //データ受信
    nn::os::SystemEventType receiveDataEvent;
    nn::ndd::AcquireReceiveDataEvent(&receiveDataEvent);
    uint32_t counter = nn::ndd::GetOldestReceiveDataCounter();
    TEST_LOG("CurrentCounter = %d\n", nn::ndd::GetCurrentReceiveDataCounter());
    TEST_LOG("NextCounter    = %d\n", nn::ndd::GetNextReceiveDataCounter());
    TEST_LOG("OldestCounter  = %d\n", counter);
    TEST_LOG("AvailableCount = %d\n", nn::ndd::GetAvailableReceiveDataCount(counter));
    for(;;)
    {
        static const int RddCountMax = 50;
        static nn::ndd::ReceiveDataDescription g_Rdd[RddCountMax];

        auto resultCount = nn::ndd::GetReceiveData(&g_Rdd[0], &counter, counter, RddCountMax);
        for(int i=0;i<resultCount;++i)
        {
            auto isValid = IsValidData(&g_Rdd[i].data[0], g_Rdd[i].dataSize);
            TEST_LOG("ReceiveResult(%d/%d) dataSize = %d, data[0] = %d, receiveTimePoint = %lld, isValidData = %d\n",
                    i + 1, resultCount, g_Rdd[i].dataSize, g_Rdd[i].data[0], g_Rdd[i].receiveTimePoint.value, isValid);
        }
        nn::os::WaitSystemEvent(&receiveDataEvent);
    }

    //WaitThread(&g_DummyDeviceThread);
    //DestroyThread(&g_DummyDeviceThread);
}
