﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <string>
#include <random>
#include <chrono>
#include <set>
#include <tuple>
#include <new>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/fs.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/kvdb/kvdb_InMemoryKeyValueStore.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentMetaDatabaseImpl.h>
#include <nn/ncm/ncm_IContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_StdAllocationPolicy.h>
#include <nn/sf/sf_DefaultAllocationPolicy.h>

namespace {
    typedef std::string String;
    typedef nn::sf::ObjectFactory<nn::sf::StdAllocationPolicy<std::allocator>> ContentMetaDatabaseFactory;
    typedef nn::sf::SharedPointer<nn::ncm::IContentMetaDatabase> ContentMetaDatabaseShared;

    const char* MountName = "saveData";

    String GetNowString()
    {
        using namespace std::chrono;

        auto duration = high_resolution_clock::now().time_since_epoch();
        time_t t = duration_cast<seconds>(duration).count();
        tm now;
        localtime_s(&now, &t);

        char stamp[32];
        sprintf_s(stamp, sizeof(stamp),
            "%04u%02u%02u%02u%02u%02u%03u",
            now.tm_year + 1900, now.tm_mon + 1, now.tm_mday, now.tm_hour, now.tm_min, now.tm_sec,
            static_cast<uint32_t>(duration_cast<milliseconds>(duration).count() % 1000));

        return String(stamp);
    }

    String CreateTemporaryRootPath()
    {
        return String(MountName)
            .append(":/")
            .append("SIGLO_NCM_TEST")
            .append("_")
            .append(GetNowString());
    }

    template<typename HandleT>
    class ScopedHandle
    {
    public:
        typedef void(*CloseFunction)(HandleT)NN_NOEXCEPT;
        explicit ScopedHandle(HandleT handle, CloseFunction func) NN_NOEXCEPT : m_Handle(handle), m_Func(func){}
        ~ScopedHandle()
        {
            m_Func(m_Handle);
        }

    private:
        HandleT         m_Handle;
        CloseFunction   m_Func;
    };

    class ScopedDirectoryHandle : private ScopedHandle<nn::fs::DirectoryHandle>
    {
    public:
        explicit ScopedDirectoryHandle(nn::fs::DirectoryHandle handle) NN_NOEXCEPT : ScopedHandle<nn::fs::DirectoryHandle>(handle, nn::fs::CloseDirectory){}
    };

    nn::Result ShowDirectoryTreeRecursively(const char* rootPath, int indent)
    {
        nn::fs::DirectoryHandle dir;
        NN_RESULT_DO(nn::fs::OpenDirectory(&dir, rootPath, static_cast<nn::fs::OpenDirectoryMode>(nn::fs::OpenDirectoryMode_File | nn::fs::OpenDirectoryMode_Directory)));
        ScopedDirectoryHandle scopedDirectoryHandle(dir);

        for (;;)
        {
            nn::fs::DirectoryEntry entry;
            int64_t numEntries;
            NN_RESULT_DO(nn::fs::ReadDirectory(&numEntries, &entry, dir, 1));
            if (numEntries == 0)
            {
                break;
            }

            for (auto i = 0; i < indent; i++)
            {
                NN_LOG("  ");
            }
            switch (entry.directoryEntryType)
            {
            case nn::fs::DirectoryEntryType_Directory:
                {
                    NN_LOG("%s/\n", entry.name);
                    ShowDirectoryTreeRecursively((std::string(rootPath) + "/" + entry.name).c_str(), indent + 1);
                }
                break;
            case nn::fs::DirectoryEntryType_File:
                {
                    NN_LOG("%s %lld bytes\n", entry.name, entry.fileSize);
                }
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result ShowDirectoryTree(const char* rootPath)
    {
        NN_LOG("Show directory tree %s\n", rootPath);

        return ShowDirectoryTreeRecursively(rootPath, 0);
    }

    void* Allocate(size_t size)
    {
        return std::malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        return std::free(p);
    }

    class ContentMetaDatabaseTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            auto rootPath = CreateTemporaryRootPath();
            NN_ABORT_UNLESS(nn::fs::CreateDirectory(rootPath.c_str()).IsSuccess());
            m_RootPath = rootPath;
        }

        virtual void TearDown()
        {
            NN_ABORT_UNLESS(nn::fs::DeleteDirectoryRecursively(GetRootPath()).IsSuccess());
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
        }

        static void SetUpTestCase()
        {
            nn::fs::SetAllocator(Allocate, Deallocate);

            const nn::fs::SystemSaveDataId SaveDataId = 0x8000000000000120; // ncm のシステムセーブデータの ID を拝借する
            const int64_t SaveDataSize = 10000 * 512 + 0x8000;
            const int64_t JournalSize = 10000 * 512 + 0x8000;
            const int64_t SaveDataFlag = 0;

            nn::fs::DisableAutoSaveDataCreation();
            auto result = nn::fs::CreateSystemSaveData(SaveDataId, SaveDataSize, JournalSize, SaveDataFlag);
            NN_ABORT_UNLESS(result.IsSuccess() || result <= nn::fs::ResultPathAlreadyExists());
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSystemSaveData(MountName, SaveDataId));
        }

        static void TearDownTestCase()
        {
            nn::fs::Unmount(MountName);
        }

        const char* GetRootPath()
        {
            return m_RootPath.c_str();
        }

        nn::ncm::PackagedContentInfo MakePackagedContentInfo(nn::ncm::ContentId id, int64_t size, nn::ncm::ContentType type)
        {
            nn::ncm::PackagedContentInfo infoData = {};
            infoData.info = nn::ncm::ContentInfo::Make(id, size, type, 0);
            return infoData;
        }

        void GetContentMetaData(std::vector<char>& out, const void* packagedContentMetaData, size_t packagedContentMetaDataSize)
        {
            nn::ncm::PackagedContentMetaReader packagedContentMetaReader(packagedContentMetaData, packagedContentMetaDataSize);

            auto contentInfo = nn::ncm::ContentInfo::Make(GetRandomContentId(), 0, nn::ncm::ContentType::Meta, 0);
            auto contentMetaSize = packagedContentMetaReader.CalculateConvertContentMetaSize();
            out.resize(contentMetaSize);
            packagedContentMetaReader.ConvertToContentMeta(out.data(), contentMetaSize, contentInfo);
        }

        typedef std::tuple<nn::ncm::ContentMetaKey, std::vector<char>> ContentKeyAndValue;

        std::vector<ContentKeyAndValue> GetRandomProgramList(int count)
        {
            std::vector<ContentKeyAndValue> list;
            for (int i = 0; i < count; i++)
            {
                auto packagedContentMetaSize = nn::ncm::PackagedContentMetaWriter::CalculateSize(nn::ncm::ContentMetaType::SystemProgram, 1, 0, 0);

                std::vector<char> packagedContentMetaData;
                packagedContentMetaData.resize(packagedContentMetaSize);
                nn::ncm::PackagedContentMetaWriter writer(packagedContentMetaData.data(), packagedContentMetaSize);
                writer.WriteHeader(GetRandomId(), GetRandomVersion(), nn::ncm::ContentMetaType::SystemProgram, nn::ncm::ContentMetaAttribute_None, 0, 1, 0);

                auto programPackagedContentInfo = MakePackagedContentInfo(GetRandomContentId(), 0, nn::ncm::ContentType::Program);
                writer.WriteContentInfo(programPackagedContentInfo, 0);

                std::vector<char> contentMetaData;
                GetContentMetaData(contentMetaData, packagedContentMetaData.data(), packagedContentMetaSize);

                nn::ncm::ContentMetaKey key = writer.GetKey();
                list.push_back(ContentKeyAndValue(key, contentMetaData));
            }

            return list;
        }

        std::vector<ContentKeyAndValue> GetRandomApplicationList(int count)
        {
            std::vector<ContentKeyAndValue> list;
            for (int i = 0; i < count; i++)
            {
                auto packagedContentMetaSize = nn::ncm::PackagedContentMetaWriter::CalculateSize(nn::ncm::ContentMetaType::Application, 2, 0, 0);
                std::vector<char> packagedContentMetaData;
                packagedContentMetaData.resize(packagedContentMetaSize);

                nn::ncm::PackagedContentMetaWriter writer(packagedContentMetaData.data(), packagedContentMetaSize);
                writer.WriteHeader(GetRandomId(), GetRandomVersion(), nn::ncm::ContentMetaType::Application, nn::ncm::ContentMetaAttribute_None, 0, 2, 0);
                nn::ncm::ApplicationMetaExtendedHeader extendedHeader = { GetRandomId(), GetRandomVersion() };
                writer.WriteExtendedHeader<nn::ncm::ApplicationMetaExtendedHeader>(extendedHeader);

                auto programContentInfo = MakePackagedContentInfo(GetRandomContentId(), 0, nn::ncm::ContentType::Program);
                writer.WriteContentInfo(programContentInfo, 0);
                auto controlContentInfo = MakePackagedContentInfo(GetRandomContentId(), 0, nn::ncm::ContentType::Control);
                writer.WriteContentInfo(controlContentInfo, 1);

                std::vector<char> contentMetaData;
                GetContentMetaData(contentMetaData, packagedContentMetaData.data(), packagedContentMetaSize);

                nn::ncm::ContentMetaKey key = writer.GetKey();
                list.push_back(ContentKeyAndValue(key, contentMetaData));
            }

            return list;
        }

        nn::Bit64 GetRandomId()
        {
            return m_RandomEngine();
        }

        uint16_t GetRandomVersion()
        {
            return std::uniform_int_distribution<uint16_t>(1, 65534)(m_RandomEngine);
        }

        nn::ncm::ContentId GetRandomContentId()
        {
            nn::ncm::ContentId contentId = {};
            auto low = m_RandomEngine();
            auto high = m_RandomEngine();
            std::memcpy(&contentId.data[0], &low, sizeof(low));
            std::memcpy(&contentId.data[8], &high, sizeof(high));

            return contentId;
        }

        std::vector<nn::ncm::ContentMetaKey> GetRandomContentMetaKeyList(int count)
        {
            std::vector<nn::ncm::ContentMetaKey> list;
            for (int i = 0; i < count; i++)
            {
                nn::ncm::ContentMetaKey key;
                key.type = static_cast<nn::ncm::ContentMetaType>(m_RandomEngine());
                key.id = m_RandomEngine();
                key.version = static_cast<uint16_t>(m_RandomEngine());

                list.push_back(key);
            }

            return list;
        }

        std::vector<std::string> GetRandomDataList(int count, size_t minLength, size_t maxLength)
        {
            std::vector<std::string> list;
            for (int i = 0; i < count; i++)
            {
                auto size = std::uniform_int_distribution<size_t>(minLength, maxLength)(m_RandomEngine);
                std::string data;
                for (size_t j = 0; j < size; j++)
                {
                    char c = static_cast<char>(m_RandomEngine());
                    data.append(1, c);
                }

                list.push_back(data);
            }

            return list;
        }

        std::string     m_RootPath;
        std::mt19937_64 m_RandomEngine;
    };

    template < typename ContainerType, typename EntryType >
    bool Contains(const ContainerType& container, EntryType entry)
    {
        for (auto& x : container)
        {
            if (x == entry)
            {
                return true;
            }
        }

        return false;
    }

    template<typename ContainerType>
    bool IsEquivalent(const ContainerType& l, const ContainerType& r)
    {
        if (l.size() != r.size())
        {
            return false;
        }

        for (auto& x : l)
        {
            if (!Contains(r, x))
            {
                return false;
            }
        }

        return true;
    }
}

TEST_F(ContentMetaDatabaseTest, ContentMetaKeyValueStore)
{
    nn::ncm::ContentMetaKeyValueStore kvs;
    NNT_EXPECT_RESULT_SUCCESS(kvs.Initialize(GetRootPath(), 10000, nn::sf::GetNewDeleteMemoryResource()));
    NNT_EXPECT_RESULT_SUCCESS(kvs.Load());

    const int TestDataCount = 10000;

    auto contentMetaKeyList = GetRandomContentMetaKeyList(TestDataCount);
    auto valueList = GetRandomDataList(TestDataCount, 16, 512);

    for (int i = 0; i < TestDataCount; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(kvs.Put(contentMetaKeyList[i], valueList[i].data(), valueList[i].size()));
    }

    auto CheckFunc = [&]()
    {
        for (int i = 0; i < TestDataCount; i++)
        {
            size_t valueSize;
            NNT_EXPECT_RESULT_SUCCESS(kvs.GetSize(&valueSize, contentMetaKeyList[i]));
            EXPECT_EQ(valueList[i].size(), valueSize);

            nn::kvdb::AutoBuffer value;
            NNT_EXPECT_RESULT_SUCCESS(value.Initialize(valueSize));
            size_t readSize;
            NNT_EXPECT_RESULT_SUCCESS(kvs.Get(&readSize, contentMetaKeyList[i], value.Get(), value.GetSize()));
            EXPECT_TRUE(std::memcmp(valueList[i].data(), value.Get(), value.GetSize()) == 0);
        }
    };


    NN_LOG("Check before Save()\n");
    CheckFunc();

    NNT_EXPECT_RESULT_SUCCESS(kvs.Save());
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));

    NN_LOG("Check after Save()\n");
    CheckFunc();

    NNT_EXPECT_RESULT_SUCCESS(kvs.Load());

    NN_LOG("Check after Load()\n");
    CheckFunc();

    NNT_EXPECT_RESULT_SUCCESS(ShowDirectoryTree(GetRootPath()));
}

TEST_F(ContentMetaDatabaseTest, SetGetRemoveProgram)
{
    nn::ncm::ContentMetaKeyValueStore kvs;
    NNT_EXPECT_RESULT_SUCCESS(kvs.Initialize(GetRootPath(), 10000, nn::sf::GetNewDeleteMemoryResource()));
    NNT_EXPECT_RESULT_SUCCESS(kvs.Load());
    auto contentMetaDatabaseShared = ContentMetaDatabaseFactory::CreateSharedEmplaced<nn::ncm::IContentMetaDatabase, nn::ncm::ContentMetaDatabaseImpl>(&kvs, MountName);
    nn::ncm::ContentMetaDatabase db(contentMetaDatabaseShared);

    auto programList = GetRandomProgramList(10);

    for (auto& contentKeyAndValue : programList)
    {
        auto program = std::get<1>(contentKeyAndValue);
        auto contentMetaKey = std::get<0>(contentKeyAndValue);
        nn::ncm::ContentMetaReader reader(program.data(), program.size());
        NNT_EXPECT_RESULT_SUCCESS(db.Set(contentMetaKey, reader.GetData(), reader.GetSize()));

        nn::ncm::SystemProgramId programId = { contentMetaKey.id };
        uint32_t version = contentMetaKey.version;

        nn::ncm::ContentId contentId;
        NNT_EXPECT_RESULT_SUCCESS(db.GetProgram(&contentId, programId, version ));
        EXPECT_EQ(reader.GetContentInfo(nn::ncm::ContentType::Program)->id, contentId);

        NNT_EXPECT_RESULT_SUCCESS(db.GetLatestProgram(&contentId, programId));
        EXPECT_EQ(reader.GetContentInfo(nn::ncm::ContentType::Program)->id, contentId);

        nn::ncm::ContentInfo infoList[16];
        int count;
        NNT_EXPECT_RESULT_SUCCESS(db.ListContentInfo(&count, infoList, 16, contentMetaKey, 0));
        EXPECT_EQ(2, count);
        for (int i = 0; i < count; i++)
        {
            EXPECT_EQ(reader.GetContentInfo(i)->id, infoList[i].id);
            EXPECT_EQ(reader.GetContentInfo(i)->GetSize(), infoList[i].GetSize());
            EXPECT_EQ(reader.GetContentInfo(i)->type, infoList[i].type);
        }

        NNT_EXPECT_RESULT_SUCCESS(db.Remove(programId, version));
        NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultContentMetaNotFound, db.GetProgram(&contentId, programId, version));
        NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultContentMetaNotFound, db.GetLatestProgram(&contentId, programId));
    }

    NNT_EXPECT_RESULT_SUCCESS(ShowDirectoryTree(GetRootPath()));
}

TEST_F(ContentMetaDatabaseTest, SetGetRemoveApplication)
{
    nn::ncm::ContentMetaKeyValueStore kvs;
    NNT_EXPECT_RESULT_SUCCESS(kvs.Initialize(GetRootPath(), 10000, nn::sf::GetNewDeleteMemoryResource()));
    NNT_EXPECT_RESULT_SUCCESS(kvs.Load());
    auto contentMetaDatabaseShared = ContentMetaDatabaseFactory::CreateSharedEmplaced<nn::ncm::IContentMetaDatabase, nn::ncm::ContentMetaDatabaseImpl>(&kvs, MountName);
    nn::ncm::ContentMetaDatabase db(contentMetaDatabaseShared);

    auto applicationList = GetRandomApplicationList(10);

    for (auto& contentKeyAndValue : applicationList)
    {
        auto application = std::get<1>(contentKeyAndValue);
        auto contentMetaKey = std::get<0>(contentKeyAndValue);

        nn::ncm::ContentMetaReader reader(application.data(), application.size());
        NNT_EXPECT_RESULT_SUCCESS(db.Set(contentMetaKey, reader.GetData(), reader.GetSize()));

        nn::ncm::ApplicationId applicationId = { contentMetaKey.id };
        uint32_t version = contentMetaKey.version;
        nn::ncm::ContentId contentId;
        NNT_EXPECT_RESULT_SUCCESS(db.GetProgram(&contentId, applicationId, version ));
        EXPECT_EQ(reader.GetContentInfo(nn::ncm::ContentType::Program)->id, contentId);
        NNT_EXPECT_RESULT_SUCCESS(db.GetControl(&contentId, applicationId, version ));
        EXPECT_EQ(reader.GetContentInfo(nn::ncm::ContentType::Control)->id, contentId);

        NNT_EXPECT_RESULT_SUCCESS(db.GetLatestProgram(&contentId, applicationId));
        EXPECT_EQ(reader.GetContentInfo(nn::ncm::ContentType::Program)->id, contentId);
        NNT_EXPECT_RESULT_SUCCESS(db.GetLatestControl(&contentId, applicationId));
        EXPECT_EQ(reader.GetContentInfo(nn::ncm::ContentType::Control)->id, contentId);

        nn::ncm::ContentInfo infoList[16];
        int count;
        NNT_EXPECT_RESULT_SUCCESS(db.ListContentInfo(&count, infoList, 16, contentMetaKey, 0));
        EXPECT_EQ(3, count);
        for (int i = 0; i < count; i++)
        {
            EXPECT_EQ(reader.GetContentInfo(i)->id, infoList[i].id);
            EXPECT_EQ(reader.GetContentInfo(i)->GetSize(), infoList[i].GetSize());
            EXPECT_EQ(reader.GetContentInfo(i)->type, infoList[i].type);
        }

        NNT_EXPECT_RESULT_SUCCESS(db.Remove(applicationId, version));
        NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultContentMetaNotFound, db.GetProgram(&contentId, applicationId, version));
        NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultContentMetaNotFound, db.GetLatestProgram(&contentId, applicationId));
        NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultContentMetaNotFound, db.GetControl(&contentId, applicationId, version));
        NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultContentMetaNotFound, db.GetLatestControl(&contentId, applicationId));
    }

    NNT_EXPECT_RESULT_SUCCESS(ShowDirectoryTree(GetRootPath()));
}

TEST_F(ContentMetaDatabaseTest, ListContentMeta)
{
    nn::ncm::ContentMetaKeyValueStore kvs;
    NNT_EXPECT_RESULT_SUCCESS(kvs.Initialize(GetRootPath(), 10000, nn::sf::GetNewDeleteMemoryResource()));
    NNT_EXPECT_RESULT_SUCCESS(kvs.Load());
    auto contentMetaDatabaseShared = ContentMetaDatabaseFactory::CreateSharedEmplaced<nn::ncm::IContentMetaDatabase, nn::ncm::ContentMetaDatabaseImpl>(&kvs, MountName);
    nn::ncm::ContentMetaDatabase db(contentMetaDatabaseShared);

    auto applicationList = GetRandomApplicationList(10);

    for (auto& contentKeyAndValue : applicationList)
    {
        auto application = std::get<1>(contentKeyAndValue);
        auto contentMetaKey = std::get<0>(contentKeyAndValue);
        nn::ncm::ContentMetaReader reader(application.data(), application.size());
        NNT_EXPECT_RESULT_SUCCESS(db.Set(contentMetaKey, reader.GetData(), reader.GetSize()));
    }

    nn::ncm::ContentMetaKey keyList[16];
    nn::ncm::ListCount keyCount = db.ListContentMeta(keyList, 16, nn::ncm::ContentMetaType::Unknown);
    EXPECT_EQ(keyCount.listed, 10);
    EXPECT_EQ(keyCount.total, 10);
    for(auto& contentKeyAndValue : applicationList)
    {
        auto application = std::get<1>(contentKeyAndValue);
        auto key = std::get<0>(contentKeyAndValue);
        bool found = false;
        for(int i = 0; i < keyCount.total; ++i)
        {
            nn::ncm::ContentMetaReader reader(application.data(), application.size());
            if(key.id == keyList[i].id)
            {
                found = true;
                break;
            }
        }
        EXPECT_TRUE(found);
    }


    NNT_EXPECT_RESULT_SUCCESS(ShowDirectoryTree(GetRootPath()));
}

TEST_F(ContentMetaDatabaseTest, ListContentMetaOfApplication)
{
    nn::ncm::ContentMetaKeyValueStore kvs;
    NNT_EXPECT_RESULT_SUCCESS(kvs.Initialize(GetRootPath(), 10000, nn::sf::GetNewDeleteMemoryResource()));
    NNT_EXPECT_RESULT_SUCCESS(kvs.Load());
    auto contentMetaDatabaseShared = ContentMetaDatabaseFactory::CreateSharedEmplaced<nn::ncm::IContentMetaDatabase, nn::ncm::ContentMetaDatabaseImpl>(&kvs, MountName);
    nn::ncm::ContentMetaDatabase db(contentMetaDatabaseShared);

    auto applicationList = GetRandomApplicationList(10);

    for (auto& contentKeyAndValue : applicationList)
    {
        auto application = std::get<1>(contentKeyAndValue);
        auto contentMetaKey = std::get<0>(contentKeyAndValue);
        nn::ncm::ContentMetaReader reader(application.data(), application.size());
        NNT_EXPECT_RESULT_SUCCESS(db.Set(contentMetaKey, reader.GetData(), reader.GetSize()));

        nn::ncm::ContentMetaKey key;
        nn::ncm::ListCount keyCount;
        keyCount = db.ListContentMeta(&key, 1, nn::ncm::ContentMetaType::Unknown);
        EXPECT_EQ(keyCount.total, 1);
        keyCount = db.ListContentMeta(&key, 1, nn::ncm::ContentMetaType::Application);
        EXPECT_EQ(keyCount.total, 1);
        keyCount = db.ListContentMeta(&key, 1, nn::ncm::ContentMetaType::AddOnContent);
        EXPECT_EQ(keyCount.total, 0);

#if 0
        nn::ncm::ApplicationId validId = reader.GetApplicationId().value();
        nn::ncm::ApplicationId invalidId = {reader.GetApplicationId().value().value + 1};

        keyCount = db.ListContentMeta(&key, 1, nn::ncm::ContentMetaType::Unknown, validId);
        EXPECT_EQ(keyCount.total, 1);
        keyCount = db.ListContentMeta(&key, 1, nn::ncm::ContentMetaType::Unknown, invalidId);
        EXPECT_EQ(keyCount.total, 0);
        keyCount = db.ListContentMeta(&key, 1, nn::ncm::ContentMetaType::Unknown, validId, 0);
        EXPECT_EQ(keyCount.total, 0);
#endif
//        NNT_EXPECT_RESULT_SUCCESS(db.Remove(validId, reader.GetKey().version));
        nn::ncm::ApplicationId validId = { contentMetaKey.id };
        NNT_EXPECT_RESULT_SUCCESS(db.Remove(validId, contentMetaKey.version));
    }

    NNT_EXPECT_RESULT_SUCCESS(ShowDirectoryTree(GetRootPath()));
}

TEST_F(ContentMetaDatabaseTest, Get)
{
    nn::ncm::ContentMetaKeyValueStore kvs;
    NNT_EXPECT_RESULT_SUCCESS(kvs.Initialize(GetRootPath(), 10000, nn::sf::GetNewDeleteMemoryResource()));
    NNT_EXPECT_RESULT_SUCCESS(kvs.Load());
    auto contentMetaDatabaseShared = ContentMetaDatabaseFactory::CreateSharedEmplaced<nn::ncm::IContentMetaDatabase, nn::ncm::ContentMetaDatabaseImpl>(&kvs, MountName);
    nn::ncm::ContentMetaDatabase db(contentMetaDatabaseShared);

    auto applicationList = GetRandomApplicationList(10);

    for (auto& contentKeyAndValue : applicationList)
    {
        auto application = std::get<1>(contentKeyAndValue);
        auto key = std::get<0>(contentKeyAndValue);
        nn::ncm::ContentMetaReader reader(application.data(), application.size());
        NNT_EXPECT_RESULT_SUCCESS(db.Set(key, reader.GetData(), reader.GetSize()));

        const size_t BufferSize = 1024 * 1024;
        std::unique_ptr<char[]> metaBuffer(new char[BufferSize]);
        size_t outSize;
        NNT_EXPECT_RESULT_SUCCESS(db.Get(&outSize, metaBuffer.get(), BufferSize, key));
        EXPECT_EQ(0, std::memcmp(metaBuffer.get(), application.data(), outSize));

        NNT_EXPECT_RESULT_SUCCESS(db.GetSize(&outSize, key));
        EXPECT_EQ(outSize, application.size());
    }

    NNT_EXPECT_RESULT_SUCCESS(ShowDirectoryTree(GetRootPath()));
}


TEST_F(ContentMetaDatabaseTest, GetApi)
{
    nn::ncm::ContentMetaKeyValueStore kvs;
    NNT_EXPECT_RESULT_SUCCESS(kvs.Initialize(GetRootPath(), 10000, nn::sf::GetNewDeleteMemoryResource()));
    NNT_EXPECT_RESULT_SUCCESS(kvs.Load());
    auto contentMetaDatabaseShared = ContentMetaDatabaseFactory::CreateSharedEmplaced<nn::ncm::IContentMetaDatabase, nn::ncm::ContentMetaDatabaseImpl>(&kvs, MountName);
    nn::ncm::ContentMetaDatabase db(contentMetaDatabaseShared);

    auto applicationList = GetRandomApplicationList(10);

    for (auto& contentKeyAndValue : applicationList)
    {
        auto application = std::get<1>(contentKeyAndValue);
        auto key = std::get<0>(contentKeyAndValue);
        nn::ncm::ContentMetaReader reader(application.data(), application.size());
        NNT_EXPECT_RESULT_SUCCESS(db.Set(key, reader.GetData(), reader.GetSize()));

        uint32_t requiredSystemVersion;
        NNT_EXPECT_RESULT_SUCCESS(db.GetRequiredSystemVersion(&requiredSystemVersion, key));
        EXPECT_EQ(reader.GetExtendedHeader<nn::ncm::ApplicationMetaExtendedHeader>()->requiredSystemVersion, requiredSystemVersion);

        nn::ncm::PatchId patchId;
        NNT_EXPECT_RESULT_SUCCESS(db.GetPatchId(&patchId, key));
        EXPECT_EQ(reader.GetExtendedHeader<nn::ncm::ApplicationMetaExtendedHeader>()->patchId, patchId);
    }

    NNT_EXPECT_RESULT_SUCCESS(ShowDirectoryTree(GetRootPath()));
}
