﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "fAudioSource.h"

#include <cmath>
#include <nn/nn_Log.h>
#include <nn/audio.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>

#include <media/stagefright/MediaBufferGroup.h>
#include <media/stagefright/foundation/ADebug.h>
#include <media/stagefright/MediaDefs.h>
#include <media/stagefright/MetaData.h>

using namespace nn;
using namespace android;

NN_ALIGNAS(4096) static char  inBuff0[8192];
NN_ALIGNAS(4096) static char  inBuff1[8192];
NN_ALIGNAS(4096) static char  inBuff2[8192];
NN_ALIGNAS(4096) static char  inBuff3[8192];

SfAudioSource::SfAudioSource(int32_t sampleRate, int32_t numChannels)
    : mStarted(false),
      mSampleRate(sampleRate),
      mNumChannels(numChannels),
      mPhase(0),
      mNumFramesReceived(0),
      mGroup(NULL) {
    CHECK(numChannels == 1 || numChannels == 2);

    NN_LOG("SfAudioSource::open In () %d %d \n", sampleRate, numChannels);

    // connect the nn audio APIs
    numDiffs = 0;

    nn::audio::FinalOutputRecorderParameter paramGameRecord { 0 };

    auto appletResourceUserId = nn::applet::AppletResourceUserId::GetInvalidId();
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
        appletResourceUserId = nn::applet::GetAppletResourceUserId();
#endif
    NN_LOG("Opening GameRecord\n");
    if (nn::audio::OpenFinalOutputRecorder(&gameRecord, &gameRecordBufferEvent, paramGameRecord).IsFailure())
    {
        paramGameRecord.sampleRate = 0;
        NN_ABORT_UNLESS(
            nn::audio::OpenFinalOutputRecorder(&gameRecord, &gameRecordBufferEvent, paramGameRecord).IsSuccess(),
            "Failed to open GameRecord."
         );
     }

    mInitCheck = OK;

}

SfAudioSource::~SfAudioSource() {
    if (mStarted) {
        stop();
    }
}

status_t SfAudioSource::initCheck() const {
    return mInitCheck;
}


status_t SfAudioSource::start(MetaData *params) {
    Mutex::Autolock autoLock(mLock);
    NN_LOG("SfAudioSource start \n");
    if (mStarted) {
        return UNKNOWN_ERROR;
    }

    if (mInitCheck != OK) {
        return NO_INIT;
    }

    mTrackMaxAmplitude = false;
    mMaxAmplitude = 0;
    mStartTimeUs = 0;
    int64_t startTimeUs;
    if (params && params->findInt64(kKeyTime, &startTimeUs)) {
        mStartTimeUs = startTimeUs;
    }

    mGroup = new MediaBufferGroup;
    mGroup->add_buffer(new MediaBuffer(kBufferSize));
    mPhase = 0;

    const size_t dataSize = 4096 ;
    const size_t bufferSize = nn::util::align_up(dataSize, nn::audio::FinalOutputRecorderBuffer::SizeGranularity);

    NN_SDK_LOG(" GameRecord input buffersize %d \n", bufferSize);

    inBuffers[0] = (char*)(((long)inBuff0) + 4095 & (~4095));
    inBuffers[1] = (char*)(((long)inBuff1) + 4095 & (~4095));
    // inBuffers[2] = (char*)(((long)inBuff2) + 4095 & (~4095));
    // inBuffers[3] = (char*)(((long)inBuff3) + 4095 & (~4095));

    // initialize membuffers and zero them
    for (int i = 0; i < numBuffers; ++i)
    {
        NN_ASSERT(inBuffers[i]);
        memset(inBuffers[i],0,bufferSize);

        nn::audio::SetFinalOutputRecorderBufferInfo(&finalOutputRecorderBuffer[i], inBuffers[i], bufferSize, dataSize);
        nn::audio::AppendFinalOutputRecorderBuffer(&gameRecord, &finalOutputRecorderBuffer[i]);
    }

    NN_ABORT_UNLESS(
        nn::audio::StartFinalOutputRecorder(&gameRecord).IsSuccess(),
        "Failed to start recording."
    );
    mStarted = true;

    return OK;
}

status_t SfAudioSource::stop() {
    Mutex::Autolock autoLock(mLock);
    if (!mStarted) {
        return UNKNOWN_ERROR;
    }

    if (mInitCheck != OK) {
        return NO_INIT;
    }

    CHECK(mStarted);
    delete mGroup;
    mGroup = NULL;

    mStarted = false;
    NN_SDK_LOG("SfAudioSink::stop()\n");
    // stop audio record
    nn::audio::StopFinalOutputRecorder(&gameRecord);
    nn::audio::CloseFinalOutputRecorder(&gameRecord);
    return OK;
}

sp<MetaData> SfAudioSource::getFormat() {
    Mutex::Autolock autoLock(mLock);
    if (mInitCheck != OK) {
        return 0;
    }

    sp<MetaData> meta = new MetaData;
    meta->setCString(kKeyMIMEType, MEDIA_MIMETYPE_AUDIO_RAW);
    meta->setInt32(kKeyChannelCount, mNumChannels);
    meta->setInt32(kKeySampleRate, mSampleRate);
    meta->setInt32(kKeyMaxInputSize, kBufferSize);

    return meta;
}

void SfAudioSource::rampVolume(
        int32_t startFrame, int32_t rampDurationFrames,
        uint8_t *data,   size_t bytes) {

    const int32_t kShift = 14;
    int32_t fixedMultiplier = (startFrame << kShift) / rampDurationFrames;
    const int32_t nChannels = 2;
    int32_t stopFrame = startFrame + bytes / sizeof(int16_t);
    int16_t *frame = (int16_t *) data;
    if (stopFrame > rampDurationFrames) {
        stopFrame = rampDurationFrames;
    }

    while (startFrame < stopFrame) {
        if (nChannels == 1) {  // mono
            frame[0] = (frame[0] * fixedMultiplier) >> kShift;
            ++frame;
            ++startFrame;
        } else {               // stereo
            frame[0] = (frame[0] * fixedMultiplier) >> kShift;
            frame[1] = (frame[1] * fixedMultiplier) >> kShift;
            frame += 2;
            startFrame += 2;
        }

        // Update the multiplier every 4 frames
        if ((startFrame & 3) == 0) {
            fixedMultiplier = (startFrame << kShift) / rampDurationFrames;
        }
    }
}

status_t SfAudioSource::read(
        MediaBuffer **out, const ReadOptions * /* options */) {
    Mutex::Autolock autoLock(mLock);
    *out = NULL;

    if (mInitCheck != OK) {
        return NO_INIT;
    }
    //  NN_LOG("SfAudioSource read \n");
    MediaBuffer *buffer;
    status_t err = mGroup->acquire_buffer(&buffer);
    if (err != OK) {
           NN_LOG("SfAudioSource error \n");
        return err;
    }

    size_t frameSize = mNumChannels * sizeof(int16_t);
    size_t numFramesPerBuffer = buffer->size() / frameSize;

    buffer->meta_data()->setInt64(
            kKeyTime, ((int64_t)mPhase * 1000000) / mSampleRate);

    mPhase += numFramesPerBuffer;
    gameRecordBufferEvent.Wait();
    // get a released buffer and write the new output
    nn::audio::FinalOutputRecorderBuffer* pFinalOutputRecorderBuffer = nullptr;
    pFinalOutputRecorderBuffer = nn::audio::GetReleasedFinalOutputRecorderBuffer(&gameRecord);

    buffer->set_range(0, numFramesPerBuffer * frameSize);
    *out = buffer;

    void* pInBuffer = nn::audio::GetFinalOutputRecorderBufferDataPointer(pFinalOutputRecorderBuffer);
    size_t inSize = nn::audio::GetFinalOutputRecorderBufferDataSize(pFinalOutputRecorderBuffer);
    size_t size = buffer->range_length();
    if( inSize != size)
    {
        numDiffs++;
        const size_t bufferSize = nn::util::align_up(size, nn::audio::FinalOutputRecorderBuffer::SizeGranularity);
        nn::audio::SetFinalOutputRecorderBufferInfo(finalOutputRecorderBuffer, pInBuffer, bufferSize, size);
    }
    // NN_SDK_LOG("inSize %d , size %d \n", inSize, size);
    memcpy(buffer->data(), pInBuffer, buffer->range_length());
    nn::audio::AppendFinalOutputRecorderBuffer(&gameRecord, pFinalOutputRecorderBuffer);

     mNumFramesReceived += buffer->range_length();
    // Mute/suppress the recording sound
    int64_t timeUs;
    CHECK(buffer->meta_data()->findInt64(kKeyTime, &timeUs));
    int64_t elapsedTimeUs = timeUs - mStartTimeUs;
    if (elapsedTimeUs < kAutoRampStartUs) {
        memset((uint8_t *) buffer->data(), 0, buffer->range_length());
    } else if (elapsedTimeUs < kAutoRampStartUs + kAutoRampDurationUs) {
        int32_t autoRampDurationFrames =
                    ((int64_t)kAutoRampDurationUs * mSampleRate + 500000LL) / 1000000LL; //Need type casting

        int32_t autoRampStartFrames =
                    ((int64_t)kAutoRampStartUs * mSampleRate + 500000LL) / 1000000LL; //Need type casting

        int32_t nFrames = mNumFramesReceived - autoRampStartFrames;
        rampVolume(nFrames, autoRampDurationFrames,
                (uint8_t *) buffer->data(), buffer->range_length());
    }

    // Track the max recording signal amplitude.
    if (mTrackMaxAmplitude) {
        trackMaxAmplitude(
            (int16_t *) buffer->data(), buffer->range_length() >> 1);
    }

    return OK;
}

void SfAudioSource::trackMaxAmplitude(int16_t *data, int nSamples) {
    for (int i = nSamples; i > 0; --i) {
        int16_t value = *data++;
        if (value < 0) {
            value = -value;
        }
        if (mMaxAmplitude < value) {
            mMaxAmplitude = value;
        }
    }
}

int16_t SfAudioSource::getMaxAmplitude() {
    // First call activates the tracking.
    if (!mTrackMaxAmplitude) {
        mTrackMaxAmplitude = true;
    }
    int16_t value = mMaxAmplitude;
    mMaxAmplitude = 0;
    ALOGV("max amplitude since last call: %d", value);
    return value;
}

