﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
 /*
 * Copyright 2013 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define LOG_TAG "NvAudioEncoder"
#include "NvAudioEncoder.h"
//#define LOG_NDEBUG = 0;
static const void *sRefBaseOwner;

NvAudioEncoder::NvAudioEncoder() {
    // aac encoder format
    kMimeTypeAac = "audio/mp4a-latm";
    kBitRateMode = "CBR";       // CBR/CQ

    kBitRate = 1536000;
    kSampleRate = 48000;
    kChannelCount = 2;
    kAacProfile = 2;         // ProfileLevel: ObjectLC = 2, ObjectHE = 5, ObjectELD = 39;
    kMaxTimeLimitSec = 180;  // 3 minutes

    aEncoder = false;
    pcmSrc = false;
    fakeAudioSrc = true;

    gTimeLimitSec = kMaxTimeLimitSec;
    gFrameBufSize = 4096;
    gWavHeaderSize = 0x68;

    wavData = NULL;
    dataSize = 0;
    gDataOffset = 0;

    audioSource = NULL;

    gStopRequested = false;
    gOutputFormat = FORMAT_MP4;
    mStopCalled = false;
}
NvAudioEncoder:: ~NvAudioEncoder() {
    //destroy
    if (mAudioThread.get()) {
        NN_LOG_V("ScoopAudio: Closing audio thread\n");
        mAudioThread->join();
        mAudioThread.clear();
        mAudioThread = NULL;
        NN_LOG_V("ScoopAudio: Closed audio thread\n");
    }
}
/**
 * Generates the presentation time for frame N, in microseconds.
 */
/*static long computePresentationTime(int frameIndex, int frameRate) {
    return frameIndex * 1000000 / frameRate;
    //return frameIndex
}*/

status_t NvAudioEncoder::start(NvMediaRecorder *aRecorder, fnEvent evenClk) {
    mAudioThread = new NvAudioEncoder::AudioRecordThread();
    mMediaRecorder = aRecorder;
    mAudioThread->mEncoder = this;
    mEventCallback = evenClk;
    mAudioThread->run( "AudioRecordThread" );
    return NO_ERROR;
}
status_t NvAudioEncoder::stop() {
    NN_LOG_V("ScoopAudio :: Stopped called");
    mStopCalled = true;
    return NO_ERROR;
}

void NvAudioEncoder::AudioRecordThread::requestExit() {
    Thread::requestExit();
}
bool NvAudioEncoder::AudioRecordThread::threadLoop()
{
    NN_LOG_V("ScoopAudio:: preparing audio encoder in threadloop\n");
    sp<MediaCodec> audioEnc = mEncoder->prepareAudioEncoder();

    NN_LOG_V("ScoopAudio:: audio encoder is ready\n");
    mEncoder->createAudioSource();
    // audio
    NN_LOG_V("ScoopAudio:: start audio encoder in threadloop\n");
    status_t err = mEncoder->runAudioEncoder(audioEnc, /*muxer*/NULL);
    if (err != NO_ERROR){
        NN_LOG_E("ScoopAudio:: encoder run fails\n");
        return false;
    }

    mEncoder->destroyAudioSource();
    audioEnc->release();
    audioEnc.clear();
    NN_LOG_V("ScoopAudio:: audio thread exit...\n");
    return false;
}

sp<MediaCodec> NvAudioEncoder::prepareAudioEncoder(){

    status_t err;
    sp<AMessage> format = new AMessage;

    format->setString("mime", kMimeTypeAac);
    format->setInt32("aac-profile", kAacProfile);
    format->setInt32("encoder", true);
    format->setInt32("bitrate", kBitRate);
    format->setInt32("sample-rate", kSampleRate);
    format->setInt32("channel-count", kChannelCount);
    format->setString("bitrate-mode", kBitRateMode);

    NN_LOG_V("ScoopAudio:: New Looper\n");
    sp<ALooper> looper = new ALooper;
    looper->setName("audiorecord_looper");
    looper->start();
    NN_LOG_V("ScoopAudio:: Creating codec\n");
    sp<MediaCodec> mAudioEncoder = MediaCodec::CreateByType(looper, kMimeTypeAac, true);

    if (mAudioEncoder == NULL) {
        NN_LOG_V( "ScoopAudio:: ERROR: unable to create %s codec instance\n",
                kMimeTypeAac);
        return mAudioEncoder;
    }
    NN_LOG_V("ScoopAudio:: configuring\n");
    err = mAudioEncoder->configure(format, NULL, NULL,
            MediaCodec::CONFIGURE_FLAG_ENCODE);
    if (err != NO_ERROR) {
        NN_LOG_V( "ScoopAudio:: ERROR: unable to configure %s codec at (err=%d)\n",
                kMimeTypeAac, err);
        mAudioEncoder->release();
        return mAudioEncoder;
    }

    NN_LOG_V("ScoopAudio:: Starting codec\n");
    err = mAudioEncoder->start();
    if (err != NO_ERROR) {
        NN_LOG_V( "ScoopAudio:: ERROR: unable to start codec (err=%d)\n", err);
        mAudioEncoder->release();
        return mAudioEncoder;
    }
    NN_LOG_V("ScoopAudio:: Codec prepared\n");
    return mAudioEncoder;
}
//#endif

status_t NvAudioEncoder::runAudioEncoder(const sp<MediaCodec>& encoder,
                                const sp<MediaMuxer>& muxer) {
    Vector<sp<ABuffer> > mInBuffers;
    Vector<sp<ABuffer> > mOutBuffers;

    static int kTimeout = 250000;
    int trackIdx = -1;
    uint32_t debugNumFrames = 0;
    status_t err;
    size_t index;

    bool inputDone = true;
    bool outputDone = false;
    int generateIndex = 0;
    // need to get the audio source via generateAudioFrame()
    uint8_t *frameData = new uint8_t[gFrameBufSize];
    long ts = 0;

    NN_LOG_V("ScoopAudio:: entered runAudioEncoder\n");

    err = encoder->getInputBuffers(&mInBuffers);
    if (err != NO_ERROR) {
        NN_LOG_V("\nScoopAudio:: Unable to get input buffers (err=%d)\n", err);
        return err;
    }
    NN_LOG_V("ScoopAudio:: entered runAudioEncoder getOutputBuffers\n");
    err = encoder->getOutputBuffers(&mOutBuffers);
    if (err != NO_ERROR) {
        NN_LOG_V("\nScoopAudio:: Unable to get output buffers (err=%d)\n", err);
        return err;
    }
    NN_LOG_V("\nScoopAudio:: Got %d input and %d output buffers",mInBuffers.size(), mOutBuffers.size());

    while (!outputDone) {
        if(!inputDone) {
            err = encoder->dequeueInputBuffer(&index,kTimeout);
            if (err == OK) {
                ts += ((gFrameBufSize / 4) * 1E6) / kSampleRate; //(frameSize/4)/SampleRate
                //ptsUsec = ts;
                long ptsUsec = ts;//computePresentationTime(generateIndex, 40 /*gFrameRate*/);
                NN_LOG_V("ScoopAudio:: Get audio system time : %lld\n", ptsUsec);
                //if (generateIndex == NUM_FRAMES) {
                if (mStopCalled) {
                    // Send an empty frame with the end-of-stream flag set.  If we set EOS
                    // on a frame with data, that frame data will be ignored, and the
                    // output will be short one frame.

                    NN_LOG_V("ScoopAudio:: Sending audio EOS as audio is done\n");
                    err = encoder->queueInputBuffer(
                                index,
                                0 /* offset */,
                                0 /* size */,
                                ptsUsec,
                                MediaCodec::BUFFER_FLAG_EOS);
                    inputDone = true;
                    if (pcmSrc)
                        delete(static_cast<char*>(wavData));
                } else {
                    NN_LOG_V("\nScoopAudio:: Filling input buffer %d\n", index);
                    // generate audio frame data from source
                    generateAudioFrame(generateIndex, frameData);
                    const sp<ABuffer> &buffer = mInBuffers.itemAt(index);
                    NN_LOG_V("\nScoopAudio:: audio Buffer capacity: %d while Frame buffer size: %d\n", buffer->capacity(), gFrameBufSize);
                    if(buffer->capacity() < gFrameBufSize){
                        NN_ASSERT(false &&"Buffer capacity is less than required\n");
                    }
                    // write in buffer frameData
                    // Need to set frame data size
                    NN_LOG_V("\ncopy audio frameData");
                    memset(buffer->data(), 0, gFrameBufSize);
                    memcpy((uint8_t *)buffer->data(),frameData, gFrameBufSize);
                    uint32_t bufferFlags = 0;

                    NN_LOG_V("\nScoopAudio:: queue audio frameData");
                    err = encoder->queueInputBuffer(
                            index,
                            0 /* offset */,
                            buffer->size(),
                            ptsUsec,
                            bufferFlags);
                }
                generateIndex++;
                NN_LOG_V("\nScoopAudio:: Gernerated Frame Index:%d\n", generateIndex);
            } else {
                 NN_LOG_V("ScoopAudio:: **filling input buffer error %d\n**", index);
            }
        }
        gStopRequested = false;
        if(!gStopRequested) {
            size_t bufIndex, offset, size;
            int64_t ptsUsec;
            uint32_t flags;
            NN_LOG_V("\nScoopAudio:: Calling dequeueOutputBuffer\n");
            err = encoder->dequeueOutputBuffer(&bufIndex,
                                               &offset,
                                               &size,
                                               &ptsUsec,
                                               &flags,
                                               kTimeout);
            NN_LOG_V("\nScoopAudio:: dequeueOutputBuffer returned %d\n", err);
            inputDone = false;
            switch (err) {
            case NO_ERROR:
                NN_LOG_V("\nScoopAudio:: Got codec buffer (%zu bytes)\n", size);
                if ((flags & MediaCodec::BUFFER_FLAG_CODECCONFIG) != 0) {
                    NN_LOG_V("\nScoopAudio:: BUFFER_FLAG_CODECCONFIG (%zu bytes)\n", size);
                    if (muxer != NULL) {
                        // ignore this -- we passed the CSD into MediaMuxer when
                        // we got the format change notification
                        size = 0;
                    }
                }
                if (size != 0) {
                    if(trackIdx == -1){
                        NN_ASSERT(false && "ScoopAudio:: ******TrackIdx != -1 error\n");
                    }
                    NN_LOG_V("\nScoopAudio:: Set System time :%d\n",ptsUsec);
                    NN_LOG_V("ScoopAudio:: Got data in buffer %zu, size=%zu, pts=%lld\n",bufIndex, size, ptsUsec);

                    {
                        err = mMediaRecorder->feedAudioData(mOutBuffers[bufIndex],ptsUsec, flags);
                        if (err != NO_ERROR) {
                            NN_LOG_V("ScoopAudio:: Failed feeding data to media recorder (err=%d)\n\n", err);
                            return err;
                        }
                }
                }
                err = encoder->releaseOutputBuffer(bufIndex);
                if (err != NO_ERROR) {
                    NN_LOG_V("\nScoopAudio:: Unable to release output buffer (err=%d)\n",err);
                    return err;
                }
                if ((flags & MediaCodec::BUFFER_FLAG_EOS) != 0) {
                    // Not expecting EOS from SurfaceFlinger.  Go with it.
                    NN_LOG_V("\nScoopAudio:: Received end-of-stream");
                    gStopRequested = true;
                    outputDone = true;
                }
                break;
            case -EAGAIN:                       // INFO_TRY_AGAIN_LATER
                NN_LOG_V("ScoopAudio:: Got -EAGAIN, looping ..Try again later");//*
                break;
            case INFO_FORMAT_CHANGED:           // INFO_OUTPUT_FORMAT_CHANGED//*
                {
                    // Format includes CSD, which we must provide to muxer.
                    NN_LOG_V("\nAudio Encoder format changed");
                    sp<AMessage> newFormat;
                    encoder->getOutputFormat(&newFormat);
                    mMediaRecorder->setEncoderFormat(newFormat, true /*isAudio*/, &trackIdx);
                    NN_LOG_V("ScoopAudio:: audio Encoder track index - %d\n", trackIdx);
                }
                break;
            case INFO_OUTPUT_BUFFERS_CHANGED:   // INFO_OUTPUT_BUFFERS_CHANGED //*
                // Not expected for an encoder; handle it anyway.
                NN_LOG_V("\nScoopAudio:: Encoder buffers changed");
                err = encoder->getOutputBuffers(&mOutBuffers);
                if (err != NO_ERROR) {
                    NN_LOG_V("\nScoopAudio:: Unable to get new output buffers (err=%d)\n", err);
                    return err;
                }
                break;
            case INVALID_OPERATION:
                NN_LOG_V("\nScoopAudio:: dequeueOutputBuffer returned INVALID_OPERATION");
                return err;
            default:
                NN_LOG_V("\nScoopAudio:: Got weird result %d from dequeueOutputBuffer\n", err);
                return err;
            }
        }
    }
    //gAudioDone = true;

    NN_LOG_V("ScoopVideo:: Sending EOS of input stream.. 1\n");
    encoder->signalEndOfInputStream();

    mEventCallback(2); //SV_AUDIO_END

    NN_LOG_V("ScoopAudio:: AAC Encoder stopping (req=%d)", gStopRequested);
    return NO_ERROR;
}//NOLINT(impl/function_size)

void NvAudioEncoder::createAudioSource() {
    audioSource = new SfAudioSource(kSampleRate, kChannelCount);
    audioSource->start();
}

void NvAudioEncoder::destroyAudioSource() {
    if (audioSource != NULL) {
        audioSource->stop();
    }
}

/**
 *  create audio frame date via different options
 */
void NvAudioEncoder::generateAudioFrame(int frameIndex, uint8_t* frameData)
{
    NN_LOG_V("\nGenerate Audio Frame data...\n");
    if (fakeAudioSrc) {
        // from fAudioSource read()
        MediaBuffer *Mbuffer;
        audioSource->read(&Mbuffer, NULL);
        memcpy(frameData, Mbuffer->data(), gFrameBufSize);
        Mbuffer->release();
        Mbuffer = NULL;
    }
    else if (pcmSrc) {
        // read from pcm data to frameData (dataSize/wavData)
        if ((dataSize - gWavHeaderSize - gDataOffset) >= gFrameBufSize) {
            memcpy(frameData, ((uint8_t*)wavData + gWavHeaderSize + gDataOffset), gFrameBufSize);
            gDataOffset += gFrameBufSize;
        }
        else {   // last chunk of the audio data in wav file data
            memcpy(frameData, ((uint8_t*)wavData + gWavHeaderSize + gDataOffset), (dataSize - gWavHeaderSize - gDataOffset));
        }
    }
    else {
    // or read audio source input
        ;
    }

    return;
}

/**
 * Since MediaCodec encoder generates the raw AAC stream which needs
 *  to be converted to a playable format, for example ADTS stream
 */
void NvAudioEncoder::addADTSHeader(char* buf, int packetLen)
{

    int profile = 2;     // AAC LC
    int freqIndex = 5;   // 48kHz
    int chanConfig = 2;  // CPE

    buf[0] = 0xFF;
    buf[1] = 0xF9;
    buf[2] = ( ((profile - 1)<<6) + (freqIndex<<2) + (chanConfig>>2) );
    buf[3] = ( ((chanConfig&3)<<6) + (packetLen>>11) );
    buf[4] = ( (packetLen&0x7FF) >> 3 );
    buf[5] = ( ((packetLen&7) << 5) + 0x1F );
    buf[6] = 0xFC;
}

// Modified from AudioMemoryPool sample
std::size_t NvAudioEncoder::aacReadWavFile(void** data, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());
    NN_LOG_V("\nWav file size: %d", size);
    *data = new char[size];
    NN_ABORT_UNLESS_NOT_NULL(*data);

    result = nn::fs::ReadFile(handle, 0, *data, size);
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    NN_LOG_V("\nWav file size: %d", size);
    return static_cast<std::size_t>(size);
}


