﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/audio.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include "AudioOutRenderer.h"

AudioOutRenderer::AudioOutRenderer()
{
    for( int i = 0; i < numBuffers; ++ i )
    {
        m_OutBuffers[i] = NULL;
    }
    m_AudioRendererThreadstacksize = 1024 * 128;
    inAudioBufferSize = 32 * 1024;
    inAudioBuffer = new char [inAudioBufferSize];
}

AudioOutRenderer::~AudioOutRenderer()
{
    if( m_AudioRendererThreadstack != NULL )
    {
        delete [] m_AudioRendererThreadstack;
    }

    for( int i = 0; i < numBuffers; ++i )
    {
        if( m_OutBuffers[i] != NULL )
        {
            delete [] (char*)m_OutBuffers[i];
        }
    }
    if( inAudioBuffer != NULL )
    {
        delete [] inAudioBuffer;
    }
}

movie::Status AudioOutRenderer::Initialize()
{
    nn::Result nnResult = nn::ResultSuccess();
    movie::Status movieStatus = movie::Status_UnknownError;
    m_Channels = 0;
    m_SampleRate = 0;
    m_FrameSize = sizeof(short);
    m_ThreadDone = false;
    m_Started = false;
    m_PresentationTimeUs = 0ll;
    m_RenderIntervalMs = 5;
    m_SampleSize = 16;
    m_AudioOutStarted = false;
    m_BufferSize = 8192 * 4;
    m_AudioOutOpened = false;
    m_RenderThreadCreated = false;
    m_AudioRendererThreadstack = NULL;
    nn::os::InitializeMutex( &m_AudioBufferVectorMutex, false, 0 );

    m_AudioRendererThreadstack = new char [m_AudioRendererThreadstacksize + 4096];
    void *mem = (void*) m_AudioRendererThreadstack;
    void *alignedStackPointer = (void*)(((uintptr_t)mem + 4095) & ~ (uintptr_t)0x0FFF);

    nnResult = nn::os::CreateThread(
            &m_AudioRendererThreadType,
            &AudioOutRendererThreadFunction,
            (void*)this,
            alignedStackPointer,
            m_AudioRendererThreadstacksize,
            nn::os::DefaultThreadPriority / 2 );
    m_RenderThreadCreated = true;
    if( nnResult.IsSuccess() )
    {
        movieStatus = movie::Status_Success;
    }

    return movieStatus;
}

movie::Status AudioOutRenderer::Finalize()
{
     nn::os::FinalizeMutex( &m_AudioBufferVectorMutex );
     return movie::Status_Success;
}

int32_t AudioOutRenderer::GetRenderInterval()
{
    return m_RenderIntervalMs;
}

movie::Status AudioOutRenderer::Open(movie::Decoder* audioDecoder, uint32_t sampleRate, int channelCount)
{
    NN_SDK_LOG("AudioOutRenderer::open() \n");
    nn::Result nnResult = nn::ResultSuccess();
    movie::Status movieStatus = movie::Status_UnknownError;

    m_InputSampleRate = sampleRate;
    m_InputChannels = channelCount;

    m_AudioDecoder = audioDecoder;
    m_SampleRate = 48000;
    m_Channels = 2;

    m_FrameSize = m_Channels * sizeof(short);
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    parameter.sampleRate = m_SampleRate;

    nnResult = nn::audio::OpenDefaultAudioOut(&m_AudioOut, parameter);
    if( nnResult.IsSuccess() )
    {
        if ((m_Channels != nn::audio::GetAudioOutChannelCount(&m_AudioOut)))
        {
            NN_SDK_LOG("AudioOutRenderer::open() failed channels: %d \n", m_Channels);
        }
        movieStatus = movie::Status_Success;
    }
    else
    {
        NN_SDK_LOG("AudioOutRenderer::open() failed sampleRate: %d \n", sampleRate);
    }

    m_SampleFormat = nn::audio::GetAudioOutSampleFormat(&m_AudioOut);
    m_SampleSize = nn::audio::GetSampleByteSize(m_SampleFormat);
    m_AudioOutOpened = true;

    return movieStatus;
}

movie::Status AudioOutRenderer::Start()
{
    NN_SDK_LOG("AudioOutRenderer::Start() \n");

    nn::Result nnResult = nn::ResultSuccess();
    movie::Status movieStatus = movie::Status_UnknownError;

    nn::os::StartThread( &m_AudioRendererThreadType );
    size_t dataSize = 1024;

    int channelCount = nn::audio::GetAudioOutChannelCount(&m_AudioOut);
    NN_UNUSED(channelCount);
    int sampleRate = nn::audio::GetAudioOutSampleRate(&m_AudioOut);
    NN_UNUSED(sampleRate);
    nn::audio::SampleFormat sampleFormat = nn::audio::GetAudioOutSampleFormat(&m_AudioOut);
    NN_UNUSED(sampleFormat);
    for( int i = 0; i < numBuffers; ++i )
    {
        m_OutBuffers[i] =  new char [m_BufferSize];
        memset(m_OutBuffers[i], 0, dataSize);
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], m_OutBuffers[i], m_BufferSize, dataSize);
        nn::audio::AppendAudioOutBuffer(&m_AudioOut, &audioOutBuffer[i]);
    }
    nnResult = nn::audio::StartAudioOut(&m_AudioOut);
    if( nnResult.IsSuccess() )
    {
        movieStatus = movie::Status_Success;
    }
    m_AudioOutStarted = true;
    m_Started = true;
    return movieStatus;
}

movie::Status AudioOutRenderer::Render()
{
    movie::Status movieStatus = movie::Status_Success;

    audioData *pcmData;
    void* buffer = NULL;
    size_t size = 0;
    size_t dataSize = 0;
    movie::Buffer pcmBufferData;

    nn::os::LockMutex( &m_AudioBufferVectorMutex );
    int pcmListSize = m_AudioCodecBuffers.size();
    if( pcmListSize > 0 )
    {
        pcmData = &m_AudioCodecBuffers.front();
        pcmBufferData.SetDataAndCapacity(inAudioBuffer, inAudioBufferSize);
        pcmBufferData.SetRange(0, inAudioBufferSize);
        if( m_AudioDecoder != NULL )
        {
            m_AudioDecoder->GetOutputBuffer(pcmData->index, &pcmBufferData);
        }
        buffer = inAudioBuffer;
        size = pcmBufferData.Size();
    }
    else
    {
        nn::os::UnlockMutex( &m_AudioBufferVectorMutex );
        return movieStatus;
    }
    nn::os::UnlockMutex( &m_AudioBufferVectorMutex );

    if( ( buffer != NULL ) && ( size > 0 ) )
    {
        nn::audio::AudioOutBuffer* audioOutBuffer = nullptr;
        audioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&m_AudioOut);
        if( audioOutBuffer == NULL )
        {
            return movieStatus;
        }
        void* outBuffer = nn::audio::GetAudioOutBufferDataPointer(audioOutBuffer);
        size_t outSize = nn::audio::GetAudioOutBufferDataSize(audioOutBuffer);
        NN_UNUSED(outSize);
        m_PresentationTimeUs = pcmData->presentationTimeUs;

        bool copyToOutBuffer = true;
        if( m_InputChannels < m_Channels )
        {
            int16_t* pIn = static_cast< int16_t* >( buffer );
            int16_t* pOut = static_cast< int16_t* >( outBuffer );
            int32_t outSamples = size / 2;
            for( int32_t index = 0; index < outSamples; index++ )
            {
                for( int32_t channel = 0; channel < m_Channels; channel++ )
                {
                    pOut[ channel ] = pIn[ index ];
                }
                pOut = &pOut[ m_Channels ];
            }
            dataSize = size * m_Channels;
            copyToOutBuffer = false;
        }
        if( copyToOutBuffer == true )
        {
            dataSize = size;
            memcpy(outBuffer, buffer, dataSize);
        }
        nn::audio::SetAudioOutBufferInfo(audioOutBuffer, outBuffer, m_BufferSize, dataSize);
        nn::audio::AppendAudioOutBuffer(&m_AudioOut, audioOutBuffer);
        if( m_AudioDecoder != NULL )
        {
            m_AudioDecoder->ReleaseOutputBufferIndex(pcmData->index);
        }
        nn::os::LockMutex( &m_AudioBufferVectorMutex );
        m_AudioCodecBuffers.erase(m_AudioCodecBuffers.begin());
        nn::os::UnlockMutex( &m_AudioBufferVectorMutex );
    }
    else
    {
        if( m_AudioDecoder != NULL )
        {
            m_AudioDecoder->ReleaseOutputBufferIndex(pcmData->index);
        }
        nn::os::LockMutex(&m_AudioBufferVectorMutex);
        m_AudioCodecBuffers.erase(m_AudioCodecBuffers.begin());
        nn::os::UnlockMutex(&m_AudioBufferVectorMutex);
    }

    return movieStatus;
}

void AudioOutRenderer::Stop()
{
    nn::os::LockMutex( &m_AudioBufferVectorMutex );
    audioData *pcmData;
    int pcmListSize = m_AudioCodecBuffers.size();
    for(int i=0; i< pcmListSize; i++)
    {
        pcmData = &m_AudioCodecBuffers.front();
        m_AudioDecoder->ReleaseOutputBufferIndex(pcmData->index);
        m_AudioCodecBuffers.erase(m_AudioCodecBuffers.begin());
    }
    nn::os::UnlockMutex( &m_AudioBufferVectorMutex );
}

bool AudioOutRenderer::DoneProcessing()
{
    return m_ThreadDone;
}

bool AudioOutRenderer::IsAudioOutStarted()
{
    return m_AudioOutStarted;
}

void AudioOutRenderer::Close()
{
    m_ThreadDone = true;
    if( m_RenderThreadCreated == true )
    {
        nn::os::WaitThread( &m_AudioRendererThreadType );
        nn::os::DestroyThread( &m_AudioRendererThreadType );
        m_RenderThreadCreated = false;
    }

    if( m_AudioOutOpened == true )
    {
        nn::audio::StopAudioOut(&m_AudioOut) ;
        nn::audio::CloseAudioOut(&m_AudioOut);
        m_AudioOutOpened = false;
    }

    audioData *pcmData;
    int pcmListSize = m_AudioCodecBuffers.size();
    for(int i=0; i< pcmListSize; i++)
    {
        pcmData = &m_AudioCodecBuffers.front();
        m_AudioDecoder->ReleaseOutputBufferIndex(pcmData->index);
        m_AudioCodecBuffers.erase(m_AudioCodecBuffers.begin());
    }

}

void AudioOutRenderer::OnOutputAvailable(int index, int64_t presentationTimeUs, uint32_t flags)
{
    audioData pcmData;
    pcmData.index = index;
    pcmData.presentationTimeUs = presentationTimeUs;
    pcmData.flags = flags;

    nn::os::LockMutex( &m_AudioBufferVectorMutex );
    m_AudioCodecBuffers.push_back(pcmData);
    nn::os::UnlockMutex( &m_AudioBufferVectorMutex );
}

void AudioOutRendererThreadFunction(void *arg)
{
    AudioOutRenderer *audioOutRenderer = (AudioOutRenderer*) arg;
    for (;;)
    {
        if( audioOutRenderer->DoneProcessing() )
        {
            break;
        }
        if( true == audioOutRenderer->IsAudioOutStarted() )
        {
            audioOutRenderer->Render();
        }
        int renderIntervalMs = audioOutRenderer->GetRenderInterval();
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( renderIntervalMs ) );
    }
}

