﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "MediaPlayerObserver.h"
#include "MediaPlayerUtilities.h"
#include "HeapTracker.h"
#include "CommandLineOptions.h"
#include "PausableLoop.h"
#include "FileHandling.h"

#include <experimental/memory_resource>

#define SOL_USE_EXPERIMENTAL_PMR_STRING

#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wunused-private-field"
#pragma clang diagnostic ignored "-Wnull-dereference"
#include <sol/sol.hpp>
#pragma clang diagnostic pop

#include <cstdint>
#include <mutex>
#include <chrono>
#include <string>
#include <cinttypes>
#include <thread>
#include <random>
#include <utility>

#include <movie/Utils.h>
#include <movie/Player.h>

#include "ThreadWrapper.h"

#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/hid/hid_Npad.h>

#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>

#include <nn/lmem/lmem_ExpHeap.h>

#ifdef MOUNT_SDCARD
    #include <nn/fs/fs_SdCardForDebug.h>
#endif

#ifdef ENABLE_PROFILER
    #include <nn/profiler.h>
#endif

///////////////////////////////////////////////////////////////

constexpr size_t g_GraphicsSystemMemorySize = 8 * 1024 * 1024;
uint8_t g_GraphicsHeap[g_GraphicsSystemMemorySize] __attribute__(( aligned(4096) ));
bool USE_HEAP_TRACKING{};

///////////////////////////////////////////////////////////////

extern "C" void nninitStartup()
{
    constexpr size_t heapSize = 256 * 1024 * 1024;
    constexpr size_t fsHeapSize = 512 * 1024;

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::SetMemoryHeapSize(heapSize) );
    static uint8_t g_FsHeapBuffer[fsHeapSize];
    static nn::lmem::HeapHandle g_FsHeap = nn::lmem::CreateExpHeap(g_FsHeapBuffer, fsHeapSize, nn::lmem::CreationOption_DebugFill);
    nn::fs::SetAllocator([](size_t sz){ return nn::lmem::AllocateFromExpHeap(g_FsHeap, sz); },
                         [](void* p, size_t){ return nn::lmem::FreeToExpHeap(g_FsHeap, p); });
}

extern "C" void nnMain()
{
    ///////////////////////////////////////////////////////////////
    NN_UTIL_SCOPE_EXIT{ if (USE_HEAP_TRACKING) {
                            MMHeap().OutputUsage();
                            CoreHeap().OutputUsage();
                            MallocHeap().OutputUsage();
                            NewHeap().OutputUsage();
                        }};
    ///////////////////////////////////////////////////////////////

    { // We use an embedded scope to make sure local destructors have been called
      // before we print our final heap tracking statistics

#ifdef ENABLE_PROFILER
    std::vector<char> profiler_buffer__(nn::profiler::MinimumBufferSize);
    nn::profiler::Initialize(profiler_buffer__.data(), profiler_buffer__.size());
    NN_UTIL_SCOPE_EXIT { nn::profiler::Finalize(); };
#endif

    // The default sol2 library uses static local std::strings in various places, and they are not
    // destructed at application exit in our environment; so they appear as leaks to our heap tracker.
    // To work around this I've modified sol2 to be able use the polymorphic_allocator string in std::pmr::string.
    // This allows us to specify a default allocator (as we do below) that uses our own custom untracked allocator.
    // (This turns out to be useful for other things we don't want to track as well, like the pmr::vector<> playlist in
    // CommandLineOptions.)

    movie::sample::UntrackedAllocator sneaky_allocator____{2};
    std::experimental::pmr::set_default_resource(&sneaky_allocator____);

    ///////////////////////////////////////////////////////////////

    movie::sample::CommandLineOptions options_{ nn::os::GetHostArgv(), nn::os::GetHostArgc(), sneaky_allocator____ };

    if (!options_.valid()) {
        NN_SDK_LOG("%s",options_.usage_string());
        return;
    }

    USE_HEAP_TRACKING = options_.useHeapTracking;

    // To handle various cleanup cases we keep a vector of calls that are executed at scope exit..
    std::vector<movie::sample::ScopeExit> exit_calls_;

    //nn::hid::InitializeDebugPad();

    ///////////////////////////////////////////////////////////////
    movie::SetAllocator(mmAllocate, mmDeallocate, mmReallocate, &MMHeap());
    nv::SetGraphicsServiceName("nvdrv:t");
    nv::SetGraphicsAllocator(mmAllocate, mmDeallocate, mmReallocate, &CoreHeap());
    nv::SetGraphicsDevtoolsAllocator(mmAllocate, mmDeallocate, mmReallocate, &CoreHeap());
    nv::InitializeGraphics(g_GraphicsHeap, g_GraphicsSystemMemorySize);

    exit_calls_.push_back([]{ NN_SDK_LOG("Cleanup: all done.. \n"); });
    exit_calls_.push_back([]{ nv::FinalizeGraphics();
                              NN_SDK_LOG("Cleanup: nv::FinalizeGraphics()\n");
                              });

    ///////////////////////////////////////////////////////////////

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());
    exit_calls_.push_back([]{ nn::fs::UnmountHostRoot(); });

    std::vector<std::string> additional_filenames___;

    if (!options_.playlist_file_.empty()) {
        if (options_.playlist_file_.find(':') == options_.playlist_file_.npos) {
            additional_filenames___ = movie::sample::get_lines_from_string( movie::sample::get_file_as_string(
                                        std::string{options_.working_folder_}.append(options_.playlist_file_).c_str() ));
        } else {
            additional_filenames___ = movie::sample::get_lines_from_string( movie::sample::get_file_as_string(
                                        std::string{options_.playlist_file_}.c_str() ));
        }

        for (auto&& s : additional_filenames___) {
            if (s.find(':') == s.npos) {
                s.insert(begin(s),
                         begin(options_.working_folder_), end(options_.working_folder_));
            }
        }

        options_.playlist_.insert(end(options_.playlist_),
                                  begin(additional_filenames___), end(additional_filenames___));
    }

    for (auto&& e : options_.playlist_) {
        NN_SDK_LOG("Playlist entry: %s\n", std::string{e}.c_str());
    }

    auto unique_uri_schemes_ = movie::sample::get_unique_uri_schemes(options_.playlist_);

    for (auto&& e : unique_uri_schemes_) {
        exit_calls_.push_back( movie::sample::get_mount_handler(e) );
    }

    std::string my_script_ = (options_.lua_file_.find(':') == options_.lua_file_.npos)
                             ? movie::sample::get_file_as_string(std::string{options_.working_folder_}.append(options_.lua_file_).c_str())
                             : movie::sample::get_file_as_string(std::string{options_.lua_file_}.c_str());

    ///////////////////////////////////////////////////////////////

    movie::PlayerConfig config;
    config.returnAudioDataToClient = false;
    config.returnVideoDataToClient = true;
    config.videoDecodeMode = options_.videoDecodeMode;
    config.returnSyncedData = true;
    config.useClientMemoryForOutput = true;
    config.autoReleaseOutputBuffers = true;
    config.audioFormat = movie::OutputFormat_AudioPcm16;
    config.videoFormat = options_.videoOutputFormat;
    config.maxResolution = options_.maxPlayerRes;

    movie::BrowserConfig browserConfig;
    browserConfig.coreMask = 0x00;
    browserConfig.sfThreadPriority = 16;
    browserConfig.audioRendererThreadPriority = 16;
    browserConfig.videoRendererThreadPriority = 16;

    ///////////////////////////////////////////////////////////////

    MediaPlayerObserver mediaPlayerObserver{config};

    ///////////////////////////////////////////////////////////////

    sol::state lua;

    lua.open_libraries();

    int32_t trackCount{};
    int32_t width{};
    int32_t height{};

    movie::sample::pausable_loop rendering_thread_;

    ///////////////////////////////////////////////////////////////

    std::mt19937 rng_{options_.rng_seed_value_};
    NN_SDK_LOG("Seeding rng with value == (%u)\n", options_.rng_seed_value_);

    ///////////////////////////////////////////////////////////////

    auto get_filenames_ =
        [it_ = begin(options_.playlist_),
          end_ = end(options_.playlist_)]() mutable {
            if (it_ == end_) {
                return std::make_tuple(false, "");
            }
            auto str_ = it_++;
            return std::make_tuple(true, (str_->empty() ? "" : str_->data()));
        };

    auto get_duration_ =
        [](movie::BrowserPlayer& player_) {
            int64_t duration_{}; player_.GetPlaybackDuration(&duration_);
            return duration_;
        };

    auto prepare_ =
        [&](movie::BrowserPlayer& player_) {
            NN_SDK_LOG("LuaPlayer: preparing..\n");
            player_.Prepare();
            mediaPlayerObserver.WaitForPrepare();

            //player_.GetPlaybackDuration(&trackDuration);    // duration is in microseconds
            //trackDuration = trackDuration / 1000;           // convert to milliseconds

            player_.GetVideoDimensions(&width, &height);
            player_.GetTrackCount(&trackCount);

            for( int32_t t = 0; t < trackCount; t++)        // TODO more than two tracks?
            {
                movie::TrackType trackType{};
                player_.GetTrackType(t, &trackType);
                switch (trackType)
                {
                    case movie::TrackType_Audio:
                        mediaPlayerObserver.SetAudioTrackNumber(t);
                        break;
                    case movie::TrackType_Video:
                        if (options_.playVideo)
                            options_.isVideoTrack = true;
                        mediaPlayerObserver.SetVideoTrackNumber(t);
                        break;
                    case movie::TrackType_Subtitle: break;
                    case movie::TrackType_ClosedCaption: break;
                    default: break;
                }
            }
        };

    auto start_ =
        [&](movie::BrowserPlayer& player_) {
           NN_SDK_LOG("LuaPlayer: issuing async Start()\n");
           player_.Start();
           mediaPlayerObserver.WaitForStart();
           NN_SDK_LOG("LuaPlayer: Start() complete\n");
           if(options_.isVideoTrack) {
               player_.GetVideoDimensions(&width, &height);
           }
           rendering_thread_.resume();
        };

    auto playback_controller_ =
        [&](movie::BrowserPlayer& player_, int64_t max_playtime_, sol::variadic_args va_) {

            std::vector<sol::thread> runners_;

            for ([[maybe_unused]] auto&& e : va_) {
                runners_.emplace_back(sol::thread::create(lua.lua_state()));
            }
            std::vector<sol::state_view> views_;

            for (auto&& e : runners_) {
                views_.emplace_back(e.state());
            }

            std::vector<sol::coroutine> coros_;
            for (int i = 0; i < va_.size(); ++i) {
                coros_.emplace_back(views_[i], va_[i].as<sol::function>());
            }

            auto const stop_time_ = std::chrono::high_resolution_clock::now() + std::chrono::microseconds{max_playtime_};

            for (auto&& e : coros_) {
                e(player_); // call coroutines once to initialize them with their parameter
                std::this_thread::sleep_for(std::chrono::microseconds(100)); // give other threads a chance to breathe
                if (std::chrono::high_resolution_clock::now() > stop_time_ || mediaPlayerObserver.IsPlaybackComplete()) {
                    return;
                }
            }

            while (true) {
                for (auto&& e : coros_) {
                    if (e.valid()) {
                        e(); // parameter is no longer needed
                        std::this_thread::sleep_for(std::chrono::microseconds(100));
                    }
                    if (std::chrono::high_resolution_clock::now() > stop_time_ || mediaPlayerObserver.IsPlaybackComplete()) {
                        break;
                    }
                }
                std::this_thread::sleep_for(std::chrono::microseconds(100));
                if (std::chrono::high_resolution_clock::now() > stop_time_ || mediaPlayerObserver.IsPlaybackComplete()) {
                    break;
                }
            }
        };

    lua.new_usertype<movie::BrowserPlayer>("Player",
        "new",             sol::factories([&](){
                               movie::BrowserPlayer* player{};
                               movie::BrowserPlayer::Create(&player, &config, &browserConfig);
                               mediaPlayerObserver.SetMoviePlayer(*player);
                               player->SetObserver(&mediaPlayerObserver);
                               return player; }),
        "SetDataSource",   &movie::BrowserPlayer::SetDataSource,
        "Prepare",         prepare_,
        "Reset",           &movie::BrowserPlayer::Reset,
        "TerminateNetworkConnection", &movie::BrowserPlayer::TerminateNetworkConnection,
        "SeekTo",          [&](movie::BrowserPlayer& player_, int64_t time_) {
                                   NN_SDK_LOG("LuaPlayer: SeekTo(%" PRId64")\n",time_);
                                   player_.SeekTo(time_);
                           },//static_cast<movie::Status(movie::BrowserPlayer::*)(int64_t)>(&movie::BrowserPlayer::SeekTo), // This static_cast is needed to disambiguate SeekTo()
        "Stop",            [&](movie::BrowserPlayer& player_) {
                               NN_SDK_LOG("LuaPlayer: issuing async Stop()\n");
                               player_.Stop();
                               mediaPlayerObserver.WaitForStop();
                               NN_SDK_LOG("LuaPlayer: Stop() complete\n");
                               rendering_thread_.pause();
                           },
        "Start",           start_,
        "Pause",           &movie::BrowserPlayer::Pause,
        "GetDuration",     get_duration_,
        "SeekRandom",      [&](movie::BrowserPlayer& player_, int start_, int end_) {
                               auto seek_target_ = std::uniform_int_distribution<int64_t>{start_,end_}(rng_);
                               NN_SDK_LOG("LuaPlayer: SeekRandom(%d, %d): seeking to %" PRId64 "\n", start_, end_, seek_target_);
                               player_.SeekTo(seek_target_);
                           },
        "SetPlaybackRate", &movie::BrowserPlayer::SetPlaybackRate,
        "ControlSequence", playback_controller_,
        sol::base_classes, sol::bases<movie::Player>{},
        sol::meta_function::garbage_collect, sol::destructor([](auto...){})
    );

    lua.set_function("get_filenames", get_filenames_);
    lua.set_function("destroy", [](movie::BrowserPlayer* ptr_) { movie::BrowserPlayer::Destroy(ptr_); });
    lua.set_function("milliseconds", [](int t) { return std::chrono::duration_cast<std::chrono::microseconds>(std::chrono::milliseconds{t}).count(); });
    lua.set_function("microseconds", [](int t) { return std::chrono::duration_cast<std::chrono::microseconds>(std::chrono::microseconds{t}).count(); });
    lua.set_function("seconds",      [](int t) { return std::chrono::duration_cast<std::chrono::microseconds>(std::chrono::seconds{t}).count();      });

    lua.set_function("wait", [](int64_t us_) { nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(us_)); });

    movie::sample::thread_wrapper
        my_lua_thread_{[&]{
                auto err_handler_ = [](lua_State*, sol::protected_function_result result) {
                                         NN_SDK_LOG("LuaPlayer: error handler triggered: \n");
                                         for (auto&& e : result) {
                                             NN_SDK_LOG("%s\n", e.as<std::string>().c_str());
                                         }
                                         return result;
                                     };

                lua.script(my_script_.c_str(), err_handler_);

                //auto hook_ = [](lua_State* state_, lua_Debug* debug_) {
                //                 lua_sethook(state_, nullptr, 0, 0);
                //                 static int count_{};
                //                 NN_SDK_LOG("hook (%d) -->\n",count_);
                //                 if (count_++ > 5) {
                //                     NN_SDK_LOG("lua is yielding out.. %d\n", count_);
                //                     lua_yield(state_, -1);
                //                     //luaL_error(state_,"oops\n");
                //                 }
                //                 // alternative is to use a jumpbuf with setjmp/longj
                //             };

                sol::thread runner = sol::thread::create(lua);
                sol::state_view view(runner.state());
                sol::coroutine cr = view["main"];

                //lua_sethook(cr.lua_state(), hook_, LUA_MASKRET | LUA_MASKCOUNT, 5);

                NN_SDK_LOG("LuaPlayer: about to execute main script..\n");

                cr(); // we run the main script as a coroutine so we can yield out early if we need to

                NN_SDK_LOG("LuaPlayer: main script has exited..\n");
                rendering_thread_.stop();
            }};

    ///////////////////////////////////////////////////////////////

    auto const start_time_
        = std::chrono::high_resolution_clock::now();

    ///////////////////////////////////////////////////////////////

    rendering_thread_.loop(
    [&]{
        if( options_.isVideoTrack )
        {
            int yOffset = 0;
            int uvOffset = 0;
            int ystride = 0;
            int nativeWidth = 0;
            int nativeHeight = 0;
            int cropWidth = 0;
            int cropHeight = 0;
            int colorSpace = 0;
            int32_t bufferIndex = -1;
            int32_t trackNumber = -1;
            int32_t bufferIndexForProperties = -1;

            if( ( config.returnVideoDataToClient == true ) && ( config.useClientMemoryForOutput == true ) )
            {
                nn::os::LockMutex( &mediaPlayerObserver.m_VideoOutputBuffersListMutex );
                uint32_t videoListSize = mediaPlayerObserver.m_VideoOutputBuffersList.size();

                if( videoListSize > 0 )
                {
                    bool videoAvailableForRendering = false;
                    MediaPlayerObserver::OutputBufferInfo videoBuffer;

                    videoBuffer = mediaPlayerObserver.m_VideoOutputBuffersList.front();
                    mediaPlayerObserver.m_VideoOutputBuffersList.erase( mediaPlayerObserver.m_VideoOutputBuffersList.begin());

                    nn::os::UnlockMutex( &mediaPlayerObserver.m_VideoOutputBuffersListMutex );

                    bufferIndex = videoBuffer.bufferIndex;
                    trackNumber = videoBuffer.trackNumber;

                    if( config.videoDecodeMode == movie::VideoDecodeMode_Cpu )
                    {
                        bufferIndexForProperties = bufferIndex;
                    }
                    else
                    {
                        //make sure the BufferIndex is one of the valid ones
                        for( int i = 0; i < mediaPlayerObserver.m_videoBufferCount; ++i )
                        {
                            if( bufferIndex == mediaPlayerObserver.m_videoBufferIndexArray[ i ] )
                            {
                                bufferIndexForProperties = bufferIndex;
                                mediaPlayerObserver.m_PresentationIndex = bufferIndex;
                                videoAvailableForRendering = true;
                                break;
                            }
                        }
                    }

                    mediaPlayerObserver.GetOutputBufferProperties(bufferIndexForProperties, &mediaPlayerObserver.m_BufferProperty);
                    mediaPlayerObserver.m_BufferProperty.FindInt32("nv12-y-stride", &ystride);
                    mediaPlayerObserver.m_BufferProperty.FindInt32("nv12-y-offset", &yOffset);
                    mediaPlayerObserver.m_BufferProperty.FindInt32("nv12-uv-offset", &uvOffset);
                    mediaPlayerObserver.m_BufferProperty.FindInt32("width", &nativeWidth);
                    mediaPlayerObserver.m_BufferProperty.FindInt32("height", &nativeHeight);
                    mediaPlayerObserver.m_BufferProperty.FindInt32("crop-width", &cropWidth);
                    mediaPlayerObserver.m_BufferProperty.FindInt32("crop-height", &cropHeight);
                    mediaPlayerObserver.m_BufferProperty.FindInt32("nv12-colorspace", &colorSpace);
                    if( mediaPlayerObserver.m_CropWidth != cropWidth || mediaPlayerObserver.m_CropHeight != cropHeight )
                    {
                        mediaPlayerObserver.m_CropWidth = cropWidth;
                        mediaPlayerObserver.m_CropHeight = cropHeight;
                        if( config.videoFormat == movie::OutputFormat_VideoColorNv12 )
                        {
                            NN_SDK_LOG("Video resolution changed, ResizeTextures : WxH %dx%d ystride %d\n", cropWidth, cropHeight, ystride);
                            if( ( cropWidth > 0 ) && ( cropHeight > 0 ) )
                            {
                                mediaPlayerObserver.ResizeTextures(cropWidth, cropHeight);
                            }
                        }
                    }

                    if( videoAvailableForRendering )
                    {
                        mediaPlayerObserver.DrawVideoNvn(mediaPlayerObserver.m_PresentationIndex, cropWidth, cropHeight, yOffset, uvOffset, ystride, colorSpace);
                    }
                }
                else
                {
                    nn::os::UnlockMutex( &mediaPlayerObserver.m_VideoOutputBuffersListMutex );
                }
            }
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
    });

    ///////////////////////////////////////////////////////////////

    auto playback_duration_
        = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::high_resolution_clock::now() - start_time_).count();
    NN_SDK_LOG("\n Total playback duration: %" PRId64 " msecs\n", playback_duration_);

    ///////////////////////////////////////////////////////////////

}   // end of embedded scope
}   //NOLINT(impl/function_size)
