﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <thread>
#include <atomic>
#include <mutex>
#include <condition_variable>

namespace movie { namespace sample {

class pausable_loop {

    enum class state { paused, playing, stopped };
    std::atomic<state> request_state_{state::paused}, worker_state_{state::paused};

    std::mutex worker_mut_, client_mut_;
    std::condition_variable worker_cv_, client_cv_;

public:
    pausable_loop() = default;

    void pause()  { request_state_ = state::paused;
                    std::unique_lock<std::mutex> l_{client_mut_};
                    client_cv_.wait(l_,[&]{ return worker_state_.load() == state::paused; });
                  }

    void resume() { request_state_ = state::playing; worker_cv_.notify_one(); }
    void stop()   { request_state_ = state::stopped; worker_cv_.notify_one();
                    std::unique_lock<std::mutex> l_{client_mut_};
                    client_cv_.wait(l_,[&]{ return worker_state_.load() == state::stopped; });
                  }

    template <typename F>
    void loop(F func) {
        std::unique_lock<std::mutex> l_{worker_mut_};
        while (true) {

            state s_ = request_state_.load();

            if (s_ == state::playing) {
                worker_state_.store(state::playing);
                func();
                continue;
            }

            if (s_ == state::paused) {
                worker_state_.store(state::paused);
                client_cv_.notify_one();
                worker_cv_.wait(l_,[&]{ s_ = request_state_.load();
                                        return s_ == state::playing || s_ == state::stopped; });
            }

            if (s_ == state::stopped) {
                worker_state_.store(state::stopped);
                break;
            }
        }

        worker_state_.store(state::stopped); // should already be state::stopped
        client_cv_.notify_one();
    }

    ~pausable_loop() {
        stop();
    }
};

}} // namespaces

