﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//
//   Simple test to see how to get a version of malloc to work based
//   on ExpHeap code, and how to redefine malloc/free to call our version.
//
//
//   lmem source code in : sdk/Programs/Alice/Sources/Libraries/lmem
//
//---------------------------------------------------------------------------


#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>


#if 1                   // if set, test our fake code...
#define IS_HORIZON
#endif


// If is Horizon, we want to use ExpHeap, else (if Win32) use standard malloc code.
#ifdef IS_HORIZON
#include <nn/lmem/lmem_ExpHeap.h>
#else
#include <memory>
#endif

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

#ifndef IS_HORIZON
void MyMallocInit(size_t heapSizeInMB)
{
    NN_UNUSED(heapSizeInMB);
}
void MyMallocFree()
{
}

void* MyMalloc(size_t size)
{
    return malloc(size);
}

void MyFree(void* p,)
{
    free(p);
}
#else       // fake out malloc/free using ExpHeap.   Need to call MyMallocInit() to make this work

nn::lmem::HeapHandle    gHeapHandle = 0;
uintptr_t               gHeapAddress = 0;
size_t                  gHeapSize = 0;
uint32_t                gAllocCount = 0, gFreeCount = 0;


void MyMallocInit(size_t heapSizeInMB)
{
    size_t heapSize = heapSizeInMB * 1024 * 1024;

    NN_ASSERT(gHeapHandle == 0,"Can only init once\n");
    nn::Result result = nn::os::AllocateMemoryBlock( &gHeapAddress, heapSize );
    NN_ASSERT( result.IsSuccess(), "AllocateMemoryBlock failed for Heap." );

    gHeapHandle = nn::lmem::CreateExpHeap((void*) gHeapAddress, heapSize,  nn::lmem::CreationOption_ZeroClear);
    NN_ASSERT( result.IsSuccess(), "CreateExpHeap failed." );

    gHeapSize = heapSize;

    NN_LOG("Allocated Heap with handle 0x%08x at 0x%08x of size 0x%08x (%d MB)\n",
        gHeapHandle, gHeapAddress, gHeapSize, gHeapSize / (1024 * 1024));
}

void MyMallocFree()
{
    if(gHeapHandle == 0)
    {
        return;
    }
    NN_LOG("Allocated/freed %d/%d blocks\n",gAllocCount, gFreeCount);
    nn::os::FreeMemoryBlock( gHeapAddress, gHeapSize);

    gHeapAddress    = 0;
    gHeapSize       = 0;
    gHeapHandle     = 0;
    gAllocCount     = 0;
    gFreeCount      = 0;
}

void* MyMalloc(size_t size)
{
    NN_ASSERT(gHeapHandle != 0 && "Forgot to init Heap");
    gAllocCount++;
    return nn::lmem::AllocateFromExpHeap(gHeapHandle, size);
}

void MyFree(void* p)
{
    gFreeCount++;
    nn::lmem::FreeToExpHeap(gHeapHandle, p);
}
#endif


// redefine malloc/free to get our version of malloc
#undef malloc
#undef free
#define malloc  MyMalloc
#define free    MyFree


extern "C" void nnMain()
{
    //int     argc = nnt::GetHostArgc();
    //char **argv = nnt::GetHostArgv();

    const size_t    heapSize = 32 * 1024 * 1024;
    uintptr_t       heapAddress;

    nn::Result result = nn::os::SetMemoryHeapSize( heapSize );
    NN_ASSERT( result.IsSuccess(), "Failed in SetMemoryHeapSize()." );

    result = nn::os::AllocateMemoryBlock( &heapAddress, heapSize );
    NN_ASSERT( result.IsSuccess(), "Failed to allocate memory block." );

    nn::os::FreeMemoryBlock( heapAddress, heapSize );

    // メモリヒープ情報の出力
    NN_LOG("Address of memory heap: 0x%08x\n", heapAddress);
    NN_LOG("Size of memory heap   : 0x%08x\n", heapSize);
    NN_LOG("\n");

    MyMallocInit(32);

    for(int j = 1; j < 1000; j++)
    {
        void *addr = malloc(j);
        NN_ASSERT(addr != NULL, "Allocate Failed\n");
        free(addr);
    }

    MyMallocFree();

    NN_LOG("Done Alloc\n");
}

