﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once


#include <nn/nn_Abort.h>
#include <nn/migration/user/migration_UserMigrationStateManager.h>

#include "nntMigration_Assertion.h"
#include "nntMigration_RamStorage.h"
#include "testUserMigrationDataTransfer_Agent.h"

// ---------------------------------------------------------------------------------------------
// Server
namespace server {

struct StateStorageTag { static const char VolumeName[12]; };
using StateStoragePolicy = nnt::migration::RamStoragePolicy<StateStorageTag>;
struct ContextStorageTag { static const char VolumeName[12]; };
using ContextStoragePolicy = nnt::migration::RamStoragePolicy<ContextStorageTag>;

const char StateStorageTag::VolumeName[] = "state";
const char ContextStorageTag::VolumeName[] = "context";

void CleanupStorage() NN_NOEXCEPT
{
    StateStoragePolicy::Storage stateStorage;
    stateStorage.Cleanup();
    ContextStoragePolicy::Storage contextStorage;
    contextStorage.Cleanup();
}

template <typename ConnectionPolicy, typename EncryptionPolicy>
inline bool Run(const nn::account::Uid& uid, nn::migration::detail::LocalUserLoginSession& loginSession, nn::migration::detail::ThreadResource&& idcThreadResource, bool toBeResumed)
{
    using ServerImpl = Server<ContextStoragePolicy, ConnectionPolicy, EncryptionPolicy>;

    typename StateStoragePolicy::Storage stateStorage;
    nn::migration::user::StateManager stateManager(stateStorage);

    ServerImpl* pServer;
    if (toBeResumed)
    {
        nn::migration::user::StateController* pController;
        NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(stateManager.ResumeServer(&pController));

        pServer = new ServerImpl(*pController, loginSession, std::move(idcThreadResource));
        pServer->Resume();
    }
    else
    {
        nn::migration::user::StateController* pController;
        NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(stateManager.StartServer(&pController));

        pServer = new ServerImpl(*pController, loginSession, std::move(idcThreadResource));
        pServer->Initialize(uid);
    }
    NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(pServer->Prepare());
    if (pServer->IsConnectionRequired())
    {
        NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(pServer->WaitConnection());
        NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(pServer->ProcessTransfer());
    }
    NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(pServer->ProcessComplete());
    stateStorage.Cleanup();
    return true;
}

} // ~namespace server

template <typename ConnectionPolicy, typename EncryptionPolicy>
inline bool RunServer(const nn::account::Uid& uid, nn::migration::detail::LocalUserLoginSession& loginSession, nn::migration::detail::ThreadResource&& idcThreadResource) NN_NOEXCEPT
{
    return server::Run<ConnectionPolicy, EncryptionPolicy>(uid, loginSession, std::move(idcThreadResource), false);
}
template <typename ConnectionPolicy, typename EncryptionPolicy>
inline bool RunResumedServer(nn::migration::detail::LocalUserLoginSession& loginSession, nn::migration::detail::ThreadResource&& idcThreadResource) NN_NOEXCEPT
{
    return server::Run<ConnectionPolicy, EncryptionPolicy>(nn::account::InvalidUid, loginSession, std::move(idcThreadResource), true);
}

// ---------------------------------------------------------------------------------------------
// Client
namespace client {

struct StateStorageTag { static const char VolumeName[12]; };
using StateStoragePolicy = nnt::migration::RamStoragePolicy<StateStorageTag>;
struct ContextStorageTag { static const char VolumeName[12]; };
using ContextStoragePolicy = nnt::migration::RamStoragePolicy<ContextStorageTag>;

const char StateStorageTag::VolumeName[] = "state";
const char ContextStorageTag::VolumeName[] = "context";

void CleanupStorage() NN_NOEXCEPT
{
    StateStoragePolicy::Storage stateStorage;
    stateStorage.Cleanup();
    ContextStoragePolicy::Storage contextStorage;
    contextStorage.Cleanup();
}

template <typename ConnectionPolicy, typename EncryptionPolicy>
inline bool Run(const nn::account::Uid& uid, nn::migration::detail::LocalUserLoginSession& loginSession, nn::migration::detail::ThreadResource&& idcThreadResource, bool toBeResumed) NN_NOEXCEPT
{
    using ClientImpl = Client<ContextStoragePolicy, ConnectionPolicy, EncryptionPolicy>;

    typename StateStoragePolicy::Storage stateStorage;
    nn::migration::user::StateManager stateManager(stateStorage);

    ClientImpl* pClient;
    if (toBeResumed)
    {
        nn::migration::user::StateController* pController;
        NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(stateManager.ResumeClient(&pController));

        pClient = new ClientImpl(*pController, loginSession, std::move(idcThreadResource));
        pClient->Resume();
    }
    else
    {
        nn::migration::user::StateController* pController;
        NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(stateManager.StartClient(&pController));

        pClient = new ClientImpl(*pController, loginSession, std::move(idcThreadResource));
        pClient->Initialize();
    }

    NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(pClient->Prepare());

    if (pClient->IsConnectionRequired())
    {
        NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(pClient->Connect());
        NN_ABORT_UNLESS(pClient->GetStorageShortfall() <= 0);

        auto total = pClient->GetTotalTransferInfo();
        nn::migration::TransferInfo current;
        while ((current = pClient->GetCurrentTransferInfo()).count < total.count)
        {
            NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(pClient->Transfer(uid, current.count));
        }
    }
    NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(pClient->Complete());
    return true;
}

} // ~namespace clients

template <typename ConnectionPolicy, typename EncryptionPolicy>
inline bool RunClient(const nn::account::Uid& uid, nn::migration::detail::LocalUserLoginSession& loginSession, nn::migration::detail::ThreadResource&& idcThreadResource) NN_NOEXCEPT
{
    return client::Run<ConnectionPolicy, EncryptionPolicy>(uid, loginSession, std::move(idcThreadResource), false);
}
template <typename ConnectionPolicy, typename EncryptionPolicy>
inline bool RunResumedClient(const nn::account::Uid& uid, nn::migration::detail::LocalUserLoginSession& loginSession, nn::migration::detail::ThreadResource&& idcThreadResource) NN_NOEXCEPT
{
    return client::Run<ConnectionPolicy, EncryptionPolicy>(uid, loginSession, std::move(idcThreadResource), true);
}
