﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>

#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/migration/migration_UserMigrationTypes.h>
#include <nn/migration/detail/migration_Authenticator.h>
#include <nn/migration/detail/migration_Cancellable.h>
#include <nn/migration/detail/migration_LoginSession.h>
#include <nn/migration/detail/migration_NetworkConnection.h>
#include <nn/migration/user/migration_UserMigrationConfig.h>
#include <nn/migration/user/migration_UserMigrationContext.h>
#include <nn/migration/user/migration_UserMigrationIdcClient.h>
#include <nn/migration/user/migration_UserMigrationIdcServer.h>
#include <nn/migration/user/migration_UserMigrationProgressMonitor.h>
#include <nn/migration/user/migration_UserMigrationStateController.h>

template <typename ContextStoragePolicy, typename ConnectionPolicy, typename EncryptionPolicy>
class Server
{
private:
    static const int ResetDelayMilliSeconds = 3 * 1000;

    // アカウントへのログインセッション
    nn::migration::detail::LocalUserLoginSession& m_LoginSession;

    // ローカル通信の接続管理
    std::atomic<bool> m_IsConnected;
    typename ConnectionPolicy::Connection m_Connection;
    typename ConnectionPolicy::ConnectionCreator m_ConnectionCreator;

    // 対向デバイスとの通信モジュール
    nn::migration::user::IdcServer<typename ConnectionPolicy::Connection, EncryptionPolicy> m_Idc;

    // 移行の状態管理
    typename ContextStoragePolicy::Storage m_ContextStorage;
    nn::migration::user::ServerContext m_Context;
    nn::migration::user::StateController& m_StateController;

    // 動作に必要なメモリ
    struct Resource
    {
        union
        {
            std::aligned_storage<1024 * 1024>::type contextWorkBuffer; // 1MiB
        } u;
    } m_Resource;

private:
    void Connect(typename ConnectionPolicy::Connection&& connection) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_IsConnected);
        m_Connection = std::move(connection);
        m_IsConnected = true;
    }
    void Disconnect(nn::TimeSpan resetDelay) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsConnected);
        m_Connection = decltype(m_Connection)();
        nn::os::SleepThread(resetDelay);
        m_ConnectionCreator.Reset();
        m_IsConnected = false;
    }
    void CleanupContext() NN_NOEXCEPT
    {
        m_StateController.Reset();
        m_ContextStorage.Cleanup();
    }

public:
    explicit Server(nn::migration::user::StateController& stateController, nn::migration::detail::LocalUserLoginSession& loginSession, nn::migration::detail::ThreadResource&& idcThreadResource) NN_NOEXCEPT
        : m_LoginSession(loginSession)
        , m_IsConnected(false)
        , m_Idc(std::move(idcThreadResource))
        , m_Context(m_ContextStorage)
        , m_StateController(stateController)
    {
    }

    void Initialize(const nn::account::Uid& uid) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(uid);
        m_ContextStorage.Cleanup();
        nn::migration::UserMigrationServerProfile profile = {};
        m_Context.CreateMigrationInfo(uid, profile);

        m_LoginSession.Initialize(m_Context.GetUid());
    }
    void Resume() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_Context.TryResume());
        NN_SDK_ASSERT(m_StateController.IsResumable());

        m_LoginSession.Initialize(m_Context.GetUid());
    }

    nn::Result Prepare() NN_NOEXCEPT
    {
        if (!m_StateController.IsResumable())
        {
            NN_MIGRATION_DETAIL_DO(m_Context.CreateMigrationList(
                &m_Resource.u.contextWorkBuffer, sizeof(m_Resource.u.contextWorkBuffer),
                nullptr));
        }

        nn::account::NetworkServiceAccountId nsaId;
        NN_MIGRATION_DETAIL_DO(m_LoginSession.GetNetworkServiceAccountId(&nsaId));
        NN_MIGRATION_DETAIL_DO(m_Idc.Initialize(m_LoginSession, nullptr));
        NN_RESULT_SUCCESS;
    }
    bool IsConnectionRequired() const NN_NOEXCEPT
    {
        return !m_StateController.IsFinalized();
    }
    nn::Result WaitConnection() NN_NOEXCEPT
    {
        nn::account::NetworkServiceAccountId nsaId;
        NN_MIGRATION_DETAIL_DO(m_LoginSession.GetNetworkServiceAccountId(&nsaId));

        nn::migration::UserMigrationServerProfile profile;
        m_Context.GetServerProfile(&profile);

        typename ConnectionPolicy::Connection connection;
        NN_MIGRATION_DETAIL_DO(m_ConnectionCreator.template CreateConnection<EncryptionPolicy>(&connection, nsaId, profile, m_Idc, nullptr));
        Connect(std::move(connection));

        NN_MIGRATION_DETAIL_DO(m_Idc.ExportUserMigrationServerInfo(m_Context, m_Connection, nullptr));

        if (m_StateController.IsResumable())
        {
            NN_MIGRATION_DETAIL_DO(m_Idc.MatchUserMigrationClientInfo(m_Context, m_Connection, nullptr));
            NN_MIGRATION_DETAIL_DO(m_Idc.AcceptUserMigrationClient(m_Connection, nullptr));
            NN_RESULT_SUCCESS;
        }

        NN_MIGRATION_DETAIL_DO(m_Idc.ImportUserMigrationClientInfo(m_Context, m_Connection, nullptr));
        NN_MIGRATION_DETAIL_DO(m_Idc.AcceptUserMigrationClient(m_Connection, nullptr));
        NN_MIGRATION_DETAIL_DO(m_Idc.ExportMigrationList(m_Context, m_Connection, nullptr));
        NN_MIGRATION_DETAIL_DO(m_Idc.WaitStateSychronizationInInitialization(m_StateController, m_Connection, nullptr));
        NN_RESULT_SUCCESS;
    }
    nn::Result ProcessTransfer() NN_NOEXCEPT
    {
        if (!m_StateController.IsInitialized())
        {
            NN_MIGRATION_DETAIL_DO(m_StateController.SetStateInTransfer());
        }

        do
        {
            NN_MIGRATION_DETAIL_DO(m_Idc.ExportSaveData(m_StateController, m_Context, m_Connection, nullptr));
        } while (!m_StateController.IsTransferDone());
        NN_RESULT_SUCCESS;
    }
    nn::Result ProcessComplete() NN_NOEXCEPT
    {
        if (m_IsConnected)
        {
            NN_MIGRATION_DETAIL_DO(m_Idc.WaitStateSychronizationFinalized(m_StateController, m_Connection, nullptr));
            Disconnect(nn::TimeSpan::FromMilliSeconds(ResetDelayMilliSeconds));
        }
        CleanupContext();
        NN_RESULT_SUCCESS;
    }
};

template <typename ContextStoragePolicy, typename ConnectionPolicy, typename EncryptionPolicy>
class Client
{
private:
    static const int ResetDelayMilliSeconds = 3 * 1000;

    // アカウントへのログインセッション
    nn::migration::detail::LocalUserLoginSession& m_LoginSession;

    // ローカル通信の接続管理
    std::atomic<bool> m_IsConnected;
    typename ConnectionPolicy::Connection m_Connection;
    typename ConnectionPolicy::ConnectionCreator m_ConnectionCreator;

    // 対向デバイスとの通信モジュール
    nn::migration::user::IdcClient<typename ConnectionPolicy::Connection, EncryptionPolicy> m_Idc;

    // 移行の状態管理
    typename ContextStoragePolicy::Storage m_ContextStorage;
    nn::migration::user::ClientContext m_Context;
    nn::migration::user::ProgressMonitor m_ProgressMonitor;
    nn::migration::user::StateController& m_StateController;

    // 動作に必要なメモリ
    struct Resource
    {
        std::aligned_storage<
            sizeof(nn::ncm::ApplicationId[nn::migration::user::UserMigrationRelatedApplicationCountMax]),
            std::alignment_of<nn::ncm::ApplicationId[nn::migration::user::UserMigrationRelatedApplicationCountMax]>::value>::type relatedAppStorage;

        union
        {
            mutable std::aligned_storage<1024 * 1024>::type contextWorkBuffer; // 1MiB
        } u;
    } m_Resource;

private:
    void Connect(typename ConnectionPolicy::Connection&& connection) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_IsConnected);
        m_Connection = std::move(connection);
        m_IsConnected = true;
    }
    void Disconnect(nn::TimeSpan resetDelay) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsConnected);
        m_Connection = decltype(m_Connection)();
        nn::os::SleepThread(resetDelay);
        m_ConnectionCreator.Reset();
        m_IsConnected = false;
    }
    void CleanupContext() NN_NOEXCEPT
    {
        m_StateController.Reset();
        m_ContextStorage.Cleanup();
    }

public:
    typedef typename std::aligned_storage<sizeof(Resource), std::alignment_of<Resource>::value>::type ResourceStorage;

public:
    Client(nn::migration::user::StateController& stateController, nn::migration::detail::LocalUserLoginSession& loginSession, nn::migration::detail::ThreadResource&& idcThreadResource) NN_NOEXCEPT
        : m_LoginSession(loginSession)
        , m_IsConnected(false)
        , m_Idc(std::move(idcThreadResource))
        , m_Context(m_ContextStorage)
        , m_ProgressMonitor(m_Context)
        , m_StateController(stateController)
    {
    }

    void Initialize() NN_NOEXCEPT
    {
        m_ContextStorage.Cleanup();
        m_Context.Initialize(&m_Resource.relatedAppStorage, sizeof(m_Resource.relatedAppStorage));
        nn::migration::UserMigrationClientProfile profile = {};
        m_Context.CreateMigrationInfo(profile);
    }
    void Resume() NN_NOEXCEPT
    {
        m_Context.Initialize(&m_Resource.relatedAppStorage, sizeof(m_Resource.relatedAppStorage));
        NN_ABORT_UNLESS(m_Context.TryResume());
        NN_SDK_ASSERT(m_StateController.IsResumable());
    }

    nn::Result Prepare() NN_NOEXCEPT
    {
        nn::account::NetworkServiceAccountId nsaId;
        NN_MIGRATION_DETAIL_DO(m_LoginSession.GetNetworkServiceAccountId(&nsaId));
        if (m_StateController.IsResumable())
        {
            NN_MIGRATION_DETAIL_THROW_UNLESS(
                m_Context.GetNetworkServiceAccountId() == nsaId,
                nn::migration::ResultUnexpectedNetworkServiceAccountLoggedIn());
        }
        else
        {
            m_Context.SetNetworkServiceAccountId(nsaId);
        }

        NN_MIGRATION_DETAIL_DO(m_Idc.Initialize(m_LoginSession, nullptr));
        NN_RESULT_SUCCESS;
    }
    bool IsConnectionRequired() const NN_NOEXCEPT
    {
        return !m_StateController.IsFinalized();
    }
    nn::Result Connect() NN_NOEXCEPT
    {
        nn::account::NetworkServiceAccountId nsaId;
        NN_MIGRATION_DETAIL_DO(m_LoginSession.GetNetworkServiceAccountId(&nsaId));

        nn::util::Uuid serverId;
        while (NN_STATIC_CONDITION(true))
        {
            size_t scanned = 0u;
            nn::migration::UserMigrationServerInfo serverInfo;
            NN_MIGRATION_DETAIL_DO(m_ConnectionCreator.ScanServers(&scanned, &serverInfo, 1, nsaId, nullptr));
            if (scanned > 0u)
            {
                serverId = serverInfo.id;
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        typename ConnectionPolicy::Connection connection;
        NN_MIGRATION_DETAIL_DO(m_ConnectionCreator.template Connect<EncryptionPolicy>(&connection, serverId, nsaId, m_Idc, nullptr));
        Connect(std::move(connection));

        if (m_StateController.IsResumable())
        {
            NN_MIGRATION_DETAIL_DO(m_Idc.MatchUserMigrationServerInfo(m_Context, m_Connection, nullptr));
            NN_MIGRATION_DETAIL_DO(m_Idc.ExportUserMigrationClientInfo(m_Context, m_Connection, nullptr));
            NN_MIGRATION_DETAIL_DO(m_Idc.WaitUserMigrationClientAccepted(m_Connection, nullptr));
            NN_RESULT_SUCCESS;
        }

        NN_MIGRATION_DETAIL_DO(m_Idc.ImportUserMigrationServerInfo(m_Context, m_Connection, nullptr));
        NN_MIGRATION_DETAIL_DO(m_Idc.ExportUserMigrationClientInfo(m_Context, m_Connection, nullptr));
        NN_MIGRATION_DETAIL_DO(m_Idc.WaitUserMigrationClientAccepted(m_Connection, nullptr));
        NN_MIGRATION_DETAIL_DO(m_Idc.ImportMigrationList(m_Context, m_Connection, nullptr));
        NN_MIGRATION_DETAIL_DO(m_Context.ResetProgressInfo(&m_Resource.u.contextWorkBuffer, sizeof(m_Resource.u.contextWorkBuffer)));
        NN_MIGRATION_DETAIL_DO(m_Idc.SynchronizeStateInInitialization(m_StateController, m_Connection, nullptr));
        NN_RESULT_SUCCESS;
    }

    int64_t GetStorageShortfall() const NN_NOEXCEPT
    {
        return static_cast<int64_t>(m_Context.CalculateCurrentShortfallOfUserSpace(
            &m_Resource.u.contextWorkBuffer, sizeof(m_Resource.u.contextWorkBuffer)));
    }
    nn::migration::TransferInfo GetTotalTransferInfo() const NN_NOEXCEPT
    {
        return m_Context.GetTotalTransferInfo();
    }
    nn::migration::TransferInfo GetCurrentTransferInfo() const NN_NOEXCEPT
    {
        return m_ProgressMonitor.GetTransferInfo();
    }
    size_t GetCurrentRelatedApplications(nn::ncm::ApplicationId* apps, size_t count) const NN_NOEXCEPT
    {
        return m_Context.GetCurrentRelatedApplications(apps, count);
    }
    nn::migration::detail::DataInfo GetDataInfo(uint32_t index) const NN_NOEXCEPT
    {
        nn::migration::detail::DataInfo dataInfo;
        auto count = m_Context.ReadMigrationList(&dataInfo, 1, static_cast<size_t>(index));
        NN_ABORT_UNLESS(count, 1u);
        return dataInfo;
    }
    nn::Result Transfer(const nn::account::Uid& uid, uint32_t index) NN_NOEXCEPT
    {
        if (!m_StateController.IsInitialized())
        {
            NN_MIGRATION_DETAIL_DO(m_StateController.SetStateInTransfer());
        }

        nn::migration::detail::DataInfo dataInfo;
        auto count = m_Context.ReadMigrationList(&dataInfo, 1, index);
        NN_ABORT_UNLESS_EQUAL(count, 1u);

        nn::migration::detail::SaveDataImporter importer(uid, dataInfo, m_ProgressMonitor);
        return m_Idc.ImportSaveData(importer, index, m_Connection, nullptr);
    }

    nn::Result Suspend() NN_NOEXCEPT
    {
        if (m_IsConnected)
        {
            NN_MIGRATION_DETAIL_DO(m_Idc.Suspend(m_Connection, nullptr));
            Disconnect(nn::TimeSpan::FromMilliSeconds(ResetDelayMilliSeconds));
        }
        NN_RESULT_SUCCESS;
    }
    nn::Result Complete() NN_NOEXCEPT
    {
        if (!m_StateController.IsFinalized())
        {
            NN_MIGRATION_DETAIL_DO(m_Idc.SynchronizeStateInFinalization(m_StateController, m_Connection, nullptr));
            NN_MIGRATION_DETAIL_DO(m_Idc.SynchronizeStateFinalized(m_StateController, m_Connection, nullptr));
            Disconnect(nn::TimeSpan::FromMilliSeconds(ResetDelayMilliSeconds));
        }
        CleanupContext();
        NN_RESULT_SUCCESS;
    }
};
