﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/migration/detail/migration_Cancellable.h>
#include <nn/migration/idc/migration_SocketConnection.h>
#include <nn/migration/idc/migration_Result.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiIpAddress.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nntest.h>

#include "testMigrationIdc_Common.h"
#include "testMigrationIdc_ConnectionTestCases.h"

using namespace nn;

class MigrationIdcSocketConnectionTest : public testing::Test
{
private:

    socket::SockAddrIn  m_ServerSocketAddress;          // Client から接続するための Server 側のアドレスとポートの情報。
    int                 m_ServerAcceptSocketDescriptor; // Server が Listen, Accept に使うソケット。
    int                 m_ServerSocketDescriptor;       // Server が Send と Receive に使うソケット（Acceptで取得）。
    int                 m_ClientSocketDescriptor = -1;  // Client が Send と Receive に使うソケット。
    os::Event           m_ConnectionReadyEvent;

public:

    typedef migration::idc::SocketConnection ConnectionType;

    MigrationIdcSocketConnectionTest() :
        m_ServerAcceptSocketDescriptor(-1),
        m_ServerSocketDescriptor(-1),
        m_ClientSocketDescriptor(-1),
        m_ConnectionReadyEvent(os::EventClearMode_ManualClear)
    {
    }

    void GetConnectionA(ConnectionType* pOutConnection) NN_NOEXCEPT
    {
        new (pOutConnection) ConnectionType(m_ServerSocketDescriptor);
    }

    void GetConnectionB(ConnectionType* pOutConnection) NN_NOEXCEPT
    {
        new (pOutConnection) ConnectionType(m_ClientSocketDescriptor);
    }

private:
    void SetupServer()
    {
        NN_LOG("-- Server startup --\n");

        if( (m_ServerAcceptSocketDescriptor = socket::Socket(socket::Family::Af_Inet, socket::Type::Sock_Stream, socket::Protocol::IpProto_Tcp)) < 0 )
        {
            NN_ABORT("server: Socket failed (error %d)\n", socket::GetLastError());
        }

        socket::SockAddrIn serverSocketAddress = { 0 };
        serverSocketAddress.sin_addr.S_addr = nn::socket::InetHtonl(socket::InAddr_Any);
        serverSocketAddress.sin_port = nn::socket::InetHtons(0);
        serverSocketAddress.sin_family = socket::Family::Af_Inet;

        if( socket::Bind(m_ServerAcceptSocketDescriptor, reinterpret_cast<nn::socket::SockAddr *>(&serverSocketAddress), sizeof(serverSocketAddress)) < 0 )
        {
            NN_ABORT("server: Bind failed (error %d)\n", nn::socket::GetLastError());
        }

        socket::SockLenT saLen = sizeof(m_ServerSocketAddress);
        if( socket::GetSockName(m_ServerAcceptSocketDescriptor, reinterpret_cast<nn::socket::SockAddr *>(&m_ServerSocketAddress), &saLen) < 0 )
        {
            NN_ABORT("server: GetSockName failed (error %d)\n", nn::socket::GetLastError());
        }

        auto result = nifm::GetCurrentPrimaryIpAddress(&m_ServerSocketAddress.sin_addr);
        if( result.IsSuccess() )
        {
            NN_LOG("server: listening for incoming messages at: %s:%d\n",
                socket::InetNtoa(m_ServerSocketAddress.sin_addr),
                static_cast<int>(socket::InetNtohs(m_ServerSocketAddress.sin_port)));
        }
        else
        {
            NN_ABORT("server: GetCurrentIpAddress failed (error 0x%08x)\n", result.GetInnerValueForDebug());
        }

        if( socket::Listen(m_ServerAcceptSocketDescriptor, 4) < 0 )
        {
            NN_ABORT("server: listen failed (error %d)\n", nn::socket::GetLastError());
        }
    }

    void CreateConnection()
    {
        RunOnThreadA([](void* arg)
            {
                auto testInstance = reinterpret_cast<MigrationIdcSocketConnectionTest*>(arg);
                socket::SockAddrIn saClientAddress = { 0 };
                socket::SockLenT clientAddressSize = sizeof(saClientAddress);
                // メモ : Accept はタイムアウトしない。
                if( (testInstance->m_ServerSocketDescriptor = nn::socket::Accept(testInstance->m_ServerAcceptSocketDescriptor, reinterpret_cast<socket::SockAddr *>(&saClientAddress), &clientAddressSize)) < 0 )
                {
                    NN_LOG("server: Accept failed (error %d)\n", nn::socket::GetLastError());
                    return;
                }
                NN_LOG("server: connection accepted from %s\n", nn::socket::InetNtoa(saClientAddress.sin_addr));
            }, this);

        RunOnThreadB([](void* arg)
            {
                auto testInstance = reinterpret_cast<MigrationIdcSocketConnectionTest*>(arg);

                NN_LOG("-- Client startup --\n");
                if( (testInstance->m_ClientSocketDescriptor = nn::socket::Socket(socket::Family::Af_Inet, socket::Type::Sock_Stream, socket::Protocol::IpProto_Tcp)) < 0 )
                {
                    NN_LOG("client: Socket failed (error %d)\n", nn::socket::GetLastError());
                    return;
                }

                NN_LOG("client: connecting to %s:%d\n",
                    socket::InetNtoa(testInstance->m_ServerSocketAddress.sin_addr),
                    static_cast<int>(socket::InetNtohs(testInstance->m_ServerSocketAddress.sin_port)));

                if( socket::Connect(testInstance->m_ClientSocketDescriptor, reinterpret_cast<socket::SockAddr *>(&testInstance->m_ServerSocketAddress), sizeof(testInstance->m_ServerSocketAddress)) < 0 )
                {
                    NN_LOG("client: Connect failed (error %d)\n", nn::socket::GetLastError());
                    return;
                }
                NN_LOG("client: established connection to server\n");
                testInstance->m_ConnectionReadyEvent.Signal();
            }, this);

        m_ConnectionReadyEvent.Wait();

        DestroyThreadA();
        DestroyThreadB();
    }

    void CleanupConnection()
    {
        m_ConnectionReadyEvent.Clear();
        socket::Close(m_ServerSocketDescriptor);
        socket::Close(m_ClientSocketDescriptor);
        socket::Close(m_ServerAcceptSocketDescriptor);
    }

protected:
    virtual void SetUp()
    {
        SetupServer();
        CreateConnection();
    }
    virtual void TearDown()
    {
        CleanupConnection();
    }

    static void SetUpTestCase()
    {
        static socket::ConfigDefaultWithMemory s_SocketConfigWithMemory;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::Initialize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(socket::Initialize(s_SocketConfigWithMemory));
        nifm::SubmitNetworkRequestAndWait();
        NN_ABORT_UNLESS(nifm::IsNetworkAvailable());
    }

    static void TearDownTestCase()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(socket::Finalize());
    }
};

TEST_F(MigrationIdcSocketConnectionTest, BasicSendReceive)
{
    ConnectionType c0, c1;
    GetConnectionA(&c0);
    GetConnectionB(&c1);
    TestBasicSendReceive<>(c0, c1);
}

TEST_F(MigrationIdcSocketConnectionTest, BasicSendReceiveLarge)
{
    ConnectionType c0, c1;
    GetConnectionA(&c0);
    GetConnectionB(&c1);
    TestBasicSendReceiveLarge<>(c0, c1);
}

/// 送信系のテスト

TEST_F(MigrationIdcSocketConnectionTest, SendPeerClose)
{
    ConnectionType c0, c1;
    GetConnectionA(&c0);
    GetConnectionB(&c1);
    TestSendPeerClose<>(c0, std::move(c1));
}

TEST_F(MigrationIdcSocketConnectionTest, SendTimeout)
{
    ConnectionType c0;
    GetConnectionA(&c0);
    TestSendTimeout<>(c0);
}

TEST_F(MigrationIdcSocketConnectionTest, SendCancel)
{
    ConnectionType c0;
    GetConnectionA(&c0);
    TestSendCancel<>(c0);
}

TEST_F(MigrationIdcSocketConnectionTest, SendCancelWhileSendBufferFull)
{
    ConnectionType c0;
    GetConnectionA(&c0);
    TestSendCancelWhileSendBufferFull<>(c0);
}

TEST_F(MigrationIdcSocketConnectionTest, SendCancelAndClose)
{
    ConnectionType c0, c1;
    GetConnectionA(&c0);
    GetConnectionB(&c1);
    TestSendCancelAndClose<>(c0, std::move(c1));
}

TEST_F(MigrationIdcSocketConnectionTest, SendLowSpeedTimeout)
{
    ConnectionType c0;
    GetConnectionA(&c0);
    TestSendLowSpeedTimeout<>(c0);
}

/// 受信系のテスト

TEST_F(MigrationIdcSocketConnectionTest, ReceivePeerClose)
{
    ConnectionType c0, c1;
    GetConnectionA(&c0);
    GetConnectionB(&c1);
    TestReceivePeerClose<>(c0, std::move(c1));
}

TEST_F(MigrationIdcSocketConnectionTest, ReceivePeerCloseWhileReceiveBufferEmpty)
{
    ConnectionType c0, c1;
    GetConnectionA(&c0);
    GetConnectionB(&c1);
    TestReceivePeerCloseWhileReceiveBufferEmpty<>(c0, std::move(c1));
}

TEST_F(MigrationIdcSocketConnectionTest, ReceivePeerCloseWithReceiveBuffer)
{
    ConnectionType c0, c1;
    GetConnectionA(&c0);
    GetConnectionB(&c1);
    TestReceivePeerCloseWithReceiveBuffer<>(c0, std::move(c1));
}

TEST_F(MigrationIdcSocketConnectionTest, ReceiveCancel)
{
    ConnectionType c0;
    GetConnectionA(&c0);
    TestReceiveCancel<>(c0);
}

TEST_F(MigrationIdcSocketConnectionTest, ReceiveCancelWhileReceiveBufferEmpty)
{
    ConnectionType c0;
    GetConnectionA(&c0);
    TestReceiveCancelWhileReceiveBufferEmpty<>(c0);
}

TEST_F(MigrationIdcSocketConnectionTest, ReceiveCancelWithReceiveBuffer)
{
    ConnectionType c0, c1;
    GetConnectionA(&c0);
    GetConnectionB(&c1);
    TestReceiveCancelWithReceiveBuffer<>(c0, c1);
}

TEST_F(MigrationIdcSocketConnectionTest, ReceiveCancelAndClose)
{
    ConnectionType c0, c1;
    GetConnectionA(&c0);
    GetConnectionB(&c1);
    TestReceiveCancelAndClose<>(c0, std::move(c1));
}

TEST_F(MigrationIdcSocketConnectionTest, ReceiveTimeout)
{
    ConnectionType c0;
    GetConnectionA(&c0);
    TestReceiveTimeout<>(c0);
}

TEST_F(MigrationIdcSocketConnectionTest, ReceiveLowSpeedTimeout)
{
    ConnectionType c0;
    GetConnectionA(&c0);
    TestReceiveLowSpeedTimeout<>(c0);
}
