﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>
#include <nn/account/account_Api.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/result/result_HandlingUtility.h>

namespace nnt { namespace migration {

inline void CleanupSaveData(const nn::account::Uid& uid) NN_NOEXCEPT
{
    nn::fs::UserId fsUid;
    std::memcpy(&fsUid, &uid, sizeof(fsUid));

    const nn::fs::SaveDataSpaceId TargetSpaceIds[] = {
        nn::fs::SaveDataSpaceId::User, nn::fs::SaveDataSpaceId::System
    };

    for (auto spaceId : TargetSpaceIds)
    {
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, spaceId));

        while (NN_STATIC_CONDITION(true))
        {
            int64_t count;
            nn::fs::SaveDataInfo info;
            NN_ABORT_UNLESS_RESULT_SUCCESS(iter->ReadSaveDataInfo(&count, &info, 1));
            if (count == 0)
            {
                break;
            }

            if (info.saveDataUserId == fsUid)
            {
                NN_LOG("Deleting %016llx for %016llx_%016llx\n", info.saveDataId, info.saveDataUserId._data[0], info.saveDataUserId._data[1]);
                auto r = nn::fs::DeleteSaveData(info.saveDataId);
                if (!(r.IsSuccess() || nn::fs::ResultTargetNotFound::Includes(r)))
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(r);
                }
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, spaceId));
            }
        }
    }
}

inline void CleanupSaveData() NN_NOEXCEPT
{
    nn::account::Uid users[nn::account::UserCountMax];
    int count;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, nn::account::UserCountMax));

    for (int i = 0; i < count; ++ i)
    {
        CleanupSaveData(users[i]);
    }
}

inline void CreateSaveData(const nn::account::Uid& uid, nn::ncm::ApplicationId appId, size_t size, size_t journalSize) NN_NOEXCEPT
{
    nn::fs::UserId fsUid;
    std::memcpy(&fsUid, &uid, sizeof(fsUid));
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::CreateSaveData(appId, fsUid, appId.value, static_cast<int64_t>(size), static_cast<int64_t>(journalSize), 0));
}

}} // ~namespace nnt::migration
