﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#if defined(NN_BUILD_TARGET_PLATFORM_WIN)
#include <nn/nn_Windows.h>
#endif
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardNetworkSystemClock.h>

namespace nnt { namespace migration {

inline void TryLogNetworkTime() NN_NOEXCEPT
{
    nn::time::PosixTime posix;
    auto r = nn::time::StandardNetworkSystemClock::GetCurrentTime(&posix);
    NN_ABORT_UNLESS(r.IsSuccess() || nn::time::ResultClockInvalid::Includes(r));
    if (!r.IsSuccess())
    {
        return;
    }

    nn::time::CalendarTime calendar;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(&calendar, nullptr, posix));

    NN_LOG("[nnt::migration]-------------------------------------------------------------------\n");
    NN_LOG("[nnt::migration] Event time stamp : %04d-%02d-%02d %02d:%02d:%02d (UTC+0)\n",
        calendar.year, calendar.month, calendar.day,
        calendar.hour, calendar.minute, calendar.second);
}

#define NNT_MIGRATION_LOG(...) \
    do { ::nnt::migration::TryLogNetworkTime(); NN_LOG(__VA_ARGS__); } while (NN_STATIC_CONDITION(false))

// bool を返す場合の結果の検査
#define NNT_MIGRATION_RETURN_FALSE_UNLESS(cond) \
    do { if (!(cond)) { ::nnt::migration::TryLogNetworkTime(); NN_LOG("%s(%d): " #cond " => false\n", __FILE__, __LINE__); return false;} } while (NN_STATIC_CONDITION(false))

#define NNT_MIGRATION_RETURN_FALSE_UNLESS_NOT(cond) \
    NNT_MIGRATION_RETURN_FALSE_UNLESS(!(cond))

#define NNT_MIGRATION_RETURN_FALSE_UNLESS_EQ(lhs, rhs) \
    NNT_MIGRATION_RETURN_FALSE_UNLESS((lhs) == (rhs))

#define NNT_MIGRATION_RETURN_FALSE_UNLESS_NE(lhs, rhs) \
    NNT_MIGRATION_RETURN_FALSE_UNLESS((lhs) != (rhs))

// bool を返す場合の結果の検査
#define NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(result) \
    do { nn::Result _r = (result); if (!_r.IsSuccess()) { ::nnt::migration::TryLogNetworkTime(); NN_LOG("%s(%d): Result(%d / %d)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription()); return false;} } while (NN_STATIC_CONDITION(false))

#define NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_INCLUDED(expected, result) \
    do { nn::Result _r = (result); if (!expected::Includes(_r)) { ::nnt::migration::TryLogNetworkTime(); NN_LOG("%s(%d): Result(%d / %d)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription()); return false;} } while (NN_STATIC_CONDITION(false))

// Result を返す場合の結果の検査
#define NNT_MIGRATION_RESULT_DO(result) \
    do { nn::Result _r = (result); if (!_r.IsSuccess()) { ::nnt::migration::TryLogNetworkTime(); NN_LOG("%s(%d): Result(%d / %d)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription()); return _r;} } while (NN_STATIC_CONDITION(false))

// Result に対する ASSERT_TRUE 的なもの
#define NNT_MIGRATION_ASSERT_RESULT_SUCCESS(result) \
    do { nn::Result _r = (result); if (!_r.IsSuccess()) { ::nnt::migration::TryLogNetworkTime(); NN_LOG("%s(%d): Result(%d / %d)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription()); ASSERT_TRUE(_r.IsSuccess());} } while (NN_STATIC_CONDITION(false))

#define NNT_MIGRATION_ASSERT_RESULT_INCLUDED(exp, result) \
    do { nn::Result _r = (result); if (!exp::Includes(_r)) { ::nnt::migration::TryLogNetworkTime(); NN_LOG("%s(%d): Result(%d / %d)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription()); ASSERT_TRUE(exp::Includes(_r));} } while (NN_STATIC_CONDITION(false))

// Result に対する EXPECT_TRUE 的なもの
#define NNT_MIGRATION_EXPECT_RESULT_SUCCESS(result) \
    do { nn::Result _r = (result); if (!_r.IsSuccess()) { ::nnt::migration::TryLogNetworkTime(); NN_LOG("%s(%d): Result(%d / %d)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription()); EXPECT_TRUE(_r.IsSuccess()); } } while (NN_STATIC_CONDITION(false))

#define NNT_MIGRATION_EXPECT_RESULT_INCLUDED(exp, result) \
    do { nn::Result _r = (result); if (!exp::Includes(_r)) { ::nnt::migration::TryLogNetworkTime(); NN_LOG("%s(%d): Result(%d / %d)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription()); EXPECT_TRUE(exp::Includes(_r)); } } while (NN_STATIC_CONDITION(false))


}} // ~namespace nnt::migration
