﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/teamcity/testTeamcity_Logger.h>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/mem.h>

#define ASSERT_NULL(p) ASSERT_TRUE(p == nullptr)
#define ASSERT_NOT_NULL(p) ASSERT_TRUE(p != nullptr)

namespace {

const size_t ThreadStackSize = 16384;                                   // スレッドのスタックサイズ
const int MaxThreadCount = 8;                                           // 作成するスレッドの最大数
NN_ALIGNAS(4096) char g_ThreadStack[MaxThreadCount][ThreadStackSize];   // 各スレッドのスタック
nn::os::ThreadType g_Thread[MaxThreadCount];                            // スレッド構造体

const int AllocateSizeCount = 16;
const int AlignmentCount = 16;
// 確保するメモリブロックのサイズ
const size_t pAllocateSize[] = {4, 8, 16, 32, 64, 128, 256, 512, 1024, 2048, 4096, 8 * 1024,
                                16 * 1024, 32 * 1024, 64 * 1024, 128 * 1024};
// 利用するアライメントのサイズ
const size_t pAlignment[] = {4, 8, 16, 32, 64, 128, 256, 512, 1024, 2048, 4096, 8 * 1024,
                             16 * 1024, 32 * 1024, 64 * 1024, 128 * 1024};

/**
 * @brief   テストに与えるパラメータです。
 */
enum StandardAllocatorTestParam
{
    StandardAllocatorTestParam_DisableThreadCache = 0,  //!< スレッドキャッシュを利用しない
    StandardAllocatorTestParam_EnableThreadCache        //!< スレッドキャッシュを利用する
};

/**
 * @brief   各スレッドに渡す情報を定義した構造体です。
 */
struct AllocateInfo
{
    nn::mem::StandardAllocator* pAllocator;
    int allocatePattern;       // 各スレッドに渡されるユニークなサイズとアライメントの組み合わせのパターン
};

/**
 * @brief   テストで利用するテストフィクスチャです。
 */
class StandardAllocatorThreadSafeTest : public ::testing::TestWithParam< std::pair<StandardAllocatorTestParam, size_t> >
{
protected:

    /**
     * @brief   テスト開始時に毎回呼び出される関数です。
     */
    virtual void SetUp()
    {
        ASSERT_TRUE(nn::os::IsVirtualAddressMemoryEnabled());
        std::pair<StandardAllocatorTestParam, size_t> pattern = GetParam();
        size_t virtualSize = pattern.second * 1024 * 1024 * 1024;
        ASSERT_LE(virtualSize, 63ull * 1024 * 1024 * 1024); // NX での最大値
        m_Param = pattern.first;

        // 利用できるコア番号とコア数の取得
        // 最大 MaxThreadCount 個取得する
        nn::Bit64 coreMask = nn::os::GetThreadAvailableCoreMask();
        m_MaxCoreCount = 0;
        for(int i = 0; i < 64; ++i)
        {
            if(coreMask & (1ULL << i))
            {
                m_CoreNumber[m_MaxCoreCount] = static_cast<int>(i);
                m_MaxCoreCount++;
                if(m_MaxCoreCount >= MaxThreadCount)
                {
                    break;
                }
            }
        }

        NN_LOG("use %d core.\n", m_MaxCoreCount);

        // ヒープを初期化
        if (pattern.first == StandardAllocatorTestParam_DisableThreadCache)
        {
            NN_LOG("ThreadCache: disabled heapSize: %zu GiB\n", pattern.second);
            m_Allocator.Initialize(nullptr, virtualSize);
        }
        else
        {
            NN_LOG("ThreadCache: enabled heapSize: %zu GiB\n", pattern.second);
            m_Allocator.Initialize(nullptr, virtualSize, true);
        }
    }

    /**
     * @brief   テスト終了時に毎回呼び出される関数です。
     */
    virtual void TearDown()
    {
        if (m_HeapStartAddr)
        {
            m_Allocator.Finalize();
        }
    }

    /**
     * @brief       スレッドの動作を開始します
     * @param[in]   functionCount   動作させるスレッド関数の数
     */
    void BeginThread(const int functionCount)
    {
        for(int i = 0; i < functionCount && i < m_MaxCoreCount; ++i)
        {
            nn::os::StartThread(&g_Thread[i]);
        }

        for(int i = 0; i < functionCount && i < m_MaxCoreCount; ++i)
        {
            nn::os::WaitThread(&g_Thread[i]);
        }

        for(int i = 0; i < functionCount && i < m_MaxCoreCount; ++i)
        {
            nn::os::DestroyThread(&g_Thread[i]);
        }
    }

    /**
     * @brief       マルチスレッドによるテストを開始します。
     * @param[in]   pFunctions      スレッド関数の配列
     * @param[in]   functionCount   pFunctions の長さ
     */
    void DoMultiThreadTest(nn::os::ThreadFunction* pFunctions, const int functionCount)
    {
        ASSERT_TRUE(functionCount <= MaxThreadCount);

        nn::Result result;

        // functions の関数を動作させる
        for(int i = 0; i < functionCount && i < m_MaxCoreCount; ++i)
        {
            pInfo[i].pAllocator = &m_Allocator;
            pInfo[i].allocatePattern = i;
            result = nn::os::CreateThread(&g_Thread[i], pFunctions[i], &pInfo[i], &g_ThreadStack[i], ThreadStackSize, nn::os::DefaultThreadPriority, m_CoreNumber[i]);
            ASSERT_TRUE(result.IsSuccess());
        }

        BeginThread(functionCount);
    }

protected:
    nn::mem::StandardAllocator m_Allocator;
    void* m_HeapStartAddr;                      // グローバルヒープの開始アドレス
    int m_CoreNumber[MaxThreadCount];           // 利用できる CPU コアの番号
    int m_MaxCoreCount;                         // 利用できる CPU コア数
    AllocateInfo pInfo[MaxThreadCount];
    StandardAllocatorTestParam m_Param;
};

// =============
// スレッド関数
// =============

/**
 * @brief   確保に失敗するまでメモリブロックの確保を行います。
 */
void Allocate(void* arg)
{
    AllocateInfo* pInfo = reinterpret_cast<AllocateInfo*>(arg);

    size_t allocateSize = pAllocateSize[0];
    size_t alignment = pAlignment[0];
    // pAllocateSize の beginIndex ～ beginIndex + range の範囲のサイズを alloc する
    int beginIndex = (pInfo->allocatePattern * 4) % AllocateSizeCount;
    int i = beginIndex;
    int j = 0;
    const int range = 4;

    allocateSize = pAllocateSize[i];

    while(pInfo->pAllocator->Allocate(allocateSize, alignment) != NULL)
    {
        if (pInfo->pAllocator->Allocate(allocateSize, alignment) == NULL) { break;  }
        i++;
        if(i >= beginIndex + range)
        {
            i = beginIndex;
        }
        j++;
        if(j >= AlignmentCount)
        {
            j = 0;
        }
        allocateSize = pAllocateSize[i];
        alignment = pAlignment[j];
    }
}

/**
 * @brief   メモリブロックの確保と解放を行います。
 * @details また、確保済みのメモリブロックが破壊されていないか確認します。
 */
void AllocateFree(void* arg)
{
    AllocateInfo* pInfo = reinterpret_cast<AllocateInfo*>(arg);

    const int repeatCount = 1000;
    size_t allocateSize = pAllocateSize[0];
    size_t alignment = pAlignment[0];
    int beginIndex = (pInfo->allocatePattern * 4) % AllocateSizeCount;
    int i = beginIndex;
    int j = 0;
    const int range = 4;

    allocateSize = pAllocateSize[i];

    for(int k = 0; k < repeatCount; ++k)
    {
        void* ptr = pInfo->pAllocator->Allocate(allocateSize, alignment);
        ASSERT_NOT_NULL(ptr);
        ASSERT_GE(pInfo->pAllocator->GetSizeOf(ptr), allocateSize);
        pInfo->pAllocator->Free(ptr);
        i++;
        if(i >= beginIndex + range)
        {
            i = beginIndex;
        }
        j++;
        if(j >= AlignmentCount)
        {
            j = 0;
        }
        allocateSize = pAllocateSize[i];
        alignment = pAlignment[j];
    }
}

/**
 * @brief   メモリブロックの再確保を行います。
 * @details また、確保済みのメモリブロックが破壊されていないか確認します。
 */
void Reallocate(void* arg)
{
    AllocateInfo* pInfo = reinterpret_cast<AllocateInfo*>(arg);

    const int repeatCount = 1000;
    size_t allocateSize = pAllocateSize[0];
    size_t reallocateSize = pAllocateSize[0];
    int beginIndex = (pInfo->allocatePattern * 4) % AllocateSizeCount;
    int i = beginIndex;
    int j = 0;
    const int range = 4;

    allocateSize = pAllocateSize[i];
    void* ptr = pInfo->pAllocator->Allocate(allocateSize);
    ASSERT_NOT_NULL(ptr);

    for (int k = 0; k < repeatCount; ++k)
    {
        ptr = pInfo->pAllocator->Reallocate(ptr, reallocateSize);
        ASSERT_NOT_NULL(ptr);
        ptr = pInfo->pAllocator->Reallocate(ptr, allocateSize);
        ASSERT_NOT_NULL(ptr);
        i++;
        if (i >= beginIndex + range)
        {
            i = beginIndex;
        }
        j++;
        if (j >= AlignmentCount)
        {
            j = 0;
        }
        allocateSize = pAllocateSize[i];
        reallocateSize = pAllocateSize[j];
    }
}

/**
 * @brief   全てのメモリブロックで実行するコールバック関数です。
 * @details 今回のテスト内容ではこの関数の中身は関係しないため、処理は適当です。
 */
int WalkCallback(void* addr, size_t size, void* userPtr)
{
    NN_UNUSED(size);
    NN_UNUSED(userPtr);
    char* str = reinterpret_cast<char*>(addr);
    str[0] = 1;

    return 0;
}

/**
 * @brief   Visit 関数を繰り返し呼びます。
 */
void Walk(void* arg)
{
    AllocateInfo* pInfo = reinterpret_cast<AllocateInfo*>(arg);
    const int repeatCount = 1000;
    for (int i = 0; i < repeatCount; ++i)
    {
        pInfo->pAllocator->WalkAllocatedBlocks(WalkCallback, nullptr);
    }
}

}   // unnamed namespace

// =======
// テスト
// =======

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
/**
 * @brief   複数スレッドを作成し、確保を繰り返します。
 */
TEST_P(StandardAllocatorThreadSafeTest, Allocate)
{
    const int FunctionCount = 8;    // 並列実行する関数の数 ( < MaxThreadCount)

    // 並列実行する関数
    // ただし、環境によっていくつのコアがあるかはわからないため、どの関数までが実行されるかは環境依存
    // 例えば、コア数が 4 なら、 functions[0] ～ functions[3] が並列実行される
    nn::os::ThreadFunction functions[FunctionCount] = { Allocate,    // コア0で実行したい関数
                                                        Allocate,    // コア1で実行したい関数
                                                        Allocate,    // コア2で実行したい関数
                                                        Allocate,    // コア3で実行したい関数
                                                        Allocate,    // コア4で実行したい関数
                                                        Allocate,    // コア5で実行したい関数
                                                        Allocate,    // コア6で実行したい関数
                                                        Allocate};   // コア7で実行したい関数
    DoMultiThreadTest(functions, FunctionCount);
}
#endif  // defined(NN_BUILD_CONFIG_OS_HORIZON)

/**
 * @brief   Allocate テストを解放も含めて行ったものです。
 */
TEST_P(StandardAllocatorThreadSafeTest, AllocateFree)
{
    const int FunctionCount = 8;
    nn::os::ThreadFunction functions[FunctionCount] = { AllocateFree,
                                                        AllocateFree,
                                                        AllocateFree,
                                                        AllocateFree,
                                                        AllocateFree,
                                                        AllocateFree,
                                                        AllocateFree,
                                                        AllocateFree};
    DoMultiThreadTest(functions, FunctionCount);
}

/**
 * @brief   Allocate テストを Reallocate に置き換えたものです。
 */
TEST_P(StandardAllocatorThreadSafeTest, Reallocate)
{
     const int FunctionCount = 8;
     nn::os::ThreadFunction functions[FunctionCount] = { Reallocate,
                                                         Reallocate,
                                                         Reallocate,
                                                         Reallocate,
                                                         Reallocate,
                                                         Reallocate,
                                                         Reallocate,
                                                         Reallocate };
     DoMultiThreadTest(functions, FunctionCount);
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
/**
 * @brief   複数スレッドで確保済みのメモリブロックに特定の処理を行います。
 * @details スレッドセーフでない場合、すべてのメモリブロックをたどっている途中にに別スレッドがメモリブロックを解放してしまうと、メモリブロックをたどれなくなります。
 */
TEST_P(StandardAllocatorThreadSafeTest, Walk)
{
     const int FunctionCount = 8;
     nn::os::ThreadFunction functions[FunctionCount] = { Walk,
                                                         Allocate,
                                                         Allocate,
                                                         Allocate,
                                                         Allocate,
                                                         Allocate,
                                                         Allocate,
                                                         Allocate };
     DoMultiThreadTest(functions, FunctionCount);
}
#endif  // #if defined(NN_BUILD_CONFIG_OS_HORIZON)

INSTANTIATE_TEST_CASE_P(SwitchThreadCache,
                        StandardAllocatorThreadSafeTest,
                        testing::Values(std::make_pair(StandardAllocatorTestParam_DisableThreadCache, 4),
                                        std::make_pair(StandardAllocatorTestParam_DisableThreadCache, 60),
                                        std::make_pair(StandardAllocatorTestParam_EnableThreadCache, 4),
                                        std::make_pair(StandardAllocatorTestParam_EnableThreadCache, 60)));
